/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.fill;

import java.awt.Color;
import java.awt.geom.Point2D;

import net.sf.jaxodraw.object.JaxoExtendedObject;
import net.sf.jaxodraw.object.JaxoFillColorObject;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoGeometry;


/** A general object that can be filled.
 * @since 2.0
 */
public abstract class JaxoFillObject extends JaxoExtendedObject
    implements JaxoFillColorObject {

    private static final long serialVersionUID = 2L;

    /** The fill color of this fillObject. */
    private Color fillColor;
    private boolean filled = true;

    // Bean getter and setter methods

    /** {@inheritDoc} */
    public final Color getFillColor() {
        return fillColor;
    }

    /** {@inheritDoc} */
    public final void setFillColor(final Color color) {
        final Color old = fillColor;
        this.fillColor = color;
        firePropertyChange("fillColor", old, fillColor);
    }

    /** {@inheritDoc} */
    public final boolean isFilled() {
        return filled;
    }

    /** {@inheritDoc} */
    public final void setFilled(final boolean value) {
        final Boolean old = Boolean.valueOf(filled);
        filled = value;
        firePropertyChange("filled", old, Boolean.valueOf(filled));
    }

    /** {@inheritDoc} */
    @Override
    public boolean isCopy(final JaxoObject comp) {
        boolean isCopy = false;

        if (comp instanceof JaxoFillObject) {
            final JaxoFillObject tmp = (JaxoFillObject) comp;
            if ((tmp.getFillColor().equals(getFillColor()))
                    && (tmp.isFilled() == isFilled()) && super.isCopy(tmp)) {
                isCopy = true;
            }
        }

        return isCopy;
    }

    /** Sets all parameters from the given object to the current one.
     * @param temp The object to copy from.
     */
    public void copyFrom(final JaxoFillObject temp) {
        super.copyFrom(temp);
        setFillColor(temp.getFillColor());
        setFilled(temp.isFilled());
    }

    /** {@inheritDoc} */
    @Override
    public void setState(final JaxoObject o) {
        if (o instanceof JaxoFillObject) {
            copyFrom((JaxoFillObject) o);
        } else {
            throw new UnsupportedOperationException("Cannot copy from super type!");
        }
    }

    /** {@inheritDoc} */
    public void rescaleObject(final int orx, final int ory, final float scale) {
        final int newRelWidth = Math.round(this.getRelSize().width * scale);
        final int newRelHeight = Math.round(this.getRelSize().height * scale);

        final Point2D newP =
            JaxoGeometry.scaledPoint(orx, ory, scale, getX(), getY());

        setLocation((int) Math.round(newP.getX()),
            (int) Math.round(newP.getY()));

        this.setRelWAndH(newRelWidth, newRelHeight);
    }

    /** {@inheritDoc} */
    @Override
    public void setPreferences() {
        super.setPreferences();
        setFilled(true);
        setFillColor(JaxoColor.getDefaultFillColor(getColor()));
    }
}
