/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object;

import java.awt.Color;
import java.awt.Component;

import java.awt.Font;
import java.awt.Point;
import javax.swing.event.ChangeListener;

import net.sf.jaxodraw.object.arrow.JaxoArrow;
import net.sf.jaxodraw.util.Location;

/**
 * A panel to edit properties of JaxoObjects.
 *
 * @since 2.0
 */
public interface JaxoObjectEditPanel {

    /** Defines an EditPanel for a simple color, of eg a vertex, a text, etc. */
    int TYPE_COLOR = 1;

    /** Defines a ColorEditPanel for a the line color of a fill object. */
    int TYPE_LINE_COLOR = 2;

    /** Defines a ColorEditPanel for a fill color. */
    int TYPE_FILL_COLOR = 3;

    /**
     * ChangeEvents will be fired everytime the object that is being
     * edited changes.
     *
     * @param l The listener to add.
     */
    void addChangeListener(ChangeListener l);

    /**
     * Display the EditPanel for the current object.
     *
     * @param parent The parent component for the edit panel window
     * @param l The location for for the edit panel window
     */
    void show(Component parent, Location l);

    /**
     * Determines wether the object has been modified.
     *
     * @return True if the object has been modified.
     */
    boolean hasChanged();

    /**
     * Set title (message key) and icon (filename) that are displayed in the dialog.
     *
     * @param newTitle the title.
     * @param newIcon the icon.
     */
    void setTitleAndIcon(String newTitle, String newIcon);

    /**
     * Adds a 3points panel at the given position.
     * @param points The points of the arc object to be edited by this panel.
     *
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void add3PointsPanel(Point[] points, int gridx, int gridy);

    /**
     * Adds a 3points panel at the given position with the given width.
     *
     * @param points The points of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     * @param gridwidth The width of the subpanel.
     */
    void add3PointsPanel(Point[] points, int gridx, int gridy, int gridwidth);

    /**
     * Adds a 4points panel at the given position.
     *
     * @param points The points of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void add4PointsPanel(Point[] points, int gridx, int gridy);

    /**
     * Adds a 4points panel at the given position with the given width.
     *
     * @param points The points of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     * @param gridwidth The width of the subpanel.
     */
    void add4PointsPanel(Point[] points, int gridx, int gridy, int gridwidth);

    /**
     * Adds a position edit panel at the given position with the given width.
     *
     * @param x The x position of the object to be edited by this panel.
     * @param y The y position of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addPositionPanel(int x, int y, int gridx, int gridy);

    /**
     * Adds a 2PointEdit panel at the given position with the given width.
     *
     * @param points The points of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void add2PointsPanel(Point[] points, int gridx, int gridy);

    /**
     * Adds a XYR panel at the given position.
     *
     * @param x The x position of the object to be edited by this panel.
     * @param y The r position of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     * @param r The radius of the object to be edited by this panel.
     */
    void addXYRPanel(int x, int y, int r, int gridx, int gridy);

    /**
     * Adds a position/size panel at the given position.
     *
     * @param x The x position of the object to be edited by this panel.
     * @param y The y position of the object to be edited by this panel.
     * @param w The width of the object to be edited by this panel.
     * @param h The height of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addCenterSizePanel(int x, int y, int w, int h, int gridx,
        int gridy);

    /**
     * Adds a dash panel for an arc object at the given position.
     *
     * @param dash The dash sizie of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addDashPanel(float dash, int gridx, int gridy);

    /**
     * Adds a stroke panel for an arc object at the given position.
     *
     * @param width The the stroke width of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addStrokePanel(float width, int gridx, int gridy);

    /**
     * Adds a scale panel for an object at the given position.
     *
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addReScalePanel(int gridx, int gridy);

    /**
     * Adds a wiggle panel for an object at the given position.
     *
     * @param amp The oamplitude of the bject to be edited by this panel.
     * @param wiggles The wiggles of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addWigglePanel(int amp, int wiggles, int gridx, int gridy);

    /**
     * Adds a wiggle panel for a group at the given position.
     *
     * @param amp The amplitude of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addWigglePanel(int amp, int gridx, int gridy);

    /**
     * Adds a symmetry panel for a photon line at the given position.
     *
     * @param symm The symmetry of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addSymmPanel(boolean symm, int gridx, int gridy);

    /**
     * Adds a stretching panel for a photon bezier at the given position.
     *
     * @param noFreqStretching The stretching parameter of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addStretchingPanel(boolean noFreqStretching, int gridx, int gridy);

    /**
     * Adds a rotation panel for a vertex at the given position.
     *
     * @param angle The rotation angle of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addRotationPanel(int angle, int gridx, int gridy);

        /**
     * Adds an arrow panel for an object at the given position.
     *
     * @param arrow The arrow of the object to be edited by this panel.
     * @param flip The flip of the arrow.
     * @param oldPosition The old position of the Arrow
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addArrowPanel(boolean arrow, boolean flip, float oldPosition, int gridx, int gridy);

    /**
     * Adds an edit panel for an arrow at the given position.
     * If the arrow does not provide an edit panel, ie if
     * {@link net.sf.jaxodraw.object.arrow.JaxoArrow#getEditPanel} returns null,
     * then nothing happens.
     *
     * @param arrow The arrow to be edited.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addArrowOptionsEditPanel(JaxoArrow arrow, int gridx, int gridy);

    /**
     * Adds a double line panel for an object at the given position.
     *
     * @param dLine The doubleLine of the object to be edited by this panel.
     * @param dlSep The double line separation of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addDoubleLinePanel(boolean dLine, float dlSep, int gridx, int gridy);

    /**
     * Adds a line color panel at the given position.
     *
     * @param color The color of the object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addLineColorPanel(Color color, int gridx, int gridy);

    /**
     * Adds a fill and a line color panel at the given positions.
     * The line panel will be disabled (dynamically) based on the
     * fill color/grayscale.
     *
     * @param color The color of the object to be edited by this panel.
     * @param fillColor The fill color of the object.
     * @param isFilled Whether the object is filled.
     * @param fillX The x position of the fill subpanel.
     * @param fillY The y position of the fill subpanel.
     * @param lineX The x position of the line subpanel.
     * @param lineY The y position of the line subpanel.
     */
    void addFillLineColorPanels(Color color, final Color fillColor,
        boolean isFilled, int fillX, int fillY, int lineX, int lineY);

    /**
     * Adds a color panel at the given position.
     *
     * @param color The color of the object to be edited by this panel.
     * @param type The color mode.
     * Should be one of {@link #TYPE_COLOR}, {@link #TYPE_LINE_COLOR} or
     *      {@link #TYPE_FILL_COLOR}.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addColorPanel(Color color, int type, int gridx, int gridy);

    /**
     * Adds a text edit panel at the given position.
     *
     * @param text The text object to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addTextPanel(String text, int gridx, int gridy);

    /**
     * Adds a postscript text font edit panel at the given position.
     *
     * @param font The font of the text to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     * @param gridwidth The width of the subpanel.
     */
    void addPSFontPanel(Font font, int gridx, int gridy, int gridwidth);

    /**
     * Adds a latex align edit panel at the given position.
     *
     * @param allign The align property of the text to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addLatexAlignPanel(int allign, int gridx, int gridy);

    /**
     * Adds a latex font size edit panel at the given position.
     *
     * @param size The text size of the text to be edited by this panel.
     * @param gridx The x position of the subpanel.
     * @param gridy The y position of the subpanel.
     */
    void addLatexFontSizePanel(int size, int gridx, int gridy);
}
