/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.logging;

import java.io.IOException;

import java.text.SimpleDateFormat;

import java.util.Date;
import java.util.Locale;
import java.util.logging.FileHandler;
import java.util.logging.Level;
import java.util.logging.LogRecord;


/**
 * File implementation of JaxoLogger.
 * Uses a FileHandler, ie logs are written to a log file.
 *
 * @since 2.0
 */
public final class JaxoFileHandler extends FileHandler {

    private static final String DEFAULT_FILE = "jaxodraw%g.log";
    private static final int DEFAULT_LIMIT = 1048576; // 1 Mb
    private static final int DEFAULT_NUM = 3;

    /**
     * Constructor.
     *
     * @param logDir The directory where log files will be stored.
     *      <b>This directory has to exist!</b>
     * @param pattern the pattern for naming the output file.
     * @param limit limit the maximum number of bytes to write to any one file.
     * @param logFiles the number of files to use.
     * @throws IOException if there are IO problems opening the files.
     */
    public JaxoFileHandler(final String logDir, final String pattern, final int limit, final int logFiles)
            throws IOException {

        super(logDir + pattern, limit, logFiles, true);

        setLevel(Level.ALL);
        setFormatter(new TimeLogFormatter());
    }

    /**
     * Constructor.
     *
     * @param logDir The directory where log files will be stored.
     *      <b>This directory has to exist!</b>
     * @throws IOException if there are IO problems opening the files.
     */
    public JaxoFileHandler(final String logDir) throws IOException {
        this(logDir, DEFAULT_FILE, DEFAULT_LIMIT, DEFAULT_NUM);
    }

    /**
     * Constructor.
     *
     * @param logDir The directory where log files will be stored.
     *      <b>This directory has to exist!</b>
     * @param pattern the pattern for naming the output file.
     * @throws IOException if there are IO problems opening the files.
     */

    public JaxoFileHandler(final String logDir, final String pattern)
            throws IOException {
        this(logDir, pattern, DEFAULT_LIMIT, DEFAULT_NUM);
    }

    /**
     * Constructor.
     *
     * @param logDir The directory where log files will be stored.
     *      <b>This directory has to exist!</b>
     * @param pattern the pattern for naming the output file.
     * @param limit the maximum number of bytes to write to any one file.
     * @throws IOException if there are IO problems opening the files.
     */

    public JaxoFileHandler(final String logDir, final String pattern, final int limit)
            throws IOException {
        this(logDir, pattern, limit, DEFAULT_NUM);
    }

      //
     // private
    //

    /**
     * Formats log records.
     */
    private static class TimeLogFormatter
            extends LogFormatter {
        /**
         * Prepends the date to standard JaxoDraw format.
         *
         * @param rec the LogRecord to format.
         */
        @Override
        public String format(final LogRecord rec) {
            final String log = super.format(rec);
            final SimpleDateFormat formatter =
                    new SimpleDateFormat("MMM d HH:mm:ss ", Locale.ENGLISH);

            final String time = formatter.format(new Date());
            return time + log;
        }
    }

}
