/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.swing;

import java.awt.Component;
import java.awt.Font;

import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;

import net.sf.jaxodraw.util.JaxoDictionary;
import net.sf.jaxodraw.util.JaxoLooknFeel;


/** Renderer for Integer values representing Font styles.
 * The text is rendered in the same font style it represents. Negative values
 * and null are interpreted as no explicit font style (taking the default).
 * @since 2.0
 */
public class JaxoFontStyleRenderer extends DefaultListCellRenderer {
    private static final long serialVersionUID = 7526471155622776147L;
    private static final JaxoDictionary LANGUAGE = new JaxoDictionary(JaxoFontStyleRenderer.class);
    private String none;
    private String plain;
    private String bold;
    private String italic;
    private String boldItalic;
    {
        JaxoLooknFeel.registerComponent(this);
    }

    /** {@inheritDoc} */
    @Override
    public void updateUI() {
        super.updateUI();

        none = LANGUAGE.value("/FontStyle.none");
        plain = LANGUAGE.value("/FontStyle.plain");
        bold = LANGUAGE.value("/FontStyle.bold");
        italic = LANGUAGE.value("/FontStyle.italic");
        boldItalic = LANGUAGE.value("/FontStyle.boldItalic");
    }

    /** {@inheritDoc} */
    @Override
    public void setFont(final Font font) {
        // Workaround JComboBox/BasicComboBoxUI
    }

    /** {@inheritDoc} */
    @Override
    public Component getListCellRendererComponent(final JList list, final Object value,
        final int index, final boolean isSelected, final boolean cellHasFocus) {
        Object newValue = value;

        if (newValue == null) {
            newValue = Integer.valueOf(-1);
        }

        String text;

        final int style = ((Integer) newValue).intValue();

        switch (style) {
            case Font.PLAIN:
                text = plain;
                break;
            case Font.BOLD:
                text = bold;
                break;
            case Font.ITALIC:
                text = italic;
                break;
            case Font.BOLD | Font.ITALIC:
                text = boldItalic;
                break;
            default:
                text = none;
                break;
        }

        super.getListCellRendererComponent(list, text, index, isSelected, cellHasFocus);

        if (style >= 0) {
            super.setFont(list.getFont().deriveFont(((Integer) newValue).intValue()));
        } else {
            super.setFont(list.getFont());
        }

        return this;
    }
}
