/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.edit;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.TitledBorder;

import net.sf.jaxodraw.gui.swing.spinner.JaxoFixedJSpinner;


/** A panel to change the x,y and radius parameters of an object.
 * @since 2.0
 */
public class JaxoXYREditPanel extends JaxoEditPanel {
    private static final long serialVersionUID = 7526471155622776147L;
    private transient JSpinner spxin;
    private transient JSpinner spyin;
    private transient JSpinner sprad;

    /** Constructor: returns a XYR edit panel with
     * default value 200 for all variables.
     */
    public JaxoXYREditPanel() {
        this(200, 200, 200);
    }

    /** Constructor: returns a XYR edit panel with
     * given values for the variables.
     * @param x Default value for the x sub-panel.
     * @param y Default value for the y sub-panel.
     * @param r Default value for the r sub-panel.
     */
    public JaxoXYREditPanel(final int x, final int y, final int r) {
        super();
        // Dimension options
        final Dimension size = new Dimension(120, 20);
        final GridLayout layout = new GridLayout(0, 2);

        final JPanel pxin = new JPanel();
        pxin.setLayout(layout);

        final SpinnerNumberModel modelXin = new SpinnerNumberModel();
        modelXin.setValue(Integer.valueOf(x));
        final JLabel xin = createLabel("X: ");
        pxin.add(xin);

        this.spxin = new JaxoFixedJSpinner(modelXin);
        spxin.setName("spxin");
        pxin.add(spxin);

        final JPanel pyin = new JPanel();
        pyin.setLayout(layout);

        final SpinnerNumberModel modelYin = new SpinnerNumberModel();
        modelYin.setValue(Integer.valueOf(y));
        final JLabel yin = createLabel("Y: ");
        pyin.add(yin);

        this.spyin = new JaxoFixedJSpinner(modelYin);
        spyin.setName("spyin");
        pyin.add(spyin);

        final JPanel prad = new JPanel();
        prad.setLayout(layout);

        final SpinnerNumberModel modelrad = new SpinnerNumberModel(r, 0, Integer.MAX_VALUE, 1);
        final JLabel rad = createI18NLabel("JaxoXYREditPanel.radius");
        prad.add(rad);

        this.sprad = new JaxoFixedJSpinner(modelrad);
        sprad.setName("sprad");
        prad.add(sprad);

        final JPanel dimSPanel = new JPanel();
        final GridBagLayout gb = new GridBagLayout();
        final GridBagConstraints sc = new GridBagConstraints();
        dimSPanel.setLayout(gb);

        sc.gridx = 0;
        sc.gridy = 0;
        pxin.setPreferredSize(size);
        dimSPanel.add(pxin, sc);

        sc.gridx = 0;
        sc.gridy = 1;
        sc.insets = new Insets(5, 0, 5, 0);
        pyin.setPreferredSize(size);
        dimSPanel.add(pyin, sc);

        sc.gridx = 1;
        sc.gridy = 0;
        sc.gridheight = 2;
        sc.insets = new Insets(0, 0, 5, 5);
        prad.setPreferredSize(size);
        dimSPanel.add(prad, sc);

        setLineBoxLayout();

        final TitledBorder dimSTitle = createI18NBorder("JaxoXYREditPanel.title");
        this.setBorder(dimSTitle);
        this.add(dimSPanel);
    }

    /** {@inheritDoc} */
    public void addEditPanelListener(final JaxoEditPanelListener listener) {
        if (listener != null) {
            spxin.addChangeListener(listener);
            spyin.addChangeListener(listener);
            sprad.addChangeListener(listener);
        }
    }
}
