/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.edit;

import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.LayoutManager;

import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.TitledBorder;

import net.sf.jaxodraw.gui.swing.JaxoTitledBorder;
import net.sf.jaxodraw.gui.swing.spinner.JaxoFixedJSpinner;
import net.sf.jaxodraw.util.JaxoLanguage;


/** A parent class for edit panel components.
 * @since 2.0
 */
public abstract class JaxoEditPanel extends JPanel {
    /** The common layout. */
    private transient GridLayout panelLayout = new GridLayout(1, 2, 10, 0);
    /** The common one-column layout. */
    private transient GridLayout oneColumnPanelLayout =
        new GridLayout(1, 1, 10, 0);
    /** The minimum size for a panel holding just a checkbox. */
    private transient Dimension checkBoxPanelSize = new Dimension(110, 60);

    /** Create without LayoutManager. */
    protected JaxoEditPanel() {
        this(null);
    }

    /**
     * Create with given LayoutManager.
     * @param l A LayoutManager, may be null.
     */
    protected JaxoEditPanel(final LayoutManager l) {
        super(l, false);
    }

    /** Set LayoutManager to a new LINE_AXIS BoxLayout */
    protected void setLineBoxLayout() {
        setLayout(new BoxLayout(this, BoxLayout.LINE_AXIS));
    }

    /** Returns a TitledBorder with translated text.
     * @param key A key to be translated by the JaxoLanguage class.
     * @return A TitledBorder with translated text.
     */
    protected TitledBorder createI18NBorder(final String key) {
        return new JaxoTitledBorder(JaxoLanguage.translate(key));
    }

    /** Returns a JLabel with translated text.
     * @param key A key to be translated by the JaxoLanguage class.
     * @return A JLabel with translated text.
     */
    protected JLabel createI18NLabel(final String key) {
        return new JLabel(JaxoLanguage.label(key), JLabel.TRAILING);
    }

    /** Returns a TitledBorder with given text.
     * @param title The title text of the border.
     * @return A TitledBorder with given text.
     */
    protected TitledBorder createBorder(final String title) {
        return new JaxoTitledBorder(title);
    }

    /** Returns a JLabel with given text.
     * @param title The text for the label.
     * @return A JLabel with given text.
     */
    protected JLabel createLabel(final String title) {
        return new JLabel(title, JLabel.TRAILING);
    }

    /** Returns a JSpinner.
     * @param value The value of the corresponding SpinnerNumberModel.
     * @param min The min value of the corresponding SpinnerNumberModel.
     * @param max The max value of the corresponding SpinnerNumberModel.
     * @param step The step size of the corresponding SpinnerNumberModel.
     * @param name The name to set for the JSpinner.
     * @return A JSpinner with the given values.
     */
    protected JSpinner getSpinner(final float value, final float min, final float max,
        final float step, final String name) {
        final SpinnerNumberModel modelstin =
            new SpinnerNumberModel(value, min, max, step);
        final JSpinner spstin = new JaxoFixedJSpinner(modelstin);
        spstin.setName(name);
        return spstin;
    }

    /** Returns a JSpinner.
    * @param value The value of the corresponding SpinnerNumberModel.
    * @param min The min value of the corresponding SpinnerNumberModel.
    * @param max The max value of the corresponding SpinnerNumberModel.
    * @param step The step size of the corresponding SpinnerNumberModel.
    * @param name The name to set for the JSpinner.
    * @return A JSpinner with the given values.
    */
    protected JSpinner getSpinner(final int value, final int min, final int max, final int step,
        final String name) {
        final SpinnerNumberModel modelstin =
            new SpinnerNumberModel(value, min, max, step);
        final JSpinner spstin = new JaxoFixedJSpinner(modelstin);
        spstin.setName(name);
        return spstin;
    }

    /** Adds the given listener as an ActionListener to this panel.
     * If listener is null, does nothing.
     * @param listener The listener to add to this panel.
     */
    public abstract void addEditPanelListener(final JaxoEditPanelListener listener);

    /**
     * Get the PanelLayout.
     * @return the PanelLayout.
     */
    protected GridLayout getPanelLayout() {
        return panelLayout;
    }

    /**
     * Set the PanelLayout.
     * @param newLayout the panelLayout.
     */
    protected void setPanelLayout(final GridLayout newLayout) {
        this.panelLayout = newLayout;
    }

    /**
     * Returns the layout of one column.
     * @return the layout of one column.
     */
    protected GridLayout getOneColumnPanelLayout() {
        return oneColumnPanelLayout;
    }

    /**
     * Sets the layout of one column.
     * @param newLayout the layout of one column.
     */
    protected void setOneColumnPanelLayout(final GridLayout newLayout) {
        this.oneColumnPanelLayout = newLayout;
    }

    /**
     * Returns the size of a single checkbox panel.
     * @return the size of a single checkbox panel.
     */
    protected Dimension getCheckBoxPanelSize() {
        return checkBoxPanelSize;
    }

    /**
     * Sets the size of a single checkbox panel.
     * @param newSize the size of a single checkbox panel.
     */
    protected void setCheckBoxPanelSize(final Dimension newSize) {
        this.checkBoxPanelSize = newSize;
    }
}
