/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.edit;

import java.awt.Color;
import java.awt.GridLayout;

import javax.swing.JCheckBox;
import javax.swing.border.TitledBorder;

import net.sf.jaxodraw.gui.panel.button.JaxoColorButton;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoLanguage;


/** A panel to change the color of an object.
 * @since 2.0
 */
public class JaxoColorEditPanel extends JaxoEditPanel {
    private static final long serialVersionUID = 7526471155622776147L;

    private transient JaxoColorButton colorButton;

    private transient JCheckBox filledBox;

    /** Constructor: returns a color edit panel with default
     * type TYPE_COLOR and default color black.
     */
    public JaxoColorEditPanel() {
        this(JaxoOptionsPanel.TYPE_COLOR, false, JaxoColor.BLACK, true);
    }

    /** Constructor: returns a color edit panel with the given color.
     * @param buttonColor The color of the button in this panel.
     */
    public JaxoColorEditPanel(final Color buttonColor) {
        this(JaxoOptionsPanel.TYPE_COLOR, false, buttonColor, true);
    }

    /** Constructor: returns a color edit panel of the given type
     * with the given color.
     * @param type The type of this edit panel. If this is not one
     * of the three types defined by this class, it gets set to TYPE_COLOR.
     * @param disabled Disables the color button of this edit panel.
     * @param buttonColor The color of the button in this panel.
     */
    public JaxoColorEditPanel(final int type, final boolean disabled, final Color buttonColor) {
        this(type, disabled, buttonColor, true);
    }

    /** Constructor: returns a fill color edit panel with the given color.
     * of the three types defined by this class, it gets set to TYPE_COLOR.
     * @param disabled Disables the color button of this edit panel.
     * @param buttonColor The color of the button in this panel.
     * @param filled If the panel is for an object that can be optionally filled.
     */
    public JaxoColorEditPanel(final boolean disabled, final Color buttonColor,
        final boolean filled) {
        this(JaxoOptionsPanel.TYPE_FILL_COLOR, disabled, buttonColor, filled);
    }

    private JaxoColorEditPanel(final int type, final boolean disabled, final Color buttonColor,
        final boolean filled) {
        super();
        int check = type;
        if ((check < JaxoOptionsPanel.TYPE_COLOR)
                || (check > JaxoOptionsPanel.TYPE_FILL_COLOR)) {
            check = JaxoOptionsPanel.TYPE_COLOR;
        }

        this.colorButton = new JaxoColorButton(buttonColor);

        final String title = getTitle(check);
        final TitledBorder panelBorder = createI18NBorder(title);

        if (disabled) {
            super.setEnabled(false);
            panelBorder.setTitleColor(JaxoColor.GRAYSCALE165);
            colorButton.setEnabled(false);
        }

        colorButton.setActionCommand(getActionCommand(check));
        this.setBorder(panelBorder);
        this.add(colorButton);

        if (check == JaxoOptionsPanel.TYPE_FILL_COLOR) {
            this.setLayout(new GridLayout(2, 0, 0, 10));
            filledBox =
                new JCheckBox(JaxoLanguage.translate(
                        "JaxoColorEditPanel.fillColor.filled"));
            filledBox.setSelected(filled);
            filledBox.setActionCommand("Filled");
            add(filledBox);

//            this.setPreferredSize(new Dimension(COLOR_PANEL_DIM.width,
//                    (COLOR_PANEL_DIM.height * 3) / 2));
        } else {
//            this.setPreferredSize(COLOR_PANEL_DIM);
            this.setLayout(new GridLayout(1, 0));
        }
    }

    /**
     * Sets the color of the button icon.
     *
     * @param color The color to set.
     */
    public void setColor(final Color color) {
        colorButton.setColor(color);
    }


    /**
     * the color of the button icon.
     *
     * @return The set color.
     */
    public Color getColor() {
        return colorButton.getColor();
    }

    /**
     * Set the button enabled.
     *
     * @param value true for enabled.
     */
    @Override
    public void setEnabled(final boolean value) {
        if (colorButton.isEnabled() != value) {
            colorButton.setEnabled(value);

            final TitledBorder b = createBorder(((TitledBorder) getBorder()).getTitle());

            if (!value) {
                b.setTitleColor(JaxoColor.GRAYSCALE165);
            }

            setBorder(b);
        }

        super.setEnabled(value);
    }

    /** Adds the given listener as an ActionListener to the color button of this panel.
     * If listener is null, does nothing.
     * @param listener The listener to add to this panel.
     */
    public void addEditPanelListener(final JaxoEditPanelListener listener) {
        if (listener != null) {
            colorButton.addActionListener(listener);

            if (filledBox != null) {
                filledBox.addActionListener(listener);
            }
        }
    }

    private String getActionCommand(final int type) {
        String result = "Color";
        if (type == JaxoOptionsPanel.TYPE_LINE_COLOR) {
            result = "Line_color";
        } else if (type == JaxoOptionsPanel.TYPE_FILL_COLOR) {
            result = "Fill_color";
        }
        return result;
    }

    private String getTitle(final int type) {
        String result = "JaxoColorEditPanel.color.title";
        if (type == JaxoOptionsPanel.TYPE_LINE_COLOR) {
            result = "JaxoColorEditPanel.lineColor.title";
        } else if (type == JaxoOptionsPanel.TYPE_FILL_COLOR) {
            result = "JaxoColorEditPanel.fillColor.title";
        }
        return result;
    }
}
