/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.edit;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Point;

import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.TitledBorder;

import net.sf.jaxodraw.gui.swing.spinner.JaxoFixedJSpinner;


/** A panel to change the x,y and radius parameters of an object.
 * @since 2.0
 */
public class Jaxo3PointsEditPanel extends JaxoEditPanel {
    private static final long serialVersionUID = 7526471155622776147L;
    private transient JSpinner spx1;
    private transient JSpinner spx2;
    private transient JSpinner spx3;
    private transient JSpinner spy1;
    private transient JSpinner spy2;
    private transient JSpinner spy3;

    /** Constructor: returns a 3Points edit panel with
     * default value 0 for all variables.
     */
    public Jaxo3PointsEditPanel() {
        this(new Point[] {new Point(), new Point(), new Point()});
    }

    /** Constructor: returns a 3Points edit panel with
     * given values for the variables.
     * @param points The initial points for this panel.
     */
    public Jaxo3PointsEditPanel(final Point[] points) {
        super();
        // Dimension options
        final Dimension size = new Dimension(120, 20);
        final GridLayout layout = new GridLayout(0, 2);

        final JPanel dimSPanel = new JPanel();
        final GridBagLayout gb = new GridBagLayout();
        final GridBagConstraints sc = new GridBagConstraints();
        dimSPanel.setLayout(gb);

        sc.gridx = 0;
        sc.gridy = 0;
        dimSPanel.add(getPx1Panel(points, size, layout), sc);

        sc.gridx = 0;
        sc.gridy = 1;
        sc.insets = new Insets(3, 0, 3, 0);
        dimSPanel.add(getPy1Panel(points, size, layout), sc);

        sc.gridx = 1;
        sc.gridy = 0;
        dimSPanel.add(getPx2Panel(points, size, layout), sc);

        sc.gridx = 1;
        sc.gridy = 1;
        sc.insets = new Insets(3, 0, 3, 0);
        dimSPanel.add(getPy2Panel(points, size, layout), sc);

        sc.gridx = 2;
        sc.gridy = 0;
        dimSPanel.add(getPx3Panel(points, size, layout), sc);

        sc.gridx = 2;
        sc.gridy = 1;
        sc.insets = new Insets(3, 0, 3, 0);
        dimSPanel.add(getPy3Panel(points, size, layout), sc);

        setLineBoxLayout();

        final TitledBorder dimSTitle =
            createI18NBorder("Jaxo3PointsEditPanel.title");

        this.setBorder(dimSTitle);
        this.add(dimSPanel);
    }

    /** {@inheritDoc} */
    public void addEditPanelListener(final JaxoEditPanelListener listener) {
        if (listener != null) {
            spx1.addChangeListener(listener);
            spx2.addChangeListener(listener);
            spx3.addChangeListener(listener);
            spy1.addChangeListener(listener);
            spy2.addChangeListener(listener);
            spy3.addChangeListener(listener);
        }
    }

      //
     // private
    //

    private JPanel getPx1Panel(final Point[] p, final Dimension size, final GridLayout layout) {
        final JPanel px1 = new JPanel();
        px1.setLayout(layout);

        final SpinnerNumberModel modelX1 = new SpinnerNumberModel();
        modelX1.setValue(Integer.valueOf(p[0].x));

        px1.add(createLabel("X1: "));

        this.spx1 = new JaxoFixedJSpinner(modelX1);
        spx1.setName("spx");
        px1.add(spx1);
        px1.setPreferredSize(size);

        return px1;
    }

    private JPanel getPx2Panel(final Point[] p, final Dimension size, final GridLayout layout) {
        final JPanel px2 = new JPanel();
        px2.setLayout(layout);

        final SpinnerNumberModel modelX2 = new SpinnerNumberModel();
        modelX2.setValue(Integer.valueOf(p[1].x));
        px2.add(createLabel("X2: "));

        this.spx2 = new JaxoFixedJSpinner(modelX2);
        spx2.setName("spx2");
        px2.add(spx2);
        px2.setPreferredSize(size);

        return px2;
    }

    private JPanel getPx3Panel(final Point[] p, final Dimension size, final GridLayout layout) {
        final JPanel px3 = new JPanel();
        px3.setLayout(layout);

        final SpinnerNumberModel modelX3 = new SpinnerNumberModel();
        modelX3.setValue(Integer.valueOf(p[2].x));
        px3.add(createLabel("X3: "));

        this.spx3 = new JaxoFixedJSpinner(modelX3);
        spx3.setName("spx3");
        px3.add(spx3);
        px3.setPreferredSize(size);

        return px3;
    }

    private JPanel getPy1Panel(final Point[] p, final Dimension size, final GridLayout layout) {
        final JPanel py1 = new JPanel();
        py1.setLayout(layout);

        final SpinnerNumberModel modelY1 = new SpinnerNumberModel();
        modelY1.setValue(Integer.valueOf(p[0].y));
        py1.add(createLabel("Y1: "));

        this.spy1 = new JaxoFixedJSpinner(modelY1);
        spy1.setName("spy");
        py1.add(spy1);
        py1.setPreferredSize(size);

        return py1;
    }

    private JPanel getPy2Panel(final Point[] p, final Dimension size, final GridLayout layout) {
        final JPanel py2 = new JPanel();
        py2.setLayout(layout);

        final SpinnerNumberModel modelY2 = new SpinnerNumberModel();
        modelY2.setValue(Integer.valueOf(p[1].y));
        py2.add(createLabel("Y2: "));

        this.spy2 = new JaxoFixedJSpinner(modelY2);
        spy2.setName("spy2");
        py2.add(spy2);
        py2.setPreferredSize(size);

        return py2;
    }

    private JPanel getPy3Panel(final Point[] p, final Dimension size, final GridLayout layout) {
        final JPanel py3 = new JPanel();
        py3.setLayout(layout);

        final SpinnerNumberModel modelY3 = new SpinnerNumberModel();
        modelY3.setValue(Integer.valueOf(p[2].y));
        py3.add(createLabel("Y3: "));

        this.spy3 = new JaxoFixedJSpinner(modelY3);
        spy3.setName("spy3");
        py3.add(spy3);
        py3.setPreferredSize(size);

        return py3;
    }
}
