/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.Color;
import java.awt.Dimension;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.border.Border;


/** Special kind of button used in the JaxoButtonPanel - most such buttons
 * can be "pressed" (modelled as the "selected" properties), indicating the
 * current mode. The button takes care that it looks good both in selected
 * and unselected state.
 * @since 2.0
 */
public class JaxoPanelButton extends JButton {
    private static final long serialVersionUID = 7526471155622776147L;
    private static final Color PRESSED_BACKGROUND;
    private static final Border UNPRESSED_BORDER;
    private static final Border PRESSED_BORDER;
    private static final Border UNPRESSED_DEFAULT_BORDER;
    private static final Border PRESSED_DEFAULT_BORDER;
    private static final Dimension BUTTON_MINIMUM_SIZE;
    private static final Dimension BUTTON_PREFERRED_SIZE;

    static {
        BUTTON_MINIMUM_SIZE = new Dimension(40, 40);
        BUTTON_PREFERRED_SIZE = new Dimension(40, 40);
        PRESSED_BACKGROUND = new Color(170, 170, 180);
        UNPRESSED_BORDER = BorderFactory.createRaisedBevelBorder();
        PRESSED_BORDER = BorderFactory.createLoweredBevelBorder();
        final Border b = BorderFactory.createLineBorder(Color.GRAY, 2);
        UNPRESSED_DEFAULT_BORDER =
            BorderFactory.createCompoundBorder(UNPRESSED_BORDER, b);
        PRESSED_DEFAULT_BORDER =
            BorderFactory.createCompoundBorder(PRESSED_BORDER, b);
    }

    private Color unpressedBackground; // LookAndFeel value
    private boolean isDefault;

    /** Constructor. */
    public JaxoPanelButton() {
        super();
        setFocusPainted(false);
        setRolloverEnabled(true);
        setMinimumSize(BUTTON_MINIMUM_SIZE);
        setPreferredSize(BUTTON_PREFERRED_SIZE);

    }

    /** The default height of a panel button.
     * @return the default height of a panel button.
     */
    public static final int getButtonHeight() {
        return BUTTON_PREFERRED_SIZE.height;
    }

    /** {@inheritDoc} */
    @Override
    public void updateUI() {
        setBackground(null);
        super.updateUI();
        unpressedBackground = getBackground();

        updateState();
    }

    /**
     * Determines if this button corresponds to the default mode.
     * Here, this may only affect the look of the button.
     * @return True if this button corresponds to the default mode.
     */
    public final boolean isDefault() {
        return isDefault;
    }

    /**
     * Sets this button as corresponding to the default mode.
     * @param value True for default.
     */
    public void setDefault(final boolean value) {
        if (isDefault != value) {
            isDefault = value;
            updateState();
        }
    }

    /**
     * Sets a button selected.
     *
     * @param value true to select.
     */
    @Override
    public void setSelected(final boolean value) {
        if (value != isSelected()) {
            super.setSelected(value);
            updateState();
        }
    }

    private void updateState() {
        setBackground(isSelected() ? PRESSED_BACKGROUND : unpressedBackground);
        setBorder(isDefault()
            ? (
                isSelected() ? PRESSED_DEFAULT_BORDER : UNPRESSED_DEFAULT_BORDER
            ) : (isSelected() ? PRESSED_BORDER : UNPRESSED_BORDER));
    }

    /** Sets the name and the action command of this button.
     * @param name The string to set as name and action command.
     */
    public void setId(final String name) {
        setName(name);
        setActionCommand(name);
    }
}
