/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionListener;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JPanel;

import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;
import net.sf.jaxodraw.util.JaxoLocalized;
import net.sf.jaxodraw.util.JaxoUtils;


/** Responsible for displaying and modifying the buttons of the action
 * button menu.
 * @since 2.0
 */
public class JaxoActionButtons extends JPanel implements PropertyChangeListener,
    JaxoLocalized {
    private static final long serialVersionUID = 7526471155622776147L;
    private static final String[] ACTION_BUTTON_ICONS =
    {"undo.png", "redo.png", "clear.png", "refresh.png"};
    private static final int[] ACTION_MODES =
    {
        JaxoConstants.UNDO, JaxoConstants.REDO, JaxoConstants.CLEAR,
        JaxoConstants.REFRESH
    };
    private static final int NOF_BUTTONS = ACTION_MODES.length;
    private static final String[] TOOLTIP_KEYS =
    {
        "Undo_the_last_operation", "Redo_the_last_operation",
        "Clear_the_screen", "Refresh_the_Screen"
    };
    private final JaxoPanelButton[] actionButtons =
        new JaxoPanelButton[NOF_BUTTONS];

    /** Constructor: lays out the action button panel. */
    public JaxoActionButtons() {
        super(new GridBagLayout());

        for (int i = 0; i < NOF_BUTTONS; i++) {
            actionButtons[i] = new JaxoPanelButton();
            actionButtons[i].setIcon(JaxoUtils.newImageIcon(
                    ACTION_BUTTON_ICONS[i]));
            actionButtons[i].setToolTipText(JaxoLanguage.translate(
                    TOOLTIP_KEYS[i]));
            actionButtons[i].setId(JaxoConstants.getModeAsString(
                    ACTION_MODES[i]));
        }

        final GridBagConstraints c =
            (GridBagConstraints) JaxoButtonPanel.BUTTON_CONSTRAINTS.clone();

        for (int y = 0; y < 1; y++) {
            for (int x = 0; x < 4; x++) {
                c.gridx = x;
                c.gridy = y;
                add(actionButtons[x + (4 * y)], c);
            }
        }

        // disable undo/redo buttons
        actionButtons[0].setEnabled(false);
        actionButtons[1].setEnabled(false);
    }

    /**
     * Applies a property change event.
     *
     * @param e the change event.
     */
    public void propertyChange(final PropertyChangeEvent e) {
        final String name = e.getPropertyName();

        if ("Jaxo.canUndo".equals(name)) {
            actionButtons[0].setEnabled(Boolean.TRUE.equals(e.getNewValue()));
        } else if ("Jaxo.canRedo".equals(name)) {
            actionButtons[1].setEnabled(Boolean.TRUE.equals(e.getNewValue()));
        }
    }

    /** Adds the given ActionListener to all buttons in this panel.
     * @param l The ActionListener to add.
     */
    public final void addActionListener(final ActionListener l) {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            actionButtons[i].addActionListener(l);
        }
    }

    /** {@inheritDoc} */
    public final void updateLanguage() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            actionButtons[i].setToolTipText(JaxoLanguage.translate(
                    TOOLTIP_KEYS[i]));
        }
    }
}
