/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.handle;

import java.awt.BasicStroke;
import java.awt.Color;

import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.graphics.JaxoGraphics2D;


/** A circle implementation for handles.
 * @since 2.0
 */
public class JaxoRoundHandle extends AbstractJaxoHandle {
    /** Constructor. */
    public JaxoRoundHandle() {
        super();
        setColor(JaxoColor.GREEN);
        setSecondaryColor(JaxoColor.CARNATION_PINK);
        setFillColor(JaxoColor.YELLOW);
        setSize(20);
        setStroke(new BasicStroke(2.0f));
    }

    /** {@inheritDoc}
     * This implementation paints circles.
     */
    public void paint(final JaxoGraphics2D g2, final int x, final int y, final boolean marked,
        final Color handleColor) {
        final int handleSize = getSize();
        final int h2 = handleSize / 2;
        if (marked) {
            g2.setColor(getFillColor());
            g2.fillOval(x - h2, y - h2, handleSize, handleSize);
        }

        g2.setColor(handleColor);
        g2.setStroke(getStroke());
        g2.drawOval(x - h2, y - h2, handleSize, handleSize);
    }

    /** {@inheritDoc}
     * This implementation checks if the point is contained within
     * a circle of length getSize() around the handle point.
     */
    public boolean contains(final int handleX, final int handleY, final int x, final int y) {
        final int l2 = getSize() / 2;
        final double distX = (double) (handleX - x);
        final double distY = (double) (handleY - y);
        final double dist = Math.sqrt((distX * distX) + (distY * distY));
        return (dist < l2);
    }
}
