/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */

package net.sf.jaxodraw.gui;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;

import java.beans.PropertyChangeListener;

import javax.swing.Scrollable;

import net.sf.jaxodraw.graph.JaxoGraph;
import net.sf.jaxodraw.gui.grid.JaxoPaintableGrid;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoLocalized;


/**
 * The drawing area.
 *
 * @since 2.1
 */
 // TODO remove: JaxoLocalized!
public interface JaxoDrawingArea extends JaxoCanvasComponent, PropertyChangeListener, Scrollable,
        JaxoLocalized {

    /** Default visualAid color. */
    Color DEFAULT_HELP_COLOR = JaxoColor.RED;

    /** Default visualAid Stroke. */
    Stroke DEFAULT_HELP_STROKE = new BasicStroke(1);

    /**
     * Clears the drawing area.
     */
    void clear();

    /**
     * Continue an edit actione that has been initiated before.
     *
     * @param p the point on this drawing area associated with the edit action.
     * @param last the last recorded action point, or null if none.
     */
    void continueEdit(Point p, Point last);

    /**
     * Carry on a select action that was started previously.
     *
     * @param p the current point of the select action.
     */
    void continueSelect(Point p);

    /**
     * Puts the current clipboard to the system clipboard.
     */
    void copyMarkedObjects();

    /**
     * Copies the current clipboard to the system clipboard
     * and removes the corresponding objects from the drawing area.
     */
    void cutMarkedObjects();

    /**
     * Determine the closest object to the given point and bring up an edit panel for it.
     *
     * @param p the test point
     */
    void editNearestObject(Point p);

    /**
     * Finalize a user input action.
     */
    void finalizeEdit();

    /**
     * End a select action.
     *
     * @param p the point where the select action was finished.
     */
    void finalizeSelect(Point p);

    /**
     * Gets the current clipboard. This never returns null.
     *
     * @return A JaxoGraph holding all the objects that are currently
     * on the clipboard, or an empty graph, if the clipboard is empty.
     */
    JaxoGraph getClipboard();

    /**
     * Return a zoom on this drawing area.
     *
     * @return a zoom on this drawing area.
     */
    JaxoZoom getZoom();

    /**
     * Groups the currently marked objects.
     */
    void groupMarkedObjects();

    /**
     * Start an edit action at a given point.
     *
     * @param p the point associated with the edit action.
     */
    void initiateEdit(Point p);

    /**
     * Start a select action.
     *
     * @param p the point where the select action was started.
     */
    void initiateSelect(Point p);

    /**
     * Mark the specified region as invalid. Currently this is also synchronous.
     *
     * @param boundingBox The region to update.
     * May be null in which case the whole image is marked for update.
     */
    void markImageInvalid(Rectangle boundingBox);

    /**
     * Moves the currently selected objects into either fore- or background.
     *
     * @param backGround True for moving into background, false for foreground.
     */
    void moveSelection(boolean backGround);

    /**
     * Displace the whole graph by a given offset.
     *
     * @param dx the offset in x direction.
     * @param dy the offset in y direction.
     */
    void moveGraph(int dx, int dy);

    /**
     * Paste the graph currently on the clipboard into this drawing area.
     */
    void pasteFromClipboard();

    /**
     * Refresh the canvas.
     * This should basically clean up any drawing operation and repaint the drawing area.
     */
    void refresh();

    /**
     * Sets the graph to be painted.
     *
     * @param value The graph to be set.
     */
    void setCanvasGraph(JaxoGraph value);

    /**
     * Sets the grid to be painted on this drawing area.
     *
     * @param value The new grid. May be null.
     */
    void setGrid(JaxoPaintableGrid value);

    /**
     * Ungroup all marked objects. This is not recursive, only one level is resolved.
     */
    void ungroupMarkedObjects();

    /**
     * Sets all objects in the current graph as not marked.
     *
     * @return true if the graph had actually contained marked objects.
     * In this case the canvas was also repainted.
     * False otherwise, in which case this method has no effect.
     */
    boolean unMarkGraph();

    /**
     * Updates the visual aid that is painted for objects in click-move mode.
     *
     * @param p the current location of the cursor.
     */
    void updatePointsAid(Point p);
}
