/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */

package net.sf.jaxodraw.gui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;

import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import java.beans.PropertyChangeListener;


/**
 * The canvas as a component.
 *
 * @since 2.1
 */
public interface JaxoCanvasComponent {
    /**
     * Adds the specified mouse listener to receive mouse events from this CanvasComponent.
     *
     * @param l the mouse listener. May be null.
     * @see java.awt.Component#addMouseListener(java.awt.event.MouseListener)
     */
    void addMouseListener(MouseListener l);

    /**
     * Adds the specified mouse motion listener to receive mouse motion events from this CanvasComponent.
     *
     * @param l the mouse motion listener. May be null.
     * @see java.awt.Component#addMouseMotionListener(java.awt.event.MouseMotionListener)
     */
    void addMouseMotionListener(MouseMotionListener l);

    /**
     * Adds a PropertyChangeListener to the listener list.
     *
     * @param listener the property change listener to be added. May be null.
     * @see java.awt.Component#addPropertyChangeListener(java.beans.PropertyChangeListener)
     */
    void addPropertyChangeListener(PropertyChangeListener listener);

    /**
     * Return this CanvasComponent as a Component.
     *
     * @return this CanvasComponent as a Component or null, if this is not a Component.
     */
    Component asComponent();

    /**
     * Background color in the region that is actually covered by the canvas
     * {@link #getCanvasSize}.
     *
     * @return The canvas background color.
     */
    Color getCanvasBackground();

    /**
     * Part of the component that is covered by the actual canvas. This includes canvas
     * background and grid, -- i.e. a rectangle with size 'canvasSize' and origin 'canvasOrigin'.
     *
     * @return Rectangle the canvas bounds.
     * @see #getCanvasSize
     * @see #getCanvasOrigin
     */
    Rectangle getCanvasBounds();

    /**
     * Origin of the actual canvas within the component. This is always the top-left corner.
     *
     * @return a Point at the origin.
     */
    Point getCanvasOrigin();

    /**
     * Canvas size. This is equal to the (inner) component size unless
     * because of viewport size tracking, the component has become
     * larger than the effective maximum canvas size.
     *
     * @return The canvas size.
     */
    Dimension getCanvasSize();

    /**
     * Return the graphics context of this CanvasComponent.
     *
     * @return the graphics context or null if this component is currently not displayable.
     * @see java.awt.Component#getGraphics()
     */
    Graphics getGraphics();

    /**
     * Maximum size of the canvas (even if there are objects with a larger bounding box).
     * The default value is larger than (0,0). If the maximum canvas size is smaller
     * (in either direction) than the minimum canvas size, the latter wins.
     *
     * @return The maximum canvas size.
     */
    Dimension getMaximumCanvasSize();

    /**
     * Minimum size of the canvas (even if there are no objects
     * or they have a smaller bounding box). The default value is (0,0).
     *
     * @return The minimum canvas size.
     */
    Dimension getMinimumCanvasSize();

    /**
     * Prints this component.
     *
     * @param g the graphics context to use for printing.
     * @see java.awt.Component#print(java.awt.Graphics)
     */
    void print(Graphics g);

    /**
     * Removes the specified mouse listener so that it no longer receives mouse events
     * from this component. This method performs no function, nor does it throw an exception,
     * if the listener specified by the argument was not previously added to this component.
     * If listener l is null, no exception is thrown and no action is performed.
     *
     * @param l the mouse listener
     * @see java.awt.Component#removeMouseListener(java.awt.event.MouseListener)
     */
    void removeMouseListener(MouseListener l);

    /**
     * Removes the specified mouse motion listener so that it no longer receives mouse motion
     * events from this component. This method performs no function, nor does it throw an
     * exception, if the listener specified by the argument was not previously added to this
     * component. If listener l is null, no exception is thrown and no action is performed.
     *
     * @param l the mouse motion listener
     * @see java.awt.Component#removeMouseMotionListener(java.awt.event.MouseMotionListener)
     */
    void removeMouseMotionListener(MouseMotionListener l);

    /**
     * Repaints this component.
     *
     * @see java.awt.Component#repaint()
     */
    void repaint();

    /**
     * Sets the color of the background.
     *
     * @param color The background color.
     */
    void setCanvasBackground(Color color);

    /**
     * Sets the maximum canvas size.
     *
     * @param value The maximum canvas size.
     */
    void setMaximumCanvasSize(Dimension value);

    /**
     * Sets the minimum canvas size.
     *
     * @param value The minimum canvas size.
     */
    void setMinimumCanvasSize(Dimension value);
}
