/*
 * Decompiled with CFR 0.152.
 */
package android.support.multidex;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.support.multidex.ZipUtil;
import android.util.Log;
import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

final class MultiDexExtractor
implements Closeable {
    private static final String TAG = "MultiDex";
    private static final String DEX_PREFIX = "classes";
    static final String DEX_SUFFIX = ".dex";
    private static final String EXTRACTED_NAME_EXT = ".classes";
    static final String EXTRACTED_SUFFIX = ".zip";
    private static final int MAX_EXTRACT_ATTEMPTS = 3;
    private static final String PREFS_FILE = "multidex.version";
    private static final String KEY_TIME_STAMP = "timestamp";
    private static final String KEY_CRC = "crc";
    private static final String KEY_DEX_NUMBER = "dex.number";
    private static final String KEY_DEX_CRC = "dex.crc.";
    private static final String KEY_DEX_TIME = "dex.time.";
    private static final int BUFFER_SIZE = 16384;
    private static final long NO_VALUE = -1L;
    private static final String LOCK_FILENAME = "MultiDex.lock";
    private final File sourceApk;
    private final long sourceCrc;
    private final File dexDir;
    private final RandomAccessFile lockRaf;
    private final FileChannel lockChannel;
    private final FileLock cacheLock;

    MultiDexExtractor(File sourceApk, File dexDir) throws IOException {
        Log.i((String)TAG, (String)("MultiDexExtractor(" + sourceApk.getPath() + ", " + dexDir.getPath() + ")"));
        this.sourceApk = sourceApk;
        this.dexDir = dexDir;
        this.sourceCrc = MultiDexExtractor.getZipCrc(sourceApk);
        File lockFile = new File(dexDir, LOCK_FILENAME);
        this.lockRaf = new RandomAccessFile(lockFile, "rw");
        try {
            this.lockChannel = this.lockRaf.getChannel();
            try {
                Log.i((String)TAG, (String)("Blocking on lock " + lockFile.getPath()));
                this.cacheLock = this.lockChannel.lock();
            }
            catch (IOException | Error | RuntimeException e) {
                MultiDexExtractor.closeQuietly(this.lockChannel);
                throw e;
            }
            Log.i((String)TAG, (String)(lockFile.getPath() + " locked"));
        }
        catch (IOException | Error | RuntimeException e) {
            MultiDexExtractor.closeQuietly(this.lockRaf);
            throw e;
        }
    }

    List<? extends File> load(Context context, String prefsKeyPrefix, boolean forceReload) throws IOException {
        List<ExtractedDex> files;
        Log.i((String)TAG, (String)("MultiDexExtractor.load(" + this.sourceApk.getPath() + ", " + forceReload + ", " + prefsKeyPrefix + ")"));
        if (!this.cacheLock.isValid()) {
            throw new IllegalStateException("MultiDexExtractor was closed");
        }
        if (!forceReload && !MultiDexExtractor.isModified(context, this.sourceApk, this.sourceCrc, prefsKeyPrefix)) {
            try {
                files = this.loadExistingExtractions(context, prefsKeyPrefix);
            }
            catch (IOException ioe) {
                Log.w((String)TAG, (String)"Failed to reload existing extracted secondary dex files, falling back to fresh extraction", (Throwable)ioe);
                files = this.performExtractions();
                MultiDexExtractor.putStoredApkInfo(context, prefsKeyPrefix, MultiDexExtractor.getTimeStamp(this.sourceApk), this.sourceCrc, files);
            }
        } else {
            if (forceReload) {
                Log.i((String)TAG, (String)"Forced extraction must be performed.");
            } else {
                Log.i((String)TAG, (String)"Detected that extraction must be performed.");
            }
            files = this.performExtractions();
            MultiDexExtractor.putStoredApkInfo(context, prefsKeyPrefix, MultiDexExtractor.getTimeStamp(this.sourceApk), this.sourceCrc, files);
        }
        Log.i((String)TAG, (String)("load found " + files.size() + " secondary dex files"));
        return files;
    }

    @Override
    public void close() throws IOException {
        this.cacheLock.release();
        this.lockChannel.close();
        this.lockRaf.close();
    }

    private List<ExtractedDex> loadExistingExtractions(Context context, String prefsKeyPrefix) throws IOException {
        Log.i((String)TAG, (String)"loading existing secondary dex files");
        String extractedFilePrefix = this.sourceApk.getName() + EXTRACTED_NAME_EXT;
        SharedPreferences multiDexPreferences = MultiDexExtractor.getMultiDexPreferences(context);
        int totalDexNumber = multiDexPreferences.getInt(prefsKeyPrefix + KEY_DEX_NUMBER, 1);
        ArrayList<ExtractedDex> files = new ArrayList<ExtractedDex>(totalDexNumber - 1);
        for (int secondaryNumber = 2; secondaryNumber <= totalDexNumber; ++secondaryNumber) {
            String fileName = extractedFilePrefix + secondaryNumber + EXTRACTED_SUFFIX;
            ExtractedDex extractedFile = new ExtractedDex(this.dexDir, fileName);
            if (extractedFile.isFile()) {
                long lastModified;
                extractedFile.crc = MultiDexExtractor.getZipCrc(extractedFile);
                long expectedCrc = multiDexPreferences.getLong(prefsKeyPrefix + KEY_DEX_CRC + secondaryNumber, -1L);
                long expectedModTime = multiDexPreferences.getLong(prefsKeyPrefix + KEY_DEX_TIME + secondaryNumber, -1L);
                if (expectedModTime != (lastModified = extractedFile.lastModified()) || expectedCrc != extractedFile.crc) {
                    throw new IOException("Invalid extracted dex: " + extractedFile + " (key \"" + prefsKeyPrefix + "\"), expected modification time: " + expectedModTime + ", modification time: " + lastModified + ", expected crc: " + expectedCrc + ", file crc: " + extractedFile.crc);
                }
            } else {
                throw new IOException("Missing extracted secondary dex file '" + extractedFile.getPath() + "'");
            }
            files.add(extractedFile);
        }
        return files;
    }

    private static boolean isModified(Context context, File archive, long currentCrc, String prefsKeyPrefix) {
        SharedPreferences prefs = MultiDexExtractor.getMultiDexPreferences(context);
        return prefs.getLong(prefsKeyPrefix + KEY_TIME_STAMP, -1L) != MultiDexExtractor.getTimeStamp(archive) || prefs.getLong(prefsKeyPrefix + KEY_CRC, -1L) != currentCrc;
    }

    private static long getTimeStamp(File archive) {
        long timeStamp = archive.lastModified();
        if (timeStamp == -1L) {
            --timeStamp;
        }
        return timeStamp;
    }

    private static long getZipCrc(File archive) throws IOException {
        long computedValue = ZipUtil.getZipCrc(archive);
        if (computedValue == -1L) {
            --computedValue;
        }
        return computedValue;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<ExtractedDex> performExtractions() throws IOException {
        String extractedFilePrefix = this.sourceApk.getName() + EXTRACTED_NAME_EXT;
        this.clearDexDir();
        ArrayList<ExtractedDex> files = new ArrayList<ExtractedDex>();
        ZipFile apk = new ZipFile(this.sourceApk);
        try {
            int secondaryNumber = 2;
            ZipEntry dexFile = apk.getEntry(DEX_PREFIX + secondaryNumber + DEX_SUFFIX);
            while (dexFile != null) {
                String fileName = extractedFilePrefix + secondaryNumber + EXTRACTED_SUFFIX;
                ExtractedDex extractedFile = new ExtractedDex(this.dexDir, fileName);
                files.add(extractedFile);
                Log.i((String)TAG, (String)("Extraction is needed for file " + extractedFile));
                int numAttempts = 0;
                boolean isExtractionSuccessful = false;
                while (numAttempts < 3 && !isExtractionSuccessful) {
                    ++numAttempts;
                    MultiDexExtractor.extract(apk, dexFile, extractedFile, extractedFilePrefix);
                    try {
                        extractedFile.crc = MultiDexExtractor.getZipCrc(extractedFile);
                        isExtractionSuccessful = true;
                    }
                    catch (IOException e) {
                        isExtractionSuccessful = false;
                        Log.w((String)TAG, (String)("Failed to read crc from " + extractedFile.getAbsolutePath()), (Throwable)e);
                    }
                    Log.i((String)TAG, (String)("Extraction " + (isExtractionSuccessful ? "succeeded" : "failed") + " '" + extractedFile.getAbsolutePath() + "': length " + extractedFile.length() + " - crc: " + extractedFile.crc));
                    if (isExtractionSuccessful) continue;
                    extractedFile.delete();
                    if (!extractedFile.exists()) continue;
                    Log.w((String)TAG, (String)("Failed to delete corrupted secondary dex '" + extractedFile.getPath() + "'"));
                }
                if (!isExtractionSuccessful) {
                    throw new IOException("Could not create zip file " + extractedFile.getAbsolutePath() + " for secondary dex (" + secondaryNumber + ")");
                }
                dexFile = apk.getEntry(DEX_PREFIX + ++secondaryNumber + DEX_SUFFIX);
            }
        }
        finally {
            try {
                apk.close();
            }
            catch (IOException e) {
                Log.w((String)TAG, (String)"Failed to close resource", (Throwable)e);
            }
        }
        return files;
    }

    private static void putStoredApkInfo(Context context, String keyPrefix, long timeStamp, long crc, List<ExtractedDex> extractedDexes) {
        SharedPreferences prefs = MultiDexExtractor.getMultiDexPreferences(context);
        SharedPreferences.Editor edit = prefs.edit();
        edit.putLong(keyPrefix + KEY_TIME_STAMP, timeStamp);
        edit.putLong(keyPrefix + KEY_CRC, crc);
        edit.putInt(keyPrefix + KEY_DEX_NUMBER, extractedDexes.size() + 1);
        int extractedDexId = 2;
        for (ExtractedDex dex : extractedDexes) {
            edit.putLong(keyPrefix + KEY_DEX_CRC + extractedDexId, dex.crc);
            edit.putLong(keyPrefix + KEY_DEX_TIME + extractedDexId, dex.lastModified());
            ++extractedDexId;
        }
        edit.commit();
    }

    private static SharedPreferences getMultiDexPreferences(Context context) {
        return context.getSharedPreferences(PREFS_FILE, Build.VERSION.SDK_INT < 11 ? 0 : 4);
    }

    private void clearDexDir() {
        File[] files = this.dexDir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                return !pathname.getName().equals(MultiDexExtractor.LOCK_FILENAME);
            }
        });
        if (files == null) {
            Log.w((String)TAG, (String)("Failed to list secondary dex dir content (" + this.dexDir.getPath() + ")."));
            return;
        }
        for (File oldFile : files) {
            Log.i((String)TAG, (String)("Trying to delete old file " + oldFile.getPath() + " of size " + oldFile.length()));
            if (!oldFile.delete()) {
                Log.w((String)TAG, (String)("Failed to delete old file " + oldFile.getPath()));
                continue;
            }
            Log.i((String)TAG, (String)("Deleted old file " + oldFile.getPath()));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void extract(ZipFile apk, ZipEntry dexFile, File extractTo, String extractedFilePrefix) throws IOException, FileNotFoundException {
        InputStream in = apk.getInputStream(dexFile);
        ZipOutputStream out = null;
        File tmp = File.createTempFile("tmp-" + extractedFilePrefix, EXTRACTED_SUFFIX, extractTo.getParentFile());
        Log.i((String)TAG, (String)("Extracting " + tmp.getPath()));
        try {
            out = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(tmp)));
            try {
                ZipEntry classesDex = new ZipEntry("classes.dex");
                classesDex.setTime(dexFile.getTime());
                out.putNextEntry(classesDex);
                byte[] buffer = new byte[16384];
                int length = in.read(buffer);
                while (length != -1) {
                    out.write(buffer, 0, length);
                    length = in.read(buffer);
                }
                out.closeEntry();
            }
            finally {
                out.close();
            }
            if (!tmp.setReadOnly()) {
                throw new IOException("Failed to mark readonly \"" + tmp.getAbsolutePath() + "\" (tmp of \"" + extractTo.getAbsolutePath() + "\")");
            }
            Log.i((String)TAG, (String)("Renaming to " + extractTo.getPath()));
            if (!tmp.renameTo(extractTo)) {
                throw new IOException("Failed to rename \"" + tmp.getAbsolutePath() + "\" to \"" + extractTo.getAbsolutePath() + "\"");
            }
        }
        finally {
            MultiDexExtractor.closeQuietly(in);
            tmp.delete();
        }
    }

    private static void closeQuietly(Closeable closeable) {
        try {
            closeable.close();
        }
        catch (IOException e) {
            Log.w((String)TAG, (String)"Failed to close resource", (Throwable)e);
        }
    }

    private static class ExtractedDex
    extends File {
        public long crc = -1L;

        public ExtractedDex(File dexDir, String fileName) {
            super(dexDir, fileName);
        }
    }
}

