"use strict";

var debug = require('debug')('updater');
var util = require('util');
var EventEmitter = require('events').EventEmitter;
var os = require('os');
var semver = require('semver');
var tmp = require('tmp');
var md5File = require('md5-file');
var fs = require('fs-extra');
var DecompressZip = require('decompress-zip');
var path = require('path');
var exec = require("child_process").exec;
var fetch = require('node-fetch');
var Download = require('@ali/ding-download').Download;

var PATH_MAPPING = {
    "web_content": "web_content"
}

var osName = 'win';

if(os.platform() === 'darwin'){
    osName = 'mac';
}

function Updater(config){
    EventEmitter.call(this);
    if(!this._checkConfig(config)){
        return;
    }
    this.conf = config;
    this.tryCompare();
    var self = this;
    this.ignoreVersions = {
        "web_content":[],
        "nw":[]
    };
    setInterval(function(){
        self.tryCompare();
    }, 24 * 60 * 60 * 1000); // 24 个小时轮询一次
}

util.inherits(Updater, EventEmitter);

Updater.prototype._checkConfig = function(config){
    return config.remoteManifest && config['web_content'] && config['web_content'].version && config["nw_version"];
}

Updater.prototype.EVENTS_NAME = {
    "WEB_CONTENT_NEED_UPDATE": "web_content_need_update",
    "WEB_CONTENT_UPDATE_SUCCESS":"web_content_update_success",
    "WEB_CONTENT_UPDATE_FAIL": "web_content_update_fail",
    "NW_IS_NOT_SUPPORTED": "nw_is_not_supported",
    "NW_IS_NOT_SATISFIED": "nw_is_not_satisfied",
    "PACKAGE_NEED_UPDATE": "package_need_update",
    "PACKAGE_DOWNLOADED" : "package_downloaded",
    "PACKAGE_UPDATE_SUCCESS" : "package_update_success",
    "PACKAGE_UPDATE_FAILED" : "package_update_failed",
    "ERROR_INFO" : "error_info",
    "NO_UPDATE": "no_update"
}

Updater.prototype.getUpdateInfo = function(){
    debug('start get update info');
    var remoteManifest = this.conf.remoteManifest;
    return fetch(remoteManifest).then(function(res){
        return res.json();
    }).then(function(remoteConfig){
        return remoteConfig[osName];
    })
}

Updater.prototype.detectUpdate = function(isForce){
    var self = this;
    debug('try detect udpate');
    return this.getUpdateInfo().then(function(platformConfig){
        try{
            var remoteWebContentVer = platformConfig['web_content'].version;
            var minNwVersion = platformConfig["min_version"];
            var packageInfo = platformConfig['package'];
            if(semver.gt(minNwVersion, self.conf['nw_version'])){
                debug('nw is not supported', JSON.stringify(platformConfig));
                return {
                    eventName: self.EVENTS_NAME["NW_IS_NOT_SUPPORTED"],
                    value: platformConfig
                };
            }

            if(osName == "win"
                && semver.gt(packageInfo['version'],self.conf['nw_version'])
                && (self.ignoreVersions.nw.indexOf(packageInfo['version']) === -1 || isForce)){
                if(packageInfo['need-reinstall'] && semver.satisfies(self.conf['nw_version'].split('-')[0],packageInfo['need-reinstall'])){
                    debug('ns is not satisfied');
                    return {
                        eventName: self.EVENTS_NAME["NW_IS_NOT_SATISFIED"],
                        value: platformConfig
                    }
                }

                if(packageInfo['need-update-v2'] && semver.satisfies(self.conf['nw_version'].split('-')[0],packageInfo['need-update-v2'])){
                    debug('pacakge need update');
                    return {
                        eventName: self.EVENTS_NAME["PACKAGE_NEED_UPDATE"],
                        value: platformConfig
                    }
                }
            }

            if(semver.gt(remoteWebContentVer, self.conf['web_content'].version)
                && (self.ignoreVersions.web_content.indexOf(remoteWebContentVer) === -1 || isForce)){
                var requireNw = platformConfig['web_content']['require_nw'].slice(2);
                if(semver.gte(self.conf['nw_version'], requireNw)){
                    debug('web content need update');
                    return {
                        eventName: self.EVENTS_NAME["WEB_CONTENT_NEED_UPDATE"],
                        value: platformConfig
                    }
                }else if(self.ignoreVersions.nw.indexOf(platformConfig['install']['version']) === -1 || isForce){
                    debug('nw is not satisfied');
                    return {
                        eventName: self.EVENTS_NAME["NW_IS_NOT_SATISFIED"],
                        value: platformConfig
                    }
                }
            }
            return {
                eventName: self.EVENTS_NAME['NO_UPDATE']
            }
        }catch(e){
            debug('get update info error', e)
            return {
                eventName: self.EVENTS_NAME['ERROR_INFO'],
                value: e
            }
        }
    }).catch(function(e){
        debug('get update info error', e)
        return {
            eventName: self.EVENTS_NAME['ERROR_INFO'],
            value: e
        }
    })
}

Updater.prototype.tryUpdate = function(detectResult, extension){
    var platformConfig = detectResult.value;
    var eventName = detectResult.eventName;
    if(eventName === this.EVENTS_NAME['WEB_CONTENT_NEED_UPDATE']){
        return this.updateWebContent(platformConfig['web_content'],extension);
    }else if(eventName === this.EVENTS_NAME['PACKAGE_NEED_UPDATE']){
        return this.fetchNewPackage(platformConfig,extension);
    }else{
        this.emit(this.EVENTS_NAME['ERROR_INFO'],'cannot update for detectResul' + eventName);
        return Promise.reject('cannot update for detectResul' + eventName);
    }
}

Updater.prototype.tryCompare = function(){
    debug('try compare');
    return this.detectUpdate().then(function(detectResult){
        var eventName = detectResult.eventName;
        this.emit(eventName, detectResult.value);
        if(eventName === this.EVENTS_NAME['WEB_CONTENT_NEED_UPDATE'] || eventName === this.EVENTS_NAME['PACKAGE_NEED_UPDATE']){
            return this.tryUpdate(detectResult);
        }else{
            return Promise.resolve('no update')
        }
    }.bind(this))
    // return this.getUpdateInfo().then(function(platformConfig){
    //     try{
    //         var remoteWebContentVer = platformConfig['web_content'].version;
    //         var minNwVersion = platformConfig["min_version"];
    //         var packageInfo = platformConfig['package'];
    //         if(semver.gt(minNwVersion, self.conf['nw_version'])){
    //             debug('nw is not supported', JSON.stringify(platformConfig));
    //             self.emit(self.EVENTS_NAME["NW_IS_NOT_SUPPORTED"], platformConfig);
    //             return {
    //                 eventName: self.EVENTS_NAME["NW_IS_NOT_SUPPORTED"],
    //                 value: platformConfig
    //             };
    //         }
    //
    //         if(osName == "win" 
    //             && semver.gt(packageInfo['version'],self.conf['nw_version']) 
    //             && (self.ignoreVersions.nw.indexOf(packageInfo['version']) === -1 || force)){
    //             if(packageInfo['need-reinstall'] && semver.satisfies(self.conf['nw_version'].split('-')[0],packageInfo['need-reinstall'])){
    //                 self.emit(self.EVENTS_NAME["NW_IS_NOT_SATISFIED"], platformConfig);
    //                 return {
    //                     eventName: self.EVENTS_NAME["NW_IS_NOT_SATISFIED"],
    //                     value: platformConfig
    //                 }
    //             }
    //
    //             if(packageInfo['need-update-v2'] && semver.satisfies(self.conf['nw_version'].split('-')[0],packageInfo['need-update-v2'])){
    //                 return self.fetchNewPackage(platformConfig,extension);
    //             }
    //         }
    //
    //         if(semver.gt(remoteWebContentVer, self.conf['web_content'].version) 
    //             && (self.ignoreVersions.web_content.indexOf(remoteWebContentVer) === -1 || force)){
    //             if(semver.satisfies(self.conf['nw_version'].split('-')[0], platformConfig['web_content']['require_nw'])){
    //                 debug('try update web content');
    //                 return self.updateWebContent(platformConfig['web_content']);
    //             }else if(self.ignoreVersions.nw.indexOf(platformConfig['install']['version']) === -1 || force){
    //                 self.emit(self.EVENTS_NAME["NW_IS_NOT_SATISFIED"], platformConfig);
    //                 return {
    //                     eventName: self.EVENTS_NAME["NW_IS_NOT_SATISFIED"],
    //                     value: platformConfig
    //                 }
    //             }
    //         }
    //         return {
    //             eventName: self.EVENTS_NAME['NO_UPDATE']
    //         }
    //     }catch(e){
    //         console.log(e);
    //         self.emit(self.EVENTS_NAME['ERROR_INFO'],e);
    //         return;
    //     }
    // }).catch(function(e){
    //     debug('get update info error', e)
    //     self.emit(self.EVENTS_NAME['ERROR_INFO'],e);
    //     return {
    //         eventName: self.EVENTS_NAME['ERROR_INFO'],
    //         value: e
    //     }
    // })
}

Updater.prototype.fetchNewPackage = function(conf,extension){
    var self = this;
    var packageInfo = conf['package'];
    try{
        var tmpDir = tmp.dirSync().name;
    }catch(e){
        console.log(e);
        self.emit(self.EVENTS_NAME['ERROR_INFO'],e);
        return;
    }
    return Promise.resolve().then(function(){
        return self._downloadPkg(packageInfo.url, tmpDir);
    }).then(function(zipFilePath){
        return self._checkMd5(zipFilePath, packageInfo.md5).then(function(){
            return self._unzip(zipFilePath);
        })
    }).then(function(assetsPath) {
        var eventName = self.EVENTS_NAME["PACKAGE_DOWNLOADED"];
        var value = {
            src : assetsPath,
            dest : process.cwd(),
            conf : conf,
            extension : extension || {},
            tmpDir : tmpDir
        }
        self.emit(eventName, value);
        return {
            eventName: eventName,
            value: value
        }
    }).catch(function(e){
        self.emit(self.EVENTS_NAME['ERROR_INFO'],e);
        return {
            eventName: self.EVENTS_NAME['ERROR_INFO'],
            value: e
        }
    })
}

Updater.prototype.addIgnoreVersion = function(type, version){
    this.ignoreVersions[type].push(version);
}

Updater.prototype.updateWebContent = function(conf,extension){
    var self = this;
    var type = 'web_content';

    try{
        var tmpDir = tmp.dirSync().name;
    }catch(e){
        console.log(e);
        self.emit(self.EVENTS_NAME['ERROR_INFO'],e);
        return Promise.resolve({
            eventName:self.EVENTS_NAME['ERROR_INFO'],
            value: e
        });
    }
    return Promise.resolve().then(function(){
        return self._downloadPkg(conf.url, tmpDir);
    }).then(function(zipFilePath){
        return self._checkMd5(zipFilePath, conf.md5).then(function(){
            return self._unzip(zipFilePath);
        })
    }).then(function(assetsPath) {
        if(osName == "win"){
            return self._moveDirForWin(assetsPath, path.join(process.cwd(), PATH_MAPPING[type]), 0);
        }else{
            return self._moveDir(assetsPath, path.join(process.cwd(), PATH_MAPPING[type]));
        }
    }).then(function(){
        if(extension){
            conf.isWebContentNotify = extension.isWebContentNotify;
        }
        self.emit(type + '_update_success', conf);
        self.removeUpdateTmpDir(tmpDir);
        self.ignoreVersions.web_content.push(conf.version);
        return {
            eventName:type + '_update_success',
            value: conf
        }
    }).catch(function(e){
        console.log(e);
        self.emit(self.EVENTS_NAME['ERROR_INFO'],e);
        self.emit(type + '_update_fail', conf);
        self.removeUpdateTmpDir(tmpDir);
        return {
            eventName: type + '_update_fail',
            value: conf
        }
    })
}

Updater.prototype.updatePackage = function(paths){
    var self = this;
    this._moveDirForWin(paths.src,paths.dest,1).then(function(){
        self.emit('package_update_success', paths);
        self.removeUpdateTmpDir(paths.tmpDir);
    }).catch(function(e){
        console.log(e);
        self.emit(self.EVENTS_NAME['ERROR_INFO'],e);
        self.removeUpdateTmpDir(paths.tmpDir);
        if(e.code === 2) return; //取消提权
        self.emit('package_update_failed', paths.conf);

    });
}

Updater.prototype.removeUpdateTmpDir = function(dir){
    if(!dir) return;
    fs.remove(dir,function(err){
        if(err){
            console.log(err);
            self.emit(self.EVENTS_NAME['ERROR_INFO'],err);
        }
    })
}

Updater.prototype._moveDir = function(from, to){
    return new Promise(function(resolve, reject){
        fs.copy(from, to, {
            clobber: true
        }, function(err){
            if(err){
                reject(err);
            }else{
                resolve(true);
            }
        })
    })
}

Updater.prototype._moveDirForWin = function(from,to,doKill){
    //var WIN_UPDATER_DIR = path.join(process.env.APPDATA,"DingDing","updater");
    var updaterPath = path.join(process.cwd(),"../../");
    var dingdingUpdater = path.join(updaterPath,"dingdingUpdater.exe");
    var desktopEnv = this.conf['desktop_env'];
    function tryCreateFile(dir){
        var tmpFolder = "tmp" + new Date().getTime();
        var tmp = path.join(dir,tmpFolder);
        return new Promise(function(resolve, reject){
            fs.mkdir(tmp,function(err){
                if(err){
                    reject(err);
                }else{
                    try{
                        fs.renameSync(tmp,tmp+'_');
                        fs.rmdirSync(tmp+'_');
                        resolve();
                    }
                    catch(e){
                        reject("rename error");
                    }
                }
            })
        })
    }
    return new Promise(function(resolve, reject){
        tryCreateFile(path.join(to,"../")).then(function(){
            var cmd = "dingdingUpdater.exe" + " \""+from+"\"" + " \""+to+ "\"" + " \""+desktopEnv+ "\"" + " "+doKill;
            exec(cmd,{cwd:updaterPath, maxBuffer: 1024*1024 },function(err){
                if(err){
                    reject(err);
                }else{
                    resolve(true);
                }
            })
        },function(){
            var cmd = "elevate.exe" + " \""+dingdingUpdater+"\""+" \""+from+"\"" + " \""+to+ "\"" + " \""+desktopEnv+ "\"" + " "+doKill;
            exec(cmd,{cwd:updaterPath,  maxBuffer: 1024*1024},function(err){
                if(err){
                    reject(err);
                }else{
                    resolve(true);
                }
            })
        })
    })
}

Updater.prototype._unzip = function(zipfile){
    return new Promise(function(resolve, reject){
        var files = [];
        var destPath = path.join(path.dirname(zipfile), 'web_content');
        fs.emptyDirSync(destPath);
        new DecompressZip(zipfile)
            .on('error', function(err){
                reject(err);
            })
            .on('extract', function(log) {
                // Setup chmodSync to fix permissions
                files.forEach(function(file) {
                    fs.chmodSync(path.join(destPath, file.path), file.mode);
                });

                resolve(destPath);
            })
            .extract({
                path: destPath,
                filter: function(entry) {
                    files.push({
                        path: entry.path,
                        mode: entry.mode.toString(8)
                    });

                    return true;
                }
            });

    })
}

Updater.prototype._checkMd5 = function(fpath, md5Value){
    return new Promise(function(resolve, reject){
        md5File(fpath, function(err, sum){
            if(sum === md5Value){
                resolve(true);
            }else{
                reject(new Error('md5 不匹配 :' + fpath));
            }
        })
    })

}

Updater.prototype._downloadPkg = function(url, dest){

    var filename = path.basename(url);
    var destinationPath = path.join(dest,filename);
    return new Promise(function(resolve, reject){
        try{
            var d = new Download(url, destinationPath);
        }catch(e){
            reject(e);
        }

        d.addListener(d.EventsName.finish, function () {
            resolve(destinationPath);
        })

        d.addListener(d.EventsName.error, function(e){
            reject(e);
        })
    })
}


exports.Updater = Updater;

/**
 * @param config
 *      {
 *        "assets":{
 *          "version": "2.8.1",
 *        },
 *        "remoteManifest":"xxxx"
 *      }
 * @returns {Updater}
 */
exports.init = function(config){
    return new Updater(config);
}

