/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.ack.ClusterStateUpdateRequest;
import org.elasticsearch.cluster.ack.ClusterStateUpdateResponse;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.RepositoriesMetadata;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.RepositoryMissingException;
import org.elasticsearch.repositories.VerificationFailure;
import org.elasticsearch.repositories.VerifyNodeRepositoryAction;
import org.elasticsearch.snapshots.RestoreService;
import org.elasticsearch.snapshots.SnapshotsService;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TransportService;

public class RepositoriesService
implements ClusterStateApplier {
    private static final Logger LOGGER = LogManager.getLogger(RepositoriesService.class);
    private final Map<String, Repository.Factory> typesRegistry;
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final VerifyNodeRepositoryAction verifyAction;
    private volatile Map<String, Repository> repositories = Collections.emptyMap();

    @Inject
    public RepositoriesService(Settings settings, ClusterService clusterService, TransportService transportService, Map<String, Repository.Factory> typesRegistry, ThreadPool threadPool) {
        this.typesRegistry = typesRegistry;
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        if (DiscoveryNode.isDataNode(settings) || DiscoveryNode.isMasterNode(settings)) {
            clusterService.addStateApplier(this);
        }
        this.verifyAction = new VerifyNodeRepositoryAction(transportService, clusterService, this);
    }

    public void registerRepository(final RegisterRepositoryRequest request, ActionListener<ClusterStateUpdateResponse> listener) {
        final RepositoryMetadata newRepositoryMetadata = new RepositoryMetadata(request.name, request.type, request.settings);
        ActionListener<ClusterStateUpdateResponse> registrationListener = request.verify ? ActionListener.delegateFailure(listener, (delegatedListener, clusterStateUpdateResponse) -> {
            if (clusterStateUpdateResponse.isAcknowledged()) {
                this.verifyRepository(request.name, ActionListener.delegateFailure(delegatedListener, (innerDelegatedListener, discoveryNodes) -> innerDelegatedListener.onResponse(clusterStateUpdateResponse)));
            } else {
                delegatedListener.onResponse(clusterStateUpdateResponse);
            }
        }) : listener;
        this.clusterService.submitStateUpdateTask(request.cause, new AckedClusterStateUpdateTask<ClusterStateUpdateResponse>((AckedRequest)request, registrationListener){

            @Override
            protected ClusterStateUpdateResponse newResponse(boolean acknowledged) {
                return new ClusterStateUpdateResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws IOException {
                RepositoriesService.this.ensureRepositoryNotInUse(currentState, request.name);
                if (!RepositoriesService.this.registerRepository(newRepositoryMetadata)) {
                    return currentState;
                }
                Metadata metadata = currentState.metadata();
                Metadata.Builder mdBuilder = Metadata.builder(currentState.metadata());
                RepositoriesMetadata repositories = (RepositoriesMetadata)metadata.custom("repositories");
                if (repositories == null) {
                    LOGGER.info("put repository [{}]", (Object)request.name);
                    repositories = new RepositoriesMetadata(Collections.singletonList(new RepositoryMetadata(request.name, request.type, request.settings)));
                } else {
                    boolean found = false;
                    ArrayList<RepositoryMetadata> repositoriesMetadata = new ArrayList<RepositoryMetadata>(repositories.repositories().size() + 1);
                    for (RepositoryMetadata repositoryMetadata : repositories.repositories()) {
                        if (repositoryMetadata.name().equals(newRepositoryMetadata.name())) {
                            found = true;
                            repositoriesMetadata.add(newRepositoryMetadata);
                            continue;
                        }
                        repositoriesMetadata.add(repositoryMetadata);
                    }
                    if (!found) {
                        LOGGER.info("put repository [{}]", (Object)request.name);
                        repositoriesMetadata.add(new RepositoryMetadata(request.name, request.type, request.settings));
                    } else {
                        LOGGER.info("update repository [{}]", (Object)request.name);
                    }
                    repositories = new RepositoriesMetadata(repositoriesMetadata);
                }
                mdBuilder.putCustom("repositories", repositories);
                return ClusterState.builder(currentState).metadata(mdBuilder).build();
            }

            @Override
            public void onFailure(String source, Exception e) {
                LOGGER.warn(() -> new ParameterizedMessage("failed to create repository [{}]", (Object)request2.name), (Throwable)e);
                super.onFailure(source, e);
            }

            @Override
            public boolean mustAck(DiscoveryNode discoveryNode) {
                return discoveryNode.isMasterNode() || discoveryNode.isDataNode();
            }
        });
    }

    public void unregisterRepository(final UnregisterRepositoryRequest request, ActionListener<ClusterStateUpdateResponse> listener) {
        this.clusterService.submitStateUpdateTask(request.cause, new AckedClusterStateUpdateTask<ClusterStateUpdateResponse>((AckedRequest)request, listener){

            @Override
            protected ClusterStateUpdateResponse newResponse(boolean acknowledged) {
                return new ClusterStateUpdateResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) {
                RepositoriesService.this.ensureRepositoryNotInUse(currentState, request.name);
                Metadata metadata = currentState.metadata();
                Metadata.Builder mdBuilder = Metadata.builder(currentState.metadata());
                RepositoriesMetadata repositories = (RepositoriesMetadata)metadata.custom("repositories");
                if (repositories != null && repositories.repositories().size() > 0) {
                    ArrayList<RepositoryMetadata> repositoriesMetadata = new ArrayList<RepositoryMetadata>(repositories.repositories().size());
                    boolean changed = false;
                    for (RepositoryMetadata repositoryMetadata : repositories.repositories()) {
                        if (Regex.simpleMatch(request.name, repositoryMetadata.name())) {
                            LOGGER.info("delete repository [{}]", (Object)repositoryMetadata.name());
                            changed = true;
                            continue;
                        }
                        repositoriesMetadata.add(repositoryMetadata);
                    }
                    if (changed) {
                        repositories = new RepositoriesMetadata(repositoriesMetadata);
                        mdBuilder.putCustom("repositories", repositories);
                        return ClusterState.builder(currentState).metadata(mdBuilder).build();
                    }
                }
                if (Regex.isMatchAllPattern(request.name)) {
                    return currentState;
                }
                throw new RepositoryMissingException(request.name);
            }

            @Override
            public boolean mustAck(DiscoveryNode discoveryNode) {
                return discoveryNode.isMasterNode() || discoveryNode.isDataNode();
            }
        });
    }

    public void verifyRepository(final String repositoryName, ActionListener<VerifyResponse> listener) {
        final Repository repository = this.repository(repositoryName);
        final boolean readOnly = repository.isReadOnly();
        this.threadPool.executor("snapshot").execute(new ActionRunnable<VerifyResponse>(listener){

            @Override
            protected void doRun() {
                String verificationToken = repository.startVerification();
                if (verificationToken != null) {
                    try {
                        RepositoriesService.this.verifyAction.verify(repositoryName, readOnly, verificationToken, ActionListener.delegateFailure(this.listener, (delegatedListener, verifyResponse) -> RepositoriesService.this.threadPool.executor("snapshot").execute(() -> {
                            try {
                                repository.endVerification(verificationToken);
                            }
                            catch (Exception e) {
                                LOGGER.warn(() -> new ParameterizedMessage("[{}] failed to finish repository verification", (Object)repositoryName), (Throwable)e);
                                delegatedListener.onFailure(e);
                                return;
                            }
                            delegatedListener.onResponse(verifyResponse);
                        })));
                    }
                    catch (Exception e) {
                        RepositoriesService.this.threadPool.executor("snapshot").execute(() -> {
                            try {
                                repository.endVerification(verificationToken);
                            }
                            catch (Exception inner) {
                                inner.addSuppressed(e);
                                LOGGER.warn(() -> new ParameterizedMessage("[{}] failed to finish repository verification", (Object)repositoryName), (Throwable)inner);
                            }
                            this.listener.onFailure(e);
                        });
                    }
                } else {
                    this.listener.onResponse(new VerifyResponse(new DiscoveryNode[0], new VerificationFailure[0]));
                }
            }
        });
    }

    @Override
    public void applyClusterState(ClusterChangedEvent event) {
        try {
            RepositoriesMetadata oldMetadata = (RepositoriesMetadata)event.previousState().getMetadata().custom("repositories");
            RepositoriesMetadata newMetadata = (RepositoriesMetadata)event.state().getMetadata().custom("repositories");
            if (oldMetadata == null && newMetadata == null || oldMetadata != null && oldMetadata.equals(newMetadata)) {
                return;
            }
            LOGGER.trace("processing new index repositories for state version [{}]", (Object)event.state().version());
            HashMap<String, Repository> survivors = new HashMap<String, Repository>();
            for (Map.Entry<String, Repository> entry : this.repositories.entrySet()) {
                if (newMetadata == null || newMetadata.repository(entry.getKey()) == null) {
                    LOGGER.debug("unregistering repository [{}]", (Object)entry.getKey());
                    this.closeRepository(entry.getValue());
                    continue;
                }
                survivors.put(entry.getKey(), entry.getValue());
            }
            HashMap<String, Repository> builder = new HashMap<String, Repository>();
            if (newMetadata != null) {
                for (RepositoryMetadata repositoryMetadata : newMetadata.repositories()) {
                    Repository repository = (Repository)survivors.get(repositoryMetadata.name());
                    if (repository != null) {
                        RepositoryMetadata previousMetadata = repository.getMetadata();
                        if (!previousMetadata.type().equals(repositoryMetadata.type()) || !previousMetadata.settings().equals(repositoryMetadata.settings())) {
                            LOGGER.debug("updating repository [{}]", (Object)repositoryMetadata.name());
                            this.closeRepository(repository);
                            repository = null;
                            try {
                                repository = this.createRepository(repositoryMetadata);
                            }
                            catch (RepositoryException ex) {
                                LOGGER.warn(() -> new ParameterizedMessage("failed to change repository [{}]", (Object)repositoryMetadata.name()), (Throwable)ex);
                            }
                        }
                    } else {
                        try {
                            repository = this.createRepository(repositoryMetadata);
                        }
                        catch (RepositoryException ex) {
                            LOGGER.warn(() -> new ParameterizedMessage("failed to create repository [{}]", (Object)repositoryMetadata.name()), (Throwable)ex);
                        }
                    }
                    if (repository == null) continue;
                    LOGGER.debug("registering repository [{}]", (Object)repositoryMetadata.name());
                    builder.put(repositoryMetadata.name(), repository);
                }
            }
            this.repositories = Collections.unmodifiableMap(builder);
        }
        catch (Exception ex) {
            LOGGER.warn("failure updating cluster state ", (Throwable)ex);
        }
    }

    public Repository repository(String repositoryName) {
        Repository repository = this.repositories.get(repositoryName);
        if (repository != null) {
            return repository;
        }
        throw new RepositoryMissingException(repositoryName);
    }

    private boolean registerRepository(RepositoryMetadata repositoryMetadata) throws IOException {
        RepositoryMetadata previousMetadata;
        Repository previous = this.repositories.get(repositoryMetadata.name());
        if (previous != null && (previousMetadata = previous.getMetadata()).equals(repositoryMetadata)) {
            return false;
        }
        Repository newRepo = this.createRepository(repositoryMetadata);
        if (previous != null) {
            this.closeRepository(previous);
        }
        HashMap<String, Repository> newRepositories = new HashMap<String, Repository>(this.repositories);
        newRepositories.put(repositoryMetadata.name(), newRepo);
        this.repositories = newRepositories;
        return true;
    }

    private void closeRepository(Repository repository) throws IOException {
        LOGGER.debug("closing repository [{}][{}]", (Object)repository.getMetadata().type(), (Object)repository.getMetadata().name());
        repository.close();
    }

    private Repository createRepository(RepositoryMetadata repositoryMetadata) {
        LOGGER.debug("creating repository [{}][{}]", (Object)repositoryMetadata.type(), (Object)repositoryMetadata.name());
        Repository.Factory factory = this.typesRegistry.get(repositoryMetadata.type());
        if (factory == null) {
            throw new RepositoryException(repositoryMetadata.name(), "repository type [" + repositoryMetadata.type() + "] does not exist");
        }
        try {
            Repository repository = factory.create(repositoryMetadata, this.typesRegistry::get);
            repository.start();
            return repository;
        }
        catch (Exception e) {
            LOGGER.warn(() -> new ParameterizedMessage("failed to create repository [{}][{}]", (Object)repositoryMetadata.type(), (Object)repositoryMetadata.name()), (Throwable)e);
            throw new RepositoryException(repositoryMetadata.name(), "failed to create repository", e);
        }
    }

    private void ensureRepositoryNotInUse(ClusterState clusterState, String repository) {
        if (SnapshotsService.isRepositoryInUse(clusterState, repository) || RestoreService.isRepositoryInUse(clusterState, repository)) {
            throw new IllegalStateException("trying to modify or unregister repository that is currently used ");
        }
    }

    public Collection<Repository> getRepositoriesList() {
        return Collections.unmodifiableCollection(this.repositories.values());
    }

    public static class RegisterRepositoryRequest
    extends ClusterStateUpdateRequest<RegisterRepositoryRequest> {
        final String cause;
        final String name;
        final String type;
        final boolean verify;
        Settings settings = Settings.EMPTY;

        public RegisterRepositoryRequest(String cause, String name, String type, boolean verify) {
            this.cause = cause;
            this.name = name;
            this.type = type;
            this.verify = verify;
        }

        public RegisterRepositoryRequest settings(Settings settings) {
            this.settings = settings;
            return this;
        }
    }

    public static class UnregisterRepositoryRequest
    extends ClusterStateUpdateRequest<UnregisterRepositoryRequest> {
        final String cause;
        final String name;

        public UnregisterRepositoryRequest(String cause, String name) {
            this.cause = cause;
            this.name = name;
        }
    }

    public static class VerifyResponse {
        private final VerificationFailure[] failures;
        private final DiscoveryNode[] nodes;

        public VerifyResponse(DiscoveryNode[] nodes, VerificationFailure[] failures) {
            this.nodes = nodes;
            this.failures = failures;
        }

        public VerificationFailure[] failures() {
            return this.failures;
        }

        public DiscoveryNode[] nodes() {
            return this.nodes;
        }

        public boolean failed() {
            return this.failures.length > 0;
        }

        public String failureDescription() {
            return Arrays.stream(this.failures).map(failure -> failure.toString()).collect(Collectors.joining(", ", "[", "]"));
        }
    }
}

