/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.discovery.ec2;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.ec2.model.DescribeInstancesRequest;
import com.amazonaws.services.ec2.model.DescribeInstancesResult;
import com.amazonaws.services.ec2.model.Filter;
import com.amazonaws.services.ec2.model.GroupIdentifier;
import com.amazonaws.services.ec2.model.Instance;
import com.amazonaws.services.ec2.model.Reservation;
import com.amazonaws.services.ec2.model.Tag;
import io.crate.common.unit.TimeValue;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.common.util.SingleObjectCache;
import org.elasticsearch.discovery.SeedHostsProvider;
import org.elasticsearch.discovery.ec2.AmazonEc2Reference;
import org.elasticsearch.discovery.ec2.AwsEc2Service;
import org.elasticsearch.transport.TransportService;

class AwsEc2SeedHostsProvider
implements SeedHostsProvider {
    private static final Logger LOGGER = LogManager.getLogger(AwsEc2SeedHostsProvider.class);
    private final TransportService transportService;
    private final AwsEc2Service awsEc2Service;
    private final boolean bindAnyGroup;
    private final Set<String> groups;
    private final Map<String, List<String>> tags;
    private final Set<String> availabilityZones;
    private final String hostType;
    private final TransportAddressesCache dynamicHosts;

    AwsEc2SeedHostsProvider(Settings settings, TransportService transportService, AwsEc2Service awsEc2Service) {
        this.transportService = transportService;
        this.awsEc2Service = awsEc2Service;
        this.hostType = AwsEc2Service.HOST_TYPE_SETTING.get(settings);
        this.dynamicHosts = new TransportAddressesCache(AwsEc2Service.NODE_CACHE_TIME_SETTING.get(settings));
        this.bindAnyGroup = AwsEc2Service.ANY_GROUP_SETTING.get(settings);
        this.groups = new HashSet<String>();
        this.groups.addAll((Collection<String>)AwsEc2Service.GROUPS_SETTING.get(settings));
        this.tags = AwsEc2Service.TAG_SETTING.getAsMap(settings);
        this.availabilityZones = new HashSet<String>();
        this.availabilityZones.addAll((Collection<String>)AwsEc2Service.AVAILABILITY_ZONES_SETTING.get(settings));
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("using host_type [{}], tags [{}], groups [{}] with any_group [{}], availability_zones [{}]", (Object)this.hostType, this.tags, this.groups, (Object)this.bindAnyGroup, this.availabilityZones);
        }
    }

    @Override
    public List<TransportAddress> getSeedAddresses(SeedHostsProvider.HostsResolver hostsResolver) {
        return (List)this.dynamicHosts.getOrRefresh();
    }

    private List<TransportAddress> fetchDynamicNodes() {
        DescribeInstancesResult descInstances;
        ArrayList<TransportAddress> dynamicHosts = new ArrayList<TransportAddress>();
        try (AmazonEc2Reference clientReference = this.awsEc2Service.client();){
            descInstances = clientReference.client().describeInstances(this.buildDescribeInstancesRequest());
        }
        catch (AmazonClientException e) {
            LOGGER.info("Exception while retrieving instance list from AWS API: {}", (Object)e.getMessage());
            LOGGER.debug("Full exception:", (Throwable)e);
            return dynamicHosts;
        }
        LOGGER.trace("finding seed nodes...");
        for (Reservation reservation : descInstances.getReservations()) {
            for (Instance instance : reservation.getInstances()) {
                if (!this.groups.isEmpty()) {
                    List instanceSecurityGroups = instance.getSecurityGroups();
                    ArrayList<String> securityGroupNames = new ArrayList<String>(instanceSecurityGroups.size());
                    ArrayList<String> securityGroupIds = new ArrayList<String>(instanceSecurityGroups.size());
                    for (GroupIdentifier sg : instanceSecurityGroups) {
                        securityGroupNames.add(sg.getGroupName());
                        securityGroupIds.add(sg.getGroupId());
                    }
                    if (this.bindAnyGroup) {
                        if (Collections.disjoint(securityGroupNames, this.groups) && Collections.disjoint(securityGroupIds, this.groups)) {
                            LOGGER.trace("filtering out instance {} based on groups {}, not part of {}", (Object)instance.getInstanceId(), (Object)instanceSecurityGroups, this.groups);
                            continue;
                        }
                    } else if (!securityGroupNames.containsAll(this.groups) && !securityGroupIds.containsAll(this.groups)) {
                        LOGGER.trace("filtering out instance {} based on groups {}, does not include all of {}", (Object)instance.getInstanceId(), (Object)instanceSecurityGroups, this.groups);
                        continue;
                    }
                }
                String address = null;
                if (this.hostType.equals("private_dns")) {
                    address = instance.getPrivateDnsName();
                } else if (this.hostType.equals("private_ip")) {
                    address = instance.getPrivateIpAddress();
                } else if (this.hostType.equals("public_dns")) {
                    address = instance.getPublicDnsName();
                } else if (this.hostType.equals("public_ip")) {
                    address = instance.getPublicIpAddress();
                } else if (this.hostType.startsWith("tag:")) {
                    String tagName = this.hostType.substring("tag:".length());
                    LOGGER.debug("reading hostname from [{}] instance tag", (Object)tagName);
                    List tags = instance.getTags();
                    for (Tag tag : tags) {
                        if (!tag.getKey().equals(tagName)) continue;
                        address = tag.getValue();
                        LOGGER.debug("using [{}] as the instance address", (Object)address);
                    }
                } else {
                    throw new IllegalArgumentException(this.hostType + " is unknown for discovery.ec2.host_type");
                }
                if (address != null) {
                    try {
                        TransportAddress[] addresses = this.transportService.addressesFromString(address);
                        for (int i = 0; i < addresses.length; ++i) {
                            LOGGER.trace("adding {}, address {}, transport_address {}", (Object)instance.getInstanceId(), (Object)address, (Object)addresses[i]);
                            dynamicHosts.add(addresses[i]);
                        }
                        continue;
                    }
                    catch (Exception e) {
                        String finalAddress = address;
                        LOGGER.warn(() -> new ParameterizedMessage("failed to add {}, address {}", (Object)instance.getInstanceId(), (Object)finalAddress), (Throwable)e);
                        continue;
                    }
                }
                LOGGER.trace("not adding {}, address is null, host_type {}", (Object)instance.getInstanceId(), (Object)this.hostType);
            }
        }
        LOGGER.debug("using dynamic transport addresses {}", dynamicHosts);
        return dynamicHosts;
    }

    private DescribeInstancesRequest buildDescribeInstancesRequest() {
        DescribeInstancesRequest describeInstancesRequest = new DescribeInstancesRequest().withFilters(new Filter[]{new Filter("instance-state-name").withValues(new String[]{"running", "pending"})});
        for (Map.Entry<String, List<String>> tagFilter : this.tags.entrySet()) {
            describeInstancesRequest.withFilters(new Filter[]{new Filter("tag:" + tagFilter.getKey()).withValues((Collection)tagFilter.getValue())});
        }
        if (!this.availabilityZones.isEmpty()) {
            describeInstancesRequest.withFilters(new Filter[]{new Filter("availability-zone").withValues(this.availabilityZones)});
        }
        return describeInstancesRequest;
    }

    private final class TransportAddressesCache
    extends SingleObjectCache<List<TransportAddress>> {
        private boolean empty;

        TransportAddressesCache(TimeValue refreshInterval) {
            super(refreshInterval, new ArrayList());
            this.empty = true;
        }

        @Override
        protected boolean needsRefresh() {
            return this.empty || super.needsRefresh();
        }

        @Override
        protected List<TransportAddress> refresh() {
            List<TransportAddress> nodes = AwsEc2SeedHostsProvider.this.fetchDynamicNodes();
            this.empty = nodes.isEmpty();
            return nodes;
        }
    }
}

