/*
 * Decompiled with CFR 0.152.
 */
package io.crate.execution.engine.collect.files;

import io.crate.analyze.CopyFromParserProperties;
import io.crate.common.collections.Tuple;
import io.crate.data.BatchIterator;
import io.crate.data.Input;
import io.crate.data.Row;
import io.crate.exceptions.Exceptions;
import io.crate.execution.dsl.phases.FileUriCollectPhase;
import io.crate.execution.engine.collect.files.FileInput;
import io.crate.execution.engine.collect.files.FileInputFactory;
import io.crate.execution.engine.collect.files.Globs;
import io.crate.execution.engine.collect.files.LineCollectorExpression;
import io.crate.execution.engine.collect.files.LineProcessor;
import io.crate.execution.engine.collect.files.URLFileInput;
import io.crate.expression.InputRow;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class FileReadingIterator
implements BatchIterator<Row> {
    private static final Logger LOGGER = LogManager.getLogger(FileReadingIterator.class);
    private static final int MAX_SOCKET_TIMEOUT_RETRIES = 5;
    private final Map<String, FileInputFactory> fileInputFactories;
    private final Boolean shared;
    private final int numReaders;
    private final int readerNumber;
    private final boolean compressed;
    private static final Pattern HAS_GLOBS_PATTERN = Pattern.compile("(.*)[^\\\\]\\*.*");
    private static final Predicate<URI> MATCH_ALL_PREDICATE = input -> true;
    private final List<UriWithGlob> urisWithGlob;
    private final Iterable<LineCollectorExpression<?>> collectorExpressions;
    private volatile Throwable killed;
    private final CopyFromParserProperties parserProperties;
    private final FileUriCollectPhase.InputFormat inputFormat;
    private Iterator<Tuple<FileInput, UriWithGlob>> fileInputsIterator = null;
    private Tuple<FileInput, UriWithGlob> currentInput = null;
    private Iterator<URI> currentInputIterator = null;
    private URI currentUri;
    private BufferedReader currentReader = null;
    private long currentLineNumber;
    private final Row row;
    private LineProcessor lineProcessor;

    private FileReadingIterator(Collection<String> fileUris, List<? extends Input<?>> inputs, Iterable<LineCollectorExpression<?>> collectorExpressions, String compression, Map<String, FileInputFactory> fileInputFactories, Boolean shared, int numReaders, int readerNumber, CopyFromParserProperties parserProperties, FileUriCollectPhase.InputFormat inputFormat) {
        this.compressed = compression != null && compression.equalsIgnoreCase("gzip");
        this.row = new InputRow(inputs);
        this.fileInputFactories = fileInputFactories;
        this.shared = shared;
        this.numReaders = numReaders;
        this.readerNumber = readerNumber;
        this.urisWithGlob = this.getUrisWithGlob(fileUris);
        this.collectorExpressions = collectorExpressions;
        this.parserProperties = parserProperties;
        this.inputFormat = inputFormat;
        this.initCollectorState();
    }

    @Override
    public Row currentElement() {
        return this.row;
    }

    @Override
    public void kill(@Nonnull Throwable throwable) {
        this.killed = throwable;
    }

    public static BatchIterator<Row> newInstance(Collection<String> fileUris, List<Input<?>> inputs, Iterable<LineCollectorExpression<?>> collectorExpressions, String compression, Map<String, FileInputFactory> fileInputFactories, Boolean shared, int numReaders, int readerNumber, CopyFromParserProperties parserProperties, FileUriCollectPhase.InputFormat inputFormat) {
        return new FileReadingIterator(fileUris, inputs, collectorExpressions, compression, fileInputFactories, shared, numReaders, readerNumber, parserProperties, inputFormat);
    }

    private void initCollectorState() {
        this.lineProcessor = new LineProcessor(this.parserProperties);
        this.lineProcessor.startCollect(this.collectorExpressions);
        ArrayList<Tuple<FileInput, UriWithGlob>> fileInputs = new ArrayList<Tuple<FileInput, UriWithGlob>>(this.urisWithGlob.size());
        for (UriWithGlob fileUri : this.urisWithGlob) {
            FileInput fileInput = this.getFileInput(fileUri.uri);
            fileInputs.add(new Tuple<FileInput, UriWithGlob>(fileInput, fileUri));
        }
        this.fileInputsIterator = fileInputs.iterator();
    }

    @Override
    public void moveToStart() {
        this.raiseIfKilled();
        this.initCollectorState();
    }

    @Override
    public boolean moveNext() {
        this.raiseIfKilled();
        try {
            if (this.currentReader != null) {
                String line = this.getLine(this.currentReader, this.currentLineNumber, 0);
                if (line == null) {
                    this.closeCurrentReader();
                    return this.moveNext();
                }
                this.lineProcessor.process(line);
                return true;
            }
            if (this.currentInputIterator != null && this.currentInputIterator.hasNext()) {
                this.advanceToNextUri(this.currentInput.v1());
                return this.moveNext();
            }
            if (this.fileInputsIterator != null && this.fileInputsIterator.hasNext()) {
                this.advanceToNextFileInput();
                return this.moveNext();
            }
            this.releaseBatchIteratorState();
            return false;
        }
        catch (IOException e) {
            this.lineProcessor.setFailure(e.getMessage());
            return true;
        }
    }

    private void advanceToNextUri(FileInput fileInput) throws IOException {
        this.currentUri = this.currentInputIterator.next();
        this.initCurrentReader(fileInput, this.currentUri);
    }

    private void advanceToNextFileInput() throws IOException {
        this.currentInput = this.fileInputsIterator.next();
        FileInput fileInput = this.currentInput.v1();
        UriWithGlob fileUri = this.currentInput.v2();
        Predicate<URI> uriPredicate = this.generateUriPredicate(fileInput, fileUri.globPredicate);
        List<URI> uris = FileReadingIterator.getUris(fileInput, fileUri.uri, fileUri.preGlobUri, uriPredicate);
        if (uris.size() > 0) {
            this.currentInputIterator = uris.iterator();
            this.advanceToNextUri(fileInput);
        } else if (fileUri.preGlobUri != null) {
            this.lineProcessor.startWithUri(fileUri.uri);
            throw new IOException("Cannot find any URI matching: " + fileUri.uri.toString());
        }
    }

    private void initCurrentReader(FileInput fileInput, URI uri) throws IOException {
        this.lineProcessor.startWithUri(uri);
        InputStream stream = fileInput.getStream(uri);
        this.currentReader = this.createBufferedReader(stream);
        this.currentLineNumber = 0L;
        this.lineProcessor.readFirstLine(this.currentUri, this.inputFormat, this.currentReader);
    }

    private void closeCurrentReader() {
        if (this.currentReader != null) {
            try {
                this.currentReader.close();
            }
            catch (IOException e) {
                LOGGER.error("Unable to close reader for {}", (Object)e, (Object)this.currentUri);
            }
            this.currentReader = null;
        }
    }

    private String getLine(BufferedReader reader, long startFrom, int retry) throws IOException {
        String line = null;
        try {
            while ((line = reader.readLine()) != null) {
                ++this.currentLineNumber;
                if (this.currentLineNumber < startFrom || line.length() == 0) {
                    continue;
                }
                break;
            }
        }
        catch (SocketTimeoutException e) {
            if (retry > 5) {
                URI uri = this.currentInput.v2().uri;
                LOGGER.info("Timeout during COPY FROM '{}' after {} retries", (Object)e, (Object)uri.toString(), (Object)retry);
                throw e;
            }
            long startLine = this.currentLineNumber + 1L;
            this.closeCurrentReader();
            this.initCurrentReader(this.currentInput.v1(), this.currentUri);
            return this.getLine(this.currentReader, startLine, retry + 1);
        }
        catch (Exception e) {
            URI uri = this.currentInput.v2().uri;
            LOGGER.info("Error during COPY FROM '{}'", (Object)e, (Object)uri.toString());
            Exceptions.rethrowUnchecked(e);
        }
        return line;
    }

    @Override
    public void close() {
        this.closeCurrentReader();
        this.releaseBatchIteratorState();
        this.killed = BatchIterator.CLOSED;
    }

    private void releaseBatchIteratorState() {
        this.fileInputsIterator = null;
        this.currentInputIterator = null;
        this.currentInput = null;
        this.currentUri = null;
    }

    @Override
    public CompletableFuture<?> loadNextBatch() {
        throw new IllegalStateException("All batches already loaded");
    }

    @Override
    public boolean allLoaded() {
        return true;
    }

    @Override
    public boolean hasLazyResultSet() {
        return true;
    }

    private List<UriWithGlob> getUrisWithGlob(Collection<String> fileUris) {
        ArrayList<UriWithGlob> uris = new ArrayList<UriWithGlob>(fileUris.size());
        for (String fileUri : fileUris) {
            URI uri = this.toURI(fileUri);
            URI preGlobUri = null;
            GlobPredicate globPredicate = null;
            Matcher hasGlobMatcher = HAS_GLOBS_PATTERN.matcher(uri.toString());
            if (hasGlobMatcher.matches()) {
                if (fileUri.startsWith("/") || fileUri.startsWith("file://")) {
                    String newPathAsString;
                    String oldPathAsString;
                    Path oldPath = Paths.get(this.toURI(hasGlobMatcher.group(1)));
                    if (!Files.isDirectory(oldPath, new LinkOption[0])) {
                        oldPath = oldPath.getParent();
                    }
                    try {
                        oldPathAsString = oldPath.toUri().toString();
                        newPathAsString = oldPath.toRealPath(new LinkOption[0]).toUri().toString();
                    }
                    catch (IOException e) {
                        continue;
                    }
                    String resolvedFileUrl = uri.toString().replace(oldPathAsString, newPathAsString);
                    uri = this.toURI(resolvedFileUrl);
                    preGlobUri = this.toURI(newPathAsString);
                } else {
                    preGlobUri = URI.create(hasGlobMatcher.group(1));
                }
                globPredicate = new GlobPredicate(uri);
            }
            uris.add(new UriWithGlob(uri, preGlobUri, globPredicate));
        }
        return uris;
    }

    private URI toURI(String fileUri) {
        if (fileUri.startsWith("/")) {
            return Paths.get(fileUri, new String[0]).toUri();
        }
        URI uri = URI.create(fileUri);
        if (uri.getScheme() == null) {
            throw new IllegalArgumentException("relative fileURIs are not allowed");
        }
        if (uri.getScheme().equals("file") && !uri.getSchemeSpecificPart().startsWith("///")) {
            throw new IllegalArgumentException("Invalid fileURI");
        }
        return uri;
    }

    @Nullable
    private FileInput getFileInput(URI fileUri) {
        FileInputFactory fileInputFactory = this.fileInputFactories.get(fileUri.getScheme());
        if (fileInputFactory != null) {
            return fileInputFactory.create();
        }
        return new URLFileInput(fileUri);
    }

    private BufferedReader createBufferedReader(InputStream inputStream) throws IOException {
        BufferedReader reader = this.compressed ? new BufferedReader(new InputStreamReader((InputStream)new GZIPInputStream(inputStream), StandardCharsets.UTF_8)) : new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
        return reader;
    }

    private static List<URI> getUris(FileInput fileInput, URI fileUri, URI preGlobUri, Predicate<URI> uriPredicate) throws IOException {
        List<URI> uris = preGlobUri != null ? fileInput.listUris(preGlobUri, uriPredicate) : (uriPredicate.test(fileUri) ? List.of(fileUri) : List.of());
        return uris;
    }

    private Predicate<URI> generateUriPredicate(FileInput fileInput, @Nullable Predicate<URI> globPredicate) {
        boolean sharedStorage = Objects.requireNonNullElse(this.shared, fileInput.sharedStorageDefault());
        Predicate<URI> moduloPredicate = sharedStorage ? input -> {
            int hash = input.hashCode();
            if (hash == Integer.MIN_VALUE) {
                hash = 0;
            }
            return Math.abs(hash) % this.numReaders == this.readerNumber;
        } : MATCH_ALL_PREDICATE;
        if (globPredicate != null) {
            return moduloPredicate.and(globPredicate);
        }
        return moduloPredicate;
    }

    private void raiseIfKilled() {
        if (this.killed != null) {
            Exceptions.rethrowUnchecked(this.killed);
        }
    }

    private static class UriWithGlob {
        final URI uri;
        final URI preGlobUri;
        @Nullable
        final Predicate<URI> globPredicate;

        UriWithGlob(URI uri, URI preGlobUri, Predicate<URI> globPredicate) {
            this.uri = uri;
            this.preGlobUri = preGlobUri;
            this.globPredicate = globPredicate;
        }
    }

    private static class GlobPredicate
    implements Predicate<URI> {
        private final Pattern globPattern;

        GlobPredicate(URI fileUri) {
            this.globPattern = Pattern.compile(Globs.toUnixRegexPattern(fileUri.toString()));
        }

        @Override
        public boolean test(@Nullable URI input) {
            return input != null && this.globPattern.matcher(input.toString()).matches();
        }
    }
}

