/*
 * Decompiled with CFR 0.152.
 */
package io.crate.blob;

import io.crate.blob.BlobCoordinator;
import io.crate.blob.DigestBlob;
import io.crate.blob.exceptions.DigestNotFoundException;
import io.crate.common.Hex;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Locale;
import java.util.NoSuchElementException;
import java.util.UUID;
import java.util.concurrent.Semaphore;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class BlobContainer {
    private static final Logger LOGGER = LogManager.getLogger(BlobContainer.class);
    private static final String[] SUB_DIRS = new String[256];
    public static final byte[] PREFIXES = new byte[256];
    private final File[] subDirs = new File[256];
    private final Path baseDirectory;
    private final Path tmpDirectory;
    private final Path varDirectory;
    private final BlobCoordinator blobCoordinator;

    public BlobContainer(Path baseDirectory) {
        this.baseDirectory = baseDirectory;
        this.tmpDirectory = baseDirectory.resolve("tmp");
        this.varDirectory = baseDirectory.resolve("var");
        this.blobCoordinator = new BlobCoordinator();
        try {
            Files.createDirectories(this.varDirectory, new FileAttribute[0]);
            this.createSubDirectories(this.varDirectory);
        }
        catch (IOException e) {
            LOGGER.error("Could not create 'var' path {}", (Object)this.varDirectory);
            throw new RuntimeException(e);
        }
        try {
            Files.createDirectories(this.tmpDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            LOGGER.error("Could not create 'tmp' path {}", (Object)this.tmpDirectory);
            throw new RuntimeException(e);
        }
    }

    private void createSubDirectories(Path parentDir) throws IOException {
        for (int i = 0; i < SUB_DIRS.length; ++i) {
            Path subDir = parentDir.resolve(SUB_DIRS[i]);
            this.subDirs[i] = subDir.toFile();
            Files.createDirectories(subDir, new FileAttribute[0]);
        }
    }

    public Iterable<File> getFiles() {
        return new RecursiveFileIterable(this.subDirs);
    }

    public byte[][] cleanAndReturnDigests(byte prefix) {
        int index = prefix & 0xFF;
        String[] names = this.cleanDigests(this.subDirs[index].list(), index);
        byte[][] digests = new byte[names.length][];
        for (int i = 0; i < names.length; ++i) {
            try {
                digests[i] = Hex.decodeHex(names[i]);
                continue;
            }
            catch (IllegalStateException ex) {
                LOGGER.error("Can't convert string {} to byte array", (Object)names[i]);
                throw ex;
            }
        }
        return digests;
    }

    private String[] cleanDigests(String[] names, int index) {
        if (names == null) {
            return null;
        }
        ArrayList<String> newNames = new ArrayList<String>(names.length);
        for (String name : names) {
            if (name.contains(".")) {
                if (new File(this.subDirs[index], name).delete()) continue;
                LOGGER.error("Could not delete {}/{}", (Object)this.subDirs[index], (Object)name);
                continue;
            }
            newNames.add(name);
        }
        return newNames.toArray(new String[newNames.size()]);
    }

    public void visitBlobs(FileVisitor<Path> visitor) throws IOException {
        Files.walkFileTree(this.varDirectory, visitor);
    }

    public Semaphore digestCoordinator(String digest) {
        return this.blobCoordinator.digestCoordinator(digest);
    }

    public Path getBaseDirectory() {
        return this.baseDirectory;
    }

    public Path getTmpDirectory() {
        return this.tmpDirectory;
    }

    public File getFile(String digest) {
        return this.varDirectory.resolve(digest.substring(0, 2)).resolve(digest).toFile();
    }

    public DigestBlob createBlob(String digest, UUID transferId) {
        return new DigestBlob(this, digest, transferId);
    }

    public RandomAccessFile getRandomAccessFile(String digest) {
        try {
            return new RandomAccessFile(this.getFile(digest), "r");
        }
        catch (FileNotFoundException e) {
            throw new DigestNotFoundException(digest);
        }
    }

    static {
        for (int i = 0; i < 256; ++i) {
            BlobContainer.SUB_DIRS[i] = String.format(Locale.ENGLISH, "%02x", i & 0xFFFFF);
            BlobContainer.PREFIXES[i] = (byte)i;
        }
    }

    private static class RecursiveFileIterable
    implements Iterable<File> {
        private final File[] subDirs;

        private RecursiveFileIterable(File[] subDirs) {
            this.subDirs = subDirs;
        }

        @Override
        public Iterator<File> iterator() {
            return new RecursiveFileIterator(this.subDirs);
        }
    }

    private static class RecursiveFileIterator
    implements Iterator<File> {
        private final File[] subDirs;
        private int subDirIndex = -1;
        private File[] files = null;
        private int fileIndex = -1;

        private RecursiveFileIterator(File[] subDirs) {
            this.subDirs = subDirs;
        }

        @Override
        public boolean hasNext() {
            if (this.files == null || this.fileIndex + 1 == this.files.length) {
                this.files = null;
                this.fileIndex = -1;
                while (this.subDirIndex + 1 < this.subDirs.length && (this.files == null || this.files.length == 0)) {
                    this.files = this.subDirs[++this.subDirIndex].listFiles();
                }
            }
            return this.files != null && this.fileIndex + 1 < this.files.length;
        }

        @Override
        public File next() {
            if (this.hasNext()) {
                return this.files[++this.fileIndex];
            }
            throw new NoSuchElementException("List of files is empty");
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("remove is unsupported for " + BlobContainer.class.getSimpleName());
        }
    }
}

