// Main app view.
(function($, UTILS) {

    $.widget('adsk.appview', {
        options: {
            title: 'Book Title',

            minNavigationWidth: 320,

            ui: {
                tabs: {
                    order: ['C', 'I', 'S', 'F'],
                    tocEnabled: true,
                    indexEnabled: true,
                    favoritesEnabled: true,
                    searchEnabled: true
                },

                tools: {
                    order: ['home', 'previous', 'parent', 'following', 'separator',
                            'favorites', 'share', 'separator', 'print'],
                    home: true,
                    previous: true,
                    parent: true,
                    following: true,
                    favorites: true,
                    share: true,
                    print: true
                },

                responsive: true,

				textDirection: 'rtl'
            },

            features: {
                tabsbarParent: 'banner',
                banner: true,
                header: true,
                navigation: true,
                splitter: true,
                breadcrums: true,
                splitter: true
            },

            labels: {
                tabs: {
                }
            }
        },

        /* Private members. */

        _$banner:       undefined,
        _$main:         undefined,
        _$breadcrumbs:  undefined,
        _$splitter:     undefined,
        _$navigation:   undefined,
        _$topic:        undefined,

        _splitterTracking:                  false,
        _splitterMoved:                     false,
        _splitterOriginalX:                 undefined,
        _navigationOriginalWidth:           undefined,
        _topicOriginalWidth:                undefined,
        _navigationOpenedWidth:             undefined,
        _topicOriginalMargin: 	            undefined,
		_dirSign: 							1,
		_dirTopicMarginProperty: 			'margin-left',

        _create: function() {
            console.log('appview: _create');

            var self = this;

            // Show "print" button only when functionality supported.
            if (!window.print) {
                this.options.ui.tools.print = false;
            }

			// Determine text direction.
			if ($('html').attr('dir') == 'rtl') {
				this.options.ui.textDirection = 'rtl';
				this._dirSign = -1;
				this._dirTopicMarginProperty = 'margin-right';
			} else {
				this.options.ui.textDirection = 'ltr';
				this._dirSign = 1;
				this._dirTopicMarginProperty = 'margin-left';
			}

            // Create layout.
            this._render();

            // Initial resize.
            this._resizeHandler();
            this._resizeTabButtons();

            // Perform transition.
            this._$main.transition();

            // Add event handlers.
            this._addEventHandlers(self);

            // Initial app resize.
            this._resizeHandler();

            // Show UI.
            this.element.removeClass('hidden');
        },

        _addEventHandlers: function(self) {
            // Add app resize handler.
            $(window).bind('resize', function() {
                self._resizeHandler();
            });

            self._addViewEventHandlers(self);
        },

        _addViewEventHandlers: function(self) {
            console.log('appview: _addEventHandlers');

            // IE7 hover/active fallback.
            if ($('html').hasClass('ie7') || $('html').hasClass('ie8')) {
                $('a')
                .mousedown(function() {
                    $(this).addClass('active');
                })
                .mouseup(function() {
                    $(this).removeClass('active');
                })
                .mouseenter(function() {
                    $(this).addClass('hover');
                })
                .mouseleave(function() {
                    $(this).removeClass('hover');
                });
            }

            // Add toolbar buttons tab bar links handlers.
            self._on('.button', {
                click: "_handleToolbarButtonClick"
            });

            // Add tab bar links handlers.
            self._on('.tab-button', {
                click: "_handleTabButtonClick"
            });

            // Splitter event handlers.
            self._on(this._$splitter, {
                mousedown: "_splitterMouseDownHandler",
                mouseup: function() {
                    // Toggle pane.
                    if (this._splitterMoved == false) {
                        this._toggleNavigationPane();
                    }
                }
            });

            self._on({
                mousemove: "_splitterMouseMoveHandler",
                mouseup: function() {
                    this._splitterTracking = false;
                    this.element.removeClass('no-user-select');
                }
            });
        },

        _toggleNavigationPane: function() {
            var navigationPaneWidth = this._$navigation.width();

            if (navigationPaneWidth > 0) {
                // Remember current width.
                this._navigationOpenedWidth = navigationPaneWidth;

                // Collapse navigation pane.
                var newTopicMargin = this._topicOriginalMargin - navigationPaneWidth;

                // Resize panes.
                this._resizeSplitPanes(0, 0, 0, 0, newTopicMargin);

                // Hide navigation pane.
                this._$navigation.css('visibility', 'hidden');
            } else {
                // Show navigation pane.
                this._$navigation.css('visibility', 'visible');

                // Open navigation pane.
                var newNavigationPaneWidth = this._navigationOpenedWidth,
                    newTopicMargin = this._topicOriginalMargin + newNavigationPaneWidth;

                this._resizeSplitPanes(0, 0, newNavigationPaneWidth, 0, newTopicMargin);
            }
        },

        _splitterMouseDownHandler: function(event) {
            this._splitterMoved = false;

            // Set up splitter.
            this._splitterOriginalX = event.clientX;
            this._navigationOriginalWidth = this._$navigation.width();
            this._topicOriginalWidth = this._$topic.width();
            this._topicOriginalMargin = parseInt(this._$topic.css(this._dirTopicMarginProperty), 10);

            // Prevent user select.
            this.element.addClass('no-user-select');

            // Acgtivate splitter tracking if pane is not collapsed.
            if (this._navigationOriginalWidth > 0) {
                this._splitterTracking = true;
            }
        },

        _splitterMouseMoveHandler: function(event) {
            this._splitterMoved = true;

            if (this._splitterTracking) {
                // Track splitter.
                var topicWidth = this._$topic.width(),
                    navigationWidth = this._$navigation.width(),
                    deltaX = this._dirSign * (event.clientX - this._splitterOriginalX),
                    minNavigationWidth = this.options.minNavigationWidth;

                if ((deltaX < 0 && navigationWidth > this.options.minNavigationWidth) ||
                    (deltaX > 0 && topicWidth > 0)) {
                    // Resize panes.
                    var newNavigationWidth = this._navigationOriginalWidth + deltaX,
                        newTopicMargin = this._topicOriginalMargin + deltaX,
                        newTopicWidth = this._topicOriginalWidth - deltaX;

                    this._resizeSplitPanes(minNavigationWidth, 0, newNavigationWidth, newTopicWidth, newTopicMargin);

                }
            }
        },

        _resizeSplitPanes: function(minNavigationWidth, minTopicWidth, newNavigationWidth, newTopicWidth, newTopicMargin) {
            if (newNavigationWidth < minNavigationWidth) {
                var delta = minNavigationWidth - newNavigationWidth;
                newNavigationWidth += delta;
                newTopicMargin += delta;
            } else if (newTopicWidth < minTopicWidth) {
                var delta = -(minTopicWidth - newTopicWidth);
                newNavigationWidth += delta;
                newTopicMargin += delta;
            }

            this._$navigation.width(newNavigationWidth);
            this._$topic.css(this._dirTopicMarginProperty, newTopicMargin + 'px');

            this._resizeTabButtons();
        },

        _resizeTabButtons: function() {
            // Stretch tab buttons for non-responsive UI.
            var tabButtons = this._$navigation.find('.tab-button'),
                tabButtonsLen = tabButtons.length;

            // Resize tab buttons if all of them are within navigation pane (non-responsive UI).
            if (tabButtonsLen) {
                var navWidth = this._$navigation.width();
                tabButtons.css('width', Math.floor(navWidth / tabButtons.length)  + 'px');

                // Show tab icons navigation pane is wide enough.
                if (navWidth >= 600) {
                    this._$navigation.addClass('wide');
                } else {
                    this._$navigation.removeClass('wide');
                }
            }
        },

        _render: function() {
            console.log('appview: _render');

            // Create app view markup.
            this.element
            .html(this._renderBanner() +
                  '<div id="main">' +
                    this._renderNavigation() +
                    this._renderSplitter() +
                    '<div id="topic"></div>' +
                  '</div>' +
                  this._renderBreadcrumbs() +
                  '<div id="share" class="hidden"></div>');

            var e = this.element;
            this._$banner       = e.find('#banner');
            this._$main         = e.find('#main');
            this._$splitter     = e.find('#splitter');
            this._$navigation   = e.find('#navigation');
            this._$tabsbar   = this._$navigation.find('.tabsbar');
            this._$topic        = e.find('#topic');
            this._$breadcrumbs  = e.find('#breadcrumbs');
        },

        _renderBanner: function() {
            console.log('appview: _renderBanner');

            var html = ''

            if (this.options.features.banner) {
                html +=  '<div id="banner">' +
                            this._renderHeader() +
                            this._renderToolbar();

                if (this.options.features.tabsbarParent == 'banner') {
                    html += this._renderTabsBar();
                }

                html += '</div>';
            }

            return html;
        },

        _renderToolbar: function() {
            var html = '<div class="toolbar"><div class="wrapper">',
                tabs = this.options.ui.tabs,
                tools = this.options.ui.tools;

            $.each(tools.order, function(index, tool) {
                switch(tool) {
                    case 'separator':
                        html += '<span class="separator"></span>';
                        break;
                    case 'home':
                        if (tools.home) {
                            html += '<a href="#" id="main-home" class="button icon home"></a>';
                        }
                        break;
                    case 'previous':
                        if (tools.previous) {
                            html += '<a href="#" id="main-previous" class="button icon previous"></a>';
                        }
                        break;
                    case 'parent':
                        if (tools.parent) {
                            html += '<a href="#" id="main-parent" class="button icon parent"></a>';
                        }
                        break;
                    case 'following':
                        if (tools.following) {
                            html += '<a href="#" id="main-following" class="button icon following"></a>';
                        }
                        break;
                    case 'favorites':
                        if (tabs.favoritesEnabled && tools.favorites) {
                            html += '<a href="#" id="main-favorites" class="button icon favorites"></a>';
                        }
                        break;
                    case 'share':
                        if (tools.share) {
                            html += '<a href="#" id="main-share" class="button icon share"></a>';
                        }
                        break;
                    case 'print':
                        if (tools.print) {
                            html += '<a href="#" id="main-print" class="button icon print"></a>';
                        }
                        break;
                }
            });

            html += '</div></div>';

            return html;
        },

        _renderTabsBar: function() {
            var that = this,
                html = '<div class="tabsbar"><div class="wrapper">',
                tabs = this.options.ui.tabs,
                caps = this.options.labels.tabs,
                pos = 0;

            $.each(tabs.order, function(index, tab) {
                switch(tab) {
                    case 'C':
                        if (tabs.tocEnabled) {
                            html += that._renderTabsBarButton('tab-button-toc', caps.contents, pos);
                            ++pos;
                        }
                        break;
                    case 'I':
                        if (tabs.indexEnabled) {
                            html += that._renderTabsBarButton('tab-button-index', caps.index, pos);
                            ++pos;
                        }
                        break;
                    case 'S':
                        if (tabs.searchEnabled) {
                            html += that._renderTabsBarButton('tab-button-search', caps.search, pos);
                            ++pos;
                        }
                        break;
                    case 'F':
                        if (tabs.favoritesEnabled) {
                            html += that._renderTabsBarButton('tab-button-favorites', caps.favorites, pos);
                            ++pos;
                        }
                        break;
                }
            });

            html += '</div></div>';

            return html;
        },

        _renderTabsBarButton: function (id, label, pos) {
            var html = '',
                classes = 'tab-button';

            if (pos == 0) {
                classes += ' opened';
            }

            html += '<a id="' + id + '" class="' + classes + '" href="#">' +
                        '<span>' + label + '</span>' +
                    '</a>';

            return html;
        },

        _renderNavigation: function() {
            console.log('appview: _renderNavigation');

            var html = '',
                position = 0,
                that = this,
                features = this.options.features,
                tabs = this.options.ui.tabs;

            if (features.navigation) {
                html += '<div id="navigation">';

                if (this.options.features.tabsbarParent == 'navigation') {
                    html += this._renderTabsBar();
                }

                $.each(tabs.order, function(index, tab) {
                    switch(tab) {
                        case 'C':
                            if (tabs.tocEnabled) {
                                html += that._renderNavigationTab('tab-toc', position);
                                ++position;
                            }
                            break;
                        case 'I':
                            if (tabs.indexEnabled) {
                                html += that._renderNavigationTab('tab-index', position);
                                ++position;
                            }
                            break;
                        case 'S':
                            if (tabs.searchEnabled) {
                                html += that._renderNavigationTab('tab-search', position);
                                ++position;
                            }
                            break;
                        case 'F':
                            if (tabs.favoritesEnabled) {
                                html += that._renderNavigationTab('tab-favorites', position);
                                ++position;
                            }
                            break;
                    }
                });

                html += '</div>';

            } else {
                this.element.addClass('no-navigation');
            }
            return html;
        },

        _renderNavigationTab: function(id, position) {
            var html = '<div id="' + id + '" class = "tab';

            // Only first tab should be visible.
            if (position > 0) {
                html += ' hidden'
            }

            html += '"></div>';

            return html;
        },

        _renderSplitter: function() {
            console.log('appview: _renderSplitter');

            if (this.options.features.navigation) {
                return '<div id="splitter"></div>';
            } else {
                return '';
            }
        },

        _renderBreadcrumbs: function() {
            console.log('appview: _renderBreadcrumbs');

            if (this.options.features.breadcrumbs) {
                return '<div id="breadcrumbs"></div>';
            } else {
                return '';
            }
        },

        _renderHeader: function() {
            console.log('appview: _renderHeader');

            if (this.options.features.header) {
                return  '<div class="header">' +
                            '<div class="wrapper">' +
                                '<a class="title" href="#">' +
                                    '<img class="product-title" src="images/product-title.png"/>' +
                                    '<span>' + this.options.title + '</span>' +
                                '</a>' +
                            '</div>' +
                        '</div>';
            } else {
                return '';
            }
        },

        _handleToolbarButtonClick: function(event) {
            console.log('appview: _handleToolbarButtonClick');

            event.preventDefault();
            event.stopPropagation();

            var $target = $(event.target);

            if (!$target.hasClass('disabled') && $target.attr('id').indexOf('main') != -1) {
                this._trigger('buttonclick', event, $target.attr('id'));
            }
        },

        _handleTabButtonClick: function(event) {
            console.log('appview: _handleTabButtonClick');

            event.preventDefault();
            event.stopPropagation();

            var $target = $(event.currentTarget);

            if (!$target.hasClass('disabled')) {
                this._trigger('tabbuttonclick', event, $target.attr('id'));
            }
        },

        _showTabButtonById: function(id) {
            var tabButtons = this._$navigation.find('.tab-button'),
                targetButton = tabButtons.filter(id);

            if (tabButtons.length) {
                tabButtons.removeClass('opened');
                targetButton.addClass('opened');
            }
        },

        _getShareButtonOffset: function() {
            console.log('appview: _getShareButtonOffset');

        },

        _resizeHandler: function() {
            console.log('appview: _resizeHandler');

            var $window = $(window);

            // Resize according to a new app height.
            var height = $window.height() - this._$banner.height() - this._$breadcrumbs.height();
            this._$main.css('height', height + 'px');

            // Resize according to a new app width.
            if (this.options.ui.responsive) {
                var width = $window.width();
                if (width > 768) {
                    // Full viewport view.
                    $(document.body).removeClass('snap');
                } else {
                    // Snap view.
                    $(document.body).addClass('snap');
                }
            } else {
                this._$navigation.find('.tab').css('height', height - this._$tabsbar.height() + 'px');
            }
        },

        _constructBreadcrumbHtml: function(breadcrumb) {
            console.log('appview: _constructBreadcrumbHtml');

            return '<li>' +
                      '<a href="#!/url=' + breadcrumb.href + '">' +
                        breadcrumb.title +
                      '</a>' +
                   '</li>' +
                   '&rarr;';
        },

        /* Public members. */

        showPreloader: function() {
            console.log('appview: showPreloader');

            $(document.body).addClass('loading');
        },

        hidePreloader: function() {
            console.log('appview: hidePreloader');

            $(document.body).removeClass('loading');
        },

        initGoToPreviousButton: function(enabled) {
            console.log('appview: initGoToPreviousButton');

            if (enabled) {
                this._$banner.find('#main-previous').removeClass('disabled');
            } else {
                this._$banner.find('#main-previous').addClass('disabled');
            }
        },

        initGoToParentButton: function(enabled) {
            console.log('appview: initGoToParentButton');

            if (enabled) {
                this._$banner.find('#main-parent').removeClass('disabled');
            } else {
                this._$banner.find('#main-parent').addClass('disabled');
            }
        },

        initGoToNextButton: function(enabled) {
            console.log('appview: initGoToNextButton');

            if (enabled) {
                this._$banner.find('#main-following').removeClass('disabled');
            } else {
                this._$banner.find('#main-following').addClass('disabled');
            }
        },

        initFavoritesButton: function(enabled) {
            console.log('appview: initFavoritesButton');

            if (enabled) {
                this._$banner.find('#main-favorites').removeClass('disabled');
            } else {
                this._$banner.find('#main-favorites').addClass('disabled');
            }
        },

        getShareButtonOffset: function() {
            console.log('appview: getShareButtonOffset');

            var offset = this.element.find('#main-share').offset();

            return offset;
        },

        showTab: function(id) {
            console.log('appview: showTab');

            var tabSelector = '#tab-' + id,
                tabButtonSelector = '#tab-button-' + id,
                tabToShow = this._$navigation.find(tabSelector),
                tabs = this._$navigation.children('.tab');

            this._showTabButtonById(tabButtonSelector);

            if (tabToShow.length > 0 && tabToShow.hasClass('hidden')) {
                // Hide all tabs.
                tabs.addClass('hidden').attr('style', '');

                // Show selected tab.
                tabToShow.transition().removeClass('hidden');

                // Recalculate tab height.
                this._resizeHandler();

                // Go to current topic in TOC.
                if (tabToShow.attr('id') == 'tab-toc') {
                    //this.expandToc(tabToShow);
                    //this.scrollTocToSelectedTopic(tabToShow);
                }
            } else {
                console.log('appview: error: tab with id ' + id + ' not found.');
            }
        },

        // Flag if topic is in Favorites or not.
        flagFavorites: function(addedToFavorites) {
            console.log('mainview: flagFavorites, added = ', addedToFavorites);

            if (addedToFavorites) {
                this._$banner.find('#main-favorites').addClass('flagged');
            } else {
                this._$banner.find('#main-favorites').removeClass('flagged');
            }
        },

        updateBreadcrumbs: function(breadcrumbs, rootTopicData) {
            var len = breadcrumbs.length,
                i,
                html = '<ol class="breadcrumbs">';

                 + this._constructBreadcrumbHtml({
                    href: rootTopicData.attr.href,
                    title: rootTopicData.data || 'Home'
                });

            for (i = 0; i < len; ++i) {
                html = html + this._constructBreadcrumbHtml(breadcrumbs[i]);
            }

            html += '</ol>';

            this._$breadcrumbs.empty();

            $(html).appendTo(this._$breadcrumbs);
        }
    });
})(jQuery);
// SIG // Begin signature block
// SIG // MIIgdwYJKoZIhvcNAQcCoIIgaDCCIGQCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // 2pGPazIC4uzh47KMxezcQWoQ2JXjRYOJZuHe/O1LPFmg
// SIG // ggoaMIIEyzCCA7OgAwIBAgIQCQja65wGHi65un4mqmKH
// SIG // ijANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xOTA1MTQwMDAw
// SIG // MDBaFw0yMDA1MTMyMzU5NTlaMHwxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIDApDYWxpZm9ybmlhMRMwEQYDVQQHDApT
// SIG // YW4gUmFmYWVsMRcwFQYDVQQKDA5BdXRvZGVzaywgSW5j
// SIG // LjERMA8GA1UECwwIU2VjdXJpdHkxFzAVBgNVBAMMDkF1
// SIG // dG9kZXNrLCBJbmMuMIIBIjANBgkqhkiG9w0BAQEFAAOC
// SIG // AQ8AMIIBCgKCAQEAkVKRoPYsoxPdlabspcuG8HYDfcWQ
// SIG // 9qyPOT2punwpVdjt2slktcpSTlydrGy8ijcbl5/k8AOJ
// SIG // afKMQvow0fNjIoNB5nsb8j2xFJPbBhHl8GT4P6DZ1OWK
// SIG // ioiUXgGHKjctUws49J4elntqvBYhKCbuPqG772QCoYFU
// SIG // P7qcFbh3PbyNB6cnKMn2vNGRqiEtV2aGg+/MiL3QPsvT
// SIG // kTWhVuTLIvwZ4kK5C5ACiQzlGOlE+C7gWuFoxtvy6gUa
// SIG // A8JAq/xLjYKwFevQwGncg3s1cuiGhx0P7Rvn6oCd2cem
// SIG // FmbK3I3rMRt+jc9wzAj3oHxvCo4f5Lfva+OpC1dETUVV
// SIG // sk0VGwIDAQABo4IBPjCCATowCQYDVR0TBAIwADAOBgNV
// SIG // HQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
// SIG // YQYDVR0gBFowWDBWBgZngQwBBAEwTDAjBggrBgEFBQcC
// SIG // ARYXaHR0cHM6Ly9kLnN5bWNiLmNvbS9jcHMwJQYIKwYB
// SIG // BQUHAgIwGQwXaHR0cHM6Ly9kLnN5bWNiLmNvbS9ycGEw
// SIG // HwYDVR0jBBgwFoAU1MAGIknrOUvdk+JcobhHdglyA1gw
// SIG // KwYDVR0fBCQwIjAgoB6gHIYaaHR0cDovL3JiLnN5bWNi
// SIG // LmNvbS9yYi5jcmwwVwYIKwYBBQUHAQEESzBJMB8GCCsG
// SIG // AQUFBzABhhNodHRwOi8vcmIuc3ltY2QuY29tMCYGCCsG
// SIG // AQUFBzAChhpodHRwOi8vcmIuc3ltY2IuY29tL3JiLmNy
// SIG // dDANBgkqhkiG9w0BAQsFAAOCAQEA0Q2+iVSEfT4xGq3A
// SIG // OADVb0rT5Z8E+PPb1x6hKknSzuaZKk/acDz9mdnFajaU
// SIG // WgIKiFJlsqJnuAM1w4lqKaEWsz3DoxvGIEwexj71Ud6a
// SIG // KlBJNcR9DdkXUZwMhbw9d78rjJAIX6tOwfSludZfdq8b
// SIG // YEkuIzQi9uYW9vHVPSCzhnUBKjwc9qTo9yBc7Fcn4FAT
// SIG // 90DzWM2hS9oc3p0pprYI2I/O4th90BEmwWkVqVgfaR0z
// SIG // VyIEmBbqB/IWLfjUBZJ/8sbkwWuWlYVRrfPxsj1e9qvb
// SIG // wE3Y28jWyGbqoUG39GhXk3uT8R8i3+T6M0c4rMfdifBC
// SIG // pceCaFTJzSye7LoeVzCCBUcwggQvoAMCAQICEHwbNTVK
// SIG // 59t050FfEWnKa6gwDQYJKoZIhvcNAQELBQAwgb0xCzAJ
// SIG // BgNVBAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5j
// SIG // LjEfMB0GA1UECxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29y
// SIG // azE6MDgGA1UECxMxKGMpIDIwMDggVmVyaVNpZ24sIElu
// SIG // Yy4gLSBGb3IgYXV0aG9yaXplZCB1c2Ugb25seTE4MDYG
// SIG // A1UEAxMvVmVyaVNpZ24gVW5pdmVyc2FsIFJvb3QgQ2Vy
// SIG // dGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMTQwNzIyMDAw
// SIG // MDAwWhcNMjQwNzIxMjM1OTU5WjCBhDELMAkGA1UEBhMC
// SIG // VVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9u
// SIG // MR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3Jr
// SIG // MTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1
// SIG // NiBDb2RlIFNpZ25pbmcgQ0EgLSBHMjCCASIwDQYJKoZI
// SIG // hvcNAQEBBQADggEPADCCAQoCggEBANeVQ9Tc32euOftS
// SIG // pLYmMQRw6beOWyq6N2k1lY+7wDDnhthzu9/r0XY/ilaO
// SIG // 6y1L8FcYTrGNpTPTC3Uj1Wp5J92j0/cOh2W13q0c8fU1
// SIG // tCJRryKhwV1LkH/AWU6rnXmpAtceSbE7TYf+wnirv+9S
// SIG // rpyvCNk55ZpRPmlfMBBOcWNsWOHwIDMbD3S+W8sS4duM
// SIG // xICUcrv2RZqewSUL+6McntimCXBx7MBHTI99w94Zzj7u
// SIG // BHKOF9P/8LIFMhlM07Acn/6leCBCcEGwJoxvAMg6ABFB
// SIG // ekGwp4qRBKCZePR3tPNgKuZsUAS3FGD/DVH0qIuE/iHa
// SIG // XF599Sl5T7BEdG9tcv8CAwEAAaOCAXgwggF0MC4GCCsG
// SIG // AQUFBwEBBCIwIDAeBggrBgEFBQcwAYYSaHR0cDovL3Mu
// SIG // c3ltY2QuY29tMBIGA1UdEwEB/wQIMAYBAf8CAQAwZgYD
// SIG // VR0gBF8wXTBbBgtghkgBhvhFAQcXAzBMMCMGCCsGAQUF
// SIG // BwIBFhdodHRwczovL2Quc3ltY2IuY29tL2NwczAlBggr
// SIG // BgEFBQcCAjAZGhdodHRwczovL2Quc3ltY2IuY29tL3Jw
// SIG // YTA2BgNVHR8ELzAtMCugKaAnhiVodHRwOi8vcy5zeW1j
// SIG // Yi5jb20vdW5pdmVyc2FsLXJvb3QuY3JsMBMGA1UdJQQM
// SIG // MAoGCCsGAQUFBwMDMA4GA1UdDwEB/wQEAwIBBjApBgNV
// SIG // HREEIjAgpB4wHDEaMBgGA1UEAxMRU3ltYW50ZWNQS0kt
// SIG // MS03MjQwHQYDVR0OBBYEFNTABiJJ6zlL3ZPiXKG4R3YJ
// SIG // cgNYMB8GA1UdIwQYMBaAFLZ3+mlIR59TEtXC6gcydgfR
// SIG // lwcZMA0GCSqGSIb3DQEBCwUAA4IBAQB/68qn6ot2Qus+
// SIG // jiBUMOO3udz6SD4Wxw9FlRDNJ4ajZvMC7XH4qsJVl5Fw
// SIG // g/lSflJpPMnx4JRGgBi7odSkVqbzHQCR1YbzSIfgy8Q0
// SIG // aCBetMv5Be2cr3BTJ7noPn5RoGlxi9xR7YA6JTKfRK9u
// SIG // QyjTIXW7l9iLi4z+qQRGBIX3FZxLEY3ELBf+1W5/muJW
// SIG // kvGWs60t+fTf2omZzrI4RMD3R3vKJbn6Kmgzm1By3qif
// SIG // 1M0sCzS9izB4QOCNjicbkG8avggVgV3rL+JR51EeyXgp
// SIG // 5x5lvzjvAUoBCSQOFsQUecFBNzTQPZFSlJ3haO8I8OJp
// SIG // nGdukAsak3HUJgLDwFojMYIVtTCCFbECAQEwgZkwgYQx
// SIG // CzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRlYyBD
// SIG // b3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1
// SIG // c3QgTmV0d29yazE1MDMGA1UEAxMsU3ltYW50ZWMgQ2xh
// SIG // c3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENBIC0gRzIC
// SIG // EAkI2uucBh4uubp+Jqpih4owDQYJYIZIAWUDBAIBBQCg
// SIG // fDAQBgorBgEEAYI3AgEMMQIwADAZBgkqhkiG9w0BCQMx
// SIG // DAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgELMQ4wDAYK
// SIG // KwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQggRLwbf8a
// SIG // 7HxLKUhnuJcJcQ69iOeNtQHHkKpwoIwCiMcwDQYJKoZI
// SIG // hvcNAQEBBQAEggEAhXf5/iG1dwNbynh2hCIkvZWVTQy4
// SIG // mUG6djawKelOYg2Af0LXO8Nzb9sIl45XBCTnh+7SCDVb
// SIG // /ceaEZNBPwg4KLgluWL47sLt5AaYEbKszdJ+6bbeDn2N
// SIG // Q6j8x943LDmx5HwRHjd3y1g8+espDNvCBMnoadCnAxkF
// SIG // QzAy6fO7fPXMIKilhDfL9aL6IPigny9ThuRrMowlHT4o
// SIG // 2ekm7T+ZYEdmAl9nh3DhruzaP2x9T4VKrFmSQDZzM3JH
// SIG // haUk1qEkhrik4vRAcVvMLr8LitPVDiRtVE71IjwQYaTq
// SIG // RnloMNZ/cE5vnV6dm42aDoRoLjX+CdIECv0dQQuBpVHM
// SIG // j2F8RqGCE24wghNqBgorBgEEAYI3AwMBMYITWjCCE1YG
// SIG // CSqGSIb3DQEHAqCCE0cwghNDAgEDMQ8wDQYJYIZIAWUD
// SIG // BAIBBQAwggENBgsqhkiG9w0BCRABBKCB/QSB+jCB9wIB
// SIG // AQYKKwYBBAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCA7
// SIG // DYabLtkLccVK2H43vwCow9cEPuP2rncb8m2XJZRUlQIV
// SIG // AOEBq77VJnJ2WgW6iWtGPJqgnyfFGA8yMDIwMDMyMzE0
// SIG // MDAwMlqggYqkgYcwgYQxCzAJBgNVBAYTAkdCMRswGQYD
// SIG // VQQIDBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcM
// SIG // B1NhbGZvcmQxGDAWBgNVBAoMD1NlY3RpZ28gTGltaXRl
// SIG // ZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBTdGFt
// SIG // cGluZyBTaWduZXIgIzGggg36MIIHBjCCBO6gAwIBAgIQ
// SIG // PRo1cjAVgmMw0BNxfoJBCDANBgkqhkiG9w0BAQwFADB9
// SIG // MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBN
// SIG // YW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYD
// SIG // VQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNl
// SIG // Y3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwHhcNMTkw
// SIG // NTAyMDAwMDAwWhcNMzAwODAxMjM1OTU5WjCBhDELMAkG
// SIG // A1UEBhMCR0IxGzAZBgNVBAgMEkdyZWF0ZXIgTWFuY2hl
// SIG // c3RlcjEQMA4GA1UEBwwHU2FsZm9yZDEYMBYGA1UECgwP
// SIG // U2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdv
// SIG // IFJTQSBUaW1lIFN0YW1waW5nIFNpZ25lciAjMTCCAiIw
// SIG // DQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAMtRUP9W
// SIG // /vx4Y3ABk1qeGPQ7U/YHryFs9aIPfR1wLYR0SIucipUF
// SIG // PVmE+ZGAeVEs2Yq3wQuaugqKzWZPA4sBuzDKq73bwE8S
// SIG // XvwKzOJFsAE4irtN59QcVJjtOVjPW8IvRZgxCvk1OLgx
// SIG // Lm20Hjly4bgqvp+MjBqlRq4LK0yZ/ixL/Ci5IjpmF9Cq
// SIG // VoohwPOWJLTQhSZruvBvZJh5pq29XNhTaysK1nKKhUbj
// SIG // DRgG2sZ7QVY2mxU+8WoRoPdm9RjQgFVjh2hm6w55VYJc
// SIG // o+1JuHGGnpM3sGuj6mJso66W6Ln9i6vG9llbADxXIBgt
// SIG // cAOnnO+S63mhx13sfLSPS9/rXfyjIN2SOOVqUTprhZxM
// SIG // oJgIaVsG5yoZ0JWTiztrigUJKdjW2tvjcvpcSi97FVaG
// SIG // Mr9/BQmdLSrPUOHmYSDbxwaAXE4URr6uV3Giqmwwkxx+
// SIG // d8sG6VfNkfXVM3Ic4drKbuvzD+x5W7snnuge/i/yu3/p
// SIG // 5dBn67gNfKQrWQOLle0iKM36LDvHFhGv49axUGdpxY71
// SIG // edCt/4fM+H+q+aLtYfjIjWnasfRRketnV9FkEetkywO9
// SIG // SVU6RUMYLCVs0S8MLW/1QTUkoPJjWRZf2aTpLE7buzES
// SIG // xm34W24D3MsVjxuNcuzbDxWQ1hJO7uIAMSWTNW9qW6US
// SIG // Y0ABirlpiDqIuA8ZAgMBAAGjggF4MIIBdDAfBgNVHSME
// SIG // GDAWgBQaofhhGSAPw0F3RSiO0TVfBhIEVTAdBgNVHQ4E
// SIG // FgQUb02GB9gyJ54sKdLQEwOAgd0FgykwDgYDVR0PAQH/
// SIG // BAQDAgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAww
// SIG // CgYIKwYBBQUHAwgwQAYDVR0gBDkwNzA1BgwrBgEEAbIx
// SIG // AQIBAwgwJTAjBggrBgEFBQcCARYXaHR0cHM6Ly9zZWN0
// SIG // aWdvLmNvbS9DUFMwRAYDVR0fBD0wOzA5oDegNYYzaHR0
// SIG // cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGlt
// SIG // ZVN0YW1waW5nQ0EuY3JsMHQGCCsGAQUFBwEBBGgwZjA/
// SIG // BggrBgEFBQcwAoYzaHR0cDovL2NydC5zZWN0aWdvLmNv
// SIG // bS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0MCMG
// SIG // CCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNv
// SIG // bTANBgkqhkiG9w0BAQwFAAOCAgEAwGjts9jUUJvv03XL
// SIG // Dzv3JNN6N0WNLO8W+1GpLB+1JbWKn10LwhsgdI1mDzbL
// SIG // qvY2DQ9+j0tKdENlrA0q9grta23FCTjtABv45dymCkAF
// SIG // R++Eygm8Q2aDv5/t24490UFksXACLQNXWxhvHCzLHrIA
// SIG // 6LoJL1uBBDW5qWNtjgjFGNHhIaz5EgoUwBLbfiWdrB0Q
// SIG // wFqlg9IfGmZV/Jsq4uw3V47l35Yw+MCTC0MY+QJvqVGv
// SIG // uFcK8xwHaTmPN5xt15GupS5J6Ures9CMvzmQDcCBzvAq
// SIG // BzoMpi1R0nLzU8b5ve/vDGlJd58sVsTpoQg9B67FHtaE
// SIG // Ise8fUMbWDhiTtEFJYTFQvgfL/bb+quMVOxFimwSTTBa
// SIG // UuWkFwki5u9v9V+GQ9+hLb1KRpKggZYsYZd/QG/YP4w1
// SIG // WqvRxqA7hWZUgO8fGvXxm7ChJ32y5wvP9i2cWBOUqYb8
// SIG // RVKiKG1/dA9SkUl66RL4qTuwkv19kRTpW21IlPLIlu4F
// SIG // OLPF7DA/4QcgBLHYi7z9sz5v8gJTBvSg7cmacqOXXwD7
// SIG // y2PQ6M10/XXJ1DZFunsSWXLt5/J6UAB4+EOaRtjfv1TU
// SIG // XrHH0bwbg/Qr5wvoR8hTnswarPb6inVTbCCFqdW4arok
// SIG // joorCJGfNwQc9m+i3TSqkf/GFS4eQhoJKU/0xs3ikaLT
// SIG // QAyOeOMwggbsMIIE1KADAgECAhAwD2+s3WaYdHypRjan
// SIG // eC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxML
// SIG // SmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJV
// SIG // U1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJT
// SIG // QSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTAeFw0xOTA1
// SIG // MDIwMDAwMDBaFw0zODAxMTgyMzU5NTlaMH0xCzAJBgNV
// SIG // BAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0
// SIG // ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1Nl
// SIG // Y3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBS
// SIG // U0EgVGltZSBTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcN
// SIG // AQEBBQADggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8g
// SIG // kL2cgCFUk7nPoD5T77NawHbWGgSlzkeDtevEzEk0y/NF
// SIG // Zbn5p2QWJgn71TJSeS7JY8ITm7aGPwEFkmZvIavVcRB5
// SIG // h/RGKs3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/8zm7
// SIG // uaQjQqzQFcN/nqJc1zjxFrJw06PE37PFcqwuCnf8DZRS
// SIG // t/wflXMkPQEovA8NT7ORAY5unSd1VdEXOzQhe5cBlK9/
// SIG // gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31DT/b0OqF
// SIG // jIpWcdEtlEzIjDzTFKKcvSb/01Mgx2Bpm1gKVPQF5/0x
// SIG // rPnIhRfHuCkZpCkvRuPd25Ffnz82Pg4wZytGtzWvlr7a
// SIG // TGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe/arKCFiHd6QV
// SIG // 6xlV/9A5VC0m7kUaOm/N14Tw1/AoxU9kgwLU++Le8bwC
// SIG // KPRt2ieKBtKWh97oaw7wW33pdmmTIBxKlyx3GSuTlZic
// SIG // l57rjsF4VsZEJd8GEpoGLZ8DXv2DolNnyrH6jaFkyYiS
// SIG // WcuoRsDJ8qb/fVfbEnb6ikEk1Bv8cqUUotStQxykSYtB
// SIG // ORQDHin6G6UirqXDTYLQjdprt9v3GEBXc/Bxo/tKfUU2
// SIG // wfeNgvq5yQ1TgH36tjlYMu9vGFCJ10+dM70atZ2h3pVB
// SIG // eqeDAgMBAAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9a
// SIG // qitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUGqH4YRkg
// SIG // D8NBd0UojtE1XwYSBFUwDgYDVR0PAQH/BAQDAgGGMBIG
// SIG // A1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYIKwYB
// SIG // BQUHAwgwEQYDVR0gBAowCDAGBgRVHSAAMFAGA1UdHwRJ
// SIG // MEcwRaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0LmNv
// SIG // bS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9y
// SIG // aXR5LmNybDB2BggrBgEFBQcBAQRqMGgwPwYIKwYBBQUH
// SIG // MAKGM2h0dHA6Ly9jcnQudXNlcnRydXN0LmNvbS9VU0VS
// SIG // VHJ1c3RSU0FBZGRUcnVzdENBLmNydDAlBggrBgEFBQcw
// SIG // AYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkq
// SIG // hkiG9w0BAQwFAAOCAgEAbVSBpTNdFuG1U4GRdd8DejIL
// SIG // LSWEEbKw2yp9KgX1vDsn9FqguUlZkClsYcu1UNviffmf
// SIG // AO9Aw63T4uRW+VhBz/FC5RB9/7B0H4/GXAn5M17qoBwm
// SIG // WFzztBEP1dXD4rzVWHi/SHbhRGdtj7BDEA+N5Pk4Yr8T
// SIG // AcWFo0zFzLJTMJWk1vSWVgi4zVx/AZa+clJqO0I3fBZ4
// SIG // OZOTlJux3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG
// SIG // 9vPWS16toytCiiGS/qhvWiVwYoFzY16gu9jc10rTPa+D
// SIG // BjgSHSSHLeT8AtY+dwS8BDa153fLnC6NIxi5o8JHHfBd
// SIG // 1qFzVwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZOqj0J
// SIG // 9QVst9JkWtgqIsJLnfE5XkzeSD2bNJaaCV+O/fexUpHO
// SIG // P4n2HKG1qXUfcb9bQ11lPVCBbqvw0NP8srMftpmWJvQ8
// SIG // eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJvHOsXTpTD
// SIG // eGUgOw9Bqh/poUGd/rG4oGUqNODeqPk85sEwu8CgYyz8
// SIG // XBYAqNDEf+oRnR4GxqZtMl20OAkrSQeq/eww2vGnL8+3
// SIG // /frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6yDVdM3aO8Xwg
// SIG // DCp3rrWiAoa6Ke60WgCxjKvj+QrJVF3UuWp0nr1Irpgx
// SIG // ggQcMIIEGAIBATCBkTB9MQswCQYDVQQGEwJHQjEbMBkG
// SIG // A1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQH
// SIG // EwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0
// SIG // ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3Rh
// SIG // bXBpbmcgQ0ECED0aNXIwFYJjMNATcX6CQQgwDQYJYIZI
// SIG // AWUDBAIBBQCgggFbMBoGCSqGSIb3DQEJAzENBgsqhkiG
// SIG // 9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjAwMzIzMTQw
// SIG // MDAyWjAvBgkqhkiG9w0BCQQxIgQgfekAWQoTrEwUaJw9
// SIG // aG1nQkA6DtouZrMTZpPTQZE9Nv4wge0GCyqGSIb3DQEJ
// SIG // EAIMMYHdMIHaMIHXMBYEFCXIrHNOSFC3+NkTkagbkkk2
// SIG // ZZ9hMIG8BBQC1luV4oNwwVcAlfqI+SPdk3+tjzCBozCB
// SIG // jqSBizCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5l
// SIG // dyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4w
// SIG // HAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAs
// SIG // BgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlv
// SIG // biBBdXRob3JpdHkCEDAPb6zdZph0fKlGNqd4LbkwDQYJ
// SIG // KoZIhvcNAQEBBQAEggIAnf+v2MxgQI1TjfNjKzHL3vQR
// SIG // d3llX4zOuJT3ghcPlkSC3prYlLPEfBe85qMi9eaAUNRX
// SIG // u6O24aFZNUvwFaSm/yv/WuaafqWjsVnHYqfC9WkARTR9
// SIG // I7YDS5F8XmOuETI46tQv5/FHyMc01vQtV5TmU6zKQdCx
// SIG // XGQC2ti5svSySOkwBlBxr7XU5MLX/yI2FXFLPXTgWsP/
// SIG // tpmbngSQRssRchdZ7YB7Mthaame4Y7rE+2SrqFYku9TE
// SIG // vutJGVuC6HRiIv0dIpHJvWqeZ5B170ehLwL9fe2DxXZY
// SIG // H8e3JNcACNVcAzEg0VUgLMCLuPXWV2s/noBTK5kO9GAi
// SIG // PLqU3xcq4/ZwumPFbCdF2ZJKblJW5gnYlsDvmHTo/Rk1
// SIG // W4gFaD0sGoX0B2q4aTonxPJRcHuVKhd8hPlqgSNompVa
// SIG // u/an/JRDM4fuhjJlfI1rT9JDFhPik6YG4FiENbIDVNCG
// SIG // ZGr1/ghm0lDwJQR/R5tiUDk+vCZp4+H0I/ABXY7ICN8h
// SIG // T6HIES7JolD0PCdnDZ9aCR3F9OlNiXRDBrG72rFjkms+
// SIG // LJzZEABODUynrJ2qY8QLQdWASQ5uvnMRi8Ry8S3RkVyJ
// SIG // 2e22BmGOLOCl9eWJlpaw0cfV7YZTsP9if/i+jRCz4M0V
// SIG // /0vgqtgHLBtSSO3MD9khjka3Ufxv33zsjyZAoHLZ4ac=
// SIG // End signature block
