// App controller.
(function ($) {
    $.widget('adsk.appcontroller', {
        options: {
            config: {
                productFullName: '',

                ui: {
                    tabs: {
                        order: [
                            'C','I','S','F'
                        ],
                        tocEnabled: true,
                        indexEnabled: true,
                        favoritesEnabled: true,
                        searchEnabled: true
                    },

                    tools: {
                        order: ['home', 'previous', 'parent', 'following', 'separator',
                                'favorites', 'share', 'separator', 'print'],
                        home: true,
                        previous: true,
                        parent: true,
                        following: true,
                        favorites: true,
                        share: true,
                        print: true
                    },

                    responsive: true
                },

                features: {
                    tabsbarParent: 'banner',
                    banner: true,
                    header: true,
                    toolbar: true,
                    breadcrumbs: true,
                    tabsbar: true,
                    navigation: true,
                    splitter: true
                }
            },
            targetId: 'browser',
            viewControllers: {},
            toc: {},
            defaultTopicData: undefined,
            urlParams: undefined
        },

        /* Private members. */

        _view: undefined,
        _defTopicReady: new $.Deferred(),
        _defTocReady: new $.Deferred(),
        _defContextIdsReady: new $.Deferred(),

        _create: function () {
            console.log('appcontroller: _create');

            var self = this;

            this._setupPolyfills();

            $.when(this._defTopicReady, this._defTocReady, this._defContextIdsReady).done(function() {
                self._addHashchangeHandler();

                // Go to default topic page if there was no hash data specified by the user.
                if (location.hash == '') {
                    var href = self.options.defaultTopicData.attr.href;

                    if (self.options.urlParams !== undefined &&
                        self.options.urlParams.length > 0) {

                        var p = self.options.urlParams;

                        // Process contextId query.
                        if (p['contextId'] !== undefined &&
                            p['contextId'] !== '' ) {

                                // Open specified topic.
                                var id = decodeURIComponent(p['contextId']).toUpperCase(),
                                    contextIdHref = self.options.config.contextIds[id];

                                if (contextIdHref !== undefined && contextIdHref != '') {
                                    href = './'.concat(contextIdHref);
                                }
                        }
                    }

                    // Go to topic page.
                    self._handleOpentopic({}, { href: href });
                } else {
                    $(window).trigger('hashchange');
                }
            });

            this._checkUrlParameters();

            this._createView();

            this._createControllers();

            // Hide main preloader.
            this._view.hidePreloader();

            // Load context id data.
            this._loadContextIdData();

            // Load and process TOC data.
            this._loadAndProcessTOC();
        },

        _setupPolyfills: function() {
            $('html').addClass($.fn.details.support ? 'details' : 'no-details');
        },

        _checkUrlParameters: function() {
            console.log('appcontroller: _checkUrlParameters');

            this.options.urlParams = $.getUrlVars();

            if (this.options.urlParams !== undefined && this.options.urlParams.length > 0) {
                var p = this.options.urlParams,
                    features = this.options.config.features,
                    tabs = this.options.config.ui.tabs;

                if (p['panels'] !== undefined && p['panels'] == 'no') {
                    // Turn off all UI features.
                    features.banner = false;
                    features.navigation = false;
                    features.splitter = false;
                    features.breadcrumbs = false;

                    // Turn off all navigation tabs too.
                    tabs.tocEnabled = false;
                    tabs.indexEnabled = false;
                    tabs.favoritesEnabled = false;
                    tabs.searchEnabled = false;
                }
            }
        },

        _createView: function () {
            console.log('appcontroller: _createView');

            var self = this;

            $('#' + this.options.targetId).appview({
                title: this.options.config.productFullName,

                features: this.options.config.features,

                ui: this.options.config.ui,

                labels: this.options.config.labels,

                create: function (event) {
                    self._view = $(this).appview('instance');
                },

                buttonclick: function (event, id) {
                    id = id.substring('main-'.length);
                    var handlerName = '_' + id + 'ButtonclickHandler';
                    self[handlerName](event);
                },

                tabbuttonclick: function (event, id) {
                    self._view.showTab(id.substring('tab-button-'.length));
                }
            });
        },

        _createControllers: function () {
            console.log('appcontroller: _createControllers');

            var config = this.options.config,
                controllers = this.options.viewControllers,
                self = this;

            self._createTocController(self, config, controllers);
            self._createFavoritesDataController(config, controllers);
            self._createTopicController(self, config, controllers);
            self._createIndexController(self, config, controllers);
            self._createFavoritesController(self, config, controllers);
            self._createSearchController(self, config, controllers);
        },

        _createTocController: function(self, config, controllers) {
            // Create TOC view controller.
            if (config.ui.tabs.tocEnabled) {
                controllers.tocController = $.adsk.toccontroller({
                    config: config,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesDataController: function(config, controllers) {
            // Create favorites controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesController = $.adsk.favoritescontroller();
            }
        },

        _createTopicController: function(self, config, controllers) {
            // Create topic view controller.
            $.adsk.topiccontroller({
                searchOptions: config.searchOptions,

                favoritesController: controllers.favoritesController,

                create: function() {
                    controllers.topicController = $(this).topiccontroller("instance");
                },

                // Called when topic view are ready to render topic.
                ready: function() {
                    self._defTopicReady.resolve();
                },

                initgotopreviousbutton: function (event, data) {
                    self._view.initGoToPreviousButton(data.enabled);
                },

                initgotoparentbutton: function (event, data) {
                    self._view.initGoToParentButton(data.enabled);
                },

                initgotonextbutton: function (event, data) {
                    self._view.initGoToNextButton(data.enabled);
                },

                initfavoritesbutton: function (event, data) {
                    self._view.initFavoritesButton(data.enabled);
                },

                setupbreadcrumbs: function(event, data) {
                    self._view.updateBreadcrumbs(data.breadcrumbs, self.options.defaultTopicData);
                },

                setuptoc: function(event, data) {
                    if (self.options.config.ui.tabs.tocEnabled) {
                        self.options.viewControllers.tocController.toggleSelectedTocTopic(data);
                    }
                },

                opentopic: function (event, data) {
                    self._handleOpentopic(event, data);
                },

                flagfavorites: function(event, data) {
                    self._view.flagFavorites(data.added);

                    if (controllers.favoritestabController) {
                        controllers.favoritestabController.refresh();
                    }
                }
            });
        },

        _createIndexController: function(self, config, controllers) {
            // Create index view controller.
            if (config.ui.tabs.indexEnabled) {
                controllers.indexController = $.adsk.indexcontroller({
                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesController: function(self, config, controllers) {
            // Create favorites view controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesTabController = $.adsk.favoritestabcontroller({
                    favoritesController: controllers.favoritesController,

                    labels: this.options.config.labels.favorites,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createSearchController: function(self, config, controllers) {
             // Create search view controller.
            if (config.ui.tabs.searchEnabled) {
                $.adsk.searchtabcontroller({
                    config: this.options.config,

                    create: function() {
                        controllers.searchTabController = $(this).searchtabcontroller("instance");
                    },

                    ready: function() {
                        if (self.options.urlParams !== undefined &&
                            self.options.urlParams.length > 0) {

                            var p = self.options.urlParams;

                            if (p['query'] !== undefined &&
                                p['query'] !== '' &&
                                p['panels'] == undefined) {
                                    // Process search query.
                                    controllers.searchTabController.doSearch(decodeURIComponent(p['query']), self);
                                    controllers.searchTabController.closeSearchOptions();
                                    self._view.showTab('search');
                            }

                        }
                    },

                    highlight: function(event, searchOptions) {
                        if (searchOptions && searchOptions.searchQuery) {
                            controllers.topicController.highlightQuery(searchOptions);
                        }
                    },

                    dosearch: function(event) {
                        self._view.showTab('search');
                    }
                });
            }
        },

        _loadAndProcessTOC: function () {
            console.log('appcontroller: _loadAndProcessTOC');

            var controllers = this.options.viewControllers,
                tabs = this.options.config.ui.tabs;

            var tocRootNodesDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.debug('TOC root nodes data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.debug('TOC root nodes data indexed: ', toc);

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocRootNodesData(toc);
                    }
                }, this)
            });

            var tocRootNodesDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeRootData',

                event: 'register-toc-root-node-data',

                loaded: function (event, data) {
                    if (tabs.tocEnabled) {
                        tocRootNodesDataProcessor.setBookTocData(data);
                    }
                }
            });

            // Load TOC root nodes data.
            tocRootNodesDataLoader.loadTocData();

            var tocDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.log('TOC data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.log('TOC data indexed: ', toc);

                    this._setTocData(toc);

                    controllers.topicController.setTocData(toc);

                    if (tabs.searchEnabled) {
                        controllers.searchTabController.setTocData(toc);
                    }

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocData(toc.tocData);
                        controllers.tocController.renderToc();
                    }

                    if (tabs.favoritesEnabled) {
                        controllers.favoritesController.setTocData(toc);
                        controllers.favoritesTabController.refresh();
                    }
                }, this)
            });

            var tocDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeData',

                event: 'register-toc-data',

                loaded: function (event, data) {
                    tocDataProcessor.setBookTocData(data);
                }
            });

            // Load TOC data.
            tocDataLoader.loadTocData();
        },

        _loadContextIdData: function() {
            console.log('appcontroller: _loadContextIdData');

            var self = this;

            // Load context id data.
            var contextIdDataLoader = $.adsk.contextiddataloader({
                config: self.options.config,

                loaded: function (event, data) {
                    $.extend(self.options.config.contextIds, data.bookData);

                    if (data._loadedData == data._totalData) {
                        self._defContextIdsReady.resolve();
                    }
                }
            });

            contextIdDataLoader.loadContextIdData();
        },

        _setTocData: function (data) {
            console.log('appcontroller: _setTocData');

            this.options.toc = data;

            // Set default topic data.
            this.options.defaultTopicData = this._getDefaultTopicData();

            // Show breadcrumbs.
            this._view.updateBreadcrumbs([], this.options.defaultTopicData);

            this._defTocReady.resolve();
        },


        _getDefaultTopicData: function () {
            console.log('appcontroller: _getDefaultTopicData, ');

            var rootTopicData,
               config = this.options.config;

            if (config.defaultPage !== '') {
                rootTopicData = {
                    // TODO: default topic title through config.
                    data: "Home",
                    attr: {
                        href: config.defaultPage,
                        // TODO: description through config.
                        desc: ""
                    }
                }
            } else if (this.options.toc && this.options.toc.tocData) {
                // Try to use first topic data from root and component books TOC data as a default.
                $.each(this.options.toc.tocData, function(bookName, bookTocData) {
                    if (bookTocData[0]) {
                        rootTopicData = bookTocData[0];
                        // break
                        return false;
                    }
                });
            }

            return rootTopicData;
        },


        _handleOpentopic: function (event, data) {
            console.log('appcontroller: _handleOpentopic');

            // Change location hash to open provided topic.
            if (data === undefined || data.href === undefined || data.href == '') {
                return;
            }

            this._setHash('#!/url=' + data.href);
        },

        _addHashchangeHandler: function () {
            console.log('appcontroller: _addHashchangeHandler');

            var self = this;

            // Listen to hash change.
            if ('onhashchange' in window) {
                $(window).on('hashchange', function (event) {
                    self._handleHashchange(event);
                });
            } else {
                setInterval($.proxy(self._handleHashchange, this), 400);
            }
        },

        _handleHashchange: function (event, data) {
            console.log('appcontroller: _handleHashchange');

            var newHash = location.hash;

            this._processHash(newHash);
        },

        _processHash: function (hash) {
            console.log('appcontroller: _processHash, ', hash);

            if (!hash || hash == '') {
                // Load default topic.
                this._handleOpentopic({}, {
                    href: this.options.config.defaultPage
                });
            }

            // If hash starts with hashbang, crop it off.
            var url = this._fetchUrlFromHash(hash);

            if (!url || url == '') {
                // Do nothing.
                return;
            }

            this._processUrl(url);
        },

        _fetchUrlFromHash: function (hash) {
            console.log('appcontroller: _fetchUrlFromHash');

            var url = '';

            if (hash.indexOf('#!') == 0) {
                url = hash.substring(2);
            } else if (hash[0] == '#') {
                url = hash.substring(1);
            } else {
                url = hash;
            }

            return url;
        },

        _processUrl: function (url) {
            console.log('appcontroller: _processUrl: ', url);

            if (url.indexOf('/url=') == 0) {
                // Resolve internal path.
                this._resolveUrl(url);
            }
        },

        _resolveUrl: function (url) {
            console.log('appcontroller: _resolveUrl: ', url);

            var href = url.substring('/url='.length);

            if (!href || href == '') {
                return;
            }

            // Load topic content.
            var data = {
                href: href
            };

            this.options.viewControllers.topicController.openTopic(data);
        },

        _setHash: function (hash) {
            console.log('appcontroller: _setHash: ', hash);

            if (!hash || hash == '') {
                return;
            }

            if (hash[0] != '#') {
                return;
            }

            location.hash = hash;
        },

        _homeButtonclickHandler: function (event) {
            console.log('appcontroller: _homeButtonclickHandler');

            // Open default topic.
            this._handleOpentopic({}, {
                href: this.options.config.defaultPage
            });
        },

        _previousButtonclickHandler: function (event) {
            console.log('appcontroller: _previousButtonclickHandler');

            this.options.viewControllers.topicController.openPreviousTopic();
        },

        _parentButtonclickHandler: function (event) {
            console.log('appcontroller: _parentButtonclickHandler');

            this.options.viewControllers.topicController.openParentTopic();
        },

        _followingButtonclickHandler: function (event) {
            console.log('appcontroller: _followingButtonclickHandler');

            this.options.viewControllers.topicController.openFollowingTopic();
        },

        _favoritesButtonclickHandler: function (event) {
            console.log('appcontroller: _favoritesButtonclickHandler');

            var controllers = this.options.viewControllers;

            controllers.topicController.addOrRemoveTopicFromFavorites();
            controllers.favoritesTabController.refresh();
        },

        _shareButtonclickHandler: function (event) {
            console.log('appcontroller: _shareButtonclickHandler');

            var controller = this.options.viewControllers.shareController;

            if (!controller) {
                controller = this.options.viewControllers.shareController = $.adsk.sharecontroller({
                    //opentopic: $.proxy(this._handleOpentopic, this)
                });
            }

            controller.refresh();
            controller.showShareMenu();
        },

        _printButtonclickHandler: function (event) {
            console.log('appcontroller: _printButtonclickHandler');

            window.print();
        }

        /* Public members. */

    });
})(jQuery);
// SIG // Begin signature block
// SIG // MIIgdgYJKoZIhvcNAQcCoIIgZzCCIGMCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // CeKZWGQstNLFioAG7EkYGAv+MHJ7iMdtv/dsrf08MRig
// SIG // ggoaMIIEyzCCA7OgAwIBAgIQCQja65wGHi65un4mqmKH
// SIG // ijANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xOTA1MTQwMDAw
// SIG // MDBaFw0yMDA1MTMyMzU5NTlaMHwxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIDApDYWxpZm9ybmlhMRMwEQYDVQQHDApT
// SIG // YW4gUmFmYWVsMRcwFQYDVQQKDA5BdXRvZGVzaywgSW5j
// SIG // LjERMA8GA1UECwwIU2VjdXJpdHkxFzAVBgNVBAMMDkF1
// SIG // dG9kZXNrLCBJbmMuMIIBIjANBgkqhkiG9w0BAQEFAAOC
// SIG // AQ8AMIIBCgKCAQEAkVKRoPYsoxPdlabspcuG8HYDfcWQ
// SIG // 9qyPOT2punwpVdjt2slktcpSTlydrGy8ijcbl5/k8AOJ
// SIG // afKMQvow0fNjIoNB5nsb8j2xFJPbBhHl8GT4P6DZ1OWK
// SIG // ioiUXgGHKjctUws49J4elntqvBYhKCbuPqG772QCoYFU
// SIG // P7qcFbh3PbyNB6cnKMn2vNGRqiEtV2aGg+/MiL3QPsvT
// SIG // kTWhVuTLIvwZ4kK5C5ACiQzlGOlE+C7gWuFoxtvy6gUa
// SIG // A8JAq/xLjYKwFevQwGncg3s1cuiGhx0P7Rvn6oCd2cem
// SIG // FmbK3I3rMRt+jc9wzAj3oHxvCo4f5Lfva+OpC1dETUVV
// SIG // sk0VGwIDAQABo4IBPjCCATowCQYDVR0TBAIwADAOBgNV
// SIG // HQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
// SIG // YQYDVR0gBFowWDBWBgZngQwBBAEwTDAjBggrBgEFBQcC
// SIG // ARYXaHR0cHM6Ly9kLnN5bWNiLmNvbS9jcHMwJQYIKwYB
// SIG // BQUHAgIwGQwXaHR0cHM6Ly9kLnN5bWNiLmNvbS9ycGEw
// SIG // HwYDVR0jBBgwFoAU1MAGIknrOUvdk+JcobhHdglyA1gw
// SIG // KwYDVR0fBCQwIjAgoB6gHIYaaHR0cDovL3JiLnN5bWNi
// SIG // LmNvbS9yYi5jcmwwVwYIKwYBBQUHAQEESzBJMB8GCCsG
// SIG // AQUFBzABhhNodHRwOi8vcmIuc3ltY2QuY29tMCYGCCsG
// SIG // AQUFBzAChhpodHRwOi8vcmIuc3ltY2IuY29tL3JiLmNy
// SIG // dDANBgkqhkiG9w0BAQsFAAOCAQEA0Q2+iVSEfT4xGq3A
// SIG // OADVb0rT5Z8E+PPb1x6hKknSzuaZKk/acDz9mdnFajaU
// SIG // WgIKiFJlsqJnuAM1w4lqKaEWsz3DoxvGIEwexj71Ud6a
// SIG // KlBJNcR9DdkXUZwMhbw9d78rjJAIX6tOwfSludZfdq8b
// SIG // YEkuIzQi9uYW9vHVPSCzhnUBKjwc9qTo9yBc7Fcn4FAT
// SIG // 90DzWM2hS9oc3p0pprYI2I/O4th90BEmwWkVqVgfaR0z
// SIG // VyIEmBbqB/IWLfjUBZJ/8sbkwWuWlYVRrfPxsj1e9qvb
// SIG // wE3Y28jWyGbqoUG39GhXk3uT8R8i3+T6M0c4rMfdifBC
// SIG // pceCaFTJzSye7LoeVzCCBUcwggQvoAMCAQICEHwbNTVK
// SIG // 59t050FfEWnKa6gwDQYJKoZIhvcNAQELBQAwgb0xCzAJ
// SIG // BgNVBAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5j
// SIG // LjEfMB0GA1UECxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29y
// SIG // azE6MDgGA1UECxMxKGMpIDIwMDggVmVyaVNpZ24sIElu
// SIG // Yy4gLSBGb3IgYXV0aG9yaXplZCB1c2Ugb25seTE4MDYG
// SIG // A1UEAxMvVmVyaVNpZ24gVW5pdmVyc2FsIFJvb3QgQ2Vy
// SIG // dGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMTQwNzIyMDAw
// SIG // MDAwWhcNMjQwNzIxMjM1OTU5WjCBhDELMAkGA1UEBhMC
// SIG // VVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9u
// SIG // MR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3Jr
// SIG // MTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1
// SIG // NiBDb2RlIFNpZ25pbmcgQ0EgLSBHMjCCASIwDQYJKoZI
// SIG // hvcNAQEBBQADggEPADCCAQoCggEBANeVQ9Tc32euOftS
// SIG // pLYmMQRw6beOWyq6N2k1lY+7wDDnhthzu9/r0XY/ilaO
// SIG // 6y1L8FcYTrGNpTPTC3Uj1Wp5J92j0/cOh2W13q0c8fU1
// SIG // tCJRryKhwV1LkH/AWU6rnXmpAtceSbE7TYf+wnirv+9S
// SIG // rpyvCNk55ZpRPmlfMBBOcWNsWOHwIDMbD3S+W8sS4duM
// SIG // xICUcrv2RZqewSUL+6McntimCXBx7MBHTI99w94Zzj7u
// SIG // BHKOF9P/8LIFMhlM07Acn/6leCBCcEGwJoxvAMg6ABFB
// SIG // ekGwp4qRBKCZePR3tPNgKuZsUAS3FGD/DVH0qIuE/iHa
// SIG // XF599Sl5T7BEdG9tcv8CAwEAAaOCAXgwggF0MC4GCCsG
// SIG // AQUFBwEBBCIwIDAeBggrBgEFBQcwAYYSaHR0cDovL3Mu
// SIG // c3ltY2QuY29tMBIGA1UdEwEB/wQIMAYBAf8CAQAwZgYD
// SIG // VR0gBF8wXTBbBgtghkgBhvhFAQcXAzBMMCMGCCsGAQUF
// SIG // BwIBFhdodHRwczovL2Quc3ltY2IuY29tL2NwczAlBggr
// SIG // BgEFBQcCAjAZGhdodHRwczovL2Quc3ltY2IuY29tL3Jw
// SIG // YTA2BgNVHR8ELzAtMCugKaAnhiVodHRwOi8vcy5zeW1j
// SIG // Yi5jb20vdW5pdmVyc2FsLXJvb3QuY3JsMBMGA1UdJQQM
// SIG // MAoGCCsGAQUFBwMDMA4GA1UdDwEB/wQEAwIBBjApBgNV
// SIG // HREEIjAgpB4wHDEaMBgGA1UEAxMRU3ltYW50ZWNQS0kt
// SIG // MS03MjQwHQYDVR0OBBYEFNTABiJJ6zlL3ZPiXKG4R3YJ
// SIG // cgNYMB8GA1UdIwQYMBaAFLZ3+mlIR59TEtXC6gcydgfR
// SIG // lwcZMA0GCSqGSIb3DQEBCwUAA4IBAQB/68qn6ot2Qus+
// SIG // jiBUMOO3udz6SD4Wxw9FlRDNJ4ajZvMC7XH4qsJVl5Fw
// SIG // g/lSflJpPMnx4JRGgBi7odSkVqbzHQCR1YbzSIfgy8Q0
// SIG // aCBetMv5Be2cr3BTJ7noPn5RoGlxi9xR7YA6JTKfRK9u
// SIG // QyjTIXW7l9iLi4z+qQRGBIX3FZxLEY3ELBf+1W5/muJW
// SIG // kvGWs60t+fTf2omZzrI4RMD3R3vKJbn6Kmgzm1By3qif
// SIG // 1M0sCzS9izB4QOCNjicbkG8avggVgV3rL+JR51EeyXgp
// SIG // 5x5lvzjvAUoBCSQOFsQUecFBNzTQPZFSlJ3haO8I8OJp
// SIG // nGdukAsak3HUJgLDwFojMYIVtDCCFbACAQEwgZkwgYQx
// SIG // CzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRlYyBD
// SIG // b3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1
// SIG // c3QgTmV0d29yazE1MDMGA1UEAxMsU3ltYW50ZWMgQ2xh
// SIG // c3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENBIC0gRzIC
// SIG // EAkI2uucBh4uubp+Jqpih4owDQYJYIZIAWUDBAIBBQCg
// SIG // fDAQBgorBgEEAYI3AgEMMQIwADAZBgkqhkiG9w0BCQMx
// SIG // DAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgELMQ4wDAYK
// SIG // KwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgSJgMnBK0
// SIG // wlPsb2490VBg/yoWcE1hzQpZ66rRtc7VVqYwDQYJKoZI
// SIG // hvcNAQEBBQAEggEALzEhk9U7SQkN4A36luJNpFJhZWun
// SIG // j41yxaLxbu+E+i6BSflRZ6U5VsS4OZUxjtm34YWgMV7t
// SIG // D8Ha77RqNOEzHjvcKuYHMhHg9/kPybBFjNdgtL06qKxu
// SIG // cg0dqArIKEgfmNsaONYTl399Gfu1e7/3srtEC3g6KK2D
// SIG // cz4eB/3D3JV2MoaB9fu3MTCJtGgEMJnt+pGHcDxWKiYc
// SIG // C/qK+DCn8EE5s5x4OyVqFz3j5fjEl40WD+o0BqiWxrOd
// SIG // p9AoZh6W3vTANqh4Fu/MJGO32HGIH9NsRvkD4o15JYT3
// SIG // d8lKSPuYBbdnyyddxkF8dNsxuHFJt3QX65RBtbWPcHPe
// SIG // KL6uiaGCE20wghNpBgorBgEEAYI3AwMBMYITWTCCE1UG
// SIG // CSqGSIb3DQEHAqCCE0YwghNCAgEDMQ8wDQYJYIZIAWUD
// SIG // BAIBBQAwggEMBgsqhkiG9w0BCRABBKCB/ASB+TCB9gIB
// SIG // AQYKKwYBBAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCAY
// SIG // URRnmsB6Y553qOku2aMHh5et3s00RJqwmVOo9SDmFwIU
// SIG // MYI1F/wuzEf2u9qJD8LBQGInT0AYDzIwMjAwMzIzMTQw
// SIG // MDQ0WqCBiqSBhzCBhDELMAkGA1UEBhMCR0IxGzAZBgNV
// SIG // BAgMEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBwwH
// SIG // U2FsZm9yZDEYMBYGA1UECgwPU2VjdGlnbyBMaW1pdGVk
// SIG // MSwwKgYDVQQDDCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1w
// SIG // aW5nIFNpZ25lciAjMaCCDfowggcGMIIE7qADAgECAhA9
// SIG // GjVyMBWCYzDQE3F+gkEIMA0GCSqGSIb3DQEBDAUAMH0x
// SIG // CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1h
// SIG // bmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNV
// SIG // BAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2Vj
// SIG // dGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTAeFw0xOTA1
// SIG // MDIwMDAwMDBaFw0zMDA4MDEyMzU5NTlaMIGEMQswCQYD
// SIG // VQQGEwJHQjEbMBkGA1UECAwSR3JlYXRlciBNYW5jaGVz
// SIG // dGVyMRAwDgYDVQQHDAdTYWxmb3JkMRgwFgYDVQQKDA9T
// SIG // ZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28g
// SIG // UlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICMxMIICIjAN
// SIG // BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAy1FQ/1b+
// SIG // /HhjcAGTWp4Y9DtT9gevIWz1og99HXAthHRIi5yKlQU9
// SIG // WYT5kYB5USzZirfBC5q6CorNZk8DiwG7MMqrvdvATxJe
// SIG // /ArM4kWwATiKu03n1BxUmO05WM9bwi9FmDEK+TU4uDEu
// SIG // bbQeOXLhuCq+n4yMGqVGrgsrTJn+LEv8KLkiOmYX0KpW
// SIG // iiHA85YktNCFJmu68G9kmHmmrb1c2FNrKwrWcoqFRuMN
// SIG // GAbaxntBVjabFT7xahGg92b1GNCAVWOHaGbrDnlVglyj
// SIG // 7Um4cYaekzewa6PqYmyjrpbouf2Lq8b2WVsAPFcgGC1w
// SIG // A6ec75LreaHHXex8tI9L3+td/KMg3ZI45WpROmuFnEyg
// SIG // mAhpWwbnKhnQlZOLO2uKBQkp2Nba2+Ny+lxKL3sVVoYy
// SIG // v38FCZ0tKs9Q4eZhINvHBoBcThRGvq5XcaKqbDCTHH53
// SIG // ywbpV82R9dUzchzh2spu6/MP7Hlbuyee6B7+L/K7f+nl
// SIG // 0GfruA18pCtZA4uV7SIozfosO8cWEa/j1rFQZ2nFjvV5
// SIG // 0K3/h8z4f6r5ou1h+MiNadqx9FGR62dX0WQR62TLA71J
// SIG // VTpFQxgsJWzRLwwtb/VBNSSg8mNZFl/ZpOksTtu7MRLG
// SIG // bfhbbgPcyxWPG41y7NsPFZDWEk7u4gAxJZM1b2pbpRJj
// SIG // QAGKuWmIOoi4DxkCAwEAAaOCAXgwggF0MB8GA1UdIwQY
// SIG // MBaAFBqh+GEZIA/DQXdFKI7RNV8GEgRVMB0GA1UdDgQW
// SIG // BBRvTYYH2DInniwp0tATA4CB3QWDKTAOBgNVHQ8BAf8E
// SIG // BAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAK
// SIG // BggrBgEFBQcDCDBABgNVHSAEOTA3MDUGDCsGAQQBsjEB
// SIG // AgEDCDAlMCMGCCsGAQUFBwIBFhdodHRwczovL3NlY3Rp
// SIG // Z28uY29tL0NQUzBEBgNVHR8EPTA7MDmgN6A1hjNodHRw
// SIG // Oi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29SU0FUaW1l
// SIG // U3RhbXBpbmdDQS5jcmwwdAYIKwYBBQUHAQEEaDBmMD8G
// SIG // CCsGAQUFBzAChjNodHRwOi8vY3J0LnNlY3RpZ28uY29t
// SIG // L1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcnQwIwYI
// SIG // KwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29t
// SIG // MA0GCSqGSIb3DQEBDAUAA4ICAQDAaO2z2NRQm+/TdcsP
// SIG // O/ck03o3RY0s7xb7UaksH7UltYqfXQvCGyB0jWYPNsuq
// SIG // 9jYND36PS0p0Q2WsDSr2Cu1rbcUJOO0AG/jl3KYKQAVH
// SIG // 74TKCbxDZoO/n+3bjj3RQWSxcAItA1dbGG8cLMsesgDo
// SIG // ugkvW4EENbmpY22OCMUY0eEhrPkSChTAEtt+JZ2sHRDA
// SIG // WqWD0h8aZlX8myri7DdXjuXfljD4wJMLQxj5Am+pUa+4
// SIG // VwrzHAdpOY83nG3Xka6lLknpSt6z0Iy/OZANwIHO8CoH
// SIG // OgymLVHScvNTxvm97+8MaUl3nyxWxOmhCD0HrsUe1oQi
// SIG // x7x9QxtYOGJO0QUlhMVC+B8v9tv6q4xU7EWKbBJNMFpS
// SIG // 5aQXCSLm72/1X4ZD36EtvUpGkqCBlixhl39Ab9g/jDVa
// SIG // q9HGoDuFZlSA7x8a9fGbsKEnfbLnC8/2LZxYE5SphvxF
// SIG // UqIobX90D1KRSXrpEvipO7CS/X2RFOlbbUiU8siW7gU4
// SIG // s8XsMD/hByAEsdiLvP2zPm/yAlMG9KDtyZpyo5dfAPvL
// SIG // Y9DozXT9dcnUNkW6exJZcu3n8npQAHj4Q5pG2N+/VNRe
// SIG // scfRvBuD9CvnC+hHyFOezBqs9vqKdVNsIIWp1bhquiSO
// SIG // iisIkZ83BBz2b6LdNKqR/8YVLh5CGgkpT/TGzeKRotNA
// SIG // DI544zCCBuwwggTUoAMCAQICEDAPb6zdZph0fKlGNqd4
// SIG // LbkwDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtK
// SIG // ZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVT
// SIG // VCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNB
// SIG // IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE5MDUw
// SIG // MjAwMDAwMFoXDTM4MDExODIzNTk1OVowfTELMAkGA1UE
// SIG // BhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
// SIG // cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2Vj
// SIG // dGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJT
// SIG // QSBUaW1lIFN0YW1waW5nIENBMIICIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAg8AMIICCgKCAgEAyBsBr9ksfoiZfQGYPyCQ
// SIG // vZyAIVSTuc+gPlPvs1rAdtYaBKXOR4O168TMSTTL80Vl
// SIG // ufmnZBYmCfvVMlJ5LsljwhObtoY/AQWSZm8hq9VxEHmH
// SIG // 9EYqzcRaydvXXUlNclYP3MnjU5g6Kh78zlhJ07/zObu5
// SIG // pCNCrNAVw3+eolzXOPEWsnDTo8Tfs8VyrC4Kd/wNlFK3
// SIG // /B+VcyQ9ASi8Dw1Ps5EBjm6dJ3VV0Rc7NCF7lwGUr3+A
// SIG // z9ERCleEyX9W4L1GnIK+lJ2/tCCwYH64TfUNP9vQ6oWM
// SIG // ilZx0S2UTMiMPNMUopy9Jv/TUyDHYGmbWApU9AXn/TGs
// SIG // +ciFF8e4KRmkKS9G493bkV+fPzY+DjBnK0a3Na+WvtpM
// SIG // YMyou58NFNQYxDCYdIIhz2JWtSFzEh79qsoIWId3pBXr
// SIG // GVX/0DlULSbuRRo6b83XhPDX8CjFT2SDAtT74t7xvAIo
// SIG // 9G3aJ4oG0paH3uhrDvBbfel2aZMgHEqXLHcZK5OVmJyX
// SIG // nuuOwXhWxkQl3wYSmgYtnwNe/YOiU2fKsfqNoWTJiJJZ
// SIG // y6hGwMnypv99V9sSdvqKQSTUG/xypRSi1K1DHKRJi0E5
// SIG // FAMeKfobpSKupcNNgtCN2mu32/cYQFdz8HGj+0p9RTbB
// SIG // 942C+rnJDVOAffq2OVgy728YUInXT50zvRq1naHelUF6
// SIG // p4MCAwEAAaOCAVowggFWMB8GA1UdIwQYMBaAFFN5v1qq
// SIG // K0rPVIDh2JvAnfKyA2bLMB0GA1UdDgQWBBQaofhhGSAP
// SIG // w0F3RSiO0TVfBhIEVTAOBgNVHQ8BAf8EBAMCAYYwEgYD
// SIG // VR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEF
// SIG // BQcDCDARBgNVHSAECjAIMAYGBFUdIAAwUAYDVR0fBEkw
// SIG // RzBFoEOgQYY/aHR0cDovL2NybC51c2VydHJ1c3QuY29t
// SIG // L1VTRVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRob3Jp
// SIG // dHkuY3JsMHYGCCsGAQUFBwEBBGowaDA/BggrBgEFBQcw
// SIG // AoYzaHR0cDovL2NydC51c2VydHJ1c3QuY29tL1VTRVJU
// SIG // cnVzdFJTQUFkZFRydXN0Q0EuY3J0MCUGCCsGAQUFBzAB
// SIG // hhlodHRwOi8vb2NzcC51c2VydHJ1c3QuY29tMA0GCSqG
// SIG // SIb3DQEBDAUAA4ICAQBtVIGlM10W4bVTgZF13wN6Mgst
// SIG // JYQRsrDbKn0qBfW8Oyf0WqC5SVmQKWxhy7VQ2+J9+Z8A
// SIG // 70DDrdPi5Fb5WEHP8ULlEH3/sHQfj8ZcCfkzXuqgHCZY
// SIG // XPO0EQ/V1cPivNVYeL9IduFEZ22PsEMQD43k+ThivxMB
// SIG // xYWjTMXMslMwlaTW9JZWCLjNXH8Blr5yUmo7Qjd8Fng5
// SIG // k5OUm7Hcsm1BbWfNyW+QPX9FcsEbI9bCVYRm5LPFZgb2
// SIG // 89ZLXq2jK0KKIZL+qG9aJXBigXNjXqC72NzXStM9r4MG
// SIG // OBIdJIct5PwC1j53BLwENrXnd8ucLo0jGLmjwkcd8F3W
// SIG // oXNXBWiap8k3ZR2+6rzYQoNDBaWLpgn/0aGUpk6qPQn1
// SIG // BWy30mRa2Coiwkud8TleTN5IPZs0lpoJX47997FSkc4/
// SIG // ifYcobWpdR9xv1tDXWU9UIFuq/DQ0/yysx+2mZYm9Dx5
// SIG // i1xkzM3uJ5rloMAMcofBbk1a0x7q8ETmMm8c6xdOlMN4
// SIG // ZSA7D0GqH+mhQZ3+sbigZSo04N6o+TzmwTC7wKBjLPxc
// SIG // FgCo0MR/6hGdHgbGpm0yXbQ4CStJB6r97DDa8acvz7f9
// SIG // +tCjhNknnvsBZne5VhDhIG7GrrH5trrINV0zdo7xfCAM
// SIG // KneutaIChrop7rRaALGMq+P5CslUXdS5anSevUiumDGC
// SIG // BBwwggQYAgEBMIGRMH0xCzAJBgNVBAYTAkdCMRswGQYD
// SIG // VQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcT
// SIG // B1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
// SIG // ZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFt
// SIG // cGluZyBDQQIQPRo1cjAVgmMw0BNxfoJBCDANBglghkgB
// SIG // ZQMEAgEFAKCCAVswGgYJKoZIhvcNAQkDMQ0GCyqGSIb3
// SIG // DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0yMDAzMjMxNDAw
// SIG // NDRaMC8GCSqGSIb3DQEJBDEiBCB4otInDTlD981tbdU3
// SIG // F5NT6ly8F6csyVFLRABCQbI73TCB7QYLKoZIhvcNAQkQ
// SIG // Agwxgd0wgdowgdcwFgQUJcisc05IULf42RORqBuSSTZl
// SIG // n2EwgbwEFALWW5Xig3DBVwCV+oj5I92Tf62PMIGjMIGO
// SIG // pIGLMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
// SIG // IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAc
// SIG // BgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwG
// SIG // A1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9u
// SIG // IEF1dGhvcml0eQIQMA9vrN1mmHR8qUY2p3gtuTANBgkq
// SIG // hkiG9w0BAQEFAASCAgBcnRbn7H/X9oTzxxwnVBrGyVaK
// SIG // xVSDWKFcxq+szjhcLrj+nWRJT8yGiJpF7hQy1MHvHm+O
// SIG // s0VTN9hPXVnaAzZcXQtXcJ1eNPzVmw0DtIGxl61QTQZv
// SIG // jGO0B2tbr5DhZV4+LTWREpdMFGK05cXlIylHFdETnaYR
// SIG // 3TatNWVT77xAN69lRAj1cW3FhA3667umQg8bhf4GgzTg
// SIG // KCAJpmXEPer5XPBVyBCScnQ+q7Rh0T9YLlG2Ta80ISC+
// SIG // SlcsnF2nMLCF6qihopcHZ/puiokvJrgcixk9iB0g9Fla
// SIG // 2BVexFS4dyVzX8k+fKGg9AW7j+3atuYC+Hbhl9K95IYN
// SIG // wgapdxR4nBZhhV2YzcIoyNdqbUVn6hXImeh7mTM+HLXq
// SIG // z2TquONJMYh0wufkx7kAQAkREpczCqybLf8RkdIHMYdD
// SIG // p3hq1VdvUzcdozCexYVqeLf9EjlCjZA7txbpDw8o+R5f
// SIG // dnD4IKbVcQpChddEWWcpoMfQUDfQ0J7V+2DlH8hxHrQE
// SIG // Z5uecnx+k5aNKTMlWfIRXs5wCsBEYy0NdchVcPPfkR8D
// SIG // O0l/N6H1HH5Q5AA+crr22mEtP/FWBiESoDawAxPLJIuM
// SIG // h+qbH+XOlgQ/VB0JBsD/KUdiKqCQ/dVzysrLTchnbqQ9
// SIG // RCpU0HAJ/6Glpx3fo+Wr54UT5XBryYb9gu53BfG2PQ==
// SIG // End signature block
