// App controller.
(function ($) {
    $.widget('adsk.appcontroller', {
        options: {
            config: {
                productFullName: '',

                ui: {
                    tabs: {
                        order: [
                            'C','I','S','F'
                        ],
                        tocEnabled: true,
                        indexEnabled: true,
                        favoritesEnabled: true,
                        searchEnabled: true
                    },

                    tools: {
                        order: ['home', 'previous', 'parent', 'following', 'separator',
                                'favorites', 'share', 'separator', 'print'],
                        home: true,
                        previous: true,
                        parent: true,
                        following: true,
                        favorites: true,
                        share: true,
                        print: true
                    },

                    responsive: true
                },

                features: {
                    tabsbarParent: 'banner',
                    banner: true,
                    header: true,
                    toolbar: true,
                    breadcrumbs: true,
                    tabsbar: true,
                    navigation: true,
                    splitter: true
                }
            },
            targetId: 'browser',
            viewControllers: {},
            toc: {},
            defaultTopicData: undefined,
            urlParams: undefined
        },

        /* Private members. */

        _view: undefined,
        _defTopicReady: new $.Deferred(),
        _defTocReady: new $.Deferred(),
        _defContextIdsReady: new $.Deferred(),

        _create: function () {
            console.log('appcontroller: _create');

            var self = this;

            this._setupPolyfills();

            $.when(this._defTopicReady, this._defTocReady, this._defContextIdsReady).done(function() {
                self._addHashchangeHandler();

                // Go to default topic page if there was no hash data specified by the user.
                if (location.hash == '') {
                    var href = self.options.defaultTopicData.attr.href;

                    if (self.options.urlParams !== undefined &&
                        self.options.urlParams.length > 0) {

                        var p = self.options.urlParams;

                        // Process contextId query.
                        if (p['contextId'] !== undefined &&
                            p['contextId'] !== '' ) {

                                // Open specified topic.
                                var id = decodeURIComponent(p['contextId']).toUpperCase(),
                                    contextIdHref = self.options.config.contextIds[id];

                                if (contextIdHref !== undefined && contextIdHref != '') {
                                    href = './'.concat(contextIdHref);
                                }
                        }
                    }

                    // Go to topic page.
                    self._handleOpentopic({}, { href: href });
                } else {
                    $(window).trigger('hashchange');
                }
            });

            this._checkUrlParameters();

            this._createView();

            this._createControllers();

            // Hide main preloader.
            this._view.hidePreloader();

            // Load context id data.
            this._loadContextIdData();

            // Load and process TOC data.
            this._loadAndProcessTOC();
        },

        _setupPolyfills: function() {
            $('html').addClass($.fn.details.support ? 'details' : 'no-details');
        },

        _checkUrlParameters: function() {
            console.log('appcontroller: _checkUrlParameters');

            this.options.urlParams = $.getUrlVars();

            if (this.options.urlParams !== undefined && this.options.urlParams.length > 0) {
                var p = this.options.urlParams,
                    features = this.options.config.features,
                    tabs = this.options.config.ui.tabs;

                if (p['panels'] !== undefined && p['panels'] == 'no') {
                    // Turn off all UI features.
                    features.banner = false;
                    features.navigation = false;
                    features.splitter = false;
                    features.breadcrumbs = false;

                    // Turn off all navigation tabs too.
                    tabs.tocEnabled = false;
                    tabs.indexEnabled = false;
                    tabs.favoritesEnabled = false;
                    tabs.searchEnabled = false;
                }
            }
        },

        _createView: function () {
            console.log('appcontroller: _createView');

            var self = this;

            $('#' + this.options.targetId).appview({
                title: this.options.config.productFullName,

                features: this.options.config.features,

                ui: this.options.config.ui,

                labels: this.options.config.labels,

                create: function (event) {
                    self._view = $(this).appview('instance');
                },

                buttonclick: function (event, id) {
                    id = id.substring('main-'.length);
                    var handlerName = '_' + id + 'ButtonclickHandler';
                    self[handlerName](event);
                },

                tabbuttonclick: function (event, id) {
                    self._view.showTab(id.substring('tab-button-'.length));
                }
            });
        },

        _createControllers: function () {
            console.log('appcontroller: _createControllers');

            var config = this.options.config,
                controllers = this.options.viewControllers,
                self = this;

            self._createTocController(self, config, controllers);
            self._createFavoritesDataController(config, controllers);
            self._createTopicController(self, config, controllers);
            self._createIndexController(self, config, controllers);
            self._createFavoritesController(self, config, controllers);
            self._createSearchController(self, config, controllers);
        },

        _createTocController: function(self, config, controllers) {
            // Create TOC view controller.
            if (config.ui.tabs.tocEnabled) {
                controllers.tocController = $.adsk.toccontroller({
                    config: config,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesDataController: function(config, controllers) {
            // Create favorites controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesController = $.adsk.favoritescontroller();
            }
        },

        _createTopicController: function(self, config, controllers) {
            // Create topic view controller.
            $.adsk.topiccontroller({
                searchOptions: config.searchOptions,

                favoritesController: controllers.favoritesController,

                create: function() {
                    controllers.topicController = $(this).topiccontroller("instance");
                },

                // Called when topic view are ready to render topic.
                ready: function() {
                    self._defTopicReady.resolve();
                },

                initgotopreviousbutton: function (event, data) {
                    self._view.initGoToPreviousButton(data.enabled);
                },

                initgotoparentbutton: function (event, data) {
                    self._view.initGoToParentButton(data.enabled);
                },

                initgotonextbutton: function (event, data) {
                    self._view.initGoToNextButton(data.enabled);
                },

                initfavoritesbutton: function (event, data) {
                    self._view.initFavoritesButton(data.enabled);
                },

                setupbreadcrumbs: function(event, data) {
                    self._view.updateBreadcrumbs(data.breadcrumbs, self.options.defaultTopicData);
                },

                setuptoc: function(event, data) {
                    if (self.options.config.ui.tabs.tocEnabled) {
                        self.options.viewControllers.tocController.toggleSelectedTocTopic(data);
                    }
                },

                opentopic: function (event, data) {
                    self._handleOpentopic(event, data);
                },

                flagfavorites: function(event, data) {
                    self._view.flagFavorites(data.added);

                    if (controllers.favoritestabController) {
                        controllers.favoritestabController.refresh();
                    }
                }
            });
        },

        _createIndexController: function(self, config, controllers) {
            // Create index view controller.
            if (config.ui.tabs.indexEnabled) {
                controllers.indexController = $.adsk.indexcontroller({
                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesController: function(self, config, controllers) {
            // Create favorites view controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesTabController = $.adsk.favoritestabcontroller({
                    favoritesController: controllers.favoritesController,

                    labels: this.options.config.labels.favorites,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createSearchController: function(self, config, controllers) {
             // Create search view controller.
            if (config.ui.tabs.searchEnabled) {
                $.adsk.searchtabcontroller({
                    config: this.options.config,

                    create: function() {
                        controllers.searchTabController = $(this).searchtabcontroller("instance");
                    },

                    ready: function() {
                        if (self.options.urlParams !== undefined &&
                            self.options.urlParams.length > 0) {

                            var p = self.options.urlParams;

                            if (p['query'] !== undefined &&
                                p['query'] !== '' &&
                                p['panels'] == undefined) {
                                    // Process search query.
                                    controllers.searchTabController.doSearch(decodeURIComponent(p['query']), self);
                                    controllers.searchTabController.closeSearchOptions();
                                    self._view.showTab('search');
                            }

                        }
                    },

                    highlight: function(event, searchOptions) {
                        if (searchOptions && searchOptions.searchQuery) {
                            controllers.topicController.highlightQuery(searchOptions);
                        }
                    },

                    dosearch: function(event) {
                        self._view.showTab('search');
                    }
                });
            }
        },

        _loadAndProcessTOC: function () {
            console.log('appcontroller: _loadAndProcessTOC');

            var controllers = this.options.viewControllers,
                tabs = this.options.config.ui.tabs;

            var tocRootNodesDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.debug('TOC root nodes data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.debug('TOC root nodes data indexed: ', toc);

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocRootNodesData(toc);
                    }
                }, this)
            });

            var tocRootNodesDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeRootData',

                event: 'register-toc-root-node-data',

                loaded: function (event, data) {
                    if (tabs.tocEnabled) {
                        tocRootNodesDataProcessor.setBookTocData(data);
                    }
                }
            });

            // Load TOC root nodes data.
            tocRootNodesDataLoader.loadTocData();

            var tocDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.log('TOC data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.log('TOC data indexed: ', toc);

                    this._setTocData(toc);

                    controllers.topicController.setTocData(toc);

                    if (tabs.searchEnabled) {
                        controllers.searchTabController.setTocData(toc);
                    }

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocData(toc.tocData);
                        controllers.tocController.renderToc();
                    }

                    if (tabs.favoritesEnabled) {
                        controllers.favoritesController.setTocData(toc);
                        controllers.favoritesTabController.refresh();
                    }
                }, this)
            });

            var tocDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeData',

                event: 'register-toc-data',

                loaded: function (event, data) {
                    tocDataProcessor.setBookTocData(data);
                }
            });

            // Load TOC data.
            tocDataLoader.loadTocData();
        },

        _loadContextIdData: function() {
            console.log('appcontroller: _loadContextIdData');

            var self = this;

            // Load context id data.
            var contextIdDataLoader = $.adsk.contextiddataloader({
                config: self.options.config,

                loaded: function (event, data) {
                    $.extend(self.options.config.contextIds, data.bookData);

                    if (data._loadedData == data._totalData) {
                        self._defContextIdsReady.resolve();
                    }
                }
            });

            contextIdDataLoader.loadContextIdData();
        },

        _setTocData: function (data) {
            console.log('appcontroller: _setTocData');

            this.options.toc = data;

            // Set default topic data.
            this.options.defaultTopicData = this._getDefaultTopicData();

            // Show breadcrumbs.
            this._view.updateBreadcrumbs([], this.options.defaultTopicData);

            this._defTocReady.resolve();
        },


        _getDefaultTopicData: function () {
            console.log('appcontroller: _getDefaultTopicData, ');

            var rootTopicData,
               config = this.options.config;

            if (config.defaultPage !== '') {
                rootTopicData = {
                    // TODO: default topic title through config.
                    data: "Home",
                    attr: {
                        href: config.defaultPage,
                        // TODO: description through config.
                        desc: ""
                    }
                }
            } else if (this.options.toc && this.options.toc.tocData) {
                // Try to use first topic data from root and component books TOC data as a default.
                $.each(this.options.toc.tocData, function(bookName, bookTocData) {
                    if (bookTocData[0]) {
                        rootTopicData = bookTocData[0];
                        // break
                        return false;
                    }
                });
            }

            return rootTopicData;
        },


        _handleOpentopic: function (event, data) {
            console.log('appcontroller: _handleOpentopic');

            // Change location hash to open provided topic.
            if (data === undefined || data.href === undefined || data.href == '') {
                return;
            }

            this._setHash('#!/url=' + data.href);
        },

        _addHashchangeHandler: function () {
            console.log('appcontroller: _addHashchangeHandler');

            var self = this;

            // Listen to hash change.
            if ('onhashchange' in window) {
                $(window).on('hashchange', function (event) {
                    self._handleHashchange(event);
                });
            } else {
                setInterval($.proxy(self._handleHashchange, this), 400);
            }
        },

        _handleHashchange: function (event, data) {
            console.log('appcontroller: _handleHashchange');

            var newHash = location.hash;

            this._processHash(newHash);
        },

        _processHash: function (hash) {
            console.log('appcontroller: _processHash, ', hash);

            if (!hash || hash == '') {
                // Load default topic.
                this._handleOpentopic({}, {
                    href: this.options.config.defaultPage
                });
            }

            // If hash starts with hashbang, crop it off.
            var url = this._fetchUrlFromHash(hash);

            if (!url || url == '') {
                // Do nothing.
                return;
            }

            this._processUrl(url);
        },

        _fetchUrlFromHash: function (hash) {
            console.log('appcontroller: _fetchUrlFromHash');

            var url = '';

            if (hash.indexOf('#!') == 0) {
                url = hash.substring(2);
            } else if (hash[0] == '#') {
                url = hash.substring(1);
            } else {
                url = hash;
            }

            return url;
        },

        _processUrl: function (url) {
            console.log('appcontroller: _processUrl: ', url);

            if (url.indexOf('/url=') == 0) {
                // Resolve internal path.
                this._resolveUrl(url);
            }
        },

        _resolveUrl: function (url) {
            console.log('appcontroller: _resolveUrl: ', url);

            var href = url.substring('/url='.length);

            if (!href || href == '') {
                return;
            }

            // Load topic content.
            var data = {
                href: href
            };

            this.options.viewControllers.topicController.openTopic(data);
        },

        _setHash: function (hash) {
            console.log('appcontroller: _setHash: ', hash);

            if (!hash || hash == '') {
                return;
            }

            if (hash[0] != '#') {
                return;
            }

            location.hash = hash;
        },

        _homeButtonclickHandler: function (event) {
            console.log('appcontroller: _homeButtonclickHandler');

            // Open default topic.
            this._handleOpentopic({}, {
                href: this.options.config.defaultPage
            });
        },

        _previousButtonclickHandler: function (event) {
            console.log('appcontroller: _previousButtonclickHandler');

            this.options.viewControllers.topicController.openPreviousTopic();
        },

        _parentButtonclickHandler: function (event) {
            console.log('appcontroller: _parentButtonclickHandler');

            this.options.viewControllers.topicController.openParentTopic();
        },

        _followingButtonclickHandler: function (event) {
            console.log('appcontroller: _followingButtonclickHandler');

            this.options.viewControllers.topicController.openFollowingTopic();
        },

        _favoritesButtonclickHandler: function (event) {
            console.log('appcontroller: _favoritesButtonclickHandler');

            var controllers = this.options.viewControllers;

            controllers.topicController.addOrRemoveTopicFromFavorites();
            controllers.favoritesTabController.refresh();
        },

        _shareButtonclickHandler: function (event) {
            console.log('appcontroller: _shareButtonclickHandler');

            var controller = this.options.viewControllers.shareController;

            if (!controller) {
                controller = this.options.viewControllers.shareController = $.adsk.sharecontroller({
                    //opentopic: $.proxy(this._handleOpentopic, this)
                });
            }

            controller.refresh();
            controller.showShareMenu();
        },

        _printButtonclickHandler: function (event) {
            console.log('appcontroller: _printButtonclickHandler');

            window.print();
        }

        /* Public members. */

    });
})(jQuery);
// SIG // Begin signature block
// SIG // MIIbRgYJKoZIhvcNAQcCoIIbNzCCGzMCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // CeKZWGQstNLFioAG7EkYGAv+MHJ7iMdtv/dsrf08MRig
// SIG // ggoaMIIEyzCCA7OgAwIBAgIQCQja65wGHi65un4mqmKH
// SIG // ijANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xOTA1MTQwMDAw
// SIG // MDBaFw0yMDA1MTMyMzU5NTlaMHwxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIDApDYWxpZm9ybmlhMRMwEQYDVQQHDApT
// SIG // YW4gUmFmYWVsMRcwFQYDVQQKDA5BdXRvZGVzaywgSW5j
// SIG // LjERMA8GA1UECwwIU2VjdXJpdHkxFzAVBgNVBAMMDkF1
// SIG // dG9kZXNrLCBJbmMuMIIBIjANBgkqhkiG9w0BAQEFAAOC
// SIG // AQ8AMIIBCgKCAQEAkVKRoPYsoxPdlabspcuG8HYDfcWQ
// SIG // 9qyPOT2punwpVdjt2slktcpSTlydrGy8ijcbl5/k8AOJ
// SIG // afKMQvow0fNjIoNB5nsb8j2xFJPbBhHl8GT4P6DZ1OWK
// SIG // ioiUXgGHKjctUws49J4elntqvBYhKCbuPqG772QCoYFU
// SIG // P7qcFbh3PbyNB6cnKMn2vNGRqiEtV2aGg+/MiL3QPsvT
// SIG // kTWhVuTLIvwZ4kK5C5ACiQzlGOlE+C7gWuFoxtvy6gUa
// SIG // A8JAq/xLjYKwFevQwGncg3s1cuiGhx0P7Rvn6oCd2cem
// SIG // FmbK3I3rMRt+jc9wzAj3oHxvCo4f5Lfva+OpC1dETUVV
// SIG // sk0VGwIDAQABo4IBPjCCATowCQYDVR0TBAIwADAOBgNV
// SIG // HQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
// SIG // YQYDVR0gBFowWDBWBgZngQwBBAEwTDAjBggrBgEFBQcC
// SIG // ARYXaHR0cHM6Ly9kLnN5bWNiLmNvbS9jcHMwJQYIKwYB
// SIG // BQUHAgIwGQwXaHR0cHM6Ly9kLnN5bWNiLmNvbS9ycGEw
// SIG // HwYDVR0jBBgwFoAU1MAGIknrOUvdk+JcobhHdglyA1gw
// SIG // KwYDVR0fBCQwIjAgoB6gHIYaaHR0cDovL3JiLnN5bWNi
// SIG // LmNvbS9yYi5jcmwwVwYIKwYBBQUHAQEESzBJMB8GCCsG
// SIG // AQUFBzABhhNodHRwOi8vcmIuc3ltY2QuY29tMCYGCCsG
// SIG // AQUFBzAChhpodHRwOi8vcmIuc3ltY2IuY29tL3JiLmNy
// SIG // dDANBgkqhkiG9w0BAQsFAAOCAQEA0Q2+iVSEfT4xGq3A
// SIG // OADVb0rT5Z8E+PPb1x6hKknSzuaZKk/acDz9mdnFajaU
// SIG // WgIKiFJlsqJnuAM1w4lqKaEWsz3DoxvGIEwexj71Ud6a
// SIG // KlBJNcR9DdkXUZwMhbw9d78rjJAIX6tOwfSludZfdq8b
// SIG // YEkuIzQi9uYW9vHVPSCzhnUBKjwc9qTo9yBc7Fcn4FAT
// SIG // 90DzWM2hS9oc3p0pprYI2I/O4th90BEmwWkVqVgfaR0z
// SIG // VyIEmBbqB/IWLfjUBZJ/8sbkwWuWlYVRrfPxsj1e9qvb
// SIG // wE3Y28jWyGbqoUG39GhXk3uT8R8i3+T6M0c4rMfdifBC
// SIG // pceCaFTJzSye7LoeVzCCBUcwggQvoAMCAQICEHwbNTVK
// SIG // 59t050FfEWnKa6gwDQYJKoZIhvcNAQELBQAwgb0xCzAJ
// SIG // BgNVBAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5j
// SIG // LjEfMB0GA1UECxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29y
// SIG // azE6MDgGA1UECxMxKGMpIDIwMDggVmVyaVNpZ24sIElu
// SIG // Yy4gLSBGb3IgYXV0aG9yaXplZCB1c2Ugb25seTE4MDYG
// SIG // A1UEAxMvVmVyaVNpZ24gVW5pdmVyc2FsIFJvb3QgQ2Vy
// SIG // dGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMTQwNzIyMDAw
// SIG // MDAwWhcNMjQwNzIxMjM1OTU5WjCBhDELMAkGA1UEBhMC
// SIG // VVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9u
// SIG // MR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3Jr
// SIG // MTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1
// SIG // NiBDb2RlIFNpZ25pbmcgQ0EgLSBHMjCCASIwDQYJKoZI
// SIG // hvcNAQEBBQADggEPADCCAQoCggEBANeVQ9Tc32euOftS
// SIG // pLYmMQRw6beOWyq6N2k1lY+7wDDnhthzu9/r0XY/ilaO
// SIG // 6y1L8FcYTrGNpTPTC3Uj1Wp5J92j0/cOh2W13q0c8fU1
// SIG // tCJRryKhwV1LkH/AWU6rnXmpAtceSbE7TYf+wnirv+9S
// SIG // rpyvCNk55ZpRPmlfMBBOcWNsWOHwIDMbD3S+W8sS4duM
// SIG // xICUcrv2RZqewSUL+6McntimCXBx7MBHTI99w94Zzj7u
// SIG // BHKOF9P/8LIFMhlM07Acn/6leCBCcEGwJoxvAMg6ABFB
// SIG // ekGwp4qRBKCZePR3tPNgKuZsUAS3FGD/DVH0qIuE/iHa
// SIG // XF599Sl5T7BEdG9tcv8CAwEAAaOCAXgwggF0MC4GCCsG
// SIG // AQUFBwEBBCIwIDAeBggrBgEFBQcwAYYSaHR0cDovL3Mu
// SIG // c3ltY2QuY29tMBIGA1UdEwEB/wQIMAYBAf8CAQAwZgYD
// SIG // VR0gBF8wXTBbBgtghkgBhvhFAQcXAzBMMCMGCCsGAQUF
// SIG // BwIBFhdodHRwczovL2Quc3ltY2IuY29tL2NwczAlBggr
// SIG // BgEFBQcCAjAZGhdodHRwczovL2Quc3ltY2IuY29tL3Jw
// SIG // YTA2BgNVHR8ELzAtMCugKaAnhiVodHRwOi8vcy5zeW1j
// SIG // Yi5jb20vdW5pdmVyc2FsLXJvb3QuY3JsMBMGA1UdJQQM
// SIG // MAoGCCsGAQUFBwMDMA4GA1UdDwEB/wQEAwIBBjApBgNV
// SIG // HREEIjAgpB4wHDEaMBgGA1UEAxMRU3ltYW50ZWNQS0kt
// SIG // MS03MjQwHQYDVR0OBBYEFNTABiJJ6zlL3ZPiXKG4R3YJ
// SIG // cgNYMB8GA1UdIwQYMBaAFLZ3+mlIR59TEtXC6gcydgfR
// SIG // lwcZMA0GCSqGSIb3DQEBCwUAA4IBAQB/68qn6ot2Qus+
// SIG // jiBUMOO3udz6SD4Wxw9FlRDNJ4ajZvMC7XH4qsJVl5Fw
// SIG // g/lSflJpPMnx4JRGgBi7odSkVqbzHQCR1YbzSIfgy8Q0
// SIG // aCBetMv5Be2cr3BTJ7noPn5RoGlxi9xR7YA6JTKfRK9u
// SIG // QyjTIXW7l9iLi4z+qQRGBIX3FZxLEY3ELBf+1W5/muJW
// SIG // kvGWs60t+fTf2omZzrI4RMD3R3vKJbn6Kmgzm1By3qif
// SIG // 1M0sCzS9izB4QOCNjicbkG8avggVgV3rL+JR51EeyXgp
// SIG // 5x5lvzjvAUoBCSQOFsQUecFBNzTQPZFSlJ3haO8I8OJp
// SIG // nGdukAsak3HUJgLDwFojMYIQhDCCEIACAQEwgZkwgYQx
// SIG // CzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRlYyBD
// SIG // b3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1
// SIG // c3QgTmV0d29yazE1MDMGA1UEAxMsU3ltYW50ZWMgQ2xh
// SIG // c3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENBIC0gRzIC
// SIG // EAkI2uucBh4uubp+Jqpih4owDQYJYIZIAWUDBAIBBQCg
// SIG // fDAQBgorBgEEAYI3AgEMMQIwADAZBgkqhkiG9w0BCQMx
// SIG // DAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgELMQ4wDAYK
// SIG // KwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgSJgMnBK0
// SIG // wlPsb2490VBg/yoWcE1hzQpZ66rRtc7VVqYwDQYJKoZI
// SIG // hvcNAQEBBQAEggEALzEhk9U7SQkN4A36luJNpFJhZWun
// SIG // j41yxaLxbu+E+i6BSflRZ6U5VsS4OZUxjtm34YWgMV7t
// SIG // D8Ha77RqNOEzHjvcKuYHMhHg9/kPybBFjNdgtL06qKxu
// SIG // cg0dqArIKEgfmNsaONYTl399Gfu1e7/3srtEC3g6KK2D
// SIG // cz4eB/3D3JV2MoaB9fu3MTCJtGgEMJnt+pGHcDxWKiYc
// SIG // C/qK+DCn8EE5s5x4OyVqFz3j5fjEl40WD+o0BqiWxrOd
// SIG // p9AoZh6W3vTANqh4Fu/MJGO32HGIH9NsRvkD4o15JYT3
// SIG // d8lKSPuYBbdnyyddxkF8dNsxuHFJt3QX65RBtbWPcHPe
// SIG // KL6uiaGCDj0wgg45BgorBgEEAYI3AwMBMYIOKTCCDiUG
// SIG // CSqGSIb3DQEHAqCCDhYwgg4SAgEDMQ0wCwYJYIZIAWUD
// SIG // BAIBMIIBDwYLKoZIhvcNAQkQAQSggf8EgfwwgfkCAQEG
// SIG // C2CGSAGG+EUBBxcDMDEwDQYJYIZIAWUDBAIBBQAEIBhR
// SIG // FGeawHpjnneo6S7ZoweHl63ezTREmrCZU6j1IOYXAhUA
// SIG // mSm16zbiBpcw1//Yxb5GeRYFrxcYDzIwMjAwMjA3MTYx
// SIG // NzA3WjADAgEeoIGGpIGDMIGAMQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAd
// SIG // BgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxMTAv
// SIG // BgNVBAMTKFN5bWFudGVjIFNIQTI1NiBUaW1lU3RhbXBp
// SIG // bmcgU2lnbmVyIC0gRzOgggqLMIIFODCCBCCgAwIBAgIQ
// SIG // ewWx1EloUUT3yYnSnBmdEjANBgkqhkiG9w0BAQsFADCB
// SIG // vTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWdu
// SIG // LCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVzdCBO
// SIG // ZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwOCBWZXJpU2ln
// SIG // biwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5
// SIG // MTgwNgYDVQQDEy9WZXJpU2lnbiBVbml2ZXJzYWwgUm9v
// SIG // dCBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTAeFw0xNjAx
// SIG // MTIwMDAwMDBaFw0zMTAxMTEyMzU5NTlaMHcxCzAJBgNV
// SIG // BAYTAlVTMR0wGwYDVQQKExRTeW1hbnRlYyBDb3Jwb3Jh
// SIG // dGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0
// SIG // d29yazEoMCYGA1UEAxMfU3ltYW50ZWMgU0hBMjU2IFRp
// SIG // bWVTdGFtcGluZyBDQTCCASIwDQYJKoZIhvcNAQEBBQAD
// SIG // ggEPADCCAQoCggEBALtZnVlVT52Mcl0agaLrVfOwAa08
// SIG // cawyjwVrhponADKXak3JZBRLKbvC2Sm5Luxjs+HPPwtW
// SIG // kPhiG37rpgfi3n9ebUA41JEG50F8eRzLy60bv9iVkfPw
// SIG // 7mz4rZY5Ln/BJ7h4OcWEpe3tr4eOzo3HberSmLU6Hx45
// SIG // ncP0mqj0hOHE0XxxxgYptD/kgw0mw3sIPk35CrczSf/K
// SIG // O9T1sptL4YiZGvXA6TMU1t/HgNuR7v68kldyd/TNqMz+
// SIG // CfWTN76ViGrF3PSxS9TO6AmRX7WEeTWKeKwZMo8jwTJB
// SIG // G1kOqT6xzPnWK++32OTVHW0ROpL2k8mc40juu1MO1DaX
// SIG // hnjFoTcCAwEAAaOCAXcwggFzMA4GA1UdDwEB/wQEAwIB
// SIG // BjASBgNVHRMBAf8ECDAGAQH/AgEAMGYGA1UdIARfMF0w
// SIG // WwYLYIZIAYb4RQEHFwMwTDAjBggrBgEFBQcCARYXaHR0
// SIG // cHM6Ly9kLnN5bWNiLmNvbS9jcHMwJQYIKwYBBQUHAgIw
// SIG // GRoXaHR0cHM6Ly9kLnN5bWNiLmNvbS9ycGEwLgYIKwYB
// SIG // BQUHAQEEIjAgMB4GCCsGAQUFBzABhhJodHRwOi8vcy5z
// SIG // eW1jZC5jb20wNgYDVR0fBC8wLTAroCmgJ4YlaHR0cDov
// SIG // L3Muc3ltY2IuY29tL3VuaXZlcnNhbC1yb290LmNybDAT
// SIG // BgNVHSUEDDAKBggrBgEFBQcDCDAoBgNVHREEITAfpB0w
// SIG // GzEZMBcGA1UEAxMQVGltZVN0YW1wLTIwNDgtMzAdBgNV
// SIG // HQ4EFgQUr2PWyqNOhXLgp7xB8ymiOH+AdWIwHwYDVR0j
// SIG // BBgwFoAUtnf6aUhHn1MS1cLqBzJ2B9GXBxkwDQYJKoZI
// SIG // hvcNAQELBQADggEBAHXqsC3VNBlcMkX+DuHUT6Z4wW/X
// SIG // 6t3cT/OhyIGI96ePFeZAKa3mXfSi2VZkhHEwKt0eYRdm
// SIG // IFYGmBmNXXHy+Je8Cf0ckUfJ4uiNA/vMkC/WCmxOM+zW
// SIG // tJPITJBjSDlAIcTd1m6JmDy1mJfoqQa3CcmPU1dBkC/h
// SIG // Hk1O3MoQeGxCbvC2xfhhXFL1TvZrjfdKer7zzf0D19n2
// SIG // A6gP41P3CnXsxnUuqmaFBJm3+AZX4cYO9uiv2uybGB+q
// SIG // ueM6AL/OipTLAduexzi7D1Kr0eOUA2AKTaD+J20UMvw/
// SIG // l0Dhv5mJ2+Q5FL3a5NPD6itas5VYVQR9x5rsIwONhSrS
// SIG // /66pYYEwggVLMIIEM6ADAgECAhB71OWvuswHP6EBIwQi
// SIG // QU0SMA0GCSqGSIb3DQEBCwUAMHcxCzAJBgNVBAYTAlVT
// SIG // MR0wGwYDVQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjEf
// SIG // MB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazEo
// SIG // MCYGA1UEAxMfU3ltYW50ZWMgU0hBMjU2IFRpbWVTdGFt
// SIG // cGluZyBDQTAeFw0xNzEyMjMwMDAwMDBaFw0yOTAzMjIy
// SIG // MzU5NTlaMIGAMQswCQYDVQQGEwJVUzEdMBsGA1UEChMU
// SIG // U3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNVBAsTFlN5
// SIG // bWFudGVjIFRydXN0IE5ldHdvcmsxMTAvBgNVBAMTKFN5
// SIG // bWFudGVjIFNIQTI1NiBUaW1lU3RhbXBpbmcgU2lnbmVy
// SIG // IC0gRzMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
// SIG // AoIBAQCvDoqq+Ny/aXtUF3FHCb2NPIH4dBV3Z5Cc/d5O
// SIG // Ap5LdvblNj5l1SQgbTD53R2D6T8nSjNObRaK5I1AjSKq
// SIG // vqcLG9IHtjy1GiQo+BtyUT3ICYgmCDr5+kMjdUdwDLNf
// SIG // W48IHXJIV2VNrwI8QPf03TI4kz/lLKbzWSPLgN4TTfkQ
// SIG // yaoKGGxVYVfR8QIsxLWr8mwj0p8NDxlsrYViaf1OhcGK
// SIG // UjGrW9jJdFLjV2wiv1V/b8oGqz9KtyJ2ZezsNvKWlYEm
// SIG // LP27mKoBONOvJUCbCVPwKVeFWF7qhUhBIYfl3rTTJrJ7
// SIG // QFNYeY5SMQZNlANFxM48A+y3API6IsW0b+XvsIqbAgMB
// SIG // AAGjggHHMIIBwzAMBgNVHRMBAf8EAjAAMGYGA1UdIARf
// SIG // MF0wWwYLYIZIAYb4RQEHFwMwTDAjBggrBgEFBQcCARYX
// SIG // aHR0cHM6Ly9kLnN5bWNiLmNvbS9jcHMwJQYIKwYBBQUH
// SIG // AgIwGRoXaHR0cHM6Ly9kLnN5bWNiLmNvbS9ycGEwQAYD
// SIG // VR0fBDkwNzA1oDOgMYYvaHR0cDovL3RzLWNybC53cy5z
// SIG // eW1hbnRlYy5jb20vc2hhMjU2LXRzcy1jYS5jcmwwFgYD
// SIG // VR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/BAQD
// SIG // AgeAMHcGCCsGAQUFBwEBBGswaTAqBggrBgEFBQcwAYYe
// SIG // aHR0cDovL3RzLW9jc3Aud3Muc3ltYW50ZWMuY29tMDsG
// SIG // CCsGAQUFBzAChi9odHRwOi8vdHMtYWlhLndzLnN5bWFu
// SIG // dGVjLmNvbS9zaGEyNTYtdHNzLWNhLmNlcjAoBgNVHREE
// SIG // ITAfpB0wGzEZMBcGA1UEAxMQVGltZVN0YW1wLTIwNDgt
// SIG // NjAdBgNVHQ4EFgQUpRMBqZ+FzBtuFh5fOzGqeTYAex0w
// SIG // HwYDVR0jBBgwFoAUr2PWyqNOhXLgp7xB8ymiOH+AdWIw
// SIG // DQYJKoZIhvcNAQELBQADggEBAEaer/C4ol+imUjPqCdL
// SIG // Ic2yuaZycGMv41UpezlGTud+ZQZYi7xXipINCNgQujYk
// SIG // +gp7+zvTYr9KlBXmgtuKVG3/KP5nz3E/5jMJ2aJZEPQe
// SIG // Sv5lzN7Ua+NSKXUASiulzMub6KlN97QXWZJBw7c/hub2
// SIG // wH9EPEZcF1rjpDvVaSbVIX3hgGd+Yqy3Ti4VmuWcI69b
// SIG // EepxqUH5DXk4qaENz7Sx2j6aescixXTN30cJhsT8kSWy
// SIG // G5bphQjo3ep0YG5gpVZ6DchEWNzm+UgUnuW/3gC9d7GY
// SIG // FHIUJN/HESwfAD/DSxTGZxzMHgajkF9cVIs+4zNbgg/F
// SIG // t4YCTnGf6WZFP3YxggJaMIICVgIBATCBizB3MQswCQYD
// SIG // VQQGEwJVUzEdMBsGA1UEChMUU3ltYW50ZWMgQ29ycG9y
// SIG // YXRpb24xHzAdBgNVBAsTFlN5bWFudGVjIFRydXN0IE5l
// SIG // dHdvcmsxKDAmBgNVBAMTH1N5bWFudGVjIFNIQTI1NiBU
// SIG // aW1lU3RhbXBpbmcgQ0ECEHvU5a+6zAc/oQEjBCJBTRIw
// SIG // CwYJYIZIAWUDBAIBoIGkMBoGCSqGSIb3DQEJAzENBgsq
// SIG // hkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjAwMjA3
// SIG // MTYxNzA3WjAvBgkqhkiG9w0BCQQxIgQgRenRSyvl0fhb
// SIG // hSm9kyHnP1/ktOX4VaxXn8Ahv4hX8A8wNwYLKoZIhvcN
// SIG // AQkQAi8xKDAmMCQwIgQgxHTOdgB9AjlODaXk3nwUxoD5
// SIG // 4oIBPP72U+9dtx/fYfgwCwYJKoZIhvcNAQEBBIIBAAIH
// SIG // L2MQCyxDy006EphKJMVHyhAqbEcaF2RMqEqCCbMVa71q
// SIG // 1xkYO8Owg8HvPXdWKR+YT9q+3weAlPtd4N2C238DLqPY
// SIG // citjmB52HCjze+5ICNfkmEO04QMYbXzLZRUW6fvQvQH+
// SIG // UoQTZXpVh7Jowrtd3JuttBFGl1MSG2XvYT/mBhxY7fxj
// SIG // H7+++yKC0BxHkO2vXK7Le8Gpi5ZPgpZiaJvhiLy6UZMr
// SIG // +4pWWF1sFsEUJwrOwe8JSiO7yz37m6doEfRkapT55a2w
// SIG // ZQIkn3lhevpZWMxdVMnPR4gc3qiuKFN0+ms0kzB0zzaA
// SIG // TD9Rsprp9Gpsc8R6UEhP9JHjuBd/Y8k=
// SIG // End signature block
