function [c,info]=wpfbt(f,wt,varargin)
%WPFBT   Wavelet Packet FilterBank Tree
%   Usage:  c=wpfbt(f,wt);
%           [c,info]=wpfbt(...);
%
%   Input parameters:
%         f   : Input data.
%         wt  : Wavelet Filterbank tree definition.
%
%   Output parameters:
%         c    : Coefficients stored in a cell-array.
%         info : Transform parameters struct.
%
%   `c=wpfbt(f,wt)` returns wavelet packet coefficients *c* obtained by
%   applying a wavelet filterbank tree defined by *wt* to the input data
%   *f*. 
%    
%   `[c,info]=wpfbt(f,wt)` additionally returns struct. `info` containing 
%   transform parameters. It can be conviniently used for the inverse 
%   transform |iwpfbt| e.g. `fhat = iwpfbt(c,info)`. It is also required 
%   by the |plotwavelets| function.
%
%   In contrast to |wfbt|, the cell array *c* contain every intermediate 
%   output of each node in the tree. `c{jj}` are ordered according to
%   nodes taken in the breadth-first order.
%
%   If *f* is row/column vector, the coefficient vectors `c{jj}` are
%   columns. If *f* is a matrix, the transformation is applied to each of
%   column of the matrix.
%
%   Scaling of intermediate outputs:
%   --------------------------------
%
%   The following flags control scaling of intermediate outputs and
%   therefore the energy relations between coefficient subbands. An 
%   intermediate output is an output of a node which is further used as an
%   input to a descendant node.
%
%      'intsqrt'
%               Each intermediate output is scaled by `1/sqrt(2)`.
%               If the filterbank in each node is orthonormal, the overall
%               undecimated transform is a tight frame.
%               This is the default.
%
%      'intnoscale'
%               No scaling of intermediate results is used. This is
%               necessaty for the |wpbest| function to correctly work with
%               the cost measures.
%
%      'intscale'
%               Each intermediate output is scaled by `1/2`.
%
%   If 'intnoscale' is used, 'intscale' must be used in |iwpfbt| (and vice
%   versa) in order to obtain a perfect reconstruction.
%
%   Please see help for |wfbt| description of possible formats of *wt* and
%   of the additional flags defining boundary handling.
%
%   Examples:
%   ---------
%
%   A simple example of calling the |wpfbt| function using the "full
%   decomposition" wavelet tree:::
%
%     f = gspi;
%     J = 6;
%     [c,info] = wpfbt(f,{'sym10',J,'full'});
%     plotwavelets(c,info,44100,'dynrange',90);
%
%   See also: wfbt, iwpfbt, wfbtinit, plotwavelets, wpbest

% AUTHOR: Zdenek Prusa

complainif_notenoughargs(nargin,2,'WPFBT');

definput.import = {'fwt','wfbtcommon'};
definput.flags.interscaling = {'intsqrt', 'intscale', 'intnoscale'};
[flags]=ltfatarghelper({},definput,varargin);

% Initialize the wavelet tree structure
wt = wfbtinit(wt,flags.forder);

%% ----- step 1 : Verify f and determine its length -------
[f,Ls]=comp_sigreshape_pre(f,upper(mfilename),0);
if(Ls<2)
   error('%s: Input signal seems not to be a vector of length > 1.',...
       upper(mfilename));
end

% Determine next legal input data length.
L = wfbtlength(Ls,wt,flags.ext);

% Pad with zeros if the safe length L differ from the Ls.
if(Ls~=L)
   f=postpad(f,L);
end

%% ----- step 3 : Run computation
wtPath = nodeBForder(0,wt);
rangeLoc = nodesLocOutRange(wtPath,wt);
c = comp_wpfbt(f,wt.nodes(wtPath),rangeLoc,flags.ext,flags.interscaling);

%% ----- Optional : Fill the info struct. -----
if nargout>1
   info.fname = 'wpfbt';
   info.wt = wt;
   info.ext = flags.ext;
   info.Lc = cellfun(@(cEl) size(cEl,1),c);
   info.Ls = Ls;
   info.fOrder = flags.forder;
   info.isPacked = 0;
   info.interscaling = flags.interscaling;
end
