!     *************************************************************************************************************************
!                                                   FLOW SIMULATOR MYELIB
!                                                   Copyright: (c) 2017 
!                                                   Author:    Shane Cline (GE Power)        
!                                                   Date:      11-Feb-2017 -> 26-Feb-2017
!     *************************************************************************************************************************
!     The myelib.f source file allows engineers to develop custom two-arm flow elements inside FLOW SIMULATOR.
!     The myelib feature is useful when the governing flow equation is LR and cannot be shared globally.
!     *************************************************************************************************************************


!     To use the Fortran UDE a UDE component and UDE element is required in the database (.flo) .
!     The GUI can be used to create the the component and element. 
!     Examples are shown here if the user needs to debug problems:

!********************************************************************************
!ELEMENT_NUMBER:       3.          UDE                                          5
!--------------------------------------------------------------------------------
!
!
!
!
!
!--------------------------------------------------------------------------------
!   I    ELTYPE     CHNUMUP     CHNUMDWN       NLU         NED        FUTURE
!   3    914809           4            2      4.0000      4.0000           0
!--------------------------------------------------------------------------------
!  NUMBER_SCALARS   NUMBER_ARRAYS
!               2               2
!--------------------------------------------------------------------------------
!            AREA              CD
!       0.2500000       0.0000000
!--------------------------------------------------------------------------------
!         TBL1_PR              20
!       0.0000000       0.0000000       0.0000000       0.0000000       0.0000000
!       0.0000000       0.0000000       0.0000000       0.0000000       0.0000000
!       0.0000000       0.0000000       0.0000000       0.0000000       0.0000000
!       0.0000000       0.0000000       0.0000000       0.0000000       0.0000000
!--------------------------------------------------------------------------------
!          TBL_CD              20
!       0.0000000       0.0000000       0.0000000       0.0000000       0.0000000
!       0.0000000       0.0000000       0.0000000       0.0000000       0.0000000
!       0.0000000       0.0000000       0.0000000       0.0000000       0.0000000
!       0.0000000       0.0000000       0.0000000       0.0000000       0.0000000
!       
! 

! Note the Element's ELTYPE matches the component's MYELIB_NUMBER
! The component's LANGUAGE is set to FORTRAN.
      
!********************************************************************************
!MYELIB_NUMBER:  914809.          MYELIB_COMPONENT                             5
!--------------------------------------------------------------------------------
!
!
!
!
!
!--------------------------------------------------------------------------------
!   I    CPTYPE     ELEMENT_NAME        SUBTYPE     LANGUAGE    FUTURE     FUTURE
!   4         9     ELIB_DEMO           RESISTIVE   FORTRAN     NONE       NONE    
!--------------------------------------------------------------------------------
!  #PROPERTY_NAME# #DATA_TYPE#     #SIZE#
!   AREA            REAL            NONE           
!   CD              REAL            NONE           
!   TBL1_PR         REAL_ARRAY      20             
!   TBL_CD          REAL_ARRAY      20             
!--------------------------------------------------------------------------------
!  #FLOW_FUNCTION#
!
!--------------------------------------------------------------------------------
!  #ERROR_TRAP#
       

      
!     *************************************************************************************************************************
!     *************************************************************************************************************************
!     The UDE_INPUT_ERROR_TRAP subroutine is where you set up input error and warning checks
!
!     Main solver subroutines typically called from this subroutine:
!
!     GET_UDE_TYPE returns the UDE element name for the UDE element number. It returns a character string 31 characters long. 
!      UDE_TYPE  = GET_UDE_TYPE(
!     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file      
!     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
!      
!      
!     ERROR_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN checks if UDE element user input is within the specified range.
!      WARN_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN is similar to ERROR_IF... but it only writes a warning message; the run will continue.
!     Both subroutines have the same arguments.
!      CALL ERROR_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN(
!     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
!     &         'AREA',             ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file
!     &         '(',                ! CHARACTER*1,      INTENT(IN)   , lower limit check method '(' or '['  :   '(' uses <=   ;   '[' uses <
!     &         0.0D0,              ! DOUBLE PRECISION, INTENT(IN)   , lower limit value
!     &         POS_INFINITY,       ! DOUBLE PRECISION, INTENT(IN)   , upper limit value, POS_INFINITY is parameter in myelib_param_mod.f90
!     &         ')',                ! CHARACTER*1,      INTENT(IN)   , upper limit check method ')' or ']'  :   ')' uses >=   ;   ']' uses >
!     &         STOP_FLAG,          ! LOGICAL,          INTENT(INOUT), .TRUE. if there is a problem
!     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed  
!
!     GET_UDE_SCALAR_PROPERTY function retrieves the value of a scalar input variable. It returns a double precision number.
!      CD(or whatever you want to call it) = GET_UDE_SCALAR_PROPERTY(
!     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
!     &         'CD',               ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file
!     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed 
!
!     GET_UDE_ARRAY_PROPERTY function retrieves the value of a table input variable or the table size. It returns a double precision number.
!      PR_NEXT(or whatever you want to call it) = GET_UDE_ARRAY_PROPERTY(
!     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
!     &         'TBL1_PR',          ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file
!     &         I,                  ! INTEGER,          INTENT(IN)   , table index to retrieve or UDE_ARRAY_SIZE to return the table size
!     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed 
!
!     WRITE_UDE_ERROR_MESSAGE subroutine writes a generic message to the screen and convhist_fi.out file            
!      CALL WRITE_UDE_ERROR_MESSAGE(
!     &         ' ERROR: UDE %ELNUM %PROPNAME IS NOT ASCENDING ',       ! CHARACTER(*),     INTENT(IN)   , message to write ("%..." text will be replaced with arguments)
!     &         UDE_ELNUM,                                              ! INTEGER,          INTENT(IN)   , element # from .flo file                       (will replace %ELNUM in MSG)
!     &         'TBL1_PR',                                              ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file (will replace %PROPNAME in MSG)
!     &         0.0D0,                                                  ! DOUBLE PRECISION, INTENT(IN)   , a number used in the error check               (will replace %VALUE1 in MSG)
!     &         0.0D0,                                                  ! DOUBLE PRECISION, INTENT(IN)   , another number used in the error check         (will replace %VALUE2 in MSG)
!     &         ISTATUS)                                                ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
!     
!     *************************************************************************************************************************
!
      SUBROUTINE UDE_INPUT_ERROR_TRAP (UDE_ELNUM, STOP_FLAG, IQ, GET_UDE_TYPE, GET_UDE_SCALAR_PROPERTY,
     &                                 GET_UDE_ARRAY_PROPERTY, ERROR_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN,
     &                                 WARN_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN, WRITE_UDE_ERROR_MESSAGE )
      
!     Do not modify the DLLEXPORT line      
      !DIR$ ATTRIBUTES DLLEXPORT :: UDE_INPUT_ERROR_TRAP
      
      USE MYELIB_PARAM_MOD         ! see myelib_param_mod.f90     for description of the parameters use 
      USE MYELIB_INTERFACE_MOD     ! see myelib_interface_mod.f90 for description of the subroutine arguments
      
      IMPLICIT NONE                ! every variable must be declared
      
!     Start: Subroutine Argument variables (do not modify)
      INTEGER, INTENT(IN)       ::   UDE_ELNUM     ! the UDE element number from the .flo file that will be checked
      INTEGER, INTENT(IN)       ::   IQ            ! the screen unit number for write commands
      LOGICAL, INTENT(OUT)      ::   STOP_FLAG     ! The main solver will stop if this is set to TRUE

!     Pointers to the function or subroutine in the main solver (do not modify)   
      PROCEDURE(GET_UDE_TYPE_INTERF)                               :: GET_UDE_TYPE
      PROCEDURE(ERROR_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN_INTERF)     :: ERROR_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN
      PROCEDURE(WARN_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN_INTERF)      :: WARN_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN      
      PROCEDURE(GET_UDE_SCALAR_PROPERTY_INTERF)                    :: GET_UDE_SCALAR_PROPERTY               
      PROCEDURE(GET_UDE_ARRAY_PROPERTY_INTERF)                     :: GET_UDE_ARRAY_PROPERTY                
      PROCEDURE(WRITE_UDE_ERROR_MESSAGE_INTERF)                    :: WRITE_UDE_ERROR_MESSAGE
!     End: Subroutine Argument variables (do not modify)      
                                
!     Start: Local variables (modify as needed)           
      CHARACTER*31              ::   UDE_TYPE                ! the name of the UDE from the .flo file 
        
      INTEGER                   ::   ISTATUS                 ! <0 means the function or subroutine has failed
      INTEGER                   ::   I, N
      DOUBLE PRECISION          ::   CD, PR_PREV, PR_NEXT
!     End: Local variables (modify as needed)      

      STOP_FLAG = .FALSE.    ! (do not modify)     
      
!     Start: UDE_TYPE check (do not modify)      
      UDE_TYPE  = GET_UDE_TYPE(UDE_ELNUM, ISTATUS)
      IF (ISTATUS < 0) THEN
         WRITE(IQ,*) ' ERROR: UDE TYPE NUMBER IS NOT REGISTERED FOR UDE ', UDE_ELNUM
         RETURN   !Stop immediately if this happens because nothing below will work anyway.  This could be a GUI issue.
      ENDIF
!     End: UDE_TYPE check (do not modify)      



!     The select case statement is needed if the model has more than 1 UDE component.        
      SELECT CASE(UDE_TYPE)
      
      CASE ('HW_WATER_FORT')           ! the name of the UDE from the .flo file

         
         CALL ERROR_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN(
     &            UDE_ELNUM,           ! INTEGER,          INTENT(IN)   , element # from .flo file
     &            'PIPE_DIA',         ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file
     &            '[',                ! CHARACTER*1,      INTENT(IN)   , lower limit check method '(' or '['  :   '(' uses <=   ;   '[' uses <
     &            0.0D0,              ! DOUBLE PRECISION, INTENT(IN)   , lower limit value
     &            1.0D4,              ! DOUBLE PRECISION, INTENT(IN)   , upper limit value, POS_INFINITY is parameter in myelib_param_mod.f90
     &            ']',                ! CHARACTER*1,      INTENT(IN)   , upper limit check method ')' or ']'  :   ')' uses >=   ;   ']' uses >
     &            STOP_FLAG,          ! LOGICAL,          INTENT(INOUT), .TRUE. if there is a problem
     &            ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
         
         CALL ERROR_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN(
     &            UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &            'PIPE_LENGTH',      ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file
     &            '[',                ! CHARACTER*1,      INTENT(IN)   , lower limit check method '(' or '['  :   '(' uses <=   ;   '[' uses <
     &            0.0D0,              ! DOUBLE PRECISION, INTENT(IN)   , lower limit value
     &            1.0D10,             ! DOUBLE PRECISION, INTENT(IN)   , upper limit value, POS_INFINITY is parameter in myelib_param_mod.f90
     &            ']',                ! CHARACTER*1,      INTENT(IN)   , upper limit check method ')' or ']'  :   ')' uses >=   ;   ']' uses >
     &            STOP_FLAG,          ! LOGICAL,          INTENT(INOUT), .TRUE. if there is a problem
     &            ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed 
         
         CALL ERROR_IF_UDE_PROPERTY_IS_OUT_OF_DOMAIN(
     &            UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &            'HW_ROUGH_COEF',    ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file
     &            '(',                ! CHARACTER*1,      INTENT(IN)   , lower limit check method '(' or '['  :   '(' uses <=   ;   '[' uses <
     &            0.0D0,              ! DOUBLE PRECISION, INTENT(IN)   , lower limit value
     &            1.0D4,              ! DOUBLE PRECISION, INTENT(IN)   , upper limit value, POS_INFINITY is parameter in myelib_param_mod.f90
     &            ']',                ! CHARACTER*1,      INTENT(IN)   , upper limit check method ')' or ']'  :   ')' uses >=   ;   ']' uses >
     &            STOP_FLAG,          ! LOGICAL,          INTENT(INOUT), .TRUE. if there is a problem
     &            ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed      
         
         
      ENDSELECT
      
      RETURN   
      ENDSUBROUTINE UDE_INPUT_ERROR_TRAP
      
!     *************************************************************************************************************************
!     *************************************************************************************************************************
!     The UDE_SOLVER subroutine is where you write your custom flow function
!
!     See the UDE_INPUT_ERROR_TRAP subroutine for descriptions of the following subroutines and functions from the main solver:
!     GET_UDE_TYPE, GET_UDE_SCALAR_PROPERTY, GET_UDE_ARRAY_PROPERTY
!
!     These are additional subroutines and functions from the main solver that are used in this subroutine:
!
!     GET_UDE_BOUNDARY_CONDITION function returns flow element upstream and downstream boundary conditions such as
!     PTS, PSEB, THETA_INLET, PHI_EXIT, etc.    It returns a double precision number.  These are usually used for the flow calculation.
!     Typical BC_ID: UDE_PTS, UDE_TTS, UDE_PSEB, UDE_RELANGL, UDE_THETA_INLET, UDE_PHI_INLET, UDE_THETA_EXIT, UDE_PHI_EXIT
!     See myelib_param_mod.f90 for more information. 
!      PTS(or whatever you want to call it) = GET_UDE_BOUNDARY_CONDITION(
!     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
!     &         UDE_PTS,            ! INTEGER,          INTENT(IN)   , BC_ID of the boundary condition to return. (UDE_PTS, UDE_PSEB, etc..)
!     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
!
!     GET_UDE_FLUID_PROPERTY function returns flow element fluid properties at the upstream, downstream and an average.
!     It returns a double precision number.  These may or may not be needed for the flow calculation.
!     See myelib_param_mod.f90 for more information. 
!      RHO(or whatever you want to call it) = GET_UDE_BOUNDARY_CONDITION(
!     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
!     &         UDE_RHO_EXIT,       ! INTEGER,          INTENT(IN)   , ID of the fluid property to return. (UDE_CP_INLET, UDE_PRANDTL_INLET etc..)
!     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
!
!     GET_UDE_FLOW_DIRECTION function returns the flow direction as determined by the solver.  It returns an integer.
!     Return values are -1, 0, 1.  Parameters REVERSED_FLOW_DIRECTION=-1, FORWARD_FLOW_DIRECTION=1, UNDEFINED_FLOW_DIRECTION=0
!     are available for clearer code.  
!     FLOW_DIRECTION(or whatever you want to call it) = GET_UDE_FLOW_DIRECTION(
!     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
!     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
!
!     SET_UDE_SOLVED_VALUE subroutine returns the results to the main solver.
!     Valid SOL_ID: UDE_MDOT1, UDE_TT_EXIT, UDE_VEL_EXIT, UDE_D_MDOT1_D_PTS, UDE_D_MDOT1_D_PSEB
!     UDE_MDOT1 is the only required solution,  the others are optional.
!     CALL SET_UDE_SOLVED_VALUE(
!     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
!     &         UDE_MDOT1,          ! INTEGER,          INTENT(IN)   , ID of the solution being sent in SOL_VAL
!     &         MDOT,               ! DOUBLE PRECISION, INTENT(IN)   , solution value
!     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed)
!
!     SET_UDE_EXTRA_RESULT subroutine writes some extra results to the .res file.  It does not affect the analysis.
!     CALL SET_UDE_EXTRA_RESULT(
!     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
!     &         "CD_RESULT",        ! CHARACTER(*),     INTENT(IN)   , Name of the extra result to write
!     &         CD,                 ! DOUBLE PRECISION, INTENT(IN)   , value of the extra result to write
!     &         "(unitless)",       ! CHARACTER(*),     INTENT(IN)   , units of the extra result to write
!     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
!
!     UDE_GET_REFERENCE_CONDITION_INTERF function will retrieve general values from the solution.  It is based on the
!     controller so it's valid inputs are the same as the controllers valid inputs.
!
!
!     *************************************************************************************************************************
!
      SUBROUTINE UDE_SOLVER (UDE_ELNUM, GET_UDE_TYPE, GET_UDE_SCALAR_PROPERTY, GET_UDE_ARRAY_PROPERTY,
     &                       GET_UDE_BOUNDARY_CONDITION, GET_UDE_FLUID_PROPERTY, GET_UDE_FLOW_DIRECTION,
     &                       UDE_GET_REFERENCE_CONDITION, SET_UDE_SOLVED_VALUE, SET_UDE_EXTRA_RESULT )

!     Do not modify the DLLEXPORT line     
      !DIR$ ATTRIBUTES DLLEXPORT :: UDE_SOLVER      
      
      USE MYELIB_PARAM_MOD         ! see myelib_param_mod.f90     for description of the parameters use 
      USE MYELIB_INTERFACE_MOD     ! see myelib_interface_mod.f90 for description of the subroutine arguments
      
      IMPLICIT NONE                ! every variable must be declared

      
!     Start: Subroutine Argument variables (do not modify)
      INTEGER, INTENT(IN)   ::   UDE_ELNUM     ! the UDE element number from the .flo file that will be processed
      
!     Pointers to the function or subroutine in the main solver (do not modify)     
      PROCEDURE(GET_UDE_TYPE_INTERF)                               :: GET_UDE_TYPE  
      PROCEDURE(GET_UDE_SCALAR_PROPERTY_INTERF)                    :: GET_UDE_SCALAR_PROPERTY               
      PROCEDURE(GET_UDE_ARRAY_PROPERTY_INTERF)                     :: GET_UDE_ARRAY_PROPERTY      
      PROCEDURE(GET_UDE_BOUNDARY_CONDITION_INTERF)                 :: GET_UDE_BOUNDARY_CONDITION      
      PROCEDURE(GET_UDE_FLUID_PROPERTY_INTERF)                     :: GET_UDE_FLUID_PROPERTY      
      PROCEDURE(GET_UDE_FLOW_DIRECTION_INTERF)                     :: GET_UDE_FLOW_DIRECTION      
      PROCEDURE(UDE_GET_REFERENCE_CONDITION_INTERF)                :: UDE_GET_REFERENCE_CONDITION      
      PROCEDURE(SET_UDE_SOLVED_VALUE_INTERF)                       :: SET_UDE_SOLVED_VALUE      
      PROCEDURE(SET_UDE_EXTRA_RESULT_INTERF)                       :: SET_UDE_EXTRA_RESULT
!     End: Subroutine Argument variables (do not modify)       
      
!     Start: Local variables (modify as needed)
      INTEGER               ::   ISTATUS, FLOW_DIRECTION
      CHARACTER*31          ::   UDE_TYPE                                            ! the name of the UDE from the .flo file 
      DOUBLE PRECISION, PARAMETER   ::  GC          = 32.17405D0
      DOUBLE PRECISION, PARAMETER   ::  PI          = 3.1415927D0
      DOUBLE PRECISION              ::  PIPE_DIA                              ! inches
      DOUBLE PRECISION              ::  PIPE_LENGTH                           ! inches
      DOUBLE PRECISION              ::  HW_ROUGH_COEF                     
      DOUBLE PRECISION              ::  PTS                                   !psia                     
      DOUBLE PRECISION              ::  PSEB                                  !psia                     
      DOUBLE PRECISION              ::  RHO_INLET                             !lbm/ft^3                     
      DOUBLE PRECISION              ::  MU_INLET                              !lbm/hr/ft                    
      DOUBLE PRECISION              ::  PIPE_AREA                             !lbm/hr/ft                    
      DOUBLE PRECISION              ::  HYD_RAD                               !in^2                    
      DOUBLE PRECISION              ::  HEAD_LOSS                             !ft                    
      DOUBLE PRECISION              ::  S                                     !ft                    
      DOUBLE PRECISION              ::  VEL                                   !ft/sec           
      DOUBLE PRECISION              ::  Q                                     !ft^3/sec           
      DOUBLE PRECISION              ::  MASS_FLOW                             !lbm/sec           
      DOUBLE PRECISION              ::  GPM                                   !gallon per minute           
      DOUBLE PRECISION              ::  RE                                    !Reynolds number (unitless)  
!     End: Local variables (modify as needed)      
      
      
      UDE_TYPE = GET_UDE_TYPE(UDE_ELNUM, ISTATUS)
      SELECT CASE(UDE_TYPE)
      
      CASE ('HW_WATER_FORT')            ! the name of the UDE from the .flo file
         
!        UDE to calculate a flowrate using the Hazen-Willimas formula
!        Restrictions: Only good for liquid Water at moderate temperatures and turbulent Reynolds number.
!                               Water fills entire pipe area
         
!        Get input variables
         PIPE_DIA = GET_UDE_SCALAR_PROPERTY(
     &              UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &              'PIPE_DIA',         ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file
     &              ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
     
         PIPE_LENGTH = GET_UDE_SCALAR_PROPERTY(
     &              UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &              'PIPE_LENGTH',      ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file
     &              ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed

         HW_ROUGH_COEF = GET_UDE_SCALAR_PROPERTY(
     &              UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &              'HW_ROUGH_COEF',    ! CHARACTER(*),     INTENT(IN)   , UDE element input property name from .flo file
     &              ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed     
     
     

         
!        Get required properties
         PTS      = GET_UDE_BOUNDARY_CONDITION(
     &             UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &             UDE_PTS,            ! INTEGER,          INTENT(IN)   , ID of the fluid property to return. (UDE_CP_INLET, UDE_PRANDTL_INLET etc..)
     &             ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
     
         PSEB    = GET_UDE_BOUNDARY_CONDITION(
     &             UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &             UDE_PSEB,           ! INTEGER,          INTENT(IN)   , ID of the fluid property to return. (UDE_CP_INLET, UDE_PRANDTL_INLET etc..)
     &             ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed     
     
         RHO_INLET = GET_UDE_BOUNDARY_CONDITION(
     &             UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &             UDE_RHO_INLET,      ! INTEGER,          INTENT(IN)   , ID of the fluid property to return. (UDE_CP_INLET, UDE_PRANDTL_INLET etc..)
     &             ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed     

         MU_INLET = GET_UDE_BOUNDARY_CONDITION(
     &             UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &             UDE_MU_INLET,       ! INTEGER,          INTENT(IN)   , ID of the fluid property to return. (UDE_CP_INLET, UDE_PRANDTL_INLET etc..)
     &             ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed    
        
         
         PIPE_AREA   = 0.25 * PI * PIPE_DIA**2.0D0                ! in^2
         HYD_RAD     = PIPE_DIA  /  4.0D0 / 12.0D0                ! ft
         HEAD_LOSS   = (PTS - PSEB) * 144.0D0 / RHO_INLET         ! ft
         S           = HEAD_LOSS /( PIPE_LENGTH / 12.0D0)
         
!        Hazen Williams water velocity equation
         VEL                  = 1.318 * HW_ROUGH_COEF * HYD_RAD**0.63 * S**0.54      ! ft/sec
         
         Q                     = VEL * PIPE_AREA / 144.                               ! ft^3/sec
         MASS_FLOW             = RHO_INLET * Q                                        ! lbm/sec
         GPM                   = Q / 0.13368                                          ! gallon per minute
         
         RE                    = RHO_INLET * VEL * PIPE_DIA / 12. / MU_INLET * 3600.  ! Reynolds number (unitless)
         
!        Return results to main solver
         CALL SET_UDE_SOLVED_VALUE(
     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &         UDE_MDOT1,          ! INTEGER,          INTENT(IN)   , ID of the solution being sent in SOL_VAL
     &         MASS_FLOW,          ! DOUBLE PRECISION, INTENT(IN)   , solution value
     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed)

         CALL SET_UDE_SOLVED_VALUE(
     &         UDE_ELNUM,          ! INTEGER,          INTENT(IN)   , element # from .flo file
     &         UDE_VEL_EXIT,       ! INTEGER,          INTENT(IN)   , ID of the solution being sent in SOL_VAL
     &         VEL,                ! DOUBLE PRECISION, INTENT(IN)   , solution value
     &         ISTATUS )           ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed)


         
!        Write some additional results to the .flo file
         CALL SET_UDE_EXTRA_RESULT(
     &         UDE_ELNUM,              ! INTEGER,          INTENT(IN)   , element # from .flo file
     &         "PIPE_DIA",             ! CHARACTER(*),     INTENT(IN)   , Name of the extra result to write
     &         PIPE_DIA,               ! DOUBLE PRECISION, INTENT(IN)   , value of the extra result to write
     &         "in",                   ! CHARACTER(*),     INTENT(IN)   , units of the extra result to write
     &         ISTATUS )               ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed

         CALL SET_UDE_EXTRA_RESULT(
     &         UDE_ELNUM,              ! INTEGER,          INTENT(IN)   , element # from .flo file
     &         "PIPE_LENGTH",          ! CHARACTER(*),     INTENT(IN)   , Name of the extra result to write
     &         PIPE_LENGTH/12.0D0,     ! DOUBLE PRECISION, INTENT(IN)   , value of the extra result to write
     &         "ft",                   ! CHARACTER(*),     INTENT(IN)   , units of the extra result to write
     &         ISTATUS )               ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed
     
         CALL SET_UDE_EXTRA_RESULT(
     &         UDE_ELNUM,              ! INTEGER,          INTENT(IN)   , element # from .flo file
     &         "ROUGHNESS_COEFF",      ! CHARACTER(*),     INTENT(IN)   , Name of the extra result to write
     &         HW_ROUGH_COEF,          ! DOUBLE PRECISION, INTENT(IN)   , value of the extra result to write
     &         "unitless",             ! CHARACTER(*),     INTENT(IN)   , units of the extra result to write
     &         ISTATUS )               ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed     
       
         CALL SET_UDE_EXTRA_RESULT(
     &         UDE_ELNUM,              ! INTEGER,          INTENT(IN)   , element # from .flo file
     &         "HEAD_LOSS_IN_PIPE",    ! CHARACTER(*),     INTENT(IN)   , Name of the extra result to write
     &         HEAD_LOSS,              ! DOUBLE PRECISION, INTENT(IN)   , value of the extra result to write
     &         "ft",                   ! CHARACTER(*),     INTENT(IN)   , units of the extra result to write
     &         ISTATUS )               ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed 
     
         CALL SET_UDE_EXTRA_RESULT(
     &         UDE_ELNUM,              ! INTEGER,          INTENT(IN)   , element # from .flo file
     &         "VOL_FLOW",             ! CHARACTER(*),     INTENT(IN)   , Name of the extra result to write
     &         Q,                      ! DOUBLE PRECISION, INTENT(IN)   , value of the extra result to write
     &         "ft^3/sec",             ! CHARACTER(*),     INTENT(IN)   , units of the extra result to write
     &         ISTATUS )               ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed 

         CALL SET_UDE_EXTRA_RESULT(
     &         UDE_ELNUM,              ! INTEGER,          INTENT(IN)   , element # from .flo file
     &         "REYNOLDS NUMBER",      ! CHARACTER(*),     INTENT(IN)   , Name of the extra result to write
     &         RE,                     ! DOUBLE PRECISION, INTENT(IN)   , value of the extra result to write
     &         "unitless",             ! CHARACTER(*),     INTENT(IN)   , units of the extra result to write
     &         ISTATUS )               ! INTEGER,          INTENT(OUT)  , <0 means the function or subroutine has failed      
         
         
      ENDSELECT
      
      RETURN   
      ENDSUBROUTINE UDE_SOLVER
