﻿/**
 *  PageItem
 */
define(function (require, exports) {

    require('browserContext');
    var path = require('../../common/path');
    var define = require('../define');
    var model = require('../model');
    var hotspotManger = require('./hotspot/hotspotmgr');
    var factory = require('./widgetfactory');
    var shellcmd = require('../../common/shellcmd');

    var ebkContext = require('../ebookcontext').Instance;
    var pageXmlParser = require('../../common/pagexmlparser').Instance;
    require('xml2json');

    /**
     *
     * @param pageProvider
     * @param pageIndex
     * @param audioPlayer
     * @constructor
     */
    function PageItem(pageProvider, pageIndex, audioPlayer) {

        var _hotspotMgr;

        /** @type {HTMLCanvasElement}  */
        var _paintCanvas;

        /** @type {HTMLCanvasElement} */
        var _hotspotCanvas;

        /** @type {boolean} 笔迹是否已经加载 */
        var _strokeLoaded = false;

        this.isStrokeChanged = false;
        this.provider = pageProvider;
        this.pageIndex = pageIndex;
        this.content = null;
        this.pageObject = null;
        this.xamlPath = '';
        this.player = audioPlayer;
        this.widgetFactory = new factory.WidgetFactory();

        this.hotspots = [];
        this.widgets = [];
        //位置数组
        this.widgetPositions = null;

        this._isModify = false;  //页面被修改
        this._isEditing = false;

        this.hotspotToolBar = null;

        this.getHotspotMgr = function () {
            return _hotspotMgr;
        };

        this.setHotspotMgr = function (hotspotMgr) {
            _hotspotMgr = hotspotMgr;
        };

        this.getStrokeLoaded = function () {
            return _strokeLoaded;
        };

        this._setStrokeLoaded = function (isLoaded) {
            _strokeLoaded = isLoaded;
        };

        /**
         * 设置笔迹Canvas
         * @param {HTMLCanvasElement} canvas
         * @private
         */
        this._setPaintCanvas = function (canvas) {
            _paintCanvas = canvas;
        };

        /**
         *  获取笔迹Canvas
         * @returns {HTMLCanvasElement}
         */
        this.getPaintCanvas = function () {
            return _paintCanvas;
        };

        this.saveStrokesToLocal = function () {
            this._savePainting();
        }

        this._setHotspotCanvas = function (canvas) {
            _hotspotCanvas = canvas;
        };

        this.getHotspotCanvas = function () {
            return _hotspotCanvas;
        };
    };

    PageItem.prototype.trigger = function (name, args) {
        this.content.triggerHandler(name, args);
    }

    PageItem.prototype.getContent = function () {
        return this.content;
    };

    /**
     * 获取页面JSON对象
     * @param {String} xaml xaml文件路径
     * @returns {JSON} 转换后的JSON对象
     * @private
     */
    PageItem.prototype._readPageObjectFromXaml = function (xaml) {
        console.log(xaml);
        var xml = shellcmd.readFile(xaml);
        if (isString(xml)) {
            if (xml.length > 0) {
                console.time('xmltojson: Page');
                var obj2 = pageXmlParser.ToJSON(xml);
                console.timeEnd('xmltojson: Page');
                return obj2;
            }
        }
        return xml;
    };

    PageItem.prototype._getUnDrmBackgroundImg = function (pageIndex) {
        var src = path.join(this.provider.getStdBookDir().replace(/\s/g, '%20'), this.pageIndex + '/' + this.pageIndex + '.jpg');
        return src;
    };

    PageItem.prototype.getBackgroud = function () {
        if (!AppCfg.useDrm) {
            return this._getUnDrmBackgroundImg(this.pageIndex);
        } else {
            var winDst = this._getDrmTmpImgPath();
            var dst = winDst.replace(/\s/g, '%20');
            if (shellcmd.isFileExistSync(winDst)) {
                return dst;
            }
        }
        return null;
    };

    PageItem.prototype._getDrmTmpImgPath = function () {
        var file = '{0}\\{0}.jpg'.format(this.pageIndex, this.pageIndex);
        var winDst = path.join(ebkContext.getSysTmpPath(), this.provider.bookCode, '{0}.jpg'.format(this.pageIndex));
        return winDst;
    };

    PageItem.prototype.loadBackground = function () {
        var _self = this;
        var def = $.Deferred();

        var w = this.provider.bookImageSize.width;
        var h = this.provider.bookImageSize.height;

        if (!AppCfg.useDrm) {
            //不使用DRM加密的资源
            var src = this._getUnDrmBackgroundImg(this.pageIndex).replace(/[\(\)]/g,function(str){return ("\\" + str)});
            this.content = $('<div class="pageItem" style="background-image:url(' + src + ');height:' + h + 'px;width:' + w + 'px;"></div>');
            def.resolve();
        } else {   //使用DRM的资源
            //不缓存,基本上用不到了
            if (!AppCfg.saveDRMImage) {
                //使用canvas直接加载
                this._loadBgByCanvas(h, w).done(function () {
                    def.resolve();
                });
            }
            else { //直接解压到Tmp 文件夹
                var file = '{0}\\{0}.jpg'.format(this.pageIndex, this.pageIndex);
                var winDst = this._getDrmTmpImgPath();
                //网页显示图片必须把空格换成%20,而客户端里不必要这么做
                var dst = winDst.replace(/\s/g, '%20');
                if (shellcmd.isFileExistSync(winDst)) {
                    //存在缓存背景图
                    dst = dst.replace(/[\(\)]/g,function(str){return ("\\" + str)});
                    _self.content = $('<div class="pageItem" style="background-image:url(' + 'file:///' + dst +
                        ');height:' + h + 'px;width:' + w + 'px;"></div>');
                    return def.resolve();
                } else {
                    //不存在缓存背景图
                    _self.provider.retriveDrmRes(file, winDst).done(function () {
                        dst = dst.replace(/[\(\)]/g,function(str){return ("\\" + str)});
                        _self.content = $('<div class="pageItem" style="background-image:url(' + 'file:///' + dst +
                            ');height:' + h + 'px;width:' + w + 'px;"></div>');
                        def.resolve();
                    });
                }
            }
        }

        return def.promise();
    };

    PageItem.prototype._getBase64Str = function (pageIndex) {
        var file = '{0}\\{0}.jpg'.format(pageIndex, pageIndex);
        return this.provider.retriveDrmRes(file, '');
    };

    PageItem.prototype._loadBgByCanvas = function (h, w) {
        var _self = this;
        var def = $.Deferred();
        this.content = $('<div class="pageItem" style="height:' + h + 'px;width:' + w + 'px;"></div>');
        var canvas = $('<canvas class="background" />').attr({width: w, height: h}).css({'position': 'absolute', 'left': '0px', 'z-index': -3});
        this._getBase64Str(this.pageIndex).done(function (base64) {
            var ctx = canvas.get(0).getContext('2d');
            var backgroundImg = new Image();
            backgroundImg.onload = function () {
                ctx.drawImage(this, 0, 0);
            }
            backgroundImg.src = base64;
            _self.content.prepend(canvas);
            def.resolve();
        });

        return def.promise();
    }

    PageItem.prototype._bindCommands = function () {
        var _self = this;
        /**
         * 删除widget
         */
        this.content.on('widgetDeleted', function (event, id, isDeleted) {
            _self._isModify = true;
            _self.removeWidgetById(id, isDeleted);
            event.stopPropagation();
        });

        this.content.on('hideAllNote', function () {
            $.each(_self.widgets, function (i, widget) {
                widget.WidgetType === define.WidgetType.Note && widget.hide();
            });
        });

        this.content.bind('insertWidget', function (event, pageOffset, widgetId, srcPos, curPos) {
            var w = _self.getWidgetById(widgetId);
            if (w) {
                var curPageItem = pageOffset ? _self.provider.getPageItem(parseInt(_self.pageIndex) + parseInt(pageOffset)) : _self;
                _self._onDrop(_self, curPageItem, w, srcPos, curPos);
            }
        });
    }

    PageItem.prototype._loadContent = function () {
        if (this.pageObject && this.content) {
            this.widgetFactory.init(this.content, path.join(ebkContext.getUserBookPath(), this.pageIndex + ""), this.xamlPath, this.player);

            this.loadBlackOverlay();
            // 创建热区
            this._loadHotspots(this.pageObject.hotspot);
            // 创建绝对定位对象
            this._loadWidget(this.pageObject.absolutelayout);
            // 创建笔迹画布
            this._loadPaintCanvas();
            //创建位置数组
            this._initWidgetPosition();
        } else {
            console.warn('pageObject or content is null');
        }
    };

    PageItem.prototype._loadWidget = function (absolutelayout) {
        if (!absolutelayout)  return;
        var value, _self = this;
        for (var key in absolutelayout) {
            if (absolutelayout.hasOwnProperty(key)) {
                value = $.makeArray(absolutelayout[key]);
                $.each(value, function (i, data) {
                    //注意约定把xaml中的类型字段的第一个字母大写后作为卡片的类型，如absolutelayout.video,则创建define.WidgetType.Video类型的卡片,这样减少许多if else
                    _self._insertWidget(define.WidgetType[key[0].toUpperCase() + key.substr(1)], data);
                });
            }
        }
    }

    /**
     * 加载笔迹画布
     * @private
     */
    PageItem.prototype._loadPaintCanvas = function () {
        var _self = this;
        var w = this.provider.bookImageSize.width;
        var h = this.provider.bookImageSize.height;
        var canvas = $('<canvas class="paint" />').attr({width: w, height: h}).css({'position': 'absolute', 'z-index': -2});

        this.content.prepend(canvas);
        this._setPaintCanvas(canvas[0]);
        _self._loadPainting();
    };

    /**
     * 加载热区画布
     * @param hotspotObj
     * @private
     */
    PageItem.prototype._loadHotspots = function (hotspotObj) {
        if (!hotspotObj) return;
        var rects = hotspotObj.Rectangle;
        if (rects) {
            if ($.isArray(rects)) {
                for (var i = 0, len = rects.length; i < len; i++) {
                    var rect = rects[i];
                    this.hotspots.push(new model.HotspotMeta(define.HotspotType.Rectangle, rect));
                }
            } else {
                this.hotspots.push(new model.HotspotMeta(define.HotspotType.Rectangle, rects));
            }
        }

        var ellipses = hotspotObj.Ellipse;
        if (ellipses) {
            if ($.isArray(ellipses)) {
                for (var i = 0, len = ellipses.length; i < len; i++) {
                    var ellipse = ellipses[i];
                    this.hotspots.push(new model.HotspotMeta(define.HotspotType.Ellipse, ellipse));
                }
            } else {
                this.hotspots.push(new model.HotspotMeta(define.HotspotType.Ellipse, ellipses));
            }
        }

        var paths = hotspotObj.Path;
        if (paths) {
            if ($.isArray(paths)) {
                for (var i = 0, len = paths.length; i < len; i++) {
                    var path = paths[i];
                    this.hotspots.push(new model.HotspotMeta(define.HotspotType.Path, path));
                }
            } else {
                this.hotspots.push(new model.HotspotMeta(define.HotspotType.Path, paths));
            }
        }

        $.each(this.hotspots, function () {
            // console.log(this.toString());
        });

        this._createHotspotCanvas(this.hotspots);
    };

    /**
     * 创建热区
     * @param {Array.<HotspotMeta>}  hotspotMetaArr
     * @private
     */
    PageItem.prototype._createHotspotCanvas = function (hotspotMetaArr) {

        var w = this.provider.bookImageSize.width;
        var h = this.provider.bookImageSize.height;
        var canvas = $('<canvas class="hotspot" />').attr({
            width: w,
            height: h
        }).css({'z-index': -1});

        var hotspotMgr = new hotspotManger.HotspotMgr(canvas[0], hotspotMetaArr);
        hotspotMgr.setIsDoubleLanguage(this.provider.isDoubleLanguage);
        if (this.pageObject.ReadLan === "Multi") {
            hotspotMgr.setIsDoubleLanguage(true);
        }
        this.setHotspotMgr(hotspotMgr);

        this.content.prepend(canvas);
        this._setHotspotCanvas(canvas[0]);

        //创建热区的同时，每个页面创建一个热区工具栏
        var hotspotToolBar = $('<div class="hotspotPopToolBar" style="position: absolute;display:none;-webkit-transform:matrix(1.3,0,0,1.4,0,0);">\
            <dl class="toolbar" style="width:80px;">\
                <dd class="toolbar_black" onMouseDown="this.className=\'toolbar_black_current\'" onmouseup="this.className=\'toolbar_black\'">\
                    <img src="assert/images/han.png" >\
                </dd>  \
                <dd class="toolbar_black" onMouseDown="this.className=\'toolbar_black_current\'" onmouseup="this.className=\'toolbar_black\'">\
                    <img src="assert/images/wei.png" >\
                </dd>\
            </dl>\
            </div>');
        this.content.prepend(hotspotToolBar);
        this.hotspotToolBar = hotspotToolBar;
        hotspotMgr.setHotspotToolBar(hotspotToolBar);
    };

    /**
     * 根据 id 获取widget
     * @param id
     */
    PageItem.prototype.getWidgetById = function (id) {
        for (var i = 0; i < this.widgets.length; i++) {
            var w = this.widgets[i];
            if (w.Id === id) {
                return w;
            }
        }
        return null;
    };

    /**
     * 删除某个Widget
     * @param id
     * @param {boolean=} deleteSrc   是否删除物理文件
     */
    PageItem.prototype.removeWidgetById = function (id, deleteSrc) {

        var deleteWidget = null;
        this.widgets = this.widgets.filter(function (w) {
            if (w.Id != id)
                return true;
            else {
                deleteWidget = w;
                return false;
            }
        });
        if (deleteWidget) {
            deleteSrc = (deleteSrc == undefined) ? true : deleteSrc;
            if (deleteSrc) {
                //避免删除有多个引用的文件
                $.each(this.widgets, function (i, e) {
                    if (e.src === deleteWidget.src) {
                        deleteSrc = false;
                        return false;
                    }
                });
            }
            this.widgetPositions[this._getWidgetPosition([deleteWidget.Content[0].offsetLeft, deleteWidget.Content[0].offsetTop])] -= 1;
            deleteWidget.destroy(deleteSrc);
            this._isModify = true;
        }
    };

    PageItem.prototype._getPageObj = function (pageIndex) {
        var _self = this;
        var def = $.Deferred();
        var tmpPath = _self.pageIndex + '/' + _self.pageIndex + '.xaml';
        var userPageXaml = path.trimPosix(path.join(ebkContext.getUserBookPath(), tmpPath));
        _self.xamlPath = userPageXaml;
        _self._getStdPageObj(pageIndex).done(function (stdPgObj) {
            console.log(stdPgObj);
            if (shellcmd.isFileExistSync(userPageXaml)) {
                var usrPgObj = _self._readPageObjectFromXaml(userPageXaml);
                if (usrPgObj != null) {
                    //使用授课资源里的卡片信息,使用原资源里的点读信息
                    stdPgObj.absolutelayout = usrPgObj.absolutelayout;
                }
            } else {
                _self._initUserPageXaml(userPageXaml);
                _self._isModify = true;
            }
            def.resolve(stdPgObj);
        });

        return def.promise();
    };

    PageItem.prototype._getStdPageObj = function (pageIndex) {

        var def = $.Deferred();
        if (AppCfg.useDrm) {
            var file = '{0}\\{0}.xaml'.format(pageIndex, pageIndex);
            if (!AppCfg.saveDRMXAML) {
                this.provider.retriveDrmRes(file, '').done(function (xml) {
                    var obj = $.xml2json(xml);
                    def.resolve(obj);
                });
            }
            else {
                var dst = path.join(ebkContext.getSysTmpPath(), this.provider.bookCode, '{0}.xaml'.format(pageIndex));
                if (shellcmd.isFileExistSync(dst)) {
                    var xml = shellcmd.readFile(dst);
                    def.resolve($.xml2json(xml));
                } else {
                    this.provider.retriveDrmRes(file, dst).done(function () {
                        var xml = shellcmd.readFile(dst);
                        var obj = $.xml2json(xml);
                        def.resolve(obj);
                    });
                }
            }
        } else {
            var tmpPath = this.pageIndex + '/' + this.pageIndex + '.xaml';
            var xaml = path.trimPosix(path.join(ebkContext.getStdBookPath(), tmpPath));
            var obj = this._readPageObjectFromXaml(xaml);
            def.resolve(obj);
        }

        return def.promise();
    };

    PageItem.prototype._initUserPageXaml = function (userPageXaml) {
        var ub = path.trimPosix(ebkContext.getUserBookPath());
        var temp = path.trimPosix(ebkContext.getTmpPath());
        var pagDir = path.trimPosix(path.dirname(userPageXaml));

        shellcmd.createDirIfNotExist(ub);
        shellcmd.createDirIfNotExist(temp);
        shellcmd.createDirIfNotExist(pagDir, 0);
    };

    /**
     * 页面加载
     * @param {function=} callback
     */
    PageItem.prototype.load = function () {
        var def = $.Deferred();
        var _self = this;
        _self._getPageObj(_self.pageIndex).done(function (ret) {
            _self.pageObject = ret;
            _self.loadBackground().done(function () {
                _self._loadContent();
                _self._bindCommands();
                _self.content.dragable = true;
                //把当前数据保存到data里,否则有些回调里取不到当前的引用对象
                _self.content.data('pagedata', _self);
                // 等页面完成加载之后才能确定笔迹Canvas或是一些Html元素的动态大小
                setTimeout(function () {
                    var hotspotMgr = _self.getHotspotMgr();
                    hotspotMgr && hotspotMgr.reRender();
                    def.resolve();
                }, 0);
            });
        });
        return def.promise()
    };

    PageItem.prototype._onDrop = function (srcPageItem, curPageItem, w, srcPos, curPos) {

        var flag = curPageItem._correctPos(curPos, w);//更正位置信息
        if (flag) {
            if (srcPageItem.pageIndex === curPageItem.pageIndex) {
                srcPageItem.widgetPositions[srcPageItem._getWidgetPosition(srcPos)] -= 1;
                w.Content.css({
                    left: curPos[0],
                    top: curPos[1]
                });
                srcPageItem._isModify = true;
            } else {
                //从别的页面拖进来
                if (w) {
                    srcPageItem.removeWidgetById(w.Id, false);

                    var _objhere = {
                        file: w.src,
                        realPath: w.realPath,
                        title: w.Title,
                        showTitleFlag : w.showTitleFlag,
                        showIconFlag : w.showIconFlag,
                        resourceId : w.resourceId,
                        left: curPos[0],
                        top: curPos[1],
                        width: w.PlayAreaWidth,
                        height: w.PlayAreaHeight,
                        type: w.type,
                        style: w.style.toString(),
                        userName: w.userName,
                        uploadTime: w.uploadTime,
                        score: w.score,
                        userAddressInfo: w.userAddressInfo,
                        description: w.description
                    }

                    //上传的地方同样要做如此处理，很挫，但还有4小时上线。
                    if(w.WidgetType == "icw") {
                        _objhere = {
                            file : w.src + ".icw",
                            insertFromOuter : true,
                            multiFiles : [w.src + ".icw"],
                            userName: w.userName,
                            left: curPos[0],
                            top: curPos[1],
                            notCopySrc : true
                        }
                    }

                    curPageItem.insertWidget(w.WidgetType,_objhere);
                }
            }
            //发送位置变化消息
            w.trigger('reposition', curPos);
        }
    }

    PageItem.prototype.onDragOver = function (event) {
        /*event.preventDefault();
         return false;*/
        event.stopPropagation();
        event.preventDefault();
    };

    /**
     * 页面卸载
     * @param {boolean=} index 默认为false，只是卸载当前页面，如果为true，则卸载整个书本页面。如从书本跳转到书架
     */
    PageItem.prototype.unload = function (index) {
        //TODO: need impl
        //console.log('PageItem::unload, pageIndex:%d', this.pageIndex);
        this._savePainting();
        this.saveXaml();

        this.content.data('pagedata', null);

        $(this.getPaintCanvas()).remove();
        this._setPaintCanvas(null);

        var hotspotMgr = this.getHotspotMgr();
        if (hotspotMgr) {
            hotspotMgr.dispose();
            this.setHotspotMgr(null);
        }
        this.content.empty();
        //释放资源
        for (var i = 0, len = this.widgets.length; i < len; ++i) {
            var widget = this.widgets[i];
            widget.releaseRes && widget.releaseRes();
        }
        !index && this.content.remove();
    };

    /**
     * 插入卡片
     * @param widgetType 卡片类型
     * @param node  卡片节点信息
     * @param {object=} options 卡片额外参数
     */
    PageItem.prototype._insertWidget = function (widgetType, node, options) {
        var _this = this,
           _fileFullPath="",
            _fileName="";
        if(options){
            _fileFullPath = options.file;
            _fileName = _fileFullPath.substr(_fileFullPath.lastIndexOf("/")+1,_fileFullPath.length-1);
        }

        if(options && options.noNewIcon){
            _this.widgetFactory.createWidget(widgetType, node, options);
            return ;
        }

        var index=0 , exsit=0;
        for(;index<_this.widgets.length;index++){
            var _exsitFileName = _this.widgets[index].File;
            if(_fileName==_exsitFileName&&_fileName!="") {
                exsit=1;
                $.Zebra_Dialog(AppImportAndExportTips.ExistDownLoadFile, {
                    'type': 'question',
                    'title': AppTips.WarmTips,
                    'buttons': [AppTips.CancelTips,AppTips.OkTips],
                    'onClose': function (capition) {
                        if (capition == AppTips.OkTips) {
                            insertWidgetAndover();
                        }
                        return;
                    }
                });

            }
        }
        if(index >= _this.widgets.length && exsit==0) insertWidgetAndover();
        /**
         * 文件不存在直接插入、若存在覆盖文件
         */
        function insertWidgetAndover(){
            var widget = _this.widgetFactory.createWidget(widgetType, node, options);
            if (widget) {
                var _self = _this;
                widget.Content.data('pagedata', _this);
                _this.widgets.push(widget);
                _this._isModify = true;
                //这里是想让这段代码在当前环境下最后执行 ---- by yczhu
            setTimeout(function () {
                var pos = [widget.Left, widget.Top];
                if (_self._correctPos(pos, widget,options))    widget.Content.css({'left': pos[0], 'top': pos[1]});
                widget.Content.show();
                //correctToBaseResolution(widget.Content, {scaleFromCenter: false, scaleOrder: false, keepScale: true, width: page_base_resolution.width, height: page_base_resolution.height});
                //通知页面加载完成
                widget.trigger('pageLoaded');
                widget.setIsEditing(_self._isEditing);
            }, 0);
                widget.Content.hide();
            }
        }

    }

    /**
     * 插入卡片，外部模块调用
     * @param {!object} widgetType 卡片类型
     * @param {!object} options  其他参数
     */
    PageItem.prototype.insertWidget = function (widgetType, options) {
        if (!options) return;
        options.file = path.toPosix(options.file || "");
        if (!widgetType) {
            var ext = path.extname(options.file);
            widgetType = define.getWidgetTypeByExt(ext);
        }
        this._insertWidget(widgetType, null, options);
    }

    PageItem.prototype.toXml = function () {
        var hotSpotXaml = "";
        if (hotSpotXaml !== "") hotSpotXaml = "\n<eb:hotspot>" + hotSpotXaml + "\n</eb:hotspot>";
        var widgetsXaml = "";
        $.each(this.widgets, function (i, e) {
            widgetsXaml += e.toXml();
        });
        if (widgetsXaml !== "") widgetsXaml = "\n<eb:absolutelayout>" + widgetsXaml + "\n</eb:absolutelayout>"

        var xml = "<eb:page background=\"" + this.pageIndex + ".jpg\" xmlns:eb=\"EBookLib\">" + hotSpotXaml + widgetsXaml + "\n</eb:page>";
        return xml;
    };

    PageItem.prototype.setIsEditing = function (isediting) {
        var _this = this;
        this._isEditing = isediting;
        $.each(_this.widgets, function () {
            this.setIsEditing(isediting);
        });
        var hotspotMgr = this.getHotspotMgr();
        if (hotspotMgr) hotspotMgr.setIsEditing(isediting);
    };

    /**
     * 保存笔迹
     * @private
     */
    PageItem.prototype._savePainting = function () {
        if (!this.isStrokeChanged) {
            return;
        }

        var paintCanvas = this.getPaintCanvas();
        var strokeBase64 = paintCanvas.toDataURL();
        var strokeFilePath = path.join(this.provider.getUserBookDir(), this.pageIndex + '/' + this.pageIndex + '.stroke.base64');
        strokeFilePath = path.trimPosix(strokeFilePath);
        shellcmd.saveFile(strokeBase64, strokeFilePath);
    };


    PageItem.prototype.saveXaml = function () {
        if (!this._isModifyChanged()) return;
        var xml = this.toXml();
        shellcmd.saveFile(xml, this.xamlPath);
        this._isModify = false;
    }


    PageItem.prototype._isModifyChanged = function () {
        if (this._isModify) return true;
        for (var i = 0, len = this.widgets.length; i < len; ++i) {
            if (this.widgets[i].isModify) return true;
        }
        return false;
    };

    /**
     * 更正位置信息，防止widget移动时超出边界和重叠时的bug
     * @param {!Array.<number>} pos
     * @param {object} widget
     * @param {object=} options
     * @returns {boolean}是否更正了位置信息
     */
    PageItem.prototype._correctPos = function (pos, widget, options) {

        if (!pos || pos.length < 2) return false;
        var flag = true;
        options = options || {};
        pos[0] = parseFloat(pos[0]);
        pos[1] = parseFloat(pos[1]);

        if (!this.widgetPositions) this._initWidgetPosition();

        //是否从外部插入，防止刚插入时重叠
        if (options.insertFromOuter) {
            var emptyPos = -1;//有任何可用的空位
            for (var i = 0; i < this.widgetPositions.length; ++i) {

                //如果当前位置可用
                if (this.widgetPositions[i] <= 0) {
                    if( Math.floor(i / this.widgetPositions.leftCount) * this.widgetPositions.topOffset >= 200){
                        emptyPos = i;
                        break;
                    }
                }
            }
            if (emptyPos < 0) {
                emptyPos = 0;
                var minPosCount = this.widgetPositions[0];
                for (var i = 0; i < this.widgetPositions.length; ++i) {
                    if (this.widgetPositions[i] < minPosCount) {
                        minPosCount = this.widgetPositions[i];
                        emptyPos = i;
                    }
                }
            }
            pos[0] = (emptyPos % this.widgetPositions.leftCount) * this.widgetPositions.leftOffset + (this.widgetPositions.leftOffset - widget.Width);
            pos[1] = Math.floor(emptyPos / this.widgetPositions.leftCount) * this.widgetPositions.topOffset + (this.widgetPositions.topOffset ) / 2;
        }

        if (isNaN(pos[0]) || isNaN(pos[1])) return false;

        // 防止超出边界
        var width = widget.Content.width(), height = widget.Content.height();
        var title = widget.Content.find('.widgetTitle:first');
        if (title.length) {
            minLeft = Math.max(title.outerWidth() - width, 0) / 2;
            width += minLeft;
        }
        var matrixTransform = this.getTransformMatrix(widget.Content);
        var offsetX = (matrixTransform[0] - 1) * width / 2, offsetY = (matrixTransform[3] - 1) * height / 2;
        var maxLeft = this.content.width() - offsetX, maxTop = this.content.height() - offsetY;
        var minLeft = offsetX, minTop = offsetY;
         //   minTop = 300//offsetY;

        pos[0] = pos[0] <= minLeft ? minLeft : pos[0], pos[1] = pos[1] <= minTop ? minTop : pos[1];

        if (pos[0] + width > maxLeft) pos[0] = maxLeft - 10 - width;
        if (pos[1] + height > maxTop) pos[1] = maxTop - 10 - height;

        pos[0] = pos[0] <= minLeft ? minLeft : pos[0], pos[1] = pos[1] <= minTop ? minTop : pos[1];

        //更正位置数组
        this.widgetPositions[this._getWidgetPosition(pos)] += 1;
        return flag;
    }

    /**
     * 初始化widget的位置数组
     * @private
     */

    PageItem.prototype._initWidgetPosition = function () {
        if (this.widgetPositions) return;
        this.widgetPositions = [];
        var matrixTransform = this.getTransformMatrix(this.content), leftOffset, topOffset;
        leftOffset = topOffset = parseInt(80 / matrixTransform[0] * 2) + 80; //每个widget的距离
        if (leftOffset < 130) leftOffset = 130;
        if (topOffset < 130) topOffset = 130;

        var maxLeft = this.content.width(), maxTop = this.content.height();

        var leftCount = Math.floor(maxLeft / leftOffset), topCount = Math.floor(maxTop / topOffset);

        for (var i = 0, len = leftCount * topCount; i < len; ++i) this.widgetPositions[i] = 0;

        $.extend(this.widgetPositions, {
            leftCount: leftCount,
            topCount: topCount,
            leftOffset: leftOffset,
            topOffset: topOffset
        });
    }

    /**
     * 根据当前位置获取在widget位置数组中的位置
     * @param {Array} pos
     * @returns {number}
     * @private
     */
    PageItem.prototype._getWidgetPosition = function (pos) {
        if (!pos || pos.length != 2 || !pos[0] || !pos[1]) return -1;
        var index = Math.floor(pos[0] / this.widgetPositions.leftOffset) + Math.floor(pos[1] / this.widgetPositions.topOffset) * this.widgetPositions.leftCount;
        return index;
    }

    /**
     * 获得变换矩阵
     * @param {jQuery} ele
     * @return {Array}
     */
    PageItem.prototype.getTransformMatrix = function (ele) {
        var matrix = [1, 0, 0, 1, 0, 0];
        if (ele) {
            var matrixTransform = ele.css('-webkit-transform');
            if (matrixTransform && matrixTransform != 'none')  matrix = matrixTransform.substr(7, matrixTransform.length - 8).split(",");
        }
        return matrix;
    };

    /**
     * 加载笔迹
     * @private
     */
    PageItem.prototype._loadPainting = function () {
        var paintCanvas = this.getPaintCanvas();
        var self = this;

        var strokeFilePath = path.join(this.provider.getUserBookDir(), this.pageIndex + '/' + this.pageIndex + '.stroke.base64');
        strokeFilePath = path.trimPosix(strokeFilePath);

        var strokeFileExist = shellcmd.isFileExistSync(strokeFilePath);
        if (strokeFileExist) {
            var strokeBase64 = shellcmd.readFile(strokeFilePath);
            var ctx = paintCanvas.getContext('2d');
            var image = new Image();
            image.src = strokeBase64;
            image.onload = function () {
                ctx.clearRect(0, 0, self.provider.bookImageSize.width, self.provider.bookImageSize.height);
                ctx.drawImage(image, 0, 0);
                self._setStrokeLoaded(true);
                $(self).trigger('strokeLoaded');
            }
        } else {
            self._setStrokeLoaded(true);
        }
    };

    /**
     * 绘制笔迹
     * @param {HTMLCanvasElement} canvas
     * @param {Bound} boundd 原始画布中的边界信息
     */
    PageItem.prototype.paintStroke = function (canvas, bound, isStrokeChanged) {
        var paintCanvas = this.getPaintCanvas();
        if (isStrokeChanged) {
            this.isStrokeChanged = isStrokeChanged;
            var w = this.provider.bookImageSize.width;
            var h = this.provider.bookImageSize.height;
            var ctx = paintCanvas.getContext('2d');
            ctx.clearRect(0, 0, w, h);
            ctx.drawImage(canvas, bound.left, bound.top, bound.width, bound.height, 0, 0, w, h);
        }
        $(paintCanvas).show();
    };

    /**
     * 恢复笔迹画布
     * @param {HTMLCanvasElement} canvas
     * @param {Bound} boundd 原始画布中的边界信息
     */
    PageItem.prototype.restoreStroke = function (canvas, bound) {
        var w = this.provider.bookImageSize.width;
        var h = this.provider.bookImageSize.height;

        var self = this;
        var paintCanvas = self.getPaintCanvas();
        var ctx = canvas.getContext('2d');

        var strokeLoaded = self.getStrokeLoaded();
        if (strokeLoaded) {
            ctx.drawImage(paintCanvas, 0, 0, w, h, bound.left, bound.top, bound.width, bound.height);
        } else {
            $(self).bind('strokeLoaded', function () {
                ctx.drawImage(paintCanvas, 0, 0, w, h, bound.left, bound.top, bound.width, bound.height);
                $(self).unbind('strokeLoaded');
            })
        }
        $(paintCanvas).hide();
        if (AppCfg.useTurnJSDoublePage) {
            $(canvas).css("zIndex", 1000);
        }
    };

    /**
     * 显示所有热区
     */
    PageItem.prototype.showAllHotspots = function () {
        var hotspotMgr = this.getHotspotMgr();
        hotspotMgr && hotspotMgr.showAllHotspots();
    }

    /**
     * 隐藏所有热区
     */
    PageItem.prototype.hideAllHotspots = function () {
        var hotspotMgr = this.getHotspotMgr();
        hotspotMgr && hotspotMgr.hideAllHotspots();
    };

    /**
     * 清空本页资源
     */
    PageItem.prototype.clearCurrentPageRes = function () {

        this.player && this.player.releaseRes();

        $.each(this.widgets, function (i, e) {
            e.destroy(false);
        });

        var userPageXaml = path.trimPosix(path.join(ebkContext.getUserBookPath(), this.pageIndex + ""));
        var args = {
            "filter": ".base64",
            "dirs": userPageXaml
        };
        shellcmd.clearDir(JSON.stringify(args));
        if (this.widgets.length) {
            this.widgets = [];
            for (var i = 0, len = this.widgetPositions.length; i < len; ++i) this.widgetPositions[i] = 0;
            this._isModify = true;
        }
    }

    /**
     * 获取本页资源 去除幕布
     */
    PageItem.prototype.getCurrentPageRes = function(){
        var tmpWidgets = [];
        var widgets = this.widgets;
        for(var i = 0;i < widgets.length;i++){
            if(widgets[i].WidgetType != "fadeCurtain"){
                tmpWidgets.push(widgets[i]);
            }
        }
        return tmpWidgets;
    }

    /**
     * 导入时重新加载widgets
     */
    PageItem.prototype.importBookRes = function () {
        var _self = this;
        this._getPageObj(this.pageIndex).done(function (ret) {
            _self.pageObject = ret;
            if (_self.pageObject.absolutelayout != null) {
                _self._loadWidget(_self.pageObject.absolutelayout);
                _self._isModify = true;
            }
        });
    }

    PageItem.prototype.loadBlackOverlay = function(){
        var _DOM = $('<div class="itemOverlay" style="display:none;height:100%;width:100%;position: absolute;top: 0px;z-index:0;background:#000;opacity: 0.4;"></div>');
        this.content.append(_DOM);
    };

    exports.PageItem = PageItem;
});