/**
 * 热区管理模块，处理热区的绘制渲染
 * Date: 2013/06/16
 * Author: yuwang@iflytek.com
 * @version 1.0
 * @module app/hotspot/hotspotmgr
 */
define(function (require, exports) {

    require('jquery');
    var _define = require('../../define');
    var _model = require('../../model');

    var LanguageType = {
        HanYu: 0,
        Uyghur: 1
    }
    /**
     * 热区线条颜色
     * @type {string}
     * @const
     */
    var STROKE_SELECTED_STYLE = 'FF0000',
        STROKE_OUTLINE_STYLE = 'FF9900';

    /**
     * 热区线条宽度
     * @type {number}
     * @const
     */
    var STROKE_SELECTED_SIZE = 3,
        STROKE_OUTLINE_SIZE = 1.5;

    /**
     *
     * @type {number}
     * @const
     */
    var HOTSPOT_CLICK_DISTANCE = 10;

    /**
     * @enum
     * @type {{Hidden: number, MouseOver: number, Selected: number, ShowOutline: number}}
     */
    var HotspotState = {
        Hidden: 0,
        MouseOver: 1,
        Selected: 2,
        ShowOutline: 3
    }

    /**
     * 热区管理类
     * @param {HTMLCanvasElement} canvas 热区画布
     * @param {Array.<_model.HotspotMeta>} hotspotMetaArr 热区元数据数组
     * @constructor
     */
    function HotspotMgr(canvas, hotspotMetaArr) {
        var _hotspots = [];
        var _canvas = canvas;
        var _ctx = _canvas.getContext('2d');
        var _self = this;
        var _mouseDownP;

        var _hotspotToolBar = null;
        this.setHotspotToolBar = function (hotspotToolBar) {
            _hotspotToolBar = hotspotToolBar;
        }
        var _isDoubleLanguage = false;
        this.setIsDoubleLanguage = function (isDouleLanguage) {
            _isDoubleLanguage = isDouleLanguage;
        }

        function _init() {
            _ctx.strokeStyle = STROKE_OUTLINE_STYLE;
            _ctx.lineWidth = STROKE_SELECTED_SIZE;

            _initHotspots(_ctx, hotspotMetaArr);
            _bindEvents();
        }

        this.dispose = function () {
            _unbindEvents();

            if (HotspotMgr._currMgr == _self) {
                HotspotMgr._currMgr = null;
            }
            if (HotspotMgr._currSelectedMgr == _self) {
                HotspotMgr._currSelectedMgr = null;
            }
            $.each(_hotspots, function () {
                this.dispose();
            });
            _hotspots = [];
            _ctx = null;
            _canvas = null;

            //console.log('HotspotMgr dispose');
        }

        function _bindEvents() {
            $(_canvas).bind('mousemove', _onMouseMove);
            $(_canvas).bind('mousedown', _onHotspotMouseDown);
            $(_canvas).bind('mouseup', _onHotspotMouseUp);
        }

        function _unbindEvents() {
            $(_canvas).unbind('mousemove');
            $(_canvas).unbind('mousedown');
            $(_canvas).unbind('mouseup');

        }

        function _onMouseMove(e) {

            if (!HotspotMgr._isEditing) {
                var point = new _model.Point(e.offsetX, e.offsetY);
                var hotspot = _getHotspot(point);

                var currHotspot = HotspotMgr._currHotspot;

                if (hotspot) {
                    if (currHotspot !== hotspot) {

                        if (currHotspot && currHotspot.getState() !== HotspotState.Selected) {
                            currHotspot.setState(HotspotMgr.isShowAllHotspot ? HotspotState.ShowOutline : HotspotState.Hidden);
                        }

                        if (hotspot.getState() !== HotspotState.Selected) {
                            hotspot.setState(HotspotState.MouseOver);
                        }

                        if (HotspotMgr._currMgr) {
                            HotspotMgr._currMgr.reRender();
                        }

                        HotspotMgr._currHotspot = hotspot;
                        HotspotMgr._currMgr = _self;

                        _self.reRender();
                    }
                } else {
                    if (currHotspot) {
                        if (currHotspot.getState() !== HotspotState.Selected) {
                            currHotspot.setState(HotspotMgr.isShowAllHotspot ? HotspotState.ShowOutline : HotspotState.Hidden);
                        }

                        HotspotMgr._currHotspot = null;
                        if (HotspotMgr._currMgr) {
                            HotspotMgr._currMgr.reRender();
                        }

                        _self.reRender();
                    }
                }
            }
        }

        function _onHotspotMouseDown(e) {
            if (!HotspotMgr._isEditing) {
                var currHotspot = HotspotMgr._currHotspot;
                if (currHotspot) {
                    var p = new _model.Point(e.offsetX, e.offsetY);
                    if (currHotspot.hitTest(p)) {
                        _mouseDownP = p;
                    }
                }
            }

        }

        function _onHotspotMouseUp(e) {
            if (!HotspotMgr._isEditing) {
                var currHotspot = HotspotMgr._currHotspot;

                //显示热区工具栏
                function showHotspotToolBar() {
                    $(".hotspotPopToolBar").hide();

                    var bound = currHotspot.getBound();
                    var popLanguageToolBarLeft = bound.left + (bound.right - bound.left) / 2 - 40;
                    var popLanguageToolBarTop = bound.bottom + 10;
                    _hotspotToolBar.css("left", popLanguageToolBarLeft);
                    _hotspotToolBar.css("top", popLanguageToolBarTop);
                    _hotspotToolBar.css("zIndex", 100);
                    _hotspotToolBar.show();
                }

                //给热区工具栏上的按钮绑定点击事件
                function bindCommandsToLanguageButtons() {
                    var btnChinese = _hotspotToolBar.children().children()[0];
                    var btnUyghur = _hotspotToolBar.children().children()[1];
                    $(btnChinese).unbind();
                    $(btnUyghur).unbind();
                    $(btnChinese).click(function () {
                        currHotspot.play(LanguageType.HanYu);
                        bindCommandsToLanguageButtons();
                    });
                    $(btnUyghur).click(function () {
                        currHotspot.play(LanguageType.Uyghur);
                        bindCommandsToLanguageButtons();
                    });
                }

                if (currHotspot) {
                    var p = new _model.Point(e.offsetX, e.offsetY);
                    if (currHotspot.hitTest(p)) {
                        if (!_mouseDownP || !p) return;
                        var distance = p.distance(_mouseDownP || p);
                        if (distance < HOTSPOT_CLICK_DISTANCE) {
                            if (_isDoubleLanguage == true) {
                                //弹出选择语言工具栏
                                showHotspotToolBar();
                                bindCommandsToLanguageButtons();
                            }
                            currHotspot.play(LanguageType.HanYu);
                            HotspotMgr.setSelectedHotspot(currHotspot, _self);
                        }
                    }
                    else {
                        if (HotspotMgr._selectHotsopt)
                            HotspotMgr._selectHotsopt.render();
                    }
                }
            }

        }

        function _clearRender() {
            _ctx.clearRect(0, 0, _canvas.width, _canvas.height);
        }

        function _initHotspots(ctx, hotspotMetaArr) {
            // console.log('hotspotMetaArr len:%d', hotspotMetaArr.length);
            for (var i = 0, len = hotspotMetaArr.length; i < len; i++) {
                var hotspotMeta = hotspotMetaArr[i];

                var hotspot;

                if (hotspotMeta.type === _define.HotspotType.Rectangle) {
                    hotspot = new RectHotspot(ctx, hotspotMeta);
                } else if (hotspotMeta.type === _define.HotspotType.Ellipse) {
                    hotspot = new EllipseHotspot(ctx, hotspotMeta);
                } else if (hotspotMeta.type === _define.HotspotType.Path) {
                    hotspot = new PathHotspot(ctx, hotspotMeta);
                } else {
                    throw new Error("not support hotSpot type:" + hotspotMeta.type);
                }

                if (app_global_playerMgr.isCurAudioBeHotspot && HotspotMgr._selectHotsopt && HotspotMgr._selectHotsopt.getAudioId() === hotspot.getAudioId()) {
                    HotspotMgr.setSelectedHotspot(hotspot, _self);
                } else if (HotspotMgr.isShowAllHotspot) {
                    hotspot.setState(HotspotState.ShowOutline);
                }

                _hotspots.push(hotspot);
            }
        }

        /**
         *
         * @param {model.Point} point
         * @returns {HotspotBase}
         */
        function _getHotspot(point) {

            var hitspotArr = [];
            for (var i = 0, len = _hotspots.length; i < len; i++) {
                if (_hotspots[i].hitTest(point)) {
                    hitspotArr.push(_hotspots[i]);
                }
            }

            var hotspot = hitspotArr[0];
            for (i = 1, len = hitspotArr.length; i < len; i++) {
                if (hitspotArr[i].getZIndex() > hotspot.getZIndex()) {
                    hotspot = hitspotArr[i];
                }
            }

            return hotspot;
        }

        this.clear = function () {
            _clearRender();
        }

        /**
         * 热区重绘
         */
        this.reRender = function () {
            _clearRender();
            $.each(_hotspots, function () {
                this.render();
            })
        }

        /**
         * 显示所有热区
         */
        this.showAllHotspots = function () {
            HotspotMgr.isShowAllHotspot = true;

            $.each(_hotspots, function () {
                var state = this.getState();
                if (state === HotspotState.Hidden) {
                    this.setState(HotspotState.ShowOutline);
                }
            })

            _self.reRender();
        }

        /**
         * 影藏所有热区
         */
        this.hideAllHotspots = function () {
            HotspotMgr.isShowAllHotspot = false;

            $.each(_hotspots, function () {
                var state = this.getState();
                if (state === HotspotState.ShowOutline) {
                    this.setState(HotspotState.Hidden);
                }
            })

            _self.reRender();
        }

        this.setIsEditing = function (isEditing) {
            if (isEditing == HotspotMgr._isEditing) return;
            HotspotMgr._isEditing = isEditing;
        }
        _init();
    }

    /**
     * 当前选中的热区
     * 注意：该属性仅在 HotspotMgr.setCurrentHotspot中赋值，其它地方只能使用，不能赋值
     * @type {HotspotBase}
     * @private
     * @static
     */
    HotspotMgr._currHotspot = null;

    /**
     * 当前选中的热区
     * 注意：该属性仅在 HotspotMgr.setPlayHotspot中赋值，其它地方只能使用，不能赋值
     * @type {HotspotBase}
     * @private
     * @static
     */
    HotspotMgr._selectHotsopt = null;

    /**
     * 当前生效的热区管理类, 每个PageItem中都有一个热区管理类
     * @type {HotspotMgr}
     * @private
     * @static
     */
    HotspotMgr._currMgr = null;

    /***
     * 当前被选中的热区管理类（即当前被选中热区所在的热区管理类）
     * @type {HotspotMgr}
     * @private
     * @static
     */
    HotspotMgr._currSelectedMgr = null;

    /**
     * 当前是否是显示全部热区状态
     * @type {boolean}
     * @static
     */
    HotspotMgr.isShowAllHotspot = false;

    /**
     * 设置当前热区
     * @param hotspot
     * @static
     */
    HotspotMgr.setSelectedHotspot = function (hotspot, hotspotMgr) {
        if (HotspotMgr._selectHotsopt !== hotspot) {
            var oldHotspot = HotspotMgr._selectHotsopt;

            // 重置前一个热区的状态
            if (oldHotspot) {
                oldHotspot.setState(HotspotMgr.isShowAllHotspot ? HotspotState.ShowOutline : HotspotState.Hidden)
            }

            // 设置当前播放
            if (hotspot) {
                hotspot.setState(HotspotState.Selected)
            }
            HotspotMgr._selectHotsopt = hotspot;

            // 清除前一次热区显示
            if (HotspotMgr._currSelectedMgr) {
                HotspotMgr._currSelectedMgr.reRender();
            }
            HotspotMgr._currSelectedMgr = hotspotMgr;
            //此处注释掉，以修复翻页和单双页切换时音频按钮状态不对的问题-------by luxing  2013-11-06
            //_hotspotMgrEventDispatcher.trigger('playHotspotChanged', {oldHotspot: oldHotspot, newHotspot: hotspot});

        }
    }


    /**
     * 编辑状态
     */
    HotspotMgr._isEditing = false;

    /**
     * @constructor
     */
    function HotspotBase() {

        /**
         * @return { number }
         */
        this.getState = function () {
            return  this._state || HotspotState.Hidden;
        };

        this.dispose = function () {

        };

        /**
         *
         * @param state
         */
        this.setState = function (state) {
            this._state = state;
        }

        /**
         * 获取热区原始数据信息
         * @return {*}
         */
        this.getContext = function () {
            throw new Error('getContext need implement');
        }

        /**
         * 获取热区原始数据信息
         * @return {model.HotspotMeta}
         */
        this.getHotspotMeta = function () {
            throw new Error('getHotSpotMeta need implement');
        };

        /**
         * 获取音频ID
         * @return {String}
         */
        this.getAudioId = function () {
            return  this.getHotspotMeta().audio;
        };

        /**
         * 获取Z轴索引
         * @return {Number}
         */
        this.getZIndex = function () {
            return  this.getHotspotMeta().zIndex;
        };

        /**
         * 获取热区图形边界信息
         * @return {model.Bound}
         */
        this.getBound = function () {
            throw new Error('getBound need implement');
        };

        /**
         *
         * @private
         */
        this._render = function () {
            throw  new Error('_render need implement');
        }

        /**
         * 判断点击是否是在热区图形区域内
         * @param {Point | _model.Point}  point  测试点
         * @return {boolean}
         */
        this.hitTest = function (point) {
            throw new Error('hitTest need implement');
        };

        /**
         * 渲染热区图形
         */
        this.render = function () {
            var state = this.getState();

            if (state == HotspotState.Hidden) {
                return;
            }

            var ctx = this.getContext();
            if (state === HotspotState.MouseOver || state === HotspotState.Selected) {
                ctx.strokeStyle = STROKE_SELECTED_STYLE;
                ctx.lineWidth = STROKE_SELECTED_SIZE;
            }
            else {
                ctx.strokeStyle = STROKE_OUTLINE_STYLE;
                ctx.lineWidth = STROKE_OUTLINE_SIZE;
            }

            this._render();
        };
    }

    /**
     * 播放热区
     */
    HotspotBase.prototype.play = function (languageType) {
        var audioId = this.getAudioId();
        if (typeof languageType == "undefined") {
            languageType = LanguageType.HanYu;
        }
        if (HotspotBase.player) {
            HotspotBase.player.play(audioId, languageType);
        } else {
            console.warn('HotspotBase::play, not found hotspot player, audioID:%s', audioId);
        }
    }

    /**
     * 暂停热区播放
     */
    HotspotBase.prototype.pause = function () {
        if (HotspotBase.player) {
            HotspotBase.player.pause()
        } else {
            console.warn('HotspotBase::play, not found hotspot player');
        }
    }


    /**
     * 继续热区播放
     */
    HotspotBase.prototype.resume = function () {
        if (HotspotBase.player) {
            HotspotBase.player.resume()
        } else {
            console.warn('HotspotBase::play, not found hotspot player');
        }
    }


    /**
     * 热区播放器
     */
    HotspotBase.player = null;

    /**
     * 矩形热区
     * @constructor
     * @param {HTMLCanvasElement}  ctx Canvas上下文
     * @param {Object} hotspotMeta 热区数据
     * @extend HotspotBase
     */
    function RectHotspot(ctx, hotspotMeta) {
        var _hotspotMeta = hotspotMeta;

        var _bound = new _model.Bound(_hotspotMeta.left, _hotspotMeta.top, _hotspotMeta.width, _hotspotMeta.height);

        /** @type {HTMLCanvasElement} */
        var _ctx = ctx;

        this._state = HotspotState.Hidden;

        /**
         * @inherit
         */
        this.getContext = function () {
            return _ctx;
        }

        /**
         * @inherit
         */
        this.getHotspotMeta = function () {
            return _hotspotMeta;
        };

        /**
         * @inherit
         */
        this.getBound = function () {
            return _bound;
        };

        /**
         * @inherit
         */
        this.hitTest = function (point) {
            return  _bound.isContainsPoint(point);
        };

        this.dispose = function () {
            this._ctx = null;
        };
    }

    RectHotspot.prototype = new HotspotBase();
    RectHotspot.prototype.constructor = RectHotspot;

    /**
     * @inherit
     */
    RectHotspot.prototype._render = function () {
        var ctx = this.getContext();
        var bound = this.getBound();
        ctx.strokeRect(bound.left, bound.top, bound.width, bound.height);
    };

    /**
     *
     * @param ctx
     * @param hotspotMeta
     * @constructor
     */
    function EllipseHotspot(ctx, hotspotMeta) {
        var _hotspotMeta = hotspotMeta;
        var _ctx = ctx;
        var _bound = new _model.Bound(_hotspotMeta.left, _hotspotMeta.top, _hotspotMeta.width, _hotspotMeta.height);

        this._state = HotspotState.Hidden;

        /**
         * @inherit
         */
        this.getContext = function () {
            return _ctx;
        };

        /**
         * @inherit
         */
        this.getHotspotMeta = function () {
            return _hotspotMeta;
        };

        /**
         * @inherit
         */
        this.getBound = function () {
            return _bound;
        };

        this.dispose = function () {
            this._ctx = null;
        };
    }

    EllipseHotspot.prototype = new HotspotBase();
    EllipseHotspot.prototype.constructor = EllipseHotspot;

    /**
     * @inherit
     */
    EllipseHotspot.prototype.hitTest = function (point) {

        var bound = this.getBound();

        // 将原始点相对于椭圆中心点进行偏移
        var x = point.x - bound.center.x;
        var y = point.y - bound.center.y;

        return Math.pow((x / (bound.width / 2)), 2) + Math.pow((y / (bound.height / 2)), 2) <= 1;
    };

    /**
     * @inherit
     */
    EllipseHotspot.prototype._render = function () {
        var ctx = this.getContext();
        var bound = this.getBound();

        var k = (bound.height / 0.75) / 2;
        var sp = new _model.Point(bound.left, bound.top + bound.height / 2);
        var ep = new _model.Point(bound.right, sp.y);

        ctx.beginPath();
        ctx.moveTo(sp.x, sp.y);
        ctx.bezierCurveTo(sp.x, sp.y + k, ep.x, ep.y + k, ep.x, ep.y);
        ctx.bezierCurveTo(ep.x, ep.y - k, sp.x, sp.y - k, sp.x, sp.y);
        ctx.closePath();
        ctx.stroke();
    }

    /**
     * 路径热区
     * @constructor
     * @param {}  ctx Canvas上下文
     * @param {Object} hotSpotMeta 热区数据
     * @extend HotspotBase
     */
    function PathHotspot(ctx, hotSpotMeta) {
        var _hotSpotMeta = hotSpotMeta;
        var _ctx = ctx;
        var _points = _getPoints(_hotSpotMeta.data);
        var _bound = _getBound(_points);
        this._state = HotspotState.Hidden;

        /**
         * @inherit
         */
        this.getContext = function () {
            return _ctx;
        };

        /**
         * @inherit
         */
        this.getHotspotMeta = function () {
            return _hotSpotMeta;
        };

        /**
         * @inherit
         */
        this.getBound = function () {
            return _bound;
        };

        /**
         *
         * @returns {Array.<model.Point>}
         */
        this.getPoints = function () {
            return _points;
        };

        /**
         * 获取数据点
         * @param dataStr
         * @return {Array.<model.Point>}
         * @private
         */
        function _getPoints(dataStr) {
            var pointStrArray = dataStr.replace(/[MLZ]/ig, '').split(' ');
            var points = [];
            for (var i = 0, pLen = pointStrArray.length; i < pLen; i++) {
                var pointInfo = pointStrArray[i].split(',')
                points.push(new _model.Point(parseInt(pointInfo[0]), parseInt(pointInfo[1])));
            }

            return  points;
        };

        /**
         * 获取图形边界信息
         * @param {Array.<Point>} points
         * @returns {model.Bound}
         * @private
         */
        function _getBound(points) {

            var left = points[0].x;
            var top = points[0].y;
            var right = left;
            var bottom = top;

            for (var i = 1, len = points.length; i < len; i++) {
                var point = points[i];
                if (point.x < left) {
                    left = point.x;
                } else if (point.x > right) {
                    right = point.x;
                }

                if (point.y < top) {
                    top = point.y;
                } else if (point.y > bottom) {
                    bottom = point.y;
                }
            }
            var bound = new _model.Bound(left, top, right - left, bottom - top);
            return bound;
        };

        this.dispose = function () {
            this._ctx = null;
        };
    }

    PathHotspot.prototype = new HotspotBase();
    PathHotspot.prototype.constructor = PathHotspot;

    /**
     * @inherit
     */
    PathHotspot.prototype.hitTest = function (point) {
        var x = point.x, y = point.y;
        var points = this.getPoints();
        var inside = false;
        for (var i = 0, j = points.length - 1, len = points.length; i < len; j = i++) {
            var xi = points[i].x, yi = points[i].y;
            var xj = points[j].x, yj = points[j].y;

            var intersect = ((yi > y) != (yj > y))
                && (x < (xj - xi) * (y - yi) / (yj - yi) + xi);
            if (intersect) inside = !inside;
        }

        return inside;
    };

    /**
     * @inherit
     */
    PathHotspot.prototype._render = function () {
        var ctx = this.getContext();
        var points = this.getPoints();
        ctx.beginPath();
        ctx.moveTo(points[0].x, points[0].y)
        for (var i = 1, len = points.length; i < len; i++) {
            ctx.lineTo(points[i].x, points[i].y);
        }
        ctx.lineTo(points[0].x, points[0].y)
        ctx.stroke();
    };

    /**
     * 热区管理类
     * @export {app.hotspot.HotspotMgr}
     */
    exports.HotspotMgr = HotspotMgr;

    exports.init = function (hotspotPlayer) {
        HotspotBase.player = hotspotPlayer;
    };
});