/**
 * 书本容器模块
 * Author: yuwang@iflytek.com
 * Date: 2013/06/19
 */
define(function (require, exports) {
    require('jquery');
    var _SlideUtil = require('../../common/slideUtil');
    var _define = require('../define');
    var _bookview = {
        TurnDoublePageView: require('./bookview/turndoublepageview').TurnDoublePageView,
        SimpleDoublePageView: require('./bookview/simpledoublepageview').SimpleDoublePageView,
        SimpleSinglePageView: require('./bookview/simplesinglepageview').SimpleSinglePageView,
        WhiteBoardView : require("./bookview/whiteboardview").WhiteBoardView
    }
    var _model = require('../model');
    var _pageProvider = require('./pageprovider').Instance;
    var CanvasManager = require('../paint/canvasmgr');
    var _canvasManager = CanvasManager.Instance;
    var trace = require('../trace').Instance;
    var shellcmd = require('../../common/shellcmd');
    var commonfunc = require('../../common/commonfunc');
    var ebkContext = require('../ebookcontext').Instance;
    var path = require('../../common/path');
    var _taskQueue = require('../../common/taskQueue').Instance;
    var _pagePreViewIcon = require('./preview/pagepreviewicon').Instance;
    var _WBViewIcon = require('./preview/whiteboardpreview').Instance;
    require('zebra_dialog');
    var _containerhelper = require("./containerhelper");
    var MatrixData = _containerhelper.MatrixData;
    var BookViewInfo = _containerhelper.BookViewInfo;
    var confirmExit = require('./common/confirmExit');
    var cardWrapper = require("../card/cardwrapper").Instance;
    var smartBar =  require('./smartbar/smartbar').Instance;

    /**
     * 课本容器类
     * 管理三种视图间的切换，实现缩放、拖动功能，
     * 转发翻页热区绘制
     * @param {jQuery} contain
     */
    function BookContainer(contain, confirmExit) {
        var _bookViews = {};
        _bookViews[_define.PageViewType.TurnJsDoublePage] = new BookViewInfo();
        _bookViews[_define.PageViewType.SimpleDoublePage] = new BookViewInfo();
        _bookViews[_define.PageViewType.SimpleSinglePage] = new BookViewInfo();
        _bookViews[_define.PageViewType.WhiteBoardView] = new BookViewInfo();

        var _bookContainer = contain;
        var _confirmExit = confirmExit;

        if (_bookContainer.length < 1) {
            throw new Error('invalid bookContainer');
        }

        /**
         * 放大最大比例
         */
        var MAX_ZOOM_RATE = Math.pow(1.2, 5);

        /**
         * 放大最小比例
         */
        var MIN_ZOOM_RATE = 1;

        /**
         * 获取ViewDOM Transform数据正则表达式
         */
        var TRANSFORM_REGEXP = /matrix\((.*)\)/;

        /** @type {BookContainer} */
        var _self = this;

        /** @type {define.PageViewType} */
        var _pageViewType = _define.PageViewType.TurnJsDoublePage;

        /** @type {BookViewBase}*/
        var _currView;

        /** @type {number} 当前页面索引 */
        var _pageIndex;
        /** @type {number} 当前白板索引 */
        var _wbIndex = 1,_maxWbIndex = 10;

        /** @type {number} 双页缩放比例 */
        var _doublePageZoomRate = 1;

        /** @type {number} 单页缩放比例 */
        var _singlePageZoomRate = 1;

        /**
         * 容器边界信息
         */
        var _containerBound = new _model.Bound(0, 0, _bookContainer.width(), _bookContainer.height());

        /** @type {boolean} 是否是拖动状态 */
        var _isDragging;

        /** @type {Point | _model.Point} 拖动起始点 */
        var _dragSrcPoint;

        /** @type {boolean} 是否正在描绘笔迹 */
        var _isPainting = false;

        var _isEditing = false;

        _self.bookMenu = null;
        _self.bookCode = '';
        _self.bookMetaInfo = null;

        _self._slideUtil = null;

        /**
         * 转发视图的beforePageChange事件
         */
        function _onBeforePageChange(e, indexInfo) {
            $(_self).trigger('beforePageChange', indexInfo);
        }

        /**
         * 转发视图的pageChanged事件
         */
        function _onPageChanged(e, indexInfo) {
            $(_self).trigger('pageChanged', indexInfo);
        }

        /**
         * 转发事件
         *
         * Caution : 这个函数很重要。自定义事件是不会冒泡的。
         * 由于存在双页视图的缘故，绘制笔迹的canvas和分页保存笔迹的canvas不是统一的
         * 这个函数主要用于canvasmgr和viewDom通信进行笔迹的保存和绘制
         *
         * 保存过程 canvasmgr ---> container ---> viewDom ---> Item.canvas ---> XXX.stroke.base64
         * 绘制过程刚好相反
         */
        function _transmitEvent() {
            $(_currView).unbind('beforePageChange')
                .unbind('pageChanged')
                .bind('beforePageChange', _onBeforePageChange)
                .bind('pageChanged', _onPageChanged) ;
        }

        /**
         * 设置当前页码
         * @param {number} pageIndex
         *
         * Caution : 这里的pageIndex参数有点危险，当心误改。
         * 这里利用js的传值特性把pageIndex的值返回
         * 有点low & ugly ，但暂时不改动
         *
         *              --------  by yczhu
         */
        _self._setPageIndex = function (pageIndex) {
            var isOldDiffWithNew = true;
            if(_pageViewType != _define.PageViewType.WhiteBoardView){
                //电子书翻页
                _pagePreViewIcon.initPages(_pageProvider.bookPages);
                pageIndex = _self._correctPageIndex(pageIndex);
                _pagePreViewIcon.setIcon(pageIndex);
                if (_pageIndex != pageIndex) {
                    _pageIndex = pageIndex;
                }
                else isOldDiffWithNew = false;
            }
            else{
                //白板页面暂时不管icon，纠正白板页码交由视图处理
                (pageIndex > _maxWbIndex) && ( pageIndex = _maxWbIndex);
                (pageIndex <= 0) && ( pageIndex = 1);
                _wbIndex = pageIndex;
                _WBViewIcon.setWBIcon(pageIndex);
            }
            return isOldDiffWithNew;
        };

        /**
         * 获取当前页码
         * @returns {number}
         */
        _self.getPageIndex = function () {
            return _pageIndex;
        };

        /**
         * 获取当前白板页码
         * @returns {number}
         */
        _self.getWbIndex = function () {
            return _wbIndex;
        };

        /**
         *  全局回调事件
         *  由browserContext转发
         *              --------- by yczhu
         */
        _self.onMessageReceived = function (obj) {
            if (obj.evt == 'mtbkunzip_end') {
                if (obj.data == _self.bookCode) {
                    _self.refresh();
                    $.Zebra_Dialog(AppTips.NewResIsLoadedMsg, {
                        'type': 'information',
                        'title': AppTips.WarmTips,
                        'buttons': [AppTips.OkTips],
                        'onClose': function (capition) {
                            if (capition == AppTips.OkTips) {

                            }
                        }
                    });
                }
            }

            if (obj.evt == "change_pageIndex") {
                if (obj.data.index) {
                    _self.setPageIndex(obj.data.index);
                }
            }

            if (obj.evt == "videocap") {
                console.log("come to videocap button solve");
                var capJson = '{"result":["' + obj.data + '"]}';
                var capOjb = JSON.parse(capJson);
                _self.insertWidget(capOjb.result,{
                    userName:ebkContext.getUser().getUserName() || ebkContext.getUser().getName(),
                    uploadTime:new Date().toLocaleString()
                });
            }


            if (obj.evt == "showvideocap") {
                //console.log(obj.data);
                var innerObj = $('<div>').append($('<image src="' + obj.data + '">').css({
                    'max-width': '100%',
                    'max-height': '100%',
                    'verticalAlign': 'middle'
                }));
                var actualHeight= Math.min($(document).height(),Config["Height"]);
                innerObj.css({textAlign: 'center', lineHeight: actualHeight + 'px'});
                cardWrapper.show(innerObj[0],"",true);
            }

            if (obj.evt == "get_pageIndex") {
                var curPageIndex = _self.getPageIndex();
                curPageIndex += "";
                var curBookCode = _self.bookCode;
                //配合电子书包项目，做的一些控制学生端的事情
                //callClient('ebagcmd', 'teachersyncturnpage', this.bookCode+"|"+curPageIndex, 0);
            }
            if (obj.evt == "get_openedWidgetId") {
                var curPageIndex = _self.getPageIndex();
                curPageIndex += "";
                var curBookCode = _self.bookCode;
                var curCardId = trace.getCurCardId();
                //配合电子书包项目，做的一些控制学生端的事情
            }

            if (obj.evt == 'saveResAndPaint') {
                if (!_isEditing) {
                    if (_canvasManager.getStrokeCanvas()) {
                        _self.paintStrokeWithoutDrag(_canvasManager.getStrokeCanvas()[0], true);
                    }
                }
                _self.dispose();
            }

            //响应上传事件
            if (obj.evt == 'uploadcallback') {
                console.log(obj.data);
                var result = obj.data.split("|");
                //清除单元名中的时间戳 unit_ddddddd
                var unitName = result[1].substr(0, result[1].lastIndexOf('_'));

                if (result[0] == "true") {
                    _taskQueue.doTask({time: 5000, status: "success", msg: unitName});
                }
                else {
                    _taskQueue.doTask({time: 3000, status: "fail", msg: unitName});
                }
            }
            if (obj.evt == "appclosing") {
                var _pageNavigatorHelper = require('../../common/pagenavigatorhelper');
                _pageNavigatorHelper.ExitAlert(_self, _canvasManager);
            }
            //屏幕分辨率发生改变
            if (obj.evt == 'displaychange') {
                _confirmExit.refreshIndex();
            }

            if(obj.evt == 'audiorecord'){
                console.log(obj.evt);
                console.log( obj.data);
                var capJson = '{"result":["' + obj.data + '"]}';
                var capOjb = JSON.parse(capJson);
                _self.insertWidget(capOjb.result,{
                    userName:ebkContext.getUser().getUserName() || ebkContext.getUser().getName(),
                    uploadTime:new Date().toLocaleString()
                });
            }

            if(obj.evt == 'resstartupload'){
                var uploadManagerexe = Config.binPath + 'iFlybookUploadMananger.exe';
                shellcmd.shellexecutewithargs(uploadManagerexe,AppRemoteSvcCfgBase.CYAPI);
            }

            if (obj.evt == 'closeApp'){
                confirmExit.ExitAlert(null, null);
            }

            if(obj.evt == 'screenshot'){
                var fullPath = obj.data;
                tipsDialog({
                    "id" : "snapshot" + new Date().getTime(),
                    "content" : "快照截图成功",
                    "action" : "openLocal",
                    "actionName" : "查看文件夹",
                    "callback" : function(){
                        fullPath = fullPath.replace(/\\\\/g,'\\');
                        shellcmd.openLocalFolder(fullPath);
                    }
                });
            }
        };

        /**
         * 获取当前课本视图类别
         * @returns {define.PageViewType}
         */
        _self.getPageViewType = function () {
            return _pageViewType;
        };

        /**
         * 加载书本
         * 基本可以理解为是加载目录和当前页码
         * @param {string} bookCode 书本编号
         * @param {number} pageIndex 页码
         */
        _self.loadPage = function (bookCode, pageIndex) {
            var def = $.Deferred();
            try {
                _pageProvider.loadBook(bookCode).done(function () {
                    _self.bookMenu = _pageProvider.bookMenu;
                    _self.bookCode = _pageProvider.bookCode;
                    _self._setPageIndex(pageIndex);
                    document.getElementById("bookTitle").innerHTML = _pageProvider.bookObj.IndexInfo.Title;  //设置课本标题  -xjzhou 2013年8月27日10:44:59
                    _self.bookMetaInfo = new _model.Book(_pageProvider.bookCode, _pageProvider.bookObj.IndexInfo);
                    def.resolve();
                });
            } catch (ex) {
                //出错统一交由上层处理 ，最终传递至index
                throw new Error("加载课本出错");
                def.resolve();
                return def.promise();
            }
            return def.promise();
        };

        /**
         * @param {define.PageViewType} pageViewType 书本视图类别
         * @param {number} pageIndex
         * @return {BookViewBase}
         */
        function _getPageView(pageViewType, pageIndex) {
            var pageView;
            switch (pageViewType) {
                case _define.PageViewType.TurnJsDoublePage:
                    pageView = new _bookview.TurnDoublePageView(_bookContainer, _pageProvider, pageIndex);
                    break;
                case _define.PageViewType.SimpleDoublePage:
                    pageView = new _bookview.SimpleDoublePageView(_bookContainer, _pageProvider);
                    break;
                case _define.PageViewType.SimpleSinglePage:
                    pageView = new _bookview.SimpleSinglePageView(_bookContainer, _pageProvider);
                    break;
                case _define.PageViewType.WhiteBoardView:
                    pageView = new _bookview.WhiteBoardView(_bookContainer, _pageProvider);
                    break;
                default:
                    throw  new Error('unknown pageViewType:' + pageViewType);
                    return null;
            }
            return pageView;
        }

        /**
         * 切换书本视图
         * @param {define.PageViewType | string} pageViewType 书本视图类别
         */
        _self.switchPageView = function (pageViewType) {
            var deferred = $.Deferred();
            if (_currView) {
                $(_self).trigger('beforeBookViewChange', {oldPageType: _pageViewType, newPageType: pageViewType});
            }

            if (_currView && _pageViewType === pageViewType) {
                return;
            }

            if (!_bookViews.hasOwnProperty(pageViewType)) {
                throw  new Error('unknown pageViewType "' + pageViewType + '"');
            } else {
                _pageProvider.unloadAllPage();
                // 清除当前视图，并保存当前视图Transform数据
                if (_currView) {
                    $(_currView).unbind('beforePageChange');
                    _bookViews[_pageViewType].transformData = _getTransformData();
                    _currView.getViewDOM().remove();
                }
                // 重新加载新视图
                _bookViews[pageViewType].pageView = _getPageView(pageViewType, _pageIndex);
                ebkContext.setWBEB(pageViewType ==  _define.PageViewType.WhiteBoardView ? "WB" : "EB");
                //设置是否编辑
                if (_isEditing) _bookViews[pageViewType].pageView.setIsEditing(_isEditing);
            }

            _currView = _bookViews[pageViewType].pageView;
            // 设置变换
            //_setTransform(_bookViews[pageViewType].transformData);
            //修复bug：非页面初始大小时切换单双页没有回到页面初始大小
            _setCurrentZoomRate(1);

            // 设置拖动
            if (_isDraggable()) {
                _setBookViewDraggable();
            }

            // 设置手势
            if (_isSwipeEnable()) {
                _setSwipable();
            }
            //_num只做临时校检，无实际意义
            var _num = (pageViewType === _define.PageViewType.WhiteBoardView) ?  _wbIndex :  Math.max(_self.getPageIndex(), _pagePreViewIcon.getNowPage());
            // 设置页码
            _currView.setPageIndex(_num).done(function (newPageIndex) {
                // 显示ViewDOM
                _currView.getViewDOM().show();

                var oldPageViewType = _pageViewType;

                _pageViewType = pageViewType;
                //如果初始的页面视图是turnjs双页，则是启动程序时，则不用激发事件
                if (oldPageViewType !== _define.PageViewType.TurnJsDoublePage) {
                    $(_self).trigger('bookViewChanged', {oldView: oldPageViewType, newView: _pageViewType});
                }
                _canvasManager.setZoomedEraserSize(1);

                //单页模式下处于最后一页再切换到双页模式下会调整页码
                if (newPageIndex && newPageIndex != _pageIndex) {
                    _pageIndex = newPageIndex;
                    _pagePreViewIcon.setIcon(_pageIndex);
                }
                (pageViewType === _define.PageViewType.WhiteBoardView) && _WBViewIcon.setWBIcon(_wbIndex);
                deferred.resolve();
            });
            return deferred.promise();
        };

        _self.isPageLoading = function () {
            return _currView.isPageLoading;
        }

        //---------------------------------------------------------------------------------------------
        //                                      Page Zoom & Drag
        //---------------------------------------------------------------------------------------------
        /**
         * 获取当前缩放比例
         * @returns {number}
         */
        function _getCurrentZoomRate() {
            return _currView.isDoublePage()? _doublePageZoomRate : _singlePageZoomRate;
        }

        function _setCurrentZoomRate(zoomRate) {
            var isDoublePage = _currView.isDoublePage();
            if (isDoublePage) {
                _doublePageZoomRate = zoomRate;
            } else {
                _singlePageZoomRate = zoomRate;
            }

            if (_isSwipeEnable()) {
                _setSwipable();
            } else {
                _setUnSwipable();
            }
        }

        /**
         * 获取课本视图原始边界信息
         * @returns {_model.Bound}
         */
        function _getBookViewBound() {
            var viewDom = _currView.getViewDOM();
            return new _model.Bound(0, 0, viewDom.width(), viewDom.height());
        }

        /**
         * 获取矩阵变换数据
         * @returns {MatrixData}
         */
        function _getTransformData() {
            var viewDom = _currView.getViewDOM();
            var transform = viewDom.css('-webkit-transform');
            var transformData;
            if (transform !== 'none') {
                var matches = TRANSFORM_REGEXP.exec(transform);
                transformData = new MatrixData(matches[1]);
            } else {
                transformData = new MatrixData();
            }

            return  transformData;
        }

        /**
         * 获取课本视图形变后的边界信息
         * @param {MatrixData=} matrixData
         * @returns {_model.Bound}
         */
        function _getBookViewTransformBound(matrixData) {

            var transformData = matrixData || _getTransformData();

            var zoomRate = transformData.a;

            var bookViewBound = _getBookViewBound();
            var offsetX = bookViewBound.width * (zoomRate - 1) / 2;
            var offsetY = bookViewBound.height * (zoomRate - 1) / 2;

            return new _model.Bound(transformData.tx - offsetX, transformData.ty - offsetY, bookViewBound.width * zoomRate, bookViewBound.height * zoomRate);
        }

        function _zoom(currZoomRate, scale, transformData) {

            var newZoomRate = currZoomRate * scale;
            if (newZoomRate < MIN_ZOOM_RATE) {
                newZoomRate = MIN_ZOOM_RATE;
            } else if (newZoomRate > MAX_ZOOM_RATE) {
                newZoomRate = MAX_ZOOM_RATE;
            }
            var newTransformData = _getZoomTransformData(newZoomRate, transformData);
            //console.log(newTransformData);
            _setTransform(newTransformData);
            _canvasManager.setZoomedEraserSize(newZoomRate);
            _setCurrentZoomRate(newZoomRate);
            $(_self).trigger('zoomRateChanged', {oldZoomRate: currZoomRate, newZoomRate: newZoomRate});
            if (_isDraggable()) {
                _setBookViewDraggable();
            }
        }

        /**
         * 获取缩放变换数据
         * @param {number} newZoomRate 新的缩放比例
         * @param {=object} transformData
         * @returns {MatrixData}
         */
        function _getZoomTransformData(newZoomRate, transformData) {

            var currTransformData = transformData ? transformData : _getTransformData();

            // 处理缩放
            var newTransformData = currTransformData.clone();
            newTransformData.a = newZoomRate;
            newTransformData.d = newZoomRate;

            // 修正边界
            _correctTransformBound(newTransformData);

            return newTransformData;
        }

        /**
         * 修正边界, 让页面缩小或拖动时周边不会出现空白
         * @param transformData
         */
        function _correctTransformBound(transformData) {
            var transformBound = _getBookViewTransformBound(transformData);

            var offsetX = 0, offsetY = 0;

            // 处理左边界
            if (transformBound.left > 0) {
                offsetX = -transformBound.left;
            }
            // 处理右边界
            else if (transformBound.right < _containerBound.right) {
                offsetX = _containerBound.right - transformBound.right;
            }

            // 处理上边界
            if (transformBound.top > 0) {
                offsetY = -transformBound.top;
            }
            // 处理下边界
            else if (transformBound.bottom < _containerBound.bottom) {
                offsetY = _containerBound.bottom - transformBound.bottom;
            }

            transformData.tx += offsetX;
            transformData.ty += offsetY;
        }

        /**
         * 设置课本视图变换
         * @param {MatrixData} transformData
         */
        function _setTransform(transformData) {
            var viewDOM = _currView.getViewDOM();

            transformData.correct();

            viewDOM.css({
                '-webkit-transform': 'matrix(' + transformData + ')',
                '-webkit-transform-origin': '50% 50%'
            });
        }

        function _isDraggable() {
            var currRoomRate = _getCurrentZoomRate();
            // 非绘制笔迹状态下，页面放大时或视图为单页时可拖动
            return !_isPainting && (currRoomRate > MIN_ZOOM_RATE || !_currView.isDoublePage());
        }

        function _setBookViewDraggable() {

            // 防止事件重复注册，先移除拖动相关事件
            _setBookViewUnDraggable();

            var viewDOM = _currView.getViewDOM();
            viewDOM.on('mousedown.containerDraggable', function (e) {
                _isDragging = true;
                _dragSrcPoint = new _model.Point(e.clientX, e.clientY);
            });

            viewDOM.on('mouseup.containerDraggable', function () {
                _isDragging = false;
            });

            viewDOM.on('mouseout.containerDraggable', function () {
                _isDragging = false;
            });

            viewDOM.on('mousemove.containerDraggable', function (e) {
                if (_isDragging) {
                    var currPoint = new _model.Point(e.clientX, e.clientY);
                    var offset = new _model.Offset(currPoint.x - _dragSrcPoint.x, currPoint.y - _dragSrcPoint.y);
                    _self.moveView(offset);
                    _dragSrcPoint = currPoint;
                }
            });
            //鼠标滚轮事件
            viewDOM.on('mousescroll.containerDraggable', function (e,x,y) {

                if(!global_scoll_flag && !global_scoll_flag2 && !global_scoll_flag3){
                    var offset = new _model.Offset(x,y);
                    _self.moveView(offset);
                }

            });
        }

        function _setBookViewUnDraggable() {
            var viewDOM = _currView.getViewDOM();
            _isDragging = false;
            viewDOM.unbind('.containerDraggable');
        }

        function _isSwipeEnable() {
            return  !_isPainting && !_self.isZoomOutEnable();
        }

        /***********************
         * 函数：判断滚轮滚动方向
         * 作者：walkingp
         * 参数：event
         * 返回：滚轮方向 1：向上 -1：向下
         *************************/
        var scrollFunc=function(e){
            var viewDOM = _currView.getViewDOM();
            e=e || window.event;
            var t1=document.getElementById("bookContainer");
            if(e.wheelDelta){//IE/Opera/Chrome
                t1.value=e.wheelDelta;
            }
            viewDOM.trigger('mousescroll.containerDraggable',[e.wheelDeltaX, e.wheelDeltaY]);
        }
        window.onmousewheel=document.onmousewheel=scrollFunc;
//        $()

        /**
         * 设置手势滑动翻页
         */
        function _setSwipable() {
            if (_currView) {
                var viewDOMEle = _currView.getViewDOM();
                var isDraggable = _isDraggable();
                if (!_self._slideUtil || viewDOMEle[0] !== _self._slideUtil.getContent()[0]) {
                    _self._slideUtil = new _SlideUtil({
                        element: viewDOMEle,
                        distance: isDraggable ? 130 : 100,
                        deltaTime: isDraggable ? 200 : 500
                    });
                    _self._slideUtil.on(viewDOMEle, 'slideLeft', _self.nextPage).on(viewDOMEle, 'slideRight', _self.previousPage);
                }
                else {
                    _self._slideUtil.setOption({
                        disabled: false
                    });
                }
            }
        }

        /**
         * 取消手势滑动翻页
         */
        function _setUnSwipable() {
            if (_currView) {
                _self._slideUtil && _self._slideUtil.setOption({
                    disabled: true
                });
            }
        }

        //---------------------------------------------------------------------------------------------
        //                                      Public Method
        //---------------------------------------------------------------------------------------------

        /**
         * @return {boolean}
         */
        _self.isZoomInEnable = function () {
            return _getCurrentZoomRate() < MAX_ZOOM_RATE;
        };

        /**
         * @return {boolean}
         */
        _self.isZoomOutEnable = function () {
            return _getCurrentZoomRate() > MIN_ZOOM_RATE;
        };

        /**
         * 缩放
         * @param {number} scale 缩放量
         * @param {object=}  transformData
         */
        _self.zoom = function (scale, transformData) {
            var currZoomRate = _getCurrentZoomRate();
            if ((scale < 1 && _self.isZoomOutEnable()) || (scale > 1 && _self.isZoomInEnable()) || (scale == 1)) {
                _zoom(currZoomRate, scale, transformData);
            }
        };

        /**
         * 缩放比例
         * @return {Number}
         */
        _self.getZoomRate = function () {
            return _getCurrentZoomRate();
        }


        /**
         * 移动视图
         * @param {Offset | _model.Offset} offset
         */
        _self.moveView = function (offset) {

            var transformData = _getTransformData();
            transformData.tx += offset.x;
            transformData.ty += offset.y;

            _correctTransformBound(transformData);
            _setTransform(transformData);
        };

        /**
         * 恢复视图的原始位置
         */
        _self.recoverView = function () {
            var transformData = _getTransformData();
            var viewDOM = _currView.getViewDOM();
            transformData.tx = _pageProvider.bookImageSize.width * (transformData.a - 1);
            transformData.ty = _pageProvider.bookImageSize.height * (transformData.d - 1);
            _correctTransformBound(transformData);
            _setTransform(transformData);
        };

        _self.insertWidget = function (files,options) {
            _currView.insertWidget(files,options);
        };

        _self.insertWidgetWithPos = function (files,pos,pageIndex,author) {
            _currView.insertWidgetWithPos(files,pos,pageIndex,author);
        };

        _self.insertSingleWidget = function (file, widgetType, cardType, options) {
            _currView.insertSingleWidget(file, widgetType, cardType, options);
        }

        _self.setIsEditing = function (isEditing) {
            if (isEditing == _isEditing) return;
            _isEditing = isEditing;
            _currView.setIsEditing(isEditing);
        };

        /**
         * 获取ViewDom
         * @returns {jQuery}
         */
        _self.getViewDOM = function () {
            return _currView.getViewDOM();
        };

        /**
         * 是否可以向后翻页
         * @returns {boolean}
         */
        _self.isNextPageEnable = function () {
            return _currView.isNextPageEnable();
        };

        /**
         * 下一页
         */
        _self.nextPage = function () {
            //如果当前页面已经加载完成
            if (!_currView.isTempViewAnimating) {
                _self.setPageIndex(((_pageViewType === _define.PageViewType.WhiteBoardView) ? _wbIndex : _pageIndex) + 1);
            }
        };

        /**
         * 是否可以向前翻页
         * @returns {boolean}
         */
        _self.isPreviousPageEnable = function () {
            return _currView.isPreviousPageEnable();
        };

        /**
         * 上一页
         */
        _self.previousPage = function () {
            //如果当前页面已经加载完成
            if (!_currView.isTempViewAnimating) {
                _self.setPageIndex(((_pageViewType === _define.PageViewType.WhiteBoardView) ? _wbIndex : _pageIndex)- 1);

            }
        };

        /**
         * 切换书本页面
         * @param {number} pageIndex
         */
        _self.setPageIndex = function (pageIndex) {
            console.log("setPageIndex -> start");
            if (_self._setPageIndex(pageIndex)) {
                _transmitEvent();
                _currView.setPageIndex(pageIndex);
                this.recoverView();
                app_global_playerMgr && app_global_playerMgr.stop(_define.PlayerType.HotspotPlayer);
            }
        };

        _self._correctPageIndex = function (pageIndex) {
            if (pageIndex <= 0) pageIndex = 1;
            else if (pageIndex >= _pageProvider.pageCount) {
                if (_currView && _currView.isDoublePage()) pageIndex = _pageProvider.pageCount - 1;
                else pageIndex = _pageProvider.pageCount;
            }
            return pageIndex;
        }

        /**
         * 将全局canvas绘制到pageItem中的canvas
         * 只有wbview处理strokes
         * @param {HTMLCanvasElement} canvas
         */
        _self.paintStroke = function (canvas, isStrokeChanged,strokes) {
            _isPainting = false;
            _currView.paintStroke(canvas, isStrokeChanged,strokes);
            _isDraggable() &&  _setBookViewDraggable();
            _isSwipeEnable() &&  _setSwipable();
        };

        _self.doEraser = function(l,t,w,h){
            _currView.doEraser(l,t,w,h);
        }

        _self.recognition = function(lan){
            _currView.recognition(lan);
        }

        /**
         *仅仅保存笔迹到page，并不将bookview设置为可移动
         * @param {HTMLCanvasElement} canvas
         */
        _self.paintStrokeWithoutDrag = function (canvas, isStrokeChanged,strokes) {
            _isPainting = false;
            _currView.paintStroke(canvas, isStrokeChanged,strokes);
        };

        _self.restoreStroke = function (canvas,canvasmgr) {
            _isPainting = true;
            _currView.restoreStroke(canvas);
            _currView.reloadStrokes && _currView.reloadStrokes(canvasmgr);
            _setBookViewUnDraggable();
            _setUnSwipable();
        };

        /**
         * 保存页面笔迹到本地
         */
        _self.saveStrokesToLocal = function () {
            _currView.saveStrokesToLocal();
        }

        /**
         * 显示全部热区
         */
        _self.showAllHotspots = function () {
            _currView.showAllHotspots();
        };

        /**
         * 影藏全部热区
         */
        _self.hideAllHotspots = function () {
            _currView.hideAllHotspots();
        };

        /**
         * 隐藏pageItem里的canvas
         */
        _self.hidePageItemCanvas = function () {
            var pages = _currView.getShowPageItem();
            $.each(pages, function () {
                $(this.getPaintCanvas()).hide();
            });
        }

        /**
         *  显示pageItem里的canvas
         */
        _self.showPageItemCanvas = function () {
            var pages = _currView.getShowPageItem();
            $.each(pages, function () {
                $(this.getPaintCanvas()).show();
            });
        }

        _self.refresh = function () {
            _currView.refresh();
        }

        /**
         * 页面卸载
         * @param {boolean=} index 默认为false，只是卸载当前页面，如果为true，则卸载整个书本页面。如从书本跳转到书架
         */
        _self.dispose = function (index) {
            index && app_global_playerMgr.destroy();
            _pageProvider.unloadAllPage(index);
        }

        /**
         * 获取当前单元UnitCode
         * @returns {*}
         */
        _self.getCurrUnit = function () {
            return _pageProvider.getUnit(_currView.getPageIndex());
        }

        _self.getUnit = function (pageIndex) {
            return _pageProvider.getUnit(pageIndex);
        }

        _self.getTransformData = function () {
            return _getTransformData();
        }

        /**
         * 获取当前单元UnitCode
         * @returns {*}
         */
        _self.getCurrCourse = function () {
            return _pageProvider.getCourse(_currView.getPageIndex());
        }

        _self.getCourse = function (pageIndex) {
            return _pageProvider.getCourse(pageIndex);
        }

        _self.getCurrPageMetadata = function () {
            return _pageProvider.getPageMetadata(_currView.getPageIndex());
        }

        _self.getPageMetadata = function (pageIndex) {
            return _pageProvider.getPageMetadata(pageIndex);
        }

        _self.getCurrentBookView = function () {
            return _currView;
        }

        _self.getContent = function () {
            return _bookContainer;
        }

        /**
         * 获取本页资源
         */
        _self.getCurrentPageRes = function(){
            console.log(_currView);
            var pages = _currView.getShowPageItem();
            var widgets = [];
            $.each(pages,function(){
                widgets = $.merge(widgets,this.getCurrentPageRes());
            })
            return widgets;
        }

        /**
         * 清空本页资源
         */
        _self.clearCurrentPageRes = function () {
            _containerhelper.clearCurrentPageRes(_currView);
        }

        /**
         * 清空书本资源
         */
        _self.clearBookRes = function () {
            _containerhelper.clearBookRes(_pageProvider);
        }

        /**
         *在拖动widget时触发这两个事件
         */
        _bookContainer.bind("swipable", function (event) {
            if (_isSwipeEnable())
                _setSwipable();
            event.stopPropagation();
        });

        _bookContainer.bind("unSwipable", function (event) {
            _setUnSwipable();
            event.stopPropagation();
        });

        _bookContainer.bind("draggable", function (event) {
            _isDraggable() && _setBookViewDraggable();
            event.stopPropagation();
        });

        _bookContainer.bind("unDraggable", function (event) {
            _setBookViewUnDraggable();
            event.stopPropagation();
        });

        _self.unloadAllPage = function(){
            _pageProvider.unloadAllPage();
        }
    }

    exports.BookContainer = BookContainer;
});