declare module GC{
    module Spread{
        module CalcEngine{
            /**
             * Represents the expression type
             * @enum {number}
             */
            export enum ExpressionType{
                /**
                 * Specifies the unknow type
                 */
                unknow= 0,
                /**
                 * Specifies the reference type
                 */
                reference= 1,
                /**
                 * Specifies the number type
                 */
                number= 2,
                /**
                 * Specifies the string type
                 */
                string= 3,
                /**
                 * Specifies the boolean type
                 */
                boolean= 4,
                /**
                 * Specifies the error type
                 */
                error= 5,
                /**
                 * Specifies the array type
                 */
                array= 6,
                /**
                 * Specifies the function type
                 */
                function= 7,
                /**
                 * Specifies the name type
                 */
                name= 8,
                /**
                 * Specifies the operator type
                 */
                operator= 9,
                /**
                 * Specifies the parenthesses type
                 */
                parentheses= 10,
                /**
                 * Specifies the missing argument type
                 */
                missingArgument= 11,
                /**
                 * Specifies the expand type
                 */
                expand= 12,
                /**
                 * Specifies the struct reference type
                 */
                structReference= 13,
                /**
                 * Specifies the spill reference type
                 */
                spillReference= 33
            }


            export class AsyncEvaluateContext{
                /**
                 * Represents an evaluate context for async functions.
                 * @class
                 * @param {Object} context The common evaluate context.
                 */
                constructor(context:  any);
                /**
                 * Set the async function evaluate result to CalcEngine, CalcEngine uses this value to recalculate the cell that contains this async function and all dependents cells.
                 * @param {Object} value The async function evaluate result.
                 */
                setAsyncResult(value:  any): void;
            }

            export class CalcError{
                /**
                 * Represents an error in calculation.
                 * @class
                 * @param {string} error The description of the error.
                 */
                constructor(error:  string);
                /**
                 * Parses the specified error from the string.
                 * @param {string} value The error string.
                 * @returns {GC.Spread.CalcEngine.CalcError} The calculation error.
                 */
                static parse(value: string): GC.Spread.CalcEngine.CalcError;
                /**
                 * Returns a string that represents this instance.
                 * @returns {string} The error string.
                 */
                toString(): string;
            }

            export class Expression{
                /**
                 * Provides the base class from which the classes that represent expression tree nodes are derived. This is an abstract class.
                 * @class
                 */
                constructor(type:  GC.Spread.CalcEngine.ExpressionType);
                /**
                 * Indicates the expression type
                 * @type {GC.Spread.CalcEngine.ExpressionType}
                 */
                type: GC.Spread.CalcEngine.ExpressionType;
            }
            module Functions{
                /**
                 * Defines a global custom function.
                 * @param {string} name The name of the function.
                 * @param {GC.Spread.CalcEngine.Functions.Function} fn The function to add.
                 */
                function defineGlobalCustomFunction(name:  string,  fn:  GC.Spread.CalcEngine.Functions.Function): GC.Spread.CalcEngine.Functions.Function;
                /**
                 * Gets all of the global functions or one global function that specified by name.
                 * @param {string} name The name of the function.
                 * @returns {GC.Spread.CalcEngine.Functions.Function} If the name is empty, return all of the global functions, otherwise, return one function with the specified name.
                 */
                function findGlobalFunction(name?:  string): any;
                /**
                 * If the name is empty, remove all of the global functions, otherwise, remove one function with the specified name.
                 * @param {string} name The name of the function.
                 */
                function removeGlobalFunction(name?:  string): void;

                export interface IFunctionDescription{
                    description: string;
                    parameters: GC.Spread.CalcEngine.Functions.IParameterDescription[];
                }


                export interface IParameterDescription{
                    name: string;
                    repeatable?: boolean;
                    optional?: boolean;
                }

                /**
                 * Represents the asynchronous function evaluate mode.
                 * @enum {number}
                 */
                export enum AsyncFunctionEvaluateMode{
                    /**
                     * enum value is 0, Specifies that the async function evaluates the changed, referenced cells.
                     */
                    onRecalculation= 0,
                    /**
                     * enum value is 1, Specifies that the async function only evaluates once.
                     */
                    calculateOnce= 1,
                    /**
                     * enum value is 2, Specifies that the async function evaluates based on the interval.
                     */
                    onInterval= 2
                }


                export class AsyncFunction{
                    /**
                     * Represents an abstract base class for defining asynchronization functions.
                     * @class
                     * @param {string} name The name of the function.
                     * @param {number} minArgs The minimum number of arguments for the function.
                     * @param {number} maxArgs The maximum number of arguments for the function.
                     * @param {Object} description The description of the function.
                     */
                    constructor(name:  string,  minArgs:  number,  maxArgs:  number,  description:  GC.Spread.CalcEngine.Functions.IFunctionDescription);
                    /**
                     * Returns the default value of the evaluated function result before getting the async result.
                     * @returns {Object} The default value of the evaluated function result before getting the async result.
                     */
                    defaultValue(): any;
                    /**
                     * Returns the result of the function applied to the arguments.
                     * @param {GC.Spread.CalcEngine.AsyncEvaluateContext} context The evaluate context
                     * @param {Object} args Arguments for the function evaluation
                     * @returns {Object} The result of the function applied to the arguments.
                     */
                    evaluateAsync(context:  GC.Spread.CalcEngine.AsyncEvaluateContext,  args:  any): any;
                    /**
                     * Decides how to re-calculate the formula.
                     * @returns {GC.Spread.CalcEngine.Functions.AsyncFunctionEvaluateMode} The evaluate mode.
                     */
                    evaluateMode(): GC.Spread.CalcEngine.Functions.AsyncFunctionEvaluateMode;
                    /**
                     * Returns the interval.
                     * @returns {number} The interval in milliseconds.
                     */
                    interval(): number;
                }

                export class Function{
                    /**
                     * Represents an abstract base class for defining functions.
                     * @class
                     * @param {string} name The name of the function.
                     * @param {number} minArgs The minimum number of arguments for the function.
                     * @param {number} maxArgs The maximum number of arguments for the function.
                     * @param {Object} functionDescription The description object of the function.
                     * @param {string} [functionDescription.description] - The description of the function.
                     * @param {Array.<Object>} [functionDescription.parameter] - The parameters' description array of the function.
                     * @param {string} functionDescription.parameter.name -  The parameter name..
                     * @param {boolean} [functionDescription.parameter.repeatable] - Whether the parameter is repeatable.
                     * @param {boolean} [functionDescription.parameter.optional] - Whether the parameter is optional.
                     */
                    constructor(name?:  string,  minArgs?:  number,  maxArgs?:  number,  description?:  GC.Spread.CalcEngine.Functions.IFunctionDescription);
                    /**
                     * Represents the maximum number of arguments for the function.
                     * @type {number}
                     */
                    maxArgs: number;
                    /**
                     * Represents the minimum number of arguments for the function.
                     * @type {number}
                     */
                    minArgs: number;
                    /**
                     * Represents the name of the function.
                     * @type {string}
                     */
                    name: string;
                    /**
                     * Represents the type name string used for supporting serialization.
                     * @type {string}
                     */
                    typeName: string;
                    /**
                     * Determines whether the function accepts array values for the specified argument.
                     * @param {number} argIndex Index of the argument.
                     * @function
                     * @returns {boolean} <c>true</c> if the function accepts array values for the specified argument; otherwise, <c>false</c>.
                     */
                    acceptsArray(argIndex:  number): boolean;
                    /**
                     * Indicates whether the function can process Error values.
                     * @param {number} argIndex Index of the argument.
                     * @returns {boolean} <c>true</c> if the function can process Error values for the specified argument; otherwise, <c>false</c>.
                     * @function
                     */
                    acceptsError(argIndex:  number): boolean;
                    /**
                     * Indicates whether the Evaluate method can process missing arguments.
                     * @param {number} argIndex Index of the argument
                     * @returns {boolean} <c>true</c> if the Evaluate method can process missing arguments; otherwise, <c>false</c>.
                     */
                    acceptsMissingArgument(argIndex:  number): boolean;
                    /**
                     * Determines whether the function accepts Reference values for the specified argument.
                     * @param {number} argIndex Index of the argument.
                     * @returns {boolean} <c>true</c> if the function accepts Reference values for the specified argument; otherwise, <c>false</c>.
                     * @function
                     */
                    acceptsReference(argIndex:  number): boolean;
                    /**
                     * Returns the description of the function.
                     * @function
                     * @returns {Object} The description of the function.
                     */
                    description(): GC.Spread.CalcEngine.Functions.IFunctionDescription;
                    /**
                     * Returns the result of the function applied to the arguments.
                     * @param {Object} args Arguments for the function evaluation
                     * @returns {Object} The result of the function applied to the arguments.
                     */
                    evaluate(args:  any): any;
                    /**
                     * Finds the branch argument.
                     * @param {Object} test The test.
                     * @returns {number} Indicates the index of the argument that would be treated as the branch condition.
                     */
                    findBranchArgument(test:  any): number;
                    /**
                     * Finds the test argument when this function is branched.
                     * @returns {number} Indicates the index of the argument that would be treated as the test condition.
                     */
                    findTestArgument(): number;
                    /**
                     * Gets a value that indicates whether this function is branched by arguments as conditional.
                     * @returns {boolean} <c>true</c> if this instance is branched; otherwise, <c>false</c>.
                     */
                    isBranch(): boolean;
                    /**
                     * Determines whether the evaluation of the function is dependent on the context in which the evaluation occurs.
                     * @returns {boolean} <c>true</c> if the evaluation of the function is dependent on the context; otherwise, <c>false</c>.
                     */
                    isContextSensitive(): boolean;
                    /**
                     * Determines whether the function is volatile while it is being evaluated.
                     * @returns {boolean} <c>true</c> if the function is volatile; otherwise, <c>false</c>.
                     */
                    isVolatile(): boolean;
                }
            }

            module LanguagePackages{
                /**
                 * Gets or Sets the language package for clacengine.
                 * @param {string} languageName the clacengine language name string;
                 * @returns {string|void} If no value is set, return the clacengine language name string;
                 * @example
                 * GC.Spread.CalcEngine.LanguagePackages.languagePackages('fr');
                 * GC.Spread.CalcEngine.LanguagePackages.languagePackages();
                 */
                function languagePackages(languageName?:  string): any;
            }

        }

        module Commands{
            /**
             * Represents the key code.
             * @enum {number}
             * @example
             * //This example creates a custom action using the enter key.
             * var activeSheet = spread.getActiveSheet();
             * spread.commandManager().register('myCmd',
             *                 function ColorAction() {
             *                     //Click on a cell and press the Enter key.
             *                     activeSheet.getCell(activeSheet.getActiveRowIndex(), activeSheet.getActiveColumnIndex()).backColor("red");
             *                 }
             *             );
             * //Map the created action to the Enter key.
             * spread.commandManager().setShortcutKey('myCmd', GC.Spread.Commands.Key.enter, false, false, false, false);
             */
            export enum Key{
                /**
                 * Indicates the left arrow key.
                 */
                left= 37,
                /**
                 * Indicates the right arrow key.
                 */
                right= 39,
                /**
                 * Indicates the up arrow key.
                 */
                up= 38,
                /**
                 * Indicates the down arrow key.
                 */
                down= 40,
                /**
                 * Indicates the Tab key.
                 */
                tab= 9,
                /**
                 * Indicates the Enter key.
                 */
                enter= 13,
                /**
                 * Indicates the Shift key.
                 */
                shift= 16,
                /**
                 * Indicates the Ctrl key.
                 */
                ctrl= 17,
                /**
                 * Indicates the space key.
                 */
                space= 32,
                /**
                 * Indicates the Alt key.
                 */
                altkey= 18,
                /**
                 * Indicates the Home key.
                 */
                home= 36,
                /**
                 * Indicates the End key.
                 */
                end= 35,
                /**
                 * Indicates the Page Up key.
                 */
                pup= 33,
                /**
                 * Indicates the Page Down key.
                 */
                pdn= 34,
                /**
                 * Indicates the Backspace key.
                 */
                backspace= 8,
                /**
                 * Indicates the Delete key.
                 */
                del= 46,
                /**
                 * Indicates the Esc key.
                 */
                esc= 27,
                /**
                 * Indicates the A key
                 */
                a= 65,
                /**
                 * Indicates the C key.
                 */
                c= 67,
                /**
                 * Indicates the V key.
                 */
                v= 86,
                /**
                 * Indicates the X key.
                 */
                x= 88,
                /**
                 * Indicates the Z key.
                 */
                z= 90,
                /**
                 * Indicates the Y key.
                 */
                y= 89
            }


            export class CommandManager{
                /**
                 * Represents a command manager.
                 * @class
                 * @param {Object} context The execution context for all commands in the command manager.
                 * @constructor
                 */
                constructor(context:  Object);
                /**
                 * Executes a command and adds the command to UndoManager.
                 * @param {Object} commandOptions The options for the command.
                 * @param {string} commandOptions.cmd The command name, the field is required.
                 * @param {Object} [commandOptions.arg1] The command argument, the name and type are decided by the command definition.
                 * @param {Object} [commandOptions.arg2] The command argument, the name and type are decided by the command definition.
                 * @param {Object} [commandOptions.argN] The command argument, the name and type are decided by the command definition.
                 * @returns {boolean} The execute command result.
                 * @example
                 * //For example, the following code executes the autoFitColumn command.
                 * var spread = GC.Spread.Sheets.findControl(document.getElementById("ss"));
                 * spread.commandManager().execute({cmd: "autoFitColumn", sheetName: "Sheet1", columns: [{col: 1}], rowHeader: false, autoFitType: GC.Spread.Sheets.AutoFitType.cell});
                 */
                execute(commandOptions:  Object): boolean;
                /**
                 * Registers a command with the command manager.
                 * @param {string} name - The name of the command.
                 * @param {Object} command - The object that defines the command.
                 * @param {number|GC.Spread.Commands.Key} key - The key code.
                 * @param {boolean} ctrl - <c>true</c> if the command uses the Ctrl key; otherwise, <c>false</c>.
                 * @param {boolean} shift - <c>true</c> if the command uses the Shift key; otherwise, <c>false</c>.
                 * @param {boolean} alt - <c>true</c> if the command uses the Alt key; otherwise, <c>false</c>.
                 * @param {boolean} meta - <c>true</c> if the command uses the Command key on the Macintosh or the Windows key on Microsoft Windows; otherwise, <c>false</c>.
                 * @example
                 * //For example, the following code registers the changeBackColor command and then executes the command.
                 *  var command = {
                 *       canUndo: true,
                 execute: function (context, options, isUndo) {
                 var Commands = GC.Spread.Sheets.Commands;
                 if (isUndo) {
                 Commands.undoTransaction(context, options);
                 return true;
                 } else {
                 Commands.startTransaction(context, options);
                 var sheet = context.getSheetFromName(options.sheetName);
                 var cell = sheet.getCell(options.row, options.col);
                 cell.backColor(options.backColor);
                 Commands.endTransaction(context, options);
                 return true;
                 }
                 }
                 };
                 *  var spread = GC.Spread.Sheets.findControl(document.getElementById("ss"));
                 *  var commandManager = spread.commandManager();
                 *  commandManager.register("changeBackColor", command);
                 *  commandManager.execute({cmd: "changeBackColor", sheetName: spread.getSheet(0).name(), row: 1, col: 2, backColor: "red"});
                 */
                register(name:  string,  command:  Object,  key?:  number|GC.Spread.Commands.Key,  ctrl?:  boolean,  shift?:  boolean,  alt?:  boolean,  meta?:  boolean): void;
                /**
                 * Binds a shortcut key to a command.
                 * @param {string} commandName The command name, setting commandName to undefined removes the bound command of the shortcut key.
                 * @param {number|GC.Spread.Commands.Key} key The key code, setting the key code to undefined removes the shortcut key of the command.
                 * @param {boolean} ctrl <c>true</c> if the command uses the Ctrl key; otherwise, <c>false</c>.
                 * @param {boolean} shift <c>true</c> if the command uses the Shift key; otherwise, <c>false</c>.
                 * @param {boolean} alt <c>true</c> if the command uses the Alt key; otherwise, <c>false</c>.
                 * @param {boolean} meta <c>true</c> if the command uses the Command key on the Macintosh or the Windows key on Microsoft Windows; otherwise, <c>false</c>.
                 * @example
                 * //This example changes the behavior of default keys.
                 * var activeSheet = spread.getActiveSheet();
                 * //Change the default Up arrow key action to "Page Up" for the active cell.
                 * spread.commandManager().setShortcutKey('navigationPageUp', GC.Spread.Commands.Key.up, false, false, false, false);
                 * //Change the default Down arrow key action to "Page Down" for the active cell.
                 * spread.commandManager().setShortcutKey('navigationPageDown', GC.Spread.Commands.Key.down, false, false, false, false);
                 */
                setShortcutKey(commandName:  string,  key?:  number|GC.Spread.Commands.Key,  ctrl?:  boolean,  shift?:  boolean,  alt?:  boolean,  meta?:  boolean): void;
            }

            export class UndoManager{
                /**
                 * Represents the undo manager.
                 * @constructor
                 */
                constructor();
                /**
                 * Gets whether the redo operation is allowed.
                 * @returns {boolean} <c>true</c> if the redo operation is allowed; otherwise, <c>false</c>.
                 */
                canRedo(): boolean;
                /**
                 * Gets whether the undo operation is allowed.
                 * @returns {boolean} <c>true</c> if the undo operation is allowed; otherwise, <c>false</c>.
                 */
                canUndo(): boolean;
                /**
                 * Clears all of the undo stack and the redo stack.
                 */
                clear(): void;
                /**
                 * Redoes the last command.
                 * @returns {boolean} <c>true</c> if the redo operation is successful; otherwise, <c>false</c>.
                 */
                redo(): boolean;
                /**
                 * Undoes the last command.
                 * @returns {boolean} <c>true</c> if the undo operation is successful; otherwise, <c>false</c>.
                 */
                undo(): boolean;
            }
        }

        module Common{

            export interface IDateTimeFormat{
                abbreviatedDayNames?: string[];
                abbreviatedMonthGenitiveNames?: string[];
                abbreviatedMonthNames?: string[];
                amDesignator?: string;
                dayNames?: string[];
                fullDateTimePattern?: string;
                longDatePattern?: string;
                longTimePattern?: string;
                monthDayPattern?: string;
                monthGenitiveNames?: string[];
                monthNames?: string[];
                pmDesignator?: string;
                shortDatePattern?: string;
                shortTimePattern?: string;
                yearMonthPattern?: string;
            }


            export interface INumberFormat{
                currencyDecimalSeparator?: string;
                currencyGroupSeparator?: string;
                currencySymbol?: string;
                numberDecimalSeparator?: string;
                numberGroupSeparator?: string;
                listSeparator?: string;
                arrayListSeparator?: string;
                arrayGroupSeparator?: string;
                dbNumber?: Object
            }


            export class CultureInfo{
                /**
                 * Represents the custom culture class. The member variable can be overwritten.
                 * @class
                 */
                constructor();
                /**
                 * Indicates the date time format fields.
                 * @type {Object}
                 * @property {Array.<string>} abbreviatedDayNames - Specifies the day formatter for "ddd".
                 * @property {Array.<string>} abbreviatedMonthGenitiveNames - Specifies the month formatter for "MMM".
                 * @property {Array.<string>} abbreviatedMonthNames - Specifies the month formatter for "MMM".
                 * @property {string} amDesignator - Indicates the AM designator.
                 * @property {Array.<string>} dayNames - Specifies the day formatter for "dddd".
                 * @property {string} fullDateTimePattern - Specifies the standard date formatter for "F".
                 * @property {string} longDatePattern - Specifies the standard date formatter for "D".
                 * @property {string} longTimePattern - Specifies the standard date formatter for "T" and "U".
                 * @property {string} monthDayPattern - Specifies the standard date formatter for "M" and "m".
                 * @property {Array.<string>} monthGenitiveNames - Specifies the formatter for "MMMM".
                 * @property {Array.<string>} monthNames - Specifies the formatter for "M" or "MM".
                 * @property {string} pmDesignator - Indicates the PM designator.
                 * @property {string} shortDatePattern - Specifies the standard date formatter for "d".
                 * @property {string} shortTimePattern - Specifies the standard date formatter for "t".
                 * @property {string} yearMonthPattern - Specifies the standard date formatter for "y" and "Y".
                 * @example
                 * // This example creates a custom culture.
                 * var myCulture = new GC.Spread.Common.CultureInfo();
                 * myCulture.NumberFormat.currencySymbol = "\u20ac"
                 * myCulture.NumberFormat.numberDecimalSeparator = ",";
                 * myCulture.NumberFormat.numberGroupSeparator = ".";
                 * myCulture.NumberFormat.arrayGroupSeparator = ";";
                 * myCulture.NumberFormat.arrayListSeparator = "\\";
                 * myCulture.NumberFormat.listSeparator = ";";
                 * myCulture.DateTimeFormat.amDesignator = "";
                 * myCulture.DateTimeFormat.pmDesignator = "";
                 * myCulture.DateTimeFormat.abbreviatedMonthNames = ["Jan", "Feb", "Mrz", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez", ""];
                 * myCulture.DateTimeFormat.abbreviatedDayNames = ["So", "Mo", "Di", "Mi", "Do", "Fr", "Sa"];
                 * myCulture.DateTimeFormat.abbreviatedMonthGenitiveNames = ["Jan", "Feb", "Mrz", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez", ""];
                 * myCulture.DateTimeFormat.dayNames = ["Sonntag", "Montag", "Dienstag", "Mittwoch", "Donnerstag", "Freitag", "Samstag"];
                 * myCulture.DateTimeFormat.fullDateTimePattern = "dddd, d. MMMM yyyy HH:mm:ss";
                 * myCulture.DateTimeFormat.longDatePattern = "dddd, d. MMMM yyyy";
                 * myCulture.DateTimeFormat.longTimePattern = "HH:mm:ss";
                 * myCulture.DateTimeFormat.monthDayPattern = "dd MMMM";
                 * myCulture.DateTimeFormat.monthNames = ["Januar", "Februar", "M\xe4rz", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember", ""];
                 * myCulture.DateTimeFormat.monthGenitiveNames = ["Januar", "Februar", "M\xe4rz", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember", ""];
                 * myCulture.DateTimeFormat.shortDatePattern = "dd.MM.yyyy";
                 * myCulture.DateTimeFormat.shortTimePattern = "HH:mm";
                 * myCulture.DateTimeFormat.yearMonthPattern = "MMMM yyyy";
                 * //add one culture
                 * GC.Spread.Common.CultureManager.addCultureInfo("de-DE", myCulture);
                 * //switch to "de-DE" culture
                 * GC.Spread.Common.CultureManager.culture("de-DE");
                 * var d = new Date();
                 * //With culture
                 * activeSheet.setValue(1, 0, new Date(d.setDate(d.getDate() + 1)));
                 * activeSheet.getCell(1, 0).formatter("mmm");
                 * var dvalue = 12345.6789;
                 * activeSheet.setColumnWidth(0, 200);
                 * activeSheet.setColumnWidth(1, 200);
                 * activeSheet.setColumnWidth(2, 200);
                 * activeSheet.setValue(0, 0, dvalue);
                 * activeSheet.getCell(0, 0).formatter("###,###.00");
                 * activeSheet.setValue(2, 0, new Date(d.setDate(d.getDate() + 1)));
                 * //With culture
                 * activeSheet.getCell(3, 0).formatter("yyyy/mmm/dddd");
                 * activeSheet.setValue(3, 0, new Date());
                 */
                DateTimeFormat: GC.Spread.Common.IDateTimeFormat;
                /**
                 * Indicates all the number format fields.
                 * @type {Object}
                 * @property {string} currencyDecimalSeparator - Indicates the currency decimal point.
                 * @property {string} currencyGroupSeparator - Indicates the currency thousand separator.
                 * @property {string} currencySymbol - Indicates the currency symbol.
                 * @property {string} numberDecimalSeparator - Indicates the decimal point.
                 * @property {string} numberGroupSeparator - Indicates the thousand separator.
                 * @property {string} listSeparator - Indicates the separator for function arguments in a formula.
                 * @property {string} arrayListSeparator - Indicates the separator for the constants in one row of an array constant in a formula.
                 * @property {string} arrayGroupSeparator - Indicates the separator for the array rows of an array constant in a formula.
                 * @property {object} dbNumber - Specifies the DBNumber characters.
                 * The dbNumber object structure as follow:
                 *  {
                 *     1: {letters: ['\u5146', '\u5343', '\u767e', '\u5341', '\u4ebf', '\u5343', '\u767e', '\u5341', '\u4e07', '\u5343', '\u767e', '\u5341', ''], // \u5146\u5343\u767e\u5341\u4ebf\u5343\u767e\u5341\u4e07\u5343\u767e\u5341
                 *         numbers: ['\u25cb', '\u4e00', '\u4e8c', '\u4e09', '\u56db', '\u4e94', '\u516d', '\u4e03', '\u516b', '\u4e5d'] }, // \u25cb\u4e00\u4e8c\u4e09\u56db\u4e94\u516d\u4e03\u516b\u4e5d
                 *     2: {letters: ['\u5146', '\u4edf', '\u4f70', '\u62fe', '\u4ebf', '\u4edf', '\u4f70', '\u62fe', '\u4e07', '\u4edf', '\u4f70', '\u62fe', ''], // \u5146\u4edf\u4f70\u62fe\u4ebf\u4edf\u4f70\u62fe\u4e07\u4edf\u4f70\u62fe
                 *         numbers: ['\u96f6', '\u58f9', '\u8d30', '\u53c1', '\u8086', '\u4f0d', '\u9646', '\u67d2', '\u634c', '\u7396']}, // \u96f6\u58f9\u8d30\u53c1\u8086\u4f0d\u9646\u67d2\u634c\u7396
                 *     3: {letters: null,
                 *         numbers: ['\uff10', '\uff11', '\uff12', '\uff13', '\uff14', '\uff15', '\uff16', '\uff17', '\uff18', '\uff19']} // \uff10\uff11\uff12\uff13\uff14\uff15\uff16\uff17\uff18\uff19
                 * };
                 * @example
                 * // This example creates a custom culture.
                 * var myCulture = new GC.Spread.Common.CultureInfo();
                 * myCulture.NumberFormat.currencySymbol = "\u20ac"
                 * myCulture.NumberFormat.numberDecimalSeparator = ",";
                 * myCulture.NumberFormat.numberGroupSeparator = ".";
                 * myCulture.NumberFormat.arrayGroupSeparator = ";";
                 * myCulture.NumberFormat.arrayListSeparator = "\\";
                 * myCulture.NumberFormat.listSeparator = ";";
                 * myCulture.DateTimeFormat.amDesignator = "";
                 * myCulture.DateTimeFormat.pmDesignator = "";
                 * myCulture.DateTimeFormat.abbreviatedMonthNames = ["Jan", "Feb", "Mrz", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez", ""];
                 * myCulture.DateTimeFormat.abbreviatedDayNames = ["So", "Mo", "Di", "Mi", "Do", "Fr", "Sa"];
                 * myCulture.DateTimeFormat.abbreviatedMonthGenitiveNames = ["Jan", "Feb", "Mrz", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez", ""];
                 * myCulture.DateTimeFormat.dayNames = ["Sonntag", "Montag", "Dienstag", "Mittwoch", "Donnerstag", "Freitag", "Samstag"];
                 * myCulture.DateTimeFormat.fullDateTimePattern = "dddd, d. MMMM yyyy HH:mm:ss";
                 * myCulture.DateTimeFormat.longDatePattern = "dddd, d. MMMM yyyy";
                 * myCulture.DateTimeFormat.longTimePattern = "HH:mm:ss";
                 * myCulture.DateTimeFormat.monthDayPattern = "dd MMMM";
                 * myCulture.DateTimeFormat.monthNames = ["Januar", "Februar", "M\xe4rz", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember", ""];
                 * myCulture.DateTimeFormat.monthGenitiveNames = ["Januar", "Februar", "M\xe4rz", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember", ""];
                 * myCulture.DateTimeFormat.shortDatePattern = "dd.MM.yyyy";
                 * myCulture.DateTimeFormat.shortTimePattern = "HH:mm";
                 * myCulture.DateTimeFormat.yearMonthPattern = "MMMM yyyy";
                 * //add one culture
                 * GC.Spread.Common.CultureManager.addCultureInfo("de-DE", myCulture);
                 * //switch to "de-DE" culture
                 * GC.Spread.Common.CultureManager.culture("de-DE");
                 * var d = new Date();
                 * //With culture
                 * activeSheet.setValue(1, 0, new Date(d.setDate(d.getDate() + 1)));
                 * activeSheet.getCell(1, 0).formatter("mmm");
                 * var dvalue = 12345.6789;
                 * activeSheet.setColumnWidth(0, 200);
                 * activeSheet.setColumnWidth(1, 200);
                 * activeSheet.setColumnWidth(2, 200);
                 * activeSheet.setValue(0, 0, dvalue);
                 * activeSheet.getCell(0, 0).formatter("###,###.00");
                 * activeSheet.setValue(2, 0, new Date(d.setDate(d.getDate() + 1)));
                 * //With culture
                 * activeSheet.getCell(3, 0).formatter("yyyy/mmm/dddd");
                 * activeSheet.setValue(3, 0, new Date());
                 */
                NumberFormat: GC.Spread.Common.INumberFormat;
            }

            export class CultureManager{
                /**
                 * Represente a culture manager.
                 * @constructor
                 */
                constructor();
                /**
                 * Adds the cultureInfo or custom language into the culture manager.
                 * @static
                 * @param {string} cultureName The culture name to set.
                 * @param {GC.Spread.Common.CultureInfo} cultureInfo The cultureInfo set to the culture.
                 * @param {object} language The custom language set to the culture. If already set, it will overwrite the old language.
                 */
                static addCultureInfo(cultureName:  string,  culture:  GC.Spread.Common.CultureInfo,  language?:  object): void;
                /**
                 * Get or set the Sheets culture.
                 * @static
                 * @param {string} cultureName The culture name to get.
                 * @returns {string}. The current culture name of Sheets.
                 */
                static culture(cultureName?:  string): string;
                /**
                 * Gets the specified cultureInfo. If no culture name, get current cultureInfo.
                 * @static
                 * @param {string | number} cultureName Culture name or culture ID
                 * @returns {GC.Spread.Common.CultureInfo} The specified cultureInfo object.
                 */
                static getCultureInfo(cultureName:  string | number): GC.Spread.Common.CultureInfo;
                /**
                 * Gets the specified custome language.
                 * @static
                 * @param {string} cultureName The culture name to get.
                 * @returns {object} The specified object. Null if not define the language.
                 */
                static getLanguage(cultureName:  string): object;
            }
        }

        module Formatter{

            export class FormatterBase{
                /**
                 * Represents a custom formatter with the specified format string.
                 * @class
                 * @param {string} format The format.
                 * @param {string} cultureName The culture name.
                 */
                constructor(format:  string,  cultureName:  string);
                /** Represents the type name string used for supporting serialization.
                 * @type {string}
                 */
                typeName: string;
                /**
                 * Formats the specified object as a string with a conditional color. This function should be overwritten.
                 * @param {Object} obj - The object with cell data to format.
                 * @param {Object} [options] - The additonal format data.
                 * @param {string} [options.conditionalForeColor] - The conditonal foreColor when format pattern contains color patter, such as [red]###.##
                 * @returns {string} The formatted string.
                 * @example
                 * //This example creates a custom formatter.
                 * var customFormatterTest = {};
                 * customFormatterTest.prototype = GC.Spread.Formatter.FormatterBase;
                 * customFormatterTest.format = function (obj, data) {
                 *     data.conditionalForeColor = "blue";
                 *     return "My format result : " + obj.toString();
                 * };
                 * customFormatterTest.parse = function (str) {
                 *     if (!str) {
                 *         return "";
                 *     }
                 *     return str;
                 * }
                 * activeSheet.getCell(1, 0).formatter(customFormatterTest);
                 * activeSheet.getCell(1, 0).value("Test");
                 */
                format(obj:  Object): string;
                /**
                 * Loads the object state from the specified JSON string.
                 * @param {Object} settings The custom formatter data from deserialization.
                 */
                fromJSON(settings:  Object): void;
                /**
                 * Parses the specified text. This function should be overwritten.
                 * @param {string} text The text.
                 * @returns {Object} The parsed object.
                 * @example
                 * //This example creates a custom formatter.
                 * var customFormatterTest = {};
                 * customFormatterTest.prototype = GC.Spread.Formatter.FormatterBase;
                 * customFormatterTest.format = function (obj, conditionalForeColor) {
                 *     conditionalForeColor.value = "blue";
                 *     return "My format result : " + obj.toString();
                 * };
                 * customFormatterTest.parse = function (str) {
                 *     if (!str) {
                 *         return "";
                 *     }
                 *     return str;
                 * }
                 * activeSheet.getCell(1, 0).formatter(customFormatterTest);
                 * activeSheet.getCell(1, 0).value("Test")
                 */
                parse(str:  string): Object;
                /**
                 * Saves the object state to a JSON string.
                 * @returns {Object} The custom formatter data.
                 */
                toJSON(): Object;
            }

            export class GeneralFormatter{
                /**
                 * Represents a formatter with the specified format mode and format string.
                 * @class
                 * @param {string} format The format.
                 * @param {string} cultureName The culture name.
                 */
                constructor(format:  string,  cultureName:  string);
                /**
                 * Formats the specified object as a string with a formatted data Object.
                 * @param {Object} obj The object with cell data to format.
                 * @param {Object} formattedData The object with formatted data.
                 * @param {Array} [formattedData.content] - The formatted data array, each item is an object that has two properties type and value, And it may contain these types: 'number', 'text', 'fillingChar', 'placeholder', 'exponent', 'decimalSeparator', 'groupSeparator', 'numberPlaceholder', 'percent', 'permille' and 'currency'. For example: {type: 'number', value: '123'}.
                 * @param {string} [formattedData.conditionalForeColor] - The conditional foreground color.
                 * @returns {string} The formatted string.
                 */
                format(obj:  Object,  formattedData:  Object): string;
                /**
                 * Gets or sets the format string for this formatter.
                 * @param {string} value The format string for this formatter.
                 * @returns {string|GC.Spread.Formatter.GeneralFormatter} If no value is set, returns the formatter string for this formatter; otherwise, returns the formatter.
                 */
                formatString(value?:  string): string | GC.Spread.Formatter.GeneralFormatter;
                /**
                 * Parses the specified text.
                 * @param {string} text The text.
                 * @returns {Object} The parsed object.
                 */
                parse(str:  string): Object;
            }
        }

        module Sheets{
            /**
             * Represents the license key for evaluation version and production version.
             */
            var LicenseKey: string;
            /**
             * Gets the Workbook instance by the host element.
             * @param {HTMLElement|string} host The host element or the host element id.
             * @returns {GC.Spread.Sheets.Workbook} The Workbook instance.
             */
            function findControl(host:  HTMLElement|string): GC.Spread.Sheets.Workbook;
            /**
             * Gets the type from the type string. This method is used for supporting the serialization of the custom object.
             * @param {string} typeString The type string.
             * @returns {Object} The type.
             */
            function getTypeFromString(typeString:  string): any;

            export interface FloatingObjectLoadedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
                element: HTMLElement;
            }


            export interface FormattedData{
                conditionalForeColor?: string | CanvasGradient | CanvasPattern;
                value?: string | CanvasGradient | CanvasPattern;
                content?: {
                type: string;
                value: string;
                }[];
            }


            export interface IActiveSheetChangedEventArgs{
                oldSheet: GC.Spread.Sheets.Worksheet;
                newSheet: GC.Spread.Sheets.Worksheet;
            }


            export interface IActiveSheetChangingEventArgs{
                oldSheet: GC.Spread.Sheets.Worksheet;
                newSheet: GC.Spread.Sheets.Worksheet;
                cancel: boolean;
            }


            export interface IBounds{
                x?: number;
                y?: number;
                width: number;
                height: number;
            }


            export interface IButtonClickedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                sheetArea: SheetArea;
            }


            export interface ICalendarOption{
                startDay?: GC.Spread.Sheets.CalendarStartDay;
                calendarPage?: GC.Spread.Sheets.CalendarPage;
                showTime?: boolean;
            }


            export interface ICellButton{
                position?: GC.Spread.Sheets.ButtonPosition;
                useButtonStyle?: boolean;
                enabled?: boolean;
                width?: number;
                caption?: string;
                imageSrc?: string;
                imageSize?: { width: number, height: number };
                captionAlign?: GC.Spread.Sheets.CaptionAlignment;
                command?: string | ((sheet: GC.Spread.Sheets.Worksheet, row: number, col: number, option: any) => void);
                imageType?: GC.Spread.Sheets.ButtonImageType;
                visibility?: GC.Spread.Sheets.ButtonVisibility;
            }


            export interface ICellChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                sheetArea: SheetArea;
                propertyName: string;
                oldValue: any;
                newValue: any;
                isUndo?: boolean;
            }


            export interface ICellClickEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sheetArea: SheetArea;
                row: number;
                col: number;
            }


            export interface ICellDoubleClickEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sheetArea: SheetArea;
                row: number;
                col: number;
            }


            export interface ICellPosition{
                row: number;
                col: number;
            }


            export interface ICellsInfo{
                row: number;
                col: number;
                rowCount: number;
                colCount: number;
                sheetName: string;
            }


            export interface IClipboardChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                copyData: string;
            }


            export interface IClipboardChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                copyData: string;
                cancel: boolean;
            }


            export interface IClipboardPastedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                cellRange: Range;
                pasteOption: ClipboardPasteOptions;
            }


            export interface IClipboardPastingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                cellRange: Range;
                pasteOption: ClipboardPasteOptions;
                cancel: boolean;
            }


            export interface IColorPickerGroup{
                name?: string;
                colors: string[][];
                needScaleColor?: boolean;
            }


            export interface IColorPickerOption{
                colorBlockSize?: number;
                groups?: GC.Spread.Sheets.IColorPickerGroup[];
            }


            export interface IColumn{
                name?: string;
                formatter?: string;
                cellType?: number;
                width?: number;
                visible?: boolean;
                resizable?: boolean;
                pageBreak?: boolean;
                displayName?: string;
                size?: number;
                value?: Function;
                starSize?: string;
            }


            export interface IColumnChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                sheetArea: SheetArea;
                propertyName: string;
                oldValue: any;
                newValue: any;
                count?: number;
                isUndo?: boolean;
            }


            export interface IColumnWidthChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                colList: number[];
                header: boolean;
            }


            export interface IColumnWidthChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                colList: any[];
                header: boolean;
                cancel: boolean;
            }


            export interface ICommentChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                comment: Comments.Comment;
                propertyName: string;
            }


            export interface ICommentRemovedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                comment: Comments.Comment;
            }


            export interface ICommentRemovingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                comment: Comments.Comment;
                cancel: boolean;
            }


            export interface IDirtyCellInfo{
                row: number;
                col: number;
                newValue: any;
                oldValue: any;
            }


            export interface IDragDropBlockCompletedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                fromRow: number;
                fromCol: number;
                toRow: number;
                toCol: number;
                rowCount: number;
                colCount: number;
                copy: boolean;
                insert: boolean;
                copyOption: CopyToOptions;
            }


            export interface IDragDropBlockEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                fromRow: number;
                fromCol: number;
                toRow: number;
                toCol: number;
                rowCount: number;
                colCount: number;
                copy: boolean;
                insert: boolean;
                copyOption: CopyToOptions;
                cancel: boolean;
            }


            export interface IDragFillBlockCompletedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                fillRange: GC.Spread.Sheets.Range;
                autoFillType: GC.Spread.Sheets.Fill.AutoFillType;
                fillDirection: GC.Spread.Sheets.Fill.FillDirection;
            }


            export interface IDragFillBlockEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                fillRange: GC.Spread.Sheets.Range;
                autoFillType: GC.Spread.Sheets.Fill.AutoFillType;
                fillDirection: GC.Spread.Sheets.Fill.FillDirection;
                cancel: boolean;
            }


            export interface IDropdown{
                type: GC.Spread.Sheets.DropDownType;
                option?: object;
                submitCommand?: string | ((sheet: GC.Spread.Sheets.Worksheet, value:any, option: any) => void);
            }


            export interface IEditChangeEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                editingText: any;
            }


            export interface IEditEndedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                editingText: Object;
            }


            export interface IEditEndingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                editor: Object;
                editingText: Object;
                cancel: boolean;
            }


            export interface IEditorStatusChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldStatus: EditorStatus;
                newStatus: EditorStatus;
            }


            export interface IEditStartingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                cancel: boolean;
            }


            export interface IEnterCellEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
            }


            export interface IFilterButtonHitInfo{
                rowFilter: GC.Spread.Sheets.Filter.RowFilterBase;
                row?: number;
                col?: number;
                x?: number;
                y?: number;
                width?: number;
                height?: number;
                sheetArea?: GC.Spread.Sheets.SheetArea;
            }


            export interface IFloatingObjectChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
                propertyName: string;
            }


            export interface IFloatingObjectRemovedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
            }


            export interface IFloatingObjectRemovingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
                cancel: boolean;
            }


            export interface IFloatingObjectSelectionChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
            }


            export interface IFormulaInfo{
                hasFormula?: boolean;
                baseRange?: GC.Spread.Sheets.Range;
                isArrayFormula?: boolean;
                formula?: string;
                formulaWithCulture?: string;
                isDynamicArray?: boolean;
            }


            export interface IFormulaRangeHitInfo{
                paramRange: GC.Spread.Sheets.IParamRange; //param range info
                inTopLeft?: boolean;
                inTopRight?: boolean;
                inBottomLeft?: boolean;
                inBottomRight?: boolean;
                inBorder?: boolean;
            }


            export interface IHitTestCellTypeHitInfo{
                x?: number;
                y?: number;
                row?: number;
                col?: number;
                cellRect?: GC.Spread.Sheets.Rect;
                sheetArea?: GC.Spread.Sheets.SheetArea;
                isReservedLocation?: boolean;
            }


            export interface IHitTestCommentHitInfo{
                x?: number;
                y?: number;
                comment?: GC.Spread.Sheets.Comments.Comment;
                area?: string;
            }


            export interface IHitTestDragInfo{
                action?: string;
                side?: string;
                outside?: boolean;
            }


            export interface IHitTestFloatingObjectHitInfo{
                x?: number;
                y?: number;
                floatingObject?: GC.Spread.Sheets.FloatingObjects.FloatingObject;
            }


            export interface IHitTestFooterCornerInfo{
                element?: string,
            }


            export interface IHitTestInformation{
                x?: number;
                y?: number;
                rowViewportIndex?: number;
                colViewportIndex?: number;
                row?: number;
                col?: number;
                hitTestType?: GC.Spread.Sheets.SheetArea;
                resizeInfo?: GC.Spread.Sheets.IHitTestResizeInfo;
                outlineHitInfo?: GC.Spread.Sheets.IHitTestOutlineHitInfo;
                filterButtonHitInfo?: GC.Spread.Sheets.IFilterButtonHitInfo;
                dragInfo?: GC.Spread.Sheets.IHitTestDragInfo;
                cellTypeHitInfo?: GC.Spread.Sheets.IHitTestCellTypeHitInfo;
                floatingObjectHitInfo?: GC.Spread.Sheets.IHitTestFloatingObjectHitInfo;
                formulaRangeHitInfo?: GC.Spread.Sheets.IFormulaRangeHitInfo;
                commentHitInfo?: GC.Spread.Sheets.IHitTestCommentHitInfo;
            }


            export interface IHitTestOutlineHitInfo{
                what?: string;
                info?: GC.Spread.Sheets.IOutlineHitInfo;
            }


            export interface IHitTestResizeInfo{
                action?: string;
                index?: number;
                sheetArea?: GC.Spread.Sheets.SheetArea;
                startY?: number;
                movingY?: number;
                startX?: number;
                movingX?: number;
            }


            export interface IHitTestScrollBarInfo{
                element?: string,
            }


            export interface IHyperlink{
                url: string;
                tooltip?: string;
                linkColor?: string;
                visitedLinkColor?: string;
                target?: number;
                drawUnderline?: boolean;
                command?: string | ((sheet: GC.Spread.Sheets.Worksheet, row: number, col: number, sheetArea?: GC.Spread.Sheets.SheetArea) => void);
            }


            export interface IIHitTestTabStripInfo{
                navButton?: string;
                sheetTab?: GC.Spread.Sheets.SheetTabInfo;
                resize?: boolean;
                blank?: boolean;
            }


            export interface IInvalidOperationEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                invalidType: InvalidOperationType;
                message: string;
            }


            export interface ILabelOptions{
                alignment: GC.Spread.Sheets.LabelAlignment;
                visibility: GC.Spread.Sheets.LabelVisibility;
                font: string;
                foreColor: string;
                margin: string;
            }


            export interface ILeaveCellEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                cancel: boolean;
            }


            export interface ILeftColumnChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldLeftCol: number;
                newLeftCol: number;
            }


            export interface IListItem{
                text?: string;
                value?: string;
                icon?: string;
            }


            export interface IListLayout{
                direction?: GC.Spread.Sheets.LayoutDirection;
                displayAs?: GC.Spread.Sheets.LayoutDisplayAs;
                collapsible?: boolean;
            }


            export interface IListOption{
                text?: string;
                layout?: GC.Spread.Sheets.IListLayout;
                multiSelect?: boolean;
                valueType?: GC.Spread.Sheets.DropdownListValue;
                onItemSelected?: (arg: MouseEvent) => string;
                items:  GC.Spread.Sheets.IListItem[] | GC.Spread.Sheets.IListOption[] | (() => HTMLElement); // a function returns a DOM element
            }


            export interface IMonthPickerOption{
                startYear?: number;
                stopYear?: number;
                height?: number;
            }


            export interface IOutlineColumnCheckStatusChanged{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number,
                col: number,
                status: boolean
            }


            export interface IOutlineHitInfo{
                index?: number;
                isExpanded?: boolean;
                level?: number;
                lineDirection?: GC.Spread.Sheets.Outlines.OutlineDirection;
                paintLine?: boolean;
            }


            export interface IParamRange{
                textOffset: number; // range text offset in formulatextbox's value
                text: string; // range text
                index: number; // index in all ranges
            }


            export interface IPictureChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                picture: FloatingObjects.Picture;
                propertyName: string;
            }


            export interface IPictureSelectionChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                picture: FloatingObjects.Picture;
            }


            export interface IProtectionOptions{
                allowSelectLockedCells?: boolean; //True or undefined if the user can select locked cells.
                allowSelectUnlockedCells?: boolean; //True or undefined if the user can select unlocked cells.
                allowSort?: boolean; //True if the user can sort ranges.
                allowFilter?: boolean; //True if the user can filter ranges.
                allowEditObjects?: boolean; //True if the user can edit floating objects.
                allowResizeRows?: boolean; //True if the user can resize rows.
                allowResizeColumns?: boolean; //True if the user can resize columns.
                allowDragInsertRows?: boolean; //True if the user can drag to insert rows.
                allowDragInsertColumns?: boolean; //True if the user can drag to insert columns.
                allowInsertRows?: boolean; //True if the user can insert rows.
                allowInsertColumns?: boolean; //True if the user can insert columns.
                allowDeleteRows?: boolean; //True if the user can delete rows.
                allowDeleteColumns?: boolean; //True if the user can delete columns.
            }


            export interface IRangeChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                rowCount: number;
                colCount: number;
                changedCells: ICellPosition[];
                action: RangeChangedAction;
                tableNames?: string[];
            }


            export interface IRangeFilterClearedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
            }


            export interface IRangeFilterClearingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
            }


            export interface IRangeFilteredEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                filterValues: any[];
            }


            export interface IRangeFilteringEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                filterValues: any[];
                conditionInfo : Object;
            }


            export interface IRangeGroupStateChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                isRowGroup: boolean;
                index: number;
                level: number;
            }


            export interface IRangeGroupStateChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                isRowGroup: boolean;
                index: number;
                level: number;
                cancel: boolean;
            }


            export interface IRangeSortedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                ascending: boolean;
            }


            export interface IRangeSortingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                ascending: boolean;
                compareFunction?: ((value1: any, value2: any) => number);
            }


            export interface IRowChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                sheetArea: SheetArea;
                propertyName: string;
                oldValue: any;
                newValue: any;
                count?: number;
                isUndo?: boolean;
            }


            export interface IRowHeightChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                rowList: number[];
                header: boolean;
            }


            export interface IRowHeightChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                rowList: any[];
                header: boolean;
                cancel: boolean;
            }


            export interface ISelectionChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldSelections: Sheets.Range[];
                newSelections: Sheets.Range[];
            }


            export interface ISelectionChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldSelections: GC.Spread.Sheets.Range[];
                newSelections: GC.Spread.Sheets.Range[];
            }


            export interface ISetBorderOptions{
                all?: boolean;
                left?: boolean;
                top?: boolean;
                right?: boolean;
                bottom?: boolean;
                outline?: boolean;
                inside?: boolean;
                innerHorizontal?: boolean;
                innerVertical?: boolean;
                diagonalUp?:boolean;
                diagonalDown?:boolean;
            }


            export interface IShapeChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                shape: GC.Spread.Sheets.Shapes.Shape;
                propertyName: string;
            }


            export interface IShapeRemovedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                shape: Shapes.Shape;
            }


            export interface IShapeRemovingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                shape: Shapes.Shape;
                cancel: boolean;
            }


            export interface IShapeSelectionChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                shape: Shapes.Shape;
            }


            export interface ISheetChangedEventArgs{
                sheetName: string;
                propertyName: string;
                sheetIndex: number;
                newValue?: boolean;
                oldValue?: boolean;
            }


            export interface ISheetChangingEventArgs{
                sheetName: string;
                propertyName: string;
                sheetIndex: number;
                cancel: boolean;
                newValue?: boolean;
                oldValue?: boolean;
            }


            export interface ISheetDefaultOption{
                rowHeight?: number;
                colHeaderRowHeight?: number;
                colWidth?: number;
                rowHeaderColWidth?: number;
            }


            export interface ISheetNameChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                oldValue: string;
                newValue: string;
            }


            export interface ISheetNameChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                oldValue: string;
                newValue: string;
                cancel: boolean;
            }


            export interface ISheetTabClickEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sheetTabIndex: number;
            }


            export interface ISheetTabDoubleClickEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sheetTabIndex: number;
            }


            export interface ISlicerChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                slicer: GC.Spread.Sheets.Slicers.Slicer;
                propertyName: string;
            }


            export interface ISliderOption{
                max?: number;
                min?: number;
                step?: number;
                direction? :GC.Spread.Sheets.LayoutDirection;
                tooltipVisible?: boolean;
                scaleVisible ?: boolean;
                marks?: number[];
                width?: number;
                height?: number;
                formatString?: string;
            }


            export interface ISortInfo{
                ascending: boolean;
                index: number;
                color?: string;
                isBackColor?: boolean;
                compareFunction?: ((value1: any, value2: any) => number);
            }


            export interface ISparklineChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sparkline: Sparklines.Sparkline;
            }


            export interface ITableFilterClearedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                table: GC.Spread.Sheets.Tables.Table;
                tableCol: number;
            }


            export interface ITableFilterClearingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                table: GC.Spread.Sheets.Tables.Table;
                tableCol: number;
            }


            export interface ITableFilteredEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                table: GC.Spread.Sheets.Tables.Table;
                tableCol: number;
                filterValues: any[];
            }


            export interface ITableFilteringEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                table: GC.Spread.Sheets.Tables.Table;
                tableCol: number;
                filterValues: any[];
                conditionInfo: Object;
            }


            export interface ITimePickerOption{
                min?: GC.Spread.Sheets.ITimePickerValue;
                max?: GC.Spread.Sheets.ITimePickerValue;
                step?: GC.Spread.Sheets.ITimePickerValue;
                formatString?: string;
                height?: number;
            }


            export interface ITimePickerValue{
                hour?: number;
                minute?: number;
                second?: number;
            }


            export interface ITopRowChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldTopRow: number;
                newTopRow: number;
            }


            export interface ITouchToolStripOpeningEventArgs{
                x: number;
                y: number;
                handled: boolean;
            }


            export interface IUserFormulaEnteredEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                formula: string;
            }


            export interface IUserZoomingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldZoomFactor: number;
                newZoomFactor: number;
            }


            export interface IValidationErrorEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                validator: DataValidation.DefaultDataValidator;
                validationResult: DataValidation.DataValidationResult;
            }


            export interface IValueChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                oldValue: any;
                newValue: any;
            }


            export interface IWorkBookDefaultOptions{
                allowUserDragDrop?: boolean;
                allowUserDragFill?: boolean;
                allowUserZoom?: boolean;
                allowUndo?: boolean;
                allowUserResize?: boolean;
                allowSheetReorder?: boolean;
                allowContextMenu?: boolean;
                allowUserDeselect?: boolean;
                defaultDragFillType?: GC.Spread.Sheets.Fill.AutoFillType;
                showDragFillSmartTag?: boolean;
                showHorizontalScrollbar?: boolean;
                showVerticalScrollbar?: boolean;
                scrollbarShowMax?: boolean;
                scrollbarMaxAlign?: boolean;
                tabStripVisible?: boolean;
                tabStripRatio?: number;
                tabEditable?: boolean;
                newTabVisible?: boolean;
                cutCopyIndicatorVisible?: boolean;
                cutCopyIndicatorBorderColor?: string;
                tabNavigationVisible?: boolean;
                backColor?: string;
                backgroundImage?: null | string;
                backgroundImageLayout?: GC.Spread.Sheets.ImageLayout;
                showResizeTip?: GC.Spread.Sheets.ShowResizeTip;
                showDragDropTip?: boolean;
                showDragFillTip?: boolean;
                scrollIgnoreHidden?: boolean;
                highlightInvalidData?: boolean;
                showScrollTip?: GC.Spread.Sheets.ShowScrollTip;
                columnResizeMode?: GC.Spread.Sheets.ResizeMode;
                rowResizeMode?: GC.Spread.Sheets.ResizeMode;
                grayAreaBackColor?: null | string;
                useTouchLayout?: boolean;
                hideSelection?: boolean;
                resizeZeroIndicator?: GC.Spread.Sheets.ResizeZeroIndicator;
                allowUserEditFormula?: boolean;
                enableFormulaTextbox?: boolean;
                referenceStyle?: number;
                allowDynamicArray?: boolean;
                autoFitType?: GC.Spread.Sheets.AutoFitType;
                calcOnDemand?: boolean;
                allowCopyPasteExcelStyle?: boolean;
                allowExtendPasteRange?: boolean;
                copyPasteHeaderOptions?: GC.Spread.Sheets.CopyPasteHeaderOptions;
                allowUserDragMerge?: boolean;
                scrollByPixel?: boolean;
                scrollPixel?: number;
                sheetCount?: number;
                font?: null | string;
                allowAutoCreateHyperlink?: boolean;
                customList?: string[][];
            }


            export interface IWorkbookHitTestInformation{
                x?: number;
                y?: number;
                worksheetHitInfo: GC.Spread.Sheets.IHitTestInformation;
                tabStripHitInfo: GC.Spread.Sheets.IIHitTestTabStripInfo;
                horizontalScrollBarHitInfo: GC.Spread.Sheets.IHitTestScrollBarInfo;
                verticalScrollBarHitInfo: GC.Spread.Sheets.IHitTestScrollBarInfo;
                footerCornerHitInfo: GC.Spread.Sheets.IHitTestFooterCornerInfo;
            }


            export interface IWorkbookOptions{
                allowUserDragMerge:boolean ; //  Whether to allow the user to drag merge cells.
                allowUserDragDrop :boolean ; // Whether to allow the user to drag and drop range data.
                allowUserDragFill :boolean ; // Whether to allow the user to drag fill a range.
                allowUserZoom :boolean ; // Whether to zoom the display by scrolling the mouse wheel while pressing the Ctrl key.
                allowUserResize :boolean ; // Whether to allow the user to resize columns and rows.
                allowUndo :boolean ; // Whether to allow the user to undo edits.
                allowSheetReorder :boolean ; // Whether the user can reorder the sheets in the Spread component.
                allowContextMenu :boolean; // Whether to allow the user to open the built-in context menu.
                allowUserDeselect :boolean; // Whether to allow the user to can use deselect in selection.
                defaultDragFillType :GC.Spread.Sheets.Fill.AutoFillType; // The default fill type.
                showDragFillSmartTag :boolean ; // Whether to display the drag fill dialog.
                showHorizontalScrollbar :boolean ; // Whether to display the horizontal scroll bar.
                showVerticalScrollbar :boolean ; // Whether to display the vertical scroll bar.
                scrollbarShowMax :boolean ; // Whether the displayed scroll bars are based on the entire number of columns and rows in the sheet.
                scrollbarMaxAlign :boolean ; // Whether the scroll bar aligns with the last row and column of the active sheet.
                tabStripVisible :boolean ; // Whether to display the sheet tab strip.
                tabStripRatio :number; // The width of the tab strip expressed as a percentage of the overall horizontal scroll bar width.
                tabEditable :boolean ; // Whether to allow the user to edit the sheet tab strip.
                newTabVisible :boolean ; // Whether the spreadsheet displays the special tab to let users insert new sheets.
                tabNavigationVisible :boolean ; // Whether to display the sheet tab navigation.
                cutCopyIndicatorVisible :boolean ; // Whether to display an indicator when copying or cutting the selected item.
                cutCopyIndicatorBorderColor :string; // The border color for the indicator displayed when the user cuts or copies the selection.
                backColor :string; // A color string used to represent the background color of the Spread component, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                backgroundImage :string; // The background image of the Spread component.
                backgroundImageLayout :GC.Spread.Sheets.ImageLayout; // The background image layout for the Spread component.
                grayAreaBackColor :string; // A color string used to represent the background color of the gray area , such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                showResizeTip :GC.Spread.Sheets.ShowResizeTip; // How to display the resize tip.
                showDragDropTip :boolean ; // Whether to display the drag-drop tip.
                showDragFillTip :boolean ; // Whether to display the drag-fill tip.
                showScrollTip :GC.Spread.Sheets.ShowScrollTip; // How to display the scroll tip.
                scrollIgnoreHidden :boolean ; // Whether the scroll bar ignores hidden rows or columns.
                highlightInvalidData :boolean ; // Whether to highlight invalid data.
                useTouchLayout :boolean ; // Whether to use touch layout to present the Spread component.
                hideSelection :boolean ; // Whether to display the selection highlighting when the Spread component does not have focus.
                resizeZeroIndicator :GC.Spread.Sheets.ResizeZeroIndicator; // The drawing policy when the row or column is resized to zero.
                allowUserEditFormula :boolean ; // Whether the user can edit formulas in a cell in the spreadsheet.
                enableFormulaTextbox :boolean ; // Whether to enable the formula text box in the spreadsheet.
                autoFitType :GC.Spread.Sheets.AutoFitType; // Whether content will be formatted to fit in cells or in cells and headers.
                referenceStyle :GC.Spread.Sheets.ReferenceStyle; // The style for cell and range references in cell formulas on this sheet.
                allowDynamicArray :boolean ; //Whether to enable the dynamic array.
                calcOnDemand :boolean; // Whether to calculate formulas only when they are demanded.
                allowCopyPasteExcelStyle :boolean ; //Whether the user can copy style from Spread Sheets then paste to Excel, or copy style from Excel then paste to Spread Sheets.
                allowExtendPasteRange :boolean ; //Whether extend paste range if the paste range is not enough for pasting.
                copyPasteHeaderOptions :GC.Spread.Sheets.CopyPasteHeaderOptions; // Which headers are included when data is copied to or pasted.
                scrollByPixel: boolean; // Whether to enable the precision scrolling by pixel.
                scrollPixel: number; // Decides scrolling by that number of pixels at a time when scrollByPixel is true. The final scrolling pixels are the result of scrolling delta multiply scrollPixel. For example, the scrolling delta is 3, and the scrollPixel is 5, the final scrolling pixels are 15.
                allowAutoCreateHyperlink: boolean; // Whether to enable auto creating hyperlink in the spreadsheet.
                columnResizeMode: GC.Spread.Sheets.ResizeMode; // Specifies the way to resize column.
                rowResizeMode: GC.Spread.Sheets.ResizeMode; // Specifies the way to resize row.
                customList?: string[][]; // The list for user to customize drag fill, prioritize matching this list in each fill. Each array item is type of string array.
            }


            export interface IWorkFlowItem{
                value: string;
                transitions: number[]|string[];
            }


            export interface IWorkFlowOption{
                items:GC.Spread.Sheets.IWorkFlowItem[];
            }


            export interface IWorkSheetGridlineOption{
                color?: string; //The grid line color
                showVerticalGridline?: boolean; //Whether to show the vertical grid line.
                showHorizontalGridline?: boolean; //Whether to show the horizontal grid line.
            }


            export interface IWorksheetOptions{
                allowCellOverflow: boolean; //indicates whether data can overflow into adjacent empty cells.
                showFormulas: boolean; //indicates whether display the formulas string not the formula result.
                sheetTabColor: string; //A color string used to represent the sheet tab color, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                frozenlineColor: string; //A color string used to represent the frozen line color, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                clipBoardOptions: GC.Spread.Sheets.ClipboardPasteOptions; //The clipboard option.
                gridline: GC.Spread.Sheets.IWorkSheetGridlineOption; //The grid line's options.
                rowHeaderVisible: boolean; //Indicates whether the row header is visible.
                colHeaderVisible: boolean; //Indicates whether the column header is visible.
                rowHeaderAutoText: GC.Spread.Sheets.HeaderAutoText; //Indicates whether the row header displays letters or numbers or is blank.
                colHeaderAutoText: GC.Spread.Sheets.HeaderAutoText; //Indicates whether the column header displays letters or numbers or is blank.
                rowHeaderAutoTextIndex: GC.Spread.Sheets.HeaderAutoText; //Specifies which row header column displays the automatic text when there are multiple row header columns.
                colHeaderAutoTextIndex: GC.Spread.Sheets.HeaderAutoText; //Specifies which column header row displays the automatic text when there are multiple column header rows.
                isProtected: boolean; //Indicates whether cells on this sheet that are marked as protected cannot be edited.
                protectionOptions: GC.Spread.Sheets.IProtectionOptions; //A value that indicates the elements that you want users to be able to change.
                selectionBackColor: string; //The selection's background color for the sheet.
                selectionBorderColor: string; //The selection's border color for the sheet.
            }


            export interface IWorkSheetSheetAreaOffsetOption{
                left?: number; //The offset left of sheet from host.
                top?: number; //The offset top of sheet from host
            }


            export interface SheetTabInfo{
                sheetName?: string;
                sheetIndex?: number;
            }

            /**
             * Represents whether the component automatically resizes cells or headers.
             * @enum {number}
             * @example
             * //This example uses the AutoFitType enumeration.
             * spread.options.autoFitType = GC.Spread.Sheets.AutoFitType.cellWithHeader;
             */
            export enum AutoFitType{
                /**
                 *  The component autofits cells.
                 */
                cell= 0,
                /**
                 *   The component autofits cells and headers.
                 */
                cellWithHeader= 1
            }

            /**
             * Specifies the type of cellbutton.
             * @enum {number}
             * @example
             * //This example sets the type of cellbutton.
             * //create config
             * leftButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     isLeft: true,
             *     imageType:GC.Spread.Sheets.ButtonImageType.always,
             * };
             * rightButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     isLeft: false,
             *     imageType:GC.Spread.Sheets.ButtonImageType.onseleciton,
             * };
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.cellButtons=[
             *      leftButtonConfig1,
             *      rightButtonConfig1
             * ];
             * sheet.setStyle(0, 0, style);
             */
            export enum ButtonImageType{
                /** Indicates the cellbutton image type is none.
                 * @type {number}
                 */
                none= 0,
                /** Indicates the cellbutton type is custom.
                 * @type {number}
                 */
                custom= 1,
                /** Indicates the cellbutton type is clear.
                 * @type {number}
                 */
                clear= 2,
                /** Indicates the cellbutton type is calcel.
                 * @type {number}
                 */
                calcel= 3,
                /** Indicates the cellbutton type is ok.
                 * @type {number}
                 */
                ok= 4,
                /** Indicates the cellbutton type is dropdown.
                 * @type {number}
                 */
                dropdown= 5,
                /** Indicates the cellbutton type is ellipsis.
                 * @type {number}
                 */
                ellipsis= 6,
                /** Indicates the cellbutton type is left.
                 * @type {number}
                 */
                left= 7,
                /** Indicates the cellbutton type is right.
                 * @type {number}
                 */
                right= 8,
                /** Indicates the cellbutton type is plus.
                 * @type {number}
                 */
                plus= 9,
                /** Indicates the cellbutton type is minus.
                 * @type {number}
                 */
                minus= 10,
                /** Indicates the cellbutton type is undo.
                 * @type {number}
                 */
                undo= 11,
                /** Indicates the cellbutton type is redo.
                 * @type {number}
                 */
                redo= 12,
                /** Indicates the cellbutton type is search.
                 * @type {number}
                 */
                search= 13,
                /** Indicates the cellbutton type is separator.
                 * @type {number}
                 */
                separator= 14,
                /** Indicates the cellbutton type is spinLeft.
                 * @type {number}
                 */
                spinLeft= 15,
                /** Indicates the cellbutton type is spinRight.
                 * @type {number}
                 */
                spinRight= 16
            }

            /**
             * Specifies the position of cellbutton.
             * @enum {number}
             * @example
             * //This example sets the position of cellbutton.
             * //create config
             * leftButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     position: GC.Spread.Sheets.ButtonPosition.right,
             *     visibility:GC.Spread.Sheets.ButtonVisibility.always,
             * };
             * rightButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     position: GC.Spread.Sheets.ButtonPosition.left,
             *     visibility:GC.Spread.Sheets.ButtonVisibility.onseleciton,
             * };
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.cellButtons=[
             *      leftButtonConfig1,
             *      rightButtonConfig1
             * ];
             * sheet.setStyle(0, 0, style);
             */
            export enum ButtonPosition{
                /** Indicates the cellbutton is in left of the cell.
                 * @type {number}
                 */
                left= 0,
                /** Indicates the cellbutton is in right of the cell.
                 * @type {number}
                 */
                right= 1
            }

            /**
             * Specifies the visibility of cellbutton.
             * @enum {number}
             * @example
             * //This example sets the visibility of cellbutton.
             * //create config
             * leftButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     isLeft: true,
             *     visibility:GC.Spread.Sheets.ButtonVisibility.always,
             * };
             * rightButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     isLeft: false,
             *     visibility:GC.Spread.Sheets.ButtonVisibility.onseleciton,
             * };
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.cellButtons=[
             *      leftButtonConfig1,
             *      rightButtonConfig1
             * ];
             * sheet.setStyle(0, 0, style);
             */
            export enum ButtonVisibility{
                /** Indicates the cellbutton alway visible.
                 * @type {number}
                 */
                always= 0,
                /** Indicates the cellbutton visible when the cell is active.
                 * @type {number}
                 */
                onSelected= 1,
                /** Indicates the cellbutton visible when the cell enter edit.
                 * @type {number}
                 */
                onEditing= 2
            }

            /**
             * Specifies the calendar's default page.
             * @enum {number}
             */
            export enum CalendarPage{
                /**
                 * Specifies the calendar's default page is year.
                 */
                year= 1,
                /**
                 * Specifies the calendar's default page is month.
                 */
                month= 2,
                /**
                 * Specifies the calendar's default page is day.
                 */
                day= 3
            }

            /**
             * Specifies the calendar's start day.
             * @enum {number}
             */
            export enum CalendarStartDay{
                /**
                 * Specifies the calendar's start day is Monday.
                 */
                monday= 1,
                /**
                 * Specifies the calendar's start day is Tuesday.
                 */
                tuesday= 2,
                /**
                 * Specifies the calendar's start day is Wednesday.
                 */
                wednesday= 3,
                /**
                 * Specifies the calendar's start day is Thursday.
                 */
                thursday= 4,
                /**
                 * Specifies the calendar's start day is Friday.
                 */
                friday= 5,
                /**
                 * Specifies the calendar's start day is Saturday.
                 */
                saturday= 6,
                /**
                 * Specifies the calendar's start day is Sunday.
                 */
                sunday= 7
            }

            /**
             * Specifies the position of caption.
             * @enum {number}
             * @example
             * //This example sets the position of caption.
             * //create config
             * leftButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     position: GC.Spread.Sheets.ButtonPosition.right,
             *     captionAlign:GC.Spread.Sheets.CaptionAlignment.right,
             * };
             * rightButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     position: GC.Spread.Sheets.ButtonPosition.left,
             *     captionAlign:GC.Spread.Sheets.CaptionAlignment.left,
             * };
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.cellButtons=[
             *      leftButtonConfig1,
             *      rightButtonConfig1
             * ];
             * sheet.setStyle(0, 0, style);
             */
            export enum CaptionAlignment{
                /** Indicates the caption is in left of the button.
                 * @type {number}
                 */
                left= 0,
                /** Indicates the caption is in right of the button.
                 * @type {number}
                 */
                right= 1
            }

            /**
             * Specifies the type of CellState.
             * @enum {number}
             */
            export enum CellStatesType{
                /** When mouse hover on the cell , cell state include "hover" state.
                 * @type {number}
                 */
                hover= 1,
                /** When the data-validation conditional evaluate fail, cell state include "invalid" state.
                 * @type {number}
                 */
                invalid= 2,
                /** When the cell is locked in protect worksheet , cell state include "readonly" state.
                 * @type {number}
                 */
                readonly= 4,
                /** When the cell is editing, cell state include "edit" state.
                 * @type {number}
                 */
                edit= 8,
                /** When the cell is focus, cell state include "active" state.
                 * @type {number}
                 */
                active= 16,
                /** When the cell is in the selection range, cell state include "selected" state.
                 * @type {number}
                 */
                selected= 32,
                /** When cell value is changed, cell state include "dirty" state.
                 * @type {number}
                 */
                dirty= 64
            }

            /**
             * Specifies what data is pasted from the Clipboard.
             * @enum {number}
             * @example
             * //This example uses the ClipboardPasteOptions enumeration.
             * activeSheet.options.clipBoardOptions = GC.Spread.Sheets.ClipboardPasteOptions.values;
             */
            export enum ClipboardPasteOptions{
                /**
                 * Pastes all data objects, including values, formatting, and formulas.
                 */
                all= 0,
                /**
                 * Pastes only values.
                 */
                values= 1,
                /**
                 * Pastes only formatting.
                 */
                formatting= 2,
                /**
                 * Pastes only formulas.
                 */
                formulas= 3,
                /**
                 * Pastes values and formatting.
                 */
                valuesAndFormatting= 4,
                /**
                 * Pastes formulas and formatting.
                 */
                formulasAndFormatting= 5
            }

            /**
             * Specifies which headers are included when data is copied to or pasted.
             * @enum {number}
             * @example
             * //This example allows you to copy and paste headers.
             * spread.options.copyPasteHeaderOptions = GC.Spread.Sheets.CopyPasteHeaderOptions.allHeaders;
             * activeSheet.setRowCount(2,GC.Spread.Sheets.SheetArea.colHeader);
             * activeSheet.setColumnCount(2,GC.Spread.Sheets.SheetArea.rowHeader);
             * activeSheet.setValue(0, 2,"Column",GC.Spread.Sheets.SheetArea.colHeader);
             * activeSheet.setValue(1, 0,"Row",GC.Spread.Sheets.SheetArea.rowHeader);
             */
            export enum CopyPasteHeaderOptions{
                /**
                 * Includes neither column nor row headers when data is copied; does not overwrite selected column or row headers when data is pasted.
                 */
                noHeaders= 0,
                /**
                 * Includes selected row headers when data is copied; overwrites selected row headers when data is pasted.
                 */
                rowHeaders= 1,
                /**
                 * Includes selected column headers when data is copied; overwrites selected column headers when data is pasted.
                 */
                columnHeaders= 2,
                /**
                 * Includes selected headers when data is copied; overwrites selected headers when data is pasted.
                 */
                allHeaders= 3
            }

            /**
             * Specifies the copy to option.
             * @enum {number}
             * @example
             * //This example uses the CopyToOption enumeration.
             * activeSheet.getCell(0,0).value("1");
             * activeSheet.copyTo(0,0,1,1,2,2,GC.Spread.Sheets.CopyToOptions.value);
             */
            export enum CopyToOptions{
                /**
                 * Indicates the type of data is pure data.
                 */
                value= 1,
                /**
                 * Indicates the type of data is a formula.
                 */
                formula= 2,
                /**
                 * Indicates the type of data is a comment.
                 */
                comment= 4,
                /**
                 * Indicates to copy a range group.
                 */
                outline= 8,
                /**
                 * Indicates the type of data is a sparkline.
                 */
                sparkline= 16,
                /**
                 * Indicates to copy a span.
                 */
                span= 32,
                /**
                 * Indicates the type of data is a style.
                 */
                style= 64,
                /**
                 * Indicates the type of data is a tag.
                 */
                tag= 128,
                /**
                 * Indicates the type of data is a binding path.
                 */
                bindingPath= 256,
                /**
                 * Indicates the type of data is a conditional format.
                 */
                conditionalFormat= 512,
                /**
                 * Indicates the type of data is a hyperlink.
                 */
                hyperlink= 1024,
                /**
                 * Indicates all types of data.
                 */
                all= 2047
            }

            /**
             * Specifies the cell value type of dropdown list result.
             * @enum {number}
             */
            export enum DropdownListValue{
                /**
                 * Use the string of list select result, use comma to split.
                 */
                string= 0,
                /**
                 * Specifies the sort will move the group containt with sort.
                 */
                array= 1
            }

            /**
             * Specifies the type of dropdown.
             * @enum {number}
             * @example
             * //This example sets the type of dropdown.
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.dropDowns=[
             *    {
             *      type:GC.Spread.Sheets.DropDownType.dateTimePicker,
             *      option: {
             *          showTime:false
             *      }
             *    }
             * ];
             * sheet.setStyle(0, 0, style);
             */
            export enum DropDownType{
                /** Indicates the type of dropdown is colorpicker.
                 * @type {number}
                 */
                colorPicker= 0,
                /** Indicates the type of dropdown is dateTimePicker.
                 * @type {number}
                 */
                dateTimePicker= 1,
                /** Indicates the type of dropdown is timePicker.
                 * @type {number}
                 */
                timePicker= 2,
                /** Indicates the type of dropdown is monthPicker.
                 * @type {number}
                 */
                monthPicker= 3,
                /** Indicates the type of dropdown is list.
                 * @type {number}
                 */
                list= 4,
                /** Indicates the type of dropdown is slider.
                 * @type {number}
                 */
                slider= 5,
                /** Indicates the type of dropdown is calculator.
                 * @type {number}
                 */
                calculator= 6,
                /** Indicates the type of dropdown is workflowList.
                 * @type {number}
                 */
                workflowList= 7
            }

            /**
             * Specifies the editor status.
             * @enum {number}
             */
            export enum EditorStatus{
                /**
                 * Cell is in Ready mode.
                 */
                ready= 0,
                /**
                 * Cell is in editing mode and can commit the input value and navigate to or select other cells when invoking navigation or selection actions.
                 */
                enter= 1,
                /**
                 * Cell is in editing mode and cannot commit the input value and navigate to or select other cells.
                 */
                edit= 2
            }

            /**
             * Specifies which default labels are displayed in headers.
             * @enum {number}
             */
            export enum HeaderAutoText{
                /**
                 *  Displays blanks in the headers.
                 */
                blank= 0,
                /**
                 *  Displays numbers in the headers.
                 */
                numbers= 1,
                /**
                 *  Displays letters in the headers.
                 */
                letters= 2
            }

            /**
             * Specifies the horizontal alignment.
             * @enum {number}
             * @example
             * //This example uses the HorizontalAlign type.
             * var style = new GC.Spread.Sheets.Style();
             * style.font = "8pt Arial";
             * style.hAlign = GC.Spread.Sheets.HorizontalAlign.center;
             * style.vAlign = GC.Spread.Sheets.VerticalAlign.center;
             * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(1,1).value("B2");
             */
            export enum HorizontalAlign{
                /**
                 *  Indicates that the cell content is left-aligned.
                 */
                left= 0,
                /**
                 *  Indicates that the cell content is centered.
                 */
                center= 1,
                /**
                 *  Indicates that the cell content is right-aligned.
                 */
                right= 2,
                /**
                 *  Indicates that the horizontal alignment is based on the value type.
                 */
                general= 3
            }

            /**
             * Specifies the horizontal position of the cell or column in the component.
             * @enum {number}
             * @example
             * //This example uses the HorizontalPosition enumeration.
             * activeSheet.setActiveCell(10,5);
             * activeSheet.showCell(10, 5, GC.Spread.Sheets.VerticalPosition.top, GC.Spread.Sheets.HorizontalPosition.center);
             */
            export enum HorizontalPosition{
                /**
                 *  Positions the cell or column to the left.
                 */
                left= 0,
                /**
                 *  Positions the cell or column in the center.
                 */
                center= 1,
                /**
                 *  Positions the cell or column to the right.
                 */
                right= 2,
                /**
                 *  Positions the cell or column to the nearest edge.
                 */
                nearest= 3
            }

            /**
             * Defines the background image layout.
             * @enum {number}
             * @example
             * var rowImage = "./css/images/quarter1.png";
             * sheet.getCell(1, -1).backgroundImage(rowImage);
             * sheet.getCell(1, -1).backgroundImageLayout(GC.Spread.Sheets.ImageLayout.center);
             */
            export enum ImageLayout{
                /** Specifies that the background image fills the area.
                 * @type {number}
                 */
                stretch= 0,
                /** Specifies that the background image displays in the center of the area.
                 * @type {number}
                 */
                center= 1,
                /** Specifies that the background image displays in the area with its original aspect ratio.
                 * @type {number}
                 */
                zoom= 2,
                /** Specifies that the background image displays in the upper left corner of the area with its original size.
                 * @type {number}
                 */
                none= 3
            }

            /**
             * Defines the IME mode to control the state of the Input Method Editor (IME).
             * @enum {number}
             * @example
             * //This example uses the imeMode method.
             * activeSheet.getRange(-1, 2, -1, 1).imeMode(GC.Spread.Sheets.ImeMode.auto);
             */
            export enum ImeMode{
                /**
                 * No change is made to the current input method editor state.
                 */
                auto= 1,
                /** All characters are entered through the IME. Users can still deactivate the IME.
                 */
                active= 2,
                /**
                 * All characters are entered without IME. Users can still activate the IME.
                 */
                inactive= 4,
                /**
                 * The input method editor is disabled and may not be activated by the user.
                 */
                disabled= 0
            }

            /**
             * Specifies paste the direction of the insertion finger.
             * @enum {number}
             */
            export enum InsertShiftCell{
                /** Specifies paste the direction right.
                 * @type {number}
                 */
                right= 0,
                /** Specifies paste the direction down.
                 * @type {number}
                 */
                down= 1
            }

            /**
             * Identifies which operation was invalid.
             * @enum {number}
             * @example
             * //This example gets the invalid type.
             * activeSheet.getCell(1,1).locked(true);
             * activeSheet.options.isProtected = true;
             * activeSheet.getCell(1,1).value(2);
             * //Bind event
             * activeSheet.bind(GC.Spread.Sheets.Events.InvalidOperation, function (e, info) {
             *         alert("Message (" + info.invalidType + ")");
             * });
             */
            export enum InvalidOperationType{
                /**
                 * Specifes the formula is invalid.
                 */
                setFormula= 0,
                /**
                 * Specifies the copy paste is invalid.
                 */
                copyPaste= 1,
                /**
                 * Specifies the drag fill is invalid.
                 */
                dragFill= 2,
                /**
                 * Specifies the drag drop is invalid.
                 */
                dragDrop= 3,
                /**
                 * Specifies the insert row is invalid.
                 */
                changePartOfArrayFormula= 4,
                /**
                 * Specifies the changed sheet name is invalid.
                 */
                changeSheetName= 5
            }

            /**
             * Specifies the cell label position.
             * @enum {number}
             * @example
             * //This example sets cell padding, alignment, and other options.
             * var type = new GC.Spread.Sheets.Style();
             * type.watermark = "User name";
             * type.cellPadding = "20";
             * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
             * activeSheet.setStyle(0, 1, type);
             * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
             * activeSheet.getRange(-1, 1, -1, 1).width(150);
             * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
             * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
             * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
             * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
             * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
             * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
             */
            export enum LabelAlignment{
                /**
                 *  Indicates that the cell label position is top-left.
                 */
                topLeft= 0,
                /**
                 *  Indicates that the cell label position is top-center.
                 */
                topCenter= 1,
                /**
                 *  Indicates that the cell label position is top-right.
                 */
                topRight= 2,
                /**
                 *  Indicates that the cell label position is bottom-left.
                 */
                bottomLeft= 3,
                /**
                 *  Indicates that the cell label position is bottom-center.
                 */
                bottomCenter= 4,
                /**
                 *  Indicates that the cell label position is bottom-right.
                 */
                bottomRight= 5
            }

            /**
             * Specifies the cell label visibility.
             * @enum {number}
             * @example
             * //This example sets label options for the watermark.
             * var type = new GC.Spread.Sheets.Style();
             * type.watermark = "User name";
             * type.cellPadding = "20";
             * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
             * activeSheet.setStyle(0, 1, type);
             * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
             * activeSheet.getRange(-1, 1, -1, 1).width(150);
             * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
             * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
             * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
             * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
             * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
             * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
             */
            export enum LabelVisibility{
                /**
                 *  Specifies to always show the watermark in the padding area and not to show the watermark in the cell area, regardless of the cell value.
                 */
                visible= 0,
                /**
                 *  Specifies to not show the watermark in the padding area, but to show the watermark in the cell area based on a value condition.
                 */
                hidden= 1,
                /**
                 *  Specifies to show the watermark in the padding area when the cell has a value or to show the watermark in the cell area if the cell does not have a value.
                 */
                auto= 2
            }

            /**
             * Specifies the layout direction.
             * @enum {number}
             */
            export enum LayoutDirection{
                /**
                 * Specifies layout direction is horizontal.
                 */
                horizontal= 0,
                /**
                 * Specifies layout direction is vertical.
                 */
                vertical= 1
            }

            /**
             * Specifies the layout display.
             * @enum {number}
             */
            export enum LayoutDisplayAs{
                /**
                 * Specifies layout display is inline.
                 */
                inline= 0,
                /**
                 * Specifies layout display is popup
                 */
                popup= 1,
                /**
                 * Specifies layout display is tree
                 */
                tree= 2
            }

            /**
             * Specifies the line drawing style for the border.
             * @enum {number}
             * @example
             * //This example creates a border.
             * var border = new GC.Spread.Sheets.LineBorder
             * border.color = "#7FFFD4";
             * border.style = GC.Spread.Sheets.LineStyle.double;
             * var cell = activeSheet.getCell(1, 1, GC.Spread.Sheets.SheetArea.viewport);
             * cell.borderLeft(border);
             */
            export enum LineStyle{
                /**
                 * Indicates a border line without a style.
                 */
                empty= 0,
                /**
                 *  Indicates a border line with a solid thin line.
                 */
                thin= 1,
                /**
                 *  Indicates a medium border line with a solid line.
                 */
                medium= 2,
                /**
                 *  Indicates a border line with dashes.
                 */
                dashed= 3,
                /**
                 *  Indicates a border line with dots.
                 */
                dotted= 4,
                /**
                 *  Indicates a thick border line with a solid line.
                 */
                thick= 5,
                /**
                 *  Indicates a double border line.
                 */
                double= 6,
                /**
                 *  Indicates a border line with all dots.
                 */
                hair= 7,
                /**
                 *  Indicates a medium border line with dashes.
                 */
                mediumDashed= 8,
                /**
                 *  Indicates a border line with dash-dot.
                 */
                dashDot= 9,
                /**
                 *  Indicates a medium border line with dash-dot-dot.
                 */
                mediumDashDot= 10,
                /**
                 *  Indicates a border line with dash-dot-dot.
                 */
                dashDotDot= 11,
                /**
                 *  Indicates a medium border line with dash-dot-dot.
                 */
                mediumDashDotDot= 12,
                /**
                 *  Indicates a slanted border line with dash-dot.
                 */
                slantedDashDot= 13
            }

            /**
             * Defines the type of action that raised the RangeChanged event.
             * @enum {number}
             */
            export enum RangeChangedAction{
                /**
                 * Indicates drag drop undo action.
                 */
                dragDrop= 0,
                /**
                 * Indicates drag fill undo action.
                 */
                dragFill= 1,
                /**
                 * Indicates clear range value undo action.
                 */
                clear= 2,
                /**
                 * Indicates paste undo action.
                 */
                paste= 3,
                /**
                 * Indicates sorting a range of cells.
                 */
                sort= 4,
                /**
                 * Indicates setting a formula in a specified range of cells .
                 */
                setArrayFormula= 5,
                /**
                 * Indicates setting a formula in a specified range of cells .
                 */
                evaluateFormula= 6
            }

            /**
             * Specifies the formula reference style.
             * @enum {number}
             * @example
             * //This example sets the reference style for the active sheet.
             * activeSheet.options.referenceStyle = GC.Spread.Sheets.ReferenceStyle.r1c1;
             */
            export enum ReferenceStyle{
                /**
                 * Indicates a1 style.
                 */
                a1= 0,
                /**
                 * Indicates r1c1 style.
                 */
                r1c1= 1
            }

            /**
             * Specifies the way resizing use.
             * @enum {number}
             * @example
             * //This example shows you how to specify the resize way.
             * spread.options.columnResizeMode = GC.Spread.Sheets.ResizeMode.split;
             * spread.options.rowResizeMode = GC.Spread.Sheets.ResizeMode.normal;
             */
            export enum ResizeMode{
                /** Specifies use normal way to resize.
                 * @type {number}
                 */
                normal= 0,
                /** Specifies use split way to resize.
                 * @type {number}
                 */
                split= 1
            }

            /**
             * Specifies the drawing policy of the row or column when it is resized to zero.
             * @enum {number}
             * @example
             * //This example displays double grid lines for the column or row with zero width or height.
             * spread.options.resizeZeroIndicator = GC.Spread.Sheets.ResizeZeroIndicator.enhanced;
             * activeSheet.getRange(-1, 2, -1, 1).width(0);
             * activeSheet.getRange(1, -1, 1, -1).height(0);
             */
            export enum ResizeZeroIndicator{
                /**
                 *  Uses the current drawing policy when the row or column is resized to zero.
                 */
                default= 0,
                /**
                 * Draws two short lines when the row or column is resized to zero.
                 */
                enhanced= 1
            }

            /**
             * Specifies how users can select items in the control.
             * @enum {number}
             * @example
             * //This example sets the selection policy.
             * activeSheet.selectionUnit(GC.Spread.Sheets.SelectionUnit.row);
             * activeSheet.selectionPolicy(GC.Spread.Sheets.SelectionPolicy.range);
             */
            export enum SelectionPolicy{
                /**
                 * Allows users to only select single items.
                 */
                single= 0,
                /**
                 * Allows users to select single items and ranges of items, but not multiple ranges.
                 */
                range= 1,
                /**
                 * Allows users to select single items and ranges of items, including multiple ranges.
                 */
                multiRange= 2
            }

            /**
             * Specifies the smallest unit users or the application can select.
             * @enum {number}
             * @example
             * //This example sets the unit type.
             * activeSheet.selectionUnit(GC.Spread.Sheets.SelectionUnit.row);
             * activeSheet.selectionPolicy(GC.Spread.Sheets.SelectionPolicy.range);
             */
            export enum SelectionUnit{
                /**
                 * Indicates that the smallest unit that can be selected is a cell.
                 */
                cell= 0,
                /**
                 * Indicates that the smallest unit that can be selected is a row.
                 */
                row= 1,
                /**
                 * Indicates that the smallest unit that can be selected is a column.
                 */
                column= 2
            }

            /**
             * Specifies the sheet area.
             * @enum {number}
             * @example
             * //Creates log text describing which area in the sheet was clicked.
             * // Use IE to see the console log text
             * activeSheet.bind(GC.Spread.Sheets.Events.CellClick, function (sender, args) {
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
             *         console.log("The column header was clicked.");
             *     }
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
             *         console.log("The row header was clicked.");
             *     }
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
             *         console.log("The corner header was clicked.");
             *     }
             *     console.log("Clicked column index: " + args.col);
             *     console.log("Clicked row index: " + args.row);
             * });
             * //Bind event
             * activeSheet.bind(GC.Spread.Sheets.Events.CellDoubleClick, function (sender, args) {
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
             *         console.log("The column header was double clicked.");
             *     }
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
             *         console.log("The row header was double clicked.");
             *     }
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
             *         console.log("The corner header was double clicked.");
             *     }
             *     console.log("Double clicked column index: " + args.col);
             *     console.log("Double clicked row index: " + args.row);
             * });
             */
            export enum SheetArea{
                /**
                 * Indicates the sheet corner.
                 */
                corner= 0,
                /**
                 * Indicates the column header.
                 */
                colHeader= 1,
                /**
                 * Indicates the row header.
                 */
                rowHeader= 2,
                /**
                 * Indicates the viewport.
                 */
                viewport= 3
            }

            /**
             * Defines how the resize tip is displayed.
             * @enum {number}
             * @example
             * //This example displays both resize tips.
             * spread.options.showDragDropTip = true;
             * spread.options.showDragFillTip = true;
             * spread.options.showScrollTip = GC.Spread.Sheets.ShowScrollTip.both;
             * spread.options.showResizeTip = GC.Spread.Sheets.ShowResizeTip.both;
             */
            export enum ShowResizeTip{
                /** Specifies that no resize tip is displayed.
                 * @type {number}
                 */
                none= 0,
                /** Specifies that only the horizontal resize tip is displayed.
                 * @type {number}
                 */
                column= 1,
                /** Specifies that only the vertical resize tip is displayed.
                 * @type {number}
                 */
                row= 2,
                /** Specifies that horizontal and vertical resize tips are displayed.
                 * @type {number}
                 */
                both= 3
            }

            /**
             * Specifies how the scroll tip is displayed.
             * @enum {number}
             * @example
             * //This example displays both resize tips.
             * spread.options.showDragDropTip = true;
             * spread.options.showDragFillTip = true;
             * spread.options.showScrollTip = GC.Spread.Sheets.ShowScrollTip.both;
             * spread.options.showResizeTip = GC.Spread.Sheets.ShowResizeTip.both;
             */
            export enum ShowScrollTip{
                /** Specifies that no scroll tip is displayed.
                 * @type {number}
                 */
                none= 0,
                /** Specifies that only the horizontal scroll tip is displayed.
                 * @type {number}
                 */
                horizontal= 1,
                /** Specifies that only the vertical scroll tip is displayed.
                 * @type {number}
                 */
                vertical= 2,
                /** Specifies that horizontal and vertical scroll tips are displayed.
                 * @type {number}
                 */
                both= 3
            }

            /**
             * Specifies the type of sorting to perform.
             * @enum {number}
             * @example
             * //This example sets the sort order of the items in the slicer.
             * //create a table
             * datas = [
             *     ["1", "NewYork", "1968/6/8", "80", "180"],
             *     ["4", "NewYork", "1972/7/3", "72", "168"],
             *     ["4", "NewYork", "1964/3/2", "71", "179"],
             *     ["5", "Washington", "1972/8/8","80", "171"],
             *     ["6", "Washington", "1986/2/2", "89", "161"],
             *     ["7", "Washington", "2012/2/15", "71", "240"]];
             * var table = activeSheet.tables.addFromDataSource("table1", 2, 2, datas);
             * dataColumns = ["Name", "City", "Birthday", "Weight", "Height"];
             * table.setColumnName(0, dataColumns[0]);
             * table.setColumnName(1, dataColumns[1]);
             * table.setColumnName(2, dataColumns[2]);
             * table.setColumnName(3, dataColumns[3]);
             * table.setColumnName(4, dataColumns[4]);
             * //create style
             * var style1 = new GC.Spread.Sheets.Slicers.SlicerStyles.light4();
             * //add a slicer to the sheet and return the slicer instance
             * var slicer = activeSheet.slicers.add("slicer1",table.name(),"Height");
             * //set slicer properties
             * slicer.position(new GC.Spread.Sheets.Point(100, 200));
             * slicer.sortState(GC.Spread.Sheets.SortState.descending);
             * slicer.style(style1);
             */
            export enum SortState{
                /** Indicates the sorting is disabled.
                 * @type {number}
                 */
                none= 0,
                /** Indicates the sorting is ascending.
                 * @type {number}
                 */
                ascending= 1,
                /** Indicates the sorting is descending.
                 * @type {number}
                 */
                descending= 2
            }

            /**
             * Represents the storage data type.
             * @enum {number}
             * @example
             * //This example uses the StorageType enumeration.
             * activeSheet.getCell(0,0).value("A1");
             * activeSheet.clear(0,0,3,3,GC.Spread.Sheets.SheetArea.viewport,GC.Spread.Sheets.StorageType.data);
             */
            export enum StorageType{
                /**
                 *  Indicates the storage data type is pure value.
                 */
                data= 1,
                /**
                 *  Indicates the storage data type is style.
                 */
                style= 2,
                /**
                 *  Indicates the storage data type is comment.
                 */
                comment= 4,
                /**
                 *  Indicates the storage data type is tag.
                 */
                tag= 8,
                /**
                 *  Indicates the storage data type is sparkline.
                 */
                sparkline= 16,
                /**
                 *  Indicates the storage data type is the axis information.
                 */
                axis= 32,
                /**
                 *  Indicates the storage data type is data binding path.
                 */
                bindingPath= 64,
                /**
                 *  Indicates the storage data type is hyperlink.
                 */
                hyperlink= 256
            }

            /**
             * Defines the type of the text decoration.
             * @enum {number}
             * @example
             * //This example uses the TextDecorationType enumeration.
             * activeSheet.getCell(0, 0).textDecoration(GC.Spread.Sheets.TextDecorationType.underline);
             * activeSheet.getRange(1, -1, 1, -1).textDecoration(GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.underline);
             * activeSheet.getRange(-1, 1, -1, 1).textDecoration(GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.lineThrough | GC.Spread.Sheets.TextDecorationType.underline);
             * var style = new GC.Spread.Sheets.Style();
             * style.textDecoration = GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.underline;
             * activeSheet.setStyle(1, 1, style, GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(0, 0).value("Test");
             * activeSheet.getCell(1, 0).value("Test");
             * activeSheet.getCell(0, 1).value("Test");
             */
            export enum TextDecorationType{
                /** Specifies to display a line below the text.
                 */
                underline= 1,
                /** Specifies to display a line through the text.
                 */
                lineThrough= 2,
                /** Specifies to display a line above the text.
                 */
                overline= 4,
                /** Specifies to display double line below the text.
                 */
                doubleUnderline= 8,
                /** Specifies normal text.
                 */
                none= 0
            }

            /**
             * Specifies cell value's type.
             * @enum {number}
             */
            export enum ValueType{
                /**
                 *  indicate normal value type.
                 */
                normal= 0,
                /**
                 *  indicate rich text value type.
                 */
                richText= 1
            }

            /**
             * Defines the type of the text vertAlign.
             * @enum {number}
             * @example
             * //This example uses the VertAlign enumeration.
             * activeSheet.setValue(1, 1, {richText:[{style:{vertAlign:GC.Spread.Sheets.Style.VertAlign.subscript},text:'SpreadJS'}]}, GC.Spread.Sheets.SheetArea.viewport);
             */
            export enum VertAlign{
                /**
                 *  indicate normal text align.
                 */
                normal= 0,
                /**
                 *  indicate superscript.
                 */
                superscript= 1,
                /**
                 *  indicate subscript.
                 */
                subscript= 2
            }

            /**
             * Specifies the vertical alignment.
             * @enum {number}
             * @example
             * //This example uses the VerticalAlign type.
             * var style = new GC.Spread.Sheets.Style();
             * style.font = "8pt Arial";
             * style.hAlign = GC.Spread.Sheets.HorizontalAlign.center;
             * style.vAlign = GC.Spread.Sheets.VerticalAlign.center;
             * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(1,1).value("B2");
             */
            export enum VerticalAlign{
                /**
                 *  Indicates that the cell content is top-aligned.
                 */
                top= 0,
                /**
                 *  Indicates that the cell content is centered.
                 */
                center= 1,
                /**
                 *  Indicates that the cell content is bottom-aligned.
                 */
                bottom= 2
            }

            /**
             * Specifies the vertical position of the cell or row in the component.
             * @enum {number}
             * @example
             * //This example uses the VerticalPosition enumeration.
             * activeSheet.setActiveCell(10,5);
             * activeSheet.showCell(10, 5, GC.Spread.Sheets.VerticalPosition.top, GC.Spread.Sheets.HorizontalPosition.center);
             */
            export enum VerticalPosition{
                /**
                 *  Positions the cell or row at the top.
                 */
                top= 0,
                /**
                 *  Positions the cell or row in the center.
                 */
                center= 1,
                /**
                 *  Positions the cell or row at the bottom.
                 */
                bottom= 2,
                /**
                 *  Positions the cell or row at the nearest edge.
                 */
                nearest= 3
            }

            /**
             * Specifies the visual state.
             * @enum {number}
             */
            export enum VisualState{
                /**
                 * Indicates normal visual state.
                 */
                normal= 0,
                /**
                 * Indicates highlight visual state.
                 */
                highlight= 1,
                /**
                 * Indicates selected visual state.
                 */
                selected= 2,
                /**
                 * Indicates active visual state.
                 */
                active= 3,
                /**
                 * Indicates hover visual state.
                 */
                hover= 4
            }


            export class CellRange{
                /**
                 * Represents a cell range in a sheet.
                 * @class
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that contains this cell range.
                 * @param {number} row The row index of the cell.
                 * @param {number} col The column index of the cell.
                 * @param {number} [rowCount] The row count of the cell. If you do not provide this parameter, it defaults to <b>1</b>.
                 * @param {number} [colCount] The column count of the cell. If you do not provide this parameter, it defaults to <b>1</b>.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If you do not provide this parameter, it defaults to <b>viewport</b>.
                 * If row is -1 and rowCount is -1, the range represents columns. For example, new GC.Spread.Sheets.CellRange(-1,4,-1,6) represents columns "E:J".
                 * If col is -1 and colCount is -1, the range represents rows. For example, new GC.Spread.Sheets.CellRange(4,-1,6,-1) represents rows "5:10".
                 */
                constructor(sheet:  GC.Spread.Sheets.Worksheet,  row:  number,  col:  number,  rowCount?:  number,  colCount?:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea);
                /**
                 * Gets the starting column index.
                 * @type {number}
                 */
                col: number;
                /**
                 * Gets the column count.
                 * @type {number}
                 */
                colCount: number;
                /**
                 * Gets the starting row index.
                 *@type {number}
                 */
                row: number;
                /**
                 * Gets the row count.
                 * @type {number}
                 */
                rowCount: number;
                /**
                 * Gets the sheet that contains this cell range.
                 * @type {GC.Spread.Sheets.Worksheet}
                 */
                sheet: GC.Spread.Sheets.Worksheet;
                /**
                 * Gets the area that contains this cell range.
                 * @type {GC.Spread.Sheets.SheetArea}
                 */
                sheetArea: GC.Spread.Sheets.SheetArea;
                /**
                 * Gets or sets the background color for the cell, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                 * @param {string} [value] The cell background color.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell background color; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).backColor("pink");
                 */
                backColor(value?:  string): any;
                /**
                 * Gets or sets the background image for the cell.
                 * @param {string} [value] The cell background image.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell background image; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(1,1).backgroundImage("images/example.jpg");
                 */
                backgroundImage(value?:  string): any;
                /**
                 * Gets or sets the background image layout for the cell.
                 * @param {GC.Spread.Sheets.ImageLayout} [value] The cell background image layout.
                 * @returns {GC.Spread.Sheets.ImageLayout|GC.Spread.Sheets.CellRange} If no value is set, returns the cell background image layout; otherwise, returns the cell.
                 * @example
                 * var layout = GC.Spread.Sheets.ImageLayout.stretch;
                 * activeSheet.getRange(-1, 1, -1, 1, GC.Spread.Sheets.SheetArea.viewport).backgroundImageLayout(layout);
                 */
                backgroundImageLayout(value?:  GC.Spread.Sheets.ImageLayout): any;
                /**
                 * Gets or sets the binding path for cell binding.
                 * @param {string} path The binding path for cell binding.
                 * @returns {string | GC.Spread.Sheets.CellRange} If no value is set, returns the binding path for cell binding; otherwise, returns the worksheet.
                 * @example
                 * //This example uses the bindingPath method.
                 * var test = {name: "John"};
                 * activeSheet.getCell(0,0).bindingPath( "name");
                 * activeSheet.setDataSource(new GC.Spread.Sheets.Bindings.CellBindingSource(test));
                 */
                bindingPath(path?:  string): any;
                /**
                 * Gets or sets the bottom border of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell bottom border line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell bottom border line; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderBottom(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 */
                borderBottom(value?:  GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the left border of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell left border line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell left border line; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderLeft(new GC.Spread.Sheets.LineBorder("green", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * @example
                 * activeSheet.getCell(1,1).borderLeft(new GC.Spread.Sheets.LineBorder("red", GC.Spread.Sheets.LineStyle.double));
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderLeft(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 */
                borderLeft(value?:  GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the right border of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell right border line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell right border line; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderRight(new GC.Spread.Sheets.LineBorder("green", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * @example
                 * activeSheet.getCell(1,1).borderRight(new GC.Spread.Sheets.LineBorder("red", GC.Spread.Sheets.LineStyle.double));
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderRight(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 */
                borderRight(value?:  GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the top border of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell top border line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell top border line; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderTop(new GC.Spread.Sheets.LineBorder("green", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * @example
                 * activeSheet.getCell(1,1).borderTop(new GC.Spread.Sheets.LineBorder("red", GC.Spread.Sheets.LineStyle.double));
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderTop(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 */
                borderTop(value?:  GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the cellButtons of the cell.
                 * @param {Object[]} value the cellButtons of the cell.
                 * @returns {Object[] |GC.Spread.Sheets.CellRange} If no value is set, returns the cellButtons of the cell; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).cellButtons([caption:"Text"]]);
                 * @example
                 * var cellButtons = activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).cellButtons();
                 */
                cellButtons(value?:  GC.Spread.Sheets.ICellButton[]): any;
                /**
                 * Gets or sets the cell padding.
                 * @param {string} [value] The cell padding.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the value of the cell padding; otherwise, returns the cell.
                 * @example
                 * // This example adds cell padding around the watermark.
                 * var type = new GC.Spread.Sheets.Style();
                 * type.watermark = "User name";
                 * type.cellPadding = "20";
                 * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
                 * activeSheet.setStyle(0, 1, type);
                 * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 * activeSheet.getRange(-1, 1, -1, 1).width(150);
                 * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
                 * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
                 * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
                 * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 */
                cellPadding(value?:  string): any;
                /**
                 * Gets or sets the cell type of the cell.
                 * @param {GC.Spread.Sheets.CellTypes.Base} [value] The cell type.
                 * @returns {GC.Spread.Sheets.CellTypes.Base|GC.Spread.Sheets.CellRange} If no value is set, returns the cell type; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 1, -1, 1, GC.Spread.Sheets.SheetArea.viewport).cellType(new GC.Spread.Sheets.CellTypes.CheckBox());
                 */
                cellType(value?:  GC.Spread.Sheets.CellTypes.Base): any;
                /**
                 * Clears the specified area.
                 * @param {GC.Spread.Sheets.StorageType} type The clear type.
                 */
                clear(type:  GC.Spread.Sheets.StorageType): void;
                /**
                 * Gets or sets the comment for the cell.
                 * @param {GC.Spread.Sheets.Comments.Comment} value The comment to set in the cell.
                 * @returns {GC.Spread.Sheets.Comments.Comment | GC.Spread.Sheets.CellRange} If no value is set, returns the comment in the cell; otherwise, returns the cell range.
                 * @example
                 * // This example creates a cell comment.
                 * var comment = new GC.Spread.Sheets.Comments.Comment();
                 * comment.text("new comment!");
                 * comment.backColor("yellow");
                 * comment.foreColor("green");
                 * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                 * activeSheet.getCell(5,5).comment(comment);
                 */
                comment(value?:  GC.Spread.Sheets.Comments.Comment): any;
                /**
                 * Gets or sets the diagonalDown of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell diagonalDown line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell diagonalDown line; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).diagonalDown(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 */
                diagonalDown(value?:  GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the diagonalUp of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell diagonalUp line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell diagonalUp line; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).diagonalUp(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 */
                diagonalUp(value?:  GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the dropDowns of the cell.
                 * @param {Object[]} [value] the dropDowns of the cell.
                 * @returns {Object[] |GC.Spread.Sheets.CellRange} If no value is set, returns the dropDowns of the cell; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).dropDowns([caption:"Text"]]);
                 * @example
                 * var dropDowns = activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).dropDowns();
                 */
                dropDowns(value?:  GC.Spread.Sheets.IDropdown[]): any;
                /**
                 * Gets or sets the font for the cell, such as "normal normal normal 20px/normal Arial".
                 * @param {string} [value] The cell font.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell's font; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).font("12pt Arial");
                 * @example
                 * activeSheet.getCell(1,1).font("8pt Arial");
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).font("12pt Arial");
                 */
                font(value?:  string): any;
                /**
                 * Gets or sets the color of the text in the cell, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                 * @param {string} [value] The color of the text.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell foreground color; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).foreColor("blue");
                 */
                foreColor(value?:  string): any;
                /**
                 * Gets or sets the formatter for the cell.
                 * @param {string | GC.Spread.Formatter.GeneralFormatter} [value] The cell formatter string or object.
                 * @returns {string | GC.Spread.Formatter.GeneralFormatter |GC.Spread.Sheets.CellRange} If no value is set, returns the cell formatter string or object; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(1,1).formatter("0.000%");
                 * @example
                 * activeSheet.getCell(1, -1).formatter("0.000%");
                 * activeSheet.getCell(1,0).value("2");
                 * @example
                 * activeSheet.getCell(-1, 0).formatter("0.000%");
                 * activeSheet.getCell(0,0).value("2");
                 */
                formatter(value?:  string | GC.Spread.Formatter.GeneralFormatter): any;
                /**
                 * Gets or sets the formula for the cell.
                 * @param {string} [value] The cell formula.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell formula; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(0,2).formula("DATEDIF(DATE(2003,1,1),DATE(2016,1,1),\"Y\")");
                 */
                formula(value?:  string): any;
                /**
                 * Gets or sets the horizontal alignment of the contents of the cell.
                 * @param {GC.Spread.Sheets.HorizontalAlign} [value] The horizontal alignment.
                 * @returns {GC.Spread.Sheets.HorizontalAlign|GC.Spread.Sheets.CellRange} If no value is set, returns the horizontal alignment of the contents of the cell; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).hAlign(GC.Spread.Sheets.HorizontalAlign.right);
                 */
                hAlign(value?:  GC.Spread.Sheets.HorizontalAlign): any;
                /**
                 * Gets or sets the height of the row in pixels.
                 * @param {number} [value] The cell row height.
                 * @returns {number|GC.Spread.Sheets.CellRange} If no value is set, returns the row height; otherwise, returns the row.
                 * @example
                 * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(90);
                 */
                height(value?:  number): any;
                /**
                 * Gets or sets the imeMode of the cell.
                 * @param {GC.Spread.Sheets.ImeMode} [value] The cell imeMode.
                 * @returns {GC.Spread.Sheets.ImeMode|GC.Spread.Sheets.CellRange} If no value is set, returns the cell imeMode; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(0, 0).imeMode(GC.Spread.Sheets.ImeMode.disabled);
                 * //or
                 * var style = new GC.Spread.Sheets.Style();
                 * style.imeMode = GC.Spread.Sheets.ImeMode.disabled;
                 * activeSheet.setStyle(0, 0, style);
                 * @example
                 * activeSheet.getRange(2, -1, 1, -1).imeMode(GC.Spread.Sheets.ImeMode.active);
                 * @example
                 * activeSheet.getRange(-1, 2, -1, 1).imeMode(GC.Spread.Sheets.ImeMode.auto);
                 */
                imeMode(value?:  GC.Spread.Sheets.ImeMode): any;
                /**
                 * Gets or sets whether the cell's text is vertical.
                 * @param {boolean} [value] Set to <c>true</c> to have the cell's text vertical.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell's text vertical; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).isVerticalText(false);
                 * activeSheet.setText(2,0,"This is a test");
                 * @example
                 * activeSheet.getCell(1,1).isVerticalText(true);
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).isVerticalText(true);
                 */
                isVerticalText(value?:  boolean): any;
                /**
                 * Gets or sets the cell label options.
                 * @param {Object} [value] The cell label options.
                 * @param {GC.Spread.Sheets.LabelAlignment} [value.alignment] - The cell label position.
                 * @param {GC.Spread.Sheets.LabelVisibility} [value.visibility] - The cell label visibility.
                 * @param {string} [value.font] - The cell label font.
                 * @param {string} [value.foreColor] - The cell label forecolor.
                 * @param {string} [value.margin] - The cell label margin.
                 * @returns {Object|GC.Spread.Sheets.CellRange} If no value is set, returns the value of the cell label options; otherwise, returns the cell.
                 * @example
                 * //This example sets label options for the watermark.
                 * var type = new GC.Spread.Sheets.Style();
                 * type.watermark = "User name";
                 * type.cellPadding = "20";
                 * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
                 * activeSheet.setStyle(0, 1, type);
                 * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 * activeSheet.getRange(-1, 1, -1, 1).width(150);
                 * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
                 * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
                 * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
                 * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 */
                labelOptions(value?:  GC.Spread.Sheets.ILabelOptions): any;
                /**
                 * Gets or sets whether the cell is locked. When the sheet is protected, the locked cell cannot be edited.
                 * @param {boolean} [value] Set to <c>true</c> to lock the cell.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell is locked; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).locked(true);
                 */
                locked(value?:  boolean): any;
                /**
                 * Gets or sets the quote Prefix the cell watermark.
                 * @param {boolean} [value] The value of the quote Prefix.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns the content of the quotePrefix; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(1,1).quotePrefix(true);
                 */
                quotePrefix(value?:  boolean): any;
                /**
                 * Gets or sets whether the row or column can be resized by the user.
                 * @param {boolean} [value] Set to <c>true</c> to make the row resizable.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the user can resize the row; otherwise, returns the row or column.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1. GC.Spread.Sheets.SheetArea.viewport).resizable(true);
                 */
                resizable(value?:  boolean): any;
                /**
                 * Sets the border for the specified area.
                 * @param {GC.Spread.Sheets.LineBorder} border The border line.
                 * @param {Object} option Determines which part of the cell range to set, the option object contains {all:true, left:true, top:true, right:true, bottom:true, diagonalUp:true, diagonalDown:true, outline:true,inside:true, innerHorizontal:true, innerVertical:true}
                 * @param {boolean} [options.all] - all border.
                 * @param {boolean} [options.left] - left border.
                 * @param {boolean} [options.top] - top border.
                 * @param {boolean} [options.right] - right border.
                 * @param {boolean} [options.bottom] - bottom border.
                 * @param {boolean} [options.outline] - outline border.
                 * @param {boolean} [options.inside] - inside border.
                 * @param {boolean} [options.innerHorizontal] - innerHorizontal border.
                 * @param {boolean} [options.innerVertical] - innerVertical border.
                 * @param {boolean} [options.diagonalUp] - diagonalUp border.
                 * @param {boolean} [options.diagonalDown] - diagonalDown border.
                 * @example
                 * //This example creates borders.
                 * sheet.getCell(1, 1).borderTop(new GC.Spread.Sheets.LineBorder("#F0F8FF",GC.Spread.Sheets.LineStyle.double));
                 * sheet.getCell(1, 1).borderLeft(new GC.Spread.Sheets.LineBorder("#F0F8FF",GC.Spread.Sheets.LineStyle.hair));
                 * sheet.getCell(1, 1).borderRight(new GC.Spread.Sheets.LineBorder("#FAEBD7",GC.Spread.Sheets.LineStyle.dashDot));
                 * sheet.getCell(1, 1).borderBottom(new GC.Spread.Sheets.LineBorder("#00FFFF",GC.Spread.Sheets.LineStyle.medium));
                 * sheet.getRange(-1, 5, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderTop(new GC.Spread.Sheets.LineBorder("#F0FFFF",GC.Spread.Sheets.LineStyle.medium));
                 * sheet.getRange(-1, 5, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderLeft(new GC.Spread.Sheets.LineBorder("#F5F5DC",GC.Spread.Sheets.LineStyle.medium));
                 * sheet.getRange(-1, 5, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderRight(new GC.Spread.Sheets.LineBorder("#FF02FF", GC.Spread.Sheets.LineStyle.dashDot));
                 * sheet.getRange(-1, 5, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderBottom (new GC.Spread.Sheets.LineBorder("#FFE4C4",GC.Spread.Sheets.LineStyle.thin));
                 * sheet.getRange(2, 2, 2, 2, GC.Spread.Sheets.SheetArea.viewport).setBorder(new GC.Spread.Sheets.LineBorder("#8A2BE2",GC.Spread.Sheets.LineStyle.thick), { all:true });
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderTop( new GC.Spread.Sheets.LineBorder("#A52A2A",GC.Spread.Sheets.LineStyle.mediumDashed));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderLeft( new GC.Spread.Sheets.LineBorder("#FF02FF",GC.Spread.Sheets.LineStyle.medium));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderRight(new GC.Spread.Sheets.LineBorder("#5F9EA0", GC.Spread.Sheets.LineStyle.dashDot));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderBottom(new GC.Spread.Sheets.LineBorder("#6495ED",GC.Spread.Sheets.LineStyle.dotted));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).diagonalUp(new GC.Spread.Sheets.LineBorder("#FF02FF",GC.Spread.Sheets.LineStyle.dotted));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).diagonalDown(new GC.Spread.Sheets.LineBorder("#6495ED",GC.Spread.Sheets.LineStyle.medium));
                 */
                setBorder(border:  GC.Spread.Sheets.LineBorder,  option:  GC.Spread.Sheets.ISetBorderOptions): void;
                /**
                 * Gets or sets the cell ellipsis property.
                 * @param {boolean} [value] Set to <c>true</c> to have the cell test show ellipsis.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell ellipsis property.
                 * @example
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).showEllipsis(false);
                 * activeSheet.setText(2,0,"This is a test");
                 * @example
                 * activeSheet.getCell(1,1).showEllipsis(true);
                 */
                showEllipsis(value?:  boolean): any;
                /**
                 * Gets or sets whether the cell shrinks the text to fit the cell size.
                 * @param {boolean} [value] Set to <c>true</c> to have the cell shrink text to fit.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell shrinks the text to fit; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).shrinkToFit(false);
                 * activeSheet.setText(2,0,"This is a test");
                 * @example
                 * activeSheet.getCell(1,1).shrinkToFit(true);
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).shrinkToFit(true);
                 */
                shrinkToFit(value?:  boolean): any;
                /**
                 * Gets or sets a value that indicates whether the user can set focus to the cell using the Tab key.
                 * @param {boolean} [value] Set to <c>true</c> to set focus to the cell using the Tab key.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the user can set focus to the cell using the Tab key; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(1,1).tabStop(false);
                 * activeSheet.getRange(1, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).tabStop(false);
                 * activeSheet.getRange(-1, 1, -1, 1, GC.Spread.Sheets.SheetArea.viewport).tabStop(false);
                 */
                tabStop(value?:  boolean): any;
                /**
                 * Gets or sets the tag for the cell.
                 * @param {Object} [value] The tag value.
                 * @returns {Object|GC.Spread.Sheets.CellRange} If no value is set, returns the tag value; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(1,1).tag("cell tag");
                 */
                tag(value?:  any): any;
                /**
                 * Gets or sets the formatted text for the cell.
                 * @param {string} [value] The cell text.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell text; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(1,1).text("cell object");
                 */
                text(value?:  string): any;
                /**
                 * Gets or sets the type of the decoration added to the cell's text.
                 * @param {GC.Spread.Sheets.TextDecorationType} [value] The type of the decoration.
                 * @returns {GC.Spread.Sheets.TextDecorationType|GC.Spread.Sheets.CellRange} If no value is set, returns the type of the decoration; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(1, -1, 1, -1).textDecoration(GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.underline);
                 */
                textDecoration(value?:  GC.Spread.Sheets.TextDecorationType): any;
                /**
                 * Gets or sets the text indent of the cell.
                 * @param {number} [value] The cell text indent.
                 * @returns {number|GC.Spread.Sheets.CellRange} If no value is set, returns the cell text indent; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).textIndent(1);
                 */
                textIndent(value?:  number): any;
                /**
                 * Gets or sets the text rotation angle of the cell.
                 * @param {number}  [value] The cell text rotation angle, form -90 to 90 angle.
                 * @returns {number|GC.Spread.Sheets.CellRange} If no value is set, returns the cell text rotation angle; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).textOrientation(66);
                 */
                textOrientation(value?:  number): any;
                /**
                 * Gets or sets the theme font for the cell.
                 * @param {string} [value] The cell's theme font.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell's theme font; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(-1, 0).themeFont("Body");
                 * activeSheet.getCell(0,0).value("Test");
                 */
                themeFont(value?:  string): any;
                /**
                 * Gets or sets the data validator for the cell.
                 * @param {GC.Spread.Sheets.DataValidation.DefaultDataValidator} value The cell data validator.
                 * @returns {GC.Spread.Sheets.DataValidation.DefaultDataValidator|GC.Spread.Sheets.CellRange} If no value is set, returns the cell data validator; otherwise, returns the cell.
                 * @example
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createDateValidator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between, new Date(2012, 11, 31), new Date(2013, 11, 31));
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Enter a date between 12/31/2012 and 12/31/2013.");
                 * dv.inputTitle("Tip");
                 * activeSheet.getCell(0,0).validator(dv);
                 * @example
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createDateValidator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between, new Date(2012, 11, 31), new Date(2013, 11, 31));
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Enter a date between 12/31/2012 and 12/31/2013.");
                 * dv.inputTitle("Tip");
                 * activeSheet.getCell(1, -1).validator(dv);
                 * @example
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createListValidator("1,2,3");
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Value must be 1,2 or 3");
                 * dv.inputTitle("tip");
                 * activeSheet.setDataValidator(-1,0,dv);
                 */
                validator(value?:  GC.Spread.Sheets.DataValidation.DefaultDataValidator): any;
                /**
                 * Gets or sets the vertical alignment of the contents of the cell.
                 * @param {GC.Spread.Sheets.VerticalAlign} [value] The vertical alignment.
                 * @returns {GC.Spread.Sheets.VerticalAlign|GC.Spread.Sheets.CellRange} If no value is set, returns the vertical alignment of the contents of the cell; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).vAlign(GC.Spread.Sheets.VerticalAlign.top);
                 */
                vAlign(value?:  GC.Spread.Sheets.VerticalAlign): any;
                /**
                 * Gets or sets the unformatted value of the cell.
                 * @param {Object} [value] The cell value.
                 * @returns {Object|GC.Spread.Sheets.CellRange} If no value is set, returns the cell value; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(1,1).value(10);
                 */
                value(value?:  any): any;
                /**
                 * Gets or sets whether the row or column is displayed.
                 * @param {boolean} [value] Set to <c>true</c> to make the row visible.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns the visible of the row or column; otherwise, returns the row or column.
                 * @example
                 * activeSheet.getCell(-1, 0).visible(false);
                 */
                visible(value?:  boolean): any;
                /**
                 * Gets or sets the content of the cell watermark.
                 * @param {string} [value] The content of the watermark.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the content of the watermark; otherwise, returns the cell.
                 * @example
                 * activeSheet.getCell(1,1).watermark("lightgreen");
                 */
                watermark(value?:  string): any;
                /**
                 * Gets or sets the width of the column in pixels.
                 * @param {number} [value] The column width.
                 * @returns {number|GC.Spread.Sheets.CellRange} If no value is set, returns the column width; otherwise, returns the column.
                 * @example
                 * activeSheet.getCell(-1, 0).width(20);
                 */
                width(value?:  number): any;
                /**
                 * Gets or sets whether the cell lets text wrap.
                 * @param {boolean} [value] Set to <c>true</c> to let text wrap within the cell.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell lets text wrap; otherwise, returns the cell.
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).wordWrap(true);
                 */
                wordWrap(value?:  boolean): any;
            }

            export class ColorScheme{
                /**
                 * Creates a ColorScheme instance.
                 * @constructor
                 * @class
                 * @classdesc Represents the theme color.
                 * @param {string} name The owner that contains the named variable.
                 * @param {string} background1 The theme color for background1.
                 * @param {string} background2 The theme color for background2.
                 * @param {string} text1 The theme color for text1.
                 * @param {string} text2 The theme color for text2.
                 * @param {string} accent1 The theme color for accent1.
                 * @param {string} accent2 The theme color for accent2.
                 * @param {string} accent3 The theme color for accent3.
                 * @param {string} accent4 The theme color for accent4.
                 * @param {string} accent5 The theme color for accent5.
                 * @param {string} accent6 The theme color for accent6.
                 * @param {string} link The color of the link.
                 * @param {string} followedLink The color of the followedLink.
                 * @example
                 * //This example creates colors for the theme.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 1 30");
                 */
                constructor(name:  string,  background1:  string,  background2:  string,  text1:  string,  text2:  string,  accent1:  string,  accent2:  string,  accent3:  string,  accent4:  string,  accent5:  string,  accent6:  string,  link:  string,  followedLink:  string);
                /**
                 * Gets or sets the accent1 theme color of the color scheme.
                 * @param {string} value The accent1 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent1 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the accent1 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 1 30");
                 */
                accent1(value?:  string): any;
                /**
                 * Gets or sets the accent2 theme color of the color scheme.
                 * @param {string} value The accent2 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent2 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the accent2 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent2("red");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 2");
                 */
                accent2(value?:  string): any;
                /**
                 * Gets or sets the accent3 theme color of the color scheme.
                 * @param {string} value The accent3 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent3 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the accent3 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent3("yellow");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 3");
                 */
                accent3(value?:  string): any;
                /**
                 * Gets or sets the accent4 theme color of the color scheme.
                 * @param {string} value The accent4 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent4 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the accent4 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent4("blue");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 4");
                 */
                accent4(value?:  string): any;
                /**
                 * Gets or sets the accent5 theme color of the color scheme.
                 * @param {string} value The accent5 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent5 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the accent5 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent5("blue");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 5");
                 */
                accent5(value?:  string): any;
                /**
                 * Gets or sets the accent6 theme color of the color scheme.
                 * @param {string} value The accent6 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent6 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the accent6 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent6("blue");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 6");
                 */
                accent6(value?:  string): any;
                /**
                 * Gets or sets the background1 theme color of the color scheme.
                 * @param {string} value The background1 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the background1 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the background1 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().background1("orange");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("background 1");
                 */
                background1(value?:  string): any;
                /**
                 *  Gets or sets the background2 theme color of the color scheme.
                 * @param {string} value The background2 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the background2 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the background2 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().background2("orange");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("background 2");
                 */
                background2(value?:  string): any;
                /**
                 * Gets or sets the followed hyperlink theme color of the color scheme.
                 * @param {string} value The followed hyperlink theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the followed hyperlink theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the followedHyperline color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().followedHyperlink("orange");
                 * ntheme.colors().hyperlink("red");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("followedHyperlink");
                 * activeSheet.getCell(2, 0).backColor("hyperlink");
                 */
                followedHyperlink(value?:  string): any;
                /**
                 * Gets the color based on the theme color.
                 * @param {string} name The theme color name.
                 * @returns {string} The theme color.
                 * @example
                 * //This example gets the theme color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().followedHyperlink("orange");
                 * ntheme.colors().hyperlink("red");
                 * ntheme.colors().name("test");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("followedHyperlink");
                 * activeSheet.getCell(2, 0).backColor("hyperlink");
                 * var cname = ntheme.getColor("accent 1");
                 * alert(cname);
                 */
                getColor(name:  string): string;
                /**
                 * Gets or sets the hyperlink theme color of the color scheme.
                 * @param {string} value The hyperlink theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the hyperlink theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the hyperline color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().followedHyperlink("orange");
                 * ntheme.colors().hyperlink("red");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("followedHyperlink");
                 * activeSheet.getCell(2, 0).backColor("hyperlink");
                 */
                hyperlink(value?:  string): any;
                /**
                 * Gets or sets the name of the color scheme.
                 * @param {string} value The name.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the name; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the theme name.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().name("green theme");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 */
                name(value?:  string): any;
                /**
                 * Gets or sets the textcolor1 theme color of the color scheme.
                 * @param {string} value The textcolor1 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the textcolor1 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the textColor1 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().textColor1("orange");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("Text 1");
                 */
                textColor1(value?:  string): any;
                /**
                 * Gets or sets the textcolor2 theme color of the color scheme.
                 * @param {string} value The textcolor2 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the textcolor2 theme color; otherwise, returns the color scheme.
                 * @example
                 * //This example sets the textColor2 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().textColor2("orange");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("Text 2");
                 */
                textColor2(value?:  string): any;
            }

            export class Events{
                /**
                 * Defines the events supported in SpreadJS.
                 * @class
                 */
                constructor();
                /**
                 * Occurs when the user has changed the active sheet.
                 * @name GC.Spread.Sheets.Workbook#ActiveSheetChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} oldSheet The old sheet.
                 * @param {GC.Spread.Sheets.Worksheet} newSheet The new sheet.
                 * @example
                 * //This example creates log text for the ActiveSheetChanged event.
                 * // Use IE to see the console log text
                 * spread.bind(GC.Spread.Sheets.Events.ActiveSheetChanged, function (sender, args) {
                 *     console.log("Active sheet has been switched.");
                 * });
                 */
                static ActiveSheetChanged: string;
                /**
                 * Occurs when the user is changing the active sheet.
                 * @name GC.Spread.Sheets.Workbook#ActiveSheetChanging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} oldSheet The old sheet.
                 * @param {GC.Spread.Sheets.Worksheet} newSheet The new sheet.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example stops the active sheet from changing.
                 * spread.bind(GC.Spread.Sheets.Events.ActiveSheetChanging, function (sender, args) {
                 *     //Cancel sheet switching.
                 *     args.cancel = true;
                 * });
                 */
                static ActiveSheetChanging: string;
                /**
                 * Occurs before the print
                 * @name GC.Spread.Sheets.WorkBook.BeforePrint
                 * @event
                 * @param {Object} iframe The print Dom
                 * @param {booble} cancel Whether cancel print
                 * @example
                 * //This example uses the BeforePrint.
                 * window.onload = function(){
                 *      var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 *      spread.bind(GC.Spread.Sheets.Events.BeforePrint, function (e, data) {
                 *          alert(data.iframe + '\n' + 'cancel: ' + data.cancel);
                 *      });
                 * }
                 */
                static BeforePrint: string;
                /**
                 * Occurs when the user clicks a button, check box, or hyperlink in a cell.
                 * @name GC.Spread.Sheets.Workbook#ButtonClicked
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of the cell.
                 * @param {number} col The column index of the cell.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area that contains the cell.
                 * @example
                 * //This example creates a button cell.
                 * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                 * cellType.buttonBackColor("#FFFF00");
                 * cellType.text("this is a button");
                 * activeSheet.setCellType(1,1,cellType);
                 * //bind
                 * spread.bind(GC.Spread.Sheets.Events.ButtonClicked, function (e, args) {
                 *     var sheet = args.sheet, row = args.row, col = args.col;
                 *     var cellType = activeSheet.getCellType(row, col);
                 *     if (cellType instanceof GC.Spread.Sheets.CellTypes.Button) {
                 *         alert("Button Clicked");
                 *     }
                 * });
                 */
                static ButtonClicked: string;
                /**
                 * Occurs when a change is made to a cell in this sheet that may require the cell to be repainted.
                 * @name GC.Spread.Sheets.Worksheet#CellChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of the cell.
                 * @param {number} col The column index of the cell.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheetArea of the cell.
                 * @param {string} propertyName The name of the cell's property that has changed.
                 * @param {boolean} isUndo Whether this event is from a undo operation.
                 * @example
                 * //This example uses the CellChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (e, info) {
                 *     if(info.sheetArea === GC.Spread.Sheets.SheetArea.viewport){
                 *         alert("Cell index (" + info.row + "," + info.col + ")");
                 *     }
                 * });
                 */
                static CellChanged: string;
                /**
                 * Occurs when the user presses down the left mouse button in a cell.
                 * @name GC.Spread.Sheets.Worksheet#CellClick
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area the clicked cell is in.
                 * @param {number} row The row index of the clicked cell.
                 * @param {number} col The column index of the clicked cell.
                 * @example
                 * //This example uses the CellClick event.
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellClick, function (sender, args) {
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
                 *         console.log("The column header was clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
                 *         console.log("The row header was clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
                 *         console.log("The corner header was clicked.");
                 *     }
                 *     console.log("Clicked column index: " + args.col);
                 *     console.log("Clicked row index: " + args.row);
                 * });
                 * //bind
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellDoubleClick, function (sender, args) {
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
                 *         console.log("The column header was double clicked.");
                 *    }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
                 *         console.log("The row header was double clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
                 *         console.log("The corner header was double clicked.");
                 *     }
                 *     console.log("Double clicked column index: " + args.col);
                 *     console.log("Double clicked row index: " + args.row);
                 * })
                 */
                static CellClick: string;
                /**
                 * Occurs when the user presses down the left mouse button twice (double-clicks) in a cell.
                 * @name GC.Spread.Sheets.Worksheet#CellDoubleClick
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area the clicked cell is in.
                 * @param {number} row The row index of the clicked cell.
                 * @param {number} col The column index of the clicked cell.
                 * @example
                 * //This example uses the CellDoubleClick event.
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellClick, function (sender, args) {
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
                 *         console.log("The column header was clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
                 *         console.log("The row header was clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
                 *         console.log("The corner header was clicked.");
                 *     }
                 *     console.log("Clicked column index: " + args.col);
                 *     console.log("Clicked row index: " + args.row);
                 * });
                 * //bind
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellDoubleClick, function (sender, args) {
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
                 *         console.log("The column header was double clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
                 *         console.log("The row header was double clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
                 *         console.log("The corner header was double clicked.");
                 *     }
                 *     console.log("Double clicked column index: " + args.col);
                 *     console.log("Double clicked row index: " + args.row);
                 * });
                 */
                static CellDoubleClick: string;
                /**
                 * Occurs when a Clipboard change occurs that affects Spread.Sheets.
                 * @name GC.Spread.Sheets.Worksheet#ClipboardChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {Object} copyData The data from Spread.Sheets that has been set into the clipboard.
                 * @param {string} copyData.text The text string of the clipboard.
                 * @param {string} copyData.html The html string of the clipboard.
                 * @example
                 * //This example uses the ClipboardChanged event.
                 * // Use IE to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardChanged, function (sender, args) {
                 *     console.log("ClipboardChanged.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardChanging, function (sender, args) {
                 *     console.log("ClipboardChanging");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardPasted, function (sender, args) {
                 *     console.log("ClipboardPasted");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardPasting, function (sender, args) {
                 *     console.log("ClipboardPasting");
                 * });
                 */
                static ClipboardChanged: string;
                /**
                 * Occurs when the Clipboard is changing due to a Spread.Sheets action.
                 * @name GC.Spread.Sheets.Worksheet#ClipboardChanging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {Object} copyData The data from Spread.Sheets that is set into the clipboard.
                 * @param {string} copyData.text The text string of the clipboard.
                 * @param {string} copyData.html The html string of the clipboard.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the ClipboardChanging event.
                 * // Use IE to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardChanged, function (sender, args) {
                 *     console.log("ClipboardChanged.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardChanging, function (sender, args) {
                 *     console.log("ClipboardChanging");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardPasted, function (sender, args) {
                 *     console.log("ClipboardPasted");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardPasting, function (sender, args) {
                 *     console.log("ClipboardPasting");
                 * });
                 */
                static ClipboardChanging: string;
                /**
                 * Occurs when the user pastes from the Clipboard.
                 * @name GC.Spread.Sheets.Worksheet#ClipboardPasted
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Range} cellRange The range that was pasted.
                 * @param {GC.Spread.Sheets.ClipboardPasteOptions} pasteOption The paste option that indicates what data is pasted from the clipboard: values, formatting, or formulas.
                 * @param {Object} pasteData The data from the clipboard that will be pasted into Spread.Sheets.
                 * @param {string} pasteData.text The text string of the clipboard.
                 * @param {string} pasteData.html The html string of the clipboard.
                 * @example
                 * //This example uses the ClipboardPasted event.
                 * // Use IE to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardChanged, function (sender, args) {
                 *     console.log("ClipboardChanged.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardChanging, function (sender, args) {
                 *     console.log("ClipboardChanging");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardPasted, function (sender, args) {
                 *     console.log("ClipboardPasted");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardPasting, function (sender, args) {
                 *     console.log("ClipboardPasting");
                 * });
                 */
                static ClipboardPasted: string;
                /**
                 * Occurs when the user is pasting from the Clipboard.
                 * @name GC.Spread.Sheets.Worksheet#ClipboardPasting
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Range} cellRange The range to paste.
                 * @param {GC.Spread.Sheets.ClipboardPasteOptions} pasteOption The paste option that indicates what data is pasted from the clipboard: values, formatting, or formulas.
                 * @param {Object} pasteData The data from the clipboard that will be pasted into Spread.Sheets.
                 * @param {string} pasteData.text The text string of the clipboard.
                 * @param {string} pasteData.html The html string of the clipboard.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the ClipboardPasting event.
                 * // Use IE to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardChanged, function (sender, args) {
                 *     console.log("ClipboardChanged.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardChanging, function (sender, args) {
                 *     console.log("ClipboardChanging");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardPasted, function (sender, args) {
                 *     console.log("ClipboardPasted");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.ClipboardPasting, function (sender, args) {
                 *     console.log("ClipboardPasting");
                 * });
                 */
                static ClipboardPasting: string;
                /**
                 * Occurs when a change is made to a column or range of columns in this sheet that may require the column or range of columns to be repainted.
                 * @name GC.Spread.Sheets.Worksheet#ColumnChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheetArea of the column.
                 * @param {string} propertyName The name of the column's property that has changed.
                 * @param {boolean} isUndo Whether this event is from a undo operation.
                 * @example
                 * //This example uses the ColumnChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.ColumnChanged, function (e, info) {
                 *     if(info.sheetArea === GC.Spread.Sheets.SheetArea.viewport){
                 *         alert("Index (" + info.col + ")");
                 *     }
                 * });
                 */
                static ColumnChanged: string;
                /**
                 * Occurs when before a change is made to a column or range of columns in this sheet that may require the column or range of columns to be repainted.
                 * @name GC.Spread.Sheets.Worksheet#ColumnChanging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheetArea of the column.
                 * @param {string} propertyName The name of the column's property that has changed.
                 * @example
                 * //This example uses the ColumnChanging event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.ColumnChanging, function (e, info) {
                 *     if(info.sheetArea === GC.Spread.Sheets.SheetArea.viewport){
                 *         alert("Index (" + info.col + ")");
                 *     }
                 * });
                 */
                static ColumnChanging: string;
                /**
                 * Occurs when the column width has changed.
                 * @name GC.Spread.Sheets.Worksheet#ColumnWidthChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {Array} colList The list of columns whose widths have changed.
                 * @param {boolean} header Whether the columns are row header columns.
                 * @example
                 * //This example uses the ColumnWidthChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.ColumnWidthChanged, function (e, info) {
                 *         alert("Column (" + info.colList + ")");
                 * });
                 */
                static ColumnWidthChanged: string;
                /**
                 * Occurs when the column width is changing.
                 * @name GC.Spread.Sheets.Worksheet#ColumnWidthChanging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {Array} colList The list of columns whose widths are changing.
                 * @param {boolean} header Whether the columns are row header columns.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the ColumnWidthChanging event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.ColumnWidthChanging, function (e, info) {
                 *         alert("Column (" + info.colList + ")");
                 * });
                 */
                static ColumnWidthChanging: string;
                /**
                 * Occurs when any comment has changed.
                 * @name GC.Spread.Sheets.Worksheet#CommentChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Comments.Comment} comment The comment that triggered the event.
                 * @param {string} propertyName The name of the comment's property that has changed.
                 * @example
                 * //This example uses the CommentChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CommentChanged, function (e, info) {
                 *     alert("changed");
                 * });
                 */
                static CommentChanged: string;
                /**
                 * Occurs when the user has removed the comment.
                 * @name GC.Spread.Sheets.Worksheet#CommentRemoved
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Comments.Comment} comment The comment has been removed.
                 * @example
                 * //This example uses the CommentRemoved event.
                 * var comment = new GC.Spread.Sheets.Comments.Comment();
                 * comment.text("new comment!");
                 * comment.backColor("orange");
                 * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                 * activeSheet.getCell(5,5).comment(comment);
                 * activeSheet.bind(GC.Spread.Sheets.Events.CommentRemoved, function (e, info) {
                 *        console.log("sheet name: " + info.sheetName);
                 * });
                 */
                static CommentRemoved: string;
                /**
                 * Occurs when the user is removing any comment.
                 * @name GC.Spread.Sheets.Worksheet#CommentRemoving
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Comments.Comment} comment The comment is being removed.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example prevents the comment from being removed.
                 * var comment = new GC.Spread.Sheets.Comments.Comment();
                 * comment.text("new comment!");
                 * comment.backColor("orange");
                 * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                 * activeSheet.getCell(5,5).comment(comment);
                 * activeSheet.bind(GC.Spread.Sheets.Events.CommentRemoving, function (e, info) {
                 *       info.cancel = true;
                 * });
                 */
                static CommentRemoving: string;
                /**
                 * Occurs when the user is dragging and dropping a range of cells.
                 * @name GC.Spread.Sheets.Worksheet#DragDropBlock
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} fromRow The row index of the top left cell of the source range (range being dragged).
                 * @param {number} fromCol The column index of the top left cell of the source range (range being dragged).
                 * @param {number} toRow The row index of the top left cell of the destination range (where selection is dropped).
                 * @param {number} toCol The column index of the bottom right cell of the destination range (where selection is dropped).
                 * @param {number} rowCount The row count of the cell range being dragged.
                 * @param {number} colCount The column count of the cell range being dragged.
                 * @param {boolean} copy Whether the source range is copied.
                 * @param {boolean} insert Whether the source range is inserted.
                 * @param {GC.Spread.Sheets.CopyToOptions} copyOption The CopyOption value for the drag and drop operation.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example creates log text for the DragDropBlock event.
                 * // Use IE to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragDropBlock, function (e, args) {
                 *         console.log("From Column:" + args.fromCol);
                 *         console.log("From Row:" + args.fromRow);
                 *         console.log("To Column:" + args.toCol);
                 *         console.log("To Row:" + args.toRow);
                 *     });
                 */
                static DragDropBlock: string;
                /**
                 * Occurs when the user completes dragging and dropping a range of cells.
                 * @name GC.Spread.Sheets.Worksheet#DragDropBlockCompleted
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} fromRow The row index of the top left cell of the source range (range being dragged).
                 * @param {number} fromCol The column index of the top left cell of the source range (range being dragged).
                 * @param {number} toRow The row index of the top left cell of the destination range (where selection is dropped).
                 * @param {number} toCol The column index of the bottom right cell of the destination range (where selection is dropped).
                 * @param {number} rowCount The row count of the cell range being dragged.
                 * @param {number} colCount The column count of the cell range being dragged.
                 * @param {boolean} copy Whether the source range is copied.
                 * @param {boolean} insert Whether the source range is inserted.
                 * @param {GC.Spread.Sheets.CopyToOptions} copyOption The CopyOption value for the drag and drop operation.
                 * @example
                 * //This example uses the DragDropBlockCompleted event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragDropBlockCompleted, function (e, args) {
                 *         alert("From Column (" + args.fromCol + ")");
                 * });
                 */
                static DragDropBlockCompleted: string;
                /**
                 * Occurs when the user is dragging to fill a range of cells.
                 * @name GC.Spread.Sheets.Worksheet#DragFillBlock
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Range} fillRange The range used for the fill operation.
                 * @param {GC.Spread.Sheets.Fill.AutoFillType} autoFillType AutoFillType value used for the fill operation.
                 * @param {GC.Spread.Sheets.Fill.FillDirection} fillDirection The direction of the fill.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the DragFillBlock event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragFillBlock, function (e, info) {
                 *         alert("Direction (" + info.fillDirection + ")");
                 * });
                 */
                static DragFillBlock: string;
                /**
                 * Occurs when the user completes dragging to fill a range of cells.
                 * @name GC.Spread.Sheets.Worksheet#DragFillBlockCompleted
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Range} fillRange The range used for the fill operation.
                 * @param {GC.Spread.Sheets.Fill.AutoFillType} autoFillType AutoFillType value used for the fill operation.
                 * @param {GC.Spread.Sheets.Fill.FillDirection} fillDirection The direction of the fill.
                 * @example
                 * //This example uses the DragFillBlockCompleted event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragFillBlockCompleted, function (e, info) {
                 *         alert("Type (" + info.autoFillType + ")");
                 * });
                 */
                static DragFillBlockCompleted: string;
                /**
                 * Occurs after user drag merge cells.
                 * @name GC.Spread.Sheets.Worksheet#DragMerged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Range} mergeRange The range that will be merged.
                 * @example
                 * //This example returns the row index.
                 * // Press Ctrl key to merge
                 * $(document).keydown(function (e) {
                 *      if (e.keyCode === 17) {
                 *         spread.options.allowUserDragMerge = true;
                 *      }
                 * });
                 * $(document).keyup(function (e) {
                 *      if (e.keyCode === 17) {
                 *         spread.options.allowUserDragMerge = false;
                 *      }
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragMerging, function (e, data) {
                 *      var mergeRange = data.mergeRange;
                 *      alert(mergeRange.row);
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragMerged, function (e, data) {
                 *      var mergeRange = data.mergeRange;
                 *      alert(mergeRange.row);
                 * });
                 */
                static DragMerged: string;
                /**
                 * Occurs before user drag merge cells.
                 * @name GC.Spread.Sheets.Worksheet#DragMerging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Range} mergeRange The range that will be merged.
                 * @param {bool} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example returns the row index.
                 * // Press Ctrl key to merge
                 * $(document).keydown(function (e) {
                 *      if (e.keyCode === 17) {
                 *          spread.options.allowUserDragMerge = true;
                 *      }
                 * });
                 * $(document).keyup(function (e) {
                 *      if (e.keyCode === 17) {
                 *          spread.options.allowUserDragMerge = false;
                 *      }
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragMerging, function (e, data) {
                 *      var mergeRange = data.mergeRange;
                 *      alert(mergeRange.row);
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragMerged, function (e, data) {
                 *      var mergeRange = data.mergeRange;
                 *      alert(mergeRange.row);
                 * });
                 */
                static DragMerging: string;
                /**
                 * Occurs when a cell is in edit mode and the text is changed.
                 * @name GC.Spread.Sheets.Worksheet#EditChange
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of cell.
                 * @param {number} col The column index of cell.
                 * @param {Object} editingText The value from the current editor.
                 * @example
                 * //This example creates log text for the EditChange event.
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditChange, function (sender, args) {
                 *     console.log("Cell (" + args.row + ", " + args.col + ") data has been changed.")
                 * });
                 */
                static EditChange: string;
                /**
                 * Occurs when a cell leaves edit mode.
                 * @name GC.Spread.Sheets.Worksheet#EditEnded
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of cell.
                 * @param {number} col The column index of cell.
                 * @param {Object} editingText The value from the current editor.
                 * @example
                 * //This example creates log text for the EditStarting and EditEnded events.
                 *  // Use IE to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditStarting, function (sender, args) {
                 *     console.log("Start cell editing.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditEnded, function (sender, args) {
                 *     console.log("Finish cell editing.");
                 * });
                 */
                static EditEnded: string;
                /**
                 * Occurs when a cell is leaving edit mode.
                 * @name GC.Spread.Sheets.Worksheet#EditEnding
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of cell.
                 * @param {number} col The column index of cell.
                 * @param {Object} editor The current editor.
                 * @param {Object} editingText The value from the current editor.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the EditEnding event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditStarting, function (sender, args) {
                 *     console.log("Start cell editing.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditEnding, function (sender, args) {
                 *     console.log("EditEnding event.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditEnded, function (sender, args) {
                 *     console.log("EditEnded event.");
                 * });
                 */
                static EditEnding: string;
                /**
                 * Occurs when the editor's status has changed.
                 * @name GC.Spread.Sheets.Worksheet#EditorStatusChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.EditorStatus} oldStatus The old status of the editor.
                 * @param {GC.Spread.Sheets.EditorStatus} newStatus The new status of the editor.
                 * @example
                 * //This example uses the EditorStatusChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditorStatusChanged, function (e, info) {
                 *         alert("Column (" + info.newStatus + ")");
                 * });
                 */
                static EditorStatusChanged: string;
                /**
                 * Occurs when a cell is entering edit mode.
                 * @name GC.Spread.Sheets.Worksheet#EditStarting
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of cell.
                 * @param {number} col The column index of cell.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example creates log text for the EditStarting and EditEnded events.
                 * // Use IE to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditStarting, function (sender, args) {
                 *     console.log("Start cell editing.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditEnded, function (sender, args) {
                 *     console.log("Finish cell editing.");
                 * });
                 */
                static EditStarting: string;
                /**
                 * Occurs when the focus enters a cell.
                 * @name GC.Spread.Sheets.Worksheet#EnterCell
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of the cell being entered.
                 * @param {number} col The column index of the cell being entered.
                 * @example
                 * //This example uses the EnterCell event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.EnterCell, function (e, info) {
                 *         alert("Cell (" + info.row + ", " + info.col +")");
                 * });
                 */
                static EnterCell: string;
                /**
                 * Occurs when any floating object has changed.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectsChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.FloatingObjects.FloatingObject} floatingObject The floating object that triggered the event.
                 * @param {string} propertyName The name of the floating object's property that has changed.
                 * @example
                 * //This example uses the FloatingObjectChanged event.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1", 10, 10, 60, 64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 * activeSheet.bind(GC.Spread.Sheets.Events.FloatingObjectChanged, function (e, info) {
                 *        alert("changed");
                 * });
                 */
                static FloatingObjectChanged: string;
                /**
                 * Occurs when the custom floating object content is loaded.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectLoaded
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.FloatingObjects.FloatingObject} floatingObject The custom floating object that triggered the event.
                 * @param {HTMLElement} element The HTMLElement of the custom floating object.
                 */
                static FloatingObjectLoaded: string;
                /**
                 * Occurs when the user has removed the floating object.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectRemoved
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.FloatingObjects.FloatingObject} floatingObject The floating object has been removed.
                 * @example
                 * //This example uses the FloatingObjectRemoved event.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1", 10, 10, 60, 64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 * activeSheet.bind(GC.Spread.Sheets.Events.FloatingObjectRemoved, function (e, info) {
                 *         alert(info.sheetName);
                 * });
                 */
                static FloatingObjectRemoved: string;
                /**
                 * Occurs when the user is removing any floating object.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectRemoving
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.FloatingObjects.FloatingObject} floatingObject The floating object is being removed.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the FloatingObjectRemoving event.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1", 10, 10, 60, 64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 * activeSheet.bind(GC.Spread.Sheets.Events.FloatingObjectRemoving, function (e, info) {
                 *         info.cancel = true;
                 * });
                 */
                static FloatingObjectRemoving: string;
                /**
                 * Occurs when the selections of the floating object have changed.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectsSelectionChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.FloatingObjects.FloatingObject} floatingObject The floating object that triggered the event.
                 * @example
                 * //This example uses the FloatingObjectSelectionChanged event.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1", 10, 10, 60, 64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.FloatingObjectSelectionChanged, function (e, info) {
                 *        console.log("sheet name: " + info.sheetName);
                 * });
                 */
                static FloatingObjectSelectionChanged: string;
                /**
                 * Occurs when an invalid operation is performed.
                 * @name GC.Spread.Sheets.Worksheet#InvalidOperation
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.InvalidOperationType} invalidType Indicates which operation was invalid.
                 * @param {string} message The description of the invalid operation.
                 * @example
                 * //This example uses the InvalidOperation event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.InvalidOperation, function (e, info) {
                 *         alert("Message (" + info.message + ")");
                 * });
                 */
                static InvalidOperation: string;
                /**
                 * Occurs when the focus leaves a cell.
                 * @name GC.Spread.Sheets.Worksheet#LeaveCell
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of the cell being left.
                 * @param {number} col The column index of the cell being left.
                 * @param {boolean} cancel Whether the operation should be canceled.
                 * @example
                 * //This example creates log text for the LeaveCell event.
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.LeaveCell, function (sender, args) {
                 *     console.log("The column index before moving: " + args.col);
                 *     console.log("The row index before moving: " + args.row);
                 * });
                 */
                static LeaveCell: string;
                /**
                 * Occurs when the left column changes.
                 * @name GC.Spread.Sheets.Worksheet#LeftColumnChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} oldLeftCol The old left column index.
                 * @param {number} newLeftCol The new left column index.
                 * @example
                 * //This example synchronizes vertical and horizontal scrolling for sheet 1 and sheet 2.
                 * var sheet1 = spread.getSheet(0),
                 * sheet2 = spread.getSheet(1);
                 * sheet1.bind(GC.Spread.Sheets.Events.TopRowChanged, function (sender, args) {
                 *     //Set the displayed top row of sheet1 to sheet2 (vertical scroll synchronization).
                 *     sheet2.showRow(args.newTopRow, GC.Spread.Sheets.VerticalPosition.top);
                 * });
                 * sheet1.bind(GC.Spread.Sheets.Events.LeftColumnChanged, function (sender, args) {
                 *     //Set the displayed left column of sheet1 to sheet2 (Horizontal scroll synchronization).
                 *     sheet2.showColumn(args.newLeftCol, GC.Spread.Sheets.HorizontalPosition.left);
                 * });
                 */
                static LeftColumnChanged: string;
                /**
                 * Occurs when the outline column check status has changed.
                 * @name GC.Spread.Sheets.Worksheet#OutlineColumnCheckStatusChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The outline column's change row index.
                 * @param {number} col The outline column's change col index.
                 * @param {boolean} status The outline column's change status
                 * @example
                 * //Removing the sparkline causes a change.
                 * activeSheet.bind(GC.Spread.Sheets.Events.OutlineColumnCheckStatusChanged, function (e, info) {
                 *      console.log("status: " + info.status);
                 * });
                 */
                static OutlineColumnCheckStatusChanged: string;
                /**
                 * Occurs when any picture has changed.
                 * @name GC.Spread.Sheets.Worksheet#PictureChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.FloatingObjects.Picture} picture The picture that triggered the event.
                 * @param {string} propertyName The name of the picture's property that has changed.
                 * @example
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.pictures.add("f2","Event.png",2,2,6,6);
                 * activeSheet.pictures.add("f1","tsoutline.png",3,0,6,6);
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.PictureChanged, function (e, info) {
                 * console.log("Sheet: " + info.sheetName);
                 * });
                 * @example
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.pictures.add("f2","Event.png",2,2,6,6);
                 * activeSheet.pictures.add("f1","tsoutline.png",3,0,6,6);
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.PictureChanged, function (e, info) {
                 * console.log("Property: " + info.propertyName);
                 * });
                 */
                static PictureChanged: string;
                /**
                 * Occurs when the selections of the picture have changed.
                 * @name GC.Spread.Sheets.Worksheet#PictureSelectionChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.FloatingObjects.Picture} picture The picture that triggered the event.
                 * @example
                 * //This example uses the PictureSelectionChanged event.
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.pictures.add("f2","Event.png",2,2,6,6);
                 * activeSheet.pictures.add("f1","tsoutline.png",3,0,6,6);
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.PictureSelectionChanged, function (e, info) {
                 * console.log("Sheet: " + info.sheetName);
                 * });
                 */
                static PictureSelectionChanged: string;
                /**
                 * Occurs when the cell range has changed.
                 * @name GC.Spread.Sheets.Worksheet#RangeChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The range's row index.
                 * @param {number} col The range's column index.
                 * @param {number} rowCount The range's row count.
                 * @param {number} colCount The range's column count.
                 * @param {Array.<string>} tableNames A collection of table names.
                 * @param {Array.<Object>} changedCells The positions of the cells whose data has changed, each position has row and col.
                 * @param {GC.Spread.Sheets.RangeChangedAction} action The type of action that raises the RangeChanged event.
                 * @example
                 * //This example returns the sheet name and action when changing the cell range in Microsoft Internet Explorer.
                 *  activeSheet.bind(GC.Spread.Sheets.Events.RangeChanged, function (sender, args) {
                 *   console.log(args.sheetName, args.action);
                 * });
                 */
                static RangeChanged: string;
                /**
                 * Occurs when a range column has just been clear filtered.
                 * @name GC.Spread.Sheets.Worksheet#RangeFilterCleared
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} col The index of the sheet column has just been clear filtered.
                 * @example
                 * //This example uses the RangeFilterCleared event.
                 * activeSheet.getCell(0,0).text("Name");
                 * activeSheet.getCell(0,1).text("Value");
                 * activeSheet.getCell(0,2).text("T/F");
                 * activeSheet.getCell(1,0).text("AW");
                 * activeSheet.getCell(1,1).text("5");
                 * activeSheet.getCell(1,2).text("T");
                 * var cellRange =new GC.Spread.Sheets.Range(0, 0, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellRange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeFilterCleared, function (e, info) {
                 *         alert("Col (" + info.col + ")");
                 * });
                 */
                static RangeFilterCleared: string;
                /**
                 * Occurs when a range column is about to be automatically clear filter.
                 * @name GC.Spread.Sheets.Worksheet#RangeFilterClearing
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} col The index of the sheet column to be automatically clear filter.
                 * @example
                 * //This example uses the RangeFilterClearing event.
                 * activeSheet.getCell(0,0).text("Name");
                 * activeSheet.getCell(0,1).text("Value");
                 * activeSheet.getCell(0,2).text("T/F");
                 * activeSheet.getCell(1,0).text("AW");
                 * activeSheet.getCell(1,1).text("5");
                 * activeSheet.getCell(1,2).text("T");
                 * var cellRange =new GC.Spread.Sheets.Range(0, 0, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellRange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeFilterClearing, function (e, info) {
                 *         alert("Col (" + info.col + ")");
                 * });
                 */
                static RangeFilterClearing: string;
                /**
                 * Occurs when a column has just been automatically filtered.
                 * @name GC.Spread.Sheets.Worksheet#RangeFiltered
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} col The index of the column that was automatically filtered.
                 * @param {Array} filterValues The values by which the column was filtered.
                 * @param {Array} conditionInfo The condition rule info that which the column was filtered.
                 * @example
                 * //This example uses the RangeFiltered event.
                 * var cellrange =new GC.Spread.Sheets.Range(0, 2, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellrange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeFiltered, function (e, info) {
                 *         alert("Col (" + info.col + ")");
                 * });
                 */
                static RangeFiltered: string;
                /**
                 * Occurs when a column is about to be automatically filtered.
                 * @name GC.Spread.Sheets.Worksheet#RangeFiltering
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} col The index of the column to be automatically filtered.
                 * @param {Array} filterValues The values by which to filter the column.
                 * @param {Object} conditionInfo The condition rule info by which to filter the column.
                 * @example
                 * //This example uses the RangeFiltering event.
                 * var cellrange =new GC.Spread.Sheets.Range(0, 2, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellrange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeFiltering, function (e, info) {
                 *        alert("Col (" + info.col + ")");
                 * });
                 */
                static RangeFiltering: string;
                /**
                 * Occurs when the user has changed the outline state (range group) for rows or columns.
                 * @name GC.Spread.Sheets.Worksheet#RangeGroupStateChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {boolean} isRowGroup Whether the outline (range group) is a group of rows.
                 * @param {number} index The index of the RangeGroupInfo object whose state has changed.
                 * @param {number} level The level of the RangeGroupInfo object whose state has changed.
                 * @example
                 * //This example uses the RangeGroupStateChanged event.
                 * activeSheet.suspendPaint();
                 * activeSheet.setRowCount(34);
                 * activeSheet.setValue(0,0,"Western");
                 * activeSheet.setValue(0,1,"Western");
                 * activeSheet.setValue(0,2,"Western");
                 * activeSheet.setValue(1,0,"A");
                 * activeSheet.setValue(1,1,"B");
                 * activeSheet.setValue(1,2,"C");
                 * activeSheet.setValue(2,0,"A");
                 * activeSheet.setValue(2,1,"B");
                 * activeSheet.setValue(2,2,"C");
                 * activeSheet.rowOutlines.group(0,2);
                 * activeSheet.columnOutlines.group(0,1);
                 * activeSheet.resumePaint();
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeGroupStateChanged, function (e, info) {
                 *         alert("Level (" + info.level + ")");
                 * });
                 */
                static RangeGroupStateChanged: string;
                /**
                 * Occurs before the user changes the outline state (range group) for rows or columns.
                 * @name GC.Spread.Sheets.Worksheet#RangeGroupStateChanging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {boolean} isRowGroup Whether the outline (range group) is a group of rows.
                 * @param {number} index The index of the RangeGroupInfo object whose state is changing.
                 * @param {number} level The level of the RangeGroupInfo object whose state is changing.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the RangeGroupStateChanging event.
                 * activeSheet.suspendPaint();
                 * activeSheet.setRowCount(34);
                 * activeSheet.setValue(0,0,"Western");
                 * activeSheet.setValue(0,1,"Western");
                 * activeSheet.setValue(0,2,"Western");
                 * activeSheet.setValue(1,0,"A");
                 * activeSheet.setValue(1,1,"B");
                 * activeSheet.setValue(1,2,"C");
                 * activeSheet.setValue(2,0,"A");
                 * activeSheet.setValue(2,1,"B");
                 * activeSheet.setValue(2,2,"C");
                 * activeSheet.rowOutlines.group(0,2);
                 * activeSheet.columnOutlines.group(0,1);
                 * activeSheet.resumePaint();
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeGroupStateChanging, function (e, info) {
                 *         alert("Level (" + info.level + ")");
                 * });
                 */
                static RangeGroupStateChanging: string;
                /**
                 * Occurs when a column has just been automatically sorted.
                 * @name GC.Spread.Sheets.Worksheet#RangeSorted
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} col The index of the column that was automatically sorted.
                 * @param {boolean} ascending Whether the automatic sort is ascending.
                 * @example
                 * //This example uses the RangeSorted event.
                 * activeSheet.setValue(0, 0, 10);
                 * activeSheet.setValue(1, 0, 100);
                 * activeSheet.setValue(2, 0, 50);
                 * activeSheet.setValue(3, 0, 40);
                 * activeSheet.setValue(4, 0, 80);
                 * activeSheet.setValue(5, 0, 1);
                 * activeSheet.setValue(6, 0, 65);
                 * activeSheet.setValue(7, 0, 20);
                 * activeSheet.setValue(8, 0, 30);
                 * activeSheet.setValue(9, 0, 35);
                 * var cellrange =new GC.Spread.Sheets.Range(0, 0, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellrange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeSorted, function (e, info) {
                 *         alert("Col (" + info.col + ", " + info.ascending +")");
                 * });
                 */
                static RangeSorted: string;
                /**
                 * Occurs when a column is about to be automatically sorted.
                 * @name GC.Spread.Sheets.Worksheet#RangeSorting
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} col The index of the column to be automatically sorted.
                 * @param {boolean} ascending Whether the automatic sort is ascending.
                 * @param {boolean} cancel Whether the operation should be canceled.
                 * @param {GC.Spread.Sheets.Range} range The range of automatic sort.
                 * @param {Function} compareFunction The customize function to use when sorting, used when value sort. function (value1, value2) {return 0;}<br />
                 *
                 * @example
                 * //This example uses the RangeSorting event.
                 * activeSheet.setValue(0, 0, 10);
                 * activeSheet.setValue(1, 0, 100);
                 * activeSheet.setValue(2, 0, 50);
                 * activeSheet.setValue(3, 0, 40);
                 * activeSheet.setValue(4, 0, 80);
                 * activeSheet.setValue(5, 0, 1);
                 * activeSheet.setValue(6, 0, 65);
                 * activeSheet.setValue(7, 0, 20);
                 * activeSheet.setValue(8, 0, 30);
                 * activeSheet.setValue(9, 0, 35);
                 * var cellrange =new GC.Spread.Sheets.Range(0, 0, 10, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellrange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeSorting, function (e, info) {
                 *         alert("Col (" + info.col + ", " + info.ascending +")");
                 *         info.compareFunction = (obj1, obj2)=>{return obj1.toString().localeCompare(obj2.toString())};
                 * });
                 */
                static RangeSorting: string;
                /**
                 * Occurs when a change is made to a row or range of rows in this sheet that may require the row or range of rows to be repainted.
                 * @name GC.Spread.Sheets.Worksheet#RowChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheetArea of the row.
                 * @param {string} propertyName The name of the row's property that has changed.
                 * @param {boolean} isUndo Whether this event is from a undo operation.
                 * @example
                 * //This example uses the RowChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.RowChanged, function (e, info) {
                 *         alert("Row (" + info.row + ")");
                 * });
                 */
                static RowChanged: string;
                /**
                 * Occurs when before a change is made to a row or range of rows in this sheet that may require the row or range of rows to be repainted.
                 * @name GC.Spread.Sheets.Worksheet#RowChanging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheetArea of the row.
                 * @param {string} propertyName The name of the row's property that has changed.
                 * @example
                 * //This example uses the RowChanging event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.RowChanging, function (e, info) {
                 *         alert("Row (" + info.row + ")");
                 * });
                 */
                static RowChanging: string;
                /**
                 * Occurs when the row height has changed.
                 * @name GC.Spread.Sheets.Worksheet#RowHeightChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {Array} rowList The list of rows whose heights have changed.
                 * @param {boolean} header Whether the columns are column header columns.
                 * @example
                 * //This example uses the RowHeightChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.RowHeightChanged, function (e, info) {
                 *         alert("Row List (" + info.rowList + ")");
                 * });
                 */
                static RowHeightChanged: string;
                /**
                 * Occurs when the row height is changing.
                 * @name GC.Spread.Sheets.Worksheet#RowHeightChanging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {Array} rowList The list of rows whose heights are changing.
                 * @param {boolean} header Whether the columns are column header columns.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the RowHeightChanging event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.RowHeightChanging, function (e, info) {
                 *         alert("Row List (" + info.rowList + ")");
                 * });
                 */
                static RowHeightChanging: string;
                /**
                 * Occurs when the selection of cells on the sheet has changed.
                 * @name GC.Spread.Sheets.Worksheet#SelectionChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {Array.<GC.Spread.Sheets.Range>} oldSelections The old selection ranges.
                 * @param {Array.<GC.Spread.Sheets.Range>} newSelections The new selection ranges.
                 * @example
                 * //This example uses the SelectionChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.SelectionChanged, function (e, info) {
                 *         alert("Name (" + info.sheetName + ")");
                 * });
                 */
                static SelectionChanged: string;
                /**
                 * Occurs when the selection of cells on the sheet is changing.
                 * @name GC.Spread.Sheets.Worksheet#SelectionChanging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {Array.<GC.Spread.Sheets.Range>} oldSelections The old selection ranges.
                 * @param {Array.<GC.Spread.Sheets.Range>} newSelections The new selection ranges.
                 * @example
                 * //This example uses the SelectionChanging event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.SelectionChanging, function (e, info) {
                 * //Use IE to see console
                 *         console.log("Name (" + info.sheetName + ")");
                 * });
                 */
                static SelectionChanging: string;
                /**
                 * Occurs when any shape has changed.
                 * @name GC.Spread.Sheets.Worksheet#ShapeChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Shapes.Shape} shape The shape that triggered the event.
                 * @param {string} propertyName The name of the shape's property that has changed.
                 * @example
                 * //This example uses the ShapeChanged event.
                 * var shape1 = activeSheet.shapes.add("shape1", GC.Spread.Sheets.Shapes.AutoShapeType.rectangle, 20, 20, 200, 200);
                 * activeSheet.bind(GC.Spread.Sheets.Events.ShapeChanged, function (e, info) {
                 *        alert("changed");
                 * });
                 */
                static ShapeChanged: string;
                /**
                 * Occurs when the user has removed the shape.
                 * @name GC.Spread.Sheets.Worksheet#ShapeRemoved
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Shapes.Shape} shape The shape has been removed.
                 * @example
                 * //This example uses the ShapeRemoved event.
                 * var shape = activeSheet.shapes.add("myShape", GC.Spread.Sheets.Shapes.AutoShapeType.diamond, 0, 90, 200, 200);
                 * activeSheet.bind(GC.Spread.Sheets.Events.ShapeRemoved, function (e, info) {
                 *         alert(info.shape.name());
                 * });
                 */
                static ShapeRemoved: string;
                /**
                 * Occurs when the user is removing any shape.
                 * @name GC.Spread.Sheets.Worksheet#ShapeRemoving
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Shapes.Shape} shape The shape is being removed.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the ShapeRemoving event.
                 * var shape = sheet.shapes.add("myShape", GC.Spread.Sheets.Shapes.AutoShapeType.diamond, 0, 90, 200, 200);
                 * activeSheet.bind(GC.Spread.Sheets.Events.ShapeRemoving, function (e, info) {
                 *         info.cancel = true;// the shape will not remove
                 * });
                 */
                static ShapeRemoving: string;
                /**
                 * Occurs when the selections of the shape have changed.
                 * @name GC.Spread.Sheets.Worksheet#ShapeSelectionChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Shapes.Shape} shape The shape that triggered the event.
                 * @example
                 * //This example uses the ShapeSelectionChanged event.
                 * var shape = sheet.shapes.add("myShape", GC.Spread.Sheets.Shapes.AutoShapeType.diamond, 0, 90, 200, 200);
                 * activeSheet.bind(GC.Spread.Sheets.Events.ShapeSelectionChanged, function (e, info) {
                 *        console.log("event info: " + info);
                 * });
                 */
                static ShapeSelectionChanged: string;
                /**
                 * After sheet changed.
                 * @name GC.Spread.Sheets.Worksheet#SheetChanged
                 * @event
                 * @param {string} sheetName The sheet's name.
                 * @param {string} propertyName The specific operation name.
                 * @param {number} sheetIndex Change the sheet position.
                 * @param {boolean} oldValue: Sheet changes the value of the previous attribute(isVisible, isSelected...).
                 * @param {boolean} newValue: Sheet changes the value of the attribute(isVisible, isSelected...).
                 * @example
                 * //This example.
                 * spread.bind(GC.Spread.Sheets.Events.SheetChanged, function (sender, args) {
                 *     var sheet = args.sheet;
                 * });
                 */
                static SheetChanged: string;
                /**
                 * Before sheet changed.
                 * @name GC.Spread.Sheets.Worksheet#SheetChanging
                 * @event
                 * @param {string} sheetName The sheet's name.
                 * @param {string} propertyName The specific operation name .
                 * @param {number} sheetIndex Change the sheet position.
                 * @param {boolean} oldValue: Sheet changes the value of the previous attribute(isVisible, isSelected...).
                 * @param {boolean} newValue: Sheet changes the value of the attribute(isVisible, isSelected...).
                 * @param {boolean} cancel Cancel the current operation.
                 * @example
                 * //This example.
                 * spread.bind(GC.Spread.Sheets.Events.SheetChanging, function (sender, args) {
                 *     var sheetIndex = args.sheetIndex;
                 *     args.cancel = true;
                 * });
                 */
                static SheetChanging: string;
                /**
                 * Occurs after the user drags and moves the sheet.
                 * @name GC.Spread.Sheets.Worksheet#SheetMoved
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} oldIndex The previous sheet index.
                 * @param {number} newIndex The new sheet index.
                 * @example
                 * //This example uses the SheetMoved event.
                 * window.onload = function(){
                 *      var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 *      var activeSheet = spread.getActiveSheet();
                 *      spread.bind(GC.Spread.Sheets.Events.SheetMoving, function (e, data) {
                 *          alert(data.sheetName + '\n' +  'oldIndex: ' + data.oldIndex + '\n' +  'newIndex: ' + data.newIndex + '\n' +  'cancel: ' + data.cancel);
                 *      });
                 *      spread.bind(GC.Spread.Sheets.Events.SheetMoved, function (e, data) {
                 *          alert(data.sheetName + '\n' +  'oldIndex: ' + data.oldIndex + '\n' +  'newIndex: ' + data.newIndex);
                 *      });
                 * }
                 */
                static SheetMoved: string;
                /**
                 * Occurs before the user drags and moves the sheet.
                 * @name GC.Spread.Sheets.Worksheet#SheetMoving
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} oldIndex The old sheet index.
                 * @param {number} newIndex A value that indicates the index will be moved to.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the SheetMoving event.
                 * window.onload = function(){
                 *      var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 *      var activeSheet = spread.getActiveSheet();
                 *      spread.bind(GC.Spread.Sheets.Events.SheetMoving, function (e, data) {
                 *          alert(data.sheetName + '\n' +  'oldIndex: ' + data.oldIndex + '\n' +  'newIndex: ' + data.newIndex + '\n' +  'cancel: ' + data.cancel);
                 *      });
                 *      spread.bind(GC.Spread.Sheets.Events.SheetMoved, function (e, data) {
                 *          alert(data.sheetName + '\n' +  'oldIndex: ' + data.oldIndex + '\n' +  'newIndex: ' + data.newIndex);
                 *      });
                 * }
                 */
                static SheetMoving: string;
                /**
                 * Occurs when the user has changed the sheet name.
                 * @name GC.Spread.Sheets.Worksheet#SheetNameChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} oldValue The sheet's old name.
                 * @param {string} newValue The sheet's new name.
                 * @example
                 * //This example uses the SheetNameChanged event.
                 * // Use IE to see the console log text
                 * spread.bind(GC.Spread.Sheets.Events.SheetNameChanging, function (sender, args) {
                 *    console.log(args.oldValue);
                 * });
                 * spread.bind(GC.Spread.Sheets.Events.SheetNameChanged, function (sender, args) {
                 *    console.log(args.newValue);
                 * });
                 */
                static SheetNameChanged: string;
                /**
                 * Occurs when the user is changing the sheet name.
                 * @name GC.Spread.Sheets.Worksheet#SheetNameChanging
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} oldValue The sheet's old name.
                 * @param {string} newValue The sheet's new name.
                 * @param {boolean} cancel A value that indicates whether the operation should be canceled.
                 * @example
                 * //This example uses the SheetNameChanging event.
                 * // Use IE to see the console log text
                 * spread.bind(GC.Spread.Sheets.Events.SheetNameChanging, function (sender, args) {
                 *    console.log(args.oldValue);
                 * });
                 * spread.bind(GC.Spread.Sheets.Events.SheetNameChanged, function (sender, args) {
                 *    console.log(args.newValue);
                 * });
                 */
                static SheetNameChanging: string;
                /**
                 * Occurs when the user clicks the sheet tab.
                 * @name GC.Spread.Sheets.Worksheet#SheetTabClick
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} sheetTabIndex The index of the sheet tab that the user clicked.
                 * @example
                 * //This example uses the SheetTabClick event.
                 * spread.bind(GC.Spread.Sheets.Events.SheetTabClick, function (e, info) {
                 *         alert("Index (" + info.sheetTabIndex + ")");
                 * });
                 */
                static SheetTabClick: string;
                /**
                 * Occurs when the user double-clicks the sheet tab.
                 * @name GC.Spread.Sheets.Worksheet#SheetTabDoubleClick
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} sheetTabIndex The index of the sheet tab that the user double-clicked.
                 * @example
                 * //This example uses the SheetTabDoubleClick event.
                 * spread.bind(GC.Spread.Sheets.Events.SheetTabDoubleClick, function (e, info) {
                 *         alert("Index (" + info.sheetTabIndex + ")");
                 * });
                 */
                static SheetTabDoubleClick: string;
                /**
                 * Occurs when any slicer has changed.
                 * @name GC.Spread.Sheets.Worksheet#SlicerChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Slicers.Slicer} slicer The slicer that triggered the event.
                 * @param {string} propertyName The name of the slicer's property that has changed.
                 * @example
                 * //This example uses the SlicerChanged event.
                 * //create a table
                 * datas = [
                 *     ["1", "NewYork", "1968/6/8", "80", "180"],
                 *     ["4", "NewYork", "1972/7/3", "72", "168"],
                 *     ["4", "NewYork", "1964/3/2", "71", "179"],
                 *     ["5", "Washington", "1972/8/8","80", "171"],
                 *     ["6", "Washington", "1986/2/2", "89", "161"],
                 *     ["7", "Washington", "2012/2/15", "71", "240"]];
                 * var table = activeSheet.tables.addFromDataSource("table1", 2, 2, datas);
                 * dataColumns = ["Name", "City", "Birthday", "Weight", "Height"];
                 * table.setColumnName(0, dataColumns[0]);
                 * table.setColumnName(1, dataColumns[1]);
                 * table.setColumnName(2, dataColumns[2]);
                 * table.setColumnName(3, dataColumns[3]);
                 * table.setColumnName(4, dataColumns[4]);
                 * //add a slicer to the sheet and return the slicer instance.
                 * var slicer = activeSheet.slicers.add("slicer1",table.name(),"Name");
                 * //change the slicer properties.
                 * slicer.width(200);
                 * slicer.height(200);
                 * slicer.position(new GC.Spread.Sheets.Point(100, 200));
                 * slicer.style(GC.Spread.Sheets.Slicers.SlicerStyles.dark4());
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.SlicerChanged, function (e, info) {
                 * console.log("name: " + info.propertyName);
                 * });
                 */
                static SlicerChanged: string;
                /**
                 * Occurs when the sparkline has changed.
                 * @name GC.Spread.Sheets.Worksheet#SparklineChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Sparklines.Sparkline} sparkline The sparkline whose property has changed.
                 * @example
                 * //Removing the sparkline causes a change.
                 * var data = new GC.Spread.Sheets.Range(1, 0, 8, 1);
                 * var setting = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * setting.options.showMarkers = true;
                 * setting.options.lineWeight = 3;
                 * setting.options.displayXAxis = true;
                 * setting.options.showFirst = true;
                 * setting.options.showLast = true;
                 * setting.options.showLow = true;
                 * setting.options.showHigh = true;
                 * setting.options.showNegative = true;
                 * setting.options.seriesColor = "Text 2 1";
                 * setting.options.firstMarkerColor = "Text 2 3";
                 * setting.options.negativeColor = "Accent 2 1";
                 * setting.options.markersColor = "Accent 3 1";
                 * setting.options.lowMarkerColor = "Accent 4 1";
                 * setting.options.highMarkerColor = "Accent 6 1";
                 * setting.options.lastMarkerColor = "Accent 6 6";
                 * setting.options.axisColor = "Text 1 1";
                 * activeSheet.addSpan(13, 0, 4, 3, null);
                 * activeSheet.setSparkline(13, 0, data, GC.Spread.Sheets.Sparklines.DataOrientation.vertical, GC.Spread.Sheets.Sparklines.SparklineType.line, setting);
                 * activeSheet.setValue(1, 0, 1);
                 * activeSheet.setValue(2, 0, -2);
                 * activeSheet.setValue(3, 0, -1);
                 * activeSheet.setValue(4, 0, 6);
                 * activeSheet.setValue(5, 0, 4);
                 * activeSheet.setValue(6, 0, -4);
                 * activeSheet.setValue(7, 0, 3);
                 * activeSheet.setValue(8, 0, 8);
                 * // Use IE to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.SparklineChanged, function (e, info) {
                 *      console.log("name: " + info.sheetName);
                 * });
                 * $("#button1").click(function () {
                 *      activeSheet.removeSparkline(13, 0);
                 * });
                 */
                static SparklineChanged: string;
                /**
                 * Occurs when the user insert/delete columns in table.
                 * @name GC.Spread.Sheets.Worksheet#TableColumnsChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {Table} table The table which is insert/delete rows.
                 * @param {string} propertyName The operation name which trigger event.
                 * @param {number} col The index of the starting column to insert/delete based on table index.
                 * @param {number} count The number of columns to insert/delete.
                 * @param {boolean} isAfter Whether insert columns before the specified column index or after. By default is false, insert before.
                 * @example
                 * //This example uses the TableColumnsChanged event.
                 * window.onload = function(){
                 *      var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 *      var activeSheet = spread.getActiveSheet();
                 *      spread.bind(GC.Spread.Sheets.Events.TableColumnsChanged, function (e, data) {});
                 * }
                 */
                static TableColumnsChanged: string;
                /**
                 * Occurs when a table column has just been clear filter.
                 * @name GC.Spread.Sheets.Worksheet#TableFilterCleared
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Tables.Table} table The table column to be automatically filtered.
                 * @param {number} tableCol The index of the table column has just been clear filter.
                 * @example
                 * //This example uses the TableFilterCleared event.
                 * activeSheet.tables.add("Table1", 0, 0, 3, 3, GC.Spread.Sheets.Tables.TableTheme.dark1);
                 * activeSheet.getCell(0,0).text("Name");
                 * activeSheet.getCell(0,1).text("Value");
                 * activeSheet.getCell(0,2).text("T/F");
                 * activeSheet.getCell(1,0).text("AW");
                 * activeSheet.getCell(1,1).text("5");
                 * activeSheet.getCell(1,2).text("T");
                 * activeSheet.bind(GC.Spread.Sheets.Events.TableFilterCleared, function (e, info) {
                 *        alert("Sheet (" + info.sheetName + ")");
                 * });
                 */
                static TableFilterCleared: string;
                /**
                 * Occurs when a table column is about to be automatically clear filter.
                 * @name GC.Spread.Sheets.Worksheet#TableFilterCleared
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Tables.Table} table The table column to be automatically filtered.
                 * @param {number} tableCol The index of the table column to be automatically clear filter.
                 * @example
                 * //This example uses the TableFilterClearing event.
                 * activeSheet.tables.add("Table1", 0, 0, 3, 3, GC.Spread.Sheets.Tables.TableTheme.dark1);
                 * activeSheet.getCell(0,0).text("Name");
                 * activeSheet.getCell(0,1).text("Value");
                 * activeSheet.getCell(0,2).text("T/F");
                 * activeSheet.getCell(1,0).text("AW");
                 * activeSheet.getCell(1,1).text("5");
                 * activeSheet.getCell(1,2).text("T");
                 * activeSheet.bind(GC.Spread.Sheets.Events.TableFilterClearing, function (e, info) {
                 *        alert("Sheet (" + info.sheetName + ")");
                 * });
                 */
                static TableFilterClearing: string;
                /**
                 * Occurs when a table column has just been automatically filtered.
                 * @name GC.Spread.Sheets.Worksheet#TableFiltered
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Tables.Table} table The table column to be automatically filtered.
                 * @param {number} col The index of the table column to be automatically filtered.
                 * @param {Array} filterValues The values by which to filter the column.
                 * @param {Object} conditionInfo The condition rule info by which to filter the column.
                 * @example
                 * //This example uses the TableFiltered event.
                 * activeSheet.tables.add("Table1", 0, 0, 3, 3, GC.Spread.Sheets.Tables.TableTheme.dark1);
                 * activeSheet.getCell(0,0).text("Name");
                 * activeSheet.getCell(0,1).text("Value");
                 * activeSheet.getCell(0,2).text("T/F");
                 * activeSheet.getCell(1,0).text("AW");
                 * activeSheet.getCell(1,1).text("5");
                 * activeSheet.getCell(1,2).text("T");
                 * activeSheet.bind(GC.Spread.Sheets.Events.TableFiltered, function (e, info) {
                 *        alert("Sheet (" + info.sheetName + ")");
                 * });
                 */
                static TableFiltered: string;
                /**
                 * Occurs when a table column is about to be automatically filtered.
                 * @name GC.Spread.Sheets.Worksheet#TableFiltering
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {GC.Spread.Sheets.Tables.Table} table The table column to be automatically filtered.
                 * @param {number} col The index of the table column to be automatically filtered.
                 * @param {Array} filterValues The values by which to filter the column.
                 * @param {Object} conditionInfo The condition rule info by which to filter the column.
                 * @example
                 * //This example uses the TableFiltering event.
                 * activeSheet.tables.add("Table1", 0, 0, 3, 3, GC.Spread.Sheets.Tables.TableTheme.dark1);
                 * activeSheet.getCell(0,0).text("Name");
                 * activeSheet.getCell(0,1).text("Value");
                 * activeSheet.getCell(0,2).text("T/F");
                 * activeSheet.getCell(1,0).text("AW");
                 * activeSheet.getCell(1,1).text("5");
                 * activeSheet.getCell(1,2).text("T");
                 * activeSheet.bind(GC.Spread.Sheets.Events.TableFiltering, function (e, info) {
                 *        alert("Sheet (" + info.sheetName + ")");
                 * });
                 */
                static TableFiltering: string;
                /**
                 * Occurs after the user resized table.
                 * @name GC.Spread.Sheets.Worksheet#TableResized
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {Table} table The table which is resized.
                 * @param {GC.Spread.Sheets.Range} oldRange The table range before resize.
                 * @param {GC.Spread.Sheets.Range} newRange The table range after resize.
                 */
                static TableResized: string;
                /**
                 * Occurs when the user resizing table by resize handler.
                 * @name GC.Spread.Sheets.Worksheet#TableResizing
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {Table} table The table which is resizing.
                 * @param {GC.Spread.Sheets.Range} oldRange The table range before resize.
                 * @param {GC.Spread.Sheets.Range} newRange The table range after resize.
                 * @example
                 * //This example uses the TableResizing and TableResized event.
                 * window.onload = function(){
                 *      var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 *      var activeSheet = spread.getActiveSheet();
                 *      spread.bind(GC.Spread.Sheets.Events.TableResizing, function (e, data) {});
                 *      spread.bind(GC.Spread.Sheets.Events.TableResized, function (e, data) {});
                 * }
                 */
                static TableResizing: string;
                /**
                 * Occurs when the user insert/delete rows in table.
                 * @name GC.Spread.Sheets.Worksheet#TableRowsChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {Table} table The table which is insert/delete rows.
                 * @param {string} propertyName The operation name which trigger event.
                 * @param {number} row The index of the starting row to insert/delete based on table index.
                 * @param {number} count The number of rows to insert/delete.
                 * @param {boolean} isAfter Whether insert rows before the specified row index or after. By default is false, insert before.
                 * @param {Array.<Object>} deletedItem The deleted rows collection in binding. The every item in array specifies deleted data item.
                 * @example
                 * //This example uses the TableRowsChanged event.
                 * window.onload = function(){
                 *      var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 *      var activeSheet = spread.getActiveSheet();
                 *      spread.bind(GC.Spread.Sheets.Events.TableRowsChanged, function (e, data) {});
                 * }
                 */
                static TableRowsChanged: string;
                /**
                 * Occurs when the top row changes.
                 * @name GC.Spread.Sheets.Worksheet#TopRowChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} oldTopRow The old top row index.
                 * @param {number} newTopRow The new top row index.
                 * @example
                 * //This example synchronizes vertical and horizontal scrolling for sheet 1 and sheet 2.
                 * var sheet1 = spread.getSheet(0),
                 * sheet2 = spread.getSheet(1);
                 * sheet1.bind(GC.Spread.Sheets.Events.TopRowChanged, function (sender, args) {
                 *     //Set the displayed top row of sheet1 to sheet2 (vertical scroll synchronization).
                 *     sheet2.showRow(args.newTopRow, GC.Spread.Sheets.VerticalPosition.top);
                 * });
                 * sheet1.bind(GC.Spread.Sheets.Events.LeftColumnChanged, function (sender, args) {
                 *     //Set the displayed left column of sheet1 to sheet2 (Horizontal scroll synchronization).
                 *     sheet2.showColumn(args.newLeftCol, GC.Spread.Sheets.HorizontalPosition.left);
                 * });
                 */
                static TopRowChanged: string;
                /**
                 * Occurs before the touch toolbar pops up.
                 * @name GC.Spread.Sheets.Worksheet#TouchToolStripOpening
                 * @event
                 * @param {number} x The <i>x</i>-coordinate of the horizontal position.
                 * @param {number} y The <i>y</i>-coordinate of the vertical position.
                 * @param {boolean} handled If <c>true</c>, the touch toolbar is prevented from popping up; otherwise, the toolbar is displayed at the default position.
                 * @example
                 * //This example uses the TouchToolStripOpening event.
                 *  activeSheet.bind(GC.Spread.Sheets.Events.TouchToolStripOpening, function (e, info) {
                 *         alert(info.x);
                 * });
                 */
                static TouchToolStripOpening: string;
                /**
                 * Occurs when the user types a formula.
                 * @name GC.Spread.Sheets.Worksheet#UserFormulaEntered
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of the cell in which the user entered a formula.
                 * @param {number} col The column index of the cell in which the user entered a formula.
                 * @param {string} formula The formula that the user entered.
                 * @example
                 * //This example uses the UserFormulaEntered event.
                 *  activeSheet.bind(GC.Spread.Sheets.Events.UserFormulaEntered, function (e, info) {
                 *         alert("Formula (" + info.formula + ")");
                 * });
                 */
                static UserFormulaEntered: string;
                /**
                 * Occurs when the user zooms.
                 * @name GC.Spread.Sheets.Worksheet#UserZooming
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} oldZoomFactor The new zoom factor.
                 * @param {number} newZoomFactor The old zoom factor.
                 * @example
                 * //This example uses the UserZooming event.
                 * spread.options.allowUserZoom = true;
                 * activeSheet.bind(GC.Spread.Sheets.Events.UserZooming, function (e, info) {
                 *      alert("Zoom (" + info.newZoomFactor + ")");
                 * });
                 */
                static UserZooming: string;
                /**
                 * Occurs when the applied cell value is invalid.
                 * @name GC.Spread.Sheets.Worksheet#ValidationError
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The cell's row index.
                 * @param {number} col The cell's column index.
                 * @param {GC.Spread.Sheets.DataValidation.DefaultDataValidator} validator The data validator that caused the error.
                 * @param {GC.Spread.Sheets.DataValidation.DataValidationResult} validationResult The policy that the user can set to determine how to process the error.
                 * @example
                 * //This example uses the ValidationError event.
                 * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.cellValueCondition);
                 * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.equalsTo);
                 * nCondition.expected(0);
                 * //When the option is false, the validation fails and the red alert is displayed.
                 * //When the option is true, the blank cell is treated as zero and the validation is successful.
                 * nCondition.treatNullValueAsZero(false);
                 * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition)
                 * validator.ignoreBlank(false);
                 * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.wholeNumber)
                 * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                 * spread.options.highlightInvalidData = true;
                 * activeSheet.setValue(0, 0, null);
                 * //Type different values in cell (0,0). This event fires if the user types an invalid value.
                 * activeSheet.bind("ValidationError", vError);
                 * function vError(sender, args) {
                 *     alert("error");
                 * }
                 */
                static ValidationError: string;
                /**
                 * Occurs when the value in the subeditor changes.
                 * @name GC.Spread.Sheets.Worksheet#ValueChanged
                 * @event
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that triggered the event.
                 * @param {string} sheetName The sheet's name.
                 * @param {number} row The row index of the cell.
                 * @param {number} col The column index of the cell.
                 * @param {Object} oldValue The old value of the cell.
                 * @param {Object} newValue The new value of the cell.
                 * @example
                 * //This example uses the ValueChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.ValueChanged, function (e, info) {
                 *         alert("Value (" + info.newValue + ")");
                 * });
                 */
                static ValueChanged: string;
            }

            export class LineBorder{
                /**
                 * Represents the line border for a border side.
                 * @class
                 * @param {string} [color] Indicates the border color and uses a format such as color name (for example, "red") or "#RGB", "#RRGGBB", "rgb(R,B,B)", "rgba(R,G,B,A)".
                 * @param {GC.Spread.Sheets.LineStyle} [style] Indicates the border line style.
                 * @example
                 * //This example creates a border.
                 * var border = new GC.Spread.Sheets.LineBorder
                 * border.color = "#7FFFD4";
                 * border.style = GC.Spread.Sheets.LineStyle.double;
                 * var cell = activeSheet.getCell(1, 1, GC.Spread.Sheets.SheetArea.viewport);
                 * cell.borderLeft(border);
                 */
                constructor(color?:  string,  style?:  GC.Spread.Sheets.LineStyle);
                /**
                 * Indicates the color of the border line. Use a known color name or HEX style color value. The default value is black.
                 * @example
                 * //This example sets the color property.
                 * var border = new GC.Spread.Sheets.LineBorder
                 * border.color = "#7FFFD4";
                 * border.style = GC.Spread.Sheets.LineStyle.double;
                 * var cell = activeSheet.getCell(1, 1, GC.Spread.Sheets.SheetArea.viewport);
                 * cell.borderLeft(border);
                 */
                color: string;
                /**
                 * Indicates the line style of the border line. The default value is empty.
                 * @example
                 * //This example sets the style property.
                 * var border = new GC.Spread.Sheets.LineBorder
                 * border.color = "#7FFFD4";
                 * border.style = GC.Spread.Sheets.LineStyle.double;
                 * var cell = activeSheet.getCell(1, 1, GC.Spread.Sheets.SheetArea.viewport);
                 * cell.borderLeft(border);
                 */
                style: GC.Spread.Sheets.LineStyle;
            }

            export class NameInfo{
                /**
                 * Represents a custom named expression that can be used by formulas.
                 * @class
                 * @param {string} name The custom expression name.
                 * @param {GC.Spread.CalcEngine.Expression} expr The custom named expression.
                 * @param {number} row The base row of the expression.
                 * @param {number} column The base column of the expression.
                 * @param {string} [comment] The custom expression comment.
                 */
                constructor(name:  string,  expr:  GC.Spread.CalcEngine.Expression,  row:  number,  column:  number,  comment?:  string);
                /**
                 * Gets the base column of the custom named expression.
                 * @returns {number} The base column.
                 */
                getColumn(): number;
                /**
                 * Gets the comment of the current NameInfo object.
                 * @returns {string} The name of the current NameInfo object.
                 */
                getComment(): string;
                /**
                 * Gets the expression.
                 * @returns {GC.Spread.CalcEngine.Expression} The expression.
                 */
                getExpression(): GC.Spread.CalcEngine.Expression;
                /**
                 * Gets the name of the current NameInfo object.
                 * @returns {string} The name of the current NameInfo object.
                 */
                getName(): string;
                /**
                 * Gets the base row of the custom named expression.
                 * @returns {number} The base row.
                 */
                getRow(): number;
            }

            export class Point{
                /**
                 * Represents an <i>x</i>- and <i>y</i>-coordinate pair in two-dimensional space.
                 * @class
                 * @param {number} x The <i>x</i>-coordinate.
                 * @param {number} y The <i>y</i>-coordinate.
                 * @example
                 * //This example creates a floating object.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1");
                 * customFloatingObject.x(10);
                 * customFloatingObject.y(10);
                 * customFloatingObject.width(60);
                 * customFloatingObject.height(64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 */
                constructor(x:  number,  y:  number);
                /**
                 * Clones a new point from the current point.
                 * @returns {GC.Spread.Sheets.Point} The cloned object.
                 */
                clone(): GC.Spread.Sheets.Point;
            }

            export class Range{
                /**
                 * Represents a range, which is described by the row index, column index, row count, and column count.
                 * @class
                 * @param {number} r The row index.
                 * @param {number} c The column index.
                 * @param {number} rc The row count.
                 * @param {number} cc The column count.
                 * @example
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                 * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                 * rule.ranges([new GC.Spread.Sheets.Range(0,0,5,1)]);
                 * rule.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between);
                 * rule.style(style);
                 * rule.value1(2);
                 * rule.value2(100);
                 * activeSheet.conditionalFormats.addRule(rule);
                 * activeSheet.setValue(0,0,1,3);
                 * activeSheet.setValue(1,0,45,3);
                 */
                constructor(r:  number,  c:  number,  rc: number,  cc: number);
                /**
                 * The column index.
                 * @example
                 * var cellrange = new GC.Spread.Sheets.Range();
                 * cellrange.col = 0;
                 * cellrange.row = 1;
                 * cellrange.colCount = 1;
                 * cellrange.rowCount = 8;
                 * activeSheet.setValue(1, 0, 1);
                 * activeSheet.setValue(2, 0, -2);
                 * activeSheet.setValue(3, 0, -1);
                 * activeSheet.setValue(4, 0, 6);
                 * activeSheet.setValue(5, 0, 4);
                 * activeSheet.setValue(6, 0, -4);
                 * activeSheet.setValue(7, 0, 3);
                 * activeSheet.setValue(8, 0, 8);
                 * var setting = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * setting.options.showMarkers = true;
                 * activeSheet.setSparkline(13, 0, cellrange, GC.Spread.Sheets.Sparklines.DataOrientation.vertical, GC.Spread.Sheets.Sparklines.SparklineType.line, setting);
                 */
                col: number;
                /**
                 * The column count.
                 * @example
                 * var cellrange = new GC.Spread.Sheets.Range();
                 * cellrange.col = 0;
                 * cellrange.row = 1;
                 * cellrange.colCount = 1;
                 * cellrange.rowCount = 8;
                 * activeSheet.setValue(1, 0, 1);
                 * activeSheet.setValue(2, 0, -2);
                 * activeSheet.setValue(3, 0, -1);
                 * activeSheet.setValue(4, 0, 6);
                 * activeSheet.setValue(5, 0, 4);
                 * activeSheet.setValue(6, 0, -4);
                 * activeSheet.setValue(7, 0, 3);
                 * activeSheet.setValue(8, 0, 8);
                 * var setting = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * setting.options.showMarkers = true;
                 * activeSheet.setSparkline(13, 0, cellrange, GC.Spread.Sheets.Sparklines.DataOrientation.vertical, GC.Spread.Sheets.Sparklines.SparklineType.line, setting);
                 */
                colCount: number;
                /**
                 * The row index.
                 * @example
                 * var cellrange = new GC.Spread.Sheets.Range();
                 * cellrange.col = 0;
                 * cellrange.row = 1;
                 * cellrange.colCount = 1;
                 * cellrange.rowCount = 8;
                 * activeSheet.setValue(1, 0, 1);
                 * activeSheet.setValue(2, 0, -2);
                 * activeSheet.setValue(3, 0, -1);
                 * activeSheet.setValue(4, 0, 6);
                 * activeSheet.setValue(5, 0, 4);
                 * activeSheet.setValue(6, 0, -4);
                 * activeSheet.setValue(7, 0, 3);
                 * activeSheet.setValue(8, 0, 8);
                 * var setting = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * setting.options.showMarkers = true;
                 * activeSheet.setSparkline(13, 0, cellrange, GC.Spread.Sheets.Sparklines.DataOrientation.vertical, GC.Spread.Sheets.Sparklines.SparklineType.line, setting);
                 */
                row: number;
                /**
                 * The row count.
                 * @example
                 * var cellrange = new GC.Spread.Sheets.Range();
                 * cellrange.col = 0;
                 * cellrange.row = 1;
                 * cellrange.colCount = 1;
                 * cellrange.rowCount = 8;
                 * activeSheet.setValue(1, 0, 1);
                 * activeSheet.setValue(2, 0, -2);
                 * activeSheet.setValue(3, 0, -1);
                 * activeSheet.setValue(4, 0, 6);
                 * activeSheet.setValue(5, 0, 4);
                 * activeSheet.setValue(6, 0, -4);
                 * activeSheet.setValue(7, 0, 3);
                 * activeSheet.setValue(8, 0, 8);
                 * var setting = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * setting.options.showMarkers = true;
                 * activeSheet.setSparkline(13, 0, cellrange, GC.Spread.Sheets.Sparklines.DataOrientation.vertical, GC.Spread.Sheets.Sparklines.SparklineType.line, setting);
                 */
                rowCount: number;
                /**
                 * Gets whether the current range contains the specified cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {number} rowCount The row count.
                 * @param {number} colCount The column count.
                 * @returns {boolean} <c>true</c> if the range contains the cell; otherwise, <c>false</c>.
                 */
                contains(row:  number,  col:  number,  rowCount:  number,  colCount:  number): boolean;
                /**
                 * Gets whether the current range contains the specified range.
                 * @param {GC.Spread.Sheets.Range} range The cell range.
                 * @returns {boolean} <c>true</c> if the current range contains the specified cell range; otherwise, <c>false</c>.
                 */
                containsRange(range:  GC.Spread.Sheets.Range): boolean;
                /**
                 * Gets whether the current range is equal to the specified range.
                 * @param {GC.Spread.Sheets.Range} range The range to compare.
                 * @returns {boolean} <c>true</c> if the current range is equal to the specified range; otherwise, <c>false</c>.
                 */
                equals(range:  GC.Spread.Sheets.Range): boolean;
                /**
                 * Gets the intersection of two cell ranges.
                 * @param {GC.Spread.Sheets.Range} range The cell range.
                 * @param {number} maxRowCount The maximum row count.
                 * @param {number} maxColumnCount The maximum column count.
                 * @returns {GC.Spread.Sheets.Range} Returns null if there is no intersection, or the cell range of the intersection.
                 */
                getIntersect(range:  GC.Spread.Sheets.Range,  maxRowCount:  number,  maxColumnCount:  number): GC.Spread.Sheets.Range;
                /**
                 * Gets whether the current range intersects with the one specified by the row and column index and the row and column count.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {number} rowCount The row count.
                 * @param {number} colCount The column count.
                 * @returns {boolean} <c>true</c> if the specified range intersects with the current range; otherwise <c>false</c>.
                 */
                intersect(row:  number,  col:  number,  rowCount:  number,  colCount:  number): boolean;
                /**
                 * Offsets the location of the range by the specified coordinates.
                 * @param {number} x The offset along the <i>x</i>-axis.
                 * @param {number} y The offset along the <i>y</i>-axis.
                 * @returns {GC.Spread.Sheets.Range} The new location.
                 */
                offset(x:  number,  y:  number): GC.Spread.Sheets.Range;
                /**
                 * Joins this range with the specified range as a union.
                 * @param {GC.Spread.Sheets.Range} range The target range.
                 * @returns {GC.Spread.Sheets.Range} Returns the union of the ranges.
                 */
                union(range:  GC.Spread.Sheets.Range): GC.Spread.Sheets.Range;
            }

            export class Rect{
                /**
                 * Represents a rectangle with a special location, and its width and height in two-dimensional space.
                 * @class
                 * @param {number} x The <i>x</i>-coordinate of the top-left corner of the rectangle.
                 * @param {number} y The <i>y</i>-coordinate of the top-left corner of the rectangle.
                 * @param {number} w The width of the rectangle.
                 * @param {number} h The height of the rectangle.
                 */
                constructor(x:  number,  y:  number,  w:  number,  h:  number);
                /**
                 * The width of the rectangle.
                 */
                height: number;
                /**
                 * The height of the rectangle.
                 */
                width: number;
                /**
                 * The <i>x</i>-coordinate of the top-left corner of the rectangle.
                 */
                x: number;
                /**
                 * The <i>y</i>-coordinate of the top-left corner of the rectangle.
                 */
                y: number;
                /**
                 * Indicates whether the rectangle contains the specified <i>x</i>-coordinate and <i>y</i>-coordinate.
                 * @param {number} x The <i>x</i>-coordinate of the point to check.
                 * @param {number} y The <i>y</i>-coordinate of the point to check.
                 * @returns {boolean} <c>true</c> if (x, y) is contained by the rectangle; otherwise, <c>false</c>.
                 */
                contains(x:  number,  y:  number): boolean;
                /**
                 * Gets the rectangle that intersects with the current rectangle.
                 * @param {GC.Spread.Sheets.Rect} rect The rectangle.
                 * @returns {GC.Spread.Sheets.Rect} The intersecting rectangle. If the two rectangles do not intersect, returns null.
                 */
                getIntersectRect(x:  number,  y:  number,  width:  number,  height:  number): GC.Spread.Sheets.Rect;
                /**
                 * Indicates whether the specified rectangle intersects with the current rectangle.
                 * @param {number} x The <i>x</i>-coordinate of the top-left corner of the rectangle.
                 * @param {number} y The <i>y</i>-coordinate of the top-left corner of the rectangle.
                 * @param {number} w The width of the rectangle.
                 * @param {number} h The height of the rectangle.
                 * @returns {boolean} <c>true</c> if the specified rectangle intersects with the current rectangle; otherwise, <c>false</c>.
                 */
                intersect(x:  number,  y:  number,  width:  number,  height:  number): boolean;
                /**
                 * Indicates whether the specified rectangle intersects with the current rectangle.
                 * @param {GC.Spread.Sheets.Rect} rect The specified rectangle.
                 * @returns {boolean} <c>true</c> if the specified rectangle intersects with the current rectangle; otherwise, <c>false</c>.
                 */
                intersectRect(rect:  GC.Spread.Sheets.Rect): boolean;
            }

            export class Style{
                /**
                 * Represents the style for a cell, row, and column.
                 * @class
                 * @param {string} backColor The background color.
                 * @param {string} foreColor The foreground color.
                 * @param {GC.Spread.Sheets.HorizontalAlign} hAlign The horizontal alignment.
                 * @param {GC.Spread.Sheets.VerticalAlign} vAlign The vertical alignment.
                 * @param {string} font The font.
                 * @param {string} themeFont The font theme.
                 * @param {string|GC.Spread.Formatter.GeneralFormatter} formatter The formatting object.
                 * @param {GC.Spread.Sheets.LineBorder} borderLeft The left border.
                 * @param {GC.Spread.Sheets.LineBorder} borderTop The top border.
                 * @param {GC.Spread.Sheets.LineBorder} borderRight The right border.
                 * @param {GC.Spread.Sheets.LineBorder} borderBottom The bottom border.
                 * @param {GC.Spread.Sheets.LineBorder} diagonalDown The diagonal with LeftTop to bottomRight.
                 * @param {GC.Spread.Sheets.LineBorder} diagonalUp The diagonal with topRight to bottomLeft.
                 * @param {boolean} locked Whether the cell, row, or column is locked.
                 * @param {number} textIndent The text indent amount.
                 * @param {boolean} wordWrap Whether words wrap within the cell or cells.
                 * @param {boolean} shrinkToFit Whether content shrinks to fit the cell or cells.
                 * @param {string} backgroundImage The background image to display.
                 * @param {GC.Spread.Sheets.CellTypes.Base} cellType The cell type.
                 * @param {GC.Spread.Sheets.ImageLayout} backgroundImageLayout The layout for the background image.
                 * @param {boolean} tabStop Whether the user can set focus to the cell using the Tab key.
                 * @param {GC.Spread.Sheets.TextDecorationType} textDecoration Specifies the decoration added to text.
                 * @param {GC.Spread.Sheets.ImeMode} imeMode Specifies the input method editor mode.
                 * @param {string} name Specifies the name.
                 * @param {string} parentName Specifies the name of the parent style.
                 * @param {string} watermark Specifies the watermark content.
                 * @param {string} cellPadding Specifies the cell padding.
                 * @param {Object} labelOptions Specifies the cell label options.
                 * @param {GC.Spread.Sheets.LabelAlignment} [labelOptions.alignment] The cell label position.
                 * @param {GC.Spread.Sheets.LabelVisibility} [labelOptions.visibility] The cell label visibility.
                 * @param {string} [labelOptions.font] The cell label font.
                 * @param {string} [labelOptions.foreColor] The cell label forecolor.
                 * @param {string} [labelOptions.margin] The cell label margin.
                 * @param {boolean} isVerticalText Whether to set the cell's text vertical.
                 * @param {number} textOrientation he cell text rotation angle.
                 * @example
                 * //This example uses a style with rules.
                 * activeSheet.setArray(0,0,[1,2,3,4,5,6,7,8,9,10]);
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.foreColor = "black";
                 * var cell = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                 * cell.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                 * cell.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.greaterThan);
                 * cell.value1(5);
                 * cell.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 1)]);
                 * cell.style(style);
                 * activeSheet.conditionalFormats.addRule(cell);
                 * var style1 = new GC.Spread.Sheets.Style();
                 * style1.foreColor = "red";
                 * var top = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                 * top.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.top10Rule);
                 * top.type(GC.Spread.Sheets.ConditionalFormatting.Top10ConditionType.top);
                 * top.rank(3);
                 * top.style(style1);
                 * top.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 1)]);
                 * top.stopIfTrue(true);
                 * activeSheet.conditionalFormats.addRule(top);
                 * @example
                 * //This example creates and sets parameters for the style.
                 * sheet.setValue(0,0, 1,3);
                 * sheet.setValue(1,0, 50,3);
                 * sheet.setValue(2,0, 100,3);
                 * sheet.setValue(3,0, 2,3);
                 * sheet.setValue(4,0, 60,3);
                 * sheet.setValue(5,0, 90,3);
                 * sheet.setValue(6,0, 3,3);
                 * sheet.setValue(7,0, 40,3);
                 * sheet.setValue(8,0, 70,3);
                 * sheet.setValue(9,0, 5,3);
                 * sheet.setValue(10,0, 35,3);
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom =  new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * sheet.conditionalFormats.addAverageRule(GC.Spread.Sheets.ConditionalFormatting.AverageConditionType.above, style, [new GC.Spread.Sheets.Range(0, 0, 10, 3)]);
                 */
                constructor(backColor?:  string,  foreColor?:  string,  hAlign?:  GC.Spread.Sheets.HorizontalAlign,  vAlign?:  GC.Spread.Sheets.VerticalAlign,  font?:  string,  themeFont?:  string,  formatter?:  string | GC.Spread.Formatter.GeneralFormatter,  borderLeft?:  GC.Spread.Sheets.LineBorder,  borderTop?:  GC.Spread.Sheets.LineBorder,  borderRight?:  GC.Spread.Sheets.LineBorder,  borderBottom?:  GC.Spread.Sheets.LineBorder,  locked?:  boolean,  textIndent?:  number,  wordWrap?:  boolean,  shrinkToFit?:  boolean,  backgroundImage?:  string,  cellType?:  GC.Spread.Sheets.CellTypes.Base,  backgroundImageLayout?:  GC.Spread.Sheets.ImageLayout,  tabStop?:  boolean,  textDecoration?:  GC.Spread.Sheets.TextDecorationType,  imeMode?:  GC.Spread.Sheets.ImeMode,  name?:  string,  parentName?:  string,  watermark?:  string,  cellPadding?:  string,  labelOptions?:  GC.Spread.Sheets.ILabelOptions,  diagonalDown?:  GC.Spread.Sheets.LineBorder,  diagonalUp?:  GC.Spread.Sheets.LineBorder,  isVerticalText?:  boolean,  textOrientation?:  number);
                /**
                 * Indicates the background color.
                 * @type {string}
                 * @example
                 * //This example sets the style backColor property.
                 * activeSheet.setArray(0,0,[1,2,3,4,5,6,7,8,9,10]);
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.foreColor = "black";
                 * var cell = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                 * cell.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                 * cell.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.greaterThan);
                 * cell.value1(5);
                 * cell.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 1)]);
                 * cell.style(style);
                 * activeSheet.conditionalFormats.addRule(cell);
                 * var style1 = new GC.Spread.Sheets.Style();
                 * style1.foreColor = "red";
                 * var top = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                 * top.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.top10Rule);
                 * top.type(GC.Spread.Sheets.ConditionalFormatting.Top10ConditionType.top);
                 * top.rank(3);
                 * top.style(style1);
                 * top.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 1)]);
                 * top.stopIfTrue(true);
                 * activeSheet.conditionalFormats.addRule(top);
                 */
                backColor: string | undefined;
                /**
                 * Indicates the background image.
                 * @type {string}
                 * @example
                 * //This example sets the backgroundImage property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "lightgreen";
                 * style.backgroundImage = "./css/images/quarter1.png";
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                backgroundImage: string | undefined;
                /**
                 * Indicates the background image layout.
                 * @type {GC.Spread.Sheets.ImageLayout}
                 * @example
                 * //This example sets the backgroundImageLayout property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "lightgreen";
                 * style.backgroundImage = "./css/images/quarter1.png";
                 * style.backgroundImageLayout  = GC.Spread.Sheets.ImageLayout.center;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                backgroundImageLayout: GC.Spread.Sheets.ImageLayout | undefined;
                /**
                 * Indicates the bottom border line.
                 * @type {GC.Spread.Sheets.LineBorder}
                 * @example
                 * //This example creates a border.
                 * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                 * cellType.buttonBackColor("#FFFF00");
                 * cellType.text("this is a button");
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.cellType = cellType
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                borderBottom: GC.Spread.Sheets.LineBorder | undefined;
                /**
                 * Indicates the left border line.
                 * @type {GC.Spread.Sheets.LineBorder}
                 * @example
                 * //This example creates a border.
                 * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                 * cellType.buttonBackColor("#FFFF00");
                 * cellType.text("this is a button");
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.cellType = cellType
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                borderLeft: GC.Spread.Sheets.LineBorder | undefined;
                /**
                 * Indicates the right border line.
                 * @type {GC.Spread.Sheets.LineBorder}
                 * @example
                 * //This example creates a border.
                 * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                 * cellType.buttonBackColor("#FFFF00");
                 * cellType.text("this is a button");
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.cellType = cellType
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                borderRight: GC.Spread.Sheets.LineBorder | undefined;
                /**
                 * Indicates the top border line.
                 * @type {GC.Spread.Sheets.LineBorder}
                 * @example
                 * //This example creates a border.
                 * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                 * cellType.buttonBackColor("#FFFF00");
                 * cellType.text("this is a button");
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.cellType = cellType
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                borderTop: GC.Spread.Sheets.LineBorder | undefined;
                /**
                 * Indicates the cellButtons of cell.
                 * @type {Object[]}
                 * @example
                 * leftButtonConfig1 = {
                 *     caption: "left",
                 *     enabled: true,
                 *     visibility:GC.Spread.Sheets.ButtonVisibility.always,
                 * };
                 * rightButtonConfig1 = {
                 *     caption: "left",
                 *     enabled: true,
                 *     visibility:GC.Spread.Sheets.ButtonVisibility.onSelected,
                 * };
                 * //create style
                 * var style = new GC.Spread.Sheets.Style();
                 * style.cellButtons=[
                 *      leftButtonConfig1,
                 *      rightButtonConfig1
                 * ];
                 * sheet.setStyle(0, 0, style);
                 */
                cellButtons: GC.Spread.Sheets.ICellButton[] | undefined;
                /**
                 * Indicates the cell padding.
                 * @type {string}
                 * @example
                 * //This example sets the cell padding for the watermark.
                 * var type = new GC.Spread.Sheets.Style();
                 * type.watermark = "User name";
                 * type.cellPadding = "20";
                 * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
                 * activeSheet.setStyle(0, 1, type);
                 * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 * activeSheet.getRange(-1, 1, -1, 1).width(150);
                 * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
                 * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
                 * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
                 * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 */
                cellPadding: string | undefined;
                /**
                 * Indicates the cell type.
                 * @type {GC.Spread.Sheets.CellTypes.Base}
                 * @example
                 * //This example creates a style and applies it to a cell.
                 * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                 * cellType.buttonBackColor("#FFFF00");
                 * cellType.text("this is a button");
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.cellType = cellType
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                cellType: GC.Spread.Sheets.CellTypes.Base | undefined;
                /**
                 * Indicates the diagonalDown border line.
                 * @type {GC.Spread.Sheets.LineBorder}
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).diagonalDown(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 */
                diagonalDown: GC.Spread.Sheets.LineBorder | undefined;
                /**
                 * Indicates the diagonalUp border line.
                 * @type {GC.Spread.Sheets.LineBorder}
                 * @example
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).diagonalUp(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 */
                diagonalUp: GC.Spread.Sheets.LineBorder | undefined;
                /**
                 * Indicates the dropdown type of cell.
                 * @type {Object[]}
                 * @example
                 * leftButtonConfig1 = {
                 *     caption: "left",
                 *     enabled: true,
                 *     isLeft: true,
                 *     visibility:GC.Spread.Sheets.ButtonVisibility.always,
                 *     command: "opendropdown"
                 * }
                 * //create style
                 * var style = new GC.Spread.Sheets.Style();
                 * style.cellButtons=[
                 *      leftButtonConfig1
                 * ];
                 * style.dropDowns= [{ type: "colorPicker" }];
                 * sheet.setStyle(0, 0, style);
                 */
                dropDowns: GC.Spread.Sheets.IDropdown[] | undefined;
                /**
                 * Indicates the font.
                 * @type {string}
                 * @example
                 * //This example sets the font property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.font = "8pt Arial";
                 * style.hAlign = GC.Spread.Sheets.HorizontalAlign.center;
                 * style.vAlign = GC.Spread.Sheets.VerticalAlign.center;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).value("B2");
                 */
                font: string | undefined;
                /**
                 * Indicates the foreground color.
                 * @type {string}
                 * @example
                 * //This example sets the foreColor property.
                 * activeSheet.setArray(0,0,[1,2,3,4,5,6,7,8,9,10]);
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.foreColor = "black";
                 * var cell = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                 * cell.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                 * cell.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.greaterThan);
                 * cell.value1(5);
                 * cell.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 1)]);
                 * cell.style(style);
                 * activeSheet.conditionalFormats.addRule(cell);
                 * var style1 = new GC.Spread.Sheets.Style();
                 * style1.foreColor = "red";
                 * var top = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                 * top.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.top10Rule);
                 * top.type(GC.Spread.Sheets.ConditionalFormatting.Top10ConditionType.top);
                 * top.rank(3);
                 * top.style(style1);
                 * top.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 1)]);
                 * top.stopIfTrue(true);
                 * activeSheet.conditionalFormats.addRule(top);
                 */
                foreColor: string | undefined;
                /**
                 * Indicates the formatter.
                 * @type {string|GC.Spread.Formatter.GeneralFormatter}
                 * @example
                 * //This example uses the formatter property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.formatter = "0.000%";
                 * style.themeFont = "Body";
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).value("11");
                 */
                formatter: string | GC.Spread.Formatter.GeneralFormatter | undefined;
                /**
                 * Indicates the horizontal alignment.
                 * @type {GC.Spread.Sheets.HorizontalAlign}
                 * @example
                 * //This example sets the hAlign property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.font = "8pt Arial";
                 * style.hAlign = GC.Spread.Sheets.HorizontalAlign.center;
                 * style.vAlign = GC.Spread.Sheets.VerticalAlign.center;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).value("B2");
                 */
                hAlign: GC.Spread.Sheets.HorizontalAlign | undefined;
                /**
                 * Indicates the Input Method Editor (IME) mode.
                 * @type {GC.Spread.Sheets.ImeMode}
                 * @example
                 * //This example sets the IME mode.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.imeMode = GC.Spread.Sheets.ImeMode.auto;
                 * activeSheet.setStyle(0, 0, style);
                 */
                imeMode: GC.Spread.Sheets.ImeMode | undefined;
                /**
                 * Indicates whether to set the text vertical.
                 * @type {boolean}
                 * @example
                 * //This example uses the rotate property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.isVerticalText = true;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                isVerticalText: boolean | undefined;
                /**
                 * Indicates the cell label options.
                 * @property {GC.Spread.Sheets.LabelAlignment} [alignment] - The cell label position.
                 * @property {GC.Spread.Sheets.LabelVisibility} [visibility] - The cell label visibility.
                 * @property {string} [font] - The cell label font.
                 * @property {string} [foreColor] - The cell label forecolor.
                 * @property {string} [margin] - The cell label margin.
                 * @type {Object}
                 * @example
                 * //This example sets label options for the watermark.
                 * var type = new GC.Spread.Sheets.Style();
                 * type.watermark = "User name";
                 * type.cellPadding = "20";
                 * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
                 * activeSheet.setStyle(0, 1, type);
                 * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 * activeSheet.getRange(-1, 1, -1, 1).width(150);
                 * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
                 * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
                 * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
                 * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 */
                labelOptions: GC.Spread.Sheets.ILabelOptions | undefined;
                /**
                 * Indicates whether a cell is marked as locked from editing.
                 * @type {boolean}
                 * @example
                 * //This example unlocks a single cell.
                 * activeSheet.options.isProtected = true;
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.locked = false;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                locked: boolean | undefined;
                /**
                 * Indicates the name.
                 * @type {string}
                 * @example
                 * //This example sets the style name.
                 * var namedStyle = new GC.Spread.Sheets.Style();
                 * namedStyle.name = "style1";
                 * namedStyle.backColor = "green";
                 * activeSheet.addNamedStyle(namedStyle);
                 * activeSheet.setStyleName(1, 1, "style1"); // cell(1,1)'s backColor is green.
                 * var namedStyle1 = new GC.Spread.Sheets.Style();
                 * namedStyle1.name = "style2";
                 * namedStyle1.parentName = "style1";
                 * namedStyle1.foreColor = "red";    // the namedStyle's foreColor is red.
                 * activeSheet.addNamedStyle(namedStyle1);
                 * activeSheet.setStyleName(2, 1, "style2");
                 */
                name: string | undefined;
                /**
                 * Indicates the name of the parent style.
                 * @type {string}
                 * @example
                 * //This example sets the parent name.
                 * var namedStyle = new GC.Spread.Sheets.Style();
                 * namedStyle.name = "style1";
                 * namedStyle.backColor = "green";
                 * activeSheet.addNamedStyle(namedStyle);
                 * activeSheet.setStyleName(1, 1, "style1"); // cell(1,1)'s backColor is green.
                 * var namedStyle1 = new GC.Spread.Sheets.Style();
                 * namedStyle1.name = "style2";
                 * namedStyle1.parentName = "style1";
                 * namedStyle1.foreColor = "red";    // the namedStyle's foreColor is red.
                 * activeSheet.addNamedStyle(namedStyle1);
                 * activeSheet.setStyleName(2, 1, "style2");
                 */
                parentName: string | undefined;
                /**
                 * Controls whether the Text out of bounds shows ellipsis.
                 * @type {boolean}
                 * @example
                 * //This example sets the showEllipsis property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.showEllipsis = true;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).text("TestTestTestTest");
                 */
                showEllipsis: boolean | undefined;
                /**
                 * Indicates whether to shrink to fit.
                 * @type {boolean}
                 * @example
                 * //This example sets the shrinkToFit property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.shrinkToFit = true;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).text("Shrink To Fit");
                 */
                shrinkToFit: boolean | undefined;
                /**
                 * Indicates whether the user can set focus to the cell using the Tab key.
                 * @type {boolean}
                 * @example
                 * //This example sets the tabStop property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.tabStop = false;
                 * style.backColor = "lightgreen";
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                tabStop: boolean | undefined;
                /**
                 * Indicates the decoration added to text.
                 * @type {GC.Spread.Sheets.TextDecorationType}
                 * @example
                 * //This example uses the textDecoration property.
                 * activeSheet.getCell(0, 0).textDecoration(GC.Spread.Sheets.TextDecorationType.underline);
                 * activeSheet.getRange(1, -1, 1, -1).textDecoration(GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.underline);
                 * activeSheet.getRange(-1, 1, -1, 1).textDecoration(GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.lineThrough | GC.Spread.Sheets.TextDecorationType.underline);
                 * var style = new GC.Spread.Sheets.Style();
                 * style.textDecoration = GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.underline;
                 * activeSheet.setStyle(1, 1, style, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(0, 0).value("Test");
                 * activeSheet.getCell(1, 0).value("Test");
                 * activeSheet.getCell(0, 1).value("Test");
                 */
                textDecoration: GC.Spread.Sheets.TextDecorationType | undefined;
                /**
                 * Indicates the number of units of indentation for text in a cell, an integer value, where an increment of 1 represents 8 pixels.
                 * @type {number}
                 * @example
                 * //This example sets the textIndent property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.textIndent = 3;
                 * // Enter multiple lines with Alt+Enter
                 * style.wordWrap = true;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).text("Test");
                 */
                textIndent: number | undefined;
                /**
                 * Indicates the text rotation angle of cell.
                 * @type {number}
                 * @example
                 * //This example sets the textOrientation property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.textOrientation = 66;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).text("TestTestTestTest");
                 */
                textOrientation: number | undefined;
                /**
                 * Indicates the font theme.
                 * @type {string}
                 * @example
                 * //This example uses the themeFont property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.formatter = "0.000%";
                 * style.themeFont = "Body";
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).value("11");
                 */
                themeFont: string | undefined;
                /**
                 * Indicates the vertical alignment.
                 * @type {GC.Spread.Sheets.VerticalAlign}
                 * @example
                 * //This example sets the vAlign property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.font = "8pt Arial";
                 * style.hAlign = GC.Spread.Sheets.HorizontalAlign.center;
                 * style.vAlign = GC.Spread.Sheets.VerticalAlign.center;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).value("B2");
                 */
                vAlign: GC.Spread.Sheets.VerticalAlign | undefined;
                /**
                 * Indicates the watermark content.
                 * @type {string}
                 * @example
                 * //The following examples add watermarks to cells, columns, and rows.
                 * var type = new GC.Spread.Sheets.Style();
                 * type.watermark = "User name";
                 * sheet.setStyle(0, 1, type);
                 * var type = new GC.Spread.Sheets.Style();
                 * type.watermark = "Password";
                 * sheet.setStyle(1, 1, type);
                 * @example
                 * var type = new GC.Spread.Sheets.Style();
                 * type.watermark = "The watermark.";
                 * activeSheet.setStyle(-1, 1, type);
                 * activeSheet.setStyle(1, -1, type);
                 * activeSheet.setStyle(2, 2, type);
                 */
                watermark: string | undefined;
                /**
                 * Indicates whether to wrap text.
                 * @type {boolean}
                 * @example
                 * //This example sets the wordWrap property.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.textIndent = 3;
                 * // Enter multiple lines with Alt+Enter
                 * style.wordWrap = true;
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(1,1).text("Test");
                 */
                wordWrap: boolean | undefined;
                /**
                 * Clones the current style.
                 * @returns {GC.Spread.Sheets.Style} The cloned style.
                 */
                clone(): GC.Spread.Sheets.Style;
            }

            export class Theme{
                /**
                 * Represents a color scheme.
                 * @class
                 * @param {string} name The name of the theme.
                 * @param {GC.Spread.Sheets.ColorScheme} colorScheme The base colors of the theme color.
                 * @param {string} headerFont The name of the heading font.
                 * @param {string} bodyFont The name of the body font.
                 * @example
                 * //This example creates a new SpreadTheme object.
                 * sheet.getCell(0, 0).backColor("accent 1");
                 * sheet.getCell(1, 0).backColor("accent 6");
                 * $("#btn").click(function () {
                 *     var custom = new GC.Spread.Sheets.Theme("Custom");
                 *     custom.colors().accent1("red");
                 *     custom.colors().accent6("green");
                 *     sheet.currentTheme(custom);
                 * })
                 */
                constructor(name:  string,  colorScheme:  ColorScheme,  headerFont:  string,  bodyFont:  string);
                /**
                 * Gets or sets the body font of the theme.
                 * @param {string} value The body font.
                 * @returns {string|GC.Spread.Sheets.Theme} If no value is set, returns the body font; otherwise, returns the theme.
                 */
                bodyFont(value?:  string): any;
                /**
                 * Gets or sets the base colors of the theme.
                 * @param {GC.Spread.Sheets.ColorScheme} value The base colors of the theme.
                 * @returns {GC.Spread.Sheets.ColorScheme|GC.Spread.Sheets.Theme} If no value is set, returns the base colors of the theme; otherwise, returns the theme.
                 */
                colors(value?:  GC.Spread.Sheets.ColorScheme): any;
                /**
                 * Gets or sets the heading font of the theme.
                 * @param {string} value The heading font.
                 * @returns {string|GC.Spread.Sheets.Theme} If no value is set, returns the heading font; otherwise, returns the theme.
                 */
                headerFont(value?:  string): any;
                /**
                 * Gets or sets the name of the theme.
                 * @param {string} value The theme name.
                 * @returns {string|GC.Spread.Sheets.Theme} If no value is set, returns the theme name; otherwise, returns the theme.
                 */
                name(value?:  string): any;
            }

            export class ThemeColors{
                /**
                 * Represents the theme color of built-in themes.
                 * @class
                 * @example
                 * //This example creates a custom theme.
                 * var custom = new GC.Spread.Sheets.Theme("CustomTheme");
                 * custom.colors().accent1("red");
                 * custom.colors().accent6("green");
                 * custom.colors().textColor1("orange");
                 * activeSheet.currentTheme(custom);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 1 30");
                 */
                constructor();
                /**
                 * The theme color of the Apex theme.
                 * @example
                 * //This example uses the Apex theme.
                 * activeSheet.currentTheme("Apex");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Apex: ColorScheme;
                /**
                 * The theme color of the Aspect theme.
                 * @example
                 * //This example uses the Aspect theme.
                 * activeSheet.currentTheme("Aspect");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Aspect: ColorScheme;
                /**
                 * The theme color of the Civic theme.
                 * @example
                 * //This example uses the Civic theme.
                 * activeSheet.currentTheme("Civic");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Civic: ColorScheme;
                /**
                 * The theme color of the Concourse theme.
                 * @example
                 * //This example uses the Concourse theme.
                 * activeSheet.currentTheme("Concourse");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Concourse: ColorScheme;
                /**
                 * The theme color of the Default theme.
                 * @example
                 * //This example sets the Default theme.
                 * activeSheet.currentTheme("Default");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Default: ColorScheme;
                /**
                 * The theme color of the Equity theme.
                 * @example
                 * //This example sets the Equity theme.
                 * activeSheet.currentTheme("Equity");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Equity: ColorScheme;
                /**
                 * The theme color of the Flow theme.
                 * @example
                 * //This example sets the Flow theme.
                 * activeSheet.currentTheme("Flow");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Flow: ColorScheme;
                /**
                 * The theme color of the Foundry theme.
                 * @example
                 * //This example sets the Foundry theme.
                 * activeSheet.currentTheme("Foundry");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Foundry: ColorScheme;
                /**
                 * The theme color of the Median theme.
                 * @example
                 * //This example sets the Median theme.
                 * activeSheet.currentTheme("Median");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Median: ColorScheme;
                /**
                 * The theme color of the Metro theme.
                 * @example
                 * //This example sets the Metro theme.
                 * activeSheet.currentTheme("Metro");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Metro: ColorScheme;
                /**
                 * The theme color of the Module theme.
                 * @example
                 * //This example sets the Module theme.
                 * activeSheet.currentTheme("Module");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Module: ColorScheme;
                /**
                 * The theme color of the Office theme.
                 * @example
                 * //This example sets the Office theme.
                 * activeSheet.currentTheme("Office");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Office: ColorScheme;
                /**
                 * The theme color of the Office 2007 theme.
                 * @example
                 * //This example uses the Office2007 theme.
                 * activeSheet.currentTheme("Office2007");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Office2007: ColorScheme;
                /**
                 * The theme color of the Opulent theme.
                 * @example
                 * //This example sets the Opulent theme.
                 * activeSheet.currentTheme("Opulent");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Opulent: ColorScheme;
                /**
                 * The theme color of the Oriel theme.
                 * @example
                 * //This example sets the Oriel theme.
                 * activeSheet.currentTheme("Oriel");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Oriel: ColorScheme;
                /**
                 * The theme color of the Origin theme.
                 * @example
                 * //This example sets the Origin theme.
                 * activeSheet.currentTheme("Origin");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Origin: ColorScheme;
                /**
                 * The theme color of the Paper theme.
                 * @example
                 * //This example sets the Paper theme.
                 * activeSheet.currentTheme("Paper");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Paper: ColorScheme;
                /**
                 * The theme color of the Solstice theme.
                 * @example
                 * //This example sets the Solstice theme.
                 * activeSheet.currentTheme("Solstice");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Solstice: ColorScheme;
                /**
                 * The theme color of the Technic theme.
                 * @example
                 * //This example sets the Technic theme.
                 * activeSheet.currentTheme("Technic");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Technic: ColorScheme;
                /**
                 * The theme color of the Trek theme.
                 * @example
                 * //This example sets the Trek theme.
                 * activeSheet.currentTheme("Trek");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Trek: ColorScheme;
                /**
                 * The theme color of the Urban theme.
                 * @example
                 * //This example sets the Urban theme.
                 * activeSheet.currentTheme("Urban");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Urban: ColorScheme;
                /**
                 * The theme color of the Verve theme.
                 * @example
                 * //This example sets the Verve theme.
                 * activeSheet.currentTheme("Verve");
                 * activeSheet.resumePaint();
                 * activeSheet.repaint();
                 */
                Verve: ColorScheme;
            }

            export class Themes{
                /**
                 * Represents all built-in themes.
                 * @class
                 */
                constructor();
                /**
                 * Indicates the Apex theme.
                 */
                Apex: Theme;
                /**
                 * Indicates the Aspect theme.
                 */
                Aspect: Theme;
                /**
                 * Indicates the Civic theme.
                 */
                Civic: Theme;
                /**
                 * Indicates the Concourse theme.
                 */
                Concourse: Theme;
                /**
                 * Indicates the Default theme.
                 */
                Default: Theme;
                /**
                 * Indicates the Equity theme.
                 */
                Equity: Theme;
                /**
                 * Indicates the Flow theme.
                 */
                Flow: Theme;
                /**
                 * Indicates the Foundry theme.
                 */
                Foundry: Theme;
                /**
                 * Indicates the Median theme.
                 */
                Median: Theme;
                /**
                 * Indicates the Metro theme.
                 */
                Metro: Theme;
                /**
                 * Indicates the Module theme.
                 */
                Module: Theme;
                /**
                 * Indicates the Office theme.
                 */
                Office: Theme;
                /**
                 * Indicates the Office 2007 theme.
                 */
                Office2007: Theme;
                /**
                 * Indicates the Opulent theme.
                 */
                Opulent: Theme;
                /**
                 * Indicates the Oriel theme.
                 */
                Oriel: Theme;
                /**
                 * Indicates the Origin theme.
                 */
                Origin: Theme;
                /**
                 * Indicates the Paper theme.
                 */
                Paper: Theme;
                /**
                 * Indicates the Solstice theme.
                 */
                Solstice: Theme;
                /**
                 * Indicates the Technic theme.
                 */
                Technic: Theme;
                /**
                 * Indicates the Trek theme.
                 */
                Trek: Theme;
                /**
                 * Indicates the Urban theme.
                 */
                Urban: Theme;
                /**
                 * Indicates the Verve theme.
                 */
                Verve: Theme;
            }

            export class Workbook{
                /**
                 * Represents a spreadsheet with the specified hosted DOM element and options setting.
                 * @class
                 * @param {HTMLElement} host - The host DOM element.
                 * @param {Object} [options] - The initialization options.<br />
                 * @param {number} [options.sheetCount] - The number of sheets.<br />
                 * @param {string} [options.font] - The tab strip font.<br />
                 * @param {boolean} [options.allowUserDragMerge] - Whether to allow the user to drag merge cells.<br />
                 * @param {boolean} [options.allowUserDragDrop] - Whether to allow the user to drag and drop range data.<br />
                 * @param {boolean} [options.allowUserDragFill] - Whether to allow the user to drag fill a range.<br />
                 * @param {boolean} [options.allowUserZoom] - Whether to zoom the display by scrolling the mouse wheel while pressing the Ctrl key.<br />
                 * @param {boolean} [options.allowUserResize] - Whether to allow the user to resize columns and rows.<br />
                 * @param {boolean} [options.allowUndo] - Whether to allow the user to undo edits.<br />
                 * @param {boolean} [options.allowSheetReorder] - Whether the user can reorder the sheets in the Spread component.<br />
                 * @param {boolean} [options.allowContextMenu] -  Whether to allow the user to open the built-in context menu.<br />
                 * @param {boolean} [options.allowUserDeselect] -  Whether to allow the user to can use deselect in selection.<br />
                 * @param {GC.Spread.Sheets.Fill.AutoFillType} [options.defaultDragFillType] - The default fill type.<br />
                 * @param {boolean} [options.showDragFillSmartTag] - Whether to display the drag fill dialog.<br />
                 * @param {boolean} [options.showHorizontalScrollbar] - Whether to display the horizontal scroll bar.<br />
                 * @param {boolean} [options.showVerticalScrollbar] - Whether to display the vertical scroll bar.<br />
                 * @param {boolean} [options.scrollbarShowMax] - Whether the displayed scroll bars are based on the entire number of columns and rows in the sheet.<br />
                 * @param {boolean} [options.scrollbarMaxAlign] - Whether the scroll bar aligns with the last row and column of the active sheet.<br />
                 * @param {boolean} [options.tabStripVisible] - Whether to display the sheet tab strip.<br />
                 * @param {number} [options.tabStripRatio] - The width of the tab strip expressed as a percentage of the overall horizontal scroll bar width.<br />
                 * @param {boolean} [options.tabEditable] - Whether to allow the user to edit the sheet tab strip.<br />
                 * @param {boolean} [options.newTabVisible] - Whether the spreadsheet displays the special tab to let users insert new sheets.<br />
                 * @param {boolean} [options.tabNavigationVisible] - Whether to display the sheet tab navigation.<br />
                 * @param {boolean} [options.cutCopyIndicatorVisible] - Whether to display an indicator when copying or cutting the selected item.<br />
                 * @param {string} [options.cutCopyIndicatorBorderColor] - The border color for the indicator displayed when the user cuts or copies the selection.<br />
                 * @param {string} [options.backColor] - A color string used to represent the background color of the Spread component, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.<br />
                 * @param {string} [options.backgroundImage] - The background image of the Spread component.<br />
                 * @param {GC.Spread.Sheets.ImageLayout} [options.backgroundImageLayout] - The background image layout for the Spread component.<br />
                 * @param {string} [options.grayAreaBackColor] - A color string used to represent the background color of the gray area , such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.<br />
                 * @param {GC.Spread.Sheets.ShowResizeTip} [options.showResizeTip] - How to display the resize tip.<br />
                 * @param {boolean} [options.showDragDropTip] -Whether to display the drag-drop tip.<br />
                 * @param {boolean} [options.showDragFillTip] - Whether to display the drag-fill tip.<br />
                 * @param {GC.Spread.Sheets.ShowScrollTip} [options.showScrollTip] - How to display the scroll tip.<br />
                 * @param {boolean} [options.scrollIgnoreHidden] - Whether the scroll bar ignores hidden rows or columns.<br />
                 * @param {boolean} [options.highlightInvalidData] - Whether to highlight invalid data.<br />
                 * @param {boolean} [options.useTouchLayout] - Whether to use touch layout to present the Spread component.<br />
                 * @param {boolean} [options.hideSelection] - Whether to display the selection highlighting when the Spread component does not have focus.<br />
                 * @param {GC.Spread.Sheets.ResizeZeroIndicator} [options.resizeZeroIndicator] - The drawing policy when the row or column is resized to zero.<br />
                 * @param {boolean} [options.allowUserEditFormula] - Whether the user can edit formulas in a cell in the spreadsheet.<br />
                 * @param {boolean} [options.enableFormulaTextbox] - Whether to enable the formula text box in the spreadsheet.<br />
                 * @param {GC.Spread.Sheets.AutoFitType} [options.autoFitType] - Whether content will be formatted to fit in cells or in cells and headers.<br />
                 * @param {GC.Spread.Sheets.ReferenceStyle} [options.referenceStyle] - the style for cell and range references in cell formulas on this sheet.
                 * @param {boolean} [options.calcOnDemand] - Whether to calculate formulas only when they are demanded.
                 * @param {boolean} [options.allowCopyPasteExcelStyle] - Whether the user can copy style from Spread Sheets then paste to Excel, or copy style from Excel then paste to Spread Sheets.
                 * @param {boolean} [options.allowExtendPasteRange] - Whether extend paste range if the paste range is not enough for pasting.
                 * @param {GC.Spread.Sheets.CopyPasteHeaderOptions} [options.copyPasteHeaderOptions] - Which headers are included when data is copied to or pasted.
                 * @param {boolean} [options.scrollByPixel] - Whether to enable the precision scrolling by pixel.
                 * @param {number} [options.scrollPixel] - Decides scrolling by that number of pixels at a time when scrollByPixel is true. The final scrolling pixels are the result of scrolling delta multiply scrollPixel. For example, the scrolling delta is 3, and the scrollPixel is 5, the final scrolling pixels are 15.
                 * @param {boolean} [options.allowAutoCreateHyperlink] - Whether to enable auto creating hyperlink in the spreadsheet.
                 * @param {GC.Spread.Sheets.ResizeMode} [options.columnResizeMode] - Specifies the way to resize column.
                 * @param {GC.Spread.Sheets.ResizeMode} [options.rowResizeMode] - Specifies the way to resize row.
                 * @param {Array} [options.customList] - The list for user to customize drag fill, prioritize matching this list in each fill. Each array item is type of string array.
                 * @example
                 * var workbook = new GC.Spread.Sheets.Workbook(document.getElementById("ss"), {sheetCount:3, font:"12pt Arial"});
                 * var workbook = new GC.Spread.Sheets.Workbook(document.getElementById("ss"), {sheetcount:3, newTabVisible:false});
                 * var workbook = new GC.Spread.Sheets.Workbook(document.getElementById("ss"), { sheetCount: 3, tabEditable: false });
                 * var workbook = new GC.Spread.Sheets.Workbook(document.getElementById("ss"), {sheetCount:3, tabStripVisible:false});
                 * var workbook = new GC.Spread.Sheets.Workbook(document.getElementById("ss"), {sheetCount:3, allowUserResize:false});
                 * var workbook = new GC.Spread.Sheets.Workbook(document.getElementById("ss"), { sheetCount: 3, allowUserZoom: false});
                 */
                constructor(host?:  HTMLElement,  options?:  GC.Spread.Sheets.IWorkBookDefaultOptions);
                /**
                 * ContextMenu for the spread.
                 * @type {GC.Spread.Sheets.ContextMenu.ContextMenu}
                 * @example
                 * //This example shows how to get contextMenu's menuData.
                 * var menuData = spread.contextMenu.menuData;
                 */
                contextMenu: GC.Spread.Sheets.ContextMenu.ContextMenu;
                /**
                 * Represents the name of the Spread control.
                 * @type {string}
                 * @example
                 * spread.name = "Spread1";
                 */
                name: string;
                /**
                 * Represents the options of the Spread control.
                 * @type {Object}
                 * @property {boolean} allowUserDragMerge - Whether to allow the user to drag merge cells.<br />
                 * @property {boolean} allowUserDragDrop - Whether to allow the user to drag and drop range data.<br />
                 * @property {boolean} allowUserDragFill - Whether to allow the user to drag fill a range.<br />
                 * @property {boolean} allowUserZoom - Whether to zoom the display by scrolling the mouse wheel while pressing the Ctrl key.<br />
                 * @property {boolean} allowUserResize - Whether to allow the user to resize columns and rows.<br />
                 * @property {boolean} allowUndo - Whether to allow the user to undo edits.<br />
                 * @property {boolean} allowSheetReorder - Whether the user can reorder the sheets in the Spread component.<br />
                 * @property {boolean} allowContextMenu - Whether to allow the user to open the built-in context menu. <br />
                 * @property {boolean} allowUserDeselect -  Whether to allow the user to can use deselect in selection.<br />
                 * @property {GC.Spread.Sheets.Fill.AutoFillType} defaultDragFillType - The default fill type.<br />
                 * @property {boolean} showDragFillSmartTag - Whether to display the drag fill dialog.<br />
                 * @property {boolean} showHorizontalScrollbar - Whether to display the horizontal scroll bar.<br />
                 * @property {boolean} showVerticalScrollbar - Whether to display the vertical scroll bar.<br />
                 * @property {boolean} scrollbarShowMax - Whether the displayed scroll bars are based on the entire number of columns and rows in the sheet.<br />
                 * @property {boolean} scrollbarMaxAlign - Whether the scroll bar aligns with the last row and column of the active sheet.<br />
                 * @property {boolean} tabStripVisible - Whether to display the sheet tab strip.<br />
                 * @property {number} tabStripRatio - The width of the tab strip expressed as a percentage of the overall horizontal scroll bar width.<br />
                 * @property {boolean} tabEditable - Whether to allow the user to edit the sheet tab strip.<br />
                 * @property {boolean} newTabVisible - Whether the spreadsheet displays the special tab to let users insert new sheets.<br />
                 * @property {boolean} tabNavigationVisible - Whether to display the sheet tab navigation.<br />
                 * @property {boolean} cutCopyIndicatorVisible - Whether to display an indicator when copying or cutting the selected item.<br />
                 * @property {string} cutCopyIndicatorBorderColor - The border color for the indicator displayed when the user cuts or copies the selection.<br />
                 * @property {string} backColor - A color string used to represent the background color of the Spread component, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.<br />
                 * @property {string} backgroundImage - The background image of the Spread component.<br />
                 * @property {GC.Spread.Sheets.ImageLayout} backgroundImageLayout - The background image layout for the Spread component.<br />
                 * @property {string} grayAreaBackColor - A color string used to represent the background color of the gray area , such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.<br />
                 * @property {GC.Spread.Sheets.ShowResizeTip} showResizeTip - How to display the resize tip.<br />
                 * @property {boolean} showDragDropTip -Whether to display the drag-drop tip.<br />
                 * @property {boolean} showDragFillTip - Whether to display the drag-fill tip.<br />
                 * @property {GC.Spread.Sheets.ShowScrollTip} showScrollTip - How to display the scroll tip.<br />
                 * @property {boolean} scrollIgnoreHidden - Whether the scroll bar ignores hidden rows or columns.<br />
                 * @property {boolean} highlightInvalidData - Whether to highlight invalid data.<br />
                 * @property {boolean} useTouchLayout - Whether to use touch layout to present the Spread component.<br />
                 * @property {boolean} hideSelection - Whether to display the selection highlighting when the Spread component does not have focus.<br />
                 * @property {GC.Spread.Sheets.ResizeZeroIndicator} resizeZeroIndicator - The drawing policy when the row or column is resized to zero.<br />
                 * @property {boolean} allowUserEditFormula - Whether the user can edit formulas in a cell in the spreadsheet.<br />
                 * @property {boolean} enableFormulaTextbox - Whether to enable the formula text box in the spreadsheet.<br />
                 * @property {GC.Spread.Sheets.AutoFitType} autoFitType - Whether content will be formatted to fit in cells or in cells and headers.<br />
                 * @property {GC.Spread.Sheets.ReferenceStyle} referenceStyle - the style for cell and range references in cell formulas on this sheet.
                 * @property {boolean} allowDynamicArray - Whether to enable dynamic array.
                 * @property {boolean} calcOnDemand - Whether to calculate formulas only when they are demanded.
                 * @property {boolean} allowCopyPasteExcelStyle - Whether the user can copy style from Spread Sheets then paste to Excel, or copy style from Excel then paste to Spread Sheets.
                 * @property {boolean} allowExtendPasteRange - Whether extend paste range if the paste range is not enough for pasting.
                 * @property {GC.Spread.Sheets.CopyPasteHeaderOptions} copyPasteHeaderOptions - Which headers are included when data is copied to or pasted.
                 * @property {boolean} scrollByPixel - Whether to enable the precision scrolling by pixel.
                 * @property {number} scrollPixel - Decides scrolling by that number of pixels at a time when scrollByPixel is true. The final scrolling pixels are the result of scrolling delta multiply scrollPixel. For example, the scrolling delta is 3, and the scrollPixel is 5, the final scrolling pixels are 15.
                 * @property {boolean} allowAutoCreateHyperlink - Whether to enable auto creating hyperlink in the spreadsheet.
                 * @property {GC.Spread.Sheets.ResizeMode} columnResizeMode - Specifies the way to resize column.
                 * @property {GC.Spread.Sheets.ResizeMode} rowResizeMode - Specifies the way to resize row.
                 * @property {Array} customList - The list for user to customize drag fill, prioritize matching this list in each fill. Each array item is type of string array.
                 * @example
                 * var workbook = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:5,showHorizontalScrollbar:false});
                 * workbook.options.allowUserDragDrop = false;
                 * workbook.options.allowUserZoom = false;
                 */
                options: IWorkbookOptions;
                /**
                 * Represents the sheet collection.
                 * @type {Array.<GC.Spread.Sheets.Worksheet>}
                 */
                sheets: GC.Spread.Sheets.Worksheet[];
                /**
                 * Represents the touch toolstrip.
                 * @type {GC.Spread.Sheets.Touch.TouchToolStrip}
                 */
                touchToolStrip: GC.Spread.Sheets.Touch.TouchToolStrip;
                /**
                 * Adds a custom function.
                 * @param {GC.Spread.CalcEngine.Functions.Function} fn The function to add.
                 */
                addCustomFunction(fn:  GC.Spread.CalcEngine.Functions.Function): void;
                /**
                 * Adds a custom name.
                 * @param {string} name The custom name.
                 * @param {string} formula The formula.
                 * @param {number} baseRow The row index.
                 * @param {number} baseCol The column index.
                 * @param {string} comment The custom comment.
                 */
                addCustomName(name:  string,  formula:  string,  baseRow:  number,  baseCol:  number,  comment:  string): void;
                /**
                 * Adds a style to the Workbook named styles collection.
                 * @param {GC.Spread.Sheets.Style} style The style to be added.
                 */
                addNamedStyle(style:  GC.Spread.Sheets.Style): void;
                /**
                 * Inserts a sheet at the specific index.
                 * @param {number} index The index at which to add a sheet.
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet to be added.
                 * @example
                 * //This example adds a sheet to the spreadsheet.
                 * spread.addSheet(0,new GC.Spread.Sheets.Worksheet("custom"));
                 */
                addSheet(index:  number,  sheet?:  GC.Spread.Sheets.Worksheet): void;
                /**
                 * Adds a SparklineEx to the SparklineEx collection.
                 * @param {GC.Spread.Sheets.Sparklines.SparklineEx} sparklineEx The SparklineEx to be added.
                 */
                addSparklineEx(sparklineEx:  GC.Spread.Sheets.Sparklines.SparklineEx): void;
                /**
                 * Binds an event to the Workbook.
                 * @param {string} type The event type.
                 * @param {Object} data Specifies additional data to pass along to the function.
                 * @param {Function} fn Specifies the function to run when the event occurs.
                 * @example
                 * //This example binds events to functions.
                 * sheet.setActiveCell(5,5);
                 * alert(sheet.getActiveColumnIndex());
                 * alert(sheet.getActiveRowIndex());
                 * spread.bind(GC.Spread.Sheets.Events.EnterCell, function (event, data) {
                 *     alert(data.col);
                 *     alert(data.row);
                 *   });
                 * spread.bind(GC.Spread.Sheets.Events.LeaveCell, function (event, data) {
                 *     alert(data.col);
                 *     alert(data.row);
                 *   });
                 */
                bind(type:  string,  data?:  any,  fn?:  Function): void;
                /**
                 * Clears all custom functions.
                 */
                clearCustomFunctions(): void;
                /**
                 * Clears custom names.
                 */
                clearCustomNames(): void;
                /**
                 * Clears all sheets in the control.
                 * @example
                 * //This example uses the clearSheets method.
                 * spread.clearSheets();
                 */
                clearSheets(): void;
                /**
                 * Gets the command manager.
                 * @returns {GC.Spread.Commands.CommandManager} The command manager.
                 * @example
                 * //This example executes a command that performs a specified action.
                 * spread.options.allowUndo = true;
                 * spread.commandManager().execute({cmd: "outlineRow", sheetName: "Sheet1", index: 3, count: 5});
                 */
                commandManager(): GC.Spread.Commands.CommandManager;
                /**
                 * Destroys the workbook and all sheets it contains.
                 * @example
                 * //This example destroys the workbook instance.
                 * spread.destroy();
                 */
                destroy(): void;
                /**
                 * Makes the Workbook component get focus or lose focus.
                 * @param {boolean} focusIn <c>false</c> makes the Workbook component lose the focus; otherwise, get focus.
                 * @example
                 * //This example sets focus to the Spread control.
                 * $("#button1").click(function () {
                 * spread.focus(true);
                 *    });
                 */
                focus(focusIn?:  boolean): void;
                /**
                 * Loads the object state from the specified JSON string.
                 * @param {Object} workbookData The spreadsheet data from deserialization.
                 * @param {Object} [deserializationOptions] - The deserialization options.
                 * @param {boolean} [deserializationOptions.ignoreStyle] - Whether to ignore the style when converting json to the workbook.
                 * @param {boolean} [deserializationOptions.ignoreFormula] - Whether to ignore the formula when converting json to the workbook.
                 * @param {boolean} [deserializationOptions.frozenColumnsAsRowHeaders] - Whether to treat the frozen columns as row headers when converting json to the workbook.
                 * @param {boolean} [deserializationOptions.frozenRowsAsColumnHeaders] - Whether to treat the frozen rows as column headers when converting json to the workbook.
                 * @param {boolean} [deserializationOptions.doNotRecalculateAfterLoad] - Whether to prevent recalculation after loading the json data.
                 * @example
                 * //This example uses the fromJSON method.
                 * activeSheet.getCell(0,0).value(123);
                 * var jsonStr = null;
                 * //export
                 * jsonStr = JSON.stringify(spread.toJSON(false));
                 * //import
                 * spread.fromJSON(JSON.parse(jsonStr));
                 * alert(jsonStr);
                 */
                fromJSON(workbookData:  Object,  deserializationOptions?:  Object): void;
                /**
                 * Gets the active sheet.
                 * @returns {GC.Spread.Sheets.Worksheet} The active sheet instance.
                 */
                getActiveSheet(): GC.Spread.Sheets.Worksheet;
                /**
                 * Gets the active sheet index of the control.
                 * @returns {number} The active sheet index.
                 * @example
                 * //This example uses the getActiveSheetIndex method.
                 * var index = spread.getActiveSheetIndex();
                 * alert(index);
                 */
                getActiveSheetIndex(): number;
                /**
                 * Gets a custom function.
                 * @param {string} name The custom function name.
                 * @returns {GC.Spread.CalcEngine.Functions.Function} The custom function.
                 */
                getCustomFunction(name:  string): void;
                /**
                 * Gets the specified custom name information.
                 * @param {string} name The custom name.
                 * @returns {GC.Spread.Sheets.NameInfo} The information for the specified custom name.
                 */
                getCustomName(name:  string): GC.Spread.Sheets.NameInfo;
                /**
                 * Gets all custom name information.
                 * @returns {Array.<GC.Spread.Sheets.NameInfo>} The type GC.Spread.Sheets.NameInfo stored in an array.
                 */
                getCustomNames(): GC.Spread.Sheets.NameInfo[];
                /**
                 * Gets the host element of the current Workbook instance.
                 * @returns {HTMLElement} host The host element of the current Workbook instance.
                 */
                getHost(): HTMLElement;
                /**
                 * Gets a style from the Workbook named styles collection which has the specified name.
                 * @param {string} name The name of the style to return.
                 * @returns {GC.Spread.Sheets.Style} Returns the specified named style.
                 */
                getNamedStyle(name:  string): GC.Spread.Sheets.Style;
                /**
                 * Gets named styles from the Workbook.
                 * @returns {Array.<GC.Spread.Sheets.Style>} The GC.Spread.Sheets.Style array of named styles.
                 */
                getNamedStyles(): GC.Spread.Sheets.Style[];
                /**
                 * Gets the specified sheet.
                 * @param {number} index The index of the sheet to return.
                 * @returns {GC.Spread.Sheets.Worksheet} The specified sheet.
                 * @example
                 * //This example gets the sheet and sets the cell forecolor.
                 * var sheet1 = spread.getSheet(1);
                 * sheet1.getCell(0,0).value("A1").foreColor("red");
                 */
                getSheet(index:  number): GC.Spread.Sheets.Worksheet;
                /**
                 * Gets the number of sheets.
                 * @returns {number} The number of sheets.
                 * @example
                 * //This example uses the getSheetCount method.
                 * var index = spread.getSheetCount();
                 * alert(index);
                 */
                getSheetCount(): number;
                /**
                 * Gets the sheet with the specified name.
                 * @param {string} name The sheet name.
                 * @returns {GC.Spread.Sheets.Worksheet} The sheet with the specified name.
                 * @example
                 * //This example gets the sheet and sets the cell forecolor.
                 * var sheet1 = spread.getSheetFromName("Sheet2");
                 * sheet1.getCell(0,0).value("A1").foreColor("red");
                 */
                getSheetFromName(name:  string): GC.Spread.Sheets.Worksheet;
                /**
                 * Gets the sheet index with the specified name.
                 * @param {string} name The sheet name.
                 * @returns {number} The sheet index.
                 * @example
                 * //This example uses the getSheetIndex method.
                 * var sheet1 = spread.getSheetIndex("Sheet2");
                 * alert(sheet1);
                 */
                getSheetIndex(name:  string): number;
                /**
                 * Performs a hit test.
                 * @param {number} x The <i>x</i>-coordinate, x relative to spread horizontal axis.
                 * @param {number} y The <i>y</i>-coordinate, y relative to spread vertical axis.
                 * @returns {Object} The hit test information. If selecting the worksheet, the worksheet information is returned. The information contains x, y, and worksheetHitInfo;
                 * If selecting the sheetsTabStrip, the sheetsTabStrip information is returned. This information contains x, y, and tabStripHitInfo;
                 * If selecting the horizontalScrollbar, the horizontalScrollbar information is returned. This information contains x, y, and horizontalScrollBarHitInfo;
                 * If selecting the verticalScrollbar, the verticalScrollbar information is returned. This information contains x, y, and verticalScrollBarHitInfo;
                 * If selecting the footerCorner, the footerCorner information is returned. This information contains x, y, and footerCornerHitInfo.
                 * @example
                 * //This example uses the hitTest method.
                 *       window.onload = function(){
                 *           var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 *           var activeeSheet = spread.getActiveSheet();
                 *           $("#ss").click(function (e) {
                 *               //Acquire cell index from mouse-clicked point of regular cells which are neither fixed rows/columns nor row/column headers.
                 *               var offset = $("#ss").offset();
                 *               var x = e.pageX - offset.left;
                 *               var y = e.pageY - offset.top;
                 *               var target = spread.hitTest(x, y);
                 *               if(target.worksheetHitInfo) {
                 *                   if(target.worksheetHitInfo.hitTestType === 0) {
                 *                       str = 'corner';
                 *                   } else if (target.worksheetHitInfo.hitTestType === 1) {
                 *                       str = 'colHeader';
                 *                   } else if (target.worksheetHitInfo.hitTestType === 2) {
                 *                       str = 'rowHeader';
                 *                   } else {
                 *                       str = 'viewport';
                 *                   }
                 *               } else if(target.tabStripHitInfo) {
                 *                   if(target.tabStripHitInfo.navButton){
                 *                       str = target.tabStripHitInfo.navButton;
                 *                   } else if(target.tabStripHitInfo.sheetTab) {
                 *                       str = target.tabStripHitInfo.sheetTab.sheetName;
                 *                   } else if(target.tabStripHitInfo.resize === true) {
                 *                       str = "resize";
                 *                   } else {
                 *                       str = "blank";
                 *                   }
                 *               } else if(target.horizontalScrollBarHitInfo) {
                 *                   str = target.horizontalScrollBarHitInfo.element;
                 *               } else if(target.verticalScrollBarHitInfo) {
                 *                   str = target.verticalScrollBarHitInfo.element;
                 *               } else if(target.footerCornerHitInfo) {
                 *                   str = target.footerCornerHitInfo.element;
                 *               }
                 *         alert(str);
                 * });
                 * }
                 */
                hitTest(x:  number,  y:  number): IWorkbookHitTestInformation;
                /**
                 * Updates the control layout information.
                 * @example
                 * //This example updates the layout.
                 * spread.invalidateLayout();
                 * spread.repaint();
                 */
                invalidateLayout(): void;
                /**
                 * Get if spread paint is suspended.
                 */
                isPaintSuspended(): boolean;
                /**
                 * Gets or sets the next control used by GC.Spread.Sheets.Actions.selectNextControl and GC.Spread.Sheets.Actions.moveToNextCellThenControl.
                 * @param {HTMLElement} value The next control. The control must have a focus method.
                 * @returns {HTMLElement|GC.Spread.Sheets.Workbook} If no value is set, returns the next control; otherwise, returns the spreadsheet.
                 */
                nextControl(value?:  HTMLElement): any;
                /**
                 *Get a page info for a sheet
                 * @param {number} sheetIndex The sheet index.
                 * @returns {Object | Array} return a page info for a sheet, If the sheet index is ignored return all sheet's page info in an array
                 * @example
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * spread.suspendPaint();
                 * var sheet = spread.getActiveSheet();
                 * for(var i=0;i<20;i++){
                 *    for(var j=0;j<20;j++){
                 *        sheet.setValue(i,j,"Row"+i+"_Column"+j);
                 *    }
                 * }
                 * var pageInfos =  spread.pageInfo(0);
                 * console.table(pageInfos.pages);
                 */
                pageInfo(sheetIndex?:  number): any;
                /**
                 * Gets or sets the previous control used by GC.Spread.Sheets.Actions.selectPreviousControl and GC.Spread.Sheets.Actions.moveToPreviousCellThenControl.
                 * @param {HTMLElement} value The previous control. The control must have a focus method.
                 * @returns {HTMLElement|GC.Spread.Sheets.Workbook} If no value is set, returns the previous control; otherwise, returns the spreadsheet.
                 */
                previousControl(value?:  HTMLElement): any;
                /**
                 *Prints the specified sheet.
                 *@param {number} sheetIndex The sheet index. If the sheet index is ignored, prints all visible sheets.
                 */
                print(sheetIndex?:  number): void;
                /**
                 * Manually refreshes the layout and rendering of the Workbook object.
                 * @example
                 * //This example uses the refresh method.
                 * spread.refresh();
                 */
                refresh(): void;
                /**
                 * Removes a custom function.
                 * @param {string} name The custom function name.
                 */
                removeCustomFunction(name:  string): void;
                /**
                 * Removes the specified custom name.
                 * @param {string} name The custom name.
                 * @example
                 * //This example uses the removeCustomName method.
                 * activeSheet.setValue(0, 0, 1);
                 * activeSheet.setValue(0, 1, 2);
                 * activeSheet.setValue(0, 2, 3);
                 * spread.addCustomName("customName1","=12", 0, 0);
                 * activeSheet.setFormula(1, 0, "customName1");
                 * //spread.removeCustomName("customName1");
                 */
                removeCustomName(name:  string): void;
                /**
                 * Removes a style from the Workbook named styles collection which has the specified name.
                 * @param {string} name The name of the style to remove.
                 * @example
                 * var namedStyle = new GC.Spread.Sheets.Style();
                 * namedStyle.name = "style1";
                 * namedStyle.backColor = "green";
                 * spread.addNamedStyle(namedStyle);
                 * activeSheet.setStyleName(1, 1, "style1"); // cell(1,1)'s backColor is  green.
                 * activeSheet.setStyleName(2, 1, "style1");
                 * var style = spread.getNamedStyle("style1");
                 * style.foreColor = "red";    // the namedStyle's foreColor is red.
                 * activeSheet.repaint(); // the foreColor of the cell(1,1) and cell(2,1) is red.
                 * activeSheet.getCell(1,1).value("test");
                 * $("#button1").click(function () {
                 *      spread.removeNamedStyle("style1");
                 * });
                 */
                removeNamedStyle(name:  string): void;
                /**
                 * Removes the specified sheet.
                 * @param {number} index The index of the sheet to remove.
                 * @example
                 * //This example removes a sheet from the spreadsheet.
                 * spread.setSheetCount(5);
                 * spread.removeSheet(0);
                 */
                removeSheet(index:  number): void;
                /**
                 * Removes a SparklineEx from the SparklineEx collection.
                 * @param {string} name The name of the SparklineEx to remove.
                 */
                removeSparklineEx(name:  string): void;
                /**
                 * Repaints the Workbook control.
                 * @example
                 * //This example updates the layout.
                 * spread.invalidateLayout();
                 * spread.repaint();
                 */
                repaint(): void;
                /**
                 * Resumes the calculation service.
                 * @param {boolean} recalcAll Specifies whether to recalculate all formulas.
                 * @example
                 * //This example uses the resumeCalcService method.
                 * spread.suspendCalcService(false);
                 * activeSheet.setValue(0,0,1);
                 * activeSheet.setValue(0,1,2);
                 * activeSheet.setValue(0,2,10);
                 * activeSheet.getCell(1,1).formula("=SUM(A1:C1)");
                 * spread.resumeCalcService(true);
                 */
                resumeCalcService(ignoreDirty:  boolean): void;
                /**
                 * Resumes the event.
                 * @example
                 * //This example suspends and resumes the event.
                 *  activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (sender, args) {
                 *     if (args.propertyName === "value") {
                 *         alert(activeSheet.getValue(args.row, args.col));
                 *     }
                 * });
                 * spread.suspendEvent();
                 * activeSheet.setValue(0, 0, "111");
                 * spread.resumeEvent();
                 * activeSheet.setValue(1, 1, "222");
                 */
                resumeEvent(): void;
                /**
                 * Resumes the paint of active sheet and tab strip.
                 */
                resumePaint(): void;
                /**
                 *Exports the specified sheet to PDF.
                 *@param {function} successCallback Call this function after successfully export. function (blob) {}.
                 *@param {function} errorCallback Call this function if an error occurs. The exception parameter object structure { errorCode: GC.Spread.Sheets.PDF.ErrorCode, errorMessage: string}.
                 *@param {Object} [options] The options for export PDF.
                 *@param {string} [options.creator] The name of the application (for example, Adobe FrameMaker\xae) that created the original document from which it was converted.
                 *@param {string} [options.title] The document\u2019s title.
                 *@param {string} [options.author] The name of the person who created the document.
                 *@param {string} [options.keywords] Keywords associated with the document.
                 *@param {string} [options.subject] The subject of the document.
                 *@param {number} sheetIndex The sheet index. If the sheet index is ignored, exports all visible sheets.
                 */
                savePDF(successCallback:  Function,  errorCallback:  Function,  options?:  Object,  sheetIndex?:  number): void;
                /**
                 * Searches the text in the cells in the specified sheet for the specified string with the specified criteria.
                 * @param {GC.Spread.Sheets.Search.SearchCondition} searchCondition The search conditions.
                 * @returns {GC.Spread.Sheets.Search.SearchResult} The search result.
                 * @example
                 * //This example searches the active sheet using the specified search condition.
                 * activeSheet.getCell(2,3).value("testSearch");
                 * var searchCondition = new GC.Spread.Sheets.Search.SearchCondition();
                 * searchCondition.searchString = "testSearch";
                 * searchCondition.startSheetIndex = spread.getActiveSheetIndex();
                 * searchCondition.endSheetIndex = spread.getActiveSheetIndex();
                 * searchCondition.searchOrder = GC.Spread.Sheets.Search.SearchOrder.nOrder;
                 * searchCondition.searchTarget = GC.Spread.Sheets.Search.SearchFoundFlags.cellText;
                 * searchCondition.searchFlags = GC.Spread.Sheets.Search.SearchFlags.ignoreCase| GC.Spread.Sheets.Search.SearchFlags.useWildCards;
                 * var searchresult= spread.search(searchCondition);
                 * var str ="[searchFoundFlag:"+ searchresult.searchFoundFlag+",\r\n foundSheetIndex:"+searchresult.foundSheetIndex+",foundRowIndex:" +
                 * searchresult.foundRowIndex+", foundColumnIndex:"+searchresult.foundColumnIndex+", foundString:"+searchresult.foundSheetIndex+"]";
                 * alert(str);
                 */
                search(searchCondition:  GC.Spread.Sheets.Search.SearchCondition): GC.Spread.Sheets.Search.SearchResult;
                /**
                 * Sets the active sheet by name.
                 * @param {string} name The name of the sheet to make the active sheet.
                 * @example
                 * //This example sets the active sheet.
                 * spread.setSheetCount(3);
                 * spread.setActiveSheet("Sheet2");
                 */
                setActiveSheet(name:  string): void;
                /**
                 * Sets the active sheet index for the control.
                 * @param {number} value The active sheet index.
                 * @example
                 * //This example uses the setActiveSheetIndex method.
                 * spread.setActiveSheetIndex(1);
                 */
                setActiveSheetIndex(value:  number): void;
                /**
                 * Sets the number of sheets.
                 * @param {number} count The number of sheets.
                 * @example
                 * spread.setSheetCount(5);
                 */
                setSheetCount(count:  number): void;
                /**
                 * Gets or sets the index of the first sheet to display in the spreadsheet.
                 * @param {number} value The index of the first sheet to display in the spreadsheet.
                 * @returns {number|GC.Spread.Sheets.Workbook} If no value is set, returns the index of the first sheet displayed in the spreadsheet; otherwise, returns the spreadsheet.
                 */
                startSheetIndex(value?:  number): any;
                /**
                 * Suspends the calculation service.
                 * @param {boolean} ignoreDirty Specifies whether to invalidate the dependency cells.
                 * @example
                 * //This example uses the suspendCalcService method.
                 * spread.suspendCalcService(false);
                 * activeSheet.setValue(0,0,1);
                 * activeSheet.setValue(0,1,2);
                 * activeSheet.setValue(0,2,10);
                 * activeSheet.getCell(1,1).formula("=SUM(A1:C1)");
                 * spread.resumeCalcService(true);
                 */
                suspendCalcService(ignoreDirty:  boolean): void;
                /**
                 * Suspends the event.
                 * @example
                 * //This example suspends and resumes the event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (sender, args) {
                 *     if (args.propertyName === "value") {
                 *         alert(activeSheet.getValue(args.row, args.col));
                 *     }
                 * });
                 * spread.suspendEvent();
                 * activeSheet.setValue(0, 0, "111");
                 * spread.resumeEvent();
                 * activeSheet.setValue(1, 1, "222");
                 */
                suspendEvent(): void;
                /**
                 * Suspends the paint of active sheet and tab strip.
                 */
                suspendPaint(): void;
                /**
                 * Saves the object state to a JSON string.
                 * @param {Object} serializationOption - The serialization options.
                 * @param {boolean} [serializationOption.includeBindingSource] - Whether to include the binding source when converting the workbook to json.
                 * @param {boolean} [serializationOption.ignoreStyle] - Whether to ignore the style when converting the workbook to json.
                 * @param {boolean} [serializationOption.ignoreFormula] - Whether to ignore the formula when converting the workbook to json.
                 * @param {boolean} [serializationOption.saveAsView] - Whether to ignore the format string when converting the workbook to json.
                 * @param {boolean} [serializationOption.rowHeadersAsFrozenColumns] - Whether to treat the row headers as frozen columns when converting the workbook to json.
                 * @param {boolean} [serializationOption.columnHeadersAsFrozenRows] - Whether to treat the column headers as frozen rows when converting the workbook to json.
                 * @param {boolean} [serializationOption.includeAutoMergedCells] - Whether to include the automatically merged cells when converting the workbook to json.
                 * @returns {Object} The spreadsheet data.
                 * @example
                 * activeSheet.getCell(0,0).value(123);
                 * var jsonStr = null;
                 * //export
                 * jsonStr = JSON.stringify(spread.toJSON(false));
                 * //import
                 * spread.fromJSON(JSON.parse(jsonStr));
                 * alert(jsonStr);
                 */
                toJSON(serializationOption?:  Object): Object;
                /**
                 * Removes the binding of an event to Workbook.
                 * @param {string} type The event type.
                 * @param {Function} fn Specifies the function to run when the event occurs.
                 * @example
                 * //This example removes the event binding.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (sender, args) {
                 *     if (args.propertyName === "value") {
                 *         alert(activeSheet.getValue(args.row, args.col));
                 *     }
                 * });
                 * activeSheet.setValue(0, 0, "111");
                 * spread.unbind(GC.Spread.Sheets.Events.CellChanged);
                 * //spread.unbindAll(); //cancel monitoring of all events.
                 * activeSheet.setValue(1, 0, "222");
                 * activeSheet.setValue(2, 0, "333");
                 * activeSheet.setValue(3, 0, "444");
                 */
                unbind(type:  string,  fn?:  Function): void;
                /**
                 * Removes the binding of all events to Workbook.
                 * @example
                 * //This example removes the event binding. Uncomment the unbindAll method to remove all event binding.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (sender, args) {
                 *     if (args.propertyName === "value") {
                 *         alert(activeSheet.getValue(args.row, args.col));
                 *     }
                 * });
                 * activeSheet.setValue(0, 0, "111");
                 * spread.unbind(GC.Spread.Sheets.Events.CellChanged);
                 * //spread.unbindAll(); //cancel monitoring of all events.
                 * activeSheet.setValue(1, 0, "222");
                 * activeSheet.setValue(2, 0, "333");
                 * activeSheet.setValue(3, 0, "444");
                 */
                unbindAll(): void;
                /**
                 * Gets the undo manager.
                 * @returns {GC.Spread.Commands.UndoManager} The undo manager.
                 */
                undoManager(): GC.Spread.Commands.UndoManager;
            }

            export class Worksheet{
                /**
                 * Represents a worksheet.
                 * @class
                 * @param {string} name The name of the Worksheet.
                 */
                constructor(name:  string);
                /**
                 * Indicates whether to generate columns automatically while binding data context.
                 * @type {boolean}
                 * @example
                 * //This example sets a data source for the sheet.
                 *  var test = [
                 *        { "Series0": 2, "Series1": 1 },
                 *        { "Series0": 4, "Series1": 2 },
                 *        { "Series0": 3, "Series1": 4 }
                 *             ];
                 * activeSheet.autoGenerateColumns = true;
                 * activeSheet.setDataSource(test, true);
                 */
                autoGenerateColumns: boolean;
                /** The cellState manager.
                 * @type {GC.Spread.Sheets.CellState.CellStateManager}
                 */
                cellStates: GC.Spread.Sheets.CellState.CellStateManager;
                /**
                 * Chart manager for the sheet.
                 * @type {GC.Spread.Sheets.Charts.ChartCollection}
                 * @example
                 * //This example shows how to add a chart.
                 * var dataRange = "A1:D4";
                 * var chart = activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 400, dataRange);
                 */
                charts: GC.Spread.Sheets.Charts.ChartCollection;
                /**
                 * Indicates the column range group.
                 * @type {GC.Spread.Sheets.Outlines.Outline}
                 */
                columnOutlines: GC.Spread.Sheets.Outlines.Outline;
                /**
                 * Comment manager for the sheet.
                 * @type {GC.Spread.Sheets.Comments.CommentManager}
                 */
                comments: GC.Spread.Sheets.Comments.CommentManager;
                /**
                 * Conditional format manager for the sheet.
                 * @type {GC.Spread.Sheets.ConditionalFormatting.ConditionalFormats}
                 * @example
                 * //This example creates a rule.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                 * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                 * rule.ranges([new GC.Spread.Sheets.Range(0,0,5,1)]);
                 * rule.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between);
                 * rule.style(style);
                 * rule.value1(2);
                 * rule.value2(100);
                 * activeSheet.conditionalFormats.addRule(rule);
                 * activeSheet.setValue(0,0,1,3);
                 * activeSheet.setValue(1,0,45,3);
                 * var ruletest = activeSheet.conditionalFormats.getRules();
                 * alert(ruletest[0].style().backColor);
                 */
                conditionalFormats: ConditionalFormatting.ConditionalFormats;
                /**
                 * Indicates the default row height and column width of the sheet.
                 * @type {Object}
                 * @example
                 * //This example sets the default row height and column width.
                 * activeSheet.suspendPaint();
                 * activeSheet.defaults.rowHeight = 40;
                 * activeSheet.defaults.colWidth = 30;
                 * activeSheet.resumePaint();
                 */
                defaults: GC.Spread.Sheets.ISheetDefaultOption;
                /**
                 * FloatingObject manager for the sheet.
                 * @type {GC.Spread.Sheets.FloatingObjects.FloatingObjectCollection}
                 * @example
                 * //This example creates a floating object.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1", 10, 10, 60, 64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 */
                floatingObjects: GC.Spread.Sheets.FloatingObjects.FloatingObjectCollection;
                /**
                 * Indicates the options of the sheet.
                 * @type {Object}
                 * @property {boolean} allowCellOverflow - Indicates whether data can overflow into adjacent empty cells.
                 * @property {boolean} showFormulas - Indicates whether display the formulas string not the formula result.
                 * @property {string} sheetTabColor - A color string used to represent the sheet tab color, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                 * @property {string} frozenlineColor - A color string used to represent the frozen line color, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                 * @property {GC.Spread.Sheets.ClipboardPasteOptions} clipBoardOptions - The clipboard option.
                 * @property {Object} gridline - The grid line's options.
                 * @property {string} gridline.color - The grid line color
                 * @property {boolean} gridline.showVerticalGridline - Whether to show the vertical grid line.
                 * @property {boolean} gridline.showHorizontalGridline - Whether to show the horizontal grid line.
                 * @property {boolean} rowHeaderVisible - Indicates whether the row header is visible.
                 * @property {boolean} colHeaderVisible - Indicates whether the column header is visible.
                 * @property {GC.Spread.Sheets.HeaderAutoText} rowHeaderAutoText - Indicates whether the row header displays letters or numbers or is blank.
                 * @property {GC.Spread.Sheets.HeaderAutoText} colHeaderAutoText - Indicates whether the column header displays letters or numbers or is blank.
                 * @property {number} rowHeaderAutoTextIndex - Specifies which row header column displays the automatic text when there are multiple row header columns.
                 * @property {number} colHeaderAutoTextIndex - Specifies which column header row displays the automatic text when there are multiple column header rows.
                 * @property {boolean} isProtected - Indicates whether cells on this sheet that are marked as protected cannot be edited.
                 * @property {Object} protectionOptions - A value that indicates the elements that you want users to be able to change.
                 * @property {boolean} [protectionOptions.allowSelectLockedCells] - True or undefined if the user can select locked cells.
                 * @property {boolean} [protectionOptions.allowSelectUnlockedCells] - True or undefined if the user can select unlocked cells.
                 * @property {boolean} [protectionOptions.allowSort] - True if the user can sort ranges.
                 * @property {boolean} [protectionOptions.allowFilter] - True if the user can filter ranges.
                 * @property {boolean} [protectionOptions.allowEditObjects] - True if the user can edit floating objects.
                 * @property {boolean} [protectionOptions.allowResizeRows] - True if the user can resize rows.
                 * @property {boolean} [protectionOptions.allowResizeColumns] - True if the user can resize columns.
                 * @property {boolean} [protectionOptions.allowDragInsertRows] - True if the user can drag to insert rows.
                 * @property {boolean} [protectionOptions.allowDragInsertColumns] - True if the user can drag to insert columns.
                 * @property {boolean} [protectionOptions.allowInsertRows] - True if the user can insert rows.
                 * @property {boolean} [protectionOptions.allowInsertColumns] - True if the user can insert columns.
                 * @property {boolean} [protectionOptions.allowDeleteRows] - True if the user can delete rows.
                 * @property {boolean} [protectionOptions.allowDeleteColumns] - True if the user can delete columns.
                 * @property {string} selectionBackColor - The selection's background color for the sheet.
                 * @property {string} selectionBorderColor -  The selection's border color for the sheet.
                 * @property {Object} sheetAreaOffset - The sheetAreaOffset's options.
                 * @property {number} sheetAreaOffset.left - The offset left of sheet from host.
                 * @property {number} sheetAreaOffset.top - The offset top of sheet from host.
                 * @example
                 * sheet.setRowCount(2,GC.Spread.Sheets.SheetArea.colHeader);
                 * sheet.setColumnCount(2,GC.Spread.Sheets.SheetArea.rowHeader);
                 * sheet.setValue(0, 2,"Column",GC.Spread.Sheets.SheetArea.colHeader);
                 * sheet.options.colHeaderAutoTextIndex = 1;
                 * sheet.options.colHeaderAutoText = GC.Spread.Sheets.HeaderAutoText.numbers;
                 */
                options: GC.Spread.Sheets.IWorksheetOptions;
                /**
                 * Gets the outline column for the sheet.
                 * @returns {GC.Spread.Sheets.OutlineColumn.OutlineColumn}
                 */
                outlineColumn: GC.Spread.Sheets.OutlineColumn.OutlineColumn;
                /**
                 * Picture manager for the sheet.
                 * @type {GC.Spread.Sheets.FloatingObjects.FloatingObjectCollection}
                 * @example
                 * //This example adds a picture.
                 * activeSheet.pictures.add("f2","Event.png",2,2,10,10);
                 * var picture = activeSheet.pictures.get("f2");
                 * picture.pictureStretch(GC.Spread.Sheets.ImageLayout.center);
                 * picture.backColor("Blue");
                 * picture.borderWidth(2);
                 * picture.borderColor("Red");
                 * picture.borderStyle("dotted");
                 * picture.borderRadius(5);
                 */
                pictures: GC.Spread.Sheets.FloatingObjects.FloatingObjectCollection;
                /**
                 * Indicates the row range group.
                 * @type {GC.Spread.Sheets.Outlines.Outline}
                 */
                rowOutlines: GC.Spread.Sheets.Outlines.Outline;
                /**
                 * Shape manager for the sheet.
                 * @type {GC.Spread.Sheets.Shapes.ShapeCollection}
                 * @example
                 * //This example shows how to add a shape.
                 * var shape = activeSheet.shapes.add("shape1", GC.Spread.Sheets.Shapes.AutoShapeType.heart, 100, 50, 100, 150);
                 */
                shapes: GC.Spread.Sheets.Shapes.ShapeCollection;
                /** The slicer manager.
                 * @type {GC.Spread.Sheets.Slicers.SlicerCollection}
                 * @example
                 * //This example adds a slicer.
                 * //create a table
                 * datas = [
                 *     ["1", "NewYork", "1968/6/8", "80", "180"],
                 *     ["4", "NewYork", "1972/7/3", "72", "168"],
                 *     ["4", "NewYork", "1964/3/2", "71", "179"],
                 *     ["5", "Washington", "1972/8/8","80", "171"],
                 *     ["6", "Washington", "1986/2/2", "89", "161"],
                 *     ["7", "Washington", "2012/2/15", "71", "240"]];
                 * var table = activeSheet.tables.addFromDataSource("table1", 2, 2, datas);
                 * dataColumns = ["Name", "City", "Birthday", "Weight", "Height"];
                 * table.setColumnName(0, dataColumns[0]);
                 * table.setColumnName(1, dataColumns[1]);
                 * table.setColumnName(2, dataColumns[2]);
                 * table.setColumnName(3, dataColumns[3]);
                 * table.setColumnName(4, dataColumns[4]);
                 * var style1 = new GC.Spread.Sheets.Slicers.SlicerStyles.light4();
                 * //add a slicer to the sheet and return the slicer instance.
                 * var slicer = activeSheet.slicers.add("slicer1",table.name(),"Height");
                 *  //change the slicer properties.
                 * slicer.position(new GC.Spread.Sheets.Point(100, 200));
                 * slicer.disableResizingAndMoving(true);
                 * slicer.style(style1);
                 */
                slicers: GC.Spread.Sheets.Slicers.SlicerCollection;
                /** The table manager.
                 * @type {GC.Spread.Sheets.Tables.TableManager}
                 * @example
                 * //This example creates a table.
                 * activeSheet.tables.add("Table1", 0, 0, 3, 3, GC.Spread.Sheets.Tables.TableThemes.dark1);
                 * activeSheet.getCell(0,0).text("Name");
                 * activeSheet.getCell(0,1).text("Value");
                 * activeSheet.getCell(0,2).text("T/F");
                 * activeSheet.getCell(1,0).text("AW");
                 * activeSheet.getCell(1,1).text("5");
                 * activeSheet.getCell(1,2).text("T");
                 */
                tables: GC.Spread.Sheets.Tables.TableManager;
                /**
                 * Adds the column or columns to the data model at the specified index.
                 * @param {number} col Column index at which to add the new columns.
                 * @param {number} count The number of columns to add.
                 * @example
                 * //This example adds columns.
                 * sheet.setValue(0, 0, "value");
                 * sheet.addRows(0, 2);
                 * sheet.addColumns(0, 2);
                 * sheet.setRowHeight(0, 50.0,GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.setColumnWidth(0, 150.0,GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.getRange(0, -1, 1, -1,GC.Spread.Sheets.SheetArea.viewport).backColor("Gray");
                 * sheet.getRange(-1, 0, -1, 1,GC.Spread.Sheets.SheetArea.viewport).backColor ("Brown");
                 */
                addColumns(col:  number,  count:  number): void;
                /**
                 * Adds a custom function.
                 * @param {GC.Spread.CalcEngine.Functions.Function} fn The function to add.
                 */
                addCustomFunction(fn:  GC.Spread.CalcEngine.Functions.Function): void;
                /**
                 * Adds a custom name.
                 * @param {string} name The custom name.
                 * @param {string} formula The formula.
                 * @param {number} baseRow The row index.
                 * @param {number} baseCol The column index.
                 * @param {string} comment The comment.
                 * @example
                 * //This example creates custom names.
                 * sheet.setValue(0, 0, 1);
                 * sheet.setValue(0, 1, 2);
                 * sheet.setValue(0, 2, 3);
                 * sheet.addCustomName("customName1","=12", 0, 0);
                 * sheet.addCustomName("customName2","Average(20,45)", 0, 0);
                 * sheet.addCustomName("customName3", "=$A$1:$C$1", 0, 0);
                 * sheet.setFormula(1, 0, "customName1");
                 * sheet.setFormula(1, 1, "customName2");
                 * sheet.setFormula(1, 2, "sum(customName3)");
                 */
                addCustomName(name:  string,  formula:  string,  baseRow:  number,  baseCol:  number,  comment:  string): void;
                /**
                 * Adds a style to the Worksheet named styles collection.
                 * @param {GC.Spread.Sheets.Style} style The style to be added.
                 * @example
                 * var namedStyle = new GC.Spread.Sheets.Style();
                 * namedStyle.name = "style1";
                 * namedStyle.backColor = "green";
                 * activeSheet.addNamedStyle(namedStyle);
                 * activeSheet.setStyleName(1, 1, "style1"); // cell(1,1)'s backColor is  green.
                 * activeSheet.setStyleName(2, 1, "style1");
                 * var style = activeSheet.getNamedStyle("style1");
                 * style.foreColor = "red";    // the namedStyle's foreColor is red.
                 * activeSheet.repaint(); // the foreColor of the cell(1,1) and cell(2,1) is red.
                 * activeSheet.getCell(1,1).value("test");
                 * $("#button1").click(function () {
                 * activeSheet.removeNamedStyle("style1");
                 *     });
                 */
                addNamedStyle(style:  GC.Spread.Sheets.Style): void;
                /**
                 * Adds rows in this worksheet.
                 * @param {number} row The index of the starting row.
                 * @param {number} count The number of rows to add.
                 * @example
                 * //This example adds rows.
                 * sheet.setValue(0, 0, "value");
                 * sheet.addRows(0, 2);
                 * sheet.addColumns(0, 2);
                 * sheet.setRowHeight(0, 50.0,GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.setColumnWidth(0, 150.0,GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.getRange(0, -1, 1, -1,GC.Spread.Sheets.SheetArea.viewport).backColor("Gray");
                 * sheet.getRange(-1, 0, -1, 1,GC.Spread.Sheets.SheetArea.viewport).backColor ("Brown");
                 */
                addRows(row:  number,  count:  number): void;
                /**
                 * Adds a cell or cells to the selection.
                 * @param {number} row The row index of the first cell to add.
                 * @param {number} column The column index of the first cell to add.
                 * @param {number} rowCount The number of rows to add.
                 * @param {number} columnCount The number of columns to add.
                 * @example
                 * //This example adds a selection and uses the selection in a rule.
                 * sheet.setValue(0,0, 1,3);
                 * sheet.setValue(1,0, 50,3);
                 * sheet.setValue(2,0, 100,3);
                 * sheet.setValue(3,0, 2,3);
                 * sheet.setValue(4,0, 60,3);
                 * sheet.setValue(5,0, 90,3);
                 * sheet.setValue(6,0, 3,3);
                 * sheet.setValue(7,0, 40,3);
                 * sheet.setValue(8,0, 70,3);
                 * sheet.setValue(9,0, 5,3);
                 * sheet.setValue(10,0, 35,3);
                 * sheet.addSelection(0,0,11,1);
                 * sheet.conditionalFormats.add3ScaleRule(1, 10, "red", 0, 50, "blue",2, 100, "yellow", sheet.getSelections());
                 */
                addSelection(row:  number,  column:  number,  rowCount:  number,  columnCount:  number): void;
                /**
                 * Adds a span of cells to this sheet in the specified sheet area.
                 * @param {number} row The row index of the cell at which to start the span.
                 * @param {number} column The column index of the cell at which to start the span.
                 * @param {number} rowCount The number of rows to span.
                 * @param {number} colCount The number of columns to span.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @example
                 * //This example creates cell spans.
                 * sheet.setRowCount(4,1);
                 * sheet.setColumnCount(4,2);
                 * sheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.colHeader);
                 * sheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.rowHeader);
                 * sheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.viewport);
                 */
                addSpan(row:  number,  col:  number,  rowCount:  number,  colCount:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Automatically fits the viewport column.
                 * @param {number} column The column index.
                 * @example
                 * //This example sets the column width based on the text.
                 * activeSheet.setValue(0, 1, "testing");
                 * activeSheet.autoFitColumn(1);
                 */
                autoFitColumn(column:  number): void;
                /**
                 * Automatically fits the viewport row.
                 * @param {number} row The row index.
                 * @example
                 * //This example sets the row height based on the text.
                 * activeSheet.setValue(0, 1, "testing\r\nmultiple\r\nlines");
                 * activeSheet.getCell(0,1).wordWrap(true);
                 * activeSheet.autoFitRow(0);
                 */
                autoFitRow(row:  number): void;
                /**
                 * Applies auto merge for a range.
                 * @param {GC.Spread.Sheets.Range} range The auto merge range.
                 * @param {GC.Spread.Sheets.AutoMerge.AutoMergeDirection} direction The auto merge direction. If this parameter is not provided, it defaults to <b>column</b>. Specially, if the direction is <b>none</b>, the auto merge for the range will be canceled.
                 * @param {GC.Spread.Sheets.AutoMerge.AutoMergeMode} mode The auto merge mode. If this parameter is not provided, it defaults to <b>free</b>.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area of the auto merge range. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @returns {Array} If no parameter is provided, returns all auto merge range infos of current worksheet. Each range info contains range, direction, mode, sheetArea.
                 * @example
                 * var range = new GC.Spread.Sheets.Range(-1, 0, -1, 1);
                 * sheet.autoMerge(range);
                 */
                autoMerge(range:  GC.Spread.Sheets.Range,  direction?:  GC.Spread.Sheets.AutoMerge.AutoMergeDirection,  mode?:  GC.Spread.Sheets.AutoMerge.AutoMergeMode,  sheetArea?:  GC.Spread.Sheets.SheetArea): GC.Spread.Sheets.AutoMerge.IRangeInfo[];
                /**
                 * Binds an event to the sheet.
                 * @param {string} type The event type.
                 * @param {Object} data Optional. Specifies additional data to pass along to the function.
                 * @param {Function} fn Specifies the function to run when the event occurs.
                 * @example
                 * //This example binds events.
                 * sheet.bind(GC.Spread.Sheets.Events.LeftColumnChanged,function(event,data)
                 *   {
                 *     var str = "----------------------------------------\n";
                 *     var title = "Event [LeftColumnChanged ] Fired";
                 *     str = str.substr(0, 4) + title + str.substr(4 + title.length);
                 *     if (typeof data == "object") {
                 *         for (var key in data) {
                 *             str += key + " : " + data[key] + "\n";
                 *           }
                 *       } else {
                 *         str += data + "\n";
                 *       }
                 *       alert(str);
                 *   });
                 * sheet.bind(GC.Spread.Sheets.Events.TopRowChanged,function(event,data)
                 *   {
                 *         var str = "----------------------------------------\n";
                 *         var title = "Event [TopRowChanged] Fired";
                 *         str = str.substr(0, 4) + title + str.substr(4 + title.length);
                 *         if (typeof data == "object") {
                 *             for (var key in data) {
                 *                 str += key + " : " + data[key] + "\n";
                 *             }
                 *         } else {
                 *             str += data + "\n";
                 *         }
                 *         alert(str);
                 *     });
                 */
                bind(type:  string,  data?:  any,  fn?:  Function): void;
                /**
                 * Binds the column using the specified data field.
                 * @param {number} index The column index.
                 * @param {string|Object} column Column information with data field. If its type is string, it is regarded as name.
                 * @example
                 * var test = [
                 * {"Series0":2,"Series1":1},
                 * {"Series0":4,"Series1":2},
                 * {"Series0":3,"Series1":4}
                 * ];
                 * sheet.setDataSource(test);
                 * sheet.bindColumn(1,"Series0");
                 * sheet.bindColumn(0,"Series1");
                 */
                bindColumn(index:  number,  column:  string | GC.Spread.Sheets.IColumn): void;
                /**
                 * Binds the columns using the specified data fields.
                 * @param {Array} columns The array of column information with data fields. If an item's type is string, the item is regarded as name.
                 * @example
                 * var datasource = [
                 *                  { name: "Alice", age: 27, birthday: "1985/08/31", position: "Beijing", isMarried: false},
                 *                  { name: "Aimee", age: 28, birthday: "1984/07/31", position: "Xi'An", isMarried: true},
                 *                  { name: "Charles", age: 29, birthday: "1983/03/31", position: "ShangHai", isMarried: true},
                 *             ];
                 * var colInfos = [
                 *                 { name: "name", displayName: "Name", size: 70, pageBread: false},
                 *                 { name: "age", displayName: "Age", size: 40, resizable: false },
                 *                 { name: "birthday", displayName: "Birthday", formatter: "d/M/yy", size: 120 },
                 *                 { name: "position", displayName: "Position", size: 50, visible: true, value: function (item){
                 *			             return 'China ' + item['position'];
                 *		           }},
                 *                 { name: "isMarried", displayName: "IsMarried", size: 50, visible: true, cellType: new GC.Spread.Sheets.CellTypes.CheckBox()}
                 *             ];
                 * activeSheet.autoGenerateColumns = true;
                 * activeSheet.setDataSource(datasource);
                 * activeSheet.bindColumns(colInfos);
                 */
                bindColumns(columns:  GC.Spread.Sheets.IColumn[]): void;
                /**
                 * Clears the specified area.
                 * @param {number} row The start row index.
                 * @param {number} column The start column index.
                 * @param {number} rowCount The number of rows to clear.
                 * @param {number} columnCount The number of columns to clear.
                 * @param {GC.Spread.Sheets.SheetArea} area The area to clear.
                 * @param {GC.Spread.Sheets.StorageType} storageType The clear type.
                 * @example
                 * //This example clears the data from the specified range.
                 * activeSheet.getCell(0,0).value("A1");
                 * activeSheet.clear(0,0,3,3,GC.Spread.Sheets.SheetArea.viewport,GC.Spread.Sheets.StorageType.data);
                 */
                clear(row:  number,  column:  number,  rowCount:  number,  colCount:  number,  area:  GC.Spread.Sheets.SheetArea,  storageType:  GC.Spread.Sheets.StorageType): void;
                /**
                 * Clears all custom functions.
                 * @example
                 * //This example clears the custom functions from the active sheet.
                 * activeSheet.clearCustomFunctions();
                 */
                clearCustomFunctions(): void;
                /**
                 * Clears custom names.
                 * @example
                 * //This example creates custom names and then clears them.
                 * activeSheet.setValue(0, 0, 1);
                 * activeSheet.setValue(0, 1, 2);
                 * activeSheet.setValue(0, 2, 3);
                 * activeSheet.addCustomName("customName1","=12", 0, 0);
                 * activeSheet.addCustomName("customName2","Average(20,45)", 0, 0);
                 * activeSheet.addCustomName("customName3", "=$A$1:$C$1", 0, 0);
                 * activeSheet.setFormula(1, 0, "customName1");
                 * activeSheet.setFormula(1, 1, "customName2");
                 * activeSheet.setFormula(1, 2, "sum(customName3)");
                 * activeSheet.clearCustomNames();
                 */
                clearCustomNames(): void;
                /**
                 * Clears the dirty, insert, and delete status from the current worksheet.
                 */
                clearPendingChanges(): void;
                /**
                 * Clears the selection.
                 * @example
                 * //This example clears the selection.
                 * sheet.addSelection(4, 0, 2, 2);
                 * sheet.clearSelection();
                 */
                clearSelection(): void;
                /**
                 * Copies data from one range to another.
                 * @param {number} fromRow The source row.
                 * @param {number} fromColumn The source column.
                 * @param {number} toRow The target row.
                 * @param {number} toColumn The target column.
                 * @param {number} rowCount The row count.
                 * @param {number} columnCount The column count.
                 * @param {GC.Spread.Sheets.CopyToOptions} option The copy option.
                 * @example
                 * //This example copies data to the specified location.
                 * activeSheet.getCell(0,0).value("1");
                 * activeSheet.copyTo(0,0,1,1,2,2,GC.Spread.Sheets.CopyToOptions.value);
                 */
                copyTo(fromRow:  number,  fromColumn:  number,  toRow:  number,  toColumn:  number,  rowCount:  number,  columnCount:  number,  option:  GC.Spread.Sheets.CopyToOptions): void;
                /**
                 * Gets or sets the current theme for the sheet.
                 * @param {string|GC.Spread.Common.Theme} value The theme name or the theme.
                 * @returns {GC.Spread.Sheets.Theme|GC.Spread.Sheets.Worksheet} If no value is set, returns the current theme; otherwise, returns the worksheet.
                 * @example
                 * //This example sets a theme.
                 * sheet.currentTheme("Civic");
                 */
                currentTheme(value?:  string | GC.Spread.Sheets.Theme): any;
                /**
                 * Deletes the columns in this sheet at the specified index.
                 * @param {number} col The index of the first column to delete.
                 * @param {number} count The number of columns to delete.
                 * @example
                 * activeSheet.getCell(0,0).value("A1");
                 * activeSheet.getCell(0,4).value("Test")
                 * activeSheet.deleteColumns(0,2);
                 * activeSheet.deleteRows(3,1);
                 */
                deleteColumns(col:  number,  count:  number): void;
                /**
                 * Deletes the rows in this worksheet at the specified index.
                 * @param {number} row The index of the first row to delete.
                 * @param {number} count The number of rows to delete.
                 * @example
                 * activeSheet.getCell(0,0).value("A1");
                 * activeSheet.getCell(0,4).value("Test")
                 * activeSheet.deleteColumns(0,2);
                 * activeSheet.deleteRows(3,1);
                 */
                deleteRows(row:  number,  count:  number): void;
                /**
                 * Returns the editor's status.
                 * @returns {GC.Spread.Sheets.EditorStatus} The editor status.
                 */
                editorStatus(): GC.Spread.Sheets.EditorStatus;
                /**
                 * Stops editing the active cell.
                 * @param {boolean} ignoreValueChange If set to <c>true</c>, does not apply the edited text to the cell.
                 * @returns {boolean} <c>true</c> when able to stop cell editing successfully; otherwise, <c>false</c>.
                 * @example
                 * //This example removes the text "123" when typing in a cell.
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditChange, function (sender,args) {
                 *                 if (args.editingText === "123") {
                 *                     activeSheet.endEdit(true);
                 *                }
                 *             });
                 */
                endEdit(ignoreValueChange?:  boolean): boolean;
                /**
                 * Fills the specified range automatically.
                 * @param {GC.Spread.Sheets.Range} startRange The fill start range.
                 * @param {GC.Spread.Sheets.Range} wholeRange The entire range to fill.
                 * @param {Object} options The range fill information.
                 * @param {GC.Spread.Sheets.Fill.FillType} [options.fillType] - Specifies how to fill the specified range.
                 *      GC.Spread.Sheets.Fill.FillType.direction:<br />
                 *              Fills the specified range in the specified direction.<br />
                 *      GC.Spread.Sheets.Fill.FillType.linear:<br />
                 *              Fills the specified range using a linear trend when the source value type is number.<br />
                 *              The next value is generated by the step and stop values.<br />
                 *              The next value is computed by adding the step value to the current cell value.<br />
                 *      GC.Spread.Sheets.Fill.FillType.growth:<br />
                 *              Fills the specified range using a growth trend when the source value type is number.<br />
                 *              The next value is generated by the step and stop values.<br />
                 *              The next value is computed by multiplying the step value with the current cell.<br />
                 *      GC.Spread.Sheets.Fill.FillType.date:<br />
                 *              Fills the specified range when the source value type is date.<br />
                 *              The next value is generated by adding the step value to the current value.<br />
                 *              The step value is affected by the fill date unit.<br />
                 *      GC.Spread.Sheets.Fill.FillType.auto:<br />
                 *              Fills the specified range automatically.<br />
                 *              When the value is a string, the value is copied to other cells.<br />
                 *              When the value is a number, the new value is generated by the TREND formula.<br />
                 * @param {GC.Spread.Sheets.Fill.FillSeries} [options.series] - The fill series.
                 * @param {GC.Spread.Sheets.Fill.FillDirection} [options.direction] - direction The fill direction.
                 * @param {number} [options.step] step - The fill step value.
                 * @param {number|Date} [options.stop] stop - The fill stop value.
                 * @param {GC.Spread.Sheets.Fill.FillDateUnit} [options.unit] - unit The fill date unit.
                 * @example
                 * activeSheet.setValue(0, 0, 5);
                 * var start = new GC.Spread.Sheets.Range(0, 0, 1, 1);
                 * var r3 = new GC.Spread.Sheets.Range(0, 0, 4, 1);
                 * activeSheet.fillAuto(start,r3, {fillType:GC.Spread.Sheets.Fill.FillType.auto, series:GC.Spread.Sheets.Fill.FillSeries.column, direction:GC.Spread.Sheets.Fill.FillDirection.down});
                 */
                fillAuto(startRange:  GC.Spread.Sheets.Range,  wholeRange:  GC.Spread.Sheets.Range,  options:  GC.Spread.Sheets.Fill.IFillOptions): void;
                /**
                 * Loads the object state from the specified JSON string.
                 * @param {Object} sheetSettings The sheet data from deserialization.
                 * @example
                 * //This example uses the fromJSON method.
                 * activeSheet.getCell(0,0).value(123);
                 * var jsonStr = null;
                 * //export
                 * jsonStr = JSON.stringify(activeSheet.toJSON());
                 * //import
                 * activeSheet.fromJSON(JSON.parse(jsonStr));
                 * alert(jsonStr);
                 */
                fromJSON(sheetSettings:  Object): void;
                /**
                 * Gets or sets the number of frozen columns of the sheet.
                 * @param {number} [colCount] The number of columns to freeze.
                 * @returns {number|GC.Spread.Sheets.Worksheet} If no value is set, returns the number of frozen columns; otherwise, returns the worksheet.
                 * @example
                 * sheet.frozenColumnCount(1);
                 */
                frozenColumnCount(colCount?:  number): any;
                /**
                 * Gets or sets the number of frozen rows of the sheet.
                 * @param {number} [rowCount] The number of rows to freeze.
                 * @returns {number|GC.Spread.Sheets.Worksheet} If no value is set, returns the number of frozen rows; otherwise, returns the worksheet.
                 * @example
                 * sheet.frozenRowCount(1);
                 */
                frozenRowCount(rowCount?:  number): any;
                /**
                 * Gets or sets the number of trailing frozen columns of the sheet.
                 * @param {number} [colCount] The number of columns to freeze at the right side of the sheet.
                 * @returns {number|GC.Spread.Sheets.Worksheet} If no value is set, returns the number of trailing frozen columns; otherwise, returns the worksheet.
                 * @example
                 * sheet.frozenTrailingColumnCount(1);
                 */
                frozenTrailingColumnCount(colCount?:  number): any;
                /**
                 * Gets or sets the number of trailing frozen rows of the sheet.
                 * @param {number} [rowCount] The number of rows to freeze at the bottom of the sheet.
                 * @returns {number|GC.Spread.Sheets.Worksheet} If no value is set, returns the number of trailing frozen rows; otherwise, returns the worksheet.
                 * @example
                 * sheet.frozenTrailingRowCount(1);
                 */
                frozenTrailingRowCount(rowCount?:  number): any;
                /**
                 * Gets the active column index for this sheet.
                 * @returns {number} The column index of the active cell.
                 * @example
                 * //This example gets the active column.
                 * sheet.setActiveCell(5,5);
                 * alert(sheet.getActiveColumnIndex());
                 * alert(sheet.getActiveRowIndex());
                 * spread.bind(GC.Spread.Sheets.Events.EnterCell, function (event, data) {
                 *     alert(data.col);
                 *     alert(data.row);
                 *   });
                 * spread.bind(GC.Spread.Sheets.Events.LeaveCell, function (event, data) {
                 *     alert(data.col);
                 *     alert(data.row);
                 *   });
                 */
                getActiveColumnIndex(): number;
                /**
                 * Gets the active row index for this sheet.
                 * @returns {number} The row index of the active cell.
                 * @example
                 * //This example gets the active row.
                 * sheet.setActiveCell(5,5);
                 * alert(sheet.getActiveColumnIndex());
                 * alert(sheet.getActiveRowIndex());
                 * spread.bind(GC.Spread.Sheets.Events.EnterCell, function (event, data) {
                 *     alert(data.col);
                 *     alert(data.row);
                 *   });
                 * spread.bind(GC.Spread.Sheets.Events.LeaveCell, function (event, data) {
                 *     alert(data.col);
                 *     alert(data.row);
                 *   });
                 */
                getActiveRowIndex(): number;
                /**
                 * Gets the actual style information for a specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} column The column index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @param {boolean} [sheetStyleOnly] If <c>true</c>, the row filter and the conditional format style are not applied to the return style;
                 * otherwise, the return style only contains the cell's inherited style.
                 * @returns {GC.Spread.Sheets.Style} Returns the cell style of the specified cell.
                 * @example
                 * //This example uses the getActualStyle method.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * var cstyle = activeSheet.getActualStyle(1,1,GC.Spread.Sheets.SheetArea.viewport, true);
                 * alert(cstyle.backColor);
                 */
                getActualStyle(row:  number,  column:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea,  sheetStyleOnly?:  boolean): GC.Spread.Sheets.Style;
                /**
                 * Gets an object array from a specified range of cells.
                 * @param {number} row The row index.
                 * @param {number} column The column index.
                 * @param {number} rowCount The row count.
                 * @param {number} colCount The column count.
                 * @param {boolean} getFormula If <c>true</c>, return formulas; otherwise, return values.
                 * @returns {Array.<Array.<Object>>} The object array from the specified range of cells.
                 * @example
                 * //This example uses the getArray method.
                 * //set value
                 * var array = [[1,2,3],[4,5],[6,7,8,9]];
                 * activeSheet.setArray(1, 2, array);
                 * //set formula
                 * var array = [["=1+1","=2+2","=3+3"],["=4+4","=5+5"],["=6+6","=7+7","=8+8","=9+9"]];
                 * activeSheet.setArray(1, 2, array, true);
                 * //get value
                 * var newArray = activeSheet.getArray(1, 2, 3, 4);
                 * //getformula
                 * var newArray = activeSheet.getArray(1, 2, 3, 4, true);
                 * //alert(newArray[0]);
                 */
                getArray(row:  number,  column:  number,  rowCount:  number,  columnCount:  number,  getFormula?:  boolean): any[];
                /**
                 * Gets the binding path of cell-level binding from the specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {string} Returns the binding path of the cell for cell-level binding.
                 * @example
                 * //This example uses the getBindingPath method.
                 * var person = {name: "Wang feng", age: 25, address: {postcode: "710075"}};
                 * var source = new GC.Spread.Sheets.Bindings.CellBindingSource(person);
                 * activeSheet.setBindingPath(0, 0, "name");
                 * activeSheet.setBindingPath(1, 1, "age");
                 * activeSheet.setBindingPath(3, 3, "address.postcode");
                 * activeSheet.setDataSource(source);
                 * alert(activeSheet.getBindingPath(0, 0, GC.Spread.Sheets.SheetArea.viewport));
                 */
                getBindingPath(row:  number,  col:  number): string;
                /**
                 * Gets the specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @returns {GC.Spread.Sheets.CellRange} The cell.
                 * @example
                 * //This example gets the cell.
                 * activeSheet.getCell(1,1).text("cell object");
                 */
                getCell(row:  number,  col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): GC.Spread.Sheets.CellRange;
                /**
                 * Gets the rectangle of the cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {number} [rowViewportIndex] Index of the row of the viewport: -1 represents column header area, 0 represents frozen row area, 1 represents viewport area, 2 represents trailing frozen row area.
                 * @param {number} [colViewportIndex] Index of the column of the viewport: -1 represents row header area, 0 represents frozen column area, 1 represents viewport area, 2 represents trailing frozen column area.
                 * @returns {GC.Spread.Sheets.Rect} Object that contains the size and location of the cell rectangle.
                 * @example
                 * //This example uses the getCellRect method.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellClick, function (e, info) {
                 *                if (info.sheetArea === GC.Spread.Sheets.SheetArea.viewport) {
                 *                    alert("Clicked cell index (" + info.row + "," + info.col + ")");
                 *                    //Acquire the coordinate information of regular cells which exist at the specified index position
                 *                    var cellRect = activeSheet.getCellRect(info.row, info.col);
                 *                    alert("X coordinate:" + cellRect.x);
                 *                    alert("Y coordinate:" + cellRect.y);
                 *                    alert("Cell width:" + cellRect.width);
                 *                    alert("Cell height:" + cellRect.height);
                 *                }
                 *            });
                 */
                getCellRect(row:  number,  col:  number,  rowViewportIndex?:  number,  colViewportIndex?:  number): GC.Spread.Sheets.Rect;
                /**
                 * Gets the cell type.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @returns {GC.Spread.Sheets.CellTypes.Base} Returns the cell type for the specified cell.
                 * @example
                 * //This example gets the cell type.
                 * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                 * cellType.buttonBackColor("#FFFF00");
                 * cellType.text("this is a button");
                 * activeSheet.getCell(0, 2).cellType(cellType);
                 * var cellType = activeSheet.getCellType(0,2,GC.Spread.Sheets.SheetArea.viewport)
                 * if (cellType instanceof GC.Spread.Sheets.CellTypes.Button) {
                 *       alert("This is a ButtonCellType");
                 * }
                 */
                getCellType(row:  number,  col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): GC.Spread.Sheets.CellTypes.Base;
                /**
                 * Gets the column count in the specified sheet area.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @returns {number} The number of columns.
                 * @example
                 * //This example gets the number of columns.
                 * var count = activeSheet.getColumnCount(GC.Spread.Sheets.SheetArea.viewport);
                 * alert(count);
                 */
                getColumnCount(sheetArea?:  GC.Spread.Sheets.SheetArea): number;
                /**
                 * Gets whether a forced page break is inserted before the specified column on this sheet when printing.
                 * @param {number} column The column index.
                 * @returns {boolean} <c>true</c> if a forced page break is inserted before the specified column; otherwise, <c>false</c>.
                 */
                getColumnPageBreak(column:  number): boolean;
                /**
                 * Gets a value that indicates whether the user can resize a specified column in the specified sheet area.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @returns {boolean} <c>true</c> if the user can resize the specified column; otherwise, <c>false</c>.
                 * @example
                 * //This example gets whether the column is resizable.
                 * sheet.setRowCount(10);
                 * sheet.setColumnCount(7);
                 * sheet.setValue(0, 0,"Western");
                 * sheet.setValue(0, 1,"Western");
                 * sheet.setValue(0, 2,"Western");
                 * sheet.setValue(1, 0,"A");
                 * sheet.setValue(1, 1,"B");
                 * sheet.setValue(1, 2,"C");
                 * sheet.setColumnResizable(0,true, GC.Spread.Sheets.SheetArea.colHeader);
                 * sheet.setRowResizable(0,true, GC.Spread.Sheets.SheetArea.rowHeader);
                 * alert( sheet.getColumnResizable(0));
                 * alert( sheet.getRowResizable(0, GC.Spread.Sheets.SheetArea.rowHeader));
                 */
                getColumnResizable(col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): boolean;
                /**
                 * Gets whether a column in the specified sheet area is displayed.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @returns {boolean} <c>true</c> if the column is visible in the sheet area; otherwise, <c>false</c>.
                 * @example
                 * //This example returns the visible and width settings for a column.
                 * var visible = activeSheet.getColumnVisible(1, GC.Spread.Sheets.SheetArea.viewport);
                 * var width = activeSheet.getColumnWidth(1, GC.Spread.Sheets.SheetArea.viewport);
                 * alert(visible);
                 * alert(width);
                 */
                getColumnVisible(col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): boolean;
                /**
                 * Gets the width in pixels or the dynamic size for the specified column in the specified sheet area.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to viewport.
                 * @param {boolean} [getDynamicSize] Whether get the dynamic size. If not given, it defaults to false. If this parameter is true, and dynamic size is not set, will return undefined.
                 * @returns {number | string} The column width in pixels or the dynamic size.
                 * @example
                 * //This example returns the visible and width settings for a column.
                 * var visible = activeSheet.getColumnVisible(1, GC.Spread.Sheets.SheetArea.viewport);
                 * var width = activeSheet.getColumnWidth(1, GC.Spread.Sheets.SheetArea.viewport);
                 * alert(visible);
                 * alert(width);
                 */
                getColumnWidth(col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea,  getDynamicSize?:  boolean): any;
                /**
                 * Gets delimited text from a range.
                 * @param {number} row The start row.
                 * @param {number} column The start column.
                 * @param {number} rowCount The row count.
                 * @param {number} columnCount The column count.
                 * @param {string} rowDelimiter The row delimiter that is appended to the end of the row.
                 * @param {string} columnDelimiter The column delimiter that is appended to the end of the column.
                 * @returns {string} The text from the range with the specified delimiters.
                 */
                getCsv(row:  number,  column:  number,  rowCount:  number,  columnCount:  number,  rowDelimiter:  string,  columnDelimiter:  string): string;
                /**
                 * Gets a custom function.
                 * @param {string} fnName The custom function name.
                 * @returns {GC.Spread.CalcEngine.Functions.Function} The custom function.
                 */
                getCustomFunction(name:  string): void;
                /**
                 * Gets the specified custom name information.
                 * @param {string} fnName The custom name.
                 * @returns {GC.Spread.Sheets.NameInfo} The information for the specified custom name.
                 * @example
                 * //This example gets the custom name and formula.
                 * activeSheet.setValue(0, 0, 1);
                 * activeSheet.setValue(0, 1, 2);
                 * activeSheet.setValue(0, 2, 3);
                 * activeSheet.addCustomName("customName1", "=12", 0, 0);
                 * activeSheet.addCustomName("customName2", "Average(20,45)", 0, 0);
                 * activeSheet.addCustomName("customName3", "=$A$1:$C$1");
                 * activeSheet.setFormula(1, 0, "customName1");
                 * activeSheet.setFormula(1, 1, "customName2");
                 * activeSheet.setFormula(1, 2, "sum(customName3)");
                 * $("#button1").click(function () {
                 *     var cname = activeSheet.getCustomName("customName2");
                 *     if (cname instanceof GC.Spread.Sheets.NameInfo) {
                 *         //get CustomName
                 *         var name = cname.getName();
                 *         //get Expression
                 *         var expression = cname.getExpression();
                 *         //get Expression String
                 *         var expStr = GC.Spread.Sheets.CalcEngine.expressionToFormula(activeSheet, expression, 0, 0);
                 *         alert("Name:" + name + ";Expression: =" + expStr);
                 *     }
                 * });
                 */
                getCustomName(name:  string): GC.Spread.Sheets.NameInfo;
                /**
                 * Gets all custom name information.
                 * @returns {Array.<GC.Spread.Sheets.NameInfo>} The type GC.Spread.Sheets.NameInfo stored in an array.
                 */
                getCustomNames(): GC.Spread.Sheets.NameInfo[];
                /**
                 * Gets the column name at the specified position.
                 * @param {number} column The column index for which the name is requested.
                 * @returns {string} The column name for data binding.
                 * @example
                 * //This example returns the name for the specified bound column.
                 * var test = [
                 *         {"Series0":2,"Series1":1},
                 *         {"Series0":4,"Series1":2},
                 *         {"Series0":3,"Series1":4}
                 *     ];
                 * activeSheet.setDataSource(test);
                 * activeSheet.bindColumn(1,"Series0");
                 * activeSheet.bindColumn(0,"Series1");
                 * var colname = activeSheet.getDataColumnName(0);
                 * alert(colname);
                 */
                getDataColumnName(column:  number): string;
                /**
                 * Gets the data item.
                 * @param {number} row The row index.
                 * @returns {Object} The row data.
                 * @example
                 * //This example uses the getDataItem method.
                 * var test = [
                 *       { "Series0": 2, "Series1": 1 },
                 *       { "Series0": 4, "Series1": 2 },
                 *       { "Series0": 3, "Series1": 4 }
                 *            ];
                 *            activeSheet.autoGenerateColumns = true;
                 *            activeSheet.setDataSource(test, false);
                 *            alert(JSON.stringify(activeSheet.getDataItem(0)));
                 */
                getDataItem(row:  number): any;
                /**
                 * Gets the data source that populates the sheet.
                 * @function
                 * @returns {Object} Returns the data source.
                 * @example
                 * var test = [
                 *         {"Series0":2,"Series1":1},
                 *         {"Series0":4,"Series1":2},
                 *         {"Series0":3,"Series1":4}
                 *     ];
                 * activeSheet.setDataSource(test);
                 * alert(activeSheet.getDataSource);
                 */
                getDataSource(): any;
                /**
                 * Gets the cell data validator.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @returns {GC.Spread.Sheets.DataValidation.DefaultDataValidator} Returns the cell data validator for the specified cell.
                 * @example
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createListValidator("1,2,3");
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Value must be 1,2 or 3");
                 * dv.inputTitle("tip");
                 * activeSheet.setDataValidator(1,1,1,1,dv,GC.Spread.Sheets.SheetArea.viewport);
                 * alert(activeSheet.getDataValidator(1,1).getValidList(activeSheet,1,1));
                 */
                getDataValidator(row:  number,  col:  number,  sheetArea:  GC.Spread.Sheets.SheetArea): GC.Spread.Sheets.DataValidation.DefaultDataValidator;
                /**
                 * Gets the default style information for the sheet.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @returns {GC.Spread.Sheets.Style} Returns the sheet's default style.
                 * @example
                 * //This example uses the getDefaultStyle method.
                 * var defaultStyle = new GC.Spread.Sheets.Style();
                 * defaultStyle.backColor = "LemonChiffon";
                 * defaultStyle.foreColor = "Red";
                 * defaultStyle.borderLeft = new GC.Spread.Sheets.LineBorder("Green");
                 * defaultStyle.borderTop = new GC.Spread.Sheets.LineBorder("Green");
                 * defaultStyle.borderRight = new GC.Spread.Sheets.LineBorder("Green");
                 * defaultStyle.borderBottom = new GC.Spread.Sheets.LineBorder("Green");
                 * activeSheet.setDefaultStyle(defaultStyle, GC.Spread.Sheets.SheetArea.viewport);
                 * var cstyle = activeSheet.getDefaultStyle(GC.Spread.Sheets.SheetArea.viewport);
                 * alert(cstyle.backColor);
                 */
                getDefaultStyle(sheetArea?:  GC.Spread.Sheets.SheetArea): GC.Spread.Sheets.Style;
                /**
                 * Gets the deleted row collection.
                 * @return {Array.<Object>} The deleted rows collection. the item in array contains two properties, row.row: specifies deleted row index, row.originalItem: specifies deleted data item.
                 */
                getDeletedRows(): any[];
                /**
                 *  Gets the dependent CellRange information object array of the cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {Array.<Object>} Returns dependent cell information object array
                 * dependentsInfo.row {number} Indicates the cellRange row index.
                 * dependentsInfo.col {number} Indicates the cellRange col index.
                 * dependentsInfo.rowCount {number} Indicates the cellRange row count.
                 * dependentsInfo.colCount {number} Indicates the cellRange colcount.
                 * dependentsInfo.sheetName {string} Indicates the workSheet name.
                 * @example
                 * sheet.getDependents(1, 1);
                 */
                getDependents(row:  number,  col:  number): GC.Spread.Sheets.ICellsInfo[];
                /**
                 * Gets the dirty cell collection.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {number} rowCount The number of rows in the range of dirty cells.
                 * @param {number} colCount The number of columns in the range of dirty cells.
                 * @return {Array} The dirty cells.
                 */
                getDirtyCells(row:  number,  col:  number,  rowCount:  number,  colCount:  number): GC.Spread.Sheets.IDirtyCellInfo[];
                /**
                 * Gets the dirty row collection.
                 * @returns {Array.<Object>} The dirty rows collection, the item in array contains three properties, row.row: specifies row index, row.item: specifies data item of current row, row.originalItem: specifies original data item of the row.
                 */
                getDirtyRows(): any[];
                /**
                 * Gets the cell formatter.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @returns {string|GC.Spread.Formatter.FormatterBase} Returns the cell formatter string or object for the specified cell.
                 * @example
                 * //This example returns the format object for the active sheet.
                 * activeSheet.getCell(0, 1).formatter("M");
                 * activeSheet.setValue(0, 1, new Date(2011, 2, 9));
                 * var style = activeSheet.getFormatter(0,1,GC.Spread.Sheets.SheetArea.viewport);
                 * alert(style);
                 */
                getFormatter(row:  number,  col:  number,  sheetArea:  GC.Spread.Sheets.SheetArea): any;
                /**
                 * Gets the formula in the specified cell in this sheet.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {string} Returns the formula string.
                 * @example
                 * //This example returns the formula in the specified cell.
                 * activeSheet.setValue(0,0,1);
                 * activeSheet.setValue(0,1,2);
                 * activeSheet.setValue(0,2,10);
                 * activeSheet.getCell(1,1).formula("=SUM(A1:C1)");
                 * var formula = activeSheet.getFormula(1,1);
                 * alert(formula);
                 */
                getFormula(row:  number,  col:  number): string;
                /**
                 * Gets the formula detail information in the specified cell in this sheet.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {Object} formulaInfo - Returns the formula information about the cell.
                 * formulaInfo.hasFormula {boolean} Indicates whether there is a formula in the cell.
                 * formulaInfo.isArrayFormula {boolean} Indicates whether the formula is an array formula.
                 * formulaInfo.formula {string} The formula string.
                 * formulaInfo.formulaWithCulture {string} The formula string with culture.
                 * @example
                 * activeSheet.setValue(0,0,1);
                 * activeSheet.setValue(0,1,2);
                 * activeSheet.setValue(0,2,10);
                 * activeSheet.getCell(1,1).formula("=SUM(A1:C1)");
                 * var test = activeSheet.getFormulaInformation(1,1, GC.Spread.Sheets.SheetArea.viewport);
                 * alert(test.formula);
                 */
                getFormulaInformation(row:  number,  col:  number): GC.Spread.Sheets.IFormulaInfo;
                /**
                 * Sets the hyperlink data for the specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * //This example uses the getHyperlink method.
                 * let firstHyperlinkData = sheet.getHyperlink(0, 2, GC.Spread.Sheets.SheetArea.viewport);
                 * let secondHyperlinkData = sheet.setHyperlink(1, 1, GC.Spread.Sheets.SheetArea.viewport);
                 */
                getHyperlink(row:  number,  col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): GC.Spread.Sheets.IHyperlink;
                /**
                 * Gets the inserted row collection.
                 * @returns {Array.<Object>} The inserted rows collection, the item in array contains two properties, row.row: specifies insert row index, row.item: specifies insert data item.
                 */
                getInsertRows(): any[];
                /**
                 * Gets a style from the Worksheet named styles collection which has the specified name.
                 * @param {string} name The name of the style to return.
                 * @returns {GC.Spread.Sheets.Style} Returns the specified named style.
                 */
                getNamedStyle(name:  string): GC.Spread.Sheets.Style;
                /**
                 * Gets named styles from the Worksheet.
                 * @returns {Array.<GC.Spread.Sheets.Style>} The GC.Spread.Sheets.Style array of named styles.
                 */
                getNamedStyles(): GC.Spread.Sheets.Style[];
                /**
                 * Gets the parent Spread object of the current sheet.
                 * @returns {GC.Spread.Sheets.Workbook} Returns the parent Spread object of the current sheet.
                 */
                getParent(): GC.Spread.Sheets.Workbook;
                /**
                 *  Gets the precedent CellRange information object array of the cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {Array.<Object>} Returns precedent cellRange information object array
                 * precedentsInfo.row {number} Indicates the cellRange row index.
                 * precedentsInfo.col {number} Indicates the cellRange col index.
                 * precedentsInfo.rowCount {number} Indicates the cellRange row count.
                 * precedentsInfo.colCount {number} Indicates the cellRange colcount.
                 * precedentsInfo.sheetName {string} Indicates the workSheet name.
                 * @example
                 * sheet.getPrecedents(1, 1);
                 */
                getPrecedents(row:  number,  col:  number): GC.Spread.Sheets.ICellsInfo[];
                /**
                 * Gets a range of cells in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {number} rowCount The row count of the range. If you do not provide this parameter, it defaults to <b>1</b>.
                 * @param {number} colCount The column count of the range. If you do not provide this parameter, it defaults to <b>1</b>.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @returns {GC.Spread.Sheets.CellRange} The cellRange.
                 * If row is -1 and rowCount is -1, the range represents columns. For example, sheet.getRange(-1,4,-1,6) returns the columns "E:J".
                 * If col is -1 and colCount is -1, the range represents rows. For example, sheet.getRange(4,-1,6,-1) returns the rows "5:10".
                 */
                getRange(row:  number,  col:  number,  rowCount?:  number,  colCount?:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): GC.Spread.Sheets.CellRange;
                /**
                 * Gets the row count in the specified sheet area.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @returns {number} The number of rows.
                 * @example
                 * //This example gets the row count.
                 * var count = activeSheet.getRowCount(GC.Spread.Sheets.SheetArea.viewport);
                 * alert(count);
                 */
                getRowCount(sheetArea?:  GC.Spread.Sheets.SheetArea): number;
                /**
                 * Gets the height in pixels or the dynamic size for the specified row in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @param {boolean} [getDynamicSize] Whether get the dynamic size. If not given, it defaults to false. If this parameter is true, and dynamic size is not set, will return undefined.
                 * @returns {number | string} The row height in pixels or the dynamic size.
                 * @example
                 * //This example returns the height for the specified row.
                 * var rheight = activeSheet.getRowHeight(1,GC.Spread.Sheets.SheetArea.viewport);
                 * alert(rheight);
                 */
                getRowHeight(row:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea,  getDynamicSize?:  boolean): any;
                /**
                 * Gets whether a forced page break is inserted before the specified row on this sheet when printing.
                 * @param {number} row The row index.
                 * @returns {boolean} <c>true</c> if a forced page break is inserted before the specified row; otherwise, <c>false</c>.
                 */
                getRowPageBreak(row:  number): boolean;
                /**
                 * Gets a value that indicates whether users can resize the specified row in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @returns {boolean} <c>true</c> if the users can resize the specified row; otherwise, <c>false</c>.
                 * @example
                 * //This example gets whether the row is resizable.
                 * sheet.setRowCount(10);
                 * sheet.setColumnCount(7);
                 * sheet.setValue(0, 0,"Western");
                 * sheet.setValue(0, 1,"Western");
                 * sheet.setValue(0, 2,"Western");
                 * sheet.setValue(1, 0,"A");
                 * sheet.setValue(1, 1,"B");
                 * sheet.setValue(1, 2,"C");
                 * sheet.setColumnResizable(0,true, GC.Spread.Sheets.SheetArea.colHeader);
                 * sheet.setRowResizable(0,true, GC.Spread.Sheets.SheetArea.rowHeader);
                 * alert( sheet.getColumnResizable(0));
                 * alert( sheet.getRowResizable(0, GC.Spread.Sheets.SheetArea.rowHeader));
                 */
                getRowResizable(row:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): boolean;
                /**
                 * Gets whether the control displays the specified row.
                 * @param {number} row The row index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @returns {boolean} <c>true</c> if the row is visible in the sheet area; otherwise, <c>false</c>.
                 * @example
                 * //This example returns the visible setting for the specified row.
                 * rvisible = activeSheet.getRowVisible(1,GC.Spread.Sheets.SheetArea.viewport);
                 * alert(rvisible);
                 */
                getRowVisible(row:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): boolean;
                /**
                 * Gets the selections in the current sheet.
                 * @returns {Array.<GC.Spread.Sheets.Range>} The type GC.Spread.Sheets.Range is stored in an Array.
                 */
                getSelections(): GC.Spread.Sheets.Range[];
                /**
                 * Gets the spans in the specified range in the specified sheet area.
                 * @param {GC.Spread.Sheets.Range} range The cell range.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @returns {Array.<GC.Spread.Sheets.Range>} An array that contains span information whose item type is GC.Spread.Sheets.Range.
                 */
                getSpans(range?:  GC.Spread.Sheets.Range,  sheetArea?:  GC.Spread.Sheets.SheetArea): GC.Spread.Sheets.Range[];
                /**
                 *  Gets the sparkline for the specified cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {GC.Spread.Sheets.Sparkline} The sparkline for the cell.
                 * @example
                 * //This example creates and gets a sparkline.
                 * var cellr = new GC.Spread.Sheets.Range(0, 0, 1, 5);
                 * var ex = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * ex.options.SeriesColor  = "Aquamarine";
                 * sheet.setValue(0, 0, 2);
                 * sheet.setValue(0, 1, 5);
                 * sheet.setValue(0, 2, 4);
                 * sheet.setValue(0, 3, -1);
                 * sheet.setValue(0, 4, 3);
                 * sheet.setSparkline(0, 5, cellr, GC.Spread.Sheets.Sparklines.DataOrientation.horizontal, GC.Spread.Sheets.Sparklines.SparklineType.column, ex);
                 * alert(sheet.getSparkline(0, 5).toString());
                 * //sheet.removeSparkline(0, 5);
                 */
                getSparkline(row:  number,  column:  number): GC.Spread.Sheets.Sparklines.Sparkline;
                /**
                 * Gets the style information for a specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} column The column index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @returns {GC.Spread.Sheets.Style} Returns the cell style of the specified cell.
                 * @example
                 * //This example uses the getStyle method.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 * var cstyle = activeSheet.getStyle(1,1,GC.Spread.Sheets.SheetArea.viewport, true);
                 * alert(cstyle.backColor);
                 */
                getStyle(row:  number,  column:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): GC.Spread.Sheets.Style;
                /**
                 * Gets the name of the style for a specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} column The column index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @returns {string} Returns the name string for the style.
                 * @example
                 * //This example uses the getStyleName method.
                 * var namedStyle = new GC.Spread.Sheets.Style();
                 * namedStyle.name = "style1";
                 * namedStyle.backColor = "green";
                 * activeSheet.addNamedStyle(namedStyle);
                 * activeSheet.setStyleName(1, 1, "style1"); // cell(1,1)'s backColor is  green.
                 * activeSheet.setStyleName(2, 1, "style1");
                 * alert(activeSheet.getStyleName(1,1,GC.Spread.Sheets.SheetArea.viewport));
                 */
                getStyleName(row:  number,  column:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): string;
                /**
                 * Gets the tag value from the specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @returns {Object} Returns the tag value of the cell.
                 * @example
                 * //This example adds and gets a cell tag.
                 * activeSheet.getRange(1, -1, 1, -1).tag("row tag");
                 * alert(activeSheet.getTag(1,-1,GC.Spread.Sheets.SheetArea.viewport));
                 */
                getTag(row:  number,  col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): any;
                /**
                 * Gets the formatted text in the cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @returns {string} Returns the formatted text of the cell.
                 * @example
                 * activeSheet.getText(1, 0);
                 */
                getText(row:  number,  col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): string;
                /**
                 * Gets the unformatted data from the specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @param {GC.Spread.Sheets.ValueType} [valueType] - Indicate the return value type is normal text or rich text, default is normal text.
                 * @returns {Object} Returns the value of the cell.
                 * @example
                 * activeSheet.getValue(1, 1, GC.Spread.Sheets.SheetArea.viewport, GC.Spread.Sheets.ValueType.richText);
                 */
                getValue(row:  number,  col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea,  valueType?:  GC.Spread.Sheets.ValueType): any;
                /**
                 * Gets the index of the bottom row in the viewport.
                 * @param {number} rowViewportIndex The index of the viewport.
                 * @returns {number} The index of the bottom row in the viewport.
                 * @example
                 * //This example returns the indices for the bottom and top rows and left and right columns in the current sheet view.
                 * var brow = activeSheet.getViewportBottomRow(1);
                 * var lcol = activeSheet.getViewportLeftColumn(1);
                 * var rcol = activeSheet.getViewportRightColumn(1);
                 * var trow = activeSheet.getViewportTopRow(1);
                 * alert(brow);
                 * alert(lcol);
                 * alert(rcol);
                 * alert(trow);
                 */
                getViewportBottomRow(rowViewportIndex:  number): number;
                /**
                 * Gets the height of the specified viewport row for the active sheet.
                 * @param {number} rowViewportIndex The index of the row viewport.
                 * @returns {number} The height of the viewport.
                 * @example
                 * //This example uses the getViewportHeight method.
                 * alert(activeSheet.getViewportHeight(1));
                 */
                getViewportHeight(rowViewportIndex:  number): number;
                /**
                 * Gets the index of the left column in the viewport.
                 * @param {number} columnViewportIndex The index of the viewport.
                 * @returns {number} The index of the left column in the viewport.
                 * @example
                 * //This example returns the indices for the bottom and top rows and left and right columns in the current sheet view.
                 * var brow = activeSheet.getViewportBottomRow(1);
                 * var lcol = activeSheet.getViewportLeftColumn(1);
                 * var rcol = activeSheet.getViewportRightColumn(1);
                 * var trow = activeSheet.getViewportTopRow(1);
                 * alert(brow);
                 * alert(lcol);
                 * alert(rcol);
                 * alert(trow);
                 */
                getViewportLeftColumn(columnViewportIndex:  number): number;
                /**
                 * Gets the index of the right column in the viewport.
                 * @param {number} columnViewportIndex The index of the viewport.
                 * @returns {number} The index of the right column in the viewport.
                 * @example
                 * //This example returns the indices for the bottom and top rows and left and right columns in the current sheet view.
                 * var brow = activeSheet.getViewportBottomRow(1);
                 * var lcol = activeSheet.getViewportLeftColumn(1);
                 * var rcol = activeSheet.getViewportRightColumn(1);
                 * var trow = activeSheet.getViewportTopRow(1);
                 * alert(brow);
                 * alert(lcol);
                 * alert(rcol);
                 * alert(trow);
                 */
                getViewportRightColumn(columnViewportIndex:  number): number;
                /**
                 * Gets the index of the top row in the viewport.
                 * @param {number} rowViewportIndex The index of the viewport.
                 * @returns {number} The index of the top row in the viewport.
                 * @example
                 * //This example returns the indices for the bottom and top rows and left and right columns in the current sheet view.
                 * var brow = activeSheet.getViewportBottomRow(1);
                 * var lcol = activeSheet.getViewportLeftColumn(1);
                 * var rcol = activeSheet.getViewportRightColumn(1);
                 * var trow = activeSheet.getViewportTopRow(1);
                 * alert(brow);
                 * alert(lcol);
                 * alert(rcol);
                 * alert(trow);
                 */
                getViewportTopRow(rowViewportIndex:  number): number;
                /**
                 * Gets the width of the specified viewport column for the active sheet.
                 * @param {number} columnViewportIndex The index of the column viewport.
                 * @returns {number} The width of the viewport
                 * @example
                 * //This example uses the getViewportWidth method.
                 * alert(activeSheet.getViewportWidth(1));
                 */
                getViewportWidth(columnViewportIndex:  number): number;
                /**
                 * Groups the sparklines.
                 * @param {Array.<GC.Spread.Sheets.Sparklines.Sparkline>} sparklines The sparklines to group.
                 * @returns {GC.Spread.Sheets.Sparklines.SparklineGroup} The sparkline group.
                 * @example
                 * //This example groups a sparkline.
                 * sheet.setValue(0, 0, "Data Range is A2-A9");
                 * sheet.setValue(1, 0, 1);
                 * sheet.setValue(2, 0, -2);
                 * sheet.setValue(3, 0, -1);
                 * sheet.setValue(4, 0, 6);
                 * sheet.setValue(5, 0, 4);
                 * sheet.setValue(6, 0, -4);
                 * sheet.setValue(7, 0, 3);
                 * sheet.setValue(8, 0, 8);
                 * var data = new GC.Spread.Sheets.Range(1, 0, 8, 1);
                 * var setting = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * var s1=  sheet.setSparkline(13, 0, data
                 *         , GC.Spread.Sheets.Sparklines.DataOrientation.vertical
                 *         , GC.Spread.Sheets.Sparklines.SparklineType.line
                 *         , setting
                 *         );
                 * var s2 =sheet.setSparkline(13, 3, data
                 *         , GC.Spread.Sheets.Sparklines.DataOrientation.vertical
                 *         , GC.Spread.Sheets.Sparklines.SparklineType.column
                 *         , setting
                 *         );
                 * var s3=  sheet.setSparkline(13, 6, data
                 *         , GC.Spread.Sheets.Sparklines.DataOrientation.vertical
                 *         , GC.Spread.Sheets.Sparklines.SparklineType.winloss
                 *         , setting
                 *         );
                 * var group = sheet.groupSparkline([s1,s2,s3]);
                 */
                groupSparkline(sparklines:  Sparklines.Sparkline[]): GC.Spread.Sheets.Sparklines.SparklineGroup;
                /**
                 * Gets whether there is a dirty, insert, or delete status for the specified range.
                 * @returns {boolean} <c>true</c> if any of the rows or cells in the range are dirty, or have been inserted or deleted; otherwise, <c>false</c>.
                 */
                hasPendingChanges(): boolean;
                /**
                 * Performs a hit test.
                 * @param {number} x The <i>x</i>-coordinate.
                 * @param {number} y The <i>y</i>-coordinate.
                 * @returns {Object} The hit test information.
                 */
                hitTest(x:  number,  y:  number): GC.Spread.Sheets.IHitTestInformation;
                /**
                 * Invalidates the sheet layout.
                 * @example
                 * //This example updates the layout.
                 * activeSheet.columnOutlines.group(0, 1);
                 * activeSheet.invalidateLayout();
                 * activeSheet.repaint();
                 */
                invalidateLayout(): void;
                /**
                 * Gets whether recording the dirty data is suspended.
                 * @returns {boolean} Whether the dirty data is suspended.
                 * @example
                 * //This example uses the isDirtySuspended method.
                 * var customers = [
                 *                { ID: 0, Name: 'A', Info1: 'Info0' },
                 *                { ID: 1, Name: 'B', Info1: 'Info1' },
                 *                { ID: 2, Name: 'C', Info1: 'Info2' },
                 *             ];
                 * activeSheet.setDataSource(customers);
                 * activeSheet.suspendDirty();
                 * alert(activeSheet.isDirtySuspended());
                 * activeSheet.resumeDirty();
                 * alert(activeSheet.isDirtySuspended());
                 */
                isDirtySuspended(): boolean;
                /**
                 * Gets whether the sheet is in edit mode.
                 * @returns {boolean} <c>true</c> if the sheet is in edit mode; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the isEditing method.
                 * alert(activeSheet.isEditing());
                 */
                isEditing(): boolean;
                /**
                 * Get if sheet paint is suspended.
                 */
                isPaintSuspended(): boolean;
                /**
                 * Gets or sets whether display a print line for the sheet.
                 * @param {boolean} [value] Whether display a print line for the sheet
                 * @returns {boolean} If no value is set, returns a value indicating whether the print line is displayed
                 */
                isPrintLineVisible(value?:  boolean): boolean;
                /**
                 * Gets or sets the selected state of the worksheet.
                 * @param {boolean} [selectedState] The selected state of the worksheet.
                 * @returns {boolean|GC.Spread.Sheets.Worksheet} If no selectedState is set, returns the worksheet selected state; otherwise, returns the worksheet.
                 * @example
                 * spread.sheets[0].isSelected();
                 * spread.sheets[1].isSelected(true);
                 */
                isSelected(selectedState?:  boolean): any;
                /**
                 * Determines whether the cell value is valid.
                 * @param {number} row The row index.
                 * @param {number} column The column index.
                 * @param {Object} value The cell value.
                 * @returns {boolean} <c>true</c> if the value is valid; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the isValid method.
                 * alert(activeSheet.isValid(0, 0, 10));
                 */
                isValid(row:  number,  column:  number,  value:  Object): boolean;
                /**
                 * Moves data from one range to another.
                 * @param {number} fromRow The source row.
                 * @param {number} fromColumn The source column.
                 * @param {number} toRow The target row.
                 * @param {number} toColumn The target column.
                 * @param {number} rowCount The row count.
                 * @param {number} columnCount The column count.
                 * @param {GC.Spread.Sheets.CopyToOptions} option The copy option.
                 * @example
                 * //This example moves the data to the specified location.
                 * activeSheet.getCell(0,0).value("A1");
                 * activeSheet.getCell(1,1).value("Test")
                 * activeSheet.moveTo(0,0,3,3,2,2,GC.Spread.Sheets.CopyToOptions.value);
                 */
                moveTo(fromRow:  number,  fromColumn:  number,  toRow:  number,  toColumn:  number,  rowCount:  number,  columnCount:  number,  option:  GC.Spread.Sheets.CopyToOptions): void;
                /**
                 * Gets or sets the name of the worksheet.
                 * @param {string} [value] The name of the worksheet.
                 * @returns {string|GC.Spread.Sheets.Worksheet} If no value is set, returns the worksheet name; otherwise, returns the worksheet.
                 * @example
                 * spread.sheets[0].name("The first sheet");
                 * spread.sheets[1].name( "The second sheet");
                 */
                name(value?:  string): any;
                /**
                 * Gets or sets the print information for the sheet.
                 * @param {GC.Spread.Sheets.Print.PrintInfo} [value] The print information for the sheet.
                 * @returns {GC.Spread.Sheets.Print.PrintInfo | GC.Spread.Sheets.Worksheet} If no value is set, returns the print information for the sheet; otherwise, returns the sheet.
                 */
                printInfo(value?:  GC.Spread.Sheets.Print.PrintInfo): any;
                /**
                 * Recalculates all the formulas in the sheet.
                 * @param {boolean} refreshAll Specifies whether to rebuild all fromula reference, custom name and custom functions.
                 * @example
                 * //This example uses the recalcAll method.
                 * activeSheet.setValue(0,0,1);
                 * activeSheet.setValue(0,1,2);
                 * activeSheet.setValue(0,2,10);
                 * activeSheet.getCell(1,1).formula("=SUM(A1:C1)");
                 * activeSheet.recalcAll();
                 */
                recalcAll(refreshAll?:  boolean): void;
                /**
                 * Removes a custom function.
                 * @param {string} fnName The custom function name.
                 * @example
                 * //This example uses the removeCustomFunction method.
                 * // Add Custom function
                 * // Type =myfunc(1)
                 * // in a cell to see the result
                 * function myfunc() {}
                 * myfunc.prototype = new GC.Spread.CalcEngine.Functions.Function("myfunc", 0, 0, {name: "myfunc",description: "This is my first function"});
                 * myfunc.prototype.evaluate = function (args) {
                 *     return 100;
                 * }
                 * spread.addCustomFunction(new myfunc());
                 * //spread.removeCustomFunction("myfunc");
                 */
                removeCustomFunction(name:  string): void;
                /**
                 * Removes the specified custom name.
                 * @param {string} fnName The custom name.
                 * @example
                 * //This example uses the removeCustomName method.
                 * activeSheet.setValue(0, 0, 1);
                 * activeSheet.setValue(0, 1, 2);
                 * activeSheet.setValue(0, 2, 3);
                 * activeSheet.addCustomName("customName1","=12", 0, 0);
                 * activeSheet.addCustomName("customName2","Average(20,45)", 0, 0);
                 * activeSheet.addCustomName("customName3", "=$A$1:$C$1", 0, 0);
                 * activeSheet.setFormula(1, 0, "customName1");
                 * activeSheet.setFormula(1, 1, "customName2");
                 * activeSheet.setFormula(1, 2, "sum(customName3)");
                 * //activeSheet.removeCustomName("customName3");
                 */
                removeCustomName(name:  string): void;
                /**
                 * Removes a style from the Worksheet named styles collection which has the specified name.
                 * @param {string} name The name of the style to remove.
                 */
                removeNamedStyle(name:  string): void;
                /**
                 * Removes the span that contains a specified anchor cell in the specified sheet area.
                 * @param {number} row The row index of the anchor cell for the span (at which spanned cells start).
                 * @param {number} col The column index of the anchor cell for the span (at which spanned cells start).
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @example
                 * activeSheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.viewport);
                 * //activeSheet.removeSpan(0, 0, GC.Spread.Sheets.SheetArea.viewport);
                 */
                removeSpan(row:  number,  col:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Removes the sparkline for the specified cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @example
                 * //This example removes a sparkline.
                 * var cellr = new GC.Spread.Sheets.Range(0, 0, 1, 5);
                 * var ex = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * ex.options.SeriesColor  = "Aquamarine";
                 * sheet.setValue(0, 0, 2);
                 * sheet.setValue(0, 1, 5);
                 * sheet.setValue(0, 2, 4);
                 * sheet.setValue(0, 3, -1);
                 * sheet.setValue(0, 4, 3);
                 * sheet.setSparkline(0, 5, cellr, GC.Spread.Sheets.Sparklines.DataOrientation.horizontal, GC.Spread.Sheets.Sparklines.SparklineType.column, ex);
                 * alert(sheet.getSparkline(0, 5).toString());
                 * //sheet.removeSparkline(0, 5);
                 */
                removeSparkline(row:  number,  col:  number): void;
                /**
                 * Repaints the specified rectangle.
                 * @param {GC.Spread.Sheets.Rect} clipRect The rectangle to repaint.
                 * @example
                 * //This example causes a repaint.
                 * var cellrange =new GC.Spread.Sheets.Range(0, 0, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellrange);
                 * sheet.rowFilter(hideRowFilter);
                 * sheet.resumePaint();
                 * sheet.repaint();
                 */
                repaint(clipRect?:  GC.Spread.Sheets.Rect): void;
                /**
                 * Resets the sheet.
                 * @example
                 * //This example uses the reset method.
                 * activeSheet.reset();
                 */
                reset(): void;
                /**
                 * Resumes the calculation service.
                 * @param {boolean} recalcAll Specifies whether to recalculate all formulas.
                 * @example
                 * //This example uses the resumeCalcService method.
                 * activeSheet.suspendCalcService(false);
                 * activeSheet.setValue(0,0,1);
                 * activeSheet.setValue(0,1,2);
                 * activeSheet.setValue(0,2,10);
                 * activeSheet.getCell(1,1).formula("=SUM(A1:C1)");
                 * activeSheet.resumeCalcService(true);
                 */
                resumeCalcService(recalcAll?:  boolean): void;
                /**
                 * Resumes recording the dirty data.
                 * @example
                 * //This example uses the resumeDirty method.
                 *  var customers = [
                 *                { ID: 0, Name: 'A', Info1: 'Info0' },
                 *                { ID: 1, Name: 'B', Info1: 'Info1' },
                 *                { ID: 2, Name: 'C', Info1: 'Info2' },
                 *             ];
                 * activeSheet.setDataSource(customers);
                 * activeSheet.suspendDirty();
                 * alert(activeSheet.isDirtySuspended());
                 * activeSheet.resumeDirty();
                 * alert(activeSheet.isDirtySuspended());
                 */
                resumeDirty(): void;
                /**
                 * Resumes the event.
                 * @example
                 * //This example suspends and resumes the event.
                 *  activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (sender, args) {
                 *     if (args.propertyName === "value") {
                 *         alert(activeSheet.getValue(args.row, args.col));
                 *     }
                 * });
                 * activeSheet.suspendEvent();
                 * activeSheet.setValue(0, 0, "111");
                 * activeSheet.resumeEvent();
                 * activeSheet.setValue(1, 1, "222");
                 */
                resumeEvent(): void;
                /**
                 * Resumes the paint.
                 */
                resumePaint(): void;
                /**
                 * Gets or sets the row filter for the sheet.
                 * @param {GC.Spread.Sheets.Filter.RowFilterBase} value The row filter for the sheet.
                 * @returns {GC.Spread.Sheets.Filter.RowFilterBase} The row filter for the sheet.
                 * @example
                 * //This example creates a row filter.
                 * sheet.rowFilter(new GC.Spread.Sheets.Filter.HideRowFilter(new GC.Spread.Sheets.Range(1,1,10,3)));
                 */
                rowFilter(value?:  GC.Spread.Sheets.Filter.RowFilterBase): GC.Spread.Sheets.Filter.RowFilterBase;
                /**
                 * Scrolls the sheet by specified pixels.
                 * When vPixels is positive, worksheet will scroll down; when vPixels is negative, worksheet will scroll up; when vPixels is 0, worksheet won't scroll in vertical direction.
                 * When hPixels is positive, worksheet will scroll right; when hPixels is negative, worksheet will scroll left; when hPixels is 0, worksheet won't scroll in horizontal direction.
                 * When Workbook's option scrollByPixel is true, worksheet will scroll to new top row/left column index and new top row/left column offset;
                 * When Workbook's option scrollByPixel is false, worksheet will scroll to new top row/left column index, and new top row/left column offset will be always 0.
                 * @param {number} vPixels The pixels to scroll in vertical direction.
                 * @param {number} hPixels The pixels to scroll in horizontal direction.
                 * @example
                 * //This example scrolls down the sheet 10 pixels and scrolls right the sheet 5 pixels.
                 * activeSheet.scroll(10, 5);
                 */
                scroll(vPixels:  number,  hPixels:  number): void;
                /**
                 * Searches the specified content.
                 * @param {GC.Spread.Sheets.Search.SearchCondition} searchCondition The search condition.
                 * @returns {GC.Spread.Sheets.Search.SearchResult} The search result.
                 * @example
                 * //This example uses the search method.
                 * activeSheet.getCell(2,3).value("testSearch");
                 * var searchCondition = new GC.Spread.Sheets.Search.SearchCondition();
                 * searchCondition.searchString = "testSearch";
                 * searchCondition.startSheetIndex = spread.getActiveSheetIndex();
                 * searchCondition.endSheetIndex = spread.getActiveSheetIndex();
                 * searchCondition.searchOrder = GC.Spread.Sheets.Search.SearchOrder.nOrder;
                 * searchCondition.searchTarget = GC.Spread.Sheets.Search.SearchFoundFlags.cellText;
                 * searchCondition.searchFlags = GC.Spread.Sheets.Search.SearchFlags.ignoreCase| GC.Spread.Sheets.Search.SearchFlags.useWildCards;
                 * var searchresult= activeSheet.search(searchCondition);
                 * var str ="[searchFoundFlag:"+ searchresult.searchFoundFlag+",\r\n foundSheetIndex:"+searchresult.foundSheetIndex+",foundRowIndex:" +
                 * searchresult.foundRowIndex+", foundColumnIndex:"+searchresult.foundColumnIndex+", foundString:"+searchresult.foundSheetIndex+"]";
                 * alert(str);
                 */
                search(searchCondition:  GC.Spread.Sheets.Search.SearchCondition): GC.Spread.Sheets.Search.SearchResult;
                /**
                 * Gets or sets whether users can select ranges of items on a sheet.
                 * @param {GC.Spread.Sheets.SelectionPolicy} [value] Whether users can select single items, ranges, or a combination of both.
                 * @returns {GC.Spread.Sheets.SelectionPolicy|GC.Spread.Sheets.Worksheet} If no value is set, returns the selection policy setting; otherwise, returns the sheet.
                 * @example
                 * //This example uses the selectionPolicy method.
                 * activeSheet.selectionUnit(GC.Spread.Sheets.SelectionUnit.row);
                 * activeSheet.selectionPolicy(GC.Spread.Sheets.SelectionPolicy.range);
                 */
                selectionPolicy(value?:  GC.Spread.Sheets.SelectionPolicy): any;
                /**
                 * Gets or sets whether users can select cells, rows, or columns on a sheet.
                 * @param {GC.Spread.Sheets.SelectionUnit} [value] Whether users can select cells, rows, or columns.
                 * @returns {GC.Spread.Sheets.SelectionUnit|GC.Spread.Sheets.Worksheet} If no value is set, returns the selection unit setting; otherwise, returns the sheet.
                 * @example
                 * //This example uses the selectionUnit method.
                 * activeSheet.selectionUnit(GC.Spread.Sheets.SelectionUnit.row);
                 * activeSheet.selectionPolicy(GC.Spread.Sheets.SelectionPolicy.range);
                 */
                selectionUnit(value?:  GC.Spread.Sheets.SelectionUnit): any;
                /**
                 * Sets the active cell for this sheet.
                 * @param {number} row The row index of the cell.
                 * @param {number} col The column index of the cell.
                 * @example
                 * //This example sets the active cell.
                 * sheet.setActiveCell(5,5);
                 * alert(sheet.getActiveColumnIndex());
                 * alert(sheet.getActiveRowIndex());
                 * spread.bind(GC.Spread.Sheets.Events.EnterCell, function (event, data) {
                 *     alert(data.col);
                 *     alert(data.row);
                 *   });
                 * spread.bind(GC.Spread.Sheets.Events.LeaveCell, function (event, data) {
                 *     alert(data.col);
                 *     alert(data.row);
                 *   });
                 */
                setActiveCell(row:  number,  col:  number): void;
                /**
                 * Sets the values in the specified two-dimensional array of objects into the specified range of cells on this sheet.
                 * @param {number} row The row index.
                 * @param {number} column The column index.
                 * @param {Array} array The array from which to set values.
                 * @param {boolean} [setFormula] If <c>true</c>, set formulas; otherwise, set values.
                 * @example
                 * //This example uses the setArray method.
                 * //set value
                 * var array = [[1,2,3],[4,5],[6,7,8,9]];
                 * activeSheet.setArray(1, 2, array);
                 * //set formula
                 * var array = [["=1+1","=2+2","=3+3"],["=4+4","=5+5"],["=6+6","=7+7","=8+8","=9+9"]];
                 * activeSheet.setArray(1, 2, array, true);
                 * //get value
                 * var newArray = activeSheet.getArray(1, 2, 3, 4);
                 * //getformula
                 * var newArray = activeSheet.getArray(1, 2, 3, 4, true);
                 * //alert(newArray[0]);
                 */
                setArray(row:  number,  column:  number,  array:  any[],  setFormula?:  boolean): void;
                /**
                 * Sets a formula in a specified cell in the specified sheet area.
                 * @param {number} row The start row index.
                 * @param {number} col The start column index.
                 * @param {number} rowCount The number of rows in range.
                 * @param {number} colCount The number of columns in range.
                 * @param {string} value The array formula to place in the specified range.
                 * @example
                 * //This example uses the setArrayFormula method.
                 * activeSheet.getCell(1,1).value(3);
                 * activeSheet.getCell(2,1).value(1);
                 * activeSheet.getCell(3,1).value(3);
                 * activeSheet.getCell(4,1).value(7);
                 * activeSheet.getCell(1,2).value(7);
                 * activeSheet.getCell(2,2).value(7);
                 * activeSheet.getCell(3,2).value(7);
                 * activeSheet.getCell(4,2).value(7);
                 * spread.allowUserEditFormula(true);
                 * activeSheet.setArrayFormula(0, 3, 4, 1, "B2:B5*C2:C5");
                 */
                setArrayFormula(row:  number,  col:  number,  rowCount:  number,  colCount:  number,  value:  string): void;
                /**
                 * Sets the binding path for cell-level binding in a specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {string} path The binding path for the cell binding source.
                 * @returns {GC.Spread.Sheets.Worksheet}
                 * @example
                 * //This example binds a cell.
                 * var test = {name: "John", gender: "male"};
                 * sheet.setBindingPath(0, 0, "name");
                 * sheet.setBindingPath(0, 1, "gender");
                 * sheet.setDataSource(new GC.Spread.Sheets.Bindings.CellBindingSource(test));
                 */
                setBindingPath(row:  number,  col:  number,  path:  string): GC.Spread.Sheets.Worksheet;
                /**
                 * Sets the cell type.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.CellTypes.Base} value The cell type.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * //This example uses the setCellType method.
                 * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                 * cellType.buttonBackColor("#FFFF00");
                 * cellType.text("this is a button");
                 * activeSheet.setCellType(1,1,cellType);
                 * spread.bind(GC.Spread.Sheets.Events.ButtonClicked, function (e, args) {
                 *     var sheet = args.sheet, row = args.row, col = args.col;
                 *     var cellType = activeSheet.getCellType(row, col);
                 *     if (cellType instanceof GC.Spread.Sheets.CellTypes.Button) {
                 *         alert("Button Clicked");
                 *     }
                 * });
                 */
                setCellType(row:  number,  col:  number,  value:  GC.Spread.Sheets.CellTypes.Base,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the column count in the specified sheet area.
                 * @param {number} colCount The column count.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @example
                 * //This example sets the number of columns.
                 * sheet.setRowCount(4,1);
                 * sheet.setColumnCount(4,2);
                 * sheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.colHeader);
                 * sheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.rowHeader);
                 * sheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.viewport);
                 */
                setColumnCount(colCount:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets whether a forced page break is inserted before the specified column on this sheet when printing.
                 * @param {number} column The column index.
                 * @param {boolean} value Set to <c>true</c> to force a page break before the specified column on this sheet when printing.
                 * @example
                 * //This example sets the page break.
                 * activeSheet.setColumnPageBreak(5, true);
                 */
                setColumnPageBreak(column:  number,  value:  boolean): void;
                /**
                 * Sets whether users can resize the specified column in the specified sheet area.
                 * @param {number} col The column index.
                 * @param {boolean} value Set to <c>true</c> to allow users to resize the column.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @example
                 * //This example sets the setColumnResizable method.
                 * sheet.setRowResizable(3,false,GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.setColumnResizable(3,false,GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.getRange(1,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).resizable(false);
                 * sheet.getRange(-1, 1, -1, 1, GC.Spread.Sheets.SheetArea.viewport).resizable(false);
                 */
                setColumnResizable(col:  number,  value:  boolean,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets whether a column in the specified sheet area is displayed.
                 * @param {number} col The column index.
                 * @param {boolean} value Whether to display the column.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @example
                 * //This example sets the specified column to be hidden.
                 * activeSheet.setColumnVisible(2,false,GC.Spread.Sheets.SheetArea.viewport);
                 */
                setColumnVisible(col:  number,  value:  boolean,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the width in pixels or dynamic size for the specified column in the specified sheet area.
                 * @param {number} col The column index.
                 * @param {number | string} value The width in pixels, or use the string with "*" to represent the dynamic size.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to viewport.
                 * @example
                 * //This example sets the column width.
                 * sheet.setValue(0, 0, "value");
                 * sheet.addRows(0, 2);
                 * sheet.addColumns(0, 2);
                 * sheet.setRowHeight(0, 50.0, GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.setColumnWidth(0, 150.0, GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.setColumnWidth(1, "2*", GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).backColor("Gray");
                 * sheet.getRange(-1, 0, -1, 1, GC.Spread.Sheets.SheetArea.viewport).backColor ("Brown");
                 */
                setColumnWidth(col:  number,  value:  number | string,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets delimited text (CSV) in the sheet.
                 * @param {number} row The start row.
                 * @param {number} column The start column.
                 * @param {string} text The delimited text.
                 * @param {string} rowDelimiter The row delimiter.
                 * @param {string} columnDelimiter The column delimiter.
                 */
                setCsv(row:  number,  column:  number,  text:  string,  rowDelimiter:  string,  columnDelimiter:  string): void;
                /**
                 * Sets the data source that populates the sheet.
                 * @param {Object} data The data source.
                 * @param {boolean} reset <c>true</c> if the sheet is reset; otherwise, <c>false</c>.
                 * @example
                 * var test = [
                 *        { "Series0": 2, "Series1": 1 },
                 *        { "Series0": 4, "Series1": 2 },
                 *        { "Series0": 3, "Series1": 4 }
                 *             ];
                 * activeSheet.autoGenerateColumns = true;
                 * activeSheet.setDataSource(test, true);
                 */
                setDataSource(data:  any,  reset?:  boolean): void;
                /**
                 * Sets the cell data validator.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.DataValidation.DefaultDataValidator} value The data validator.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createListValidator("1,2,3");
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Value must be 1,2 or 3");
                 * dv.inputTitle("tip");
                 * activeSheet.setDataValidator(1,1,dv,GC.Spread.Sheets.SheetArea.viewport);
                 * alert(activeSheet.getDataValidator(1,1).getValidList(activeSheet,1,1));
                 */
                setDataValidator(row:  number,  col:  number,  value:  GC.Spread.Sheets.DataValidation.DefaultDataValidator,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the cell data validator.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {number} rowCount The row count.
                 * @param {number} colCount The column count.
                 * @param {GC.Spread.Sheets.DataValidation.DefaultDataValidator} value The data validator.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createListValidator("1,2,3");
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Value must be 1,2 or 3");
                 * dv.inputTitle("tip");
                 * activeSheet.setDataValidator(1,1,1,1,dv,GC.Spread.Sheets.SheetArea.viewport);
                 * alert(activeSheet.getDataValidator(1,1).getValidList(activeSheet,1,1));
                 */
                setDataValidator(row:  number,  col:  number,  rowCount:  number,  colCount:  number,  value:  GC.Spread.Sheets.DataValidation.DefaultDataValidator,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the default style information for the sheet.
                 * @param {GC.Spread.Sheets.Style} style The style to set.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 */
                setDefaultStyle(style:  GC.Spread.Sheets.Style,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the cell formatter.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {string | GC.Spread.Formatter.FormatterBase} value The formatter string or object.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * //This example sets the format object for the active sheet.
                 * activeSheet.setValue(2, 3, new Date(2011, 2, 9));
                 * activeSheet.setFormatter(2,3,"M",GC.Spread.Sheets.SheetArea.viewport);
                 */
                setFormatter(row:  number,  col:  number,  value:  string | GC.Spread.Formatter.FormatterBase,  sheetArea:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets a formula in a specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {string} value The formula to place in the specified cell.
                 * @example
                 * //This example sets the formula for the specified cell.
                 * activeSheet.setValue(0, 2, 3);
                 * activeSheet.setFormula(1,1,"C1+D1",GC.Spread.Sheets.SheetArea.viewport);
                 */
                setFormula(row:  number,  col:  number,  value:  string): void;
                /**
                 * Sets the hyperlink data for the specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {Object} value The hyperlink data to set for the specified cell.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * //This example uses the setHyperlink method.
                 * sheet.setHyperlink(0, 2, {
                 *    url: 'https://www.grapecity.com/spreadjs',
                 *    tooltip: 'baidu',
                 *    linkColor: 'blue',
                 *    visitedLinkColor: 'red',
                 *    target: GC.Spread.Sheets.Hyperlink.HyperlinkTargetType.blank,
                 *    command: 'navigationLeft'
                 * }, GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.setHyperlink(1, 1, {
                 *    url: 'https://www.grapecity.com/spreadjs',
                 *    tooltip: 'baidu',
                 *    setUnderlineToStyle: true;
                 *    setForeColorToStyle: true;
                 *    target: GC.Spread.Sheets.Hyperlink.HyperlinkTargetType.top,
                 *    command: function() { console.log('Only show this message when click the hyperlink.')}
                 * }, GC.Spread.Sheets.SheetArea.viewport);
                 */
                setHyperlink(row:  number,  col:  number,  value:  GC.Spread.Sheets.IHyperlink,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the row count in the specified sheet area.
                 * @param {number} rowCount The row count.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @example
                 * //This example sets the row count.
                 * sheet.setRowCount(4,1);
                 * sheet.setColumnCount(4,2);
                 * sheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.colHeader);
                 * sheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.rowHeader);
                 * sheet.addSpan(0,0,3,3,GC.Spread.Sheets.SheetArea.viewport);
                 */
                setRowCount(rowCount:  number,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the height in pixels or dynamic size for the specified row in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number | string} value The height in pixels, or use the string with "*" to represent the dynamic size.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @example
                 * //This example sets the row height.
                 * sheet.setValue(0, 0, "value");
                 * sheet.addRows(0, 2);
                 * sheet.addColumns(0, 2);
                 * sheet.setRowHeight(0, 50.0, GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.setRowHeight(1, "3*", GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.setColumnWidth(0, 150.0, GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).backColor("Gray");
                 * sheet.getRange(-1, 0, -1, 1, GC.Spread.Sheets.SheetArea.viewport).backColor ("Brown");
                 */
                setRowHeight(row:  number,  value:  number | string,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets whether a forced page break is inserted before the specified row on this sheet when printing.
                 * @param {number} row The row index.
                 * @param {boolean} value Set to <c>true</c> to force a page break before the specified row on this sheet when printing.
                 * @example
                 * activeSheet.setRowPageBreak(3, true);
                 */
                setRowPageBreak(row:  number,  value:  boolean): void;
                /**
                 * Sets whether users can resize the specified row in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {boolean} value Set to <c>true</c> to let the users resize the specified row.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @example
                 * //This example prevents certain rows and columns from being resized.
                 * sheet.setRowResizable(3,false,GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.setColumnResizable(3,false,GC.Spread.Sheets.SheetArea.viewport);
                 * sheet.getRange(1,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).resizable(false);
                 * sheet.getRange(-1, 1, -1, 1, GC.Spread.Sheets.SheetArea.viewport).resizable(false);
                 */
                setRowResizable(row:  number,  value:  boolean,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets whether the control displays the specified row in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {boolean} value Set to <c>true</c> to display the specified row.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not given, it defaults to <b>viewport</b>.
                 * @example
                 * //This example sets the specified row to be hidden.
                 * activeSheet.setRowVisible(1,false,GC.Spread.Sheets.SheetArea.viewport);
                 */
                setRowVisible(row:  number,  value:  boolean,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the selection to a cell or a range and sets the active cell to the first cell.
                 * @param {number} row The row index of the first cell to add.
                 * @param {number} column The column index of the first cell to add.
                 * @param {number} rowCount The number of rows to add.
                 * @param {number} columnCount The number of columns to add.
                 * @example
                 * //This example selects a range of cells.
                 * sheet.setValue(0,0, 1,3);
                 * sheet.setValue(1,0, 50,3);
                 * sheet.setValue(2,0, 100,3);
                 * sheet.setValue(3,0, 2,3);
                 * sheet.setValue(4,0, 60,3);
                 * sheet.setValue(5,0, 90,3);
                 * sheet.setValue(6,0, 3,3);
                 * sheet.setValue(7,0, 40,3);
                 * sheet.setValue(8,0, 70,3);
                 * sheet.setValue(9,0, 5,3);
                 * sheet.setValue(10,0, 35,3);
                 * sheet.setSelection(0,0,11,1);
                 * sheet.conditionalFormats.add3ScaleRule(1, 10, "red", 0, 50, "blue",2, 100, "yellow", sheet.getSelections());
                 */
                setSelection(row:  number,  column:  number,  rowCount:  number,  columnCount:  number): void;
                /**
                 * Sets the sparkline for a cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.Range | string} dataRange The data range.
                 * @param {GC.Spread.Sheets.Sparklines.DataOrientation} dataOrientation The data orientation.
                 * @param {GC.Spread.Sheets.Sparklines.SparklineType} sparklineType The sparkline type.
                 * @param {GC.Spread.Sheets.Sparklines.SparklineSetting} sparklineSetting The sparkline setting.
                 * @param {GC.Spread.Sheets.Range | string} dateAxisRange The date axis range.
                 * @param {GC.Spread.Sheets.Sparklines.DataOrientation} dateAxisOrientation The date axis range orientation.
                 * @returns {GC.Spread.Sheets.Sparklines.Sparkline} The sparkline.
                 * @example
                 * //This example creates a sparkline for the specified range.
                 * var data = new GC.Spread.Sheets.Range(1, 0, 8, 1);
                 * var setting = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * setting.options.showMarkers = true;
                 * setting.options.lineWeight = 3;
                 * setting.options.displayXAxis = true;
                 * setting.options.showFirst = true;
                 * setting.options.showLast = true;
                 * setting.options.showLow = true;
                 * setting.options.showHigh = true;
                 * setting.options.showNegative = true;
                 * setting.options.seriesColor = "Text 2 1";
                 * setting.options.firstMarkerColor = "Text 2 3";
                 * setting.options.negativeColor = "Accent 2 1";
                 * setting.options.markersColor = "Accent 3 1";
                 * setting.options.lowMarkerColor = "Accent 4 1";
                 * setting.options.highMarkerColor = "Accent 6 1";
                 * setting.options.lastMarkerColor = "Accent 6 6";
                 * setting.options.axisColor ="Text 1 1";
                 * sheet.addSpan(13, 0, 4, 3, null);
                 * sheet.setSparkline(13, 0, data, GC.Spread.Sheets.Sparklines.DataOrientation.Vertical, GC.Spread.Sheets.Sparklines.SparklineType.line, setting);
                 * sheet.setValue(1, 0, 1);
                 * sheet.setValue(2, 0, -2);
                 * sheet.setValue(3, 0, -1);
                 * sheet.setValue(4, 0, 6);
                 * sheet.setValue(5, 0, 4);
                 * sheet.setValue(6, 0, -4);
                 * sheet.setValue(7, 0, 3);
                 * sheet.setValue(8, 0, 8);
                 */
                setSparkline(row:  number,  col:  number,  dataRange:  GC.Spread.Sheets.Range | string,  dataOrientation:  GC.Spread.Sheets.Sparklines.DataOrientation,  sparklineType:  GC.Spread.Sheets.Sparklines.SparklineType,  sparklineSetting:  GC.Spread.Sheets.Sparklines.SparklineSetting,  dateAxisRange?:  GC.Spread.Sheets.Range | string,  dateAxisOrientation?:  GC.Spread.Sheets.Sparklines.DataOrientation): GC.Spread.Sheets.Sparklines.Sparkline;
                /**
                 * Sets the style information for a specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} column The column index.
                 * @param {GC.Spread.Sheets.Style} value The cell style.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * //This example uses the setStyle method.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "lightgreen";
                 * style.backgroundImage = "./css/images/quarter1.png";
                 * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
                 */
                setStyle(row:  number,  col:  number,  value:  GC.Spread.Sheets.Style,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the specified style name for a specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} column The column index.
                 * @param {string} value The name of the style to set.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * var namedStyle = new GC.Spread.Sheets.Style();
                 * namedStyle.name = "style1";
                 * namedStyle.backColor = "green";
                 * activeSheet.addNamedStyle(namedStyle);
                 * activeSheet.setStyleName(1, 1, "style1"); // cell(1,1)'s backColor is  green.
                 * activeSheet.setStyleName(2, 1, "style1");
                 */
                setStyleName(row:  number,  column:  number,  value:  string,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the tag value for the specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {Object} tag The tag value to set for the specified cell.
                 * @param {GC.Spread.Sheets.SheetArea} sheetArea The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * activeSheet.setTag(1,1,"test");
                 */
                setTag(row:  number,  col:  number,  tag:  any,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the formatted text in the cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {string} value The text for the specified cell.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @example
                 * activeSheet.setText(1, 0, "10");
                 */
                setText(row:  number,  col:  number,  value:  string,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                /**
                 * Sets the value for the specified cell in the specified sheet area.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {Object} value The value to set for the specified cell. if the value is rich text format, should include a richText field which type is a rich text style array.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                 * @param {boolean} ignoreRecalc Whether to ignore recalculation.
                 * @example
                 * //This example uses the setValue method.
                 * sheet.setValue(0,2,"ColumnHeader", GC.Spread.Sheets.SheetArea.colHeader);
                 * sheet.setValue(2,0,{richText:[{style:{font:'bold 24px Arial'},text:'SpreadJS'}]}, GC.Spread.Sheets.SheetArea.rowHeader);
                 * sheet.setValue(1, 1, {richText:[{style:{vertAlign: GC.Spread.Sheets.VertAlign.subscript},text:'SpreadJS'}]}, GC.Spread.Sheets.SheetArea.viewport);
                 */
                setValue(row:  number,  col:  number,  value:  any,  sheetArea?:  GC.Spread.Sheets.SheetArea,  ignoreRecalc?:  boolean): void;
                /**
                 * Moves the view of a cell to the specified position in the viewport.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.VerticalPosition} verticalPosition The vertical position in which to display the cell.
                 * @param {GC.Spread.Sheets.HorizontalPosition} horizontalPosition The horizontal position in which to display the cell.
                 * @example
                 * //This example uses the showCell method.
                 * //Set cell (3,3) as the active cell
                 * activeSheet.setActiveCell(3, 3);
                 * //Display the active cell at top left
                 * activeSheet.showCell(3, 3, GC.Spread.Sheets.VerticalPosition.top, GC.Spread.Sheets.HorizontalPosition.left);
                 */
                showCell(row:  number,  col:  number,  verticalPosition:  GC.Spread.Sheets.VerticalPosition,  horizontalPosition:  GC.Spread.Sheets.HorizontalPosition): void;
                /**
                 * Moves the view of a column to the specified position in the viewport.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Sheets.HorizontalPosition} horizontalPosition The horizontal position in which to display the column.
                 * @example
                 * activeSheet.showColumn(9, GC.Spread.Sheets.HorizontalPosition.left);
                 */
                showColumn(col:  number,  horizontalPosition:  GC.Spread.Sheets.HorizontalPosition): void;
                /**
                 * Gets or sets whether the column outline (range group) is visible.
                 * @param {boolean} value Whether to display the column outline.
                 * @returns {boolean | GC.Spread.Sheets.Worksheet} If no value is set, returns a value that indicates whether the column outline is displayed on this sheet; otherwise, returns the worksheet.
                 * @example
                 * //This example uses the showColumnOutline method.
                 * activeSheet.showColumnOutline(false);
                 */
                showColumnOutline(value?:  boolean): any;
                /**
                 * Moves the view of a row to the specified position in the viewport.
                 * @param {number} row The row index.
                 * @param {GC.Spread.Sheets.VerticalPosition} verticalPosition The vertical position in which to display the row.
                 * @example
                 * activeSheet.showRow(9, GC.Spread.Sheets.VerticalPosition.top);
                 */
                showRow(row:  number,  verticalPosition:  GC.Spread.Sheets.VerticalPosition): void;
                /**
                 * Gets or sets whether the row outline (range group) is visible.
                 * @param {boolean} value Whether to display the row outline.
                 * @returns {boolean | GC.Spread.Sheets.Worksheet} If no value is set, returns a value that indicates whether the row outline is displayed on this sheet; otherwise, returns the worksheet.
                 * @example
                 * //This example uses the showRowOutline method.
                 * activeSheet.showRowOutline(false);
                 */
                showRowOutline(value?:  boolean): any;
                /**
                 * Sorts a range of cells in this sheet in the data model.
                 * @param {number} row The index of the starting row of the block of cells to sort.
                 * @param {number} column The index of the starting column of the block of cells to sort.
                 * @param {number} rowCount The number of rows in the block of cells.
                 * @param {number} columnCount The number of columns in the block of cells.
                 * @param {boolean} byRows Set to <c>true</c> to sort by rows, and <c>false</c> to sort by columns.
                 * @param {Object} sortInfo The SortInfo object with sort criteria and information about how to perform the sort. For example, [{index:0,ascending:true,compareFunction: Function}]
                 * @param {number} sortInfo.index The index of the column or row on which to sort.
                 * @param {boolean} sortInfo.ascending Whether the sort order is ascending.
                 * @param {Function} sortInfo.compareFunction Whether the sort order is ascending.  function (value1, value2) {return 0;}.
                 * @returns {boolean} <c>true</c> if the data is sorted successfully; otherwise, <c>false</c>.
                 * @example
                 * //This example sorts a range.
                 * sheet.setValue(0,0,"112");
                 * sheet.setValue(1,0,"10");
                 * sheet.setValue(2,0,"223");
                 * sheet.setValue(3,0,"20");
                 * sheet.setValue(4,0,"334");
                 * sheet.setValue(5,0,"30");
                 * function pinyinCompare (obj1, obj2) {
                 *     return obj1.toString().localeCompare(obj2.toString(), 'zh');
                 * }
                 * sheet.sortRange(0, 0, 6, 1, true, [
                 *                 {index:0, ascending:true, compareFunction: pinyinCompare}
                 *                 ]);
                 */
                sortRange(row:  number,  column:  number,  rowCount:  number,  columnCount:  number,  byRows:  boolean,  sortInfo:  GC.Spread.Sheets.ISortInfo[]): boolean;
                /**
                 * Starts to edit the cell.
                 * @param {boolean} selectAll Set to <c>true</c> to select all the text in the cell.
                 * @param {string} defaultText The default text to display while editing the cell.
                 * @example
                 * //This example uses the startEdit method.
                 * activeSheet.setActiveCell(5,5);
                 * activeSheet.startEdit(true, "Test");
                 */
                startEdit(selectAll?:  boolean,  defaultText?:  string): void;
                /**
                 * Suspends the calculation service.
                 * @param {boolean} ignoreDirty Specifies whether to invalidate the dependency cells.
                 * @example
                 * //This example uses the suspendCalcService method.
                 * activeSheet.suspendCalcService(false);
                 * activeSheet.setValue(0,0,1);
                 * activeSheet.setValue(0,1,2);
                 * activeSheet.setValue(0,2,10);
                 * activeSheet.getCell(1,1).formula("=SUM(A1:C1)");
                 * activeSheet.resumeCalcService(true);
                 */
                suspendCalcService(ignoreDirty?:  boolean): void;
                /**
                 * Suspends recording the dirty data.
                 * @example
                 * //This example uses the suspendDirty method.
                 *  var customers = [
                 *                { ID: 0, Name: 'A', Info1: 'Info0' },
                 *                { ID: 1, Name: 'B', Info1: 'Info1' },
                 *                { ID: 2, Name: 'C', Info1: 'Info2' },
                 *             ];
                 * activeSheet.setDataSource(customers);
                 * activeSheet.suspendDirty();
                 * alert(activeSheet.isDirtySuspended());
                 * activeSheet.resumeDirty();
                 * alert(activeSheet.isDirtySuspended());
                 */
                suspendDirty(): void;
                /**
                 * Suspends the event.
                 * @example
                 * //This example suspends and resumes the event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (sender, args) {
                 *     if (args.propertyName === "value") {
                 *         alert(activeSheet.getValue(args.row, args.col));
                 *     }
                 * });
                 * activeSheet.suspendEvent();
                 * activeSheet.setValue(0, 0, "111");
                 * activeSheet.resumeEvent();
                 * activeSheet.setValue(1, 1, "222");
                 */
                suspendEvent(): void;
                /**
                 * Suspends the paint.
                 */
                suspendPaint(): void;
                /**
                 * Gets or sets the tag value for the current sheet.
                 * @param {Object} value The tag value to set for the current sheet.
                 * @returns {Object | GC.Spread.Sheets.Worksheet} If no value is set, returns the tag value of the current sheet; otherwise, returns the worksheet.
                 * @example
                 * //This example sets the sheet tag.
                 * activeSheet.tag("test");
                 * alert(activeSheet.tag());
                 */
                tag(value?:  any): any;
                /**
                 * Saves the object state to a JSON string.
                 * @param {Object} serializationOption Serialization option that contains the <i>includeBindingSource</i> argument. See the Remarks for more information.
                 * @returns {Object} The sheet data.
                 * @example
                 * //This example uses the toJSON method.
                 * activeSheet.getCell(0,0).value(123);
                 * var jsonStr = null;
                 * //export
                 * jsonStr = JSON.stringify(activeSheet.toJSON());
                 * //import
                 * activeSheet.fromJSON(JSON.parse(jsonStr));
                 * alert(jsonStr);
                 */
                toJSON(serializationOption?:  Object): Object;
                /**
                 * Removes the binding of an event to the sheet.
                 * @param {string} type The event type.
                 * @param {Function} fn Specifies the function for which to remove the binding.
                 * @example
                 * //This example unbinds the event after setting the first value.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (sender, args) {
                 *     if (args.propertyName === "value") {
                 *         alert(activeSheet.getValue(args.row, args.col));
                 *     }
                 * });
                 * activeSheet.setValue(0, 0, "111");
                 * activeSheet.unbind(GC.Spread.Sheets.Events.CellChanged);
                 * activeSheet.setValue(1, 0, "222");
                 * activeSheet.setValue(2, 0, "333");
                 * activeSheet.setValue(3, 0, "444");
                 */
                unbind(type:  string,  fn?:  Function): void;
                /**
                 * Removes the binding of all events to the sheet.
                 * @example
                 * //This example cancels monitoring of all events.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (sender, args) {
                 *     if (args.propertyName === "value") {
                 *         alert(activeSheet.getValue(args.row, args.col));
                 *     }
                 * });
                 * activeSheet.setValue(0, 0, "111");
                 * activeSheet.unbindAll(); //cancel monitoring of all events.
                 * activeSheet.setValue(1, 0, "222");
                 * activeSheet.setValue(2, 0, "333");
                 * activeSheet.setValue(3, 0, "444");
                 */
                unbindAll(): void;
                /**
                 * Ungroups the sparklines in the specified group.
                 * @param {GC.Spread.Sheets.Sparklines.SparklineGroup} group The sparkline group.
                 * @example
                 * //This example uses the ungroupSparkline method.
                 * activeSheet.setValue(0, 0, "Data Range is A2-A9");
                 * activeSheet.setValue(1, 0, 1);
                 * activeSheet.setValue(2, 0, -2);
                 * activeSheet.setValue(3, 0, -1);
                 * activeSheet.setValue(4, 0, 6);
                 * activeSheet.setValue(5, 0, 4);
                 * activeSheet.setValue(6, 0, -4);
                 * activeSheet.setValue(7, 0, 3);
                 * activeSheet.setValue(8, 0, 8);
                 * var data = new GC.Spread.Sheets.Range(1, 0, 8, 1);
                 * var setting = new GC.Spread.Sheets.Sparklines.SparklineSetting();
                 * var s1=  activeSheet.setSparkline(11, 0, data, GC.Spread.Sheets.Sparklines.DataOrientation.Vertical, GC.Spread.Sheets.Sparklines.SparklineType.line, setting);
                 * var s2 =activeSheet.setSparkline(11, 3, data, GC.Spread.Sheets.Sparklines.DataOrientation.Vertical, GC.Spread.Sheets.Sparklines.SparklineType.column, setting);
                 * var s3=  activeSheet.setSparkline(11, 6, data, GC.Spread.Sheets.Sparklines.DataOrientation.Vertical, GC.Spread.Sheets.Sparklines.SparklineType.winloss, setting);
                 * var group = activeSheet.groupSparkline([s1,s2,s3]);
                 * //activeSheet.ungroupSparkline(group);
                 */
                ungroupSparkline(group:  GC.Spread.Sheets.Sparklines.SparklineGroup): void;
                /**
                 * Sets whether the worksheet is displayed.
                 * @param {boolean} [value] Whether the worksheet is displayed.
                 * @returns {boolean|GC.Spread.Sheets.Worksheet} If you call this function without a parameter, it returns a boolean indicating whether the sheet is visible;
                 * otherwise, it returns the current worksheet object.
                 * @example
                 * activeSheet.visible(false);
                 */
                visible(value?:  boolean): any;
                /**
                 * Gets or sets the zoom factor for the sheet.
                 * @param {number} [factor] The zoom factor.
                 * @returns {number|GC.Spread.Sheets.Worksheet} If no value is set, returns the zoom factor; otherwise, returns the worksheet.
                 * @example
                 * //This example zooms the sheet.
                 * spread.options.allowUserZoom = false;
                 * sheet.zoom(3);
                 */
                zoom(factor?:  number): any;
            }
            module AutoMerge{

                export interface IRangeInfo{
                    range: GC.Spread.Sheets.Range;
                    direction: GC.Spread.Sheets.AutoMerge.AutoMergeDirection;
                    mode: GC.Spread.Sheets.AutoMerge.AutoMergeMode;
                    sheetArea: GC.Spread.Sheets.SheetArea;
                }

                /**
                 * Indicates the auto merge direction.
                 * @enum {number}
                 */
                export enum AutoMergeDirection{
                    /**
                     * Indicates to cancel the auto merge.
                     */
                    none= 0,
                    /**
                     * Indicates to apply the auto merge in column direction.
                     */
                    column= 1,
                    /**
                     * Indicates to apply the auto merge in row direction.
                     */
                    row= 2,
                    /**
                     * Indicates to apply the auto merge in column direction preferentially then in row direction.
                     */
                    columnRow= 3,
                    /**
                     * Indicates to apply the auto merge in row direction preferentially then in column direction.
                     */
                    rowColumn= 4
                }

                /**
                 * Indicates the auto merge mode.
                 * @enum {number}
                 */
                export enum AutoMergeMode{
                    /**
                     * Indicates to apply the auto merge when neighboring cells have same value.
                     */
                    free= 0,
                    /**
                     * Indicates to apply the auto merge when neighboring cells have same value and the corresponding cells in previous row or column are merged automatically.
                     */
                    restricted= 1
                }

            }

            module Barcode{

                export class Codabar extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the Codabar.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class Code128 extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the Code128.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class Code39 extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the Code39.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class Code49 extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the Code49.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class Code93 extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the Code93.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class DataMatrix extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the DataMatrix.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class EAN13 extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the EAN13.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class EAN8 extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the EAN8.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class GS1_128 extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the GS1_128.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class PDF417 extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the PDF417.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }

                export class QRCode extends GC.Spread.Sheets.Sparklines.SparklineEx{
                    /**
                     * Represents the class for the QRCode.
                     * @extends GC.Spread.Sheets.Sparklines.SparklineEx
                     * @class
                     */
                    constructor();
                }
            }

            module Bindings{

                export class CellBindingSource{
                    /**
                     * Represents a source for cell binding.
                     * @param {Object} source The data source.
                     * @class
                     * @example
                     * var person = {name: "Wang feng", age: 25, address: {postcode: "710075"}};
                     * var source = new GC.Spread.Sheets.Bindings.CellBindingSource(person);
                     * activeSheet.setBindingPath(0, 0, "name");
                     * activeSheet.setBindingPath(1, 1, "age");
                     * activeSheet.setBindingPath(3, 3, "address.postcode");
                     * activeSheet.setDataSource(source);
                     */
                    constructor(source:  Object);
                    /**
                     * Gets the wrapped data source for cell binding.
                     * @returns {Object} The original data source.
                     * @example
                     * //This example gets the name.
                     * var person = { name: "Wang feng", age: 25, address: { postcode: "710075" } };
                     * var source = new GC.Spread.Sheets.Bindings.CellBindingSource(person);
                     * activeSheet.setBindingPath(0, 0, "name");
                     * activeSheet.setBindingPath(1, 1, "age");
                     * activeSheet.setBindingPath(3, 3, "address.postcode");
                     * activeSheet.setDataSource(source);
                     * alert(source.getSource().name);
                     */
                    getSource(): Object;
                    /**
                     * Gets the value of the source by the binding path.
                     * @param {string} path The binding path.
                     * @returns {Object} Returns the value of the binding source at the specified path.
                     * @example
                     * //This example gets the value.
                     * var person = {name: "Wang feng", age: 25, address: {postcode: "710075"}};
                     * var source = new GC.Spread.Sheets.Bindings.CellBindingSource(person);
                     * activeSheet.setBindingPath(0, 0, "name");
                     * activeSheet.setBindingPath(1, 1, "age");
                     * activeSheet.setBindingPath(3, 3, "address.postcode");
                     * activeSheet.setDataSource(source);
                     * alert(source.getValue("name"));
                     */
                    getValue(path:  string): Object;
                    /**
                     * Sets the value of the source by the binding path.
                     * @param {string} path The row index.
                     * @param {Object} value The value to set.
                     * @example
                     * //This example sets the name value.
                     * var person = {name: "Wang feng", age: 25, address: {postcode: "710075"}};
                     * var source = new GC.Spread.Sheets.Bindings.CellBindingSource(person);
                     * activeSheet.setBindingPath(0, 0, "name");
                     * activeSheet.setBindingPath(1, 1, "age");
                     * activeSheet.setBindingPath(3, 3, "address.postcode");
                     * activeSheet.setDataSource(source);
                     * source.setValue("name", "test");
                     * activeSheet.resumePaint();
                     * activeSheet.repaint();
                     */
                    setValue(path:  string,  value:  Object): void;
                }
            }

            module CalcEngine{
                /**
                 * Evaluates the specified formula.
                 * @param {object} context The evaluation context; in general, you should use the active sheet object.
                 * @param {string} formula The formula string.
                 * @param {number} [baseRow] The base row index of the formula.
                 * @param {number} [baseColumn] The base column index of the formula.
                 * @param {boolean} [useR1C1] Whether to use the r1c1 reference style.
                 * @example
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById('ss'), { sheetCount: 1 });
                 * sheet = spread.getSheet(0);
                 * sheet.setValue(0, 0, 1);
                 * sheet.setValue(1, 0, 2);
                 * // Using EvaluateFormula() method to evaluate formula without setting formula in sheet's cell
                 * var result = GC.Spread.Sheets.CalcEngine.evaluateFormula(sheet, "SUM(A1:A2)", 0, 0);
                 * console.log("SUM(A1:A2) = " + result);
                 * @returns {object} The evaluated formula result.
                 */
                function evaluateFormula(context:  Object,  formula:  string,  baseRow?:  number,  baseColumn?:  number,  useR1C1?:  boolean): any;
                /**
                 * Unparse the specified expression tree to formula string.
                 * @param {object} context The context; in general, you should use the active sheet object.
                 * @param {GC.Spread.CalcEngine.Expression} expression The expression tree.
                 * @param {number} [baseRow] The base row index of the formula.
                 * @param {number} [baseColumn] The base column index of the formula.
                 * @param {boolean} [useR1C1] Whether to use the r1c1 reference style.
                 * @returns {string} The formula string.
                 * @example
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById('ss'), { sheetCount: 1 });
                 * sheet = spread.getSheet(0);
                 * sheet.setValue(0, 0, 1);
                 * sheet.setValue(0, 1, 2);
                 * sheet.setValue(0, 2, 3);
                 * sheet.addCustomName("customName1", "=12", 0, 0);
                 * sheet.addCustomName("customName2", "Average(20,45)", 0, 0);
                 * sheet.addCustomName("customName3", "=$A$1:$C$1");
                 * sheet.setFormula(1, 0, "customName1");
                 * sheet.setFormula(1, 1, "customName2");
                 * sheet.setFormula(1, 2, "sum(customName3)");
                 * var cname = sheet.getCustomName("customName2");
                 * if (cname instanceof GC.Spread.Sheets.NameInfo) {
                 *     // Get CustomName
                 *     var name = cname.getName();
                 *     // Get Expression
                 *     var expression = cname.getExpression();
                 *     // Get Expression String
                 *     var expStr = GC.Spread.Sheets.CalcEngine.expressionToFormula(sheet, expression, 0, 0);
                 *     console.log("Name: " + name + " ; Expression: " + expStr);
                 * }
                 */
                function expressionToFormula(context:  Object,  expression:  GC.Spread.CalcEngine.Expression,  baseRow?:  number,  baseColumn?:  number,  useR1C1?:  boolean): string;
                /**
                 * Parse the specified formula to expression tree.
                 * @param {object} context The context; in general, you should use the active sheet object.
                 * @param {string} formula The formula string.
                 * @param {number} [baseRow] The base row index of the formula.
                 * @param {number} [baseColumn] The base column index of the formula.
                 * @param {boolean} [useR1C1] Whether to use the r1c1 reference style.
                 * @returns {GC.Spread.CalcEngine.Expression} The expression tree.
                 * @example
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById('ss'), { sheetCount: 1 });
                 * sheet = spread.getSheet(0);
                 * sheet.setValue(0, 0, 1);
                 * sheet.setValue(0, 1, 2);
                 * sheet.setValue(0, 2, 3);
                 * sheet.getCell(4, 4).formula("=SUM(A1:C1)");
                 * var formula = sheet.getFormula(4, 4);
                 * var expression = GC.Spread.Sheets.CalcEngine.formulaToExpression(sheet, formula, 0, 0);
                 * console.log("Function Name is: " + expression.functionName);
                 */
                function formulaToExpression(context:  Object,  formula:  string,  baseRow?:  number,  baseColumn?:  number,  useR1C1?:  boolean): GC.Spread.CalcEngine.Expression;
                /**
                 * Converts a formula string to the specified cell ranges.
                 * @param {GC.Spread.Sheets.Worksheet} sheet The base sheet.
                 * @param {string} formula The formula.
                 * @param {number} baseRow The base row index of the formula.
                 * @param {number} baseCol The base column index of the formula.
                 * @returns {Array} The cell ranges that refers to the formula string.
                 */
                function formulaToRanges(sheet:  GC.Spread.Sheets.Worksheet,  formula:  string,  baseRow?:  number,  baseCol?:  number): Object[];
                /**
                 * Attempts to find a value for one cell that produces the desired formula result in another cell.
                 * @param {GC.Spread.Sheets.Worksheet} changingSheet The sheet that contains the cell that you want to adjust.
                 * @param {number} changingRow The row index of the cell that contains the value that you want to adjust.
                 * @param {number} changingColumn The column index of the cell that contains the value that you want to adjust.
                 * @param {GC.Spread.Sheets.Worksheet} formulaSheet The sheet that contains the formula that you want to resolve.
                 * @param {number} formulaRow The row index of the cell that contains the formula that you want to resolve.
                 * @param {number} formulaColumn The column index of the cell that contains the formula that you want to resolve.
                 * @param {number} desiredResult The formula result that you want.
                 * @example
                 * // This sample shows how to use the goal seek.
                 * // Loan amount is 10000, term is 18 months and pay 600 each month, evaluate what interest rate you will need to secure in order to meet your loan goal.
                 * sheet.setValue(0, 1, 10000); // Loan Amount
                 * sheet.setValue(1, 1, 18); // Term in Months
                 * sheet.setFormatter(2, 1, "0%"); // Interest Rate
                 * sheet.setFormatter(3, 1, "0.00");
                 * sheet.setFormula(3, 1, "PMT(B3/12,B2,B1)"); // Payment
                 * GC.Spread.Sheets.CalcEngine.goalSeek(sheet, 2, 1, sheet, 3, 1, -600); // result in B3 is 10%
                 */
                function goalSeek(changingSheet:  GC.Spread.Sheets.Worksheet,  changingRow:  number,  changingColumn:  number,  formulaSheet:  GC.Spread.Sheets.Worksheet,  formulaRow:  number,  formulaColumn:  number,  desiredResult:  number): void;
                /**
                 * Converts the specified cell range to a formula string.
                 * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell range in the sheet.
                 * @param {number} [baseRow] The base row index of the formula.
                 * @param {number} [baseCol] The base column index of the formula.
                 * @param {GC.Spread.Sheets.CalcEngine.RangeReferenceRelative} [rangeReferenceRelative] Whether the range reference is relative or absolute.
                 * @param {boolean} [useR1C1] Whether to use the r1c1 reference style.
                 * @returns {string} The formula string that refers to the specified cell range.
                 * @example
                 * spread = new GC.Spread.Sheets.Workbook(document.getElementById('ss'), { sheetCount: 1 });
                 * sheet = spread.getSheet(0);
                 * // Setting Value
                 * sheet.setValue(0, 0, 1, 3);
                 * sheet.setValue(1, 0, 50, 3);
                 * sheet.setValue(2, 0, 100, 3);
                 * sheet.setValue(3, 0, 2, 3);
                 * sheet.setValue(4, 0, 60, 3);
                 * sheet.setValue(5, 0, 90, 3);
                 * sheet.setValue(6, 0, 3, 3);
                 * sheet.setValue(7, 0, 40, 3);
                 * sheet.clearSelection();
                 * // Adding selections
                 * sheet.addSelection(0, 0, 3, 1);
                 * sheet.addSelection(5, 0, 2, 1);
                 * var ranges = sheet.getSelections();
                 * // getting range string
                 * var rangesStr = GC.Spread.Sheets.CalcEngine.rangesToFormula(ranges);
                 * // creating formula using selected ranges
                 * var formula = "Sum(" + rangesStr + ")";
                 * // setting formula in Sheet's cell
                 * sheet.setFormula(5, 5, formula, GC.Spread.Sheets.SheetArea.viewport);
                 */
                function rangesToFormula(ranges:  GC.Spread.Sheets.Range[],  baseRow?:  number,  baseCol?:  number,  rangeReferenceRelative?:  GC.Spread.Sheets.CalcEngine.RangeReferenceRelative,  useR1C1?:  boolean): string;
                /**
                 * Converts the specified cell range to a formula string.
                 * @param {GC.Spread.Sheets.Range} range The cell range in the sheet.
                 * @param {number} [baseRow] The base row index of the formula.
                 * @param {number} [baseCol] The base column index of the formula.
                 * @param {GC.Spread.Sheets.CalcEngine.RangeReferenceRelative} [rangeReferenceRelative] Whether the range reference is relative or absolute.
                 * @param {boolean} [useR1C1] Whether to use the r1c1 reference style.
                 * @returns {string} The formula string that refers to the specified cell range.
                 * @example
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById('ss'), { sheetCount: 1 });
                 * sheet = spread.getSheet(0);
                 * // setting value
                 * sheet.setValue(0, 0, 1, 3);
                 * sheet.setValue(1, 0, 50, 3);
                 * sheet.setValue(2, 0, 100, 3);
                 * sheet.setValue(3, 0, 2, 3);
                 * sheet.setValue(4, 0, 60, 3);
                 * sheet.setValue(5, 0, 90, 3);
                 * sheet.clearSelection();
                 * // adding selection
                 * sheet.addSelection(2, 0, 3, 1);
                 * var range = sheet.getSelections();
                 * // Getting range string
                 * var rangeStr = GC.Spread.Sheets.CalcEngine.rangeToFormula(range[0]);
                 * // creating formula using selected range
                 * var formula = "Sum(" + rangeStr + ")";
                 * // setting formula in Sheet's cell
                 * sheet.setFormula(5, 5, formula, GC.Spread.Sheets.SheetArea.viewport);
                 */
                function rangeToFormula(range:  GC.Spread.Sheets.Range,  baseRow?:  number,  baseCol?:  number,  rangeReferenceRelative?:  GC.Spread.Sheets.CalcEngine.RangeReferenceRelative,  useR1C1?:  boolean): string;
                /**
                 * Specifies whether the range reference is relative or absolute.
                 * @enum {number}
                 */
                export enum RangeReferenceRelative{
                    /**
                     * Specifies all reference is absolute
                     */
                    allAbsolute= 0,
                    /**
                     * Specifies start row is relative
                     */
                    startRowRelative= 1,
                    /**
                     * Specifies start column is relative
                     */
                    startColRelative= 2,
                    /**
                     * Specifies end row is relative
                     */
                    endRowRelative= 4,
                    /**
                     * Specifies end column is relative
                     */
                    endColRelative= 8,
                    /**
                     * Specifies row is relative
                     */
                    rowRelative= 5,
                    /**
                     * Specifies column is relative
                     */
                    colRelative= 10,
                    /**
                     * Specifies all reference is relative
                     */
                    allRelative= 15
                }

            }

            module CellState{

                export class CellStateManager{
                    /**
                     * Represents a cellstae manager that can manage all cell state in a sheet.
                     * @class
                     * @param {GC.Spread.Sheets.Worksheet} sheet The worksheet.
                     */
                    constructor(sheet:  GC.Spread.Sheets.Worksheet);
                    /**
                     * @description Add cell state instance for range , which will apply style of cellState when the state of range cell matched.
                     * @param range {GC.Spread.Sheets.Range} The scope range which will apply style of cellState.
                     * @param state {GC.Spread.Sheets.CellStatesType} Which state will use style.
                     * @param style {GC.Spread.Sheets.Style} Which style when the state is matched.
                     * @param sheetArea {GC.Spread.Sheets.SheetArea} The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                     * @example
                     * var style = new  GC.Spread.Sheets.Style();
                     * style.backColor = 'blue';
                     * style.foreColor = 'red';
                     * var range = new GC.Spread.Sheets.Range(1,2,3,3);
                     * sheet.cellStates.add(range,GC.Spread.Sheets.CellStatesType.hover,style,101);
                     */
                    add(range:  GC.Spread.Sheets.Range,  state:  GC.Spread.Sheets.CellStatesType,  style:  GC.Spread.Sheets.Style,  sheetArea?:  GC.Spread.Sheets.SheetArea): void;
                    /**
                     * @description Clear all style by range, after clear there are no style will be applied when the cell state is matched.
                     * @param range {GC.Spread.Sheets.Range} The scope range for clear.
                     * @param sheetArea {GC.Spread.Sheets.SheetArea} The sheet area. If this parameter is not provided, it defaults to <b>viewport</b>.
                     * @example
                     * var range = new GC.Spread.Sheets.Range(1,2,3,3);
                     * sheet.cellStates.clear(range, GC.Spread.Sheets.SheetArea.viewport);
                     */
                    clear(range:  GC.Spread.Sheets.Range,  sheetArea:  GC.Spread.Sheets.SheetArea): void;
                }
            }

            module CellTypes{

                export interface ICelltypeItemOption{
                    text: string;
                    value: number | string;
                }


                export interface IItemSpacing{
                    horizontal?: number;
                    vertical?: number;
                }

                /**
                 * Specifies the text alignment for check box cells.
                 * @enum {number}
                 * @example
                 * //This example creates a check box cell.
                 * var cellType1 = new GC.Spread.Sheets.CellTypes.CheckBox();
                 * cellType1.caption("caption");
                 * cellType1.textTrue("true");
                 * cellType1.textFalse("false");
                 * cellType1.textIndeterminate("indeterminate");
                 * cellType1.textAlign(GC.Spread.Sheets.CellTypes.CheckBoxTextAlign.bottom);
                 * cellType1.isThreeState(true);
                 * activeSheet.getCell(1, 1).cellType(cellType1);
                 */
                export enum CheckBoxTextAlign{
                    /**
                     * Specifies text is on top of the check box.
                     */
                    top= 0,
                    /**
                     * Specifies text is below the check box.
                     */
                    bottom= 1,
                    /**
                     * Specifies text is to the left of the check box.
                     */
                    left= 2,
                    /**
                     * Specifies text is to the right of the check box.
                     */
                    right= 3
                }

                /**
                 * Specifies the extend direction for radio list cells.
                 * @enum {number}
                 * @example
                 * //This example creates a radio button list cell.
                 * var cellType2 = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                 * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                 * cellType2.direction(GC.Spread.Sheets.CellTypes.Direction.vertical);
                 * activeSheet.getCell(2, 2).cellType(cellType2);
                 */
                export enum Direction{
                    /**
                     * Specifies item extend by horizontal.
                     */
                    horizontal= 0,
                    /**
                     * Specifies item extend by vertical.
                     */
                    vertical= 1
                }

                /**
                 * Represents the editor type of text cell type.
                 * @enum {number}
                 * @example
                 * //This example shows how to change the editor of text cell type to textarea.
                 * var tempStyle = new GC.Spread.Sheets.Style();
                 * tempStyle.cellType = new GC.Spread.Sheets.CellTypes.Text(GC.Spread.Sheets.CellTypes.EditorType.textarea);
                 * activeSheet.setDefaultStyle(tempStyle);
                 */
                export enum EditorType{
                    /**
                     *  Use textarea element as the editor of text cell type.
                     */
                    textarea= 0,
                    /**
                     *  Use editorable div element as the editor of text cell type.
                     */
                    editableDiv= 1
                }

                /**
                 *  Specifies what is written out to the data model for a selected item from
                 *  certain cell types that offer a selection of multiple values.
                 * @readonly
                 * @enum {number}
                 * @example
                 * //This example uses the EditorValueType enumeration.
                 * var cellType2 = new GC.Spread.Sheets.CellTypes.ComboBox();
                 * cellType2.items(["a","b","c"]);
                 * cellType2.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
                 * activeSheet.getCell(2, 2).cellType(cellType2);
                 */
                export enum EditorValueType{
                    /**
                     *  Writes the text value of the selected item to the model.
                     */
                    text= 0,
                    /**
                     * Writes the index of the selected item to the model.
                     */
                    index= 1,
                    /**
                     *  Writes the corresponding data value of the selected item to the model.
                     */
                    value= 2
                }

                /**
                 * Specifies the hyperlink's target type.
                 * @enum {number}
                 * @example
                 * //This example creates a hyperlink cell.
                 * var cellType = new GC.Spread.Sheets.CellTypes.HyperLink();
                 * cellType.linkColor("blue");
                 * cellType.visitedLinkColor("#FFFF00");
                 * cellType.text("GrapeCity");
                 * cellType.linkToolTip("Company Web Site");
                 * cellType.target(GC.Spread.Sheets.CellTypes.HyperLinkTargetType.self);
                 * activeSheet.getCell(0, 2).cellType(cellType).value("http://www.grapecity.com/");
                 */
                export enum HyperLinkTargetType{
                    /**
                     * Opens the hyperlinked document in a new window or tab.
                     */
                    blank= 0,
                    /**
                     * Opens the hyperlinked document in the same frame where the user clicked.
                     */
                    self= 1,
                    /**
                     * Opens the hyperlinked document in the parent frame.
                     */
                    parent= 2,
                    /**
                     * Opens the hyperlinked document in the full body of the window.
                     */
                    top= 3
                }

                /**
                 * Specifies the text selection mode for buttonList cells.
                 * @enum {number}
                 * @example
                 * //This example creates a buttonList cell.
                 * var cellType2 = new GC.Spread.Sheets.CellTypes.ButtonList();
                 * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                 * cellType2.selectionMode(GC.Spread.Sheets.CellTypes.SelectionMode.single);
                 * activeSheet.getCell(2, 2).cellType(cellType2);
                 */
                export enum SelectionMode{
                    /**
                     * Specifies selection mode is single.
                     */
                    single= 0,
                    /**
                     * Specifies selection mode is multiple.
                     */
                    multiple= 1
                }

                /**
                 * Specifies the text alignment for check box cells.
                 * @enum {number}
                 * @example
                 * //This example creates a check box cell.
                 * var cellType = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                 * cellType.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                 * cellType.textAlign(GC.Spread.Sheets.CellTypes.TextAlign.left);
                 * activeSheet.getCell(2, 2).cellType(cellType);
                 */
                export enum TextAlign{
                    /**
                     * Specifies text is on the top.
                     */
                    //top = 0, = 0,
                    /**
                     * Specifies text is on the bottom.
                     */
                    //bottom = 1, = 1,
                    /**
                     * Specifies text is on the left.
                     */
                    left= 2,
                    /**
                     * Specifies text is on the right.
                     */
                    right= 3
                }


                export class Base{
                    /**
                     * Represents the base class for the other cell type classes.
                     * @class
                     */
                    constructor();
                    /**
                     * Represents the type name string used for supporting serialization.
                     * @type {string}
                     */
                    typeName: string;
                    /**
                     * Activates the editor, including setting properties or attributes for the editor and binding events for the editor.
                     * @param {Object} editorContext The DOM element that was created by the createEditorElement method.
                     * @param {GC.Spread.Sheets.Style} cellStyle The cell's actual style.
                     * @param {GC.Spread.Sheets.Rect} cellRect The cell's layout information.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     */
                    activateEditor(editorContext:  HTMLElement,  cellStyle:  GC.Spread.Sheets.Style,  cellRect:  GC.Spread.Sheets.Rect,  context?:  any): void;
                    /**
                     * Creates a DOM element then returns it.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {HTMLElement} Returns a DOM element.
                     */
                    createEditorElement(context?:  any): HTMLElement;
                    /**
                     * Deactivates the editor, such as unbinding events for editor.
                     * @param {Object} editorContext The DOM element that was created by the createEditorElement method.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     */
                    deactivateEditor(editorContext:  HTMLElement,  context?:  any): void;
                    /**
                     * Focuses the editor DOM element.
                     * @param {HTMLElement} editorContext The DOM element that was created by the createEditorElement method.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     */
                    focus(editorContext:  HTMLElement,  context?:  any): void;
                    /**
                     * Formats a value with the specified format to a string.
                     * @param {Object} value The object value to format.
                     * @param {GC.Spread.Formatter.GeneralFormatter | string} format The format.
                     * @param {Object} formattedData the formatted data.
                     * @param {Array} [formattedData.content] - The formatted data array, each item is an object that has two properties type and value, And it may contain these types: 'number', 'text', 'fillingChar', 'placeholder', 'exponent', 'decimalSeparator', 'groupSeparator', 'numberPlaceholder', 'percent', 'permille' and 'currency'. For example: {type: 'number', value: '123'}.
                     * @param {string} [formattedData.conditionalForeColor] - The conditional foreground color.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {string} Returns the formatted string.
                     */
                    format(value:  any,  format:  GC.Spread.Formatter.GeneralFormatter | string,  formattedData?:  GC.Spread.Sheets.FormattedData,  context?:  any): string;
                    /**
                     * Loads the object state from the specified JSON string.
                     * @param {Object} settings The cell type data from deserialization.
                     */
                    fromJSON(settings:  any): void;
                    /**
                     * Gets a cell's height that can be used to handle the row's automatic fit.
                     * @param {Object} value The cell's value.
                     * @param {string} text The cell's text.
                     * @param {GC.Spread.Sheets.Style} cellStyle The cell's actual value.
                     * @param {number} zoomFactor The current sheet's zoom factor.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {number} Returns the cell's height that can be used to handle the row's automatic fit.
                     */
                    getAutoFitHeight(value:  any,  text:  string,  cellStyle:  GC.Spread.Sheets.Style,  zoomFactor:  number,  context?:  any): number;
                    /**
                     * Gets a cell's width that can be used to handle the column's automatic fit.
                     * @param {Object} value The cell's value.
                     * @param {string} text The cell's text.
                     * @param {GC.Spread.Sheets.Style} cellStyle The cell's actual value.
                     * @param {number} zoomFactor The current sheet's zoom factor.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {number} Returns the cell's width that can be used to handle the column's automatic fit.
                     */
                    getAutoFitWidth(value:  any,  text:  string,  cellStyle:  GC.Spread.Sheets.Style,  zoomFactor:  number,  context?:  any): number;
                    /**
                     * Gets the editor's value.
                     * @param {Object} editorContext The DOM element that was created by the createEditorElement method.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {Object} Returns the editor's value.
                     */
                    getEditorValue(editorContext:  HTMLElement,  context?:  any): any;
                    /**
                     * Gets the cell type's hit information.
                     * @param {number} x <i>x</i>-coordinate of pointer's current location relative to the canvas.
                     * @param {number} y <i>y</i>-coordinate of pointer's current location relative to the canvas.
                     * @param {GC.Spread.Sheets.Style} cellStyle The current cell's actual style.
                     * @param {GC.Spread.Sheets.Rect} cellRect The current cell's layout information.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {Object} Returns an object that contains the <i>x</i>, <i>y</i>, <i>row</i>, <i>col</i>, <i>cellRect</i>, and <i>sheetArea</i> parameters, and a value to indicate <i>isReservedLocation</i>.
                     * <i>isReservedLocation</i> is <c>true</c> if the hit test is in a special area that the cell type needs to handle; otherwise, <c>false</c>.
                     */
                    getHitInfo(x:  number,  y:  number,  cellStyle:  GC.Spread.Sheets.Style,  cellRect:  GC.Spread.Sheets.Rect,  context?:  any): GC.Spread.Sheets.IHitTestCellTypeHitInfo;
                    /**
                     * Whether the editing value has changed.
                     * @param {Object} oldValue Old editing value.
                     * @param {Object} newValue New editing value.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {boolean} <c>true</c> if oldValue equals newValue; otherwise, <c>false</c>.
                     */
                    isEditingValueChanged(oldValue:  any,  newValue:  any,  context?:  any): boolean;
                    /**
                     * Whether this cell type is aware of IME.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {boolean} <c>true</c> if the cell type is aware of IME; otherwise, <c>false</c>.
                     */
                    isImeAware(context?:  any): boolean;
                    /**
                     * Whether the cell type handles the keyboard event itself.
                     * @param {KeyboardEvent} e The KeyboardEvent.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {boolean} Returns <c>true</c> if the cell type handles the keyboard event itself; otherwise, <c>false</c>.
                     */
                    isReservedKey(e:  KeyboardEvent,  context?:  any): boolean;
                    /**
                     * Paints a cell on the canvas.
                     * @param {CanvasRenderingContext2D} ctx The canvas's two-dimensional context.
                     * @param {Object} value The cell's value.
                     * @param {number} x <i>x</i>-coordinate relative to the canvas.
                     * @param {number} y <i>y</i>-coordinate relative to the canvas.
                     * @param {number} w The cell's width.
                     * @param {number} h The cell's height.
                     * @param {GC.Spread.Sheets.Style} style The cell's actual style.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     */
                    paint(ctx:  CanvasRenderingContext2D,  value:  any,  x:  number,  y:  number,  w:  number,  h:  number,  style:  GC.Spread.Sheets.Style,  context?:  any): void;
                    /**
                     * Paints the cell content area on the canvas.
                     * @param {CanvasRenderingContext2D} ctx The canvas's two-dimensional context.
                     * @param {Object} value The cell's value.
                     * @param {number} x <i>x</i>-coordinate relative to the canvas.
                     * @param {number} y <i>y</i>-coordinate relative to the canvas.
                     * @param {number} w The cell content area's width.
                     * @param {number} h The cell content area's height.
                     * @param {GC.Spread.Sheets.Style} style The cell's actual style.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     */
                    paintContent(ctx:  CanvasRenderingContext2D,  value:  any,  x:  number,  y:  number,  w:  number,  h:  number,  style:  GC.Spread.Sheets.Style,  context?:  any): void;
                    /**
                     * Parses the text with the specified format string to an object.
                     * @param {string} text The parse text string.
                     * @param {string} formatStr The parse format string.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {Object} The parsed object.
                     */
                    parse(text:  string,  formatStr:  string,  context?:  any): any;
                    /**
                     * Processes key down in display mode.
                     * @param {KeyboardEvent} event The KeyboardEvent.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {boolean} Returns <c>true</c> if the process is successful; otherwise, <c>false</c>.
                     */
                    processKeyDown(event:  KeyboardEvent,  context?:  any): boolean;
                    /**
                     * Processes key up in display mode.
                     * @param {KeyboardEvent} event The KeyboardEvent.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {boolean} Returns <c>true</c> if the process is successful; otherwise, <c>false</c>.
                     */
                    processKeyUp(event:  KeyboardEvent,  context?:  any): boolean;
                    /**
                     * Processes mouse down in display mode.
                     * @param {Object} hitInfo The hit test information returned by the getHitInfo method. See the Remarks for more information.
                     * @returns {boolean} Returns <c>true</c> if the process is successful; otherwise, <c>false</c>.
                     */
                    processMouseDown(hitInfo:  GC.Spread.Sheets.IHitTestCellTypeHitInfo): boolean;
                    /**
                     * Processes mouse enter in display mode.
                     * @param {Object} hitInfo The hit test information returned by the getHitInfo method. See the Remarks for more information.
                     * @returns {boolean} Returns <c>true</c> if the process is successful; otherwise, <c>false</c>.
                     */
                    processMouseEnter(hitInfo:  GC.Spread.Sheets.IHitTestCellTypeHitInfo): boolean;
                    /**
                     * Processes mouse leave in display mode.
                     * @param {Object} hitInfo The hit test information returned by the getHitInfo method. See the Remarks for more information.
                     * @returns {boolean} Returns <c>true</c> if the process is successful; otherwise, <c>false</c>.
                     */
                    processMouseLeave(hitInfo:  GC.Spread.Sheets.IHitTestCellTypeHitInfo): boolean;
                    /**
                     * Processes mouse move in display mode.
                     * @param {Object} hitInfo The hit test information returned by the getHitInfo method. See the Remarks for more information.
                     * @returns {boolean} Returns <c>true</c> if the process is successful; otherwise, <c>false</c>.
                     */
                    processMouseMove(hitInfo:  GC.Spread.Sheets.IHitTestCellTypeHitInfo): boolean;
                    /**
                     * Processes mouse up in display mode.
                     * @param {Object} hitInfo The hit test information returned by the getHitInfo method. See the Remarks for more information.
                     * @returns {boolean} Returns <c>true</c> if the process is successful; otherwise, <c>false</c>.
                     */
                    processMouseUp(hitInfo:  GC.Spread.Sheets.IHitTestCellTypeHitInfo): boolean;
                    /**
                     * Selects all the text in the editor DOM element.
                     * @param {Object} editorContext The DOM element that was created by the createEditorElement method.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     */
                    selectAll(editorContext:  HTMLElement,  context?:  any): void;
                    /**
                     * Sets the editor's value.
                     * @param {Object} editorContext The DOM element that was created by the createEditorElement method.
                     * @param {Object} value The value returned from the active cell.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     */
                    setEditorValue(editorContext:  HTMLElement,  value:  any,  context?:  any): void;
                    /**
                     * Saves the object state to a JSON string.
                     * @returns {Object} The cell type data.
                     */
                    toJSON(): any;
                    /**
                     * Updates the editor's size.
                     * @param {Object} editorContext The DOM element that was created by the createEditorElement method.
                     * @param {GC.Spread.Sheets.Style} cellStyle The cell's actual style.
                     * @param {GC.Spread.Sheets.Rect} cellRect The cell's layout information.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     * @returns {GC.Spread.Sheets.Rect} Returns the new size for cell wrapper element, it should contain two properties 'width' and 'height'.
                     */
                    updateEditor(editorContext:  HTMLElement,  cellStyle:  GC.Spread.Sheets.Style,  cellRect:  GC.Spread.Sheets.Rect,  context?:  any): GC.Spread.Sheets.Rect;
                    /**
                     * Updates the cell wrapper element size.
                     * @param {HTMLElement} editorContext The DOM element that was created by the createEditorElement method.
                     * @param {GC.Spread.Sheets.Rect} editorBounds The cell wrapper element's new size.
                     * @param {number} editorBounds.x - The cell wrapper element's x position.
                     * @param {number} editorBounds.y - The cell wrapper element's y position.
                     * @param {number} editorBounds.width - The cell wrapper element's new width value.
                     * @param {number} editorBounds.height - The cell wrapper element's new height value.
                     * @param {GC.Spread.Sheets.Style} cellStyle The cell's actual style.
                     */
                    updateEditorContainer(editorContext:  HTMLElement,  editorBounds:  GC.Spread.Sheets.Rect,  cellStyle:  GC.Spread.Sheets.Style): void;
                    /**
                     * Updates the editor's ime-mode.
                     * @param {Object} editorContext The DOM element that was created by the createEditorElement method.
                     * @param {GC.Spread.Sheets.ImeMode} imeMode The ime-mode from cell's actual style.
                     * @param {Object} context The context associated with the cell type. See the Remarks for more information.
                     */
                    updateImeMode(editorContext:  HTMLElement,  imeMode:  GC.Spread.Sheets.ImeMode,  context?:  any): void;
                }

                export class Button extends Base{
                    /**
                     * Represents a button cell.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     * @example
                     * //This example creates a button cell.
                     * var spread = new GC.Spread.Sheets.Workbook();
                     * var sheet = spread.getActiveSheet();
                     * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                     * cellType.buttonBackColor("#FFFF00");
                     * cellType.text("this is a button");
                     * sheet.setCellType(1,1,cellType);
                     * //Bind event
                     * spread.bind(GC.Spread.Sheets.Events.ButtonClicked, function (e, args) {
                     *     var sheet = args.sheet, row = args.row, col = args.col;
                     *     var cellType = sheet.getCellType(row, col);
                     *     if (cellType instanceof GC.Spread.Sheets.CellTypes.Button) {
                     *         alert("Button Clicked");
                     *     }
                     * });
                     */
                    constructor();
                    /**
                     * Gets or sets the button's background color.
                     * @param {string} value The button's background color.
                     * @returns {string | GC.Spread.Sheets.CellTypes.Button} If no value is set, returns the background color; otherwise, returns the button cell type.
                     * @example
                     * //This example creates a button cell.
                     * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                     * cellType.buttonBackColor("#FFFF00");
                     * cellType.text("this is a button");
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     */
                    buttonBackColor(value?:  string): any;
                    /**
                     * Gets or sets the button's bottom margin in pixels relative to the cell.
                     * @param {number} value The button's bottom margin relative to the cell.
                     * @returns {number | GC.Spread.Sheets.CellTypes.Button} If no value is set, returns the bottom margin in pixels; otherwise, returns the button cell type.
                     * @example
                     * //This example creates a button cell and sets its margins.
                     * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                     * cellType.buttonBackColor("#FFFF00");
                     * cellType.text("this is a button");
                     * cellType.marginTop(5).marginRight(8).marginBottom(10).marginLeft(12);
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     * activeSheet.setColumnWidth(2, 120.0,GC.Spread.Sheets.SheetArea.viewport);
                     * activeSheet.setRowHeight(0, 90.0,GC.Spread.Sheets.SheetArea.viewport);
                     */
                    marginBottom(value?:  number): any;
                    /**
                     * Gets or sets the button's left margin in pixels relative to the cell.
                     * @param {number} value The button's left margin relative to the cell.
                     * @returns {number | GC.Spread.Sheets.CellTypes.Button} If no value is set, returns the left margin in pixels; otherwise, returns the button cell type.
                     * @example
                     * //This example creates a button cell and sets its margins.
                     * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                     * cellType.buttonBackColor("#FFFF00");
                     * cellType.text("this is a button");
                     * cellType.marginTop(5).marginRight(8).marginBottom(10).marginLeft(12);
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     * activeSheet.setColumnWidth(2, 120.0,GC.Spread.Sheets.SheetArea.viewport);
                     * activeSheet.setRowHeight(0, 90.0,GC.Spread.Sheets.SheetArea.viewport);
                     */
                    marginLeft(value?:  number): any;
                    /**
                     * Gets or sets the button's right margin in pixels relative to the cell.
                     * @param {number} value The button's right margin relative to the cell.
                     * @returns {number | GC.Spread.Sheets.CellTypes.Button} If no value is set, returns the right margin in pixels; otherwise, returns the button cell type.
                     * @example
                     * //This example creates a button cell and sets its margins.
                     * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                     * cellType.buttonBackColor("#FFFF00");
                     * cellType.text("this is a button");
                     * cellType.marginTop(5).marginRight(8).marginBottom(10).marginLeft(12);
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     * activeSheet.setColumnWidth(2, 120.0,GC.Spread.Sheets.SheetArea.viewport);
                     * activeSheet.setRowHeight(0, 90.0,GC.Spread.Sheets.SheetArea.viewport);
                     */
                    marginRight(value?:  number): any;
                    /**
                     * Gets or sets the button's top margin in pixels relative to the cell.
                     * @param {number} value The button's top margin relative to the cell.
                     * @returns {number | GC.Spread.Sheets.CellTypes.Button} If no value is set, returns the top margin in pixels; otherwise, returns the button cell type.
                     * @example
                     * //This example creates a button cell and sets its margins.
                     * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                     * cellType.buttonBackColor("#FFFF00");
                     * cellType.text("this is a button");
                     * cellType.marginTop(5).marginRight(8).marginBottom(10).marginLeft(12);
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     * activeSheet.setColumnWidth(2, 120.0,GC.Spread.Sheets.SheetArea.viewport);
                     * activeSheet.setRowHeight(0, 90.0,GC.Spread.Sheets.SheetArea.viewport);
                     */
                    marginTop(value?:  number): any;
                    /**
                     * Gets or sets the button's content.
                     * @param {string} value The button's content.
                     * @returns {string | GC.Spread.Sheets.CellTypes.Button} If no value is set, returns the content; otherwise, returns the button cell type.
                     * @example
                     * //This example creates a button cell.
                     * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                     * cellType.buttonBackColor("#FFFF00");
                     * cellType.text("this is a button");
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     */
                    text(value?:  string): any;
                }

                export class ButtonList extends Base{
                    /**
                     * Represents an editable buttonList cell.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     * @example
                     * //This example creates a buttonList cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    constructor();
                    /**
                     * Gets or sets the buttonList list's orders.
                     * @param {GC.Spread.Sheets.CellTypes.Direction} value Whether the order is vertical.
                     * @returns {GC.Spread.Sheets.CellTypes.Direction | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns whether the buttonList list's orders is vertical; otherwise, returns the checkbox list cellType.
                     * @example
                     * //This example creates a buttonList list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.direction(GC.Spread.Sheets.CellTypes.Direction.vertical);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    direction(value?:  GC.Spread.Sheets.CellTypes.Direction): any;
                    /**
                     * Gets or sets the buttonList list's layout is autofit.
                     * @param {boolean} value Whether the layout is autofit.
                     * @returns {boolean | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns whether the layout is autofit, returns the buttonList list cellType.
                     * @example
                     * //This example creates a buttonList list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.isFlowLayout(true);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    isFlowLayout(value?:  boolean): any;
                    /**
                     * Gets or sets the items for the buttonList list.
                     * @param {Array} items The items for the buttonList list.
                     * @returns {Array | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns the items array; otherwise, returns the buttonList list cellType.
                     * @example
                     * //This example creates a buttonList list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    items(items?:  GC.Spread.Sheets.CellTypes.ICelltypeItemOption[] | string[]): any;
                    /**
                     * Gets or sets the space for two items in the buttonList.
                     * @param {object} value the space for two items in the buttonList.
                     * @returns {object | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns the space for two items in the buttonList; otherwise, returns the buttonList cellType.
                     * @example
                     * //This example creates a buttonList cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.itemSpacing({
                     *      horizontal:80,
                     *      vertical:20
                     * });
                     */
                    itemSpacing(value?:  GC.Spread.Sheets.CellTypes.IItemSpacing): any;
                    /**
                     * Gets or sets the items for the buttonList list's column count.
                     * @param {number} value The column count for the buttonList list.
                     * @returns {number | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns the column count; otherwise, returns the buttonList list cellType.
                     * @example
                     * //This example creates a buttonList list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.maxColumnCount(2);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    maxColumnCount(value?:  number): any;
                    /**
                     * Gets or sets the items for the buttonList list's row count.
                     * @param {number} value The row count for the buttonList list.
                     * @returns {number | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns the row count; otherwise, returns the buttonList list cellType.
                     * @example
                     * //This example creates buttonList list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.maxRowCount(2);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    maxRowCount(value?:  number): any;
                    /**
                     * Gets or sets the buttonList's padding in pixels relative to the cell.
                     * @param {string} value The buttonList's padding relative to the cell.
                     * @returns {string | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns the padding in pixels; otherwise, returns the buttonList cell type.
                     * @example
                     * //This example creates a buttonList cell and sets its padding.
                     * var cellType = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType.padding("5");
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     * activeSheet.setColumnWidth(2, 120.0,GC.Spread.Sheets.SheetArea.viewport);
                     * activeSheet.setRowHeight(0, 90.0,GC.Spread.Sheets.SheetArea.viewport);
                     */
                    padding(value?:  string): any;
                    /**
                     * Gets or sets the selected buttonList's background color.
                     * @param {string} value The selected buttonList's background color.
                     * @returns {string | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns the background color; otherwise, returns the buttonList cell type.
                     * @example
                     * //This example creates a buttonList cell.
                     * var cellType = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType.selectedBackColor("#FFFF00");
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     */
                    selectedBackColor(value?:  string): any;
                    /**
                     * Gets or sets the selected buttonList's fore color.
                     * @param {string} value The selected buttonList's fore color.
                     * @returns {string | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns the fore color; otherwise, returns the buttonList cell type.
                     * @example
                     * //This example creates a buttonList cell.
                     * var cellType = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType.selectedForeColor("#FFFF00");
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     */
                    selectedForeColor(value?:  string): any;
                    /**
                     * Gets or sets the buttonList's select mode.
                     * @param {GC.Spread.Sheets.CellTypes.SelectionMode} value The selected buttonList's select mode.
                     * @returns {GC.Spread.Sheets.CellTypes.SelectionMode | GC.Spread.Sheets.CellTypes.ButtonList} If no value is set, returns the select mode; otherwise, returns the buttonList cell type.
                     * @example
                     * //This example creates a buttonList cell.
                     * var cellType = new GC.Spread.Sheets.CellTypes.ButtonList();
                     * cellType.selectionMode(GC.Spread.Sheets.CellTypes.SelectionMode.single);
                     * activeSheet.getCell(0, 2).cellType(cellType);
                     */
                    selectionMode(value?:  GC.Spread.Sheets.CellTypes.SelectionMode): any;
                }

                export class CheckBox extends Base{
                    /**
                     * Represents a check box cell.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     * @example
                     * //This example creates a check box cell.
                     * var cellType1 = new GC.Spread.Sheets.CellTypes.CheckBox();
                     * cellType1.caption("caption");
                     * cellType1.textTrue("true");
                     * cellType1.textFalse("false");
                     * cellType1.textIndeterminate("indeterminate");
                     * cellType1.textAlign(GC.Spread.Sheets.CellTypes.CheckBoxTextAlign.bottom);
                     * cellType1.isThreeState(true);
                     * activeSheet.getCell(1, 1).cellType(cellType1);
                     */
                    constructor();
                    /**
                     * Gets or sets the caption of the cell type.
                     * @param {string} value The caption of the cell type.
                     * @returns {string | GC.Spread.Sheets.CellTypes.CheckBox} If no value is set, returns the caption; otherwise, returns the check box cell type.
                     * @example
                     * //This example creates a check box cell.
                     * var cellType1 = new GC.Spread.Sheets.CellTypes.CheckBox();
                     * cellType1.caption("caption");
                     * cellType1.textTrue("true");
                     * cellType1.textFalse("false");
                     * cellType1.textIndeterminate("indeterminate");
                     * cellType1.textAlign(GC.Spread.Sheets.CellTypes.CheckBoxTextAlign.bottom);
                     * cellType1.isThreeState(true);
                     * activeSheet.getCell(1, 1).cellType(cellType1);
                     */
                    caption(value?:  string): any;
                    /**
                     * Gets or sets a value that indicates whether the check box supports three states.
                     * @param {boolean} value Whether the check box supports three states.
                     * @returns {boolean | GC.Spread.Sheets.CellTypes.CheckBox} If no value is set, returns whether the check box supports three states; otherwise, returns the check box cell type.
                     * @example
                     * //This example creates a check box cell.
                     * var cellType1 = new GC.Spread.Sheets.CellTypes.CheckBox();
                     * cellType1.caption("caption");
                     * cellType1.textTrue("true");
                     * cellType1.textFalse("false");
                     * cellType1.textIndeterminate("indeterminate");
                     * cellType1.textAlign(GC.Spread.Sheets.CellTypes.CheckBoxTextAlign.bottom);
                     * cellType1.isThreeState(true);
                     * activeSheet.getCell(1, 1).cellType(cellType1);
                     */
                    isThreeState(value?:  boolean): any;
                    /**
                     * Gets or sets the text alignment relative to the check box.
                     * @param {GC.Spread.Sheets.CellTypes.CheckBoxTextAlign} value The text alignment relative to the check box.
                     * @returns {GC.Spread.Sheets.CellTypes.CheckBoxTextAlign | GC.Spread.Sheets.CellTypes.CheckBox} If no value is set, returns the text alignment relative to the check box; otherwise, returns the check box cell type.
                     * @example
                     * //This example creates a check box cell.
                     * var cellType1 = new GC.Spread.Sheets.CellTypes.CheckBox();
                     * cellType1.caption("caption");
                     * cellType1.textTrue("true");
                     * cellType1.textFalse("false");
                     * cellType1.textIndeterminate("indeterminate");
                     * cellType1.textAlign(GC.Spread.Sheets.CellTypes.CheckBoxTextAlign.bottom);
                     * cellType1.isThreeState(true);
                     * activeSheet.getCell(1, 1).cellType(cellType1);
                     */
                    textAlign(value?:  GC.Spread.Sheets.CellTypes.CheckBoxTextAlign): any;
                    /**
                     * Gets or sets the text in the cell when the cell's value is <c>false</c>.
                     * @param {string} value The text in the cell when the cell's value is <c>false</c>.
                     * @returns {string | GC.Spread.Sheets.CellTypes.CheckBox} If no value is set, returns the text in the cell when the cell's value is <c>false</c>. If a value is set, returns the check box cell type.
                     * @example
                     * //This example creates a check box cell.
                     * var cellType1 = new GC.Spread.Sheets.CellTypes.CheckBox();
                     * cellType1.caption("caption");
                     * cellType1.textTrue("true");
                     * cellType1.textFalse("false");
                     * cellType1.textIndeterminate("indeterminate");
                     * cellType1.textAlign(GC.Spread.Sheets.CellTypes.CheckBoxTextAlign.bottom);
                     * cellType1.isThreeState(true);
                     * activeSheet.getCell(1, 1).cellType(cellType1);
                     */
                    textFalse(value?:  string): any;
                    /**
                     * Gets or sets the text in the cell when the cell's value is indeterminate (neither <c>true</c> nor <c>false</c>).
                     * @param {string} value The text in the cell when the cell's value is indeterminate.
                     * @returns {string | GC.Spread.Sheets.CellTypes.CheckBox} If no value is set, returns the text in the cell when the cell's value is indeterminate. If a value is set, returns the check box cell type.
                     * @example
                     * //This example creates a check box cell.
                     * var cellType1 = new GC.Spread.Sheets.CellTypes.CheckBox();
                     * cellType1.caption("caption");
                     * cellType1.textTrue("true");
                     * cellType1.textFalse("false");
                     * cellType1.textIndeterminate("indeterminate");
                     * cellType1.textAlign(GC.Spread.Sheets.CellTypes.CheckBoxTextAlign.bottom);
                     * cellType1.isThreeState(true);
                     * activeSheet.getCell(1, 1).cellType(cellType1);
                     */
                    textIndeterminate(value?:  string): any;
                    /**
                     * Gets or sets the text in the cell when the cell's value is <c>true</c>.
                     * @param {string} value The text when the cell's value is <c>true</c>.
                     * @returns {string | GC.Spread.Sheets.CellTypes.CheckBox} If no value is set, returns the text when the cell's value is <c>true</c>. If a value is set, returns the check box cell type.
                     * @example
                     * //This example creates a check box cell.
                     * var cellType1 = new GC.Spread.Sheets.CellTypes.CheckBox();
                     * cellType1.caption("caption");
                     * cellType1.textTrue("true");
                     * cellType1.textFalse("false");
                     * cellType1.textIndeterminate("indeterminate");
                     * cellType1.textAlign(GC.Spread.Sheets.CellTypes.CheckBoxTextAlign.bottom);
                     * cellType1.isThreeState(true);
                     * activeSheet.getCell(1, 1).cellType(cellType1);
                     */
                    textTrue(value?:  string): any;
                }

                export class CheckBoxList extends Base{
                    /**
                     * Represents an editable CheckBoxList cell.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     * @example
                     * //This example creates a CheckBoxList cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.CheckBoxList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    constructor();
                    /**
                     * Gets or sets the checkbox list's orders.
                     * @param {GC.Spread.Sheets.CellTypes.Direction} value Whether the order is vertical.
                     * @returns {GC.Spread.Sheets.CellTypes.Direction | GC.Spread.Sheets.CellTypes.CheckBoxList} If no value is set, returns whether the checkbox list's orders is vertical; otherwise, returns the checkbox list cellType.
                     * @example
                     * //This example creates a checkbox list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.CheckBoxList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.direction(GC.Spread.Sheets.CellTypes.Direction.vertical);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    direction(value?:  GC.Spread.Sheets.CellTypes.Direction): any;
                    /**
                     * Gets or sets the checkbox list's layout is autofit.
                     * @param {boolean} value Whether the layout is autofit.
                     * @returns {boolean | GC.Spread.Sheets.CellTypes.CheckBoxList} If no value is set, returns whether the layout is autofit, returns the checkbox list cellType.
                     * @example
                     * //This example creates a checkbox list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.CheckBoxList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.isFlowLayout(true);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    isFlowLayout(value?:  boolean): any;
                    /**
                     * Gets or sets the items for the checkbox list.
                     * @param {Array} items The items for the checkbox list.
                     * @returns {Array | GC.Spread.Sheets.CellTypes.CheckBoxList} If no value is set, returns the items array; otherwise, returns the checkbox list cellType.
                     * @example
                     * //This example creates a checkbox list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.CheckBoxList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    items(items?:  GC.Spread.Sheets.CellTypes.ICelltypeItemOption[] | string[]): any;
                    /**
                     * Gets or sets the space for two items in the checkbox list.
                     * @param {object} value the space for two items in the checkbox list.
                     * @returns {object | GC.Spread.Sheets.CellTypes.CheckBoxList} If no value is set, returns the space for two items in the checkbox list; otherwise, returns the checkbox list cellType.
                     * @example
                     * //This example creates a checkbox list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.CheckBoxList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.itemSpacing({
                     *      horizontal:80,
                     *      vertical:20
                     * });
                     */
                    itemSpacing(value?:  GC.Spread.Sheets.CellTypes.IItemSpacing): any;
                    /**
                     * Gets or sets the items for the checkbox list's column count.
                     * @param {number} value The column count for the checkbox list.
                     * @returns {number | GC.Spread.Sheets.CellTypes.CheckBoxList} If no value is set, returns the column count; otherwise, returns the checkbox list cellType.
                     * @example
                     * //This example creates a checkbox list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.CheckBoxList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.maxColumnCount(2);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    maxColumnCount(value?:  number): any;
                    /**
                     * Gets or sets the items for the checkbox list's row count.
                     * @param {number} value The row count for the checkbox list.
                     * @returns {number | GC.Spread.Sheets.CellTypes.CheckBoxList} If no value is set, returns the row count; otherwise, returns the checkbox list cellType.
                     * @example
                     * //This example creates checkbox list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.CheckBoxList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.maxRowCount(2);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    maxRowCount(value?:  number): any;
                    /**
                     * Gets or sets the text of checkbox's position, only support left and right .
                     * @param {GC.Spread.Sheets.CellTypes.TextAlign} value the text of checkbox's position.
                     * @returns {GC.Spread.Sheets.CellTypes.TextAlign | GC.Spread.Sheets.CellTypes.CheckBoxList} If no value is set, returns the text of checkbox's position, returns the checkbox list cellType.
                     * @example
                     * //This example creates a checkbox list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.CheckBoxList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.textAlign(GC.Spread.Sheets.CellTypes.TextAlign.left);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    textAlign(value?:  GC.Spread.Sheets.CellTypes.TextAlign): any;
                }

                export class ColumnHeader extends Base{
                    /**
                     * Represents the painter of the column header cells.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     */
                    constructor();
                }

                export class ComboBox extends Base{
                    /**
                     * Represents an editable combo box cell.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     * @example
                     * //This example creates a combo box cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ComboBox();
                     * cellType2.items(["a","b","c"]);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    constructor();
                    /**
                     * Gets or sets whether the combo box is editable.
                     * @param {boolean} value Whether the combo box is editable.
                     * @returns {boolean | GC.Spread.Sheets.CellTypes.ComboBox} If no value is set, returns whether the combo box is editable; otherwise, returns the combo box cellType.
                     * @example
                     * //This example sets the editable method.
                     * var items2 = ["a", "ab", "abc", "apple", "boy", "cat", "dog"];
                     * var eComboBoxCellType = new GC.Spread.Sheets.CellTypes.ComboBox().items(items2).editable(true);
                     * activeSheet.getCell(1, 3).cellType(eComboBoxCellType);
                     * activeSheet.setColumnWidth(0,120);
                     * activeSheet.setColumnWidth(2,120);
                     */
                    editable(value?:  boolean): any;
                    /**
                     * Gets or sets the value that is written to the underlying data model.
                     * @param {GC.Spread.Sheets.CellTypes.EditorValueType} value The type of editor value.
                     * @returns {GC.Spread.Sheets.CellTypes.EditorValueType | GC.Spread.Sheets.CellTypes.ComboBox} If no value is set, returns the type of editor value; otherwise, returns the combo box cellType.
                     * @example
                     * //This example gets the type.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ComboBox();
                     * cellType2.items(["a","b","c"]);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     * alert(cellType2.editorValueType());
                     */
                    editorValueType(value?:  GC.Spread.Sheets.CellTypes.EditorValueType): any;
                    /**
                     * Gets or sets the height of each item.
                     * @param {number} value The height of each item.
                     * @returns {number | GC.Spread.Sheets.CellTypes.ComboBox} If no value is set, returns the height of each item; otherwise, returns the  combo box cellType.
                     * @example
                     * //This example sets the item height.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ComboBox();
                     * cellType2.items(["a","b","c"]);
                     * cellType2.itemHeight(30);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    itemHeight(value?:  number): any;
                    /**
                     * Gets or sets the items for the drop-down list in the combo box.
                     * @param {Array} items The items for the combo box.
                     * @returns {Array | GC.Spread.Sheets.CellTypes.ComboBox} If no value is set, returns the items array; otherwise, returns the  combo box cellType.
                     * @example
                     * //This example creates a combo box cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ComboBox();
                     * cellType2.items(["a","b","c"]);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    items(items?:  any[]): any;
                    /**
                     * Gets or sets the maximum item count of the drop-down list per page.
                     * @param {number} value The maximum item count of the drop-down list per page.
                     * @returns {number | GC.Spread.Sheets.CellTypes.ComboBox} If no value is set, returns the maximum item count of the drop-down list per page; otherwise, returns the  combo box cellType.
                     * @example
                     * //This example shows three items in the list at a time.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.ComboBox();
                     * cellType2.items(["a", "b", "c", "d", "e", "f", "g", "h"]);
                     * cellType2.maxDropDownItems(3);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     * });
                     */
                    maxDropDownItems(value?:  number): any;
                }

                export class Corner extends Base{
                    /**
                     * Represents the painter of the corner cell.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     */
                    constructor();
                }

                export class HyperLink extends Base{
                    /**
                     * Represents the hyperlink cell.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     * @example
                     * var cellType = new GC.Spread.Sheets.CellTypes.HyperLink();
                     * cellType.linkColor("blue");
                     * cellType.visitedLinkColor("#FF2235");
                     * cellType.text("GrapeCity");
                     * cellType.linkToolTip("Company Web Site");
                     * activeSheet.getCell(1, 1).cellType(cellType).value("http://www.grapecity.com/");
                     * activeSheet.getCell(1, -1).height(30);
                     * @example
                     * var cellType = new GC.Spread.Sheets.CellTypes.HyperLink();
                     * cellType.linkColor("blue");
                     * cellType.visitedLinkColor("#FF2235");
                     * cellType.text("GrapeCity");
                     * cellType.linkToolTip("Company Web Site");
                     * activeSheet.getCell(0, 2).cellType(cellType).value("formula.html");
                     */
                    constructor();
                    /**
                     *  Gets or sets whether to move to the active cell when clicked.
                     * @param {boolean} value Whether to move to the active cell when clicked.
                     * @returns {boolean | GC.Spread.Sheets.CellTypes.HyperLink} If no value is set, returns a value that indicates whether to move to the active cell; otherwise, returns the hyperlink cell type.
                     * @example
                     * //This example uses the activeOnClick method.
                     * var h1 = new GC.Spread.Sheets.CellTypes.HyperLink();
                     * h1.text("GrapeCity");
                     * h1.linkToolTip("link to GrapeCity Web page");
                     * h1.linkColor("rgb(0, 100, 200)");
                     * h1.visitedLinkColor("rgb(0, 200, 100)");
                     * h1.activeOnClick(true);
                     * activeSheet.setCellType(1, 1, h1);
                     * activeSheet.getCell(1, 1, GC.Spread.Sheets.SheetArea.viewport).value("http://www.grapecity.com/").hAlign(GC.Spread.Sheets.HorizontalAlign.center);
                     */
                    activeOnClick(value?:  boolean): any;
                    /**
                     * Gets or sets the color of the hyperlink.
                     * @param {string} value The hyperlink color.
                     * @returns {string | GC.Spread.Sheets.CellTypes.HyperLink} If no value is set, returns the hyperlink color; otherwise, returns the hyperLink cell type.
                     * @example
                     * //This example creates a hyperlink cell.
                     * cellType = new GC.Spread.Sheets.CellTypes.HyperLink();
                     * cellType.linkColor("blue");
                     * cellType.visitedLinkColor("#FF2235");
                     * cellType.text("GrapeCity");
                     * cellType.linkToolTip("Company Web Site");
                     * activeSheet.getCell(1, 1).cellType(cellType).value("http://www.grapecity.com/");
                     * activeSheet.getCell(1, -1).height(30);
                     */
                    linkColor(value?:  string): any;
                    /**
                     * Gets or sets the tooltip for the hyperlink.
                     * @param {string} value The tooltip text.
                     * @returns {string | GC.Spread.Sheets.CellTypes.HyperLink} If no value is set, returns the tooltip text; otherwise, returns the hyperLink cell type.
                     * @example
                     * //This example creates a hyperlink cell.
                     * cellType = new GC.Spread.Sheets.CellTypes.HyperLink();
                     * cellType.linkColor("blue");
                     * cellType.visitedLinkColor("#FF2235");
                     * cellType.text("GrapeCity");
                     * cellType.linkToolTip("Company Web Site");
                     * activeSheet.getCell(1, 1).cellType(cellType).value("http://www.grapecity.com/");
                     * activeSheet.getCell(1, -1).height(30);
                     */
                    linkToolTip(value?:  string): any;
                    /**
                     *  Gets or sets the callback of the hyperlink, If execute the function will represent the context for the callback.
                     * @param {Function} value The callback of the hyperlink.
                     * @returns {Function | GC.Spread.Sheets.CellTypes.HyperLink} If no value is set, return a value that indicates the callback of the hyperlink; otherwise, returns the hyperlink cell type.
                     * @example
                     * //This example sets the tab color when selecting the hyperlink.
                     * var h2 = new GC.Spread.Sheets.CellTypes.HyperLink();
                     * h2.text("set sheet tab style");
                     * h2.linkToolTip("set sheet tab style");
                     * h2.linkColor("blue");
                     * h2.visitedLinkColor("#FF2235");
                     * activeSheet.getCell(2, 1).cellType(h2).value("set sheet tab style").hAlign(GC.Spread.Sheets.HorizontalAlign.center);
                     * h2.activeOnClick(true);
                     *             h2.onClickAction(function () {
                     *                 var setSheetTabColor = {
                     *                     canUndo: true,
                     *                     execute: function (context, options, isUndo) {
                     *                         activeSheet.name("Hyperlink");
                     *                         activeSheet.options.sheetTabColor = "red";
                     *                     }
                     *                 };
                     *                 var commandManager = spread.commandManager();
                     *                 var commandName = "setSheetTabStyle";
                     *                 commandManager.register(commandName, setSheetTabColor, null, false, false, false, false);
                     *                 commandManager.execute({cmd: commandName})
                     *             });
                     */
                    onClickAction(value?:  Function): any;
                    /**
                     *  Gets or sets the type for the hyperlink's target.
                     * @param {GC.Spread.Sheets.CellTypes.HyperLinkTargetType} value The hyperlink's target type.
                     * @returns {GC.Spread.Sheets.CellTypes.HyperLinkTargetType | GC.Spread.Sheets.CellTypes.HyperLink} If no value is set, returns the hyperlink's target type; otherwise, returns the hyperLink cell type.
                     * @example
                     * //This example creates a hyperlink cell.
                     * var cellType = new GC.Spread.Sheets.CellTypes.HyperLink();
                     * cellType.linkColor("blue");
                     * cellType.visitedLinkColor("#FFFF00");
                     * cellType.text("GrapeCity");
                     * cellType.linkToolTip("Company Web Site");
                     * cellType.target(GC.Spread.Sheets.CellTypes.HyperLinkTargetType.self);
                     * activeSheet.getCell(0, 2).cellType(cellType).value("http://www.grapecity.com/");
                     */
                    target(value?:  GC.Spread.Sheets.CellTypes.HyperLinkTargetType): any;
                    /**
                     * Gets or sets the text string for the hyperlink.
                     * @param {string} value The text displayed in the hyperlink.
                     * @returns {string | GC.Spread.Sheets.CellTypes.HyperLink} If no value is set, returns the text in the hyperlink; otherwise, returns the hyperLink cell type.
                     * @example
                     * //This example creates a hyperlink cell.
                     * var cellType = new GC.Spread.Sheets.CellTypes.HyperLink();
                     * cellType.linkColor("blue");
                     * cellType.visitedLinkColor("#FFFF00");
                     * cellType.text("GrapeCity");
                     * cellType.linkToolTip("Company Web Site");
                     * cellType.target(GC.Spread.Sheets.CellTypes.HyperLinkTargetType.self);
                     * activeSheet.getCell(0, 2).cellType(cellType).value("http://www.grapecity.com/");
                     */
                    text(value?:  string): any;
                    /**
                     * Gets or sets the color of visited links.
                     * @param {string} value The visited link color.
                     * @returns {string | GC.Spread.Sheets.CellTypes.HyperLink} If no value is set, returns the visited link color; otherwise, returns the hyperLink cell type.
                     * @example
                     * //This example creates a hyperlink cell.
                     * var cellType = new GC.Spread.Sheets.CellTypes.HyperLink();
                     * cellType.linkColor("blue");
                     * cellType.visitedLinkColor("#FFFF00");
                     * cellType.text("GrapeCity");
                     * cellType.linkToolTip("Company Web Site");
                     * cellType.target(GC.Spread.Sheets.CellTypes.HyperLinkTargetType.self);
                     * activeSheet.getCell(0, 2).cellType(cellType).value("http://www.grapecity.com/");
                     */
                    visitedLinkColor(value?:  string): any;
                }

                export class RadioButtonList extends Base{
                    /**
                     * Represents an editable radio button list cell.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     * @example
                     * //This example creates a radio button list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    constructor();
                    /**
                     * Gets or sets the radio button list's orders.
                     * @param {GC.Spread.Sheets.CellTypes.Direction} value the order is vertical.
                     * @returns {GC.Spread.Sheets.CellTypes.Direction | GC.Spread.Sheets.CellTypes.RadioButtonList} If no value is set, returns whether the radio button list's orders is vertical; otherwise, returns the radio button list cellType.
                     * @example
                     * //This example creates a radio button list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.direction(GC.Spread.Sheets.CellTypes.Direction.vertical);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    direction(value?:  GC.Spread.Sheets.CellTypes.Direction): any;
                    /**
                     * Gets or sets the radio button list's layout is autofit.
                     * @param {boolean} value Whether the layout is autofit.
                     * @returns {boolean | GC.Spread.Sheets.CellTypes.RadioButtonList} If no value is set, returns whether the layout is autofit, returns the radio button list cellType.
                     * @example
                     * //This example creates a radio button list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.isFlowLayout(true);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    isFlowLayout(value?:  boolean): any;
                    /**
                     * Gets or sets the items for the radio-button list.
                     * @param {Array} items The items for the radio button list.
                     * @returns {Array | GC.Spread.Sheets.CellTypes.RadioButtonList} If no value is set, returns the items array; otherwise, returns the radio button list cellType.
                     * @example
                     * //This example creates a radio button list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    items(items?:  GC.Spread.Sheets.CellTypes.ICelltypeItemOption[] | string[]): any;
                    /**
                     * Gets or sets the space for two items in the radio button list.
                     * @param {object} value the space for two items in the radio button list.
                     * @returns {object | GC.Spread.Sheets.CellTypes.RadioButtonList} If no value is set, returns the space for two items in the radio button list; otherwise, returns the radio button list cellType.
                     * @example
                     * //This example creates a radio button list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.itemSpacing({
                     *      horizontal:80,
                     *      vertical:20
                     * });
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    itemSpacing(value?:  GC.Spread.Sheets.CellTypes.IItemSpacing): any;
                    /**
                     * Gets or sets the items for the radio button list's column count.
                     * @param {number} value The column count for the radio button list.
                     * @returns {number | GC.Spread.Sheets.CellTypes.RadioButtonList} If no value is set, returns the column count; otherwise, returns the radio button list cellType.
                     * @example
                     * //This example creates a radio button list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.maxColumnCount(2);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    maxColumnCount(value?:  number): any;
                    /**
                     * Gets or sets the items for the radio button list's row count.
                     * @param {number} value The row count for the radio button list.
                     * @returns {number | GC.Spread.Sheets.CellTypes.RadioButtonList} If no value is set, returns the row count; otherwise, returns the radio button list cellType.
                     * @example
                     * //This example creates radio button list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.maxRowCount(2);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    maxRowCount(value?:  number): any;
                    /**
                     * Gets or sets the text of radio button's position, only support left and right .
                     * @param {GC.Spread.Sheets.CellTypes.TextAlign} value the text of radio button's position.
                     * @returns {GC.Spread.Sheets.CellTypes.TextAlign | GC.Spread.Sheets.CellTypes.RadioButtonList} If no value is set, returns the text of radio button's position, returns the radio button list cellType.
                     * @example
                     * //This example creates a radio button list cell.
                     * var cellType2 = new GC.Spread.Sheets.CellTypes.RadioButtonList();
                     * cellType2.items([{text:"a",value:1},{text:"b",value:2},{text:"c",value:3}]);
                     * cellType2.textAlign(GC.Spread.Sheets.CellTypes.TextAlign.left);
                     * activeSheet.getCell(2, 2).cellType(cellType2);
                     */
                    textAlign(value?:  GC.Spread.Sheets.CellTypes.TextAlign): any;
                }

                export class RangeTemplate extends GC.Spread.Sheets.CellTypes.Base{
                    /**
                     * @description RangeTemplate provide a template from a range of referenced worksheet,it can apply to a cell.It will render the cell same as the tempalte and fill data different.If the param row,col, rowCount, colCount not set , it will use the whole sheet as the range scope.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @param {GC.Spread.Sheets.Worksheet} sheet  the referenced worksheet, the sheet could be an individual sheet outside the workbook.
                     * @param {number} [row] the template scope start row.
                     * @param {number} [col] the template scope start col.
                     * @param {number} [rowCount] the template scope row count.
                     * @param {number} [colCount] the template scope col count.
                     * @class
                     * @example
                     * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"), { sheetCount: 2 });
                     * var sheet = spread.getActiveSheet();
                     * var sheet2 = spread.getSheetFromName("Sheet2");
                     * var celltype = new GC.Spread.Sheets.CellTypes.RangeTemplate(sheet2, 0, 0, 6, 4);
                     * sheet.getRange(0,0,2,2).cellType(celltype);
                     */
                    constructor(sheet:  GC.Spread.Sheets.Worksheet,  row?:  number,  col?:  number,  rowCount?:  number,  colCount?:  number);
                }

                export class RowHeader extends Base{
                    /**
                     * Represents the painter of the row header cells.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     */
                    constructor();
                }

                export class Text extends Base{
                    /**
                     * Represents a text cell type.
                     * @extends GC.Spread.Sheets.CellTypes.Base
                     * @class
                     * @param {GC.Spread.Sheets.CellTypes.EditorType} editorType The editor type of the text cell type.
                     */
                    constructor();
                }
            }

            module Charts{

                export interface ErrorBarItem{
                    type?: number;
                    valueType?: number;
                    noEndCap?: boolean;
                    value?: number;
                    custom?: {
                    positive: string;
                    negative: string;
                    };
                    style?: {
                    color?: string;
                    width?: number;
                    transparency?: number;
                    dashStyle?: GC.Spread.Sheets.Charts.ILineStyle;
                    visible?: boolean;
                    }
                }


                export interface ErrorBarItems{
                    vertical?: GC.Spread.Sheets.Charts.ErrorBarItem;
                    horizontal?: GC.Spread.Sheets.Charts.ErrorBarItem;
                }


                export interface IAxes{
                    primaryCategory?: IAxis;
                    primaryValue?: IAxis;
                    secondaryCategory?: IAxis;
                    secondaryValue?: IAxis;
                }


                export interface IAxis{
                    visible?: boolean;
                    tickLabelPosition?: GC.Spread.Sheets.Charts.TickLabelPosition;
                    lineStyle?: GC.Spread.Sheets.Charts.IBorder;
                    style?: GC.Spread.Sheets.Charts.IAxisStyle;
                    majorTickPosition?: GC.Spread.Sheets.Charts.TickMark;
                    minorTickPosition?: GC.Spread.Sheets.Charts.TickMark;
                    majorUnit?: number;
                    minorUnit?: number;
                    min?: number | Date;
                    max?: number | Date;
                    format?: string;
                    title?: GC.Spread.Sheets.Charts.IAxisTitle;
                    majorGridLine?: GC.Spread.Sheets.Charts.IGridLine;
                    minorGridLine?: GC.Spread.Sheets.Charts.IGridLine;
                    labelAngle?: number;
                    scaling?: GC.Spread.Sheets.Charts.IScaling;
                    displayUnit?: GC.Spread.Sheets.Charts.IDisplayUnit;
                }


                export interface IAxisStyle{
                    color?: string;
                    transparency?: number;
                    fontFamily?: string;
                    fontSize?: number;
                }


                export interface IAxisTitle{
                    text?: string;
                    color?: string;
                    transparency?: number;
                    fontFamily?: string;
                    fontSize?: number;
                }


                export interface IBorder{
                    color?: string;
                    width?: number;
                    transparency?: number;
                    dashStyle?: GC.Spread.Sheets.Charts.LineType;
                }


                export interface IChartArea{
                    backColor?: string;
                    backColorTransparency?: number;
                    fontFamily?: string;
                    fontSize?: number;
                    color?: string;
                    transparency?: number;
                    border?: GC.Spread.Sheets.Charts.IBorder;
                }


                export interface IChartTextStyle{
                    color?: string;
                    fontFamily?: string;
                    fontSize?: number | string;
                    transparency?: number;
                }


                export interface IChartTitle{
                    text?: string;
                    fontFamily?: string;
                    fontSize?: number;
                    color?: string;
                    transparency?:number;
                }


                export interface IDataLabels{
                    showValue?: boolean;
                    showSeriesName?: boolean;
                    showCategoryName?: boolean;
                    showPercentage?: boolean;
                    position?: GC.Spread.Sheets.Charts.DataLabelPosition;
                    format?: string;
                    color?: string;
                    transparency?: number;
                }


                export interface IDataPointPart{
                    fillColor?: string;
                    idx?: number;
                    transparency?: number;
                }


                export interface IDisplayUnit{
                    unit?: number | GC.Spread.Sheets.Charts.DisplayUnit;
                    visible?: boolean;
                    style?: IChartTextStyle;
                }


                export interface IGridLine{
                    color?: string;
                    transparency?: number;
                    visible?: boolean;
                    width?: number;
                }


                export interface IHoverStyle{
                    color?:string;
                    transparency?:number;
                    borderStyle?: GC.Spread.Sheets.Charts.IBorder;
                    symbolStyle?: GC.Spread.Sheets.Charts.IHoverSymbolStyle;
                }


                export interface IHoverSymbolStyle{
                    color?:string;
                    transparency?:number;
                    borderStyle?: GC.Spread.Sheets.Charts.IBorder;
                }


                export interface ILegend{
                    position?: GC.Spread.Sheets.Charts.LegendPosition;
                    visible?: boolean;
                    backColor?: string;
                    backColorTransparency?: number;
                    borderStyle?: GC.Spread.Sheets.Charts.IBorder;
                    color?: string;
                    fontFamily?: string;
                    fontSize?: number;
                }


                export interface ILineStyle{
                    color?: string;
                    width?: number;
                    transparency?: number;
                    dashStyle?: GC.Spread.Sheets.Charts.LineType;
                }


                export interface IScaling{
                    orientation?: number;
                    logBase?: number;
                }


                export interface ISeries{
                    chartType?: GC.Spread.Sheets.Charts.ChartType;
                    axisGroup?: GC.Spread.Sheets.Charts.AxisGroup;
                    backColor?: string;
                    backColorTransparency?: number;
                    border?: GC.Spread.Sheets.Charts.IBorder;
                    startAngle?: number;
                    name?: string;
                    xValues?: string;
                    yValues?: string;
                    bubbleSizes?: string;
                    datalabels?: any;
                    symbol?: GC.Spread.Sheets.Charts.ISymbol;
                    errorBars?: GC.Spread.Sheets.Charts.ErrorBarItems;
                    trendlines?: GC.Spread.Sheets.Charts.TrendlineItem[];
                }


                export interface ISeriesSymbolBorder{
                    color?: string;
                    colorTransparency?: number;
                    transparency?: number;
                    width?: number;
                    lineType?: number;
                }


                export interface ISymbol{
                    fill: string;
                    fillColorTransparency: number;
                    size: number;
                    shape: GC.Spread.Sheets.Charts.SymbolShape;
                    border: GC.Spread.Sheets.Charts.ISeriesSymbolBorder;
                }


                export interface ITrendline{
                    type?: GC.Spread.Sheets.Charts.TrendlineType;
                    order?: number;
                    period?: number;
                    intercept?: number;
                    forward?: number;
                    backward?: number;
                    displayEquation?: boolean;
                    displayRSquared?: boolean;
                    fontFamily?: string;
                    fontSize?: number;
                    fontColor?: string;
                    name?: string;
                    style?: GC.Spread.Sheets.Charts.ILineStyle;
                }


                export interface TrendlineItem{
                    type?: GC.Spread.Sheets.Charts.TrendlineType;
                    display?: string;
                    order?: number;
                    period?: number;
                    label?: string;
                    style?: GC.Spread.Sheets.Charts.ILineStyle;
                    intercept?: number;
                    forward?: number;
                    backward?: number;
                    displayEquation?: boolean;
                    displayRSquared?: boolean;
                    name?: string;
                    fontFamily?: string;
                    fontSize?: number;
                    fontColor?: string;
                }

                /**
                 * Specifies the type of axis group.
                 * @enum {number}
                 */
                export enum AxisGroup{
                    /**
                     * Primary axis group.
                     */
                    primary= 0,
                    /**
                     * Secondary axis group.
                     */
                    secondary= 1
                }

                /**
                 * Specifies the chart type.
                 * @enum {number}
                 */
                export enum ChartType{
                    /**
                     * Combo
                     */
                    combo= 0,
                    /**
                     * Scatter
                     */
                    xyScatter= 1,
                    /**
                     * Radar
                     */
                    radar= 2,
                    /**
                     * Doughnut
                     */
                    doughnut= 3,
                    /**
                     * Area
                     */
                    area= 8,
                    /**
                     * Line
                     */
                    line= 9,
                    /**
                     * Pie
                     */
                    pie= 10,
                    /**
                     * Bubble
                     */
                    bubble= 11,
                    /**
                     * Clustered Column
                     */
                    columnClustered= 12,
                    /**
                     * Stacked Column
                     */
                    columnStacked= 13,
                    /**
                     * 100% Stacked Column
                     */
                    columnStacked100= 14,
                    /**
                     * Clustered Bar
                     */
                    barClustered= 18,
                    /**
                     * Stacked Bar
                     */
                    barStacked= 19,
                    /**
                     * 100% Stacked Bar
                     */
                    barStacked100= 20,
                    /**
                     * Stacked Line
                     */
                    lineStacked= 24,
                    /**
                     * 100% Stacked Line
                     */
                    lineStacked100= 25,
                    /**
                     * Line with Markers
                     */
                    lineMarkers= 26,
                    /**
                     * Stacked Line with Markers
                     */
                    lineMarkersStacked= 27,
                    /**
                     * 100% Stacked Line with Markers
                     */
                    lineMarkersStacked100= 28,
                    /**
                     * Scatter with Smoothed Lines
                     */
                    xyScatterSmooth= 33,
                    /**
                     * Scatter with Smoothed Lines and No Data Markers
                     */
                    xyScatterSmoothNoMarkers= 34,
                    /**
                     * Scatter with Lines.
                     */
                    xyScatterLines= 35,
                    /**
                     * Scatter with Lines and No Data Markers
                     */
                    xyScatterLinesNoMarkers= 36,
                    /**
                     * Stacked Area
                     */
                    areaStacked= 37,
                    /**
                     * 100% Stacked Area
                     */
                    areaStacked100= 38,
                    /**
                     * Radar with data makers
                     */
                    radarMarkers= 42,
                    /**
                     * Filled Radar
                     */
                    radarFilled= 43,
                    /**
                     * High-Low-Close
                     */
                    stockHLC= 49,
                    /**
                     * Open-High-Low-Close
                     */
                    stockOHLC= 50,
                    /**
                     * Volume-High-Low-Close
                     */
                    stockVHLC= 51,
                    /**
                     * Volume-Open-High-Low-Close
                     */
                    stockVOHLC= 52,
                    /**
                     * Box & Whisker
                     */
                    boxWhisker= 53,
                    /**
                     * Funnel
                     */
                    funnel= 54,
                    /**
                     * Pareto
                     */
                    paretoLine= 55,
                    /**
                     * map
                     */
                    regionMap= 56,
                    /**
                     * sunburst
                     */
                    sunburst= 57,
                    /**
                     * tree map
                     */
                    treemap= 58,
                    /**
                     * Waterfull
                     */
                    waterfall= 59,
                    /**
                     *  Histogram
                     */
                    clusteredColumn= 60
                }

                /**
                 * Specifies where the data label is positioned.
                 * @enum {number}
                 */
                export enum DataLabelPosition{
                    /**
                     * Adjust data label position automatically.
                     */
                    bestFit= 0,
                    /**
                     * Data label below point.
                     */
                    below= 1,
                    /**
                     * Data label centered on data point or inside bar or pie.
                     */
                    center= 2,
                    /**
                     * Data label positioned arbitrarily.
                     */
                    insideBase= 3,
                    /**
                     * Data label positioned arbitrarily.
                     */
                    insideEnd= 4,
                    /**
                     * Data label positioned at bottom of bar or pie.
                     */
                    left= 5,
                    /**
                     * Data label positioned at top of bar or pie.
                     */
                    outsideEnd= 6,
                    /**
                     * Data label positioned at top of bar or pie.
                     */
                    right= 7,
                    /**
                     * Data label above point.
                     */
                    above= 8
                }

                /**
                 * Specifies the way of the chart display blank data.
                 * @enum {number}
                 */
                export enum DisplayBlanksAs{
                    /**
                     * Specifies display empty cells as connected
                     */
                    connected= 0,
                    /**
                     * Specifies display empty cells as gaps
                     */
                    gaps= 1,
                    /**
                     * Specifies display empty cells as zero
                     */
                    zero= 2
                }

                /**
                 * Specifies the built-in type of axis display unit.
                 * @enum {number}
                 */
                export enum DisplayUnit{
                    /**
                     * The hundreds of built-in type.
                     */
                    hundreds = 0,
                    /**
                     * The thousands of built-in type.
                     */
                    thousands = 1,
                    /**
                     * The ten thousands of built-in type.
                     */
                    tenThousands = 2,
                    /**
                     * The hundred thousands of built-in type.
                     */
                    hundredThousands = 3,
                    /**
                     * The millions of built-in type.
                     */
                    millions = 4,
                    /**
                     * The ten millions of built-in type.
                     */
                    tenMillions = 5,
                    /**
                     * The hundred millions of built-in type.
                     */
                    hundredMillions = 6,
                    /**
                     * The billions of built-in type.
                     */
                    billions = 7,
                    /**
                     * The trillions of built-in type.
                     */
                    trillions = 8
                }

                /**
                 * Specifies the type of error bar in series
                 * enum {number}
                 */
                export enum ErrorBarType{
                    /**
                     * Specifies each error bar has both minus and plus type at each data point.
                     */
                    both= 0,
                    /**
                     * Specifies each error bar has minus type at each data point.
                     */
                    minus= 1,
                    /**
                     * Specifies each error bar has plus type at each data point.
                     */
                    plus= 2
                }

                /**
                 * Specifies the value type of error bar in series
                 * enum {number}
                 */
                export enum ErrorBarValueType{
                    /**
                     * Specifies error bar has custom value type, each error bars has it own value, each values may be different.
                     */
                    custom= 0,
                    /**
                     * Specifies each error bar has fixed value type at each data point.
                     */
                    fixedValue= 1,
                    /**
                     * Specifies each error bar has percentage value type at each data point.
                     */
                    percentage= 2,
                    /**
                     * Specifies each error bar has calculated standard deviation value type at each data point.
                     */
                    standardDeviation= 3,
                    /**
                     * Specifies each error bar has calculated standard error value type at each data point.
                     */
                    standardError= 4
                }

                /**
                 * Specifies the position of the legend on a chart.
                 * @enum {number}
                 */
                export enum LegendPosition{
                    /**
                     * Above the chart.
                     */
                    top= 1,
                    /**
                     * To the right of the chart.
                     */
                    right= 2,
                    /**
                     * To the left of the chart.
                     */
                    left= 3,
                    /**
                     * Below the chart.
                     */
                    bottom= 4,
                    /**
                     * In the upper right-hand corner of the chart border.
                     */
                    topRight= 5
                }

                /**
                 * Specifies the type of line in series
                 * enum {number}
                 */
                export enum LineType{
                    /**
                     * Specifies a solid line type
                     */
                    solid= 0,
                    /**
                     * Specifies a dot line type
                     */
                    dot= 1,
                    /**
                     * Specifies a dash line type
                     */
                    dash= 2,
                    /**
                     * Specifies a large dash line type
                     */
                    lgDash= 3,
                    /**
                     * Specifies a dash and a dot line type
                     */
                    dashDot= 4,
                    /**
                     * Specifies a large dash and a dot line type
                     */
                    lgDashDot= 5,
                    /**
                     * Specifies a large dash and a dot and a dot line type
                     */
                    lgDashDotDot= 6,
                    /**
                     * Specifies a small dash line type
                     */
                    sysDash= 7,
                    /**
                     * Specifies a small dot line type
                     */
                    sysDot= 8,
                    /**
                     * Specifies a small dash and a dot line type
                     */
                    sysDashDot= 9,
                    /**
                     * Specifies a small dash and a dot and a dot line type
                     */
                    sysDashDotDot= 10
                }

                /**
                 * Specifies whether the values corresponding to a particular data series are in rows or columns.
                 * @enum {number}
                 */
                export enum RowCol{
                    /**
                     * Data series is in a column.
                     */
                    rows= 0,
                    /**
                     * Data series is in a row.
                     */
                    columns= 1
                }

                /**
                 * Specifies the shape of symbol in series
                 * enum {number}
                 */
                export enum SymbolShape{
                    /**
                     * Specifies a circle shall be drawn at each data point.
                     */
                    circle= 0,
                    /**
                     * Specifies a dash shall be drawn at each data point.
                     */
                    dash= 1,
                    /**
                     * Specifies a diamond shall be drawn at each data point.
                     */
                    diamond= 2,
                    /**
                     * Specifies a dot shall be drawn at each data point.
                     */
                    dot= 3,
                    /**
                     * Doesn't draw any symbol at each data point.
                     */
                    none= 4,
                    /**
                     * Specifies a picture shall be drawn at each data point.
                     */
                    picture= 5,
                    /**
                     * Specifies a plus shall be drawn at each data point.
                     */
                    plus= 6,
                    /**
                     * Specifies a square shall be drawn at each data point.
                     */
                    square= 7,
                    /**
                     * Specifies a star shall be drawn at each data point.
                     */
                    star= 8,
                    /**
                     * Specifies a triangle shall be drawn at each data point.
                     */
                    triangle= 9,
                    /**
                     * Specifies an X shall be drawn at each data point.
                     */
                    x= 10
                }

                /**
                 * Specifies the position of tick-mark labels on the specified axis.
                 * @enum {number}
                 */
                export enum TickLabelPosition{
                    /**
                     * Next to axis (where axis is not at either side of the chart).
                     */
                    nextToAxis= 2,
                    /**
                     * No tick marks.
                     */
                    none= 3
                }

                /**
                 * Specifies the position of major and minor tick marks for an axis.
                 * @enum {number}
                 */
                export enum TickMark{
                    /**
                     * Crosses the axis.
                     */
                    cross= 0,
                    /**
                     * Inside the axis.
                     */
                    inside= 1,
                    /**
                     * No mark.
                     */
                    none= 2,
                    /**
                     * Outside the axis.
                     */
                    outside= 3
                }

                /**
                 * Specifies how the trendline that smoothes out fluctuations in the data is calculated.
                 * @enum {number}
                 */
                export enum TrendlineType{
                    /**
                     * Uses an equation to calculate the least squares fit through points.
                     */
                    exponential= 0,
                    /**
                     * Uses the linear equation y = mx + b to calculate the least squares fit through points.
                     */
                    linear= 1,
                    /**
                     * Uses the equation y = c ln x + b to calculate the least squares fit through points.
                     */
                    logarithmic= 2,
                    /**
                     * Uses a sequence of averages computed from parts of the data series. The number
                     * of points equals the total number of points in the series less the number
                     * specified for the period.
                     */
                    movingAverage= 3,
                    /**
                     * Uses an equation to calculate the least squares fit through points.
                     */
                    polynomial= 4,
                    /**
                     * Uses an equation to calculate the least squares fit through points.
                     */
                    power= 5
                }


                export class Chart extends GC.Spread.Sheets.FloatingObjects.FloatingObject{
                    /**
                     * Represents a chart.
                     * @extends GC.Spread.Sheets.FloatingObjects.FloatingObject
                     * @class
                     * @param {GC.Spread.Sheets.Worksheet} sheet The host sheet of the chart.
                     * @param {string} name The name of the chart.
                     * @param {GC.Spread.Sheets.Charts.ChartType} chartType The type of the chart.
                     * @param {number} x The <i>x</i> location of the chart.
                     * @param {number} y The <i>y</i> location of the chart.
                     * @param {number} width The width of the chart.
                     * @param {number} height The height of the chart.
                     * @param {string?} dataRange The formula string of data range for the chart.
                     * @param {GC.Spread.Sheets.Charts.RowCol?} dataOrientation The orientation of data for series.
                     */
                    constructor(sheet:  GC.Spread.Sheets.Worksheet,  name:  string,  chartType:  GC.Spread.Sheets.Charts.ChartType,  x:  number,  y:  number,  width:  number,  height:  number,  dataRange?:  string,  dataOrientation?:  GC.Spread.Sheets.Charts.RowCol);
                    /**
                     * Gets or sets the chart axes of the chart.
                     * @param {Object} value The chart axes of the chart.
                     * @param {Object} value.primaryCategory The primary category axis of the chart.
                     * @param {Object} value.primaryValue The primary value axis of the chart.
                     * @param {Object} value.secondaryCategory The secondary category axis of the chart.
                     * @param {Object} value.secondaryValue The secondary value axis of the chart.
                     *
                     * The primaryCategory, primaryValue, secondaryCategory and secondaryValue have same type.
                     * @param {boolean} value.primaryCategory.visible Indicates if the specified axis should be shown.
                     * @param {GC.Spread.Sheets.Charts.TickLabelPosition} value.primaryCategory.tickLabelPosition The tick label position.
                     * @param {Object} value.primaryCategory.lineStyle The line style of the primary category axis.
                     * @param {string} value.primaryCategory.lineStyle.color The line color of the primary category axis.
                     * @param {number} value.primaryCategory.lineStyle.transparency The transparency of the chart primary category axis line color.
                     * @param {number} value.primaryCategory.lineStyle.width The line width of the primary category axis.
                     * @param {Object} value.primaryCategory.style The style of the primary category axis.
                     * @param {string} value.primaryCategory.style.color The color of the primary category axis.
                     * @param {number} value.primaryCategory.style.transparency The transparency of the chart primary category axis color.
                     * @param {string} value.primaryCategory.style.fontFamily The font family of the primary category axis.
                     * @param {number} value.primaryCategory.style.fontSize The font size of the primary category axis, its unit is pixel.
                     * @param {GC.Spread.Sheets.Charts.TickMark} value.primaryCategory.majorTickPosition The major tick position of the primary category axis.
                     * @param {GC.Spread.Sheets.Charts.TickMark} value.primaryCategory.minorTickPosition The minor tick position of the primary category axis.
                     * @param {GC.Spread.Sheets.Charts.TimeUnit} value.primaryCategory.baseUnit The base unit scale of the date primary category axis.
                     * @param {number} value.primaryCategory.majorUnit The major unit of the primary category axis.
                     * @param {GC.Spread.Sheets.Charts.TimeUnit} value.primaryCategory.majorUnitScale The major unit scale of the date primary category axis.
                     * @param {number} value.primaryCategory.minorUnit The minor unit of the primary category axis.
                     * @param {GC.Spread.Sheets.Charts.TimeUnit} value.primaryCategory.minorUnitScale The minor unit scale of the date primary category axis.
                     * @param {number | Date} value.primaryCategory.min The minimum value of the related axis. (for value / date axis only)
                     * @param {number | Date} value.primaryCategory.max The maximum value of the related axis. (for value / date axis only)
                     * @param {string} value.primaryCategory.format The format of the primary category axis.
                     * @param {Object} value.primaryCategory.title The title of the primary category axis.
                     * @param {string} value.primaryCategory.title.text The title text of the primary category axis.
                     * @param {string} value.primaryCategory.title.color The title color of the primary category axis.
                     * @param {number} value.primaryCategory.title.transparency The transparency of the primary category axis color.
                     * @param {string} value.primaryCategory.title.fontFamily The title font family of the primary category axis.
                     * @param {number} value.primaryCategory.title.fontSize The title font size of the primary category axis, its unit is pixel.
                     * @param {Object} value.primaryCategory.majorGridLine The major grid line of the primary category axis.
                     * @param {Object} value.primaryCategory.minorGridLine The minor grid line of the primary category axis.
                     * @param {number} value.primaryCategory.labelAngle The label angle of the primary category axis.
                     * @param {Object} value.primaryCategory.scaling The scaling informations of the primary category axis.
                     * @param {number} value.primaryCategory.scaling.logBase The logarithmic scaling base value of the primary category axis.
                     * @param {Object} value.primaryCategory.displayUnit The display unit informations of the primary category axis.
                     * @param {number | GC.Spread.Sheets.Charts.DisplayUnit} value.primaryCategory.displayUnit.unit The built-in display unit string or custom number display unit of the primary category axis.
                     * @param {boolean} value.primaryCategory.displayUnit.visible The display unit label visible of the primary category axis.
                     * @param {Object} value.primaryCategory.displayUnit.style The display unit label style of the primary category axis.
                     * @param {string} value.primaryCategory.displayUnit.style.color The display unit label text color of the primary category axis.
                     * @param {number} value.primaryCategory.displayUnit.style.transparency The display unit label text transparency of the primary category axis color.
                     * @param {string} value.primaryCategory.displayUnit.style.fontFamily The display unit label text font family of the primary category axis.
                     * @param {number} value.primaryCategory.displayUnit.style.fontSize The display unit label text font size of the primary category axis, its unit is pixel.
                     * @returns {Object | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the chart axes of the chart; otherwise, returns the chart.
                     *
                     * The majorGridLine and minorGridLine have same type.
                     * @param {string} value.primaryCategory.majorGridLine.color The color of the major grid line.
                     * @param {boolean} value.primaryCategory.majorGridLine.visible The visibility of the major grid line.
                     * @param {number} value.primaryCategory.majorGridLine.width The width of the major grid line.
                     * @param {number} value.primaryCategory.majorGridLine.transparency The transparency of the major grid line color.
                     *
                     * @returns {Object | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the chart axes of the chart; otherwise, returns the chart.
                     */
                    axes(value?:  GC.Spread.Sheets.Charts.IAxes): any;
                    /**
                     * Gets or sets the chart area style of the chart.
                     * @param {Object} value The chart area style of the chart.
                     * @param {string} value.backColor The background color of the chart area.
                     * @param {number} value.backColorTransparency The transparency of the chart area backColor.
                     * @param {string} value.fontFamily The font family of the chart area.
                     * @param {number} value.fontSize The font size of the chart area, its unit is pixel.
                     * @param {string} value.color The color of the chart area.
                     * @param {number} value.transparency The transparency of the chart area color.
                     * @param {Object} value.border The border of the chart area.
                     * @param {string} value.border.color The border color of the chart area.
                     * @param {number} value.border.transparency The border transparency of the chart area.
                     * @param {number} value.border.width The border width of the chart area.
                     * @param {GC.Spread.Sheets.Charts.LineDashStyle} value.border.dashStyle The border dash style of the chart area.
                     * @returns {Object | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the chart area style of the chart; otherwise, returns the chart.
                     */
                    chartArea(value?:  GC.Spread.Sheets.Charts.IChartArea): any;
                    /**
                     * Gets or sets the type of the chart.
                     * @param {GC.Spread.Sheets.Charts.ChartType} value The type of the chart.
                     * @returns {GC.Spread.Sheets.Charts.ChartType | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the type of the chart; otherwise, returns the chart.
                     */
                    chartType(value?:  GC.Spread.Sheets.Charts.ChartType): any;
                    /**
                     * Gets or sets the chart data labels style of the chart.
                     * @param {Object} value The chart data labels style of the chart.
                     * @param {boolean} value.showValue Whether to show value in data labels.
                     * @param {boolean} value.showSeriesName Whether to show series name in data labels.
                     * @param {boolean} value.showCategoryName Whether to show category name in data labels.
                     * @param {boolean} value.showPercentage Whether to show the percent value in data labels.
                     * @param {GC.Spread.Sheets.Charts.DataLabelPosition} value.position The position of the chart data labels.
                     * @param {string} value.format The format of the chart data labels.
                     * @param {string} value.color The color of the chart data labels.
                     * @param {number} value.transparency The transparency of the chart data labels color.
                     * @returns {Object | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the chart data labels style of the chart; otherwise, returns the chart.
                     */
                    dataLabels(value?:  GC.Spread.Sheets.Charts.IDataLabels): any;
                    /**
                     * Gets or sets the whole data range of the chart as formula string.
                     * @param {string} value The formula string of the data range for the chart.
                     * @returns {string | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the formula string of the whole data range for the chart; otherwise, returns the chart.
                     */
                    dataRange(value?:  string): any;
                    /**
                     * Gets or sets the way that the chart display blank data.
                     * @param {GC.Spread.Sheets.Charts.DisplayBlanksAs} value the way that the chart display blank data.
                     * @returns {GC.Spread.Sheets.Charts.DisplayBlanksAs | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the way that the chart display blank data, otherwise, returns the chart.
                     */
                    displayBlanksAs(value?:  GC.Spread.Sheets.Charts.DisplayBlanksAs): any;
                    /**
                     * Gets or sets whether to show #N/A cells as blank cells.
                     * @param {boolean} value that whether to show #N/A cells as blank cells.
                     * @returns { boolean | GC.Spread.Sheets.Charts.Chart} If no value is set, returns whether to show #N/A cells as blank cells, otherwise, returns the chart.
                     */
                    displayNaAsBlank(value?:  boolean): boolean | GC.Spread.Sheets.Charts.Chart;
                    /**
                     * Gets or sets the style when user hover over the dataPoint.
                     * @param {Object} value The hover style of the dataPoint been hovered.
                     * @param {string} value.color The color of the dataPoint been hovered.
                     * @param {number} value.transparency The color transparency of the dataPoint been hovered.
                     * @param {Object} value.borderStyle The border of the dataPoint been hovered.
                     * @param {string} value.borderStyle.color The border color of the dataPoint been hovered.
                     * @param {number} value.borderStyle.width The border width of the dataPoint been hovered.
                     * @param {number} value.borderStyle.transparency The border color transparency of the dataPoint been hovered.
                     * @param {GC.Spread.Sheets.Charts.LineType} value.borderStyle.dashStyle The border dash style of the dataPoint been hovered.
                     * @param {Object} value.symbolStyle The symbol style of the dataPoint been hovered.
                     * @param {string} value.symbolStyle.color The symbol's color of the dataPoint been hovered.
                     * @param {number} value.symbolStyle.transparency The symbol's transparency of the dataPoint been hovered.
                     * @param {Object} value.symbolStyle.borderStyle The symbol's border style of the dataPoint been hovered.
                     * @param {number} value.symbolStyle.borderStyle.color The symbol's border color of the dataPoint been hovered.
                     * @param {number} value.symbolStyle.borderStyle.width The symbol's border width of the dataPoint been hovered.
                     * @param {number} value.symbolStyle.borderStyle.transparency The symbol's border transparency of the dataPoint been hovered.
                     * @param {GC.Spread.Sheets.Charts.LineType} value.symbolStyle.borderStyle.dashStyle The symbol's border dash style of the dataPoint been hovered.
                     * @returns {Object | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the current hover style of the chart; otherwise, returns the chart.
                     */
                    hoverStyle(value?:  GC.Spread.Sheets.Charts.IHoverStyle): any;
                    /**
                     * Gets or sets the way that if the chart display hidden rows and columns data.
                     * @param {boolean} value the value that if the chart display hidden rows and columns data.
                     * @returns {boolean | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the value that if the chart display hidden rows and columns data, otherwise, returns the chart.
                     */
                    ignoreHidden(value?:  boolean): boolean | GC.Spread.Sheets.Charts.Chart;
                    /**
                     * Gets or sets the legend of the chart.
                     * @param {Object} value The legend of the chart.
                     * @param {GC.Spread.Sheets.Charts.LegendPosition} value.position The position of the chart legend.
                     * @param {boolean} value.visible The visibility of the chart legend.
                     * @param {string} value.backColor The backgroundColor of the chart legend.
                     * @param {number} value.backColorTransparency The transparency of the chart legend color
                     * @param {Object} value.borderStyle The borderStyle of the chart legend.
                     * @param {string} value.borderStyle.color The border color of the chart legend.
                     * @param {number} value.borderStyle.width The border width of the chart legend.
                     * @param {number} value.borderStyle.transparency The transparency of the chart legend border color
                     * @param {string} value.color The color of the chart legend text.
                     * @param {string} value.fontFamily The font family of the chart legend text.
                     * @param {number} value.fontSize The font size of the chart legend text.
                     * @param {boolean} value.showLegendWithoutOverlapping whether the legend display without overlapping chart area
                     * @param {Object} value.layout The layout of the chart legend.
                     * @param {number} value.layout.x The x position of the chart legend, it's percentage, and the base line is chart's left edge.
                     * @param {number} value.layout.y The y position of the chart legend, it's percentage, and the base line is chart's top edge.
                     * @param {number} value.layout.width The width of the chart legend, it's percentage, and it's based on the chart's width.
                     * @param {number} value.layout.height The height of the chart legend, it's percentage, and it's based on the chart's height.
                     * @returns {Object | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the legend of the chart; otherwise, returns the chart.
                     */
                    legend(value?:  GC.Spread.Sheets.Charts.ILegend): any;
                    /**
                     * Gets the series collection of the chart.
                     * @returns {GC.Spread.Sheets.Charts.SeriesCollection} Returns the series collection of the chart.
                     */
                    series(): GC.Spread.Sheets.Charts.SeriesCollection;
                    /**
                     * Switches the data orientation between rows and columns.
                     * @returns {boolean} Returns true when data orienetation is changable and successful switched; otherwise, false.
                     */
                    switchDataOrientation(): boolean;
                    /**
                     * Gets or sets the title of the chart.
                     * @param {Object} value The title of the chart.
                     * @param {string} value.text The text of the chart title.
                     * @param {string} value.fontFamily The font family of the chart title.
                     * @param {number} value.fontSize The font size of the chart title, its unit is pixel.
                     * @param {string} value.color The color of the chart title.
                     * @param {number} value.transparency The transparency of the chart title color
                     * @returns {Object | GC.Spread.Sheets.Charts.Chart} If no value is set, returns the title of the chart; otherwise, returns the chart.
                     */
                    title(value?:  GC.Spread.Sheets.Charts.IChartTitle): any;
                    /**
                     * Gets or sets whether apply animation to the chart.
                     * @param {boolean} value whether apply animation to the chart.
                     * @returns {boolean | GC.Spread.Sheets.Charts.Chart} If no value is set, returns whether apply animation to the chart; otherwise, returns the chart.
                     */
                    useAnimation(value?:  boolean): any;
                }

                export class ChartCollection{
                    /**
                     * Represents a chart manager that managers all charts in a sheet.
                     * @class
                     * @param {GC.Spread.Sheets.Worksheet} sheet The worksheet.
                     */
                    constructor();
                    /**
                     * Adds a chart to the sheet.
                     * @param {string} name The name of the chart that will be added to the sheet.
                     * @param {GC.Spread.Sheets.Charts.ChartType} chartType The type of the chart.
                     * @param {number} x The x location of the chart.
                     * @param {number} y The y location of the chart.
                     * @param {number} width The width of the chart.
                     * @param {number} height The height of the chart.
                     * @param {string} dataRange The formula string of data range for the chart.
                     * @param {GC.Spread.Sheets.Charts.RowCol} dataOrientation The orientation of data for series.
                     * @return {GC.Spread.Sheets.Charts.Chart} The chart that has been added to the sheet.
                     * @example
                     * //This example shows how to add a chart.
                     * var dataRange = "A1:D4";
                     * var chart = activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 400, dataRange);
                     */
                    add(name:  string,  chartType:  GC.Spread.Sheets.Charts.ChartType,  x:  number,  y:  number,  width:  number,  height:  number,  dataRange?:  string,  dataOrientation?:  GC.Spread.Sheets.Charts.RowCol): GC.Spread.Sheets.Charts.Chart;
                    /**
                     * Gets all of the charts in the sheet.
                     * @return {Array.<GC.Spread.Sheets.Charts.Chart>} The collection of all the charts in the sheet.
                     * @example
                     * var dataRange = "A1:D4";
                     * activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 180, dataRange);
                     * var dataRange2 = "A20:D24";
                     * activeSheet.charts.add('Chart2', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 220, 600, 180, dataRange2);
                     * var charts = activeSheet.charts.all();
                     * for (var i = 0; i &lt; charts.length; i++) {
                     *     alert("Name of chart " + i + " is:  " + charts[i].name())
                     * }
                     */
                    all(): GC.Spread.Sheets.Charts.Chart[];
                    /**
                     * Removes all charts in the sheet.
                     */
                    clear(): void;
                    /**
                     * Gets a chart from the sheet by the indicate name.
                     * @param {string} name The name of the chart.
                     * @return {GC.Spread.Sheets.Charts.Chart} The chart in the sheet with the indicate name.
                     * @example
                     * var dataRange = "A1:D4";
                     * activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 400, dataRange);
                     * //button
                     * $("#button1").click(function () {
                     *  var chart = activeSheet.charts.get("f2");
                     * });
                     */
                    get(name:  string): GC.Spread.Sheets.Charts.Chart;
                    /**
                     * Removes a chart from the sheet by the indicate name.
                     * @param {string} name The name of the chart.
                     * @example
                     * var dataRange = "A1:D4";
                     * activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 400, dataRange);
                     * //button
                     * $("#button1").click(function () {
                     *      activeSheet.resumePaint();
                     *      activeSheet.charts.remove("f2");
                     *      activeSheet.repaint();
                     * });
                     */
                    remove(name:  string): void;
                    /**
                     * Gets or sets the z-index of chart.
                     * @param {string} name The name of the chart.
                     * @param {number} zIndex The z-index of the chart.
                     * @return {number | *} If the parameter 'zIndex' is null or undefined,it will return the z-index of the chart with the indicate name.
                     * @example
                     * var dataRange = "A1:D4";
                     * activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 180, dataRange);
                     * var dataRange2 = "A20:D24";
                     * activeSheet.charts.add('Chart2', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 180, 600, 180, dataRange2);
                     * activeSheet.charts.zIndex('Chart1', 897);
                     * activeSheet.charts.zIndex('Chart2', 890);
                     */
                    zIndex(name:  string,  zIndex?:  number): any;
                }

                export class DataPoints{
                    /**
                     * Represents the dataPoint collection that managers all dataPoints in a chart series.
                     * @class
                     */
                    constructor();
                    /**
                     * Gets all dataPoints or a specified dataPoint from dataPoints collection.
                     * @param {number} index The index of the dataPoint.
                     * @return {Object | Object[]} The dataPoint of the chart series.
                     * @example
                     * // This example shows how to get a dataPoint.
                     * var dataRange = new GC.Spread.Sheets.Range(0, 0, 16, 4);
                     * var chart = activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.sunburst, 250, 20, 600, 400, dataRange);
                     * var dataPoints = chart.series().dataPoints();
                     * var dataPoint = dataPoints.get(0);
                     */
                    get(index?:  number): any;
                    /**
                     * Updates the specified dataPoint's property.
                     * @param {number} index The index of the dataPoint.
                     * @param {Object} dataPoint The data point of the chart.
                     * @param {string} dataPoint.fillColor The color of the data point.
                     * @param {number} dataPoint.transparency The transparency of the data point color.
                     * @example
                     * // This example shows how to update the property of a dataPoint
                     * var dataRange = new GC.Spread.Sheets.Range(0, 0, 16, 4);
                     * var chart = activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.sunburst, 250, 20, 600, 400, dataRange);
                     * var dataPoints = chart.series().dataPoints();
                     * var dataPoint = {fillColor: 'rgba(255,255,0,0,7)'};
                     * dataPoints.set(0, dataPoint);
                     */
                    set(index:  number,  dataPoint:  GC.Spread.Sheets.Charts.IDataPointPart): void;
                }

                export class SeriesCollection{
                    /**
                     * Represents the series manager that managers all series in a chart.
                     * @class
                     */
                    constructor();
                    /**
                     * Adds a new series to series collection.
                     * @param {Object} seriesItem The series of the chart.
                     * @param {Object} seriesItem.symbol, seriesItem The symbol of the series.
                     * @param {string} seriesItem.symbol.fill The symbol fill color of the series.
                     * @param {number} seriesItem.symbol.size The symbol size of the series.
                     * @param {GC.Spread.Sheets.Charts.SymbolShape} seriesItem.symbol.shape The symbol shape of the series.
                     * @param {Object} seriesItem.symbol.border The symbol border of the series.
                     * @param {string} seriesItem.symbol.border.color The symbol border color of the series.
                     * @param {number} seriesItem.symbol.border.width The symbol border width of the series.
                     * @param {GC.Spread.Sheets.Charts.LineType} seriesItem.symbol.border.lineType The symbol border line Type of the series.
                     * @param {GC.Spread.Sheets.Charts.ChartType} seriesItem.chartType The chart type of the series.
                     * @param {GC.Spread.Sheets.Charts.AxisGroup} seriesItem.axisGroup The axis group of the series.
                     * @param {string} seriesItem.backColor The background color of the series.
                     * @param {number} seriesItem.backColorTransparency The transparency of the series background color.
                     * @param {string} seriesItem.border The border of the series.
                     * @param {string} seriesItem.border.color The border color of the series.
                     * @param {number} seriesItem.border.transparency The transparency of the series border color.
                     * @param {number} seriesItem.border.width The border width of the series.
                     * @param {GC.Spread.Sheets.Charts.LineDashStyle} seriesItem.border.lineType The border line type of the series.
                     * @param {number} seriesItem.startAngle The first slice angle of the chart whose chart type is pie. The default value is 0, which represents the 12 o'clock position.
                     * @param {string} seriesItem.name The name formula of the series.
                     * @param {string} seriesItem.xValues The x values formula of the series.
                     * @param {string} seriesItem.yValues The y values formula of the series.
                     * @param {string} seriesItem.bubbleSizes The bubble sizes formula of the series. This is used for bubble chart.
                     * @example
                     * // This example shows how to add a new series.
                     * var dataRange = "A1:D4";
                     * var chart = activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 400, dataRange);
                     * chart.series.add({
                     *     chartType: GC.Spread.Sheets.Charts.ChartType.columnClustered,
                     *     axisGroup: GC.Spread.Sheets.Charts.AxisGroup.primary,
                     *     backColor: {
                     *         color: "lightblue",
                     *         width: 2
                     *     },
                     *     xValues: "A2:A4",
                     *     yValues: "B2:B4"
                     * });
                     */
                    add(series:  GC.Spread.Sheets.Charts.ISeries | GC.Spread.Sheets.Charts.ISeries[]): void;
                    /**
                     * Gets the dataPoints of the chart series.
                     * @returns {GC.Spread.Sheets.Charts.DataPoints} Returns the dataPoints of the chart series if exist.
                     */
                    dataPoints(): GC.Spread.Sheets.Charts.DataPoints;
                    /**
                     * Gets all series or a specified series from series collection.
                     * @param {number} index The index of the series.
                     * @return {Object | Object[]} The series of the chart.
                     * @example
                     * // This example shows how to get a series.
                     * var dataRange = "A1:D4";
                     * var chart = activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 400, dataRange);
                     * var series1 = chart.series().get(0);
                     */
                    get(index?:  number): any;
                    /**
                     * Removes a specified series from series collection.
                     * @param {number} index The index of the series.
                     * @example
                     * // This example shows how to remove a specified series.
                     * var dataRange = "A1:D4";
                     * var chart = activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 400, dataRange);
                     * chart.series.remove(0);
                     */
                    remove(index:  number): void;
                    /**
                     * Updates the specified series's property.
                     * @param {number} index The index of the series.
                     * @param {Object} seriesItem The series of the chart.
                     * @param {Object} seriesItem.symbol, seriesItem The symbol of the series.
                     * @param {string} seriesItem.symbol.fill The symbol fill color of the series.
                     * @param {number} seriesItem.symbol.fillColorTransparency The transparency of the symbol fill color.
                     * @param {number} seriesItem.symbol.size The symbol size of the series.
                     * @param {GC.Spread.Sheets.Charts.SymbolShape} seriesItem.symbol.shape The symbol shape of the series.
                     * @param {Object} seriesItem.symbol.border The symbol border of the series.
                     * @param {string} seriesItem.symbol.border.color The symbol border color of the series.
                     * @param {number} seriesItem.symbol.border.colorTransparency The transparency of the symbol border color.
                     * @param {number} seriesItem.symbol.border.width The symbol border width of the series.
                     * @param {GC.Spread.Sheets.Charts.LineType} seriesItem.symbol.border.lineType The symbol border line Type of the series.
                     * @param {Object} seriesItem.errorbars The error bars of the series.
                     * @param {Object} seriesItem.errorbars.vertical The vertical error bar of the series, each series may has different direction error bar(s) based on chart type.
                     * @param {GC.Spread.Sheets.Charts.ErrorBarType} seriesItem.errorbars.vertical.type The error bar type.
                     * @param {GC.Spread.Sheets.Charts.ErrorBarValueType} seriesItem.errorbars.vertical.valuetype The error bar value type.
                     * @param {boolean} seriesItem.errorbars.vertical.noEndCap The error bar has end cap or not.
                     * @param {number} seriesItem.errorbars.vertical.value The error bar value, just take effect on the FixedValue(1) / Percentage(2) / StandardDeviation(3) value type.
                     * @param {Object} seriesItem.errorbars.vertical.custom The error bar custom formulas, it contains positive formula and negative formula, just take effect on the Custom(0) value type.
                     * @param {string} seriesItem.errorbars.vertical.custom.positive The error bar custom positive formula.
                     * @param {string} seriesItem.errorbars.vertical.custom.negative The error bar custom negative formula.
                     * @param {Object} seriesItem.errorbars.vertical.style The error bar styles.
                     * @param {string} seriesItem.errorbars.vertical.style.color The error bar color.
                     * @param {number} seriesItem.errorbars.vertical.style.transparency The error bar transparency.
                     * @param {number} seriesItem.errorbars.vertical.style.width The error bar width.
                     * @param {Object} seriesItem.errorbars.horizontal The horizontal error bar of the series, each series may has different direction error bar(s) based on chart type.
                     * @param {GC.Spread.Sheets.Charts.ErrorBarType} seriesItem.errorbars.horizontal.type The error bar type.
                     * @param {GC.Spread.Sheets.Charts.ErrorBarValueType} seriesItem.errorbars.horizontal.valuetype The error bar value type.
                     * @param {boolean} seriesItem.errorbars.horizontal.noEndCap The error bar has end cap or not.
                     * @param {number} seriesItem.errorbars.horizontal.value The error bar value, just take effect on the FixedValue(1) / Percentage(2) / StandardDeviation(3) value type.
                     * @param {Object} seriesItem.errorbars.horizontal.custom The error bar custom formulas, it contains positive formula and negative formula, just take effect on the Custom(0) value type.
                     * @param {string} seriesItem.errorbars.horizontal.custom.positive The error bar custom positive formula.
                     * @param {string} seriesItem.errorbars.horizontal.custom.negative The error bar custom negative formula.
                     * @param {Object} seriesItem.errorbars.horizontal.style The error bar styles.
                     * @param {string} seriesItem.errorbars.horizontal.style.color The error bar color.
                     * @param {number} seriesItem.errorbars.horizontal.style.transparency The error bar transparency.
                     * @param {number} seriesItem.errorbars.horizontal.style.width The error bar width.
                     * @param {GC.Spread.Sheets.Charts.LineDashStyle} seriesItem.errorbars.horizontal.style.dashStyle The error bar dash style.
                     * @param {Object[]} seriesItem.trendlines The trendlines of the series.
                     * @param {GC.Spread.Sheets.Charts.TrendlineType} seriesItem.trendlines[0].type The type of the trendline.
                     * @param {number} seriesItem.trendlines[0].order The order of the polynomial trendline.
                     * @param {number} seriesItem.trendlines[0].period The period of the movingAverage trendline.
                     * @param {number} seriesItem.trendlines[0].intercept The intercept of the trendline.
                     * @param {number} seriesItem.trendlines[0].forward The forward of the trendline.
                     * @param {number} seriesItem.trendlines[0].backward The backward of the trendline.
                     * @param {boolean} seriesItem.trendlines[0].displayEquation Whether display the equation of the trendline.
                     * @param {boolean} seriesItem.trendlines[0].displayRSquared Whether display the R squared of the trendline.
                     * @param {string} seriesItem.trendlines[0].fontFamily The fontFamily of the trendline.
                     * @param {number} seriesItem.trendlines[0].fontSize The fontSize of the trendline.
                     * @param {string} seriesItem.trendlines[0].fontColor The fontColor of the trendline.
                     * @param {string} seriesItem.trendlines[0].name The name of the trendline.
                     * @param {Object} seriesItem.trendlines[0].style The line style of the trendline.
                     * @param {string} seriesItem.trendlines[0].style.color The color of the trendline.
                     * @param {number} seriesItem.trendlines[0].style.transparency The transparency of the trendline.
                     * @param {number} seriesItem.trendlines[0].style.width The width of the trendline.
                     * @param {GC.Spread.Sheets.Charts.LineDashStyle} seriesItem.trendlines[0].style.dashStyle The dash style of the trendline.
                     * @param {GC.Spread.Sheets.Charts.ChartType} seriesItem.chartType The chart type of the series.
                     * @param {GC.Spread.Sheets.Charts.AxisGroup} seriesItem.axisGroup The axis group of the series.
                     * @param {string} seriesItem.backColor The background color of the series.
                     * @param {number} seriesItem.backColorTransparency The transparency of the series background color.
                     * @param {Object} seriesItem.border The border of the series.
                     * @param {string} seriesItem.border.color The border color of the series.
                     * @param {GC.Spread.Sheets.Charts.LineDashStyle} seriesItem.border.lineType The border line type of the series.
                     * @param {number} seriesItem.border.transparency The transparency of the series border color.
                     * @param {number} seriesItem.border.width The border width of the series.
                     * @param {number} seriesItem.startAngle The first slice angle of the chart whose chart type is pie. The default value is 0, which represents the 12 o'clock position.
                     * @param {string} seriesItem.name The name formula of the series.
                     * @param {string} seriesItem.xValues The x values formula of the series.
                     * @param {string} seriesItem.yValues The y values formula of the series.
                     * @param {string} seriesItem.bubbleSizes The bubble sizes formula of the series. This is used for bubble chart.
                     * @param {Array} seriesItem.trendlines The trendlines of the series.
                     * @param {string} seriesItem.bubbleSizes The bubble sizes formula of the series. This is used for bubble chart.
                     * @example
                     * // This example shows how to update the property of a series.
                     * var dataRange = "A1:D4";
                     * var chart = activeSheet.charts.add('Chart1', GC.Spread.Sheets.Charts.ChartType.columnClustered, 250, 20, 600, 400, dataRange);
                     * var series1 = chart.series().get(0);
                     * series1.backColor = "red";
                     * chart.series().set(0, series1);
                     */
                    set(index:  number,  series:  GC.Spread.Sheets.Charts.ISeries): void;
                }
            }

            module Commands{
                /**
                 * Represents the command used to automatically resize the column in a sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.columns {Array} The resize columns; each item is an object which has a col.<br />
                 * options.rowHeader {boolean} Whether the resized columns are in the row header area.<br />
                 * options.autoFitType {GC.Spread.Sheets.AutoFitType} Whether the auto-fit action includes the header text.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * var columns = [ { col: 3 } ];
                 * spread.options.allowUndo = true;
                 * spread.commandManager().execute({cmd: "autoFitColumn", sheetName: "Sheet1", columns: columns, isRowHeader: false, autoFitType: GC.Spread.Sheets.AutoFitType.cell});
                 */
                var autoFitColumn: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, columns: Object[], rowHeader: boolean, autoFitType: GC.Spread.Sheets.AutoFitType}, isUndo: boolean): any};
                /**
                 * Represents the command used to automatically resize the row in a sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.rows {Array} The resize rows; each item is an object which has a row.<br />
                 * options.columnHeader {boolean} Whether the resized rows are in the column header area.<br />
                 * options.autoFitType {GC.Spread.Sheets.AutoFitType} Whether the auto-fit action includes the header text.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * spread.options.allowUndo = true;
                 * var rows = [ { row: 3 } ];
                 * spread.commandManager().execute({cmd: "autoFitRow", sheetName: "Sheet1", rows: rows, isColHeader: false, autoFitType: GC.Spread.Sheets.AutoFitType.cell});
                 */
                var autoFitRow: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, rows: Object[], columnHeader: boolean, autoFitType: GC.Spread.Sheets.AutoFitType}, isUndo: boolean): any};
                /**
                 * Represents the command used to stop cell editing and cancel input.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var cancelInput: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to switch the formula reference between relative, absolute, and mixed when editing formulas.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var changeFormulaReference: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to clear the cell value.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * // clear selected cells with the tab key
                 * spread.commandManager().setShortcutKey('clear', GC.Spread.Commands.Key.tab, false, false, false, false); // Tab key
                 */
                var clear: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to clear the active cell value and enters edit mode.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var clearAndEditing: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to clear cell values on a worksheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.ranges {Array} The clear cell value ranges whose item type is GC.Spread.Sheets.Range.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * spread.options.allowUndo = true;
                 * spread.commandManager().execute({cmd: "clearValues", sheetName: "Sheet1", ranges: [new GC.Spread.Sheets.Range(8, 5, 2, 1)]});
                 */
                var clearValues: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, ranges:GC.Spread.Sheets.Range[]}, isUndo: boolean): any};
                /**
                 * Represents the command used for a Clipboard paste on the worksheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.fromSheet {GC.Spread.Sheets.Worksheet} The source sheet.<br />
                 * options.fromRanges {Array} The source range array which item type is GC.Spread.Sheets.Range.<br />
                 * options.pastedRanges {Array} The target range array which item type is GC.Spread.Sheets.Range.<br />
                 * options.isCutting {boolean} Whether the operation is cutting or copying.<br />
                 * options.clipboardText {string} The text on the clipboard.<br />
                 * options.pasteOption {GC.Spread.Sheets.ClipboardPasteOptions} The Clipboard pasting option that indicates which content to paste.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the clipboardPaste method.
                 * activeSheet.setValue(0, 0, 1, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.setValue(1, 0, 2, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.setFormula(2, 0, "=A1+A2", GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.setValue(0, 1, 3, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.setValue(1, 1, 4, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.setFormula(2, 1, "=B1+B2", GC.Spread.Sheets.SheetArea.viewport);
                 * var fromRange = [new GC.Spread.Sheets.Range(0, 0, 3, 2)];
                 * var toRanges = [new GC.Spread.Sheets.Range(5, 0, 3, 2)];
                 * $("#button1").click(function () {
                 *     //Cut Paste Action
                 *     spread.commandManager().execute({cmd: "clipboardPaste", sheetName: "Sheet1", fromSheet: activeSheet, fromRanges: fromRange, pastedRanges: toRanges, isCutting: true, clipboardText: "", pasteOption: GC.Spread.Sheets.ClipboardPasteOptions.all});
                 * });
                 * $("#button2").click(function () {
                 *     spread.commandManager().execute({cmd: "clipboardPaste", sheetName: "Sheet1", fromSheet: activeSheet, fromRanges: fromRange, pastedRanges: toRanges, isCutting: false, clipboardText: "", pasteOption: GC.Spread.Sheets.ClipboardPasteOptions.all});
                 * });
                 * //Add button controls to page
                 * &lt;input type="button" id="button1" value="button1"/&gt;
                 * &lt;input type="button" id="button2" value="button2"/&gt;
                 */
                var clipboardPaste: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, fromSheet: GC.Spread.Sheets.Worksheet, fromRanges: GC.Spread.Sheets.Range[], pastedRanges: GC.Spread.Sheets.Range[], isCutting: boolean, clipboardText: string, pasteOption: GC.Spread.Sheets.ClipboardPasteOptions}, isUndo: boolean): any};
                /**
                 * Represents the command used to commit the cell editing and sets the array formula to the active range.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var commitArrayFormula: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to stop cell editing and moves the active cell to the next row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var commitInputNavigationDown: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to stop cell editing and moves the active cell to the previous row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var commitInputNavigationUp: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to copy the selected item text to the Clipboard.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var copy: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to cut the selected item text to the Clipboard.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 ** options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var cut: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command for deleting the floating objects.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var deleteFloatingObjects: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to drag and copy the floating objects on the sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.floatingObjects {Array} The names array of floating objects.<br />
                 * options.offsetX The horizontal offset.<br />
                 * options.offsetY The vertical offset.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise <c>false</c>.
                 */
                var dragCopyFloatingObjects: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, floatingObjects: string[], offsetX: number, offsetY: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to drag a range and drop it onto another range on the worksheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * commandOptions {Object} The options of the operation.<br />
                 * commandOptions.sheetName {string} The sheet name.<br />
                 * commandOptions.fromRow {number} The source row index for the drag drop.<br />
                 * commandOptions.fromColumn {number} The source column index for the drag drop.<br />
                 * commandOptions.toRow {number} The destination row index for the drag drop.<br />
                 * commandOptions.toColumn {number} The destination column index for the drag drop.<br />
                 * commandOptions.rowCount {number} The row count for the drag drop.<br />
                 * commandOptions.columnCount {number} The column count for the drag drop.<br />
                 * commandOptions.copy {boolean} If set to <c>true</c> copy; otherwise, cut if <c>false</c>.<br />
                 * commandOptions.insert {boolean} If set to <c>true</c> inserts the drag data in the drop row or column.<br />
                 * commandOptions.option {GC.Spread.Sheets.CopyToOptions} Indicates the content type to drag and drop.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * spread.options.allowUndo = true;
                 * spread.commandManager().execute({cmd: "dragDrop",  sheetName: "Sheet1", fromRow:2, fromColumn:1, toRow:12, toColumn:2, rowCount:2, columnCount:2, copy: true, insert: false, option: GC.Spread.Sheets.CopyToOptions.value});
                 */
                var dragDrop: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, fromRow: number, fromColumn: number, toRow: number, toColumn: number, rowCount: number, columnCount: number, copy: boolean, insert: boolean, option: GC.Spread.Sheets.CopyToOptions}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to apply a new value to a cell on the worksheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.row {number} The row index of the cell.<br />
                 * options.col {number} The column index of the cell.<br />
                 * options.newValue {Object} The new value of the cell.<br />
                 * options.autoFormat {boolean} Whether to format the new value automatically.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var editCell: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, row: number, col: number, newValue: any, autoFormat: boolean}, isUndo: boolean): any};
                /**
                 * Represents the command to expand or collapse a column range group.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.index {number} The outline summary index.<br />
                 * options.level {number} The outline level.<br />
                 * options.collapsed {boolean} Whether to make the outline collapsed or expanded.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var expandColumnOutline: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, index: number, level: number, collapsed: boolean}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to expand or collapse column range groups on the same level.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.level {number} The outline level.<br />
                 * isUndo {boolean} <c>true</c> if this an undo operation; otherwise, <c>false</c>.
                 */
                var expandColumnOutlineForLevel: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, level: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command to expand or collapse a row range group.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.index {number} The outline summary index.<br />
                 * options.level {number} The outline level.<br />
                 * options.collapsed {boolean} Whether to make the outline collapsed or expanded.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var expandRowOutline: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, index: number, level: number, collapsed: boolean}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to expand or collapse row range groups on the same level.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.level {number} The outline level.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var expandRowOutlineForLevel: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, level: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to drag and fill a range on the sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.startRange {GC.Spread.Sheets.Range} The start range.<br />
                 * options.fillRange {GC.Spread.Sheets.Range} The fill range.<br />
                 * options.autoFillType {GC.Spread.Sheets.Fill.AutoFillType} The auto fill type.<br />
                 * options.fillDirection {GC.Spread.Sheets.Fill.FillDirection} The fill direction.<br />
                 * isUndo {boolean} <c>true</c> if an undo operation; otherwise, <c>false</c>.
                 * @example
                 * spread.options.allowUndo = true;
                 * var srange = new GC.Spread.Sheets.Range(10, 5, 1, 1);
                 * var frange = new GC.Spread.Sheets.Range(11, 5, 5, 1);
                 * spread.commandManager().execute({cmd: "fill", sheetName: "Sheet1", startRange: srange, fillRange: frange, autoFillType: GC.Spread.Sheets.Fill.AutoFillType.fillSeries, fillDirection: GC.Spread.Sheets.Fill.FillDirection.down });
                 */
                var fill: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, startRange: GC.Spread.Sheets.Range, fillRange: GC.Spread.Sheets.Range, autoFillType: GC.Spread.Sheets.Fill.AutoFillType, fillDirection: GC.Spread.Sheets.Fill.FillDirection}, isUndo: boolean): boolean};
                /**
                 * Represents the command for moving floating objects.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.floatingObjects {Array} The names array of floating objects.<br />
                 * options.offsetX The horizontal offset.<br />
                 * options.offsetY The vertical offset.
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var moveFloatingObjects: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, floatingObjects: string[], offsetX: number, offsetY: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to move the active cell to the next cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example maps the moveToNextCell action.
                 * spread.focus();
                 * spread.commandManager().setShortcutKey('moveToNextCell', GC.Spread.Commands.Key.a, false, false, false, false); // a
                 */
                var moveToNextCell: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to select the next control if the active cell is the last visible cell; otherwise, move the active cell to the next cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example maps the moveToNextCellThenControl action.
                 * spread.commandManager().setShortcutKey('moveToNextCellThenControl', GC.Spread.Commands.Key.tab, false, false, false, false); // Tab key
                 * spread.commandManager().setShortcutKey('moveToPreviousCellThenControl', GC.Spread.Commands.Key.tab, false, true, false, false); // Shift key and Tab key
                 */
                var moveToNextCellThenControl: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the previous cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the moveToPreviousCell action.
                 * spread.commandManager().setShortcutKey('moveToPreviousCell', GC.Spread.Commands.Key.a, false, false, false, false); // a
                 */
                var moveToPreviousCell: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to select the previous control if the active cell is the first visible cell; otherwise, move the active cell to the previous cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example maps the moveToPreviousCellThenControl action.
                 * spread.commandManager().setShortcutKey('moveToNextCellThenControl', GC.Spread.Commands.Key.tab, false, false, false, false); // Tab key
                 * spread.commandManager().setShortcutKey('moveToPreviousCellThenControl', GC.Spread.Commands.Key.tab, false, true, false, false); // Shift key and Tab key
                 */
                var moveToPreviousCellThenControl: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the last row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example maps the navigationBottom action to the Tab key.
                 * spread.commandManager().setShortcutKey('navigationDown', GC.Spread.Commands.Key.tab, false, false, false, false); // Tab key
                 * spread.commandManager().setShortcutKey('navigationBottom', GC.Spread.Commands.Key.tab, false, true, false, false); // Shift key and Tab key
                 */
                var navigationBottom: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the next row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example maps the navigationDown key.
                 * spread.commandManager().setShortcutKey('navigationDown', GC.Spread.Commands.Key.tab, false, false, false, false); // Tab key
                 * spread.commandManager().setShortcutKey('navigationBottom', GC.Spread.Commands.Key.tab, false, true, false, false); // Shift key and Tab key
                 */
                var navigationDown: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the last column.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationEnd method.
                 * spread.commandManager().setShortcutKey('navigationEnd', GC.Spread.Commands.Key.tab,  false, false, false, false); // Tab key
                 */
                var navigationEnd: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the last column without regard to frozen trailing columns.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationEnd2 action.
                 * spread.commandManager().setShortcutKey("navigationEnd2", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationEnd2: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the first cell in the sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationFirst action.
                 * spread.commandManager().setShortcutKey("navigationFirst", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationFirst: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the first column.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationHome action.
                 * spread.commandManager().setShortcutKey("navigationHome", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationHome: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the first column without regard to frozen columns.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 ** options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationHome2 action.
                 * spread.commandManager().setShortcutKey("navigationHome2", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationHome2: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the last cell in the sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationLast action.
                 * spread.commandManager().setShortcutKey("navigationLast", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationLast: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the previous column.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationLeft action.
                 * spread.commandManager().setShortcutKey("navigationLeft", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationLeft: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active sheet to the next sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationNextSheet action.
                 * spread.commandManager().setShortcutKey("navigationNextSheet", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationNextSheet: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell down one page of rows.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationPageDown action.
                 * spread.commandManager().setShortcutKey("navigationPageDown", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationPageDown: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell up one page of rows.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationPageUp action.
                 * spread.commandManager().setShortcutKey("navigationPageUp", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationPageUp: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active sheet to the previous sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationPreviousSheet action.
                 * spread.commandManager().setShortcutKey("navigationNextSheet", GC.Spread.Commands.Key.a, false, false, false, false);
                 * spread.commandManager().setShortcutKey("navigationPreviousSheet", GC.Spread.Commands.Key.c, false, false, false, false);
                 */
                var navigationPreviousSheet: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}, isUndo: boolean): any};
                /**
                 * Represents the command used to move the active cell to the next column.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationRight action.
                 * spread.commandManager().setShortcutKey("navigationRight", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationRight: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the first row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the navigationTop action.
                 * spread.commandManager().setShortcutKey("navigationTop", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var navigationTop: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to move the active cell to the previous row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example maps the navigationUp action to a.
                 * spread.commandManager().setShortcutKey('navigationUp', GC.Spread.Commands.Key.a, false, false, false, false); // a
                 */
                var navigationUp: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to open a calculator picker in specified cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.sheetArea {GC.Spread.Sheets.SheetArea} The sheet area.<br />
                 * options.row {number} The rowIndex.<br />
                 * options.col {number} The columnIndex.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var openCalculator: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, row: number, col: number, sheetArea: GC.Spread.Sheets.SheetArea}, isUndo: boolean): any};
                /**
                 * Represents the command used to open a color picker in specified cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.sheetArea {GC.Spread.Sheets.SheetArea} The sheet area.<br />
                 * options.row {number} The rowIndex.<br />
                 * options.col {number} The columnIndex.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var openColorPicker: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, row: number, col: number, sheetArea: GC.Spread.Sheets.SheetArea}, isUndo: boolean): any};
                /**
                 * Represents the command used to open a datetime picker in specified cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.sheetArea {GC.Spread.Sheets.SheetArea} The sheet area.<br />
                 * options.row {number} The rowIndex.<br />
                 * options.col {number} The columnIndex.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var openDateTimePicker: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, row: number, col: number, sheetArea: GC.Spread.Sheets.SheetArea}, isUndo: boolean): any};
                /**
                 * Represents the command used to open a list picker in specified cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.sheetArea {GC.Spread.Sheets.SheetArea} The sheet area.<br />
                 * options.row {number} The rowIndex.<br />
                 * options.col {number} The columnIndex.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var openList: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, row: number, col: number, sheetArea: GC.Spread.Sheets.SheetArea}, isUndo: boolean): any};
                /**
                 * Represents the command used to open a month picker in specified cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.sheetArea {GC.Spread.Sheets.SheetArea} The sheet area.<br />
                 * options.row {number} The rowIndex.<br />
                 * options.col {number} The columnIndex.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var openMonthPicker: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, row: number, col: number, sheetArea: GC.Spread.Sheets.SheetArea}, isUndo: boolean): any};
                /**
                 * Represents the command used to open a slider picker in specified cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.sheetArea {GC.Spread.Sheets.SheetArea} The sheet area.<br />
                 * options.row {number} The rowIndex.<br />
                 * options.col {number} The columnIndex.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var openSlider: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, row: number, col: number, sheetArea: GC.Spread.Sheets.SheetArea}, isUndo: boolean): any};
                /**
                 * Represents the command used to open a time picker in specified cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.sheetArea {GC.Spread.Sheets.SheetArea} The sheet area.<br />
                 * options.row {number} The rowIndex.<br />
                 * options.col {number} The columnIndex.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var openTimePicker: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, row: number, col: number, sheetArea: GC.Spread.Sheets.SheetArea}, isUndo: boolean): any};
                /**
                 * Represents the command used to open the url of the hyperlink cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.url {string} The url string.<br />
                 * options.target {HyperlinkTargetType} The target type, it's default is blank.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var openUrl: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: { sheetName: string, url: string, target?: GC.Spread.Sheets.Hyperlink.HyperlinkTargetType }, isUndo: boolean): any};
                /**
                 * Represents the command used to open a workflow list picker in specified cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.sheetArea {GC.Spread.Sheets.SheetArea} The sheet area.<br />
                 * options.row {number} The rowIndex.<br />
                 * options.col {number} The columnIndex.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var openWorkflowList: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, row: number, col: number, sheetArea: GC.Spread.Sheets.SheetArea}, isUndo: boolean): any};
                /**
                 * Represents the command for grouping a column outline (range group) on a sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.index {number} The outline starting index.<br />
                 * options.count {number} The number of rows or columns to group or ungroup in the outline.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example creates a group.
                 * spread.options.allowUndo = true;
                 * spread.commandManager().execute({cmd: "outlineColumn", sheetName: "Sheet1", index: 3, count: 5});
                 */
                var outlineColumn: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, index: number, count: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command for grouping a row outline (range group) on a sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.index {number} The outline starting index.<br />
                 * options.count {number} The number of rows or columns to group or ungroup in the outline.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example undoes an action.
                 * spread.options.allowUndo = true;
                 * spread.commandManager().execute({cmd: "outlineRow", sheetName: "Sheet1", index: 3, count: 5});
                 */
                var outlineRow: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, index: number, count: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to paste the selected items from the Clipboard to the current sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.shiftCells {GC.Spread.Sheets.InsertShiftCell} The inserted data needs to be moved in the direction.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var paste: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, shiftCells?: GC.Spread.Sheets.InsertShiftCell}): any};
                /**
                 * Represents the command for pasting the floating objects on the sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var pasteFloatingObjects: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to perform a redo of the most recently undone edit or action.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the undo or redo action.
                 * $("#button1").click(function () {
                 *    if(spread.undoManager().canUndo()){
                 *        spread.undoManager().undo();
                 *    }
                 * });
                 * $("#button2").click(function () {
                 *     if(spread.undoManager().canRedo()){
                 *         spread.undoManager().redo();
                 *     }
                 * });
                 */
                var redo: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command for ungrouping a column outline (range group) on a sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.index {number} The outline starting index.<br />
                 * options.count {number} The number of rows or columns to group or ungroup in the outline.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var removeColumnOutline: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, index: number, count: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command for ungrouping a row outline (range group) on a sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.index {number} The outline starting index.<br />
                 * options.count {number} The number of rows or columns to group or ungroup in the outline.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var removeRowOutline: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, index: number, count: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to rename a worksheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.name {string} The sheet's new name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example renames a sheet.
                 * spread.commandManager().execute({cmd: "renameSheet", sheetName: "Sheet1", name: "SheetName"});
                 */
                var renameSheet: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, name: string}, isUndo: boolean): any};
                /**
                 * Represents the command used to resize the column on a worksheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.columns {Array} The resize columns; each item is an object which has firstCol and lastCol.<br />
                 * options.size {number} The size of the column that is being resized.<br />
                 * options.rowHeader {boolean} Whether the column being resized is in the row header area.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var resizeColumn: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, columns: Object[], size: number, rowHeader: boolean}, isUndo: boolean): any};
                /**
                 * Represents the command for resizing floating objects.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.floatingObjects {Array} The names array of floating objects.<br />
                 * options.offsetX The offset left.<br />
                 * options.offsetY The offset top.<br />
                 * options.offsetWidth The offset width.<br />
                 * options.offsetHeight The offset height.
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var resizeFloatingObjects: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, floatingObjects: string[], offsetX: number, offsetY: number, offsetWidth: number, offsetHeight: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to resize the row on a worksheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.rows {Array} The resize rows; each item is an object which has firstRow and lastRow.<br />
                 * options.size {number} The size of the row that is being resized.<br />
                 * options.columnHeader {boolean} Whether the row being resized is in the column header area.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var resizeRow: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, rows: Object[], size: number, columnHeader: boolean}, isUndo: boolean): any};
                /**
                 * Represents the command used to extend the selection to the last row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionBottom action.
                 * spread.commandManager().setShortcutKey('selectionBottom', GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionBottom: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection down one row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionDown action.
                 * spread.commandManager().setShortcutKey('selectionDown', GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionDown: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection to the last column.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionEnd action.
                 * spread.focus();
                 * spread.commandManager().setShortcutKey('selectionEnd', GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionEnd: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection to the first cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionFirst action.
                 * spread.focus();
                 * spread.commandManager().setShortcutKey('selectionFirst', GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionFirst: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection to the first column.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionHome action.
                 * spread.focus();
                 * spread.commandManager().setShortcutKey('selectionHome', GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionHome: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection to the last cell.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionLast action.
                 * spread.focus();
                 * spread.commandManager().setShortcutKey('selectionLast', GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionLast: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection one column to the left.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionLeft action.
                 * spread.commandManager().setShortcutKey('selectionLeft', GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionLeft: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}, isUndo: boolean): any};
                /**
                 * Represents the command used to extend the selection down to include one page of rows.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionPageDown action.
                 * spread.commandManager().setShortcutKey('selectionPageDown', GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionPageDown: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection up to include one page of rows.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionPageUp action.
                 * spread.commandManager().setShortcutKey('selectionPageUp', GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionPageUp: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection one column to the right.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionRight action.
                 * spread.commandManager().setShortcutKey("selectionRight", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionRight: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection to the first row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionTop action.
                 * spread.commandManager().setShortcutKey("selectionTop", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionTop: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to extend the selection up one row.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the selectionUp action.
                 * spread.commandManager().setShortcutKey("selectionUp", GC.Spread.Commands.Key.a, false, false, false, false);
                 */
                var selectionUp: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to select the next control.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example maps the selectNextControl action to the Tab key.
                 * spread.commandManager().setShortcutKey('selectNextControl', GC.Spread.Commands.Key.tab, false, false, false, false); // Tab key
                 * spread.commandManager().setShortcutKey('selectPreviousControl', GC.Spread.Commands.Key.tab, false, true, false, false); // Shift key and Tab key
                 */
                var selectNextControl: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to select the previous control.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 ** options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example maps the selectPreviousControl action to the Shift + Tab key combination.
                 * spread.commandManager().setShortcutKey('selectNextControl', GC.Spread.Commands.Key.tab, false, false, false, false); // Tab key
                 * spread.commandManager().setShortcutKey('selectPreviousControl', GC.Spread.Commands.Key.tab, false, true, false, false); // Shift key and Tab key
                 */
                var selectPreviousControl: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to split resize the column on a worksheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.columns {Array} The resize columns; each item is an object which has firstCol and lastCol.<br />
                 * options.size {number} The size of the column that is being resized.<br />
                 * options.rowHeader {boolean} Whether the column being resized is in the row header area.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var splitResizeColumn: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: { sheetName: string, columns: Object[], size: number, rowHeader: boolean }, isUndo: boolean): any};
                /**
                 * Represents the command used to split resize the row on a worksheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.rows {Array} The resize rows; each item is an object which has firstRow and lastRow.<br />
                 * options.size {number} The size of the row that is being resized.<br />
                 * options.columnHeader {boolean} Whether the row being resized is in the column header area.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var splitResizeRow: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: { sheetName: string, rows: Object[], size: number, columnHeader: boolean }, isUndo: boolean): any};
                /**
                 * Represents the command used to delete columns for table.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.tableName {string} The table name.<br />
                 * options.col {number} The index of the starting column to delete, the col index is based on table index.<br />
                 * options.count {number} The number of columns to delete.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var tableDeleteColumns: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, tableName: string, col: number, count: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to delete rows for table.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.tableName {string} The table name.<br />
                 * options.row {number} The index of the starting row to delete, the row index is based on table index.<br />
                 * options.count {number} The number of rows to delete.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var tableDeleteRows: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, tableName: string, row: number, count: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to insert columns for table.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.tableName {string} The table name.<br />
                 * options.col {number} The index of the starting column to insert, the col index is based on table index.<br />
                 * options.count {number} The number of columns to insert.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var tableInsertColumns: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, tableName: string, col: number, count: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to insert rows for table.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.tableName {string} The table name.<br />
                 * options.row {number} The index of the starting row to insert, the row index is based on table index.<br />
                 * options.count {number} The number of rows to insert.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var tableInsertRows: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, tableName: string, row: number, count: number}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to resize table.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.tableName {string} The table name.<br />
                 * options.resizeToRange {GC.Spread.Sheets.Range} The resized table range.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var tableResize: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, tableName: string, resizeToRange: GC.Spread.Sheets.Range}, isUndo: boolean): boolean};
                /**
                 * Represents the command used to perform an undo of the most recent edit or action.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 * @example
                 * //This example uses the undo or redo action.
                 * $("#button1").click(function () {
                 *      if(spread.undoManager().canUndo()){
                 *          spread.undoManager().undo();
                 *      }
                 *  });
                 * $("#button2").click(function() {
                 *      if(spread.undoManager().canRedo()){
                 *          spread.undoManager().redo();
                 *      }
                 * });
                 */
                var undo: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string}): any};
                /**
                 * Represents the command used to zoom the sheet.
                 * @property {boolean} canUndo - indicates whether the command supports undo and redo operations.
                 * @property {function} execute - performs an execute or undo operation.<br />
                 * The arguments of the execute method are as follows.<br />
                 * context {GC.Spread.Sheets.Workbook} The context of the operation.<br />
                 * options {Object} The options of the operation.<br />
                 * options.sheetName {string} The sheet name.<br />
                 * options.zoomFactor {number} The zoom factor.<br />
                 * isUndo {boolean} <c>true</c> if this is an undo operation; otherwise, <c>false</c>.
                 */
                var zoom: { canUndo: boolean, execute(context: GC.Spread.Sheets.Workbook, options: {sheetName: string, zoomFactor: number}, isUndo: boolean): any};
                /**
                 * Ends a transaction. During the transaction, the changes of the data model will be saved.
                 * @param {GC.Spread.Sheets.Workbook} context The context of the operation.
                 * @param {Object} options The options of the operation.<br />
                 *  @param  {string|Array<string>} [options.sheetName] The sheet name. If the current transaction need change multiple worksheets, the sheetName could be a string Array.
                 */
                function endTransaction(context:  GC.Spread.Sheets.Workbook,  options:  any): void;
                /**
                 * Starts a transaction. During the transaction, the changes of the data model will be saved.
                 * @param {GC.Spread.Sheets.Workbook} context The context of the operation.
                 * @param {Object} options The options of the operation.
                 * @param {string|Array<string>} [options.sheetName] The sheet name. If the current transaction need change multiple worksheets, the sheetName could be a string Array.
                 */
                function startTransaction(context:  GC.Spread.Sheets.Workbook,  options:  any): void;
                /**
                 * Undo the changes made in a transaction.
                 * @param {GC.Spread.Sheets.Workbook} context The context of the operation.
                 * @param {Object} options The options of the operation.<br />
                 * @param {string|Array<string>} [options.sheetName] The sheet name. If the current transaction need change multiple worksheets, the sheetName could be a string Array.
                 */
                function undoTransaction(context:  GC.Spread.Sheets.Workbook,  options:  any): void;
            }

            module Comments{
                /**
                 * Defines the comment state.
                 * @enum {number}
                 * @example
                 * //This example gets the comment state.
                 * var comment = new GC.Spread.Sheets.Comments.Comment();
                 * comment.text("new comment!");
                 * comment.backColor("yellow");
                 * comment.foreColor("green");
                 * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                 * activeSheet.getCell(5,5).comment(comment);
                 * alert(comment.commentState());
                 */
                export enum CommentState{
                    /**
                     * Specifies that the comment is in an active state.
                     */
                    active= 1,
                    /**
                     *Specifies that the comment is in an editing state.
                     */
                    edit= 2,
                    /**
                     * Specifies that the comment is in a normal state.
                     */
                    normal= 3
                }

                /**
                 * Defines when the comment is displayed.
                 * @enum {number}
                 * @example
                 * //This example uses the DisplayMode enumeration.
                 * var comment = new GC.Spread.Sheets.Comments.Comment();
                 * comment.text("new comment!");
                 * comment.backColor("yellow");
                 * comment.foreColor("green");
                 * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                 * activeSheet.getCell(5,5).comment(comment);
                 */
                export enum DisplayMode{
                    /**
                     *  Specifies that the comment is always displayed.
                     */
                    alwaysShown= 1,
                    /**
                     *  Specifies that the comment is displayed only when the pointer hovers over the comment's owner cell.
                     */
                    hoverShown= 2
                }


                export class Comment{
                    /**
                     * Represents a comment.
                     * @class
                     * @param {string} [text] The text of the comment.
                     * @example
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.autoSize(true);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    constructor(text?:  string);
                    /**
                     * Gets or sets whether the comment automatically sizes based on its content.
                     * @param {boolean} value Whether the comment automatically sizes.
                     * @returns {boolean | GC.Spread.Sheets.Comments.Comment} If no value is set, returns whether the comment automatically sizes; otherwise, returns the comment.
                     * @example
                     * //This example uses the autoSize method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.autoSize(true);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    autoSize(value?:  boolean): any;
                    /**
                     * Gets or sets the background color of the comment.
                     * @param {string} value The background color of the comment.
                     * @returns {string | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the background color of the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the backColor method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    backColor(value?:  string): any;
                    /**
                     * Gets or sets the border color for the comment.
                     * @param {string} value The border color for the comment.
                     * @returns {string | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the border color for the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the borderColor method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.borderWidth(2);
                     * comment.borderStyle("dotted");
                     * comment.borderColor("red");
                     * activeSheet.getCell(5,5).comment(comment);
                     * activeSheet.suspendPaint();
                     * activeSheet.resumePaint();
                     */
                    borderColor(value?:  string): any;
                    /**
                     * Gets or sets the border style for the comment.
                     * @param {string} value The border style for the comment.
                     * @returns {string | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the border style for the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the borderStyle method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.borderWidth(2);
                     * comment.borderStyle("dotted");
                     * comment.borderColor("red");
                     * activeSheet.getCell(5,5).comment(comment);
                     * activeSheet.suspendPaint();
                     * activeSheet.resumePaint();
                     */
                    borderStyle(value?:  string): any;
                    /**
                     * Gets or sets the border width for the comment.
                     * @param {number} value The border width for the comment.
                     * @returns {number | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the border width for the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the borderWidth method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.borderWidth(2);
                     * comment.borderStyle("dotted");
                     * comment.borderColor("red");
                     * activeSheet.getCell(5,5).comment(comment);
                     * activeSheet.suspendPaint();
                     * activeSheet.resumePaint();
                     */
                    borderWidth(value?:  number): any;
                    /**
                     * Gets or sets the state of the comment.
                     * @param {GC.Spread.Sheets.Comments.CommentState} value The state of the comment.
                     * @returns {GC.Spread.Sheets.Comments.CommentState | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the state of the comment; otherwise, returns the comment.
                     * @example
                     * //This example gets the comment state.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * activeSheet.getCell(5,5).comment(comment);
                     * alert(comment.commentState());
                     */
                    commentState(value?:  GC.Spread.Sheets.Comments.CommentState): any;
                    /**
                     * Gets or sets the display mode for the comment.
                     * @param {GC.Spread.Sheets.Comments.DisplayMode} value The display mode for the comment.
                     * @returns {GC.Spread.Sheets.Comments.DisplayMode | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the display mode for the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the displayMode method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    displayMode(value?:  GC.Spread.Sheets.Comments.DisplayMode): any;
                    /**
                     * Gets or sets whether the comment dynamically moves.
                     * @param {boolean} value Whether the comment dynamically moves.
                     * @returns {boolean | GC.Spread.Sheets.Comments.Comment} If no value is set, returns whether the comment dynamically moves; otherwise, returns the comment.
                     * @example
                     * //This example uses the dynamicMove method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.dynamicMove(true);
                     * comment.dynamicSize(true);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    dynamicMove(value?:  boolean): any;
                    /**
                     * Gets or sets whether the comment is dynamically sized.
                     * @param {boolean} value Whether the comment is dynamically sized.
                     * @returns {boolean | GC.Spread.Sheets.Comments.Comment} If no value is set, returns whether the comment is dynamically sized; otherwise, returns the comment.
                     * @example
                     * //This example uses the dynamicSize method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.dynamicMove(true);
                     * comment.dynamicSize(true);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    dynamicSize(value?:  boolean): any;
                    /**
                     * Gets or sets the font family for the comment.
                     * @param {string} value The font family for the comment.
                     * @returns {string | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the font family for the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the fontFamily method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.fontFamily("Comic Sans MS");
                     * comment.fontSize("10pt");
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    fontFamily(value?:  string): any;
                    /**
                     * Gets or sets the font size for the comment. Valid value is numbers followed by "pt" (required), such as "12pt".
                     * @param {string} value The font size for the comment.
                     * @returns {string | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the font size for the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the fontSize method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.fontFamily("Comic Sans MS");
                     * comment.fontSize("10pt");
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    fontSize(value?:  string): any;
                    /**
                     * Gets or sets the font style of the comment.
                     * @param {string} value The font style of the comment.
                     * @returns {string | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the font style of the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the fontStyle method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.fontFamily("Comic Sans MS");
                     * comment.fontStyle("normal");
                     * comment.fontWeight("normal");
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    fontStyle(value?:  string): any;
                    /**
                     * Gets or sets the font weight for the comment.
                     * @param {string} value The font weight for the comment.
                     * @returns {string | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the font weight for the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the fontWeight method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.fontFamily("Comic Sans MS");
                     * comment.fontStyle("normal");
                     * comment.fontWeight("normal");
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    fontWeight(value?:  string): any;
                    /**
                     * Gets or sets the text color for the comment.
                     * @param {string} value The text color for the comment.
                     * @returns {string | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the text color for the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the foreColor method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    foreColor(value?:  string): any;
                    /**
                     * Gets or sets the height of the comment.
                     * @param {number} value The height of the comment.
                     * @returns {number | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the height of the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the height method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.height(50);
                     * comment.width(90);
                     * activeSheet.getCell(5,5).comment(comment);
                     * activeSheet.suspendPaint();
                     * activeSheet.resumePaint();
                     */
                    height(value?:  number): any;
                    /**
                     * Gets or sets the horizontal alignment of the comment.
                     * @param {GC.Spread.Sheets.HorizontalAlign} value The horizontal alignment of the comment.
                     * @returns {GC.Spread.Sheets.HorizontalAlign | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the horizontal alignment of the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the horizontalAlign method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.horizontalAlign(GC.Spread.Sheets.HorizontalAlign.center);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    horizontalAlign(value?:  GC.Spread.Sheets.HorizontalAlign): any;
                    /**
                     * Gets or sets the location of the comment.
                     * @param {GC.Spread.Sheets.Point} value The location of the comment.
                     * @returns {GC.Spread.Sheets.Point | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the location of the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the location method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.location(new GC.Spread.Sheets.Point(10, 10));
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    location(value?:  GC.Spread.Sheets.Point): any;
                    /**
                     * Gets or sets the locked setting for the comment.
                     * @param {boolean} value The locked setting for the comment.
                     * @returns {boolean | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the locked setting for the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the locked method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.lockText(false);
                     * comment.locked(false);
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * activeSheet.options.isProtected = true;
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    locked(value?:  boolean): any;
                    /**
                     * Gets or sets the locked text for the comment.
                     * @param {boolean} value The locked text for the comment.
                     * @returns {boolean | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the locked text for the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the lockText method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.lockText(false);
                     * comment.locked(false);
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * activeSheet.options.isProtected = true;
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    lockText(value?:  boolean): any;
                    /**
                     * Gets or sets the opacity of the comment.
                     * @param {number} value The opacity of the comment.
                     * @returns {number | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the opacity of the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the opacity.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.opacity(10);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    opacity(value?:  number): any;
                    /**
                     * Gets or sets the padding for the comment.
                     * @param {GC.Spread.Sheets.Comments.Padding} value The padding for the comment.
                     * @returns {GC.Spread.Sheets.Comments.Padding | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the padding for the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the padding method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.padding(new GC.Spread.Sheets.Comments.Padding(2, 2, 2, 2));
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    padding(value?:  GC.Spread.Sheets.Comments.Padding): any;
                    /**
                     * Gets or sets whether the comment displays a shadow.
                     * @param {boolean} value Whether the comment displays a shadow.
                     * @returns {boolean | GC.Spread.Sheets.Comments.Comment} If no value is set, returns whether the comment displays a shadow; otherwise, returns the comment.
                     * @example
                     * //This example uses the showShadow method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.borderWidth(2);
                     * comment.borderStyle("dotted");
                     * comment.borderColor("red");
                     * comment.showShadow(true);
                     * activeSheet.getCell(5,5).comment(comment);
                     * activeSheet.suspendPaint();
                     * activeSheet.resumePaint();
                     */
                    showShadow(value?:  boolean): any;
                    /**
                     * Gets or sets the text of the comment.
                     * @param {string} value The text of the comment.
                     * @returns {string | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the text of the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the text method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * activeSheet.getCell(5,5).comment(comment);
                     */
                    text(value?:  string): any;
                    /**
                     * Gets or sets the text decoration for the comment.
                     * @param {GC.Spread.Sheets.TextDecorationType} value The text decoration for the comment.
                     * @returns {GC.Spread.Sheets.TextDecorationType | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the text decoration for the comment; otherwise, returns the comment.
                     * @example
                     * //This example uses the textDecoration method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.textDecoration(GC.Spread.Sheets.TextDecorationType.underline);
                     * activeSheet.getCell(5,5).comment(comment);
                     * activeSheet.suspendPaint();
                     * activeSheet.resumePaint();
                     */
                    textDecoration(value?:  GC.Spread.Sheets.TextDecorationType): any;
                    /**
                     * Gets or sets the width of the comment.
                     * @param {number} value The width of the comment.
                     * @returns {number | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the width of the comment; otherwise, returns the comment.
                     * @example
                     * //This example sets the width method.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * comment.height(50);
                     * comment.width(90);
                     * activeSheet.getCell(5,5).comment(comment);
                     * activeSheet.suspendPaint();
                     * activeSheet.resumePaint();
                     */
                    width(value?:  number): any;
                    /**
                     * Gets or sets the z-index of the comment.
                     * @param {number} value The z-index of the comment.
                     * @returns {number | GC.Spread.Sheets.Comments.Comment} If no value is set, returns the z-index of the comment; otherwise, returns the comment.
                     * @example
                     * //This example gets the index.
                     * var comment = new GC.Spread.Sheets.Comments.Comment();
                     * comment.text("new comment!");
                     * comment.backColor("yellow");
                     * comment.foreColor("green");
                     * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                     * activeSheet.getCell(5,5).comment(comment);
                     * alert(comment.zIndex());
                     */
                    zIndex(value?:  number): any;
                }

                export class CommentManager{
                    /**
                     * Represents a comment manager that can manage all comments in a sheet.
                     * @class
                     * @param {GC.Spread.Sheets.Worksheet} sheet The worksheet.
                     */
                    constructor(sheet:  GC.Spread.Sheets.Worksheet);
                    /**
                     * Adds a comment to the cell for the indicated row and column.
                     * @param {number} row The row index of the cell.
                     * @param {number} col The column index of the cell.
                     * @param {string} text The text of the comment.
                     * @returns {GC.Spread.Sheets.Comments.Comment} The comment that has been added to the cell.
                     */
                    add(row:  number,  col:  number,  text:  string): GC.Spread.Sheets.Comments.Comment;
                    /**
                     * Gets all comments in the sheet.
                     * @returns {Array.<GC.Spread.Sheets.Comments.Comment>}
                     */
                    all(): GC.Spread.Sheets.Comments.Comment[];
                    /**
                     * Clears all of the comments in the indicated range on the sheet. When the range is not specified, it clears all the comments in the sheet.
                     * @param {GC.Spread.Sheets.Range} range The range that you want clear all comments from.
                     */
                    clear(range:  GC.Spread.Sheets.Range): void;
                    /**
                     * Gets the comment in the cell with the indicated row and column.
                     * @param {number} row The row index of the cell.
                     * @param {number} col The column index of the cell.
                     * @returns {GC.Spread.Sheets.Comments.Comment} The comment in the indicated cell.
                     */
                    get(row:  number,  col:  number): GC.Spread.Sheets.Comments.Comment;
                    /**
                     *Removes the comment from the cell for the indicated row and column.
                     * @param {number} row The row index of the cell.
                     * @param {number} col The column index of the cell.
                     */
                    remove(row:  number,  col:  number): void;
                }

                export class Padding{
                    /**
                     * Represents the padding information.
                     * @class
                     * @param {number} top The top padding.
                     * @param {number} right The right padding.
                     * @param {number} bottom The bottom padding.
                     * @param {number} left The left padding.
                     */
                    constructor(top?:  number,  right?:  number,  bottom?:  number,  left?:  number);
                }
            }

            module ConditionalFormatting{

                export interface IIconInfo{
                    iconSetType:GC.Spread.Sheets.ConditionalFormatting.IconSetType;
                    iconIndex:number;
                }

                /**
                 * Specifies the average condition type.
                 * @enum {number}
                 * @example
                 * //This example creates an average rule.
                 * activeSheet.setValue(0,0, 1,3);
                 * activeSheet.setValue(1,0, 50,3);
                 * activeSheet.setValue(2,0, 100,3);
                 * activeSheet.setValue(3,0, 2,3);
                 * activeSheet.setValue(4,0, 60,3);
                 * activeSheet.setValue(5,0, 90,3);
                 * activeSheet.setValue(6,0, 3,3);
                 * activeSheet.setValue(7,0, 40,3);
                 * activeSheet.setValue(8,0, 70,3);
                 * activeSheet.setValue(9,0, 5,3);
                 * activeSheet.setValue(10,0, 35,3);
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                 * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                 * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.averageRule);
                 * rule.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 3)]);
                 * rule.style(style);
                 * rule.type(GC.Spread.Sheets.ConditionalFormatting.AverageConditionType.above);
                 * activeSheet.conditionalFormats.addRule(rule);
                 */
                export enum AverageConditionType{
                    /** Specifies the above average condition.
                     * @type {number}
                     */
                    above= 0,
                    /** Specifies the below average condition.
                     * @type {number}
                     */
                    below= 1,
                    /** Specifies the above average or equal average condition.
                     * @type {number}
                     */
                    equalOrAbove= 2,
                    /** Specifies the below average or equal average condition.
                     * @type {number}
                     */
                    equalOrBelow= 3,
                    /** Specifies the above standard deviation condition.
                     * @type {number}
                     */
                    above1StdDev= 4,
                    /** Specifies the below standard deviation condition.
                     * @type {number}
                     */
                    below1StdDev= 5,
                    /** Specifies above the two standard deviation condition.
                     * @type {number}
                     */
                    above2StdDev= 6,
                    /** Specifies below the two standard deviation condition.
                     * @type {number}
                     */
                    below2StdDev= 7,
                    /** Specifies above the three standard deviation condition.
                     * @type {number}
                     */
                    above3StdDev= 8,
                    /** Specifies below the three standard deviation condition.
                     * @type {number}
                     */
                    below3StdDev= 9
                }

                /**
                 * Specifies the data bar direction.
                 * @enum {number}
                 * @example
                 * //This example creates a data bar rule.
                 * activeSheet.setValue(0,0,1,3);
                 * activeSheet.setValue(1,0,15,3);
                 * activeSheet.setValue(2,0,25,3);
                 * activeSheet.setValue(3,0,-1,3);
                 * var dataBarRule = new GC.Spread.Sheets.ConditionalFormatting.DataBarRule();
                 * dataBarRule.ranges([new GC.Spread.Sheets.Range(0,0,4,1)]);
                 * dataBarRule.minType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * dataBarRule.minValue(-1);
                 * dataBarRule.maxType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * dataBarRule.maxValue(40);
                 * dataBarRule.color("green");
                 * dataBarRule.showBorder(true);
                 * dataBarRule.borderColor("orange");
                 * dataBarRule.dataBarDirection(GC.Spread.Sheets.ConditionalFormatting.BarDirection.leftToRight);
                 * dataBarRule.negativeFillColor("yellow");
                 * dataBarRule.useNegativeFillColor(true);
                 * dataBarRule.negativeBorderColor("red");
                 * dataBarRule.useNegativeBorderColor(true);
                 * dataBarRule.axisPosition(GC.Spread.Sheets.ConditionalFormatting.DataBarAxisPosition.automatic);
                 * dataBarRule.axisColor("blue");
                 * dataBarRule.showBarOnly(false);
                 * activeSheet.conditionalFormats.addRule(dataBarRule);
                 */
                export enum BarDirection{
                    /** Specifies whether to display the data bar from left to right.
                     * @type {number}
                     */
                    leftToRight= 0,
                    /** Specifies whether to display the data bar from right to left.
                     * @type {number}
                     */
                    rightToLeft= 1
                }

                /**
                 * Specifies the color compare type.
                 * @enum {number}
                 * @example
                 * //This example filters using colors.
                 * activeSheet.suspendPaint();
                 * var rowFilter = new GC.Spread.Sheets.Filter.HideRowFilter(new GC.Spread.Sheets.Range(-1, 0, -1, 1));
                 * activeSheet.rowFilter(rowFilter);
                 * activeSheet.getCell(0, 0).value("A1").backColor("blue");
                 * activeSheet.getCell(1, 0).value("A2").backColor("yellow");
                 * activeSheet.getCell(2, 0).value("A3").backColor("red");
                 * activeSheet.getCell(3, 0).value("A4").backColor("green");
                 * activeSheet.getCell(4, 0).value("A5").backColor("yellow");
                 * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.colorCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.ColorCompareType.backgroundColor, expected: "yellow"});
                 * var filter = activeSheet.rowFilter();
                 * filter.addFilterItem(0, nCondition);
                 * filter.filter(0);
                 * activeSheet.resumePaint();
                 */
                export enum ColorCompareType{
                    /** Indicates whether the cell background color is equal to a specified color.
                     * @type {number}
                     */
                    backgroundColor= 0,
                    /** Indicates whether the cell foreground color is equal to a specified color.
                     * @type {number}
                     */
                    foregroundColor= 1
                }

                /**
                 * Specifies the comparison operator.
                 * @enum {number}
                 * @example
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createDateValidator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between, new Date(2012, 11, 31), new Date(2013, 11, 31));
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Enter a date between 12/31/2012 and 12/31/2013.");
                 * dv.inputTitle("Tip");
                 * activeSheet.getCell(1, -1).validator(dv);
                 */
                export enum ComparisonOperators{
                    /** Determines whether a cell value is equal to the parameter value.
                     * @type {number}
                     */
                    equalsTo= 0,
                    /** Determines whether a cell value is not equal to the parameter value.
                     * @type {number}
                     */
                    notEqualsTo= 1,
                    /** Determines whether a cell value is greater than the parameter value.
                     * @type {number}
                     */
                    greaterThan= 2,
                    /** Determines whether a cell value is greater than or equal to the parameter value.
                     * @type {number}
                     */
                    greaterThanOrEqualsTo= 3,
                    /** Determines whether a cell value is less than the parameter value.
                     * @type {number}
                     */
                    lessThan= 4,
                    /** Determines whether a cell value is less than or equal to the parameter value.
                     * @type {number}
                     */
                    lessThanOrEqualsTo= 5,
                    /** Determines whether a cell value is between the two parameter values.
                     * @type {number}
                     */
                    between= 6,
                    /** Determines whether a cell value is not between the two parameter values.
                     * @type {number}
                     */
                    notBetween= 7
                }

                /**
                 * Specifies the condition type.
                 * @enum {number}
                 */
                export enum ConditionType{
                    /** Specifies the relation condition.
                     * @type {number}
                     */
                    relationCondition= 0,
                    /** Specifies the number condition.
                     * @type {number}
                     */
                    numberCondition= 1,
                    /** Specifies the text condition.
                     * @type {number}
                     */
                    textCondition= 2,
                    /** Specifies the color condition.
                     * @type {number}
                     */
                    colorCondition= 3,
                    /** Specifies the formula condition.
                     * @type {number}
                     */
                    formulaCondition= 4,
                    /** Specifies the date condition.
                     * @type {number}
                     */
                    dateCondition= 5,
                    /** Specifies the dateex condition.
                     * @type {number}
                     */
                    dateExCondition= 6,
                    /** Specifies the text length condition.
                     * @type {number}
                     */
                    textLengthCondition= 7,
                    /** Specifies the top 10 condition.
                     * @type {number}
                     */
                    top10Condition= 8,
                    /** Specifies the unique condition.
                     * @type {number}
                     */
                    uniqueCondition= 9,
                    /** Specifies the average condition.
                     * @type {number}
                     */
                    averageCondition= 10,
                    /** Specifies the cell value condition.
                     * @type {number}
                     */
                    cellValueCondition= 11,
                    /** Specifies the area condition.
                     * @type {number}
                     */
                    areaCondition= 12
                }

                /**
                 * Specifies the custom value type.
                 * @enum {number}
                 * @example
                 * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.formulaCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.CustomValueType.nonEmpty});
                 * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                 * validator.ignoreBlank(false);
                 * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                 * spread.options.highlightInvalidData = true;
                 */
                export enum CustomValueType{
                    /** Indicates whether the cell value is empty or null.
                     * @type {number}
                     */
                    empty= 0,
                    /** Indicates whether the cell value is not empty or null.
                     * @type {number}
                     */
                    nonEmpty= 1,
                    /** Indicates whether the cell value contains a calculation error.
                     * @type {number}
                     */
                    error= 2,
                    /** Indicates whether the cell value does not contain a calculation error.
                     * @type {number}
                     */
                    nonError= 3,
                    /** Indicates whether the cell value is a formula.
                     * @type {number}
                     */
                    formula= 4
                }

                /**
                 * Specifies the position of the data bar's axis.
                 * @enum {number}
                 * @example
                 * //This example creates a data bar rule.
                 * activeSheet.setValue(0,0,1,3);
                 * activeSheet.setValue(1,0,15,3);
                 * activeSheet.setValue(2,0,25,3);
                 * activeSheet.setValue(3,0,-1,3);
                 * var dataBarRule = new GC.Spread.Sheets.ConditionalFormatting.DataBarRule();
                 * dataBarRule.ranges([new GC.Spread.Sheets.Range(0,0,4,1)]);
                 * dataBarRule.minType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * dataBarRule.minValue(-1);
                 * dataBarRule.maxType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * dataBarRule.maxValue(40);
                 * dataBarRule.color("green");
                 * dataBarRule.showBorder(true);
                 * dataBarRule.borderColor("orange");
                 * dataBarRule.dataBarDirection(GC.Spread.Sheets.ConditionalFormatting.BarDirection.leftToRight);
                 * dataBarRule.negativeFillColor("yellow");
                 * dataBarRule.useNegativeFillColor(true);
                 * dataBarRule.negativeBorderColor("red");
                 * dataBarRule.useNegativeBorderColor(true);
                 * dataBarRule.axisPosition(GC.Spread.Sheets.ConditionalFormatting.DataBarAxisPosition.automatic);
                 * dataBarRule.axisColor("blue");
                 * dataBarRule.showBarOnly(false);
                 * activeSheet.conditionalFormats.addRule(dataBarRule);
                 */
                export enum DataBarAxisPosition{
                    /** Specifies whether to display at a variable position based on negative values.
                     * @type {number}
                     */
                    automatic= 0,
                    /** Specifies whether to display at the cell midpoint.
                     * @type {number}
                     */
                    cellMidPoint= 1,
                    /** Specifies whether to display value bars in the same direction as positive values.
                     * @type {number}
                     */
                    none= 2
                }

                /**
                 * Specifies the date compare type.
                 * @enum {number}
                 * @example
                 * //This example validates cell data.
                 * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.dateCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.DateCompareType.before, expected: new Date(2012, 11, 31)});
                 * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                 * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                 * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                 * spread.options.highlightInvalidData = true;
                 * activeSheet.setValue(0, 0, new Date(2012, 12, 12));
                 */
                export enum DateCompareType{
                    /** Indicates whether the date time is equal to a certain time.
                     * @type {number}
                     */
                    equalsTo= 0,
                    /** Indicates whether the date time is not equal to a certain time.
                     * @type {number}
                     */
                    notEqualsTo= 1,
                    /** Indicates whether the date time is before a certain time.
                     * @type {number}
                     */
                    before= 2,
                    /** Indicates whether the date time is before or equal to a certain time.
                     * @type {number}
                     */
                    beforeEqualsTo= 3,
                    /** Indicates whether the date time is after a certain time.
                     * @type {number}
                     */
                    after= 4,
                    /** Indicates whether the date time is after or equal to a certain time.
                     * @type {number}
                     */
                    afterEqualsTo= 5
                }

                /**
                 * Specifies the date occurring type.
                 * @enum {number}
                 * @example
                 * //This example creates a rule.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule(GC.Spread.Sheets.ConditionalFormatting.RuleType.dateOccurringRule, [new GC.Spread.Sheets.Range(0,0,10,1)], style, null, null, null, null, null, GC.Spread.Sheets.ConditionalFormatting.DateOccurringType.nextWeek);
                 * activeSheet.conditionalFormats.addRule(rule);
                 * var d = new Date();
                 * activeSheet.setValue(0, 0, d);
                 * activeSheet.setValue(1, 0, new Date(d.setDate(d.getDate()+1)));
                 * activeSheet.setValue(2, 0, new Date(d.setDate(d.getDate()+5)));
                 * activeSheet.setValue(3, 0,new Date(d.setDate(d.getDate()+6)));
                 * activeSheet.setValue(4, 0,new Date(d.setDate(d.getDate()+7)));
                 * activeSheet.setValue(5, 0, new Date(d.setDate(d.getDate()+8)));
                 */
                export enum DateOccurringType{
                    /** Specifies today.
                     * @type {number}
                     */
                    today= 0,
                    /** Specifies yesterday.
                     * @type {number}
                     */
                    yesterday= 1,
                    /** Specifies tomorrow.
                     * @type {number}
                     */
                    tomorrow= 2,
                    /** Specifies the last seven days.
                     * @type {number}
                     */
                    last7Days= 3,
                    /** Specifies this month.
                     * @type {number}
                     */
                    thisMonth= 4,
                    /** Specifies last month.
                     * @type {number}
                     */
                    lastMonth= 5,
                    /** Specifies next month.
                     * @type {number}
                     */
                    nextMonth= 6,
                    /** Specifies this week.
                     * @type {number}
                     */
                    thisWeek= 7,
                    /** Specifies last week.
                     * @type {number}
                     */
                    lastWeek= 8,
                    /** Specifies next week.
                     * @type {number}
                     */
                    nextWeek= 9,
                    /** Specifies next Quarter.
                     * @type {number}
                     */
                    nextQuarter= 10,
                    /** Specifies this Quarter.
                     * @type {number}
                     */
                    thisQuarter= 11,
                    /** Specifies last Quarter.
                     * @type {number}
                     */
                    lastQuarter= 12,
                    /** Specifies next Year.
                     * @type {number}
                     */
                    nextYear= 13,
                    /** Specifies this Year.
                     * @type {number}
                     */
                    thisYear= 14,
                    /** Specifies last Year.
                     * @type {number}
                     */
                    lastYear= 15
                }

                /**
                 * Specifies the general operator.
                 * @enum {number}
                 * @example
                 * //This example validates a cell value.
                 * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.cellValueCondition);
                 * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.GeneralComparisonOperators.equalsTo);
                 * nCondition.expected(0);
                 * //When the option is false, the validation fails and the red alert is displayed.
                 * //When the option is true, the blank cell is treated as zero and the validation is successful.
                 * nCondition.treatNullValueAsZero(false);
                 * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                 * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                 * validator.ignoreBlank(false);
                 * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                 * spread.options.highlightInvalidData = true;
                 * activeSheet.setValue(0, 0, null);
                 * alert(validator.value2());
                 */
                export enum GeneralComparisonOperators{
                    /** Indicates whether the number is equal to a specified number.
                     * @type {number}
                     */
                    equalsTo= 0,
                    /** Indicates whether the number is not equal to a specified number.
                     * @type {number}
                     */
                    notEqualsTo= 1,
                    /** Indicates whether the number is greater than a specified number.
                     * @type {number}
                     */
                    greaterThan= 2,
                    /** Indicates whether the number is greater than or equal to a specified number.
                     * @type {number}
                     */
                    greaterThanOrEqualsTo= 3,
                    /** Indicates whether the number is less than a specified number.
                     * @type {number}
                     */
                    lessThan= 4,
                    /** Indicates whether the number is less than or equal to a specified number.
                     * @type {number}
                     */
                    lessThanOrEqualsTo= 5
                }

                /**
                 * Specifies the icon set.
                 * @enum {number}
                 * @example
                 * //This example creates a rule.
                 * activeSheet.setValue(0,0,1,3);
                 * activeSheet.setValue(1,0,15,3);
                 * activeSheet.setValue(2,0,25,3);
                 * activeSheet.setValue(3,0,-1,3);
                 * var iconSetRule = new GC.Spread.Sheets.ConditionalFormatting.IconSetRule();
                 * iconSetRule.ranges([new GC.Spread.Sheets.Range(0,0,4,1)]);
                 * iconSetRule.iconSetType(GC.Spread.Sheets.ConditionalFormatting.IconSetType.fourTrafficLights);
                 * var iconCriteria = iconSetRule.iconCriteria();
                 * iconCriteria[0] = new GC.Spread.Sheets.ConditionalFormatting.IconCriterion(true, GC.Spread.Sheets.ConditionalFormatting.IconValueType.number, 1);
                 * iconCriteria[1] = new GC.Spread.Sheets.ConditionalFormatting.IconCriterion(true, GC.Spread.Sheets.ConditionalFormatting.IconValueType.number, 10);
                 * iconCriteria[2] = new GC.Spread.Sheets.ConditionalFormatting.IconCriterion(true, GC.Spread.Sheets.ConditionalFormatting.IconValueType.number, 20);
                 * iconSetRule.reverseIconOrder(false);
                 * iconSetRule.showIconOnly(false);
                 * activeSheet.conditionalFormats.addRule(iconSetRule);
                 */
                export enum IconSetType{
                    /** Specifies three colored arrows.
                     * @type {number}
                     */
                    threeArrowsColored= 0,
                    /** Specifies three gray arrows.
                     * @type {number}
                     */
                    threeArrowsGray= 1,
                    /** Specifies three trangles.
                     * @type {number}
                     */
                    threeTriangles= 2,
                    /** Specifies three stars.
                     * @type {number}
                     */
                    threeStars= 3,
                    /** Specifies three flags.
                     * @type {number}
                     */
                    threeFlags= 4,
                    /** Specifies three traffic lights (unrimmed).
                     * @type {number}
                     */
                    threeTrafficLightsUnrimmed= 5,
                    /** Specifies three traffic lights (rimmed).
                     * @type {number}
                     */
                    threeTrafficLightsRimmed= 6,
                    /** Specifies three signs.
                     * @type {number}
                     */
                    threeSigns= 7,
                    /** Specifies three symbols (circled).
                     * @type {number}
                     */
                    threeSymbolsCircled= 8,
                    /** Specifies three symbols (uncircled).
                     * @type {number}
                     */
                    threeSymbolsUncircled= 9,
                    /** Specifies four colored arrows.
                     * @type {number}
                     */
                    fourArrowsColored= 10,
                    /** Specifies four gray arrows.
                     * @type {number}
                     */
                    fourArrowsGray= 11,
                    /** Specifies four red to black icons.
                     * @type {number}
                     */
                    fourRedToBlack= 12,
                    /** Specifies four ratings.
                     * @type {number}
                     */
                    fourRatings= 13,
                    /** Specifies four traffic lights.
                     * @type {number}
                     */
                    fourTrafficLights= 14,
                    /** Specifies five colored arrows.
                     * @type {number}
                     */
                    fiveArrowsColored= 15,
                    /** Specifies five gray arrows.
                     * @type {number}
                     */
                    fiveArrowsGray= 16,
                    /** Specifies five ratings.
                     * @type {number}
                     */
                    fiveRatings= 17,
                    /** Specifies five quarters.
                     * @type {number}
                     */
                    fiveQuarters= 18,
                    /** Specifies five boxes.
                     * @type {number}
                     */
                    fiveBoxes= 19,
                    /** Specifies no cell icon.
                     * @type {number}
                     */
                    noIcons= 20
                }

                /**
                 * Specifies the icon value type.
                 * @enum {number}
                 * @example
                 * //This example creates a rule.
                 * activeSheet.setValue(0,0,1,3);
                 * activeSheet.setValue(1,0,15,3);
                 * activeSheet.setValue(2,0,25,3);
                 * activeSheet.setValue(3,0,-1,3);
                 * var iconSetRule = new GC.Spread.Sheets.ConditionalFormatting.IconSetRule();
                 * iconSetRule.ranges([new GC.Spread.Sheets.Range(0,0,4,1)]);
                 * iconSetRule.iconSetType(GC.Spread.Sheets.ConditionalFormatting.IconSetType.fourTrafficLights);
                 * var iconCriteria = iconSetRule.iconCriteria();
                 * iconCriteria[0] = new GC.Spread.Sheets.ConditionalFormatting.IconCriterion(true, GC.Spread.Sheets.ConditionalFormatting.IconValueType.number, 1);
                 * iconCriteria[1] = new GC.Spread.Sheets.ConditionalFormatting.IconCriterion(true, GC.Spread.Sheets.ConditionalFormatting.IconValueType.number, 10);
                 * iconCriteria[2] = new GC.Spread.Sheets.ConditionalFormatting.IconCriterion(true, GC.Spread.Sheets.ConditionalFormatting.IconValueType.number, 20);
                 * iconSetRule.reverseIconOrder(false);
                 * iconSetRule.showIconOnly(false);
                 * activeSheet.conditionalFormats.addRule(iconSetRule);
                 */
                export enum IconValueType{
                    /** Indicates whether to return a specified number directly.
                     * @type {number}
                     */
                    number= 1,
                    /** Indicates whether to return the percentage of a cell value in a specified cell range.
                     * @type {number}
                     */
                    percent= 4,
                    /** Indicates whether to return the result of a formula calculation.
                     * @type {number}
                     */
                    formula= 7,
                    /** Indicates whether to return the percentile of a cell value in a specified cell range.
                     * @type {number}
                     */
                    percentile= 5
                }

                /**
                 * Specifies the relation operator.
                 * @enum {number}
                 * @example
                 * //This example validates data.
                 * var condition1 = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.dateCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.DateCompareType.afterEqualsTo, expected: new Date(2012, 11, 31)});
                 * var condition2 = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.dateCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.DateCompareType.beforeEqualsTo, expected: new Date(2013, 11, 31)});
                 * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.relationCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.LogicalOperators.and, item1: condition1, item2: condition2});
                 * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                 * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                 * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                 * spread.options.highlightInvalidData = true;
                 * activeSheet.setValue(0, 0, new Date(2012, 11, 25));
                 */
                export enum LogicalOperators{
                    /** Specifies the Or relation.
                     * @type {number}
                     */
                    or= 0,
                    /** Specifies the And relation.
                     * @type {number}
                     */
                    and= 1
                }

                /**
                 * Specifies the quarter type.
                 * @enum {number}
                 */
                export enum QuarterType{
                    /** Indicates the first quarter of a year.
                     * @type {number}
                     */
                    quarter1= 0,
                    /** Indicates the second quarter of a year.
                     * @type {number}
                     */
                    quarter2= 1,
                    /** Indicates the third quarter of a year.
                     * @type {number}
                     */
                    quarter3= 2,
                    /** Indicates the fourth quarter of a year.
                     * @type {number}
                     */
                    quarter4= 3
                }

                /**
                 * Specifies the rule type.
                 * @enum {number}
                 */
                export enum RuleType{
                    /** Specifies the base rule of the condition.
                     * @type {number}
                     */
                    conditionRuleBase= 0,
                    /** Specifies the cell value rule.
                     * @type {number}
                     */
                    cellValueRule= 1,
                    /** Specifies the specific text rule.
                     * @type {number}
                     */
                    specificTextRule= 2,
                    /** Specifies the formula rule.
                     * @type {number}
                     */
                    formulaRule= 3,
                    /** Specifies the date occurring rule.
                     * @type {number}
                     */
                    dateOccurringRule= 4,
                    /** Specifies the top 10 rule.
                     * @type {number}
                     */
                    top10Rule= 5,
                    /** Specifies the unique rule.
                     * @type {number}
                     */
                    uniqueRule= 6,
                    /** Specifies the duplicate rule.
                     * @type {number}
                     */
                    duplicateRule= 7,
                    /** Specifies the average rule.
                     * @type {number}
                     */
                    averageRule= 8,
                    /** Specifies the two scale rule.
                     * @type {number}
                     */
                    twoScaleRule= 10,
                    /** Specifies the three scale rule.
                     * @type {number}
                     */
                    threeScaleRule= 11,
                    /** Specifies the data bar rule.
                     * @type {number}
                     */
                    dataBarRule= 12,
                    /** Specifies the icon set rule.
                     * @type {number}
                     */
                    iconSetRule= 13
                }

                /**
                 * Specifies the scale value type.
                 * @enum {number}
                 * @example
                 * //This example creates a data bar rule.
                 * activeSheet.setValue(0,0,1,3);
                 * activeSheet.setValue(1,0,15,3);
                 * activeSheet.setValue(2,0,25,3);
                 * activeSheet.setValue(3,0,-1,3);
                 * var dataBarRule = new GC.Spread.Sheets.ConditionalFormatting.DataBarRule();
                 * dataBarRule.ranges([new GC.Spread.Sheets.Range(0,0,4,1)]);
                 * dataBarRule.minType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * dataBarRule.minValue(-1);
                 * dataBarRule.maxType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * dataBarRule.maxValue(40);
                 * dataBarRule.color("green");
                 * dataBarRule.showBorder(true);
                 * dataBarRule.borderColor("orange");
                 * dataBarRule.dataBarDirection(GC.Spread.Sheets.ConditionalFormatting.BarDirection.leftToRight);
                 * dataBarRule.negativeFillColor("yellow");
                 * dataBarRule.useNegativeFillColor(true);
                 * dataBarRule.negativeBorderColor("red");
                 * dataBarRule.useNegativeBorderColor(true);
                 * dataBarRule.axisPosition(GC.Spread.Sheets.ConditionalFormatting.DataBarAxisPosition.automatic);
                 * dataBarRule.axisColor("blue");
                 * dataBarRule.showBarOnly(false);
                 * activeSheet.conditionalFormats.addRule(dataBarRule);
                 */
                export enum ScaleValueType{
                    /** Indicates whether to return a specified number directly.
                     * @type {number}
                     */
                    number= 0,
                    /** Indicates whether to return the lowest value in a specified cell range.
                     * @type {number}
                     */
                    lowestValue= 1,
                    /** Indicates whether to return the highest value in a specified cell range.
                     * @type {number}
                     */
                    highestValue= 2,
                    /** Indicates whether to return the percentage of a cell value in a specified cell range.
                     * @type {number}
                     */
                    percent= 3,
                    /** Indicates whether to return the percentile of a cell value in a specified cell range.
                     * @type {number}
                     */
                    percentile= 4,
                    /** Indicates whether to return the automatic minimum value in a specified range.
                     * @type {number}
                     */
                    automin= 5,
                    /** Indicates whether to return the result of a formula calculation.
                     * @type {number}
                     */
                    formula= 6,
                    /** Indicates whether to return the automatic maximum value in a specified range.
                     * @type {number}
                     */
                    automax= 7
                }

                /**
                 * Specifies the text compare type.
                 * @enum {number}
                 * @example
                 * //This example uses the TextCompareType enumeration.
                 * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.textCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.TextCompareType.contains, expected: "test"});
                 * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                 * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                 * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                 * spread.options.highlightInvalidData = true;
                 * activeSheet.setValue(0, 0, "testing");
                 * //Type text in 0,0 that does not contain "test" to see invalid symbol
                 */
                export enum TextCompareType{
                    /** Indicates whether the string is equal to a specified string.
                     * @type {number}
                     */
                    equalsTo= 0,
                    /** Indicates whether the string is not equal to a specified string.
                     * @type {number}
                     */
                    notEqualsTo= 1,
                    /** Indicates whether the string starts with a specified string.
                     * @type {number}
                     */
                    beginsWith= 2,
                    /** Indicates whether the string does not start with a specified string.
                     * @type {number}
                     */
                    doesNotBeginWith= 3,
                    /** Indicates whether the string ends with a specified string.
                     * @type {number}
                     */
                    endsWith= 4,
                    /** Indicates whether the string does not end with a specified string.
                     * @type {number}
                     */
                    doesNotEndWith= 5,
                    /** Indicates whether the string contains a specified string.
                     * @type {number}
                     */
                    contains= 6,
                    /** Indicates whether the string does not contain a specified string.
                     * @type {number}
                     */
                    doesNotContain= 7
                }

                /**
                 * Specifies the text comparison operator.
                 * @enum {number}
                 * @example
                 * //This example creates a rule.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * var ranges=[new GC.Spread.Sheets.Range(0,0,10,1)];
                 * activeSheet.conditionalFormats.addSpecificTextRule(GC.Spread.Sheets.ConditionalFormatting.TextComparisonOperators.contains, "test", style, ranges);
                 * activeSheet.setValue(0, 0, "testing");
                 * activeSheet.setValue(1, 0, "test");
                 * activeSheet.setValue(2, 0, "a");
                 * activeSheet.setValue(3, 0, "t");
                 */
                export enum TextComparisonOperators{
                    /** Determines whether a cell value contains the parameter value.
                     * @type {number}
                     */
                    contains= 0,
                    /** Determines whether a cell value does not contain the parameter value.
                     * @type {number}
                     */
                    doesNotContain= 1,
                    /** Determines whether a cell value begins with the parameter value.
                     * @type {number}
                     */
                    beginsWith= 2,
                    /** Determines whether a cell value ends with the parameter value.
                     * @type {number}
                     */
                    endsWith= 3
                }

                /**
                 * Specifies the top 10 condition type.
                 * @enum {number}
                 * @example
                 * //This example uses the Top10ConditionType enumeration.
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = "red";
                 * var ranges=[new GC.Spread.Sheets.Range(0,0,10,1)];
                 * activeSheet.conditionalFormats.addTop10Rule(GC.Spread.Sheets.ConditionalFormatting.Top10ConditionType.top, 2, style, ranges);
                 * activeSheet.setValue(0, 0, 1);
                 * activeSheet.setValue(1, 0, 50);
                 * activeSheet.setValue(2, 0, 11);
                 * activeSheet.setValue(3, 0, 5);
                 */
                export enum Top10ConditionType{
                    /** Specifies the top condition.
                     * @type {number}
                     */
                    top= 0,
                    /** Specifies the bottom condition.
                     * @type {number}
                     */
                    bottom= 1
                }


                export class Condition{
                    /**
                     * Represents a conditional item using the parameter object.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ConditionType} conditionType
                     * @param {Object} args
                     * @constructor
                     * @example
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.averageCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.AverageConditionType.above});
                     * nCondition.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 3)]);
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, 5);
                     * activeSheet.setValue(1, 0, 15);
                     */
                    constructor(conditionType:  GC.Spread.Sheets.ConditionalFormatting.ConditionType,  args:  Object);
                    /**
                     * Gets or sets the rule compare type.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.LogicalOperators | GC.Spread.Sheets.ConditionalFormatting.GeneralComparisonOperators | GC.Spread.Sheets.ConditionalFormatting.TextCompareType | GC.Spread.Sheets.ConditionalFormatting.ColorCompareType | GC.Spread.Sheets.ConditionalFormatting.DateCompareType} value The rule compare type.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.LogicalOperators | GC.Spread.Sheets.ConditionalFormatting.GeneralComparisonOperators | GC.Spread.Sheets.ConditionalFormatting.TextCompareType | GC.Spread.Sheets.ConditionalFormatting.ColorCompareType | GC.Spread.Sheets.ConditionalFormatting.DateCompareType | GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set, returns the rule compare type; otherwise, returns the condition.
                     * @example
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.textCondition);
                     * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.TextCompareType.contains);
                     * nCondition.expected("test");
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, "testing");
                     */
                    compareType(value?:  GC.Spread.Sheets.ConditionalFormatting.LogicalOperators | GC.Spread.Sheets.ConditionalFormatting.GeneralComparisonOperators | GC.Spread.Sheets.ConditionalFormatting.TextCompareType | GC.Spread.Sheets.ConditionalFormatting.ColorCompareType | GC.Spread.Sheets.ConditionalFormatting.DateCompareType): any;
                    /**
                     * Evaluates the condition using the specified evaluator.
                     * @param {Object} evaluator The evaluator that can evaluate an expression or a function.
                     * @param {number} baseRow The base row index for evaluation.
                     * @param {number} baseColumn The base column index for evaluation.
                     * @param {Object} actualObj The actual value of object1 for evaluation.
                     * @returns {boolean} <c>true</c> if the result is successful; otherwise, <c>false</c>.
                     */
                    evaluate(evaluator:  Object,  baseRow:  number,  baseColumn:  number,  actualObj:  Object): boolean;
                    /**
                     * Gets or sets the expected value.
                     * @param {Object} value The expected value.
                     * @returns {object | GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set, returns the expected value; otherwise, returns the condition.
                     * @example
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.dateCondition);
                     * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.DateCompareType.before);
                     * nCondition.expected(new Date(2012, 11, 31));
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, new Date(2012, 12, 12));
                     */
                    expected(value?:  any): any;
                    /**
                     * Gets or sets the expected formula.
                     * @param {string | number} formulaOrBaseRow The expected formula or base row.
                     * @param {number} baseColumn The base column.
                     * @returns {string | GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set or baseRow and baseColumn is set, returns the expected formula; otherwise, returns the condition.
                     * @example
                     * var textLengthCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.textLengthCondition);
                     * textLengthCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.GeneralComparisonOperators.greaterThan);
                     * textLengthCondition.formula("$C$1"); // formula used to calculate a number.
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(textLengthCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, "abcf");
                     * //Set value 3 to $C$1, after this code, the value in Cell(0,0) is valid.
                     * activeSheet.setValue(0, 2, 3);
                     * //Set value 5 to $C$1, after this code, the value in Cel(0,0) is invalid.
                     * // activeSheet.setValue(0, 2, 5);
                     */
                    formula(formulaOrBaseRow?:  string | number,  baseColumn ?:  number): any;
                    /**
                     * Creates a date extend condition object from the specified day.
                     * @static
                     * @param {number} day The day.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} A date extend condition object.
                     */
                    static fromDay(day:  number): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Creates the area condition from formula data.
                     * @static
                     * @param {string} formula The formula that specifies a range that contains data items.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} The area condition.
                     */
                    static fromFormula(formula:  string): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Creates a date extend condition object from the specified month.
                     * @static
                     * @param {number} month The month. The first month is 0.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} A date extend condition object.
                     */
                    static fromMonth(month:  number): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Creates a date extend condition object from the specified quarter.
                     * @static
                     * @param {GC.Spread.Sheets.ConditionalFormatting.QuarterType} quarter The quarter.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} A date extend condition object.
                     */
                    static fromQuarter(quarter:  GC.Spread.Sheets.ConditionalFormatting.QuarterType): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Creates the area condition from source data.
                     * @static
                     * @param {string} expected The expected source that separates each data item with a comma (",").
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} The area condition.
                     */
                    static fromSource(expected:  string): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Creates a date extend condition object from the specified week.
                     * @static
                     * @param {number} week The week.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} A date extend condition object.
                     */
                    static fromWeek(week:  number): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Creates a date extend condition object from the specified year.
                     * @static
                     * @param {number} year The year.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} A date extend condition object.
                     */
                    static fromYear(year:  number): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Gets the expected value.
                     * @constructor
                     * @param {Object} evaluator The evaluator that can evaluate an expression or a function.
                     * @param {number} baseRow The base row index for evaluation.
                     * @param {number} baseColumn The base column index for evaluation.
                     * @returns {Object} The expected value.
                     */
                    getExpected(evaluator:  Object,  baseRow:  number,  baseColumn:  number): Object;
                    /**
                     * Returns the list of valid data items.
                     * @param {Object} evaluator The evaluator that can evaluate an expression or a function.
                     * @param {number} baseRow The base row index for evaluation.
                     * @param {number} baseColumn The base column index for evaluation.
                     * @returns {Array} The list of valid data items.
                     */
                    getValidList(evaluator:  Object,  baseRow:  number,  baseColumn:  number): any[];
                    /**
                     * Gets or sets whether to ignore the blank cell.
                     * @param {boolean} value Whether to ignore the blank cell.
                     * @returns {boolean | GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set, returns whether to ignore the blank cell; otherwise, returns the condition.
                     * @example
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.textCondition);
                     * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.TextCompareType.contains);
                     * nCondition.expected("te?t");
                     * nCondition.ignoreBlank(true);
                     * nCondition.ignoreCase(true);
                     * nCondition.useWildCards(true);
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, "testing");
                     */
                    ignoreBlank(value?:  boolean): any;
                    /**
                     * Gets or sets whether to ignore case when performing the comparison.
                     * @param {boolean} value Whether to ignore case when performing the comparison.
                     * @returns {boolean | GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set, returns whether to ignore case when performing the comparison; otherwise, returns the condition.
                     * @example
                     * //This example creates a text condition.
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.textCondition);
                     * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.TextCompareType.contains);
                     * nCondition.expected("te?t");
                     * nCondition.ignoreBlank(true);
                     * nCondition.ignoreCase(true);
                     * nCondition.useWildCards(true);
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, "testing");
                     */
                    ignoreCase(value?:  boolean): any;
                    /**
                     * Gets or sets the first condition.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.Condition} value The first condition.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set, returns the first condition; otherwise, returns the relation condition.
                     * @example
                     * //This example validates a date.
                     * var condition1 = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.dateCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.DateCompareType.afterEqualsTo, expected: new Date(2012, 11, 31)});
                     * var condition2 = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.dateCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.DateCompareType.beforeEqualsTo, expected: new Date(2013, 11, 31)});
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.relationCondition);
                     * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.LogicalOperators.and);
                     * nCondition.item1(condition1);
                     * nCondition.item2(condition2);
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, new Date(2012, 11, 25));
                     */
                    item1(value?:  GC.Spread.Sheets.ConditionalFormatting.Condition): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Gets or sets the second condition.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.Condition} value The second condition.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set, returns the second condition; otherwise, returns the relation condition.
                     * @example
                     * //This example validates a date.
                     * var condition1 = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.dateCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.DateCompareType.afterEqualsTo, expected: new Date(2012, 11, 31)});
                     * var condition2 = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.dateCondition, {compareType: GC.Spread.Sheets.ConditionalFormatting.DateCompareType.beforeEqualsTo, expected: new Date(2013, 11, 31)});
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.relationCondition);
                     * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.LogicalOperators.and);
                     * nCondition.item1(condition1);
                     * nCondition.item2(condition2);
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, new Date(2012, 11, 25));
                     */
                    item2(value?:  GC.Spread.Sheets.ConditionalFormatting.Condition): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Gets or sets the condition ranges.
                     * @param {Array.<GC.Spread.Sheets.Range>} value The condition ranges.
                     * @returns {Array.<GC.Spread.Sheets.Range> | GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set, returns the condition ranges; otherwise, returns the condition.
                     * @example
                     * //This example creates a unique condition.
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.uniqueCondition);
                     * nCondition.expected(true);
                     * nCondition.ranges([new GC.Spread.Sheets.Range(0, 0, 5, 1)]);
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, 5);
                     */
                    ranges(value?:  GC.Spread.Sheets.Range[]): any;
                    /**
                     * Resets this instance.
                     * @example
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.numberCondition);
                     * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.greaterThan);
                     * nCondition.expected(5);
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, 5);
                     * //Create a button
                     * $("#button1").click(function () {
                     * activeSheet.suspendPaint();
                     * nCondition.reset();
                     * activeSheet.resumePaint();
                     *     });
                     */
                    reset(): void;
                    /**
                     * Gets or sets whether to treat the null value in a cell as zero.
                     * @param {boolean} value Whether to treat the null value in a cell as zero.
                     * @returns {boolean | GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set, returns whether to treat the null value in a cell as zero; otherwise, returns the condition.
                     * @example
                     * //This example sets the treatNullValueAsZero method.
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.cellValueCondition);
                     * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.GeneralComparisonOperators.equalsTo);
                     * nCondition.expected(0);
                     * //When the option is false, the validation fails and the red alert is displayed.
                     * //When the option is true, the blank cell is treated as zero and the validation is successful.
                     * nCondition.treatNullValueAsZero(false);
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * validator.ignoreBlank(false);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, null);
                     */
                    treatNullValueAsZero(value?:  boolean): any;
                    /**
                     * Gets or sets whether to compare strings using wildcards.
                     * @param {boolean} value Whether to compare strings using wildcards.
                     * @returns {boolean | GC.Spread.Sheets.ConditionalFormatting.Condition} If no value is set, returns whether to compare strings using wildcards; otherwise, returns the condition.
                     * @example
                     * //This example allows wildcards.
                     * var nCondition = new GC.Spread.Sheets.ConditionalFormatting.Condition(GC.Spread.Sheets.ConditionalFormatting.ConditionType.textCondition);
                     * nCondition.compareType(GC.Spread.Sheets.ConditionalFormatting.TextCompareType.contains);
                     * nCondition.expected("te?t");
                     * nCondition.ignoreBlank(true);
                     * nCondition.ignoreCase(true);
                     * nCondition.useWildCards(true);
                     * var validator = new GC.Spread.Sheets.DataValidation.DefaultDataValidator(nCondition);
                     * validator.type(GC.Spread.Sheets.DataValidation.CriteriaType.custom);
                     * activeSheet.getCell(0, 0, GC.Spread.Sheets.SheetArea.viewport).validator(validator);
                     * spread.options.highlightInvalidData = true;
                     * activeSheet.setValue(0, 0, "testing");
                     */
                    useWildCards(value?:  boolean): any;
                }

                export class ConditionalFormats{
                    /**
                     * Represents a format condition class.
                     * @class
                     * @param {GC.Spread.Sheets.Worksheet} worksheet The sheet.
                     */
                    constructor(worksheet:  GC.Spread.Sheets.Worksheet);
                    /**
                     * Adds the two scale rule to the rule collection.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ScaleValueType} minType The minimum scale type.
                     * @param {number} minValue The minimum scale value.
                     * @param {string} minColor The minimum scale color.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ScaleValueType} maxType The maximum scale type.
                     * @param {number} maxValue The maximum scale value.
                     * @param {string} maxColor The maximum scale color.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The two scale rule added to the rule collection.
                     * @example
                     * //This example uses the add2ScaleRule method.
                     * activeSheet.conditionalFormats.add2ScaleRule(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number,10,"Red",GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number,100,"Yellow", [new GC.Spread.Sheets.Range(0,0,10,3)]);
                     * activeSheet.setValue(0,0, 1,3);
                     * activeSheet.setValue(1,0, 50,3);
                     * activeSheet.setValue(2,0, 100,3);
                     */
                    add2ScaleRule(minType:  GC.Spread.Sheets.ConditionalFormatting.ScaleValueType,  minValue:  number,  minColor:  string,  maxType:  GC.Spread.Sheets.ConditionalFormatting.ScaleValueType,  maxValue:  number,  maxColor:  string,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds the three scale rule to the rule collection.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ScaleValueType} minType The minimum scale type.
                     * @param {number} minValue The minimum scale value.
                     * @param {string} minColor The minimum scale color.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ScaleValueType} midType The midpoint scale type.
                     * @param {number} midValue The midpoint scale value.
                     * @param {string} midColor The midpoint scale color.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ScaleValueType} maxType The maximum scale type.
                     * @param {number} maxValue The maximum scale value.
                     * @param {string} maxColor The maximum scale color.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The three scale rule added to the rule collection.
                     * @example
                     * //This example uses the add3ScaleRule method.
                     * activeSheet.setValue(0,0, 1,3);
                     * activeSheet.setValue(1,0, 50,3);
                     * activeSheet.setValue(2,0, 100,3);
                     * activeSheet.conditionalFormats.add3ScaleRule(1, 10, "red", 0, 50, "blue",2, 100, "yellow", [new GC.Spread.Sheets.Range(0, 0, 10, 3)]);
                     */
                    add3ScaleRule(minType:  GC.Spread.Sheets.ConditionalFormatting.ScaleValueType,  minValue:  number,  minColor:  string,  midType:  GC.Spread.Sheets.ConditionalFormatting.ScaleValueType,  midValue:  number,  midColor:  string,  maxType:  GC.Spread.Sheets.ConditionalFormatting.ScaleValueType,  maxValue:  number,  maxColor:  string,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds an average rule to the rule collection.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.AverageConditionType} type The average condition type.
                     * @param {GC.Spread.Sheets.Style} style The style that is applied to the cell when the condition is met.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The average rule added to the rule collection.
                     * @example
                     * //This example uses the addAverageRule method.
                     * activeSheet.setValue(0,0, 1,3);
                     * activeSheet.setValue(1,0, 50,3);
                     * activeSheet.setValue(2,0, 100,3);
                     * activeSheet.setValue(3,0, 2,3);
                     * activeSheet.setValue(4,0, 60,3);
                     * activeSheet.setValue(5,0, 90,3);
                     * activeSheet.setValue(6,0, 3,3);
                     * activeSheet.setValue(7,0, 40,3);
                     * activeSheet.setValue(8,0, 70,3);
                     * activeSheet.setValue(9,0, 5,3);
                     * activeSheet.setValue(10,0, 35,3);
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * style.borderLeft =new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                     * style.borderTop = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                     * style.borderRight = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                     * style.borderBottom = new GC.Spread.Sheets.LineBorder("blue",GC.Spread.Sheets.LineStyle.medium);
                     * activeSheet.conditionalFormats.addAverageRule(GC.Spread.Sheets.ConditionalFormatting.AverageConditionType.above,style,[new GC.Spread.Sheets.Range(0, 0, 10, 3)]);
                     */
                    addAverageRule(type:  GC.Spread.Sheets.ConditionalFormatting.AverageConditionType,  style:  GC.Spread.Sheets.Style,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds the cell value rule to the rule collection.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators} comparisionOperator The comparison operator.
                     * @param {Object} value1 The first value.
                     * @param {Object} value2 The second value.
                     * @param {GC.Spread.Sheets.Style} style The style that is applied to the cell when the condition is met.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The cell value rule added to the rule collection.
                     * @example
                     * //This example uses the addCellValueRule method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var ranges=[new GC.Spread.Sheets.Range(0,0,5,1)];
                     * activeSheet.conditionalFormats.addCellValueRule(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between, 2, 100, style, ranges);
                     * activeSheet.setValue(0,0,1,3);
                     * activeSheet.setValue(1,0,45,3);
                     */
                    addCellValueRule(comparisionOperator:  GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators,  value1:  Object,  value2:  Object,  style:  GC.Spread.Sheets.Style,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds a data bar rule to the rule collection.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ScaleValueType} minType The minimum scale type.
                     * @param {number} minValue The minimum scale value.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ScaleValueType} maxType The maximum scale type.
                     * @param {number} maxValue The maximum scale value.
                     * @param {string} color The color data bar to show on the view.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The data bar rule added to the rule collection.
                     * @example
                     * //This example uses the addDataBarRule method.
                     * activeSheet.setValue(0,0,1,3);
                     * activeSheet.setValue(1,0,15,3);
                     * activeSheet.setValue(2,0,25,3);
                     * activeSheet.setValue(3,0,-1,3);
                     * activeSheet.conditionalFormats.addDataBarRule(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number, -1, GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number, 40, "orange", [new GC.Spread.Sheets.Range(0,0,4,1)]);
                     */
                    addDataBarRule(minType:  GC.Spread.Sheets.ConditionalFormatting.ScaleValueType,  minValue:  number,  maxType:  GC.Spread.Sheets.ConditionalFormatting.ScaleValueType,  maxValue:  number,  color:  string,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds the date occurring rule to the rule collection.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.DateOccurringType} type The data occurring type.
                     * @param {GC.Spread.Sheets.Style} style The style that is applied to the cell when the condition is met.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The date occurring rule added to the rule collection.
                     * @example
                     * //This example uses the addDateOccurringRule method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var d = new Date();
                     * activeSheet.setValue(0, 0, d);
                     * activeSheet.setValue(1, 0, new Date(d.setDate(d.getDate()+1)));
                     * activeSheet.setValue(2, 0, new Date(d.setDate(d.getDate()+5)));
                     * activeSheet.setValue(3, 0,new Date(d.setDate(d.getDate()+6)));
                     * activeSheet.setValue(4, 0,new Date(d.setDate(d.getDate()+7)));
                     * activeSheet.setValue(5, 0, new Date(d.setDate(d.getDate()+8)));
                     * activeSheet.conditionalFormats.addDateOccurringRule(GC.Spread.Sheets.ConditionalFormatting.DateOccurringType.nextWeek, style, [new GC.Spread.Sheets.Range(0,0,10,1)]);
                     */
                    addDateOccurringRule(type: GC.Spread.Sheets.ConditionalFormatting.DateOccurringType,  style:  GC.Spread.Sheets.Style,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds a duplicate rule to the rule collection.
                     * @param {GC.Spread.Sheets.Style} style The style that is applied to the cell when the condition is met.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The duplicate rule added to the rule collection.
                     * @example
                     * //This example uses the addDuplicateRule method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "yellow";
                     * var ranges=[new GC.Spread.Sheets.Range(0,0,10,1)];
                     * activeSheet.conditionalFormats.addDuplicateRule(style, ranges);
                     * activeSheet.setValue(0, 0, 50);
                     * activeSheet.setValue(1, 0, 50);
                     * activeSheet.setValue(2, 0, 11);
                     * activeSheet.setValue(3, 0, 5);
                     */
                    addDuplicateRule(style:  GC.Spread.Sheets.Style,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds the formula rule to the rule collection.
                     * @param {string} formula The condition formula.
                     * @param {GC.Spread.Sheets.Style} style The style that is applied to the cell when the condition is met.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The formula rule added to the rule collection.
                     * @example
                     * //This example uses the addFormulaRule method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var ranges = [new GC.Spread.Sheets.Range(0, 0, 2, 1)];
                     * activeSheet.conditionalFormats.addFormulaRule("=A1=B1+C1", style, ranges);
                     * activeSheet.setValue(0, 0, 2,3);
                     * activeSheet.setValue(0, 1, 1,3);
                     * activeSheet.setValue(0, 2,1,3);
                     * activeSheet.setValue(1, 0, 1,3);
                     */
                    addFormulaRule(formula:  string,  style:  GC.Spread.Sheets.Style,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds an icon set rule to the rule collection.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.IconSetType} iconSetType The type of icon set.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The icon set rule added to the rule collection.
                     * @example
                     * //This example uses the addIconSetRule method.
                     * activeSheet.setValue(0,0,1,3);
                     * activeSheet.setValue(1,0,15,3);
                     * activeSheet.setValue(2,0,25,3);
                     * activeSheet.setValue(3,0,-1,3);
                     * activeSheet.conditionalFormats.addIconSetRule(GC.Spread.Sheets.ConditionalFormatting.IconSetType.fourTrafficLights, [new GC.Spread.Sheets.Range(0,0,4,1)]);
                     */
                    addIconSetRule(iconSetTye:  GC.Spread.Sheets.ConditionalFormatting.IconSetType,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds the rule.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} rule The rule to add.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase}
                     * @example
                     * //This example uses the addRule method.
                     * var scale = new GC.Spread.Sheets.ConditionalFormatting.ScaleRule();
                     * scale.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.threeScaleRule);
                     * scale.midColor("red");
                     * scale.midType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                     * scale.midValue(50);
                     * scale.maxColor("blue");
                     * scale.maxType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                     * scale.maxValue(100);
                     * scale.minColor("yellow");
                     * scale.minType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                     * scale.minValue(10);
                     * scale.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 3)]);
                     * activeSheet.conditionalFormats.addRule(scale);
                     * activeSheet.setValue(0,0, 1,3);
                     * activeSheet.setValue(1,0, 50,3);
                     * activeSheet.setValue(2,0, 100,3);
                     */
                    addRule(rule:  GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds the text rule to the rule collection.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.TextComparisonOperators} comparisonOperator The comparison operator.
                     * @param {string} text The text for comparison.
                     * @param {GC.Spread.Sheets.Style} style The style that is applied to the cell when the condition is met.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied to items whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The text rule added to the rule collection.
                     * @example
                     * //This example uses the addSpecificTextRule method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var ranges=[new GC.Spread.Sheets.Range(0,0,10,1)];
                     * activeSheet.conditionalFormats.addSpecificTextRule(GC.Spread.Sheets.ConditionalFormatting.TextComparisonOperators.contains, "test", style, ranges);
                     * activeSheet.setValue(0, 0, "testing");
                     * activeSheet.setValue(1, 0, "test");
                     * activeSheet.setValue(2, 0, "a");
                     * activeSheet.setValue(3, 0, "t");
                     */
                    addSpecificTextRule(comparisionOperator:  GC.Spread.Sheets.ConditionalFormatting.TextComparisonOperators,  text:  string,  style:  GC.Spread.Sheets.Style,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds the top 10 rule or bottom 10 rule to the collection based on the Top10CondtionType object.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.Top10ConditionType} type The top 10 condition.
                     * @param {number} rank The number of top or bottom items to apply the style to.
                     * @param {GC.Spread.Sheets.Style} style The style that is applied to the cell when the condition is met.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The top 10 rule added to the rule collection.
                     * @example
                     * //This example uses the addTop10Rule method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var ranges=[new GC.Spread.Sheets.Range(0,0,10,1)];
                     * activeSheet.conditionalFormats.addTop10Rule(GC.Spread.Sheets.ConditionalFormatting.Top10ConditionType.top, 2, style, ranges);
                     * activeSheet.setValue(0, 0, 1);
                     * activeSheet.setValue(1, 0, 50);
                     * activeSheet.setValue(2, 0, 11);
                     * activeSheet.setValue(3, 0, 5);
                     */
                    addTop10Rule(type:  GC.Spread.Sheets.ConditionalFormatting.Top10ConditionType,  rank:  number,  style:  GC.Spread.Sheets.Style,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Adds a unique rule to the rule collection.
                     * @param {GC.Spread.Sheets.Style} style The style that is applied to the cell when the condition is met.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The cell ranges where the rule is applied whose item type is GC.Spread.Sheets.Range.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The unique rule added to the rule collection.
                     * @example
                     * //This example uses the addUniqueRule method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "green";
                     * activeSheet.setValue(0, 0, 50);
                     * activeSheet.setValue(1, 0, 50);
                     * activeSheet.setValue(2, 0, 11);
                     * activeSheet.setValue(3, 0, 5);
                     * activeSheet.conditionalFormats.addUniqueRule(style, [new GC.Spread.Sheets.Range(0,0,10,1)]);
                     */
                    addUniqueRule(style:  GC.Spread.Sheets.Style,  ranges:  GC.Spread.Sheets.Range[]): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Removes all rules.
                     * @example
                     * //This example uses the clearRule method.
                     * activeSheet.setValue(0,0, 1,3);
                     * activeSheet.setValue(1,0, 50,3);
                     * activeSheet.setValue(2,0, 100,3);
                     * activeSheet.conditionalFormats.add2ScaleRule(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number, 10, "red", GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number, 100, "yellow", [new GC.Spread.Sheets.Range(0, 0, 10, 3)]);
                     * // Remove comment in front of method to test
                     * //activeSheet.conditionalFormats.clearRule();
                     */
                    clearRule(): void;
                    /**
                     * Determines whether the specified cell contains a specified rule.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} rule The rule for which to check.
                     * @param {number} row The row index.
                     * @param {number} column The column index.
                     * @returns {boolean} <c>true</c> if the specified cell contains a specified rule; otherwise, <c>false</c>.
                     * @example
                     * //This example checks to see if a cell has a specified rule.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                     * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                     * rule.ranges([new GC.Spread.Sheets.Range(0,0,5,1)]);
                     * rule.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between);
                     * rule.style(style);
                     * rule.value1(2);
                     * rule.value2(100);
                     * activeSheet.conditionalFormats.addRule(rule);
                     * activeSheet.setValue(0,0,1,3);
                     * activeSheet.setValue(1,0,45,3);
                     * var ruletest = activeSheet.conditionalFormats.containsRule(rule, 0, 0);
                     * alert(ruletest);
                     */
                    containsRule(rule:  GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase,  row:  number,  column:  number): boolean;
                    /**
                     * Gets the number of rule objects in the collection.
                     * @returns {number} The number of rule objects in the collection.
                     * @example
                     * //This example counts the rules.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                     * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                     * rule.ranges([new GC.Spread.Sheets.Range(0,0,5,1)]);
                     * rule.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between);
                     * rule.style(style);
                     * rule.value1(2);
                     * rule.value2(100);
                     * activeSheet.conditionalFormats.addRule(rule);
                     * activeSheet.setValue(0,0,1,3);
                     * activeSheet.setValue(1,0,45,3);
                     * var ruletest = activeSheet.conditionalFormats.count();
                     * alert(ruletest);
                     */
                    count(): number;
                    /**
                     * Gets the rule using the index.
                     * @param {number} index The index from which to get the rule.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} The rule from the index.
                     * @example
                     * //This example uses the getRule method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                     * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                     * rule.ranges([new GC.Spread.Sheets.Range(0,0,5,1)]);
                     * rule.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between);
                     * rule.style(style);
                     * rule.value1(2);
                     * rule.value2(100);
                     * activeSheet.conditionalFormats.addRule(rule);
                     * activeSheet.setValue(0,0,1,3);
                     * activeSheet.setValue(1,0,45,3);
                     * var ruletest = activeSheet.conditionalFormats.getRule(0);
                     * alert(ruletest.value1());
                     */
                    getRule(index:  number): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase;
                    /**
                     * Gets the conditional rules from the cell at the specified row and column.
                     * @param {number} row The row index.
                     * @param {number} column The column index.
                     * @returns {Array.<GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase>} The conditional rules.
                     * @example
                     * //This example uses the getRules method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                     * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                     * rule.ranges([new GC.Spread.Sheets.Range(0,0,5,1)]);
                     * rule.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between);
                     * rule.style(style);
                     * rule.value1(2);
                     * rule.value2(100);
                     * activeSheet.conditionalFormats.addRule(rule);
                     * activeSheet.setValue(0,0,1,3);
                     * activeSheet.setValue(1,0,45,3);
                     * var ruletest = activeSheet.conditionalFormats.getRules();
                     * alert(ruletest[0].style().backColor);
                     */
                    getRules(row:  number,  column:  number): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase[];
                    /**
                     * Removes a rule object from the ConditionalFormats object.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} rule The rule object to remove from the ConditionalFormats object.
                     * @example
                     * //This example uses the removeRule method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                     * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                     * rule.ranges([new GC.Spread.Sheets.Range(0,0,5,1)]);
                     * rule.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between);
                     * rule.style(style);
                     * rule.value1(2);
                     * rule.value2(100);
                     * activeSheet.conditionalFormats.addRule(rule);
                     * activeSheet.setValue(0,0,1,3);
                     * activeSheet.setValue(1,0,45,3);
                     * activeSheet.conditionalFormats.removeRule(rule);
                     */
                    removeRule(rule:  GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase): void;
                    /**
                     * Removes the rules from a specified cell range.
                     * @param {number} row The row index of the first cell in the range.
                     * @param {number} column The column index of the first cell in the range.
                     * @param {number} rowCount The number of rows in the range.
                     * @param {number} columnCount The number of columns in the range.
                     * @example
                     * //This example uses the removeRuleByRange method.
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * var rule = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                     * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                     * rule.ranges([new GC.Spread.Sheets.Range(0,0,5,1)]);
                     * rule.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between);
                     * rule.style(style);
                     * rule.value1(2);
                     * rule.value2(100);
                     * activeSheet.conditionalFormats.addRule(rule);
                     * activeSheet.setValue(0,0,1,3);
                     * activeSheet.setValue(1,0,45,3);
                     * activeSheet.conditionalFormats.removeRuleByRange(0, 0, 5, 1);
                     */
                    removeRuleByRange(row:  number,  column:  number,  rowCount:  number,  columnCount:  number): void;
                }

                export class ConditionRuleBase{
                    /**
                     * Represents a formatting base rule class as the specified style.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.RuleType} ruleType
                     * @param {GC.Spread.Sheets.Style} style The style for the rule.
                     * @param {Array.<GC.Spread.Sheets.Range>} ranges The range array.
                     * @class
                     */
                    constructor(ruleType:  GC.Spread.Sheets.ConditionalFormatting.RuleType,  style:  GC.Spread.Sheets.Style,  ranges:  GC.Spread.Sheets.Range[]);
                    /**
                     * Gets or sets the base condition of the rule.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.Condition} value The base condition of the rule.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition | GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} If no value is set, returns the base condition of the rule; otherwise, returns the condition rule.
                     */
                    condition(value?:  GC.Spread.Sheets.ConditionalFormatting.Condition): any;
                    /**
                     * Determines whether the range of cells contains the cell at the specified row and column.
                     * @param {number} row The row index.
                     * @param {number} column The column index.
                     * @returns {boolean} <c>true</c> if the range of cells contains the cell at the specified row and column; otherwise, <c>false</c>.
                     */
                    contains(row:  number,  column:  number): boolean;
                    /**
                     * Creates condition for the rule.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.Condition} The condition.
                     */
                    createCondition(): GC.Spread.Sheets.ConditionalFormatting.Condition;
                    /**
                     * Returns the cell style of the rule if the cell satisfies the condition.
                     * @param {Object} evaluator The object that can evaluate a condition.
                     * @param {number} baseRow The row index.
                     * @param {number} baseColumn The column index.
                     * @param {Object} actual The actual value.
                     * @returns {GC.Spread.Sheets.Style} The cell style of the rule.
                     */
                    evaluate(evaluator:  Object,  baseRow:  number,  baseColumn:  number,  actual:  Object): GC.Spread.Sheets.Style;
                    /**
                     * Gets the style of the base rule.
                     * @returns {GC.Spread.Sheets.Style}
                     */
                    getExpected(): GC.Spread.Sheets.Style;
                    /**
                     * Specifies whether the range for this rule intersects another range.
                     * @param {number} row The row index.
                     * @param {number} column The column index.
                     * @param {number} rowCount The number of rows.
                     * @param {number} columnCount The number of columns.
                     * @returns {boolean} <c>true</c> if the range for this rule intersects another range; otherwise, <c>false</c>.
                     */
                    intersects(row:  number,  column:  number,  rowCount:  number,  columnCount:  number): boolean;
                    /**
                     * Specifies whether this rule is a scale rule.
                     * @returns {boolean} <c>true</c> if this rule is a scale rule; otherwise, <c>false</c>.
                     */
                    isScaleRule(): boolean;
                    /**
                     * Gets or sets the priority of the rule.
                     * @param {number} value The priority of the rule.
                     * @returns {number | GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} If no value is set, returns the priority of the rule; otherwise, returns the condition rule.
                     */
                    priority(value?:  number): any;
                    /**
                     * Gets or sets the condition rule ranges.
                     * @param {Array.<GC.Spread.Sheets.Range>} value The condition rule ranges.
                     * @returns {Array.<GC.Spread.Sheets.Range> | GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} If no value is set, returns the condition rule ranges; otherwise, returns the condition rule.
                     * @example
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "green";
                     * var ranges = [new GC.Spread.Sheets.Range(0, 0, 10, 1)];
                     * activeSheet.conditionalFormats.addUniqueRule(style, ranges);
                     * activeSheet.setValue(0, 0, 50);
                     * activeSheet.setValue(1, 0, 50);
                     * activeSheet.setValue(2, 0, 11);
                     * activeSheet.setValue(3, 0, 5);
                     */
                    ranges(value?:  GC.Spread.Sheets.Range[]): any;
                    /**
                     * Resets the rule.
                     */
                    reset(): void;
                    /**
                     * Gets or sets the condition rule type.
                     * @param {GC.Spread.Sheets.ConditionalFormatting.RuleType} value The condition rule type.
                     * @returns {GC.Spread.Sheets.ConditionalFormatting.RuleType | GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} If no value is set, returns the condition rule type; otherwise, returns the condition rule.
                     * @example
                     * //This example uses the ruleType method.
                     * activeSheet.setArray(0,0,[1,2,3,4,5,6,7,8,9,10]);
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * style.foreColor = "black";
                     * var cell = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                     * cell.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                     * cell.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.greaterThan);
                     * cell.value1(5);
                     * cell.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 1)]);
                     * cell.style(style);
                     * activeSheet.conditionalFormats.addRule(cell);
                     * var style1 = new GC.Spread.Sheets.Style();
                     * style1.foreColor = "red";
                     * var top = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                     * top.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.top10Rule);
                     * top.type(GC.Spread.Sheets.ConditionalFormatting.Top10ConditionType.top);
                     * top.rank(3);
                     * top.style(style1);
                     * top.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 1)]);
                     * top.stopIfTrue(true);
                     * activeSheet.conditionalFormats.addRule(top);
                     */
                    ruleType(value?:  GC.Spread.Sheets.ConditionalFormatting.RuleType): any;
                    /**
                     * Gets or sets whether rules with lower priority are applied before this rule.
                     * @param {boolean} value Whether rules with lower priority are applied before this rule.
                     * @returns {boolean | GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} If no value is set, returns whether the rules with lower priority are not applied before this rule; otherwise, returns the condition rule.
                     * @example
                     * //This example applies multiple rules.
                     * activeSheet.setArray(0,0,[1,2,3,4,5,6,7,8,9,10]);
                     * var style = new GC.Spread.Sheets.Style();
                     * style.backColor = "red";
                     * style.foreColor = "black";
                     * var cell = new GC.Spread.Sheets.ConditionalFormatting.NormalConditionRule();
                     * cell.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.cellValueRule);
                     * cell.operator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.greaterThan);
                     * cell.value1(5);
                     * cell.ranges([new GC.Spread.Sheets.Range(0, 0, 10, 1)]);
                     * cell.style(style);
                     * activeSheet.conditionalFormats.addRule(cell);
                     * var style1 = new GC.Spread.Sheets.Style();
                     * style1.foreCo