/*
 * Decompiled with CFR 0.152.
 */
package edu.cmu.sphinx.linguist.dictionary;

import edu.cmu.sphinx.linguist.acoustic.Context;
import edu.cmu.sphinx.linguist.acoustic.Unit;
import edu.cmu.sphinx.linguist.acoustic.UnitManager;
import edu.cmu.sphinx.linguist.dictionary.Dictionary;
import edu.cmu.sphinx.linguist.dictionary.Pronunciation;
import edu.cmu.sphinx.linguist.dictionary.Word;
import edu.cmu.sphinx.linguist.g2p.G2PConverter;
import edu.cmu.sphinx.linguist.g2p.Path;
import edu.cmu.sphinx.util.Timer;
import edu.cmu.sphinx.util.TimerPool;
import edu.cmu.sphinx.util.props.ConfigurationManagerUtils;
import edu.cmu.sphinx.util.props.PropertyException;
import edu.cmu.sphinx.util.props.PropertySheet;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.logging.Logger;

public class TextDictionary
implements Dictionary {
    protected Logger logger;
    protected URL wordDictionaryFile;
    protected URL fillerDictionaryFile;
    protected List<URL> addendaUrlList;
    private String wordReplacement;
    protected URL g2pModelFile;
    protected int g2pMaxPron = 0;
    protected UnitManager unitManager;
    protected Map<String, String> dictionary;
    protected Map<String, Word> wordDictionary;
    protected G2PConverter g2pDecoder;
    protected static final String FILLER_TAG = "-F-";
    protected Set<String> fillerWords;
    protected boolean allocated;

    public TextDictionary(String wordDictionaryFile, String fillerDictionaryFile, List<URL> addendaUrlList, boolean addSilEndingPronunciation, String wordReplacement, UnitManager unitManager) throws MalformedURLException, ClassNotFoundException {
        this(ConfigurationManagerUtils.resourceToURL(wordDictionaryFile), ConfigurationManagerUtils.resourceToURL(fillerDictionaryFile), addendaUrlList, wordReplacement, unitManager);
    }

    public TextDictionary(URL wordDictionaryFile, URL fillerDictionaryFile, List<URL> addendaUrlList, String wordReplacement, UnitManager unitManager) {
        this.logger = Logger.getLogger(this.getClass().getName());
        this.wordDictionaryFile = wordDictionaryFile;
        this.fillerDictionaryFile = fillerDictionaryFile;
        this.addendaUrlList = addendaUrlList;
        this.wordReplacement = wordReplacement;
        this.unitManager = unitManager;
    }

    public TextDictionary(URL wordDictionaryFile, URL fillerDictionaryFile, List<URL> addendaUrlList, boolean addSilEndingPronunciation, String wordReplacement, UnitManager unitManager, URL g2pModelFile, int g2pMaxPron) {
        this(wordDictionaryFile, fillerDictionaryFile, addendaUrlList, wordReplacement, unitManager);
        this.g2pModelFile = g2pModelFile;
        this.g2pMaxPron = g2pMaxPron;
    }

    public TextDictionary() {
    }

    @Override
    public void newProperties(PropertySheet ps) throws PropertyException {
        this.logger = ps.getLogger();
        this.wordDictionaryFile = ConfigurationManagerUtils.getResource("dictionaryPath", ps);
        this.fillerDictionaryFile = ConfigurationManagerUtils.getResource("fillerPath", ps);
        this.addendaUrlList = ps.getResourceList("addenda");
        this.wordReplacement = ps.getString("wordReplacement");
        this.unitManager = (UnitManager)ps.getComponent("unitManager");
        this.g2pModelFile = ConfigurationManagerUtils.getResource("g2pModelPath", ps);
        this.g2pMaxPron = ps.getInt("g2pMaxPron");
    }

    public URL getWordDictionaryFile() {
        return this.wordDictionaryFile;
    }

    public URL getFillerDictionaryFile() {
        return this.fillerDictionaryFile;
    }

    @Override
    public void allocate() throws IOException {
        if (!this.allocated) {
            this.dictionary = new HashMap<String, String>();
            this.wordDictionary = new HashMap<String, Word>();
            Timer loadTimer = TimerPool.getTimer(this, "Load Dictionary");
            this.fillerWords = new HashSet<String>();
            loadTimer.start();
            this.logger.info("Loading dictionary from: " + this.wordDictionaryFile);
            this.loadDictionary(this.wordDictionaryFile.openStream(), false);
            this.loadCustomDictionaries(this.addendaUrlList);
            this.logger.info("Loading filler dictionary from: " + this.fillerDictionaryFile);
            this.loadDictionary(this.fillerDictionaryFile.openStream(), true);
            if (this.g2pModelFile != null && !this.g2pModelFile.getPath().equals("")) {
                this.g2pDecoder = new G2PConverter(this.g2pModelFile);
            }
            loadTimer.stop();
        }
    }

    @Override
    public void deallocate() {
        if (this.allocated) {
            this.dictionary = null;
            this.g2pDecoder = null;
            this.allocated = false;
        }
    }

    protected void loadDictionary(InputStream inputStream, boolean isFillerDict) throws IOException {
        String line;
        InputStreamReader isr = new InputStreamReader(inputStream);
        BufferedReader br = new BufferedReader(isr);
        while ((line = br.readLine()) != null) {
            if ((line = line.trim()).isEmpty()) continue;
            int spaceIndex = this.getSpaceIndex(line);
            if (spaceIndex < 0) {
                throw new Error("Error loading word: " + line);
            }
            String word = line.substring(0, spaceIndex);
            if (this.dictionary.containsKey(word)) {
                String wordWithIdx;
                int index = 2;
                while (this.dictionary.containsKey(wordWithIdx = String.format("%s(%d)", word, index++))) {
                }
                word = wordWithIdx;
            }
            if (isFillerDict) {
                this.dictionary.put(word, FILLER_TAG + line);
                this.fillerWords.add(word);
                continue;
            }
            this.dictionary.put(word, line);
        }
        br.close();
        isr.close();
        inputStream.close();
    }

    private int getSpaceIndex(String line) {
        for (int i = 0; i < line.length(); ++i) {
            if (line.charAt(i) != ' ' && line.charAt(i) != '\t') continue;
            return i;
        }
        return -1;
    }

    protected Unit getCIUnit(String name, boolean isFiller) {
        return this.unitManager.getUnit(name, isFiller, Context.EMPTY_CONTEXT);
    }

    @Override
    public Word getSentenceStartWord() {
        return this.getWord("<s>");
    }

    @Override
    public Word getSentenceEndWord() {
        return this.getWord("</s>");
    }

    @Override
    public Word getSilenceWord() {
        return this.getWord("<sil>");
    }

    @Override
    public Word getWord(String text) {
        Word wordObject = this.wordDictionary.get(text);
        if (wordObject != null) {
            return wordObject;
        }
        String word = this.dictionary.get(text);
        if (word == null) {
            this.logger.info("The dictionary is missing a phonetic transcription for the word '" + text + "'");
            if (this.wordReplacement != null) {
                wordObject = this.getWord(this.wordReplacement);
            } else if (this.g2pModelFile != null && !this.g2pModelFile.getPath().equals("")) {
                this.logger.info("Generating phonetic transcription(s) for the word '" + text + "' using g2p model");
                wordObject = this.extractPronunciation(text);
                this.wordDictionary.put(text, wordObject);
            }
        } else {
            wordObject = this.processEntry(text);
        }
        return wordObject;
    }

    private Word extractPronunciation(String text) {
        ArrayList<Path> paths = this.g2pDecoder.phoneticize(text, this.g2pMaxPron);
        LinkedList<Pronunciation> pronunciations = new LinkedList<Pronunciation>();
        for (Path p : paths) {
            int unitCount = p.getPath().size();
            ArrayList<Unit> units = new ArrayList<Unit>(unitCount);
            for (String token : p.getPath()) {
                units.add(this.getCIUnit(token, false));
            }
            if (units.size() == 0) {
                units.add(UnitManager.SILENCE);
            }
            pronunciations.add(new Pronunciation(units));
        }
        Pronunciation[] pronunciationsArray = pronunciations.toArray(new Pronunciation[pronunciations.size()]);
        Word wordObject = this.createWord(text, pronunciationsArray, false);
        for (Pronunciation pronunciation : pronunciationsArray) {
            pronunciation.setWord(wordObject);
        }
        return wordObject;
    }

    private Word createWord(String text, Pronunciation[] pronunciation, boolean isFiller) {
        Word word = new Word(text, pronunciation, isFiller);
        this.dictionary.put(text, word.toString());
        return word;
    }

    private Word processEntry(String word) {
        String line;
        LinkedList<Pronunciation> pronunciations = new LinkedList<Pronunciation>();
        int count = 0;
        boolean isFiller = false;
        do {
            String lookupWord = word;
            if (++count > 1) {
                lookupWord = lookupWord + '(' + count + ')';
            }
            if ((line = this.dictionary.get(lookupWord)) == null) continue;
            StringTokenizer st = new StringTokenizer(line);
            String tag = st.nextToken();
            isFiller = tag.startsWith(FILLER_TAG);
            int unitCount = st.countTokens();
            ArrayList<Unit> units = new ArrayList<Unit>(unitCount);
            for (int i = 0; i < unitCount; ++i) {
                String unitName = st.nextToken();
                units.add(this.getCIUnit(unitName, isFiller));
            }
            pronunciations.add(new Pronunciation(units));
        } while (line != null);
        Pronunciation[] pronunciationsArray = pronunciations.toArray(new Pronunciation[pronunciations.size()]);
        Word wordObject = this.createWord(word, pronunciationsArray, isFiller);
        for (Pronunciation pronunciation : pronunciationsArray) {
            pronunciation.setWord(wordObject);
        }
        this.wordDictionary.put(word, wordObject);
        return wordObject;
    }

    public String toString() {
        TreeMap<String, String> sorted = new TreeMap<String, String>(this.dictionary);
        StringBuilder result = new StringBuilder();
        for (Map.Entry entry : sorted.entrySet()) {
            result.append((String)entry.getKey());
            result.append("   ").append((String)entry.getValue()).append('\n');
        }
        return result.toString();
    }

    @Override
    public Word[] getFillerWords() {
        Word[] fillerWordArray = new Word[this.fillerWords.size()];
        int index = 0;
        for (String spelling : this.fillerWords) {
            fillerWordArray[index++] = this.getWord(spelling);
        }
        return fillerWordArray;
    }

    public void dump() {
        System.out.print(this.toString());
    }

    private void loadCustomDictionaries(List<URL> addenda) throws IOException {
        if (addenda != null) {
            for (URL addendumUrl : addenda) {
                this.logger.info("Loading addendum dictionary from: " + addendumUrl);
                this.loadDictionary(addendumUrl.openStream(), false);
            }
        }
    }
}

