var parser = require('../lib/keyParser');

var path = require('path'),
    assert = require('assert'),
    inspect = require('util').inspect;

var group = path.basename(__filename, '.js') + '/';

[
  { source: [
      '-----BEGIN RSA PRIVATE KEY-----',
      'MIIBywIBAAJhANXy0sLFQFgcY6sfyqlrBBNAcow7MWjQS0flUEj8HmrdTX+denZ5',
      'sArVEBWIoIcCWt0DWGz/mZDS2XPqyy9IbEnhwtZ3bpHmw11J1k61odAwy5sE3P4S',
      'kS/Svl7Wxjcr5wIDAQABAmEAx50pG2QiZpMGQvEEixE32iZ1OXU44kwFEO1Y/N+O',
      'GbKkzf5F9igPIewsd4rkC1MbcVPYTN6a7WlFh2dP1kC5wdPAdVjBf1p7Ii4t4HZt',
      'OokidduJzCmC+TulDCNLFmlBAjEA/6SUxcxRXJV7GHBmu+FIbnR6/drHaBvMkNKZ',
      '/tGIq1KfrHIWrhGl2vYqot6hRgoXAjEA1j9VAyuDK2NhQG/WMmf8WV+OMeJQ7vpn',
      '3KmtPc/CsWfPNYeQagfwlj0SLGwz8J6xAjEA4qIRkllUN/P9MOlU6nMKN4HkF3EB',
      'HvamIC3Uf+dadifFjqeGG8NhegCKQ+GCsAj5AjAecs3AL5Tqu6r8n2hPrZudkAkD',
      'aeiarhfIsoyedkJzOA+oyeVqTEqS74b/rTkMf2ECMBrts0oFmVJYo77riKQBYbR7',
      '/8feisYSPHSg7hwaoXSwU+jU4g39w7rgvO+WkGuzyQ==',
      '-----END RSA PRIVATE KEY-----'
    ].join('\n'),
    expected: {
      fulltype: undefined,
      type: 'rsa',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: new Buffer([
        48, 130, 1, 203, 2, 1, 0, 2, 97, 0, 213, 242, 210, 194, 197, 64, 88,
        28, 99, 171, 31, 202, 169, 107, 4, 19, 64, 114, 140, 59, 49, 104, 208,
        75, 71, 229, 80, 72, 252, 30, 106, 221, 77, 127, 157, 122, 118, 121, 176,
        10, 213, 16, 21, 136, 160, 135, 2, 90, 221, 3, 88, 108, 255, 153, 144,
        210, 217, 115, 234, 203, 47, 72, 108, 73, 225, 194, 214, 119, 110, 145, 230,
        195, 93, 73, 214, 78, 181, 161, 208, 48, 203, 155, 4, 220, 254, 18, 145,
        47, 210, 190, 94, 214, 198, 55, 43, 231, 2, 3, 1, 0, 1, 2, 97,
        0, 199, 157, 41, 27, 100, 34, 102, 147, 6, 66, 241, 4, 139, 17, 55,
        218, 38, 117, 57, 117, 56, 226, 76, 5, 16, 237, 88, 252, 223, 142, 25,
        178, 164, 205, 254, 69, 246, 40, 15, 33, 236, 44, 119, 138, 228, 11, 83,
        27, 113, 83, 216, 76, 222, 154, 237, 105, 69, 135, 103, 79, 214, 64, 185,
        193, 211, 192, 117, 88, 193, 127, 90, 123, 34, 46, 45, 224, 118, 109, 58,
        137, 34, 117, 219, 137, 204, 41, 130, 249, 59, 165, 12, 35, 75, 22, 105,
        65, 2, 49, 0, 255, 164, 148, 197, 204, 81, 92, 149, 123, 24, 112, 102,
        187, 225, 72, 110, 116, 122, 253, 218, 199, 104, 27, 204, 144, 210, 153, 254,
        209, 136, 171, 82, 159, 172, 114, 22, 174, 17, 165, 218, 246, 42, 162, 222,
        161, 70, 10, 23, 2, 49, 0, 214, 63, 85, 3, 43, 131, 43, 99, 97,
        64, 111, 214, 50, 103, 252, 89, 95, 142, 49, 226, 80, 238, 250, 103, 220,
        169, 173, 61, 207, 194, 177, 103, 207, 53, 135, 144, 106, 7, 240, 150, 61,
        18, 44, 108, 51, 240, 158, 177, 2, 49, 0, 226, 162, 17, 146, 89, 84,
        55, 243, 253, 48, 233, 84, 234, 115, 10, 55, 129, 228, 23, 113, 1, 30,
        246, 166, 32, 45, 212, 127, 231, 90, 118, 39, 197, 142, 167, 134, 27, 195,
        97, 122, 0, 138, 67, 225, 130, 176, 8, 249, 2, 48, 30, 114, 205, 192,
        47, 148, 234, 187, 170, 252, 159, 104, 79, 173, 155, 157, 144, 9, 3, 105,
        232, 154, 174, 23, 200, 178, 140, 158, 118, 66, 115, 56, 15, 168, 201, 229,
        106, 76, 74, 146, 239, 134, 255, 173, 57, 12, 127, 97, 2, 48, 26, 237,
        179, 74, 5, 153, 82, 88, 163, 190, 235, 136, 164, 1, 97, 180, 123, 255,
        199, 222, 138, 198, 18, 60, 116, 160, 238, 28, 26, 161, 116, 176, 83, 232,
        212, 226, 13, 253, 195, 186, 224, 188, 239, 150, 144, 107, 179, 201,
      ]),
      privateOrig: new Buffer([
        45, 45, 45, 45, 45, 66, 69, 71, 73, 78, 32, 82, 83, 65, 32, 80, 82,
        73, 86, 65, 84, 69, 32, 75, 69, 89, 45, 45, 45, 45, 45, 10, 77,
        73, 73, 66, 121, 119, 73, 66, 65, 65, 74, 104, 65, 78, 88, 121, 48,
        115, 76, 70, 81, 70, 103, 99, 89, 54, 115, 102, 121, 113, 108, 114, 66,
        66, 78, 65, 99, 111, 119, 55, 77, 87, 106, 81, 83, 48, 102, 108, 85,
        69, 106, 56, 72, 109, 114, 100, 84, 88, 43, 100, 101, 110, 90, 53, 10,
        115, 65, 114, 86, 69, 66, 87, 73, 111, 73, 99, 67, 87, 116, 48, 68,
        87, 71, 122, 47, 109, 90, 68, 83, 50, 88, 80, 113, 121, 121, 57, 73,
        98, 69, 110, 104, 119, 116, 90, 51, 98, 112, 72, 109, 119, 49, 49, 74,
        49, 107, 54, 49, 111, 100, 65, 119, 121, 53, 115, 69, 51, 80, 52, 83,
        10, 107, 83, 47, 83, 118, 108, 55, 87, 120, 106, 99, 114, 53, 119, 73,
        68, 65, 81, 65, 66, 65, 109, 69, 65, 120, 53, 48, 112, 71, 50, 81,
        105, 90, 112, 77, 71, 81, 118, 69, 69, 105, 120, 69, 51, 50, 105, 90,
        49, 79, 88, 85, 52, 52, 107, 119, 70, 69, 79, 49, 89, 47, 78, 43,
        79, 10, 71, 98, 75, 107, 122, 102, 53, 70, 57, 105, 103, 80, 73, 101,
        119, 115, 100, 52, 114, 107, 67, 49, 77, 98, 99, 86, 80, 89, 84, 78,
        54, 97, 55, 87, 108, 70, 104, 50, 100, 80, 49, 107, 67, 53, 119, 100,
        80, 65, 100, 86, 106, 66, 102, 49, 112, 55, 73, 105, 52, 116, 52, 72,
        90, 116, 10, 79, 111, 107, 105, 100, 100, 117, 74, 122, 67, 109, 67, 43,
        84, 117, 108, 68, 67, 78, 76, 70, 109, 108, 66, 65, 106, 69, 65, 47,
        54, 83, 85, 120, 99, 120, 82, 88, 74, 86, 55, 71, 72, 66, 109, 117,
        43, 70, 73, 98, 110, 82, 54, 47, 100, 114, 72, 97, 66, 118, 77, 107,
        78, 75, 90, 10, 47, 116, 71, 73, 113, 49, 75, 102, 114, 72, 73, 87,
        114, 104, 71, 108, 50, 118, 89, 113, 111, 116, 54, 104, 82, 103, 111, 88,
        65, 106, 69, 65, 49, 106, 57, 86, 65, 121, 117, 68, 75, 50, 78, 104,
        81, 71, 47, 87, 77, 109, 102, 56, 87, 86, 43, 79, 77, 101, 74, 81,
        55, 118, 112, 110, 10, 51, 75, 109, 116, 80, 99, 47, 67, 115, 87, 102,
        80, 78, 89, 101, 81, 97, 103, 102, 119, 108, 106, 48, 83, 76, 71, 119,
        122, 56, 74, 54, 120, 65, 106, 69, 65, 52, 113, 73, 82, 107, 108, 108,
        85, 78, 47, 80, 57, 77, 79, 108, 85, 54, 110, 77, 75, 78, 52, 72,
        107, 70, 51, 69, 66, 10, 72, 118, 97, 109, 73, 67, 51, 85, 102, 43,
        100, 97, 100, 105, 102, 70, 106, 113, 101, 71, 71, 56, 78, 104, 101, 103,
        67, 75, 81, 43, 71, 67, 115, 65, 106, 53, 65, 106, 65, 101, 99, 115,
        51, 65, 76, 53, 84, 113, 117, 54, 114, 56, 110, 50, 104, 80, 114, 90,
        117, 100, 107, 65, 107, 68, 10, 97, 101, 105, 97, 114, 104, 102, 73, 115,
        111, 121, 101, 100, 107, 74, 122, 79, 65, 43, 111, 121, 101, 86, 113, 84,
        69, 113, 83, 55, 52, 98, 47, 114, 84, 107, 77, 102, 50, 69, 67, 77,
        66, 114, 116, 115, 48, 111, 70, 109, 86, 74, 89, 111, 55, 55, 114, 105,
        75, 81, 66, 89, 98, 82, 55, 10, 47, 56, 102, 101, 105, 115, 89, 83,
        80, 72, 83, 103, 55, 104, 119, 97, 111, 88, 83, 119, 85, 43, 106, 85,
        52, 103, 51, 57, 119, 55, 114, 103, 118, 79, 43, 87, 107, 71, 117, 122,
        121, 81, 61, 61, 10, 45, 45, 45, 45, 45, 69, 78, 68, 32, 82, 83,
        65, 32, 80, 82, 73, 86, 65, 84, 69, 32, 75, 69, 89, 45, 45, 45,
        45, 45,
      ]),
      public: undefined,
      publicOrig: undefined
    },
    what: 'Unencrypted RSA private key (OpenSSH format)'
  },
  { source: [
      '-----BEGIN RSA PRIVATE KEY-----',
      'Proc-Type: 4,ENCRYPTED',
      'DEK-Info: AES-128-CBC,E9F7397F94E585BCE4C739D5C5226A2D',
      '',
      'mshPNeXq8XAh79N3F1vKUOCsWoTofhz/Kpt/ddYdgnGG9L5wGDoveJQtpsa5u+A4',
      'hCkAWA7KRCUKbOxZd8lG00cm7HILN7eHCKSoSkyA+X1jN7v1tSGPjiN2RSZFHCOI',
      '8jEF0fyTeg9GZzFaiyyD5p1niy7uCssEeo8AzbIBX4yLDR18mmBiMhn0FxVu/YFn',
      'rfkm+ciE4WhkqdCUOkKiqIux7RI6O9ttUuZlhwVvoKZqujMeNrFppOC9kF5hAi6Y',
      'oHvrweE8QmimEUIMAwdDrI1/jglFw2x2K5sRxPIIFJC1aHR8nXLTuO1g2/g6aqls',
      'jGkwlQU4cU5pnkxkLUVCSHi6A6nwbdjQTIIDofpcckQVFHZaBCJx9UF691NR05Dc',
      'GthFFfNGXWU6vbwKia/LDX8R98ay4Ci2zoDXxe7QBnvnWbs2m69HPh6opmo5gKL+',
      'y65Sf6MAi5udIJLVfgXQBgRyX3glS7/rH8YbrrNY9Iop1KcSWXAIC4zlCoO3tBJB',
      '9464YBD/R+qOvVMcBGxTjabudi00ougeTgsS7idDTmfc4HYIls0E7yWqzgspsws5',
      'w7K4crSgm+u3oypViUSre2qSvAHTC5gXth8LVCnDZuA=',
      '-----END RSA PRIVATE KEY-----'
    ].join('\n'),
    expected: {
      fulltype: undefined,
      type: 'rsa',
      extra: [ 'E9F7397F94E585BCE4C739D5C5226A2D' ],
      comment: undefined,
      encryption: 'aes-128-cbc',
      private: new Buffer([
        154, 200, 79, 53, 229, 234, 241, 112, 33, 239, 211, 119, 23, 91, 202, 80, 224,
        172, 90, 132, 232, 126, 28, 255, 42, 155, 127, 117, 214, 29, 130, 113, 134,
        244, 190, 112, 24, 58, 47, 120, 148, 45, 166, 198, 185, 187, 224, 56, 132,
        41, 0, 88, 14, 202, 68, 37, 10, 108, 236, 89, 119, 201, 70, 211, 71,
        38, 236, 114, 11, 55, 183, 135, 8, 164, 168, 74, 76, 128, 249, 125, 99,
        55, 187, 245, 181, 33, 143, 142, 35, 118, 69, 38, 69, 28, 35, 136, 242,
        49, 5, 209, 252, 147, 122, 15, 70, 103, 49, 90, 139, 44, 131, 230, 157,
        103, 139, 46, 238, 10, 203, 4, 122, 143, 0, 205, 178, 1, 95, 140, 139,
        13, 29, 124, 154, 96, 98, 50, 25, 244, 23, 21, 110, 253, 129, 103, 173,
        249, 38, 249, 200, 132, 225, 104, 100, 169, 208, 148, 58, 66, 162, 168, 139,
        177, 237, 18, 58, 59, 219, 109, 82, 230, 101, 135, 5, 111, 160, 166, 106,
        186, 51, 30, 54, 177, 105, 164, 224, 189, 144, 94, 97, 2, 46, 152, 160,
        123, 235, 193, 225, 60, 66, 104, 166, 17, 66, 12, 3, 7, 67, 172, 141,
        127, 142, 9, 69, 195, 108, 118, 43, 155, 17, 196, 242, 8, 20, 144, 181,
        104, 116, 124, 157, 114, 211, 184, 237, 96, 219, 248, 58, 106, 169, 108, 140,
        105, 48, 149, 5, 56, 113, 78, 105, 158, 76, 100, 45, 69, 66, 72, 120,
        186, 3, 169, 240, 109, 216, 208, 76, 130, 3, 161, 250, 92, 114, 68, 21,
        20, 118, 90, 4, 34, 113, 245, 65, 122, 247, 83, 81, 211, 144, 220, 26,
        216, 69, 21, 243, 70, 93, 101, 58, 189, 188, 10, 137, 175, 203, 13, 127,
        17, 247, 198, 178, 224, 40, 182, 206, 128, 215, 197, 238, 208, 6, 123, 231,
        89, 187, 54, 155, 175, 71, 62, 30, 168, 166, 106, 57, 128, 162, 254, 203,
        174, 82, 127, 163, 0, 139, 155, 157, 32, 146, 213, 126, 5, 208, 6, 4,
        114, 95, 120, 37, 75, 191, 235, 31, 198, 27, 174, 179, 88, 244, 138, 41,
        212, 167, 18, 89, 112, 8, 11, 140, 229, 10, 131, 183, 180, 18, 65, 247,
        142, 184, 96, 16, 255, 71, 234, 142, 189, 83, 28, 4, 108, 83, 141, 166,
        238, 118, 45, 52, 162, 232, 30, 78, 11, 18, 238, 39, 67, 78, 103, 220,
        224, 118, 8, 150, 205, 4, 239, 37, 170, 206, 11, 41, 179, 11, 57, 195,
        178, 184, 114, 180, 160, 155, 235, 183, 163, 42, 85, 137, 68, 171, 123, 106,
        146, 188, 1, 211, 11, 152, 23, 182, 31, 11, 84, 41, 195, 102, 224,
      ]),
      privateOrig: new Buffer([
        45, 45, 45, 45, 45, 66, 69, 71, 73, 78, 32, 82, 83, 65, 32, 80, 82,
        73, 86, 65, 84, 69, 32, 75, 69, 89, 45, 45, 45, 45, 45, 10, 80,
        114, 111, 99, 45, 84, 121, 112, 101, 58, 32, 52, 44, 69, 78, 67, 82,
        89, 80, 84, 69, 68, 10, 68, 69, 75, 45, 73, 110, 102, 111, 58, 32,
        65, 69, 83, 45, 49, 50, 56, 45, 67, 66, 67, 44, 69, 57, 70, 55,
        51, 57, 55, 70, 57, 52, 69, 53, 56, 53, 66, 67, 69, 52, 67, 55,
        51, 57, 68, 53, 67, 53, 50, 50, 54, 65, 50, 68, 10, 10, 109, 115,
        104, 80, 78, 101, 88, 113, 56, 88, 65, 104, 55, 57, 78, 51, 70, 49,
        118, 75, 85, 79, 67, 115, 87, 111, 84, 111, 102, 104, 122, 47, 75, 112,
        116, 47, 100, 100, 89, 100, 103, 110, 71, 71, 57, 76, 53, 119, 71, 68,
        111, 118, 101, 74, 81, 116, 112, 115, 97, 53, 117, 43, 65, 52, 10, 104,
        67, 107, 65, 87, 65, 55, 75, 82, 67, 85, 75, 98, 79, 120, 90, 100,
        56, 108, 71, 48, 48, 99, 109, 55, 72, 73, 76, 78, 55, 101, 72, 67,
        75, 83, 111, 83, 107, 121, 65, 43, 88, 49, 106, 78, 55, 118, 49, 116,
        83, 71, 80, 106, 105, 78, 50, 82, 83, 90, 70, 72, 67, 79, 73, 10,
        56, 106, 69, 70, 48, 102, 121, 84, 101, 103, 57, 71, 90, 122, 70, 97,
        105, 121, 121, 68, 53, 112, 49, 110, 105, 121, 55, 117, 67, 115, 115, 69,
        101, 111, 56, 65, 122, 98, 73, 66, 88, 52, 121, 76, 68, 82, 49, 56,
        109, 109, 66, 105, 77, 104, 110, 48, 70, 120, 86, 117, 47, 89, 70, 110,
        10, 114, 102, 107, 109, 43, 99, 105, 69, 52, 87, 104, 107, 113, 100, 67,
        85, 79, 107, 75, 105, 113, 73, 117, 120, 55, 82, 73, 54, 79, 57, 116,
        116, 85, 117, 90, 108, 104, 119, 86, 118, 111, 75, 90, 113, 117, 106, 77,
        101, 78, 114, 70, 112, 112, 79, 67, 57, 107, 70, 53, 104, 65, 105, 54,
        89, 10, 111, 72, 118, 114, 119, 101, 69, 56, 81, 109, 105, 109, 69, 85,
        73, 77, 65, 119, 100, 68, 114, 73, 49, 47, 106, 103, 108, 70, 119, 50,
        120, 50, 75, 53, 115, 82, 120, 80, 73, 73, 70, 74, 67, 49, 97, 72,
        82, 56, 110, 88, 76, 84, 117, 79, 49, 103, 50, 47, 103, 54, 97, 113,
        108, 115, 10, 106, 71, 107, 119, 108, 81, 85, 52, 99, 85, 53, 112, 110,
        107, 120, 107, 76, 85, 86, 67, 83, 72, 105, 54, 65, 54, 110, 119, 98,
        100, 106, 81, 84, 73, 73, 68, 111, 102, 112, 99, 99, 107, 81, 86, 70,
        72, 90, 97, 66, 67, 74, 120, 57, 85, 70, 54, 57, 49, 78, 82, 48,
        53, 68, 99, 10, 71, 116, 104, 70, 70, 102, 78, 71, 88, 87, 85, 54,
        118, 98, 119, 75, 105, 97, 47, 76, 68, 88, 56, 82, 57, 56, 97, 121,
        52, 67, 105, 50, 122, 111, 68, 88, 120, 101, 55, 81, 66, 110, 118, 110,
        87, 98, 115, 50, 109, 54, 57, 72, 80, 104, 54, 111, 112, 109, 111, 53,
        103, 75, 76, 43, 10, 121, 54, 53, 83, 102, 54, 77, 65, 105, 53, 117,
        100, 73, 74, 76, 86, 102, 103, 88, 81, 66, 103, 82, 121, 88, 51, 103,
        108, 83, 55, 47, 114, 72, 56, 89, 98, 114, 114, 78, 89, 57, 73, 111,
        112, 49, 75, 99, 83, 87, 88, 65, 73, 67, 52, 122, 108, 67, 111, 79,
        51, 116, 66, 74, 66, 10, 57, 52, 54, 52, 89, 66, 68, 47, 82, 43,
        113, 79, 118, 86, 77, 99, 66, 71, 120, 84, 106, 97, 98, 117, 100, 105,
        48, 48, 111, 117, 103, 101, 84, 103, 115, 83, 55, 105, 100, 68, 84, 109,
        102, 99, 52, 72, 89, 73, 108, 115, 48, 69, 55, 121, 87, 113, 122, 103,
        115, 112, 115, 119, 115, 53, 10, 119, 55, 75, 52, 99, 114, 83, 103, 109,
        43, 117, 51, 111, 121, 112, 86, 105, 85, 83, 114, 101, 50, 113, 83, 118,
        65, 72, 84, 67, 53, 103, 88, 116, 104, 56, 76, 86, 67, 110, 68, 90,
        117, 65, 61, 10, 45, 45, 45, 45, 45, 69, 78, 68, 32, 82, 83, 65,
        32, 80, 82, 73, 86, 65, 84, 69, 32, 75, 69, 89, 45, 45, 45, 45,
        45,
      ]),
      public: undefined,
      publicOrig: undefined
    },
    what: 'Encrypted RSA private key (OpenSSH format)'
  },
  { source: 'ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRLlnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0fxoLSF3lxQFJ+32/rS0=',
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 7, 115, 115, 104, 45, 114, 115, 97, 0, 0, 0, 3, 1, 0,
        1, 0, 0, 0, 97, 0, 229, 233, 220, 75, 250, 24, 49, 192, 99, 97,
        102, 186, 181, 242, 240, 43, 108, 208, 116, 82, 163, 113, 189, 201, 95, 250,
        135, 34, 68, 185, 103, 71, 147, 186, 175, 247, 51, 2, 28, 146, 59, 231,
        142, 115, 148, 133, 239, 7, 177, 103, 53, 168, 252, 35, 144, 223, 187, 68,
        139, 218, 222, 78, 231, 24, 20, 212, 16, 88, 91, 241, 224, 79, 2, 68,
        83, 254, 80, 244, 122, 133, 5, 244, 127, 26, 11, 72, 93, 229, 197, 1,
        73, 251, 125, 191, 173, 45,
      ]),
      publicOrig: new Buffer([
        115, 115, 104, 45, 114, 115, 97, 32, 65, 65, 65, 65, 66, 51, 78, 122, 97,
        67, 49, 121, 99, 50, 69, 65, 65, 65, 65, 68, 65, 81, 65, 66, 65,
        65, 65, 65, 89, 81, 68, 108, 54, 100, 120, 76, 43, 104, 103, 120, 119,
        71, 78, 104, 90, 114, 113, 49, 56, 118, 65, 114, 98, 78, 66, 48, 85,
        113, 78, 120, 118, 99, 108, 102, 43, 111, 99, 105, 82, 76, 108, 110, 82,
        53, 79, 54, 114, 47, 99, 122, 65, 104, 121, 83, 79, 43, 101, 79, 99,
        53, 83, 70, 55, 119, 101, 120, 90, 122, 87, 111, 47, 67, 79, 81, 51,
        55, 116, 69, 105, 57, 114, 101, 84, 117, 99, 89, 70, 78, 81, 81, 87,
        70, 118, 120, 52, 69, 56, 67, 82, 70, 80, 43, 85, 80, 82, 54, 104,
        81, 88, 48, 102, 120, 111, 76, 83, 70, 51, 108, 120, 81, 70, 74, 43,
        51, 50, 47, 114, 83, 48, 61,
      ])
    },
    what: 'RSA public key (OpenSSH format)'
  },
  { source: 'ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAAYQD3txsEf0HAKElAFUvIXzsM98gfPlIbG4/GlqbYYBulkHu6z0laOdoT14Zx2M+3q+9RjhTZjHxyMfePdcgNK9z98V6tOz5bIQhtMS8tl1Tnw5qZByGqpqOKf665ev62LaM= testing-ssh2-from-node.js',
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing-ssh2-from-node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 7, 115, 115, 104, 45, 114, 115, 97, 0, 0, 0, 3, 1, 0,
        1, 0, 0, 0, 97, 0, 247, 183, 27, 4, 127, 65, 192, 40, 73, 64,
        21, 75, 200, 95, 59, 12, 247, 200, 31, 62, 82, 27, 27, 143, 198, 150,
        166, 216, 96, 27, 165, 144, 123, 186, 207, 73, 90, 57, 218, 19, 215, 134,
        113, 216, 207, 183, 171, 239, 81, 142, 20, 217, 140, 124, 114, 49, 247, 143,
        117, 200, 13, 43, 220, 253, 241, 94, 173, 59, 62, 91, 33, 8, 109, 49,
        47, 45, 151, 84, 231, 195, 154, 153, 7, 33, 170, 166, 163, 138, 127, 174,
        185, 122, 254, 182, 45, 163, 45, 122, 203, 98, 158, 15, 172, 178, 29, 190,
        126, 186, 38, 250, 122, 29, 122, 59,
      ]),
      publicOrig: new Buffer([
        115, 115, 104, 45, 114, 115, 97, 32, 65, 65, 65, 65, 66, 51, 78, 122, 97,
        67, 49, 121, 99, 50, 69, 65, 65, 65, 65, 68, 65, 81, 65, 66, 65,
        65, 65, 65, 89, 81, 68, 51, 116, 120, 115, 69, 102, 48, 72, 65, 75,
        69, 108, 65, 70, 85, 118, 73, 88, 122, 115, 77, 57, 56, 103, 102, 80,
        108, 73, 98, 71, 52, 47, 71, 108, 113, 98, 89, 89, 66, 117, 108, 107,
        72, 117, 54, 122, 48, 108, 97, 79, 100, 111, 84, 49, 52, 90, 120, 50,
        77, 43, 51, 113, 43, 57, 82, 106, 104, 84, 90, 106, 72, 120, 121, 77,
        102, 101, 80, 100, 99, 103, 78, 75, 57, 122, 57, 56, 86, 54, 116, 79,
        122, 53, 98, 73, 81, 104, 116, 77, 83, 56, 116, 108, 49, 84, 110, 119,
        53, 113, 90, 66, 121, 71, 113, 112, 113, 79, 75, 102, 54, 54, 53, 101,
        118, 54, 50, 76, 97, 77, 61, 32, 116, 101, 115, 116, 105, 110, 103, 45,
        115, 115, 104, 50, 45, 102, 114, 111, 109, 45, 110, 111, 100, 101, 46, 106,
        115,
      ])
    },
    what: 'RSA public key with comment (OpenSSH format)'
  },
  { source: [
      '-----BEGIN DSA PRIVATE KEY-----',
      'MIIBuwIBAAKBgQD7v1kR31D2NhZIzOjJL1hPzvw79K3eWZqQEgqKmeB+P9MhOx51',
      'MOZrvs3hZoYqmsNxT/Y29EwQ+o+SXpTYjJhw/s2vR0AeJBj32l8weD804+T+S8yr',
      'MJBdgsc5AbV2XYKnsNzl65kAEOLPCobUNysKqYnJ8naYCYL3jjwGhW36iwIVANqh',
      'gZgvnVpVXyYx1GlceA5/8mI1AoGBAOyjuQhGoOW4hILR02WrqwCEPqhRyMp87dMU',
      '9z0PQKKrACmiQpEUawvtR4aMB7Xzy+f6MRLwdonQISzswopa7Gwl3CYLiOdKFFIQ',
      'zDdymwGovTjN2fQm8v/UHdafZBPImGBDzf+iykNBhZc7UP1rciMzKOnT3BTF/al8',
      'hhBy0bjsAoGAYrykryPYWUGArKS6NO9Ijtwc7SbbNXZCuBcl6hc1Hdr1UEEKQcsW',
      '5fajF5Ut9ZLwcggsWSKZtZk9NA7xybmObuDEU7oiFFNRAbBkam/v/3a3bwvSHGKg',
      'q15cvC313zg1ii9NXylvBIoIS8EWfekl1LMvHC7NdJ41wPYhhmPrUk4CFHJjavJH',
      'kb6tDtemyiAq3N9aQj/D',
      '-----END DSA PRIVATE KEY-----'
    ].join('\n'),
    expected: {
      fulltype: undefined,
      type: 'dss',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: new Buffer([
        48, 130, 1, 187, 2, 1, 0, 2, 129, 129, 0, 251, 191, 89, 17, 223, 80,
        246, 54, 22, 72, 204, 232, 201, 47, 88, 79, 206, 252, 59, 244, 173, 222,
        89, 154, 144, 18, 10, 138, 153, 224, 126, 63, 211, 33, 59, 30, 117, 48,
        230, 107, 190, 205, 225, 102, 134, 42, 154, 195, 113, 79, 246, 54, 244, 76,
        16, 250, 143, 146, 94, 148, 216, 140, 152, 112, 254, 205, 175, 71, 64, 30,
        36, 24, 247, 218, 95, 48, 120, 63, 52, 227, 228, 254, 75, 204, 171, 48,
        144, 93, 130, 199, 57, 1, 181, 118, 93, 130, 167, 176, 220, 229, 235, 153,
        0, 16, 226, 207, 10, 134, 212, 55, 43, 10, 169, 137, 201, 242, 118, 152,
        9, 130, 247, 142, 60, 6, 133, 109, 250, 139, 2, 21, 0, 218, 161, 129,
        152, 47, 157, 90, 85, 95, 38, 49, 212, 105, 92, 120, 14, 127, 242, 98,
        53, 2, 129, 129, 0, 236, 163, 185, 8, 70, 160, 229, 184, 132, 130, 209,
        211, 101, 171, 171, 0, 132, 62, 168, 81, 200, 202, 124, 237, 211, 20, 247,
        61, 15, 64, 162, 171, 0, 41, 162, 66, 145, 20, 107, 11, 237, 71, 134,
        140, 7, 181, 243, 203, 231, 250, 49, 18, 240, 118, 137, 208, 33, 44, 236,
        194, 138, 90, 236, 108, 37, 220, 38, 11, 136, 231, 74, 20, 82, 16, 204,
        55, 114, 155, 1, 168, 189, 56, 205, 217, 244, 38, 242, 255, 212, 29, 214,
        159, 100, 19, 200, 152, 96, 67, 205, 255, 162, 202, 67, 65, 133, 151, 59,
        80, 253, 107, 114, 35, 51, 40, 233, 211, 220, 20, 197, 253, 169, 124, 134,
        16, 114, 209, 184, 236, 2, 129, 128, 98, 188, 164, 175, 35, 216, 89, 65,
        128, 172, 164, 186, 52, 239, 72, 142, 220, 28, 237, 38, 219, 53, 118, 66,
        184, 23, 37, 234, 23, 53, 29, 218, 245, 80, 65, 10, 65, 203, 22, 229,
        246, 163, 23, 149, 45, 245, 146, 240, 114, 8, 44, 89, 34, 153, 181, 153,
        61, 52, 14, 241, 201, 185, 142, 110, 224, 196, 83, 186, 34, 20, 83, 81,
        1, 176, 100, 106, 111, 239, 255, 118, 183, 111, 11, 210, 28, 98, 160, 171,
        94, 92, 188, 45, 245, 223, 56, 53, 138, 47, 77, 95, 41, 111, 4, 138,
        8, 75, 193, 22, 125, 233, 37, 212, 179, 47, 28, 46, 205, 116, 158, 53,
        192, 246, 33, 134, 99, 235, 82, 78, 2, 20, 114, 99, 106, 242, 71, 145,
        190, 173, 14, 215, 166, 202, 32, 42, 220, 223, 90, 66, 63, 195,
      ]),
      privateOrig: new Buffer([
        45, 45, 45, 45, 45, 66, 69, 71, 73, 78, 32, 68, 83, 65, 32, 80, 82,
        73, 86, 65, 84, 69, 32, 75, 69, 89, 45, 45, 45, 45, 45, 10, 77,
        73, 73, 66, 117, 119, 73, 66, 65, 65, 75, 66, 103, 81, 68, 55, 118,
        49, 107, 82, 51, 49, 68, 50, 78, 104, 90, 73, 122, 79, 106, 74, 76,
        49, 104, 80, 122, 118, 119, 55, 57, 75, 51, 101, 87, 90, 113, 81, 69,
        103, 113, 75, 109, 101, 66, 43, 80, 57, 77, 104, 79, 120, 53, 49, 10,
        77, 79, 90, 114, 118, 115, 51, 104, 90, 111, 89, 113, 109, 115, 78, 120,
        84, 47, 89, 50, 57, 69, 119, 81, 43, 111, 43, 83, 88, 112, 84, 89,
        106, 74, 104, 119, 47, 115, 50, 118, 82, 48, 65, 101, 74, 66, 106, 51,
        50, 108, 56, 119, 101, 68, 56, 48, 52, 43, 84, 43, 83, 56, 121, 114,
        10, 77, 74, 66, 100, 103, 115, 99, 53, 65, 98, 86, 50, 88, 89, 75,
        110, 115, 78, 122, 108, 54, 53, 107, 65, 69, 79, 76, 80, 67, 111, 98,
        85, 78, 121, 115, 75, 113, 89, 110, 74, 56, 110, 97, 89, 67, 89, 76,
        51, 106, 106, 119, 71, 104, 87, 51, 54, 105, 119, 73, 86, 65, 78, 113,
        104, 10, 103, 90, 103, 118, 110, 86, 112, 86, 88, 121, 89, 120, 49, 71,
        108, 99, 101, 65, 53, 47, 56, 109, 73, 49, 65, 111, 71, 66, 65, 79,
        121, 106, 117, 81, 104, 71, 111, 79, 87, 52, 104, 73, 76, 82, 48, 50,
        87, 114, 113, 119, 67, 69, 80, 113, 104, 82, 121, 77, 112, 56, 55, 100,
        77, 85, 10, 57, 122, 48, 80, 81, 75, 75, 114, 65, 67, 109, 105, 81,
        112, 69, 85, 97, 119, 118, 116, 82, 52, 97, 77, 66, 55, 88, 122, 121,
        43, 102, 54, 77, 82, 76, 119, 100, 111, 110, 81, 73, 83, 122, 115, 119,
        111, 112, 97, 55, 71, 119, 108, 51, 67, 89, 76, 105, 79, 100, 75, 70,
        70, 73, 81, 10, 122, 68, 100, 121, 109, 119, 71, 111, 118, 84, 106, 78,
        50, 102, 81, 109, 56, 118, 47, 85, 72, 100, 97, 102, 90, 66, 80, 73,
        109, 71, 66, 68, 122, 102, 43, 105, 121, 107, 78, 66, 104, 90, 99, 55,
        85, 80, 49, 114, 99, 105, 77, 122, 75, 79, 110, 84, 51, 66, 84, 70,
        47, 97, 108, 56, 10, 104, 104, 66, 121, 48, 98, 106, 115, 65, 111, 71,
        65, 89, 114, 121, 107, 114, 121, 80, 89, 87, 85, 71, 65, 114, 75, 83,
        54, 78, 79, 57, 73, 106, 116, 119, 99, 55, 83, 98, 98, 78, 88, 90,
        67, 117, 66, 99, 108, 54, 104, 99, 49, 72, 100, 114, 49, 85, 69, 69,
        75, 81, 99, 115, 87, 10, 53, 102, 97, 106, 70, 53, 85, 116, 57, 90,
        76, 119, 99, 103, 103, 115, 87, 83, 75, 90, 116, 90, 107, 57, 78, 65,
        55, 120, 121, 98, 109, 79, 98, 117, 68, 69, 85, 55, 111, 105, 70, 70,
        78, 82, 65, 98, 66, 107, 97, 109, 47, 118, 47, 51, 97, 51, 98, 119,
        118, 83, 72, 71, 75, 103, 10, 113, 49, 53, 99, 118, 67, 51, 49, 51,
        122, 103, 49, 105, 105, 57, 78, 88, 121, 108, 118, 66, 73, 111, 73, 83,
        56, 69, 87, 102, 101, 107, 108, 49, 76, 77, 118, 72, 67, 55, 78, 100,
        74, 52, 49, 119, 80, 89, 104, 104, 109, 80, 114, 85, 107, 52, 67, 70,
        72, 74, 106, 97, 118, 74, 72, 10, 107, 98, 54, 116, 68, 116, 101, 109,
        121, 105, 65, 113, 51, 78, 57, 97, 81, 106, 47, 68, 10, 45, 45, 45,
        45, 45, 69, 78, 68, 32, 68, 83, 65, 32, 80, 82, 73, 86, 65, 84,
        69, 32, 75, 69, 89, 45, 45, 45, 45, 45,
      ]),
      public: undefined,
      publicOrig: undefined
    },
    what: 'Unencrypted DSA private key (OpenSSH format)'
  },
  { source: [
      '-----BEGIN DSA PRIVATE KEY-----',
      'Proc-Type: 4,ENCRYPTED',
      'DEK-Info: AES-128-CBC,D6218FEC397F57918B06DC3F9EC33487',
      '',
      'OjtbW9WChM2baMDRXW/IYhmqAmlaulb8pQd0rDcFYGrAS/5PnA96R3X6f9tdooSv',
      'uPrQ37oCI7FfDrRmVy6pn6QOLxLZHk7LViBAA7q8VytRHUyezauicTkhc6/nfxQU',
      'tYN5dNajSl/aFuv/IitaYtuG2ORvXLLJhR3sYe5fs2ai14qtULdWxtrZ4cACXPSz',
      'Ij29/lUfeBIYxNFFDF5hgsyO+jHLHO8bJCgXLCEb1o/Im68kQT3LjmvLbdhVM2+V',
      '5c+YVDW/yXKpaSDNqctJFsETP3SsI5k9PVijSeND4U27W9HAoukJmG8jNBXmMOKW',
      'MtOEp0Yw5+fu0OPTACMcK7UoiN90RTFLNcDT9kKjeQwqk37sZBNyOpnQ57Mu6YWX',
      'BS5nnT4bK/WpyWUgGJ+MVzcp6pErBUy3SxlV2LN/LIZq+YzME55sygtwNSuVhpBH',
      '+9UwrUIlaT8Ru//OdRKY5Vp8AqomjlAzCkBxnmC9XhVB0r4zqsqgVyzSfsX+oGLe',
      'dbixpbeMXe3WoAowx6LmqojuP46MIL80/9BH6Xr51UqJj156bYXashJtgpYaAQIn',
      '8YB9sXf4S027sgHxK0Xanw==',
      '-----END DSA PRIVATE KEY-----'
    ].join('\n'),
    expected: {
      fulltype: undefined,
      type: 'dss',
      extra: [ 'D6218FEC397F57918B06DC3F9EC33487' ],
      comment: undefined,
      encryption: 'aes-128-cbc',
      private: new Buffer([
        58, 59, 91, 91, 213, 130, 132, 205, 155, 104, 192, 209, 93, 111, 200, 98, 25,
        170, 2, 105, 90, 186, 86, 252, 165, 7, 116, 172, 55, 5, 96, 106, 192,
        75, 254, 79, 156, 15, 122, 71, 117, 250, 127, 219, 93, 162, 132, 175, 184,
        250, 208, 223, 186, 2, 35, 177, 95, 14, 180, 102, 87, 46, 169, 159, 164,
        14, 47, 18, 217, 30, 78, 203, 86, 32, 64, 3, 186, 188, 87, 43, 81,
        29, 76, 158, 205, 171, 162, 113, 57, 33, 115, 175, 231, 127, 20, 20, 181,
        131, 121, 116, 214, 163, 74, 95, 218, 22, 235, 255, 34, 43, 90, 98, 219,
        134, 216, 228, 111, 92, 178, 201, 133, 29, 236, 97, 238, 95, 179, 102, 162,
        215, 138, 173, 80, 183, 86, 198, 218, 217, 225, 192, 2, 92, 244, 179, 34,
        61, 189, 254, 85, 31, 120, 18, 24, 196, 209, 69, 12, 94, 97, 130, 204,
        142, 250, 49, 203, 28, 239, 27, 36, 40, 23, 44, 33, 27, 214, 143, 200,
        155, 175, 36, 65, 61, 203, 142, 107, 203, 109, 216, 85, 51, 111, 149, 229,
        207, 152, 84, 53, 191, 201, 114, 169, 105, 32, 205, 169, 203, 73, 22, 193,
        19, 63, 116, 172, 35, 153, 61, 61, 88, 163, 73, 227, 67, 225, 77, 187,
        91, 209, 192, 162, 233, 9, 152, 111, 35, 52, 21, 230, 48, 226, 150, 50,
        211, 132, 167, 70, 48, 231, 231, 238, 208, 227, 211, 0, 35, 28, 43, 181,
        40, 136, 223, 116, 69, 49, 75, 53, 192, 211, 246, 66, 163, 121, 12, 42,
        147, 126, 236, 100, 19, 114, 58, 153, 208, 231, 179, 46, 233, 133, 151, 5,
        46, 103, 157, 62, 27, 43, 245, 169, 201, 101, 32, 24, 159, 140, 87, 55,
        41, 234, 145, 43, 5, 76, 183, 75, 25, 85, 216, 179, 127, 44, 134, 106,
        249, 140, 204, 19, 158, 108, 202, 11, 112, 53, 43, 149, 134, 144, 71, 251,
        213, 48, 173, 66, 37, 105, 63, 17, 187, 255, 206, 117, 18, 152, 229, 90,
        124, 2, 170, 38, 142, 80, 51, 10, 64, 113, 158, 96, 189, 94, 21, 65,
        210, 190, 51, 170, 202, 160, 87, 44, 210, 126, 197, 254, 160, 98, 222, 117,
        184, 177, 165, 183, 140, 93, 237, 214, 160, 10, 48, 199, 162, 230, 170, 136,
        238, 63, 142, 140, 32, 191, 52, 255, 208, 71, 233, 122, 249, 213, 74, 137,
        143, 94, 122, 109, 133, 218, 178, 18, 109, 130, 150, 26, 1, 2, 39, 241,
        128, 125, 177, 119, 248, 75, 77, 187, 178, 1, 241, 43, 69, 218, 159,
      ]),
      privateOrig: new Buffer([
        45, 45, 45, 45, 45, 66, 69, 71, 73, 78, 32, 68, 83, 65, 32, 80, 82,
        73, 86, 65, 84, 69, 32, 75, 69, 89, 45, 45, 45, 45, 45, 10, 80,
        114, 111, 99, 45, 84, 121, 112, 101, 58, 32, 52, 44, 69, 78, 67, 82,
        89, 80, 84, 69, 68, 10, 68, 69, 75, 45, 73, 110, 102, 111, 58, 32,
        65, 69, 83, 45, 49, 50, 56, 45, 67, 66, 67, 44, 68, 54, 50, 49,
        56, 70, 69, 67, 51, 57, 55, 70, 53, 55, 57, 49, 56, 66, 48, 54,
        68, 67, 51, 70, 57, 69, 67, 51, 51, 52, 56, 55, 10, 10, 79, 106,
        116, 98, 87, 57, 87, 67, 104, 77, 50, 98, 97, 77, 68, 82, 88, 87,
        47, 73, 89, 104, 109, 113, 65, 109, 108, 97, 117, 108, 98, 56, 112, 81,
        100, 48, 114, 68, 99, 70, 89, 71, 114, 65, 83, 47, 53, 80, 110, 65,
        57, 54, 82, 51, 88, 54, 102, 57, 116, 100, 111, 111, 83, 118, 10, 117,
        80, 114, 81, 51, 55, 111, 67, 73, 55, 70, 102, 68, 114, 82, 109, 86,
        121, 54, 112, 110, 54, 81, 79, 76, 120, 76, 90, 72, 107, 55, 76, 86,
        105, 66, 65, 65, 55, 113, 56, 86, 121, 116, 82, 72, 85, 121, 101, 122,
        97, 117, 105, 99, 84, 107, 104, 99, 54, 47, 110, 102, 120, 81, 85, 10,
        116, 89, 78, 53, 100, 78, 97, 106, 83, 108, 47, 97, 70, 117, 118, 47,
        73, 105, 116, 97, 89, 116, 117, 71, 50, 79, 82, 118, 88, 76, 76, 74,
        104, 82, 51, 115, 89, 101, 53, 102, 115, 50, 97, 105, 49, 52, 113, 116,
        85, 76, 100, 87, 120, 116, 114, 90, 52, 99, 65, 67, 88, 80, 83, 122,
        10, 73, 106, 50, 57, 47, 108, 85, 102, 101, 66, 73, 89, 120, 78, 70,
        70, 68, 70, 53, 104, 103, 115, 121, 79, 43, 106, 72, 76, 72, 79, 56,
        98, 74, 67, 103, 88, 76, 67, 69, 98, 49, 111, 47, 73, 109, 54, 56,
        107, 81, 84, 51, 76, 106, 109, 118, 76, 98, 100, 104, 86, 77, 50, 43,
        86, 10, 53, 99, 43, 89, 86, 68, 87, 47, 121, 88, 75, 112, 97, 83,
        68, 78, 113, 99, 116, 74, 70, 115, 69, 84, 80, 51, 83, 115, 73, 53,
        107, 57, 80, 86, 105, 106, 83, 101, 78, 68, 52, 85, 50, 55, 87, 57,
        72, 65, 111, 117, 107, 74, 109, 71, 56, 106, 78, 66, 88, 109, 77, 79,
        75, 87, 10, 77, 116, 79, 69, 112, 48, 89, 119, 53, 43, 102, 117, 48,
        79, 80, 84, 65, 67, 77, 99, 75, 55, 85, 111, 105, 78, 57, 48, 82,
        84, 70, 76, 78, 99, 68, 84, 57, 107, 75, 106, 101, 81, 119, 113, 107,
        51, 55, 115, 90, 66, 78, 121, 79, 112, 110, 81, 53, 55, 77, 117, 54,
        89, 87, 88, 10, 66, 83, 53, 110, 110, 84, 52, 98, 75, 47, 87, 112,
        121, 87, 85, 103, 71, 74, 43, 77, 86, 122, 99, 112, 54, 112, 69, 114,
        66, 85, 121, 51, 83, 120, 108, 86, 50, 76, 78, 47, 76, 73, 90, 113,
        43, 89, 122, 77, 69, 53, 53, 115, 121, 103, 116, 119, 78, 83, 117, 86,
        104, 112, 66, 72, 10, 43, 57, 85, 119, 114, 85, 73, 108, 97, 84, 56,
        82, 117, 47, 47, 79, 100, 82, 75, 89, 53, 86, 112, 56, 65, 113, 111,
        109, 106, 108, 65, 122, 67, 107, 66, 120, 110, 109, 67, 57, 88, 104, 86,
        66, 48, 114, 52, 122, 113, 115, 113, 103, 86, 121, 122, 83, 102, 115, 88,
        43, 111, 71, 76, 101, 10, 100, 98, 105, 120, 112, 98, 101, 77, 88, 101,
        51, 87, 111, 65, 111, 119, 120, 54, 76, 109, 113, 111, 106, 117, 80, 52,
        54, 77, 73, 76, 56, 48, 47, 57, 66, 72, 54, 88, 114, 53, 49, 85,
        113, 74, 106, 49, 53, 54, 98, 89, 88, 97, 115, 104, 74, 116, 103, 112,
        89, 97, 65, 81, 73, 110, 10, 56, 89, 66, 57, 115, 88, 102, 52, 83,
        48, 50, 55, 115, 103, 72, 120, 75, 48, 88, 97, 110, 119, 61, 61, 10,
        45, 45, 45, 45, 45, 69, 78, 68, 32, 68, 83, 65, 32, 80, 82, 73,
        86, 65, 84, 69, 32, 75, 69, 89, 45, 45, 45, 45, 45,
      ]),
      public: undefined,
      publicOrig: undefined
    },
    what: 'Encrypted DSA private key (OpenSSH format)'
  },
  { source: 'ssh-dss AAAAB3NzaC1kc3MAAACBAL1FzdhtSYN22YgbInypxa//RyKiZAvWxNzDKNF0cYvabH6ESjdwdEcW96H7pHaKb69hoCHKZzfWmdpj93CcezGnorWvUaO4nAvbR+l/gH+AEYzHmJsECvM/nWnmPltwbvrPi/NmXCNf5aeqQlPbL44vZZjFRcCEwOomGY2FWUKJAAAAFQDBD5Y6VM8slL2I/HhqTmnoVFuGmQAAAIAUkwKajnUklN4CX/ZWEhIOZWCRa6ZKCosICBc/yC4RGJldqsXaM0PCmdy0dmcv0XuOZ4Nb92aiZnWdGRXTN5KJ405WVOQLBB3LpebA6qeyDGiVt/iPOxgJGZy0LIWiqWuzAAf199cQ2BkWmGTC6AAJi1depXC+KVAP+HwXOLjSUAAAAIBeoyKKEiVDQOpB0KAr5RLH3/+5BP+mR+52gTAbxZbxQPYyH7mqq2/PI1A0bjnvnLe45a/iVumoIjP/atMtVJ6w103Ex5JGMwVB7uZBWeyJ4m3MqQdPf1/+r+UCoOaL7stK3rWeTCydb82z34ejX0dZobAPGj2F7SyZy7jM8THgcw==',
    expected: {
      fulltype: 'ssh-dss',
      type: 'dss',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 7, 115, 115, 104, 45, 100, 115, 115, 0, 0, 0, 129, 0, 189,
        69, 205, 216, 109, 73, 131, 118, 217, 136, 27, 34, 124, 169, 197, 175, 255,
        71, 34, 162, 100, 11, 214, 196, 220, 195, 40, 209, 116, 113, 139, 218, 108,
        126, 132, 74, 55, 112, 116, 71, 22, 247, 161, 251, 164, 118, 138, 111, 175,
        97, 160, 33, 202, 103, 55, 214, 153, 218, 99, 247, 112, 156, 123, 49, 167,
        162, 181, 175, 81, 163, 184, 156, 11, 219, 71, 233, 127, 128, 127, 128, 17,
        140, 199, 152, 155, 4, 10, 243, 63, 157, 105, 230, 62, 91, 112, 110, 250,
        207, 139, 243, 102, 92, 35, 95, 229, 167, 170, 66, 83, 219, 47, 142, 47,
        101, 152, 197, 69, 192, 132, 192, 234, 38, 25, 141, 133, 89, 66, 137, 0,
        0, 0, 21, 0, 193, 15, 150, 58, 84, 207, 44, 148, 189, 136, 252, 120,
        106, 78, 105, 232, 84, 91, 134, 153, 0, 0, 0, 128, 20, 147, 2, 154,
        142, 117, 36, 148, 222, 2, 95, 246, 86, 18, 18, 14, 101, 96, 145, 107,
        166, 74, 10, 139, 8, 8, 23, 63, 200, 46, 17, 24, 153, 93, 170, 197,
        218, 51, 67, 194, 153, 220, 180, 118, 103, 47, 209, 123, 142, 103, 131, 91,
        247, 102, 162, 102, 117, 157, 25, 21, 211, 55, 146, 137, 227, 78, 86, 84,
        228, 11, 4, 29, 203, 165, 230, 192, 234, 167, 178, 12, 104, 149, 183, 248,
        143, 59, 24, 9, 25, 156, 180, 44, 133, 162, 169, 107, 179, 0, 7, 245,
        247, 215, 16, 216, 25, 22, 152, 100, 194, 232, 0, 9, 139, 87, 94, 165,
        112, 190, 41, 80, 15, 248, 124, 23, 56, 184, 210, 80, 0, 0, 0, 128,
        94, 163, 34, 138, 18, 37, 67, 64, 234, 65, 208, 160, 43, 229, 18, 199,
        223, 255, 185, 4, 255, 166, 71, 238, 118, 129, 48, 27, 197, 150, 241, 64,
        246, 50, 31, 185, 170, 171, 111, 207, 35, 80, 52, 110, 57, 239, 156, 183,
        184, 229, 175, 226, 86, 233, 168, 34, 51, 255, 106, 211, 45, 84, 158, 176,
        215, 77, 196, 199, 146, 70, 51, 5, 65, 238, 230, 65, 89, 236, 137, 226,
        109, 204, 169, 7, 79, 127, 95, 254, 175, 229, 2, 160, 230, 139, 238, 203,
        74, 222, 181, 158, 76, 44, 157, 111, 205, 179, 223, 135, 163, 95, 71, 89,
        161, 176, 15, 26, 61, 133, 237, 44, 153, 203, 184, 204, 241, 49, 224, 115,
      ]),
      publicOrig: new Buffer([
        115, 115, 104, 45, 100, 115, 115, 32, 65, 65, 65, 65, 66, 51, 78, 122, 97,
        67, 49, 107, 99, 51, 77, 65, 65, 65, 67, 66, 65, 76, 49, 70, 122,
        100, 104, 116, 83, 89, 78, 50, 50, 89, 103, 98, 73, 110, 121, 112, 120,
        97, 47, 47, 82, 121, 75, 105, 90, 65, 118, 87, 120, 78, 122, 68, 75,
        78, 70, 48, 99, 89, 118, 97, 98, 72, 54, 69, 83, 106, 100, 119, 100,
        69, 99, 87, 57, 54, 72, 55, 112, 72, 97, 75, 98, 54, 57, 104, 111,
        67, 72, 75, 90, 122, 102, 87, 109, 100, 112, 106, 57, 51, 67, 99, 101,
        122, 71, 110, 111, 114, 87, 118, 85, 97, 79, 52, 110, 65, 118, 98, 82,
        43, 108, 47, 103, 72, 43, 65, 69, 89, 122, 72, 109, 74, 115, 69, 67,
        118, 77, 47, 110, 87, 110, 109, 80, 108, 116, 119, 98, 118, 114, 80, 105,
        47, 78, 109, 88, 67, 78, 102, 53, 97, 101, 113, 81, 108, 80, 98, 76,
        52, 52, 118, 90, 90, 106, 70, 82, 99, 67, 69, 119, 79, 111, 109, 71,
        89, 50, 70, 87, 85, 75, 74, 65, 65, 65, 65, 70, 81, 68, 66, 68,
        53, 89, 54, 86, 77, 56, 115, 108, 76, 50, 73, 47, 72, 104, 113, 84,
        109, 110, 111, 86, 70, 117, 71, 109, 81, 65, 65, 65, 73, 65, 85, 107,
        119, 75, 97, 106, 110, 85, 107, 108, 78, 52, 67, 88, 47, 90, 87, 69,
        104, 73, 79, 90, 87, 67, 82, 97, 54, 90, 75, 67, 111, 115, 73, 67,
        66, 99, 47, 121, 67, 52, 82, 71, 74, 108, 100, 113, 115, 88, 97, 77,
        48, 80, 67, 109, 100, 121, 48, 100, 109, 99, 118, 48, 88, 117, 79, 90,
        52, 78, 98, 57, 50, 97, 105, 90, 110, 87, 100, 71, 82, 88, 84, 78,
        53, 75, 74, 52, 48, 53, 87, 86, 79, 81, 76, 66, 66, 51, 76, 112,
        101, 98, 65, 54, 113, 101, 121, 68, 71, 105, 86, 116, 47, 105, 80, 79,
        120, 103, 74, 71, 90, 121, 48, 76, 73, 87, 105, 113, 87, 117, 122, 65,
        65, 102, 49, 57, 57, 99, 81, 50, 66, 107, 87, 109, 71, 84, 67, 54,
        65, 65, 74, 105, 49, 100, 101, 112, 88, 67, 43, 75, 86, 65, 80, 43,
        72, 119, 88, 79, 76, 106, 83, 85, 65, 65, 65, 65, 73, 66, 101, 111,
        121, 75, 75, 69, 105, 86, 68, 81, 79, 112, 66, 48, 75, 65, 114, 53,
        82, 76, 72, 51, 47, 43, 53, 66, 80, 43, 109, 82, 43, 53, 50, 103,
        84, 65, 98, 120, 90, 98, 120, 81, 80, 89, 121, 72, 55, 109, 113, 113,
        50, 47, 80, 73, 49, 65, 48, 98, 106, 110, 118, 110, 76, 101, 52, 53,
        97, 47, 105, 86, 117, 109, 111, 73, 106, 80, 47, 97, 116, 77, 116, 86,
        74, 54, 119, 49, 48, 51, 69, 120, 53, 74, 71, 77, 119, 86, 66, 55,
        117, 90, 66, 87, 101, 121, 74, 52, 109, 51, 77, 113, 81, 100, 80, 102,
        49, 47, 43, 114, 43, 85, 67, 111, 79, 97, 76, 55, 115, 116, 75, 51,
        114, 87, 101, 84, 67, 121, 100, 98, 56, 50, 122, 51, 52, 101, 106, 88,
        48, 100, 90, 111, 98, 65, 80, 71, 106, 50, 70, 55, 83, 121, 90, 121,
        55, 106, 77, 56, 84, 72, 103, 99, 119, 61, 61,
      ])
    },
    what: 'DSA public key (OpenSSH format)'
  },
  { source: 'ssh-dss AAAAB3NzaC1kc3MAAACBAILCaN5QbaErJBa0nNoCY2QvvgcSd7WMgIgQ2qPAPH3Sav+9SWZB0SlHP8QYmV/ntBufaRw9ZdIop6esZBoXmNXDxjcfwpJAzrT3qOMfHXeNcc7w/6dHp+2DlkWD3yzK07SheIZnGNtF1fpfrUvkHFA8UX96awWRLnYuSvWvAK9dAAAAFQCrfeadO0tCCwgnq97Skk0Ng/xV6wAAAIBcVYaMkjXNgfvcUS4pwMabY8rFNGbYo9Jgv7IU2LGFEzD6yFbp2t1sQw031EYlad4n3BMTrZdE6K2/SMi3f3yRr0Z6S6nVSbo7/kIT8/tGJAj9/8RHdCihrFUMaAJNzdk5dmbyaCzexLmLXlKfUVW9A1WOsASjeJBwpFrJ8deuVAAAAIAu/zdW1qxdUVd0b7sjWFFZpp3l6HgOXlZpaOYchIoybs7n2dkYwmvn9glNcLCvRqiZpm2oEZIg1dy8RaH5b7AqY2huwmv+hLZ1XkUFyS2Ae2AKux5rslmiKmvglbDY0Rng1Wj3r3/N0KcKke1g1ohEUaQfXnz7VVx2YoVxhT1dTQ== testing ssh2 from node.js',
    expected: {
      fulltype: 'ssh-dss',
      type: 'dss',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 7, 115, 115, 104, 45, 100, 115, 115, 0, 0, 0, 129, 0, 130,
        194, 104, 222, 80, 109, 161, 43, 36, 22, 180, 156, 218, 2, 99, 100, 47,
        190, 7, 18, 119, 181, 140, 128, 136, 16, 218, 163, 192, 60, 125, 210, 106,
        255, 189, 73, 102, 65, 209, 41, 71, 63, 196, 24, 153, 95, 231, 180, 27,
        159, 105, 28, 61, 101, 210, 40, 167, 167, 172, 100, 26, 23, 152, 213, 195,
        198, 55, 31, 194, 146, 64, 206, 180, 247, 168, 227, 31, 29, 119, 141, 113,
        206, 240, 255, 167, 71, 167, 237, 131, 150, 69, 131, 223, 44, 202, 211, 180,
        161, 120, 134, 103, 24, 219, 69, 213, 250, 95, 173, 75, 228, 28, 80, 60,
        81, 127, 122, 107, 5, 145, 46, 118, 46, 74, 245, 175, 0, 175, 93, 0,
        0, 0, 21, 0, 171, 125, 230, 157, 59, 75, 66, 11, 8, 39, 171, 222,
        210, 146, 77, 13, 131, 252, 85, 235, 0, 0, 0, 128, 92, 85, 134, 140,
        146, 53, 205, 129, 251, 220, 81, 46, 41, 192, 198, 155, 99, 202, 197, 52,
        102, 216, 163, 210, 96, 191, 178, 20, 216, 177, 133, 19, 48, 250, 200, 86,
        233, 218, 221, 108, 67, 13, 55, 212, 70, 37, 105, 222, 39, 220, 19, 19,
        173, 151, 68, 232, 173, 191, 72, 200, 183, 127, 124, 145, 175, 70, 122, 75,
        169, 213, 73, 186, 59, 254, 66, 19, 243, 251, 70, 36, 8, 253, 255, 196,
        71, 116, 40, 161, 172, 85, 12, 104, 2, 77, 205, 217, 57, 118, 102, 242,
        104, 44, 222, 196, 185, 139, 94, 82, 159, 81, 85, 189, 3, 85, 142, 176,
        4, 163, 120, 144, 112, 164, 90, 201, 241, 215, 174, 84, 0, 0, 0, 128,
        46, 255, 55, 86, 214, 172, 93, 81, 87, 116, 111, 187, 35, 88, 81, 89,
        166, 157, 229, 232, 120, 14, 94, 86, 105, 104, 230, 28, 132, 138, 50, 110,
        206, 231, 217, 217, 24, 194, 107, 231, 246, 9, 77, 112, 176, 175, 70, 168,
        153, 166, 109, 168, 17, 146, 32, 213, 220, 188, 69, 161, 249, 111, 176, 42,
        99, 104, 110, 194, 107, 254, 132, 182, 117, 94, 69, 5, 201, 45, 128, 123,
        96, 10, 187, 30, 107, 178, 89, 162, 42, 107, 224, 149, 176, 216, 209, 25,
        224, 213, 104, 247, 175, 127, 205, 208, 167, 10, 145, 237, 96, 214, 136, 68,
        81, 164, 31, 94, 124, 251, 85, 92, 118, 98, 133, 113, 133, 61, 93, 77,
        11, 94, 178, 216, 167, 130, 203, 33, 217, 250, 232, 154, 122, 29, 122, 59,
      ]),
      publicOrig: new Buffer([
        115, 115, 104, 45, 100, 115, 115, 32, 65, 65, 65, 65, 66, 51, 78, 122, 97,
        67, 49, 107, 99, 51, 77, 65, 65, 65, 67, 66, 65, 73, 76, 67, 97,
        78, 53, 81, 98, 97, 69, 114, 74, 66, 97, 48, 110, 78, 111, 67, 89,
        50, 81, 118, 118, 103, 99, 83, 100, 55, 87, 77, 103, 73, 103, 81, 50,
        113, 80, 65, 80, 72, 51, 83, 97, 118, 43, 57, 83, 87, 90, 66, 48,
        83, 108, 72, 80, 56, 81, 89, 109, 86, 47, 110, 116, 66, 117, 102, 97,
        82, 119, 57, 90, 100, 73, 111, 112, 54, 101, 115, 90, 66, 111, 88, 109,
        78, 88, 68, 120, 106, 99, 102, 119, 112, 74, 65, 122, 114, 84, 51, 113,
        79, 77, 102, 72, 88, 101, 78, 99, 99, 55, 119, 47, 54, 100, 72, 112,
        43, 50, 68, 108, 107, 87, 68, 51, 121, 122, 75, 48, 55, 83, 104, 101,
        73, 90, 110, 71, 78, 116, 70, 49, 102, 112, 102, 114, 85, 118, 107, 72,
        70, 65, 56, 85, 88, 57, 54, 97, 119, 87, 82, 76, 110, 89, 117, 83,
        118, 87, 118, 65, 75, 57, 100, 65, 65, 65, 65, 70, 81, 67, 114, 102,
        101, 97, 100, 79, 48, 116, 67, 67, 119, 103, 110, 113, 57, 55, 83, 107,
        107, 48, 78, 103, 47, 120, 86, 54, 119, 65, 65, 65, 73, 66, 99, 86,
        89, 97, 77, 107, 106, 88, 78, 103, 102, 118, 99, 85, 83, 52, 112, 119,
        77, 97, 98, 89, 56, 114, 70, 78, 71, 98, 89, 111, 57, 74, 103, 118,
        55, 73, 85, 50, 76, 71, 70, 69, 122, 68, 54, 121, 70, 98, 112, 50,
        116, 49, 115, 81, 119, 48, 51, 49, 69, 89, 108, 97, 100, 52, 110, 51,
        66, 77, 84, 114, 90, 100, 69, 54, 75, 50, 47, 83, 77, 105, 51, 102,
        51, 121, 82, 114, 48, 90, 54, 83, 54, 110, 86, 83, 98, 111, 55, 47,
        107, 73, 84, 56, 47, 116, 71, 74, 65, 106, 57, 47, 56, 82, 72, 100,
        67, 105, 104, 114, 70, 85, 77, 97, 65, 74, 78, 122, 100, 107, 53, 100,
        109, 98, 121, 97, 67, 122, 101, 120, 76, 109, 76, 88, 108, 75, 102, 85,
        86, 87, 57, 65, 49, 87, 79, 115, 65, 83, 106, 101, 74, 66, 119, 112,
        70, 114, 74, 56, 100, 101, 117, 86, 65, 65, 65, 65, 73, 65, 117, 47,
        122, 100, 87, 49, 113, 120, 100, 85, 86, 100, 48, 98, 55, 115, 106, 87,
        70, 70, 90, 112, 112, 51, 108, 54, 72, 103, 79, 88, 108, 90, 112, 97,
        79, 89, 99, 104, 73, 111, 121, 98, 115, 55, 110, 50, 100, 107, 89, 119,
        109, 118, 110, 57, 103, 108, 78, 99, 76, 67, 118, 82, 113, 105, 90, 112,
        109, 50, 111, 69, 90, 73, 103, 49, 100, 121, 56, 82, 97, 72, 53, 98,
        55, 65, 113, 89, 50, 104, 117, 119, 109, 118, 43, 104, 76, 90, 49, 88,
        107, 85, 70, 121, 83, 50, 65, 101, 50, 65, 75, 117, 120, 53, 114, 115,
        108, 109, 105, 75, 109, 118, 103, 108, 98, 68, 89, 48, 82, 110, 103, 49,
        87, 106, 51, 114, 51, 47, 78, 48, 75, 99, 75, 107, 101, 49, 103, 49,
        111, 104, 69, 85, 97, 81, 102, 88, 110, 122, 55, 86, 86, 120, 50, 89,
        111, 86, 120, 104, 84, 49, 100, 84, 81, 61, 61, 32, 116, 101, 115, 116,
        105, 110, 103, 32, 115, 115, 104, 50, 32, 102, 114, 111, 109, 32, 110, 111,
        100, 101, 46, 106, 115,
      ])
    },
    what: 'DSA public key (OpenSSH format) with comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: "testing ssh2 from node.js"',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 3, 1, 0, 1, 0, 0, 0, 97, 0, 229, 233, 220, 75, 250,
        24, 49, 192, 99, 97, 102, 186, 181, 242, 240, 43, 108, 208, 116, 82, 163,
        113, 189, 201, 95, 250, 135, 34, 68, 185, 103, 71, 147, 186, 175, 247, 51,
        2, 28, 146, 59, 231, 142, 115, 148, 133, 239, 7, 177, 103, 53, 168, 252,
        35, 144, 223, 187, 68, 139, 218, 222, 78, 231, 24, 20, 212, 16, 88, 91,
        241, 224, 79, 2, 68, 83, 254, 80, 244, 122, 133, 5, 244, 127, 26, 11,
        72, 93, 229, 197, 1, 73, 251, 125, 191, 173, 45,
      ]),
      publicOrig: new Buffer([
        45, 45, 45, 45, 32, 66, 69, 71, 73, 78, 32, 83, 83, 72, 50, 32, 80,
        85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45, 10, 67,
        111, 109, 109, 101, 110, 116, 58, 32, 34, 116, 101, 115, 116, 105, 110, 103,
        32, 115, 115, 104, 50, 32, 102, 114, 111, 109, 32, 110, 111, 100, 101, 46,
        106, 115, 34, 10, 65, 65, 65, 65, 66, 51, 78, 122, 97, 67, 49, 121,
        99, 50, 69, 65, 65, 65, 65, 68, 65, 81, 65, 66, 65, 65, 65, 65,
        89, 81, 68, 108, 54, 100, 120, 76, 43, 104, 103, 120, 119, 71, 78, 104,
        90, 114, 113, 49, 56, 118, 65, 114, 98, 78, 66, 48, 85, 113, 78, 120,
        118, 99, 108, 102, 43, 111, 99, 105, 82, 76, 10, 108, 110, 82, 53, 79,
        54, 114, 47, 99, 122, 65, 104, 121, 83, 79, 43, 101, 79, 99, 53, 83,
        70, 55, 119, 101, 120, 90, 122, 87, 111, 47, 67, 79, 81, 51, 55, 116,
        69, 105, 57, 114, 101, 84, 117, 99, 89, 70, 78, 81, 81, 87, 70, 118,
        120, 52, 69, 56, 67, 82, 70, 80, 43, 85, 80, 82, 54, 104, 81, 88,
        48, 10, 102, 120, 111, 76, 83, 70, 51, 108, 120, 81, 70, 74, 43, 51,
        50, 47, 114, 83, 48, 61, 10, 45, 45, 45, 45, 32, 69, 78, 68, 32,
        83, 83, 72, 50, 32, 80, 85, 66, 76, 73, 67, 32, 75, 69, 89, 32,
        45, 45, 45, 45,
      ])
    },
    what: 'RSA public key (RFC4716 format) with quoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: testing ssh2 from node.js',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 3, 1, 0, 1, 0, 0, 0, 97, 0, 229, 233, 220, 75, 250,
        24, 49, 192, 99, 97, 102, 186, 181, 242, 240, 43, 108, 208, 116, 82, 163,
        113, 189, 201, 95, 250, 135, 34, 68, 185, 103, 71, 147, 186, 175, 247, 51,
        2, 28, 146, 59, 231, 142, 115, 148, 133, 239, 7, 177, 103, 53, 168, 252,
        35, 144, 223, 187, 68, 139, 218, 222, 78, 231, 24, 20, 212, 16, 88, 91,
        241, 224, 79, 2, 68, 83, 254, 80, 244, 122, 133, 5, 244, 127, 26, 11,
        72, 93, 229, 197, 1, 73, 251, 125, 191, 173, 45,
      ]),
      publicOrig: new Buffer([
        45, 45, 45, 45, 32, 66, 69, 71, 73, 78, 32, 83, 83, 72, 50, 32, 80,
        85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45, 10, 67,
        111, 109, 109, 101, 110, 116, 58, 32, 116, 101, 115, 116, 105, 110, 103, 32,
        115, 115, 104, 50, 32, 102, 114, 111, 109, 32, 110, 111, 100, 101, 46, 106,
        115, 10, 65, 65, 65, 65, 66, 51, 78, 122, 97, 67, 49, 121, 99, 50,
        69, 65, 65, 65, 65, 68, 65, 81, 65, 66, 65, 65, 65, 65, 89, 81,
        68, 108, 54, 100, 120, 76, 43, 104, 103, 120, 119, 71, 78, 104, 90, 114,
        113, 49, 56, 118, 65, 114, 98, 78, 66, 48, 85, 113, 78, 120, 118, 99,
        108, 102, 43, 111, 99, 105, 82, 76, 10, 108, 110, 82, 53, 79, 54, 114,
        47, 99, 122, 65, 104, 121, 83, 79, 43, 101, 79, 99, 53, 83, 70, 55,
        119, 101, 120, 90, 122, 87, 111, 47, 67, 79, 81, 51, 55, 116, 69, 105,
        57, 114, 101, 84, 117, 99, 89, 70, 78, 81, 81, 87, 70, 118, 120, 52,
        69, 56, 67, 82, 70, 80, 43, 85, 80, 82, 54, 104, 81, 88, 48, 10,
        102, 120, 111, 76, 83, 70, 51, 108, 120, 81, 70, 74, 43, 51, 50, 47,
        114, 83, 48, 61, 10, 45, 45, 45, 45, 32, 69, 78, 68, 32, 83, 83,
        72, 50, 32, 80, 85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45,
        45, 45,
      ])
    },
    what: 'RSA public key (RFC4716 format) with unquoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: "testing ssh2 \\',
      'from node.js"',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 3, 1, 0, 1, 0, 0, 0, 97, 0, 229, 233, 220, 75, 250,
        24, 49, 192, 99, 97, 102, 186, 181, 242, 240, 43, 108, 208, 116, 82, 163,
        113, 189, 201, 95, 250, 135, 34, 68, 185, 103, 71, 147, 186, 175, 247, 51,
        2, 28, 146, 59, 231, 142, 115, 148, 133, 239, 7, 177, 103, 53, 168, 252,
        35, 144, 223, 187, 68, 139, 218, 222, 78, 231, 24, 20, 212, 16, 88, 91,
        241, 224, 79, 2, 68, 83, 254, 80, 244, 122, 133, 5, 244, 127, 26, 11,
        72, 93, 229, 197, 1, 73, 251, 125, 191, 173, 45,
      ]),
      publicOrig: new Buffer([
        45, 45, 45, 45, 32, 66, 69, 71, 73, 78, 32, 83, 83, 72, 50, 32, 80,
        85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45, 10, 67,
        111, 109, 109, 101, 110, 116, 58, 32, 34, 116, 101, 115, 116, 105, 110, 103,
        32, 115, 115, 104, 50, 32, 92, 10, 102, 114, 111, 109, 32, 110, 111, 100,
        101, 46, 106, 115, 34, 10, 65, 65, 65, 65, 66, 51, 78, 122, 97, 67,
        49, 121, 99, 50, 69, 65, 65, 65, 65, 68, 65, 81, 65, 66, 65, 65,
        65, 65, 89, 81, 68, 108, 54, 100, 120, 76, 43, 104, 103, 120, 119, 71,
        78, 104, 90, 114, 113, 49, 56, 118, 65, 114, 98, 78, 66, 48, 85, 113,
        78, 120, 118, 99, 108, 102, 43, 111, 99, 105, 82, 76, 10, 108, 110, 82,
        53, 79, 54, 114, 47, 99, 122, 65, 104, 121, 83, 79, 43, 101, 79, 99,
        53, 83, 70, 55, 119, 101, 120, 90, 122, 87, 111, 47, 67, 79, 81, 51,
        55, 116, 69, 105, 57, 114, 101, 84, 117, 99, 89, 70, 78, 81, 81, 87,
        70, 118, 120, 52, 69, 56, 67, 82, 70, 80, 43, 85, 80, 82, 54, 104,
        81, 88, 48, 10, 102, 120, 111, 76, 83, 70, 51, 108, 120, 81, 70, 74,
        43, 51, 50, 47, 114, 83, 48, 61, 10, 45, 45, 45, 45, 32, 69, 78,
        68, 32, 83, 83, 72, 50, 32, 80, 85, 66, 76, 73, 67, 32, 75, 69,
        89, 32, 45, 45, 45, 45,
      ])
    },
    what: 'RSA public key (RFC4716 format) with multi-line quoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: testing ssh2 \\',
      'from node.js',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 3, 1, 0, 1, 0, 0, 0, 97, 0, 229, 233, 220, 75, 250,
        24, 49, 192, 99, 97, 102, 186, 181, 242, 240, 43, 108, 208, 116, 82, 163,
        113, 189, 201, 95, 250, 135, 34, 68, 185, 103, 71, 147, 186, 175, 247, 51,
        2, 28, 146, 59, 231, 142, 115, 148, 133, 239, 7, 177, 103, 53, 168, 252,
        35, 144, 223, 187, 68, 139, 218, 222, 78, 231, 24, 20, 212, 16, 88, 91,
        241, 224, 79, 2, 68, 83, 254, 80, 244, 122, 133, 5, 244, 127, 26, 11,
        72, 93, 229, 197, 1, 73, 251, 125, 191, 173, 45,
      ]),
      publicOrig: new Buffer([
        45, 45, 45, 45, 32, 66, 69, 71, 73, 78, 32, 83, 83, 72, 50, 32, 80,
        85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45, 10, 67,
        111, 109, 109, 101, 110, 116, 58, 32, 116, 101, 115, 116, 105, 110, 103, 32,
        115, 115, 104, 50, 32, 92, 10, 102, 114, 111, 109, 32, 110, 111, 100, 101,
        46, 106, 115, 10, 65, 65, 65, 65, 66, 51, 78, 122, 97, 67, 49, 121,
        99, 50, 69, 65, 65, 65, 65, 68, 65, 81, 65, 66, 65, 65, 65, 65,
        89, 81, 68, 108, 54, 100, 120, 76, 43, 104, 103, 120, 119, 71, 78, 104,
        90, 114, 113, 49, 56, 118, 65, 114, 98, 78, 66, 48, 85, 113, 78, 120,
        118, 99, 108, 102, 43, 111, 99, 105, 82, 76, 10, 108, 110, 82, 53, 79,
        54, 114, 47, 99, 122, 65, 104, 121, 83, 79, 43, 101, 79, 99, 53, 83,
        70, 55, 119, 101, 120, 90, 122, 87, 111, 47, 67, 79, 81, 51, 55, 116,
        69, 105, 57, 114, 101, 84, 117, 99, 89, 70, 78, 81, 81, 87, 70, 118,
        120, 52, 69, 56, 67, 82, 70, 80, 43, 85, 80, 82, 54, 104, 81, 88,
        48, 10, 102, 120, 111, 76, 83, 70, 51, 108, 120, 81, 70, 74, 43, 51,
        50, 47, 114, 83, 48, 61, 10, 45, 45, 45, 45, 32, 69, 78, 68, 32,
        83, 83, 72, 50, 32, 80, 85, 66, 76, 73, 67, 32, 75, 69, 89, 32,
        45, 45, 45, 45,
      ])
    },
    what: 'RSA public key (RFC4716 format) with multi-line unquoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: ""',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: '',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 3, 1, 0, 1, 0, 0, 0, 97, 0, 229, 233, 220, 75, 250,
        24, 49, 192, 99, 97, 102, 186, 181, 242, 240, 43, 108, 208, 116, 82, 163,
        113, 189, 201, 95, 250, 135, 34, 68, 185, 103, 71, 147, 186, 175, 247, 51,
        2, 28, 146, 59, 231, 142, 115, 148, 133, 239, 7, 177, 103, 53, 168, 252,
        35, 144, 223, 187, 68, 139, 218, 222, 78, 231, 24, 20, 212, 16, 88, 91,
        241, 224, 79, 2, 68, 83, 254, 80, 244, 122, 133, 5, 244, 127, 26, 11,
        72, 93, 229, 197, 1, 73, 251, 125, 191, 173, 45,
      ]),
      publicOrig: new Buffer([
        45, 45, 45, 45, 32, 66, 69, 71, 73, 78, 32, 83, 83, 72, 50, 32, 80,
        85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45, 10, 67,
        111, 109, 109, 101, 110, 116, 58, 32, 34, 34, 10, 65, 65, 65, 65, 66,
        51, 78, 122, 97, 67, 49, 121, 99, 50, 69, 65, 65, 65, 65, 68, 65,
        81, 65, 66, 65, 65, 65, 65, 89, 81, 68, 108, 54, 100, 120, 76, 43,
        104, 103, 120, 119, 71, 78, 104, 90, 114, 113, 49, 56, 118, 65, 114, 98,
        78, 66, 48, 85, 113, 78, 120, 118, 99, 108, 102, 43, 111, 99, 105, 82,
        76, 10, 108, 110, 82, 53, 79, 54, 114, 47, 99, 122, 65, 104, 121, 83,
        79, 43, 101, 79, 99, 53, 83, 70, 55, 119, 101, 120, 90, 122, 87, 111,
        47, 67, 79, 81, 51, 55, 116, 69, 105, 57, 114, 101, 84, 117, 99, 89,
        70, 78, 81, 81, 87, 70, 118, 120, 52, 69, 56, 67, 82, 70, 80, 43,
        85, 80, 82, 54, 104, 81, 88, 48, 10, 102, 120, 111, 76, 83, 70, 51,
        108, 120, 81, 70, 74, 43, 51, 50, 47, 114, 83, 48, 61, 10, 45, 45,
        45, 45, 32, 69, 78, 68, 32, 83, 83, 72, 50, 32, 80, 85, 66, 76,
        73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45,
      ])
    },
    what: 'RSA public key (RFC4716 format) with empty quoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: ',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: '',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 3, 1, 0, 1, 0, 0, 0, 97, 0, 229, 233, 220, 75, 250,
        24, 49, 192, 99, 97, 102, 186, 181, 242, 240, 43, 108, 208, 116, 82, 163,
        113, 189, 201, 95, 250, 135, 34, 68, 185, 103, 71, 147, 186, 175, 247, 51,
        2, 28, 146, 59, 231, 142, 115, 148, 133, 239, 7, 177, 103, 53, 168, 252,
        35, 144, 223, 187, 68, 139, 218, 222, 78, 231, 24, 20, 212, 16, 88, 91,
        241, 224, 79, 2, 68, 83, 254, 80, 244, 122, 133, 5, 244, 127, 26, 11,
        72, 93, 229, 197, 1, 73, 251, 125, 191, 173, 45,
      ]),
      publicOrig: new Buffer([
        45, 45, 45, 45, 32, 66, 69, 71, 73, 78, 32, 83, 83, 72, 50, 32, 80,
        85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45, 10, 67,
        111, 109, 109, 101, 110, 116, 58, 32, 10, 65, 65, 65, 65, 66, 51, 78,
        122, 97, 67, 49, 121, 99, 50, 69, 65, 65, 65, 65, 68, 65, 81, 65,
        66, 65, 65, 65, 65, 89, 81, 68, 108, 54, 100, 120, 76, 43, 104, 103,
        120, 119, 71, 78, 104, 90, 114, 113, 49, 56, 118, 65, 114, 98, 78, 66,
        48, 85, 113, 78, 120, 118, 99, 108, 102, 43, 111, 99, 105, 82, 76, 10,
        108, 110, 82, 53, 79, 54, 114, 47, 99, 122, 65, 104, 121, 83, 79, 43,
        101, 79, 99, 53, 83, 70, 55, 119, 101, 120, 90, 122, 87, 111, 47, 67,
        79, 81, 51, 55, 116, 69, 105, 57, 114, 101, 84, 117, 99, 89, 70, 78,
        81, 81, 87, 70, 118, 120, 52, 69, 56, 67, 82, 70, 80, 43, 85, 80,
        82, 54, 104, 81, 88, 48, 10, 102, 120, 111, 76, 83, 70, 51, 108, 120,
        81, 70, 74, 43, 51, 50, 47, 114, 83, 48, 61, 10, 45, 45, 45, 45,
        32, 69, 78, 68, 32, 83, 83, 72, 50, 32, 80, 85, 66, 76, 73, 67,
        32, 75, 69, 89, 32, 45, 45, 45, 45,
      ])
    },
    what: 'RSA public key (RFC4716 format) with empty unquoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 3, 1, 0, 1, 0, 0, 0, 97, 0, 229, 233, 220, 75, 250,
        24, 49, 192, 99, 97, 102, 186, 181, 242, 240, 43, 108, 208, 116, 82, 163,
        113, 189, 201, 95, 250, 135, 34, 68, 185, 103, 71, 147, 186, 175, 247, 51,
        2, 28, 146, 59, 231, 142, 115, 148, 133, 239, 7, 177, 103, 53, 168, 252,
        35, 144, 223, 187, 68, 139, 218, 222, 78, 231, 24, 20, 212, 16, 88, 91,
        241, 224, 79, 2, 68, 83, 254, 80, 244, 122, 133, 5, 244, 127, 26, 11,
        72, 93, 229, 197, 1, 73, 251, 125, 191, 173, 45,
      ]),
      publicOrig: new Buffer([
        45, 45, 45, 45, 32, 66, 69, 71, 73, 78, 32, 83, 83, 72, 50, 32, 80,
        85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45, 10, 65,
        65, 65, 65, 66, 51, 78, 122, 97, 67, 49, 121, 99, 50, 69, 65, 65,
        65, 65, 68, 65, 81, 65, 66, 65, 65, 65, 65, 89, 81, 68, 108, 54,
        100, 120, 76, 43, 104, 103, 120, 119, 71, 78, 104, 90, 114, 113, 49, 56,
        118, 65, 114, 98, 78, 66, 48, 85, 113, 78, 120, 118, 99, 108, 102, 43,
        111, 99, 105, 82, 76, 10, 108, 110, 82, 53, 79, 54, 114, 47, 99, 122,
        65, 104, 121, 83, 79, 43, 101, 79, 99, 53, 83, 70, 55, 119, 101, 120,
        90, 122, 87, 111, 47, 67, 79, 81, 51, 55, 116, 69, 105, 57, 114, 101,
        84, 117, 99, 89, 70, 78, 81, 81, 87, 70, 118, 120, 52, 69, 56, 67,
        82, 70, 80, 43, 85, 80, 82, 54, 104, 81, 88, 48, 10, 102, 120, 111,
        76, 83, 70, 51, 108, 120, 81, 70, 74, 43, 51, 50, 47, 114, 83, 48,
        61, 10, 45, 45, 45, 45, 32, 69, 78, 68, 32, 83, 83, 72, 50, 32,
        80, 85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45,
      ])
    },
    what: 'RSA public key (RFC4716 format) with no comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      '',
      '',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 3, 1, 0, 1, 0, 0, 0, 97, 0, 229, 233, 220, 75, 250,
        24, 49, 192, 99, 97, 102, 186, 181, 242, 240, 43, 108, 208, 116, 82, 163,
        113, 189, 201, 95, 250, 135, 34, 68, 185, 103, 71, 147, 186, 175, 247, 51,
        2, 28, 146, 59, 231, 142, 115, 148, 133, 239, 7, 177, 103, 53, 168, 252,
        35, 144, 223, 187, 68, 139, 218, 222, 78, 231, 24, 20, 212, 16, 88, 91,
        241, 224, 79, 2, 68, 83, 254, 80, 244, 122, 133, 5, 244, 127, 26, 11,
        72, 93, 229, 197, 1, 73, 251, 125, 191, 173, 45,
      ]),
      publicOrig: new Buffer([
        45, 45, 45, 45, 32, 66, 69, 71, 73, 78, 32, 83, 83, 72, 50, 32, 80,
        85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45, 10, 10,
        10, 65, 65, 65, 65, 66, 51, 78, 122, 97, 67, 49, 121, 99, 50, 69,
        65, 65, 65, 65, 68, 65, 81, 65, 66, 65, 65, 65, 65, 89, 81, 68,
        108, 54, 100, 120, 76, 43, 104, 103, 120, 119, 71, 78, 104, 90, 114, 113,
        49, 56, 118, 65, 114, 98, 78, 66, 48, 85, 113, 78, 120, 118, 99, 108,
        102, 43, 111, 99, 105, 82, 76, 10, 108, 110, 82, 53, 79, 54, 114, 47,
        99, 122, 65, 104, 121, 83, 79, 43, 101, 79, 99, 53, 83, 70, 55, 119,
        101, 120, 90, 122, 87, 111, 47, 67, 79, 81, 51, 55, 116, 69, 105, 57,
        114, 101, 84, 117, 99, 89, 70, 78, 81, 81, 87, 70, 118, 120, 52, 69,
        56, 67, 82, 70, 80, 43, 85, 80, 82, 54, 104, 81, 88, 48, 10, 102,
        120, 111, 76, 83, 70, 51, 108, 120, 81, 70, 74, 43, 51, 50, 47, 114,
        83, 48, 61, 10, 45, 45, 45, 45, 32, 69, 78, 68, 32, 83, 83, 72,
        50, 32, 80, 85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45,
        45,
      ])
    },
    what: 'RSA public key (RFC4716 format) with blank lines'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'AAAAB3NzaC1kc3MAAACBAI4CR+tgz/kZnbmKJjccZNWKqcifcxLy/pTMnRLh5SrzQ44R8T',
      'QSbQTNnJHVpc6ucW369+TtQRpRmr4iU65ttZkZqCJ1jawXAnxcpIrVq/xBfsXfCIr26LaG',
      'qNm//vYBAPs5j648Fhg6AHyM/LewPSnek3fE8gIB9gtCsPei5L7lAAAAFQDJ3k6TdJK/eo',
      'LgUqg2bUB94Mjg9wAAAIAp9Q3SyooklLGMPdG2kj0vMWF+cVtChcjGjemhPeHXyjD5/in8',
      'gWYFifVqoYKBJhAaPP5HpPtA3BEGgIkLVGRzCk3tnwXbutGbwt7PzutbJ9LKiiR7z1HLqn',
      'E+r0//6Nhqz/ZNQPWZRbcG0l1EIQNgjuZkSzJLLO15TSPZeVhZcQAAAIAUAYobyKQ3Rl4q',
      'XAcOP8uOt5q7YHMy43JmZJnTCQ7X95TuNv5kFj7a5RkDU9HjgzSg5aabVsJAD6wDy5/rm+',
      'kJA9uXGOvHfiToh+tRjDp2SrWDERfPCRDl8oTJr36p1zqFXERwTJsNJF32QClszhzk7NYv',
      'G6kcaQC8Hhm9yccRoA==',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-dss',
      type: 'dss',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0, 0, 0, 129, 0, 142, 2, 71, 235, 96, 207, 249, 25, 157, 185, 138, 38,
        55, 28, 100, 213, 138, 169, 200, 159, 115, 18, 242, 254, 148, 204, 157, 18,
        225, 229, 42, 243, 67, 142, 17, 241, 52, 18, 109, 4, 205, 156, 145, 213,
        165, 206, 174, 113, 109, 250, 247, 228, 237, 65, 26, 81, 154, 190, 34, 83,
        174, 109, 181, 153, 25, 168, 34, 117, 141, 172, 23, 2, 124, 92, 164, 138,
        213, 171, 252, 65, 126, 197, 223, 8, 138, 246, 232, 182, 134, 168, 217, 191,
        254, 246, 1, 0, 251, 57, 143, 174, 60, 22, 24, 58, 0, 124, 140, 252,
        183, 176, 61, 41, 222, 147, 119, 196, 242, 2, 1, 246, 11, 66, 176, 247,
        162, 228, 190, 229, 0, 0, 0, 21, 0, 201, 222, 78, 147, 116, 146, 191,
        122, 130, 224, 82, 168, 54, 109, 64, 125, 224, 200, 224, 247, 0, 0, 0,
        128, 41, 245, 13, 210, 202, 138, 36, 148, 177, 140, 61, 209, 182, 146, 61,
        47, 49, 97, 126, 113, 91, 66, 133, 200, 198, 141, 233, 161, 61, 225, 215,
        202, 48, 249, 254, 41, 252, 129, 102, 5, 137, 245, 106, 161, 130, 129, 38,
        16, 26, 60, 254, 71, 164, 251, 64, 220, 17, 6, 128, 137, 11, 84, 100,
        115, 10, 77, 237, 159, 5, 219, 186, 209, 155, 194, 222, 207, 206, 235, 91,
        39, 210, 202, 138, 36, 123, 207, 81, 203, 170, 113, 62, 175, 79, 255, 232,
        216, 106, 207, 246, 77, 64, 245, 153, 69, 183, 6, 210, 93, 68, 33, 3,
        96, 142, 230, 100, 75, 50, 75, 44, 237, 121, 77, 35, 217, 121, 88, 89,
        113, 0, 0, 0, 128, 20, 1, 138, 27, 200, 164, 55, 70, 94, 42, 92,
        7, 14, 63, 203, 142, 183, 154, 187, 96, 115, 50, 227, 114, 102, 100, 153,
        211, 9, 14, 215, 247, 148, 238, 54, 254, 100, 22, 62, 218, 229, 25, 3,
        83, 209, 227, 131, 52, 160, 229, 166, 155, 86, 194, 64, 15, 172, 3, 203,
        159, 235, 155, 233, 9, 3, 219, 151, 24, 235, 199, 126, 36, 232, 135, 235,
        81, 140, 58, 118, 74, 181, 131, 17, 23, 207, 9, 16, 229, 242, 132, 201,
        175, 126, 169, 215, 58, 133, 92, 68, 112, 76, 155, 13, 36, 93, 246, 64,
        41, 108, 206, 28, 228, 236, 214, 47, 27, 169, 28, 105, 0, 188, 30, 25,
        189, 201, 199, 17, 160,
      ]),
      publicOrig: new Buffer([
        45, 45, 45, 45, 32, 66, 69, 71, 73, 78, 32, 83, 83, 72, 50, 32, 80,
        85, 66, 76, 73, 67, 32, 75, 69, 89, 32, 45, 45, 45, 45, 10, 65,
        65, 65, 65, 66, 51, 78, 122, 97, 67, 49, 107, 99, 51, 77, 65, 65,
        65, 67, 66, 65, 73, 52, 67, 82, 43, 116, 103, 122, 47, 107, 90, 110,
        98, 109, 75, 74, 106, 99, 99, 90, 78, 87, 75, 113, 99, 105, 102, 99,
        120, 76, 121, 47, 112, 84, 77, 110, 82, 76, 104, 53, 83, 114, 122, 81,
        52, 52, 82, 56, 84, 10, 81, 83, 98, 81, 84, 78, 110, 74, 72, 86,
        112, 99, 54, 117, 99, 87, 51, 54, 57, 43, 84, 116, 81, 82, 112, 82,
        109, 114, 52, 105, 85, 54, 53, 116, 116, 90, 107, 90, 113, 67, 74, 49,
        106, 97, 119, 88, 65, 110, 120, 99, 112, 73, 114, 86, 113, 47, 120, 66,
        102, 115, 88, 102, 67, 73, 114, 50, 54, 76, 97, 71, 10, 113, 78, 109,
        47, 47, 118, 89, 66, 65, 80, 115, 53, 106, 54, 52, 56, 70, 104, 103,
        54, 65, 72, 121, 77, 47, 76, 101, 119, 80, 83, 110, 101, 107, 51, 102,
        69, 56, 103, 73, 66, 57, 103, 116, 67, 115, 80, 101, 105, 53, 76, 55,
        108, 65, 65, 65, 65, 70, 81, 68, 74, 51, 107, 54, 84, 100, 74, 75,
        47, 101, 111, 10, 76, 103, 85, 113, 103, 50, 98, 85, 66, 57, 52, 77,
        106, 103, 57, 119, 65, 65, 65, 73, 65, 112, 57, 81, 51, 83, 121, 111,
        111, 107, 108, 76, 71, 77, 80, 100, 71, 50, 107, 106, 48, 118, 77, 87,
        70, 43, 99, 86, 116, 67, 104, 99, 106, 71, 106, 101, 109, 104, 80, 101,
        72, 88, 121, 106, 68, 53, 47, 105, 110, 56, 10, 103, 87, 89, 70, 105,
        102, 86, 113, 111, 89, 75, 66, 74, 104, 65, 97, 80, 80, 53, 72, 112,
        80, 116, 65, 51, 66, 69, 71, 103, 73, 107, 76, 86, 71, 82, 122, 67,
        107, 51, 116, 110, 119, 88, 98, 117, 116, 71, 98, 119, 116, 55, 80, 122,
        117, 116, 98, 74, 57, 76, 75, 105, 105, 82, 55, 122, 49, 72, 76, 113,
        110, 10, 69, 43, 114, 48, 47, 47, 54, 78, 104, 113, 122, 47, 90, 78,
        81, 80, 87, 90, 82, 98, 99, 71, 48, 108, 49, 69, 73, 81, 78, 103,
        106, 117, 90, 107, 83, 122, 74, 76, 76, 79, 49, 53, 84, 83, 80, 90,
        101, 86, 104, 90, 99, 81, 65, 65, 65, 73, 65, 85, 65, 89, 111, 98,
        121, 75, 81, 51, 82, 108, 52, 113, 10, 88, 65, 99, 79, 80, 56, 117,
        79, 116, 53, 113, 55, 89, 72, 77, 121, 52, 51, 74, 109, 90, 74, 110,
        84, 67, 81, 55, 88, 57, 53, 84, 117, 78, 118, 53, 107, 70, 106, 55,
        97, 53, 82, 107, 68, 85, 57, 72, 106, 103, 122, 83, 103, 53, 97, 97,
        98, 86, 115, 74, 65, 68, 54, 119, 68, 121, 53, 47, 114, 109, 43, 10,
        107, 74, 65, 57, 117, 88, 71, 79, 118, 72, 102, 105, 84, 111, 104, 43,
        116, 82, 106, 68, 112, 50, 83, 114, 87, 68, 69, 82, 102, 80, 67, 82,
        68, 108, 56, 111, 84, 74, 114, 51, 54, 112, 49, 122, 113, 70, 88, 69,
        82, 119, 84, 74, 115, 78, 74, 70, 51, 50, 81, 67, 108, 115, 122, 104,
        122, 107, 55, 78, 89, 118, 10, 71, 54, 107, 99, 97, 81, 67, 56, 72,
        104, 109, 57, 121, 99, 99, 82, 111, 65, 61, 61, 10, 45, 45, 45, 45,
        32, 69, 78, 68, 32, 83, 83, 72, 50, 32, 80, 85, 66, 76, 73, 67,
        32, 75, 69, 89, 32, 45, 45, 45, 45,
      ])
    },
    what: 'DSA public key (RFC4716 format) with no comment'
  },
].forEach(function(v) {
  require('buffer').INSPECT_MAX_BYTES = Infinity;
  var result = parser(v.source);
  assert.deepEqual(result,
                   v.expected,
                   '[' + group + v.what + ']: parser output mismatch.'
                   + '\nSaw: ' + inspect(result)
                   + '\nExpected: ' + inspect(v.expected));
});
