--[[ 

	Martin Karer / Sezz, 2016
	Wardrobe BoE mailer
	
--]]

if (IsTrialAccount() or IsRestrictedAccount()) then return; end -- Mail is disabled for Trial accounts.

local addonName, ns = ...;
local S, C;
if (SezzUI) then
	S, C = unpack(SezzUI);
else
	S, C = ns.S, ns.C;
end

if (addonName == "WardrobeTools" and SezzUI) then return; end

-----------------------------------------------------------------------------

local addon = S:CreateModule("AppearanceMailer"):AddDefaultHandlers();
local tooltip = S.ScanningTooltip;

-- Lua API
local select, strlen, tinsert, tremove, strlower, next = select, string.len, table.insert, table.remove, string.lower, next;

-- WoW API
local GetItemClassInfo, GetItemSubClassInfo, GameTooltip_Hide, ClearCursor, PickupContainerItem, ClickSendMailItemButton, SendMail, GetItemInfo, C_TransmogCollection = GetItemClassInfo, GetItemSubClassInfo, GameTooltip_Hide, ClearCursor, PickupContainerItem, ClickSendMailItemButton, SendMail, GetItemInfo, C_TransmogCollection;
local ATTACHMENTS_MAX_SEND, LE_ITEM_CLASS_ARMOR, LE_ITEM_ARMOR_CLOTH, LE_ITEM_ARMOR_LEATHER, LE_ITEM_ARMOR_MAIL, LE_ITEM_ARMOR_PLATE, LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_AXE1H, LE_ITEM_WEAPON_AXE2H, LE_ITEM_WEAPON_MACE1H, LE_ITEM_WEAPON_MACE2H, LE_ITEM_WEAPON_SWORD1H, LE_ITEM_WEAPON_SWORD2H, LE_ITEM_WEAPON_WARGLAIVE, LE_ITEM_WEAPON_DAGGER, LE_ITEM_WEAPON_UNARMED, LE_ITEM_WEAPON_POLEARM, LE_ITEM_WEAPON_STAFF, LE_ITEM_WEAPON_WAND, LE_ITEM_WEAPON_BOWS, LE_ITEM_WEAPON_CROSSBOW, LE_ITEM_WEAPON_GUNS, LE_ITEM_ARMOR_SHIELD, LE_ITEM_ARMOR_GENERIC, INVTYPE_RANGED, INVTYPE_HOLDABLE = ATTACHMENTS_MAX_SEND, LE_ITEM_CLASS_ARMOR, LE_ITEM_ARMOR_CLOTH, LE_ITEM_ARMOR_LEATHER, LE_ITEM_ARMOR_MAIL, LE_ITEM_ARMOR_PLATE, LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_AXE1H, LE_ITEM_WEAPON_AXE2H, LE_ITEM_WEAPON_MACE1H, LE_ITEM_WEAPON_MACE2H, LE_ITEM_WEAPON_SWORD1H, LE_ITEM_WEAPON_SWORD2H, LE_ITEM_WEAPON_WARGLAIVE, LE_ITEM_WEAPON_DAGGER, LE_ITEM_WEAPON_UNARMED, LE_ITEM_WEAPON_POLEARM, LE_ITEM_WEAPON_STAFF, LE_ITEM_WEAPON_WAND, LE_ITEM_WEAPON_BOWS, LE_ITEM_WEAPON_CROSSBOW, LE_ITEM_WEAPON_GUNS, LE_ITEM_ARMOR_SHIELD, LE_ITEM_ARMOR_GENERIC, INVTYPE_RANGED, INVTYPE_HOLDABLE;

-----------------------------------------------------------------------------
-- Pre-Legion support
-----------------------------------------------------------------------------

if (not S:IsBeta()) then
	local weapon, armor = GetAuctionItemClasses();

	LE_ITEM_CLASS_ARMOR = 4;
	LE_ITEM_ARMOR_CLOTH = 1;
	LE_ITEM_ARMOR_LEATHER = 2;
	LE_ITEM_ARMOR_MAIL = 3;
	LE_ITEM_ARMOR_PLATE = 4;
	LE_ITEM_CLASS_WEAPON = 2;
	LE_ITEM_WEAPON_AXE1H = 0;
	LE_ITEM_WEAPON_AXE2H = 1;
	LE_ITEM_WEAPON_MACE1H = 4;
	LE_ITEM_WEAPON_MACE2H = 5;
	LE_ITEM_WEAPON_SWORD1H = 7;
	LE_ITEM_WEAPON_SWORD2H = 8;
	LE_ITEM_WEAPON_WARGLAIVE = 9;
	LE_ITEM_WEAPON_DAGGER = 15;
	LE_ITEM_WEAPON_UNARMED = 13;
	LE_ITEM_WEAPON_POLEARM = 6;
	LE_ITEM_WEAPON_STAFF = 10;
	LE_ITEM_WEAPON_WAND = 19;
	LE_ITEM_WEAPON_BOWS = 2;
	LE_ITEM_WEAPON_CROSSBOW = 18;
	LE_ITEM_WEAPON_GUNS = 3;
	LE_ITEM_ARMOR_SHIELD = 6;
	LE_ITEM_ARMOR_GENERIC = 0;

	local itemClasses = {
		[LE_ITEM_CLASS_ARMOR] = armor,
		[LE_ITEM_CLASS_WEAPON] = weapon,
	};

	local itemClassIDs = {
		[armor] = LE_ITEM_CLASS_ARMOR,
		[weapon] = LE_ITEM_CLASS_WEAPON,
	};

	GetItemClassInfo = function(itemClassID)
		return itemClasses[itemClassID];
	end

	local axe1, axe2, bow, gun, mace1, mace2, polearm, sword1, sword2, staff, fist, _, dagger, thrown, xbow, wand = GetAuctionItemSubClasses(1);
	local misc, cloth, leather, mail, plate, _, shield = GetAuctionItemSubClasses(2);

	local itemSubClasses = {
		[LE_ITEM_CLASS_ARMOR] = {
			[LE_ITEM_ARMOR_CLOTH] = cloth,
			[LE_ITEM_ARMOR_LEATHER] = leather,
			[LE_ITEM_ARMOR_MAIL] = mail,
			[LE_ITEM_ARMOR_PLATE] = plate,
			[LE_ITEM_ARMOR_SHIELD] = shield,
			[LE_ITEM_ARMOR_GENERIC] = misc,
		},
		[LE_ITEM_CLASS_WEAPON] = {
			[LE_ITEM_WEAPON_AXE1H] = axe1,
			[LE_ITEM_WEAPON_AXE2H] = axe2,
			[LE_ITEM_WEAPON_MACE1H] = mace1,
			[LE_ITEM_WEAPON_MACE2H] = mace2,
			[LE_ITEM_WEAPON_SWORD1H] = sword1,
			[LE_ITEM_WEAPON_SWORD2H] = sword2,
			[LE_ITEM_WEAPON_DAGGER] = dagger,
			[LE_ITEM_WEAPON_UNARMED] = fist,
			[LE_ITEM_WEAPON_POLEARM] = polearm,
			[LE_ITEM_WEAPON_STAFF] = staff,
			[LE_ITEM_WEAPON_WAND] = wand,
			[LE_ITEM_WEAPON_BOWS] = bow,
			[LE_ITEM_WEAPON_CROSSBOW] = xbow,
			[LE_ITEM_WEAPON_GUNS] = gun,
		},
	};

	local itemSubClassIDs = {
		[armor] = {
			[cloth] = LE_ITEM_ARMOR_CLOTH,
			[leather] = LE_ITEM_ARMOR_LEATHER,
			[mail] = LE_ITEM_ARMOR_MAIL,
			[plate] = LE_ITEM_ARMOR_PLATE,
			[shield] = LE_ITEM_ARMOR_SHIELD,
			[misc] = LE_ITEM_ARMOR_GENERIC,
		},
		[weapon] = {
			[axe1] = LE_ITEM_WEAPON_AXE1H,
			[axe2] = LE_ITEM_WEAPON_AXE2H,
			[mace1] = LE_ITEM_WEAPON_MACE1H,
			[mace2] = LE_ITEM_WEAPON_MACE2H,
			[sword1] = LE_ITEM_WEAPON_SWORD1H,
			[sword2] = LE_ITEM_WEAPON_SWORD2H,
			[dagger] = LE_ITEM_WEAPON_DAGGER,
			[fist] = LE_ITEM_WEAPON_UNARMED,
			[polearm] = LE_ITEM_WEAPON_POLEARM,
			[staff] = LE_ITEM_WEAPON_STAFF,
			[wand] = LE_ITEM_WEAPON_WAND,
			[bow] = LE_ITEM_WEAPON_BOWS,
			[xbow] = LE_ITEM_WEAPON_CROSSBOW,
			[gun] = LE_ITEM_WEAPON_GUNS,
		},
	};

	GetItemSubClassInfo = function(itemClassID, itemSubClassID)
		return itemSubClasses[itemClassID] and itemSubClasses[itemClassID][itemSubClassID];
	end

	local GetItemInfo_Blizzard = GetItemInfo;
	GetItemInfo = function(itemID)
		local _, _, quality, _, _, class, subclass, _, equipSlot = GetItemInfo_Blizzard(itemID);
		return nil, nil, quality, nil, nil, nil, nil, nil, equipSlot, nil, nil, itemClassIDs[class], itemSubClassIDs[class] and itemSubClassIDs[class][subclass];
	end

	C_TransmogCollection = {
		PlayerHasTransmog = function()
			return false;
		end,
	};
end

-----------------------------------------------------------------------------
-- Settings
-----------------------------------------------------------------------------

local settingsAnchor;

local CreateSettingsHeader = function(itemClassID)
	local frame = CreateFrame("Frame", nil, addon.configFrame);
	frame:SetHeight(18);

	if (settingsAnchor) then
		frame:SetPoint("TOP", settingsAnchor, "BOTTOM", 0, 0);
	else
		frame:SetPoint("TOP", addon.configFrame, "TOP", 0, -30);
	end

	frame:SetPoint("LEFT");
	frame:SetPoint("RIGHT");

	local label = frame:CreateFontString(nil, "BACKGROUND", "GameFontNormal");
	label:SetPoint("TOP");
	label:SetPoint("BOTTOM");
	label:SetJustifyH("CENTER");
	label:SetText(GetItemClassInfo(itemClassID));

	local left = frame:CreateTexture(nil, "BACKGROUND");
	left:SetHeight(8);
	left:SetPoint("LEFT", 14, 0);
	left:SetPoint("RIGHT", label, "LEFT", -5, 0);
	left:SetTexture("Interface\\Tooltips\\UI-Tooltip-Border");
	left:SetTexCoord(0.81, 0.94, 0.5, 1);
	left:SetPoint("RIGHT", label, "LEFT", -5, 0);

	local right = frame:CreateTexture(nil, "BACKGROUND");
	right:SetHeight(8);
	right:SetPoint("RIGHT", -10, 0);
	right:SetPoint("LEFT", label, "RIGHT", 5, 0);
	right:SetTexture("Interface\\Tooltips\\UI-Tooltip-Border");
	right:SetTexCoord(0.81, 0.94, 0.5, 1);

	settingsAnchor = frame;
end

local CreateSettingsEditBox = function(itemClassID, itemSubClassID, customText)
	local frame = CreateFrame("Frame", nil, addon.configFrame);
	frame:SetHeight(26);
	frame:SetPoint("TOP", settingsAnchor, "BOTTOM", 0, 0);
	frame:SetPoint("LEFT");
	frame:SetPoint("RIGHT");
	frame.itemClassID = itemClassID;
	frame.itemSubClassID = itemSubClassID;

	local label = frame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
	label:SetPoint("TOPLEFT", 14, 0);
	label:SetPoint("BOTTOMRIGHT", -230, 0);
	label:SetJustifyH("LEFT");
	label:SetText(customText or GetItemSubClassInfo(itemClassID, itemSubClassID));

	local editbox = CreateFrame("EditBox", nil, frame, "InputBoxTemplate")
	editbox:SetAutoFocus(false)
	editbox:SetFontObject(ChatFontNormal)
	editbox:SetTextInsets(0, 0, 3, 3)
	editbox:SetMaxLetters(256)
	editbox:SetPoint("RIGHT", -10, 0);
	editbox:SetPoint("LEFT", label, "RIGHT", 5, 0);
	editbox:SetHeight(19)
	frame.editbox = editbox;

	settingsAnchor = frame;
end

local ToggleSettings = function()
	if (addon.configFrame:IsShown()) then
		addon.configFrame:Hide();
	else
		addon.configFrame:Show();
	end
end

local UpdateEditBoxes = function()
	-- Fill edit box text with saved settings
	for _, frame in pairs({ addon.configFrame:GetChildren() }) do
		if (frame.itemClassID and frame.itemSubClassID) then
			frame.editbox:SetText(addon.ADB.recipients[S.myRealm][frame.itemClassID][frame.itemSubClassID] or "");
		end
	end
end

local SaveSettings = function()
	-- Save edit box text to profile
	for _, frame in pairs({ addon.configFrame:GetChildren() }) do
		if (frame.itemClassID and frame.itemSubClassID) then
			local recipient = frame.editbox:GetText();
			addon.ADB.recipients[S.myRealm][frame.itemClassID][frame.itemSubClassID] = (strlen(recipient) >= 2 and recipient or nil);
		end
	end

	-- Whoever is able to use LE_ITEM_WEAPON_BOWS can also use LE_ITEM_WEAPON_CROSSBOW + LE_ITEM_WEAPON_GUNS
	addon.ADB.recipients[S.myRealm][LE_ITEM_CLASS_WEAPON][LE_ITEM_WEAPON_CROSSBOW] = addon.ADB.recipients[S.myRealm][LE_ITEM_CLASS_WEAPON][LE_ITEM_WEAPON_BOWS];
	addon.ADB.recipients[S.myRealm][LE_ITEM_CLASS_WEAPON][LE_ITEM_WEAPON_GUNS] = addon.ADB.recipients[S.myRealm][LE_ITEM_CLASS_WEAPON][LE_ITEM_WEAPON_BOWS];
end

local SettingsFrame_OnShow = function(self)
	addon.configButton:SetText("<");
	UpdateEditBoxes();
end

local SettingsFrame_OnHide = function(self)
	addon.configButton:SetText(">");
	SaveSettings();
end

local CreateSettingsFrame = function()
	if (addon.configFrame) then return; end

	local frame = CreateFrame("Frame", "SezzUIAppearanceMailerRecipients", InboxFrame, "UIPanelDialogTemplate");
	frame:Hide();
	frame:SetPoint("TOPLEFT", 335, 4);
	frame:SetSize(384, 570);
	frame:EnableMouse(true);
	frame:SetScript("OnShow", SettingsFrame_OnShow);
	frame:SetScript("OnHide", SettingsFrame_OnHide);

	addon.configFrame = frame;

	-- Title
	local title = frame:CreateFontString();
	title:SetFontObject(GameFontNormalCenter);
	title:SetPoint("TOPLEFT", "$parentTitleBG");
	title:SetPoint("BOTTOMRIGHT", "$parentTitleBG");
	title:SetText("Appearance Mailer Recipients");

	-- Armor
	CreateSettingsHeader(LE_ITEM_CLASS_ARMOR);
	CreateSettingsEditBox(LE_ITEM_CLASS_ARMOR, LE_ITEM_ARMOR_CLOTH);
	CreateSettingsEditBox(LE_ITEM_CLASS_ARMOR, LE_ITEM_ARMOR_LEATHER);
	CreateSettingsEditBox(LE_ITEM_CLASS_ARMOR, LE_ITEM_ARMOR_MAIL);
	CreateSettingsEditBox(LE_ITEM_CLASS_ARMOR, LE_ITEM_ARMOR_PLATE);

	-- Weapons
	CreateSettingsHeader(LE_ITEM_CLASS_WEAPON);
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_AXE1H);
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_AXE2H);
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_MACE1H);
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_MACE2H);
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_SWORD1H);
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_SWORD2H);
	if (S:IsBeta()) then
		CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_WARGLAIVE);
	end
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_DAGGER);
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_UNARMED);

	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_POLEARM);
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_STAFF);

	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_WAND);
	CreateSettingsEditBox(LE_ITEM_CLASS_WEAPON, LE_ITEM_WEAPON_BOWS, INVTYPE_RANGED);

	CreateSettingsEditBox(LE_ITEM_CLASS_ARMOR, LE_ITEM_ARMOR_SHIELD);
	CreateSettingsEditBox(LE_ITEM_CLASS_ARMOR, LE_ITEM_ARMOR_GENERIC, INVTYPE_HOLDABLE);
end

-----------------------------------------------------------------------------
-- Buttons
-----------------------------------------------------------------------------

local ShowTooltip = function(self)
	if (self.tooltipText) then
		GameTooltip:SetOwner(self, "ANCHOR_RIGHT");
		GameTooltip:AddLine(self.tooltipText, 1, 1, 1);
		GameTooltip:SetClampedToScreen(true);
		GameTooltip:Show();
	end
end

local SendButton_OnClick = function(self)
	if (addon.configFrame:IsVisible()) then
		SaveSettings();
	end
	addon:QueueMails();
end

local SendButton_ShowTooltip = function(self)
	if (addon.configFrame:IsVisible()) then
		SaveSettings();
	end

	local queue = addon:QueueMails(true);

	if (next(queue)) then
		-- show tooltip
		GameTooltip:SetOwner(self, "ANCHOR_RIGHT");

		for recipient, items in pairs(queue) do
			GameTooltip:AddLine(strupper(recipient), 1, 1, 1);

			for _, t in pairs(items) do
				GameTooltip:AddLine(GetContainerItemLink(t[1], t[2]));
			end
		end

		GameTooltip:SetClampedToScreen(true);
		GameTooltip:Show();
	else
		GameTooltip_Hide();
	end
end

local ResizePostalButtons = function()
	if (not PostalSelectOpenButton or not PostalSelectReturnButton or PostalSelectOpenButton:GetWidth() == 76) then return; end

	PostalSelectOpenButton:ClearAllPoints();
	PostalSelectOpenButton:SetPoint("TOPLEFT", "MailFramePortraitFrame", "RIGHT", 0, -2);
	PostalSelectOpenButton:SetWidth(76);

	PostalSelectReturnButton:SetWidth(76);
	PostalSelectReturnButton:ClearAllPoints();
	PostalSelectReturnButton:SetPoint("LEFT", "PostalSelectOpenButton", "RIGHT", 4, 0);

	addon.sendButton:SetWidth(76);
	addon.sendButton:SetHeight(25);
	addon.sendButton:SetText("Send Tmog");
	addon.sendButton:ClearAllPoints();
	addon.sendButton:SetPoint("LEFT", "PostalSelectReturnButton", "RIGHT", 4, 0);
	addon.configButton:SetHeight(25);
end

local CreateButtons = function()
	local buttonHeight = OpenMailReplyButton:GetHeight();

	if (not addon.sendButton) then
		local sendButton = CreateFrame("Button", nil, InboxFrame, "UIPanelButtonTemplate");
		sendButton:SetHeight(buttonHeight);
		sendButton:SetWidth(OpenMailReplyButton:GetWidth() * 1.5 - 16);
		sendButton:SetPoint("TOPLEFT", "MailFramePortraitFrame", "RIGHT", OpenMailReplyButton:GetWidth() * 1.5 + 4, -2);
		sendButton:SetText("Send Transmogs");
		sendButton:RegisterEvent("MAIL_INBOX_UPDATE");
		sendButton:RegisterEvent("UI_ERROR_MESSAGE");
		sendButton:RegisterEvent("MAIL_CLOSED");
		sendButton:SetScript("OnClick", SendButton_OnClick);
		sendButton:SetScript("OnEnter", SendButton_ShowTooltip);
		sendButton:SetScript("OnLeave", GameTooltip_Hide);

		addon.sendButton = sendButton;
	end

	if (not addon.configButton) then
		local configButton = CreateFrame("Button", nil, InboxFrame, "UIPanelButtonTemplate");
		configButton.tooltipText = "Configure Recipients";
		configButton:SetSize(buttonHeight, buttonHeight);
		configButton:SetPoint("LEFT", addon.sendButton, "RIGHT", 4, 0);
		configButton:SetText(">");
		configButton:SetScript("OnEnter", ShowTooltip);
		configButton:SetScript("OnLeave", GameTooltip_Hide);
		configButton:SetScript("OnClick", ToggleSettings);

		addon.configButton = configButton;
	end

	if (IsAddOnLoaded("Postal") and Postal and Postal.IsEnabled and Postal:IsEnabled()) then
		local Postal_Select = Postal:GetModule("Select", true);
		if (Postal_Select and Postal_Select:IsEnabled()) then
			if (PostalSelectOpenButton and PostalSelectReturnButton) then
				ResizePostalButtons();
			else
				hooksecurefunc(Postal_Select, "OnEnable", ResizePostalButtons);
			end
		end
	end
end

-----------------------------------------------------------------------------
-- Module Loader
-----------------------------------------------------------------------------

addon.OnInitialize = function(self)
	self:LoadSettings(true);
	self:SetEnabledState(self.DB.enabled);
end

addon.InitializeProfile = function(self)
	if (not self.ADB.recipients or type(self.ADB.recipients) ~= "table") then
		self.ADB.recipients = {};
	end

	if (not self.ADB.recipients[S.myRealm]) then
		self.ADB.recipients[S.myRealm] = {};
	end

	if (not self.ADB.recipients[S.myRealm][LE_ITEM_CLASS_ARMOR]) then
		self.ADB.recipients[S.myRealm][LE_ITEM_CLASS_ARMOR] = {};
	end

	if (not self.ADB.recipients[S.myRealm][LE_ITEM_CLASS_WEAPON]) then
		self.ADB.recipients[S.myRealm][LE_ITEM_CLASS_WEAPON] = {};
	end
end

addon.OnEnable = function(self)
	self:InitializeProfile();

	self:RegisterEvent("MAIL_CLOSED");

	if (not self.configFrame) then
		if (InboxFrame:IsShown()) then
			-- already at a mailbox, create frames
			self:MAIL_SHOW();
		else
			-- create frames when visiting the mailbox
			self:RegisterEvent("MAIL_SHOW");
		end
	end

	-- show buttons
	if (self.configFrame) then
		self.sendButton:Show();
		self.configButton:Show();
	end
end

addon.OnDisable = function(self)
	-- hide buttons
	if (self.configFrame) then
		self.sendButton:Hide();
		self.configButton:Hide();
		self.configFrame:Hide();
	end

	-- disable events
	self:UnregisterAllEvents();
end

-----------------------------------------------------------------------------
-- UI Events
-----------------------------------------------------------------------------

addon.BAG_UPDATE_DELAYED = function(self)
	if (self.sendButton:IsEnabled() and self.sendButton:IsMouseOver()) then
		SendButton_ShowTooltip(self.sendButton);
	end
end

addon.MAIL_SHOW = function(self)
	CreateButtons();
	CreateSettingsFrame();
	self:RegisterEvent("BAG_UPDATE_DELAYED");
end

addon.MAIL_CLOSED = function(self)
	if (self.configFrame) then
		self.configFrame:Hide();
		self.configButton:Enable();
		self.sendButton:Enable();
	end
	self.active = false;
	wipe(self.queue);
	self:UnregisterEvent("BAG_UPDATE_DELAYED");
end

-----------------------------------------------------------------------------
-- Mailing
-----------------------------------------------------------------------------

local myName = strlower(S.myName);
local myNameFull = myName.."-"..strlower(gsub(S.myRealm, "%s", ""));

local IsBagItemTradable = function(bag, slot)
	tooltip:SetBagItem(bag, slot);

	for i = 1, 8 do
		local text = tooltip.L[i];
		if (text == ITEM_SOULBOUND or text == ITEM_ACCOUNTBOUND or text == ITEM_BIND_TO_BNETACCOUNT or text == ITEM_BNETACCOUNTBOUND) then
			return false;
		end
	end

	return true;
end

addon.queue = {};

addon.QueueMails = function(self, displayOnly)
	local queue = (displayOnly and {} or self.queue);

	-- iterate through bags and build mailing queue
	local bag, slot;
	for bag = 0, 4 do
		for slot = 1, GetContainerNumSlots(bag) do
			local itemID = GetContainerItemID(bag, slot);
			if (itemID) then
				local _, _, quality, _, _, _, _, _, equipSlot, _, _, itemClassID, itemSubClassID = GetItemInfo(itemID);
				local recipient = (itemClassID and itemSubClassID and self.ADB.recipients[S.myRealm][itemClassID] and self.ADB.recipients[S.myRealm][itemClassID][itemSubClassID] and strlower(self.ADB.recipients[S.myRealm][itemClassID][itemSubClassID]) or nil);

				if (quality >= 2 and recipient and recipient ~= myName and recipient ~= myNameFull
					and (not (itemClassID == LE_ITEM_CLASS_ARMOR and itemSubClassID == LE_ITEM_ARMOR_GENERIC) or (itemClassID == LE_ITEM_CLASS_ARMOR and itemSubClassID == LE_ITEM_ARMOR_GENERIC and equipSlot == "INVTYPE_HOLDABLE"))
					and not C_TransmogCollection.PlayerHasTransmog(itemID) and IsBagItemTradable(bag, slot)) then
					-- LE_ITEM_CLASS_ARMOR, LE_ITEM_ARMOR_GENERIC is used for offhands and jewelery and propably more, filtering it by using INVTYPE_HOLDABLE

					if (not queue[recipient]) then
						queue[recipient] = {};
					end

					tinsert(queue[recipient], { bag, slot });
				end
			end
		end
	end

	-- start sending mails if at least one item is found
	if (next(queue) and not displayOnly) then
		self.active = true;
		self.configButton:Disable();
		self.sendButton:Disable();

		-- enable events
		self:RegisterEvent("MAIL_FAILED");
		self:RegisterEvent("MAIL_SUCCESS");

		-- process queue
		self:ProcessQueue();
	elseif (displayOnly) then
		return queue;
	end
end

addon.ProcessQueue = function(self)
	local recipient, items = next(self.queue);
	
	if (recipient) then
		-- clear message
		ClearSendMail();

		-- attach items
		local index = 1;
		while (index <= ATTACHMENTS_MAX_SEND and #items > 0) do
			local item = tremove(items);

			ClearCursor();
			PickupContainerItem(item[1], item[2]);
			ClickSendMailItemButton(index);
			index = index + 1;
		end

		-- remove recipient when no more items left
		if (#items == 0) then
			self.queue[recipient] = nil;
		end

		-- send message
		SendMail(recipient, "Appearance Collector Items", "");
	else
		-- done
		self:Print("All items sent!");
		self:UnregisterEvent("MAIL_FAILED");
		self:UnregisterEvent("MAIL_SUCCESS");
		self.configButton:Enable();
		self.sendButton:Enable();

		if (self.sendButton:IsMouseOver()) then
			GameTooltip_Hide();
		end
	end
end

addon.MAIL_FAILED = function(self, event)
	-- TOOD: Notify user, that something went wrong...
	self:ProcessQueue();
end

addon.MAIL_SUCCESS = function(self, event)
	-- Mail sent - proceed with the next one
	self:ProcessQueue();
end

-- Move "Inbox Full" Warning
InboxTooMuchMail:ClearAllPoints();
InboxTooMuchMail:SetPoint("TOPLEFT", 48, 5);

local iconWarning = select(2, InboxTooMuchMail:GetRegions());
iconWarning:SetSize(16, 16);
iconWarning:ClearAllPoints();
iconWarning:SetPoint("RIGHT", "$parentText", "LEFT", 0, -1);
