--[[ 

	Martin Karer / Sezz, 2016
	Wardrobe appearance collector for BoE items

	Known issues:
	- The API sometimes doesn't seem to know whether an appearance is collected or not, nothing I can do about that.
	
--]]

local addonName, ns = ...;
local S, C;
if (SezzUI) then
	S, C = unpack(SezzUI);
else
	S, C = ns.S, ns.C;
end

if (not S:IsBeta() or (addonName == "WardrobeTools" and SezzUI)) then return; end
local addon = S:CreateModule("AppearanceCollector", "SezzUIAppearanceCollector", "Button", "SecureActionButtonTemplate,ActionButtonTemplate");

-----------------------------------------------------------------------------

-- Lua API
local strlower, select, tonumber = string.lower, select, tonumber;

-- WoW API/Constants
local GetContainerNumSlots, GetContainerItemID, GetContainerItemLink, InCombatLockdown, SetOverrideBindingClick, ClearOverrideBindings, GetItemInfo, SaveEquipmentSet, DeleteEquipmentSet = GetContainerNumSlots, GetContainerItemID, GetContainerItemLink, InCombatLockdown, SetOverrideBindingClick, ClearOverrideBindings, GetItemInfo, SaveEquipmentSet, DeleteEquipmentSet;
local TRANSMOGRIFY_TOOLTIP_APPEARANCE_UNKNOWN, ITEM_SOULBOUND, ITEM_ACCOUNTBOUND, ITEM_BIND_TO_BNETACCOUNT, ITEM_BNETACCOUNTBOUND, LE_ITEM_CLASS_ARMOR, LE_ITEM_CLASS_WEAPON = TRANSMOGRIFY_TOOLTIP_APPEARANCE_UNKNOWN, ITEM_SOULBOUND, ITEM_ACCOUNTBOUND, ITEM_BIND_TO_BNETACCOUNT, ITEM_BNETACCOUNTBOUND, LE_ITEM_CLASS_ARMOR, LE_ITEM_CLASS_WEAPON;
local GameTooltip_Hide = GameTooltip_Hide;

-----------------------------------------------------------------------------

local tooltip = S.ScanningTooltip;
local equipmentSet = "SezzAC";

local CanCollectAppearance = function(bag, slot)
	tooltip:SetBagItem(bag, slot);

	for i = tooltip:NumLines(), 1, -1 do
		local text = tooltip.L[i];
		if (text == TRANSMOGRIFY_TOOLTIP_APPEARANCE_UNKNOWN) then
			return true;
		end
	end

	return false;
end

local IsBagItemTradable = function(bag, slot)
	tooltip:SetBagItem(bag, slot);

	for i = 1, 8 do
		local text = tooltip.L[i];
		if (text == ITEM_SOULBOUND or text == ITEM_ACCOUNTBOUND or text == ITEM_BIND_TO_BNETACCOUNT or text == ITEM_BNETACCOUNTBOUND) then
			return false;
		end
	end

	return true;
end

local GetNextItem = function()
	local bag, slot;
	for bag = 0, 4 do
		for slot = 1, GetContainerNumSlots(bag) do
			local link = GetContainerItemLink(bag, slot);
			local itemID = GetContainerItemID(bag, slot);

			if (link and itemID and not C_TransmogCollection.PlayerHasTransmog(itemID) and IsBagItemTradable(bag, slot) and CanCollectAppearance(bag, slot)) then
				local _, _, _, _, reqLevel, _, _, _, equipSlot, _, _, itemClassID = GetItemInfo(link);
				if ((itemClassID == LE_ITEM_CLASS_ARMOR or itemClassID == LE_ITEM_CLASS_WEAPON) and (not reqLevel or (reqLevel > 0 and reqLevel <= S.myLevel))) then
					return bag, slot, equipSlot;
				end
			end
		end
	end
end

local ShowTooltip = function(self)
	GameTooltip:SetOwner(self, "ANCHOR_TOP");

	if (self.bag and self.slot) then
		GameTooltip:SetBagItem(self.bag, self.slot);
	else
		GameTooltip:AddLine("Close AppearanceCollector", 1, 1, 1);
		GameTooltip:AddLine("There are no more uncollected appearances in your inventory.");
		GameTooltip:AddLine("Click here to close the addon and re-equip your old gear.");
	end

	GameTooltip:SetClampedToScreen(true);
	GameTooltip:Show();
end

addon.BAG_UPDATE_DELAYED = function(self, event)
	if (not InCombatLockdown()) then
		self:Update();
	end
end

addon.PLAYER_REGEN_DISABLED = function(self, event)
	self.icon:SetVertexColor(0.5, 0.5, 0.5);
	self.icon:SetDesaturated(true);
	self:SetAttribute("type", nil);
	self:UnbindMouseWheel();
end

addon.PLAYER_REGEN_ENABLED = function(self, event)
	self.icon:SetVertexColor(1, 1, 1);
	self.icon:SetDesaturated(false);
	self:SetAttribute("type", "macro");
	self:BindMouseWheel();
end

addon.Update = function(self, shutdown)
	local bag, slot, equipSlot = GetNextItem();

	if (not shutdown and bag and slot and equipSlot) then
		local texture = GetContainerItemInfo(bag, slot);

		self.bag = bag;
		self.slot = slot;
		self.equipSlot = equipSlot;

		self:SetAttribute("macrotext", "/use "..bag.." "..slot.."\n/click StaticPopup1Button1");
		self.icon:SetTexture(texture);
	else
		-- No more items left
		self.bag = nil;
		self.slot = nil;
		self.equipSlot = nil;
		self:SetAttribute("macrotext", nil);
		self.icon:SetTexture("Interface\\PaperDollInfoFrame\\UI-GearManager-Undo");
	end
end

addon.OnInitialize = function(self)
	if (not C.modules.AppearanceCollector) then
		-- Standalone defaults
		C.modules.AppearanceCollector = {
			anchor = "CENTER",
			x = 300,
			y = 0,
		};
	end

	self:LoadSettings();
end

addon.OnEnable = function(self)
	self:Print(GREEN_FONT_COLOR_CODE.."ON"..FONT_COLOR_CODE_CLOSE);

	if (not CollectionsJournal) then
		CollectionsJournal_LoadUI();
	end

	if (not IsAddOnLoaded("Blizzard_Collections")) then
		self:PrintError("Couln't load Blizzard_Collections - this is required to check if an appearance is already collected.");
		self:SetEnabledState(false);
		return;
	end

	-- Create button
	if (not self:GetAttribute("type")) then
		self:SetPoint(self.DB.anchor, self.DB.x, self.DB.y);
		self:SetSize(64, 64);
		self:SetScript("OnEnter", ShowTooltip);
		self:SetScript("OnLeave", GameTooltip_Hide);
		self:RegisterForClicks("LeftButtonUp", "RightButtonUp");
		self:SetMovable(true);
		self:RegisterForDrag("LeftButton");
		self:EnableMouse(true);
		S.SkinActionButton(self);

		self:SetScript("OnDragStart", function(self)
			if (IsAltKeyDown()) then
				self:StartMoving();
			end
		end);

		self:SetScript("OnDragStop", function(self)
			self:StopMovingOrSizing();

			local anchor, _, _, x, y = self:GetPoint();
			self.DB.anchor = anchor;
			self.DB.x = x;
			self.DB.y = y;
		end);

		self:HookScript("PreClick", function(self, button)
			if (not InCombatLockdown() and button == "RightButton" and self.equipSlot) then
				-- Remove equip macro
				self:Update(true);
			end
		end);

		self:HookScript("OnClick", function(self, button)
			if (not InCombatLockdown()) then
				if (button == "RightButton" or not self.bag) then
					-- Disable
					self:Toggle();
				end
			end
		end);
	end

	-- Setup equipment set
	PaperDollFrame_ClearIgnoredSlots();
	PaperDollEquipmentManagerPane_Update();
	SaveEquipmentSet(equipmentSet, 897143);

	-- Setup events
	self:RegisterEvent("BAG_UPDATE_DELAYED");
	self:RegisterEvent("PLAYER_REGEN_DISABLED");
	self:RegisterEvent("PLAYER_REGEN_ENABLED");
	self:PLAYER_REGEN_ENABLED();

	-- Update button
	self:Update();
	self:Show();
end

addon.OnDisable = function(self)
	self:Print(GRAY_FONT_COLOR_CODE.."OFF"..FONT_COLOR_CODE_CLOSE);
	self:Hide();
	self:SetAttribute("macrotext", nil);
	self:UnregisterAllEvents();
	self:UnbindMouseWheel();

	-- Equip normal gear and remove equipment set
	EquipmentManager_EquipSet(equipmentSet);
	DeleteEquipmentSet(equipmentSet);
end

addon.BindMouseWheel = function(self)
	SetOverrideBindingClick(self, true, "MOUSEWHEELUP", self:GetName());
	SetOverrideBindingClick(self, true, "MOUSEWHEELDOWN", self:GetName());
end

addon.UnbindMouseWheel = function(self)
	ClearOverrideBindings(self);
end

addon.Toggle = function(self)
	if (not InCombatLockdown()) then
		self:SetEnabledState(not self.enabledState);
	else
		self:Print("Sorry, you can't toggle this addon while in combat!");
	end
end

S:RegisterSlashCommand("AC", "/ac", function(args)
	if (args and args ~= "") then
		local args = strlower(args);

		if (args  == "reset") then
			addon.DB.anchor = "CENTER";
			addon.DB.x = 300;
			addon.DB.y = 0;
			addon:ClearAllPoints();
			addon:SetPoint(addon.DB.anchor, addon.DB.x, addon.DB.y);
		else
			addon:Print("Usage:")
			addon:Print("- Type "..ORANGE_FONT_COLOR_CODE.."/ac"..FONT_COLOR_CODE_CLOSE.." to toggle the button");
			addon:Print("- Left clicking the button equips the displayed item, right clicking closes the addon (you can also use the mousewheel while the button is displayed to quickly equip all items and close the addon when done)");
			addon:Print("- To move the button press and hold the ALT key and drag it to the desired position");
			addon:Print("Slash commands:");
			addon:Print(" - "..ORANGE_FONT_COLOR_CODE.."/ac reset"..FONT_COLOR_CODE_CLOSE.." reset button position");
		end
	else
		addon:Toggle();
	end
end);
