--[[ based upon FishermansFriend by Ammo; fixed to work in Classic by Firedancer ]]--

local FISHINGTEXTURE = "Interface\\Icons\\Trade_Fishing"
local db 					-- our saved variables database
local overrideOn = false 	-- do we have the override click active?
local fishingName = ""		-- fishing skill name, will be filled from spellbook
local isFishing = nil		-- flag if we're in fishing mode or not
local resetClickToMove = nil -- will be set to true if we have clicktomove overridden
local resetAutoLoot = nil	-- will be set to true if we have overriden autoloot
local ACTIONDOUBLEWAIT = 0.4 -- seconds to wait for the next click for doubleclicking
local lastClickTime = nil 	-- used to store the last click time for doubleclick detection
local recheckAfterCombat = nil -- toggle to recheck fishing mode after combat ends

local defaults = {
	profile = {
		sounds = true,
		autoloot = true,
		soundVars = { 
			["Sound_MasterVolume"] = true,
			["Sound_MusicVolume"] = true,
			["Sound_AmbienceVolume"] = true,
			["Sound_SFXVolume"] = true,
		},
		ignoreVars = {
			["MapWaterSounds"] = true,
		},
		soundVolume = {
			["Sound_MasterVolume"] = 1.0,
			["Sound_MusicVolume"] = 0.0,
			["Sound_AmbienceVolume"] = 0.0,
			["Sound_SFXVolume"] = 1.0,
		},
	}
}

local L = LibStub("AceLocale-3.0"):GetLocale("EasyFishingClassic")

local function giveOptions()
	local options = {
		type = "group",
		icon = FISHINGTEXTURE,
		name = L["Easy Fishing Classic"],
		get = function( k ) return db[k.arg] end,
		set = function( k, v ) db[k.arg] = v end,
		args = {
			fishdesc = {
				type = "description", order = 0,
				name = L["Easy Fishing Classic will automatically detect if you are wielding a fishing rod. When you are, you can double rightclick to cast your line. You can toggle the features of Easy Fishing Classic below."],
			},
			autoloot = {
				name = L["Auto Loot"],
				desc = L["Turn on Auto Loot when fishing."],
				type = "toggle",
				arg = "autoloot",
				order = 1, width = "full",
			},
			sounds = {
				name = L["Enhance Sounds"],
				desc = L["Enhance sounds for fishing, lowering all sounds but the bobber (and potentially other sound effects)."],
				type = "toggle",
				arg = "sounds",
				order = 4, width = "full",
			},
			volume = {
				name = L["Volume Settings"],
				desc = L["Set the volumes for the various types of sound when fishing."],
				type = "group",
				inline = true,
				get = function( k ) return db["soundVolume"][k.arg] end,
				set = function( k, v ) db["soundVolume"][k.arg] = v end,
				disabled = function() return not db.sounds end,
				args = {
					master = {
						name = L["Master Volume"],
						arg = "Sound_MasterVolume",
						type = "range",
						min = 0, max = 1,
					},
					sfx = {
						name = L["SFX Volume"],
						arg = "Sound_SFXVolume",
						type = "range",
						min = 0, max = 1,
					},
					music = {
						name = L["Music Volume"],
						arg = "Sound_MusicVolume",
						type = "range",
						min = 0, max = 1,
					},
					ambience = {
						name = L["Ambience Volume"],
						arg = "Sound_AmbienceVolume",
						type = "range",
						min = 0, max = 1,
					},
				}
			}
		}
	}
	return options
end

EFC = LibStub("AceAddon-3.0"):NewAddon("EasyFishingClassic", "AceConsole-3.0", "AceEvent-3.0", "AceTimer-3.0", "AceHook-3.0")

function EFC:OnInitialize()
	self.db = LibStub("AceDB-3.0"):New("EFC_DB", defaults, "Default")
	db = self.db.profile
	
	LibStub("AceConfigRegistry-3.0"):RegisterOptionsTable("Easy Fishing Classic", giveOptions)
	local f = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("Easy Fishing Classic", L["Easy Fishing Classic"])
end

function EFC:OnEnable()
	isFishing = nil
	
	self:CreateButton()
	self:HookScript(WorldFrame, "OnMouseDown", "WorldFrameOnMouseDown")
	self:RegisterEvent("UNIT_INVENTORY_CHANGED")
	self:RegisterEvent("UNIT_AURA", "UNIT_INVENTORY_CHANGED")
	self:RegisterEvent("SPELLS_CHANGED")
	self:RegisterEvent("PLAYER_LOGOUT", "StopFishingMode")
	self:RegisterEvent("LEARNED_SPELL_IN_TAB", "SPELLS_CHANGED")
	self:RegisterEvent("SKILL_LINES_CHANGED", "SPELLS_CHANGED")
	
	self:RegisterEvent("PLAYER_REGEN_ENABLED")
	
	-- intial setup
	self:SPELLS_CHANGED() -- find the fishing skill
	self:UNIT_INVENTORY_CHANGED( "", "player" ) -- find the fishing pole
end

-- Make sure we stop fishing OnDisable to restore old values of mouselook etc.
function EFC:OnDisable()
	self:StopFishingMode()
end

-- Remove the overridebinding click after clicking or after setting a lure
local function CleanupOverride()
	local wait = false
	if InCombatLockdown() then
		wait = true
	elseif overrideOn then
		ClearOverrideBindings(efcButton)
		overrideOn = false
	end

	if wait then
		EFC:ScheduleTimer( CleanupOverride, 1 )
	end
end

function EFC:CreateButton()
	if efcButton then return end
	
	efcButton = CreateFrame("Button", "EasyFishingClassicButton", UIParent, "SecureActionButtonTemplate")
	-- place outside viewable area
	efcButton:SetPoint("LEFT", UIParent, "RIGHT", 10000, 0)
	efcButton:EnableMouse(true)
	efcButton:RegisterForClicks("RightButtonUp")
	efcButton:Hide()
	efcButton:SetScript("PostClick", CleanupOverride)
end

local function CheckForDoubleClick()
	if lastClickTime then
		local pressTime = GetTime()
		local doubleTime = pressTime - lastClickTime
		lastClickTime = pressTime
		if ( doubleTime < ACTIONDOUBLEWAIT ) then
			return true
		end
	end
	lastClickTime = GetTime()
	return false
end

function EFC:WorldFrameOnMouseDown( this, button )
	if button == "RightButton" and not InCombatLockdown() and 
		isFishing and fishingName and CheckForDoubleClick() then
			-- We're stealing the mouse-up event, make sure we exit MouseLook
			if IsMouselooking() then MouselookStop() end
			efcButton:SetAttribute("type", "spell")
			efcButton:SetAttribute("spell", fishingName)
			SetOverrideBindingClick(efcButton, true, "BUTTON2", "EasyFishingClassicButton")
			overrideOn = true
	end
end

function EFC:PLAYER_REGEN_ENABLED()
	if recheckAfterCombat then
		recheckAfterCombat = nil
		self:UNIT_INVENTORY_CHANGED("", "player") -- check if we have a fishing pole
	end
end

function EFC:UNIT_INVENTORY_CHANGED( event, unit )
	if unit ~= "player" then return end -- we only care about the player
	if InCombatLockdown() then
		recheckAfterCombat = true -- we'll check back for the pole after combat
		return
	end
	local usingFishingPole = IsUsableSpell(fishingName or "")
	-- enable / disable fishing mode only if we have to
	if not isFishing and usingFishingPole then
		self:StartFishingMode()
	elseif isFishing and not usingFishingPole then 
		self:StopFishingMode()
	end
end

function EFC:SPELLS_CHANGED()
	local fishing = GetSpellInfo(7620) or GetSpellInfo(7731) or GetSpellInfo(7732) or GetSpellInfo(18248)
	if (GetSpellInfo(fishing)) then
		fishingName = fishing
	end
end

function EFC:StartFishingMode()
	if isFishing then return end
	
	-- Disable Click-to-Move if we're fishing
	if GetCVar("autointeract") == "1" then
		resetClickToMove = true
		SetCVar("autointeract", "0")
	end
	-- autoloot on when fishing
	if db.autoloot and not GetCVarBool("autoLootDefault") then
		resetAutoLoot = true
		SetCVar("autoLootDefault", "1")
	end
	-- sound enhancement
	if db.sounds then self:EnhanceSound() end
	isFishing = true
end

function EFC:StopFishingMode()
	if not isFishing then return end
	
	if resetClickToMove then
		-- Re-enable Click-to-Move if we changed it
		SetCVar("autointeract", "1")
		resetClickToMove = nil
	end
	if resetAutoLoot then
		SetCVar("autoLootDefault", "0")
		resetAutoLoot = nil
	end
	self:RestoreSound()
	isFishing = false
end

function EFC:EnhanceSound()
	if db.isEnhanced then return end
	for k, v in pairs( db.soundVars ) do
		db.soundVars[k] = tonumber(GetCVar(k))
	end
	for k, v in pairs( db.ignoreVars ) do
		db.soundVars[k] = nil
	end
	for k, v in pairs( db.soundVolume ) do 
		SetCVar(k, v)
	end
	db.isEnhanced = true
end

function EFC:RestoreSound()
	if not db.isEnhanced then return end
	for k, v in pairs( db.soundVars ) do
		SetCVar(k, v)
		db.soundVars[k] = true
	end
	db.isEnhanced = nil
end
