#version 120
#extension GL_ARB_shader_texture_lod : enable

/*
 _______ _________ _______  _______  _
(  ____ \\__   __/(  ___  )(  ____ )( )
| (    \/   ) (   | (   ) || (    )|| |
| (_____    | |   | |   | || (____)|| |
(_____  )   | |   | |   | ||  _____)| |
      ) |   | |   | |   | || (      (_)
/\____) |   | |   | (___) || )       _
\_______)   )_(   (_______)|/       (_)

This work is licensed under the Creative Commons Attribution-NonCommercial-NoDerivatives 4.0 International License.
 To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-nd/4.0/.

*/
/////////////////////////CONFIGURABLE VARIABLES////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////CONFIGURABLE VARIABLES////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#define BANDING_FIX_FACTOR 1.0

#define SMOOTH_SKY

#define CLOUD_PLANE_Cloud			//New 2D clouds!
#define CLOUD_PLANE_Coverage 0.40		// CLOUD_PLANE_Coverage. 0.20 = Lowest Cover. 0.60 = Highest Cover [0.20 0.30 0.40 0.50 0.60 0.70]
//#define CLOUD_COVERAGE CLOUD_PLANE_Coverage + rainy * 0.335;		//Default is 0.49

#define Water_Refraction

#define VOLUMETRIC_LIGHT			//True GodRays, not 2D ScreenSpace
//----------GodRays----------//
#define GODRAYS
	#ifdef VOLUMETRIC_LIGHT
		float exposure = 0.00008;			//godrays intensity 0.0009 is default
	#else
		float exposure = 0.0009;
	#endif
	const float grdensity = 1.0;
	const int NUM_SAMPLES = 10;			//increase this for better quality at the cost of performance /8 is default
	const float Moon_exposure = 0.001;			//Moonrays intensity 0.0009 is default, increase to make brighter

#define MOONRAYS					//Make sure if you enable/disable this to do the same in Composite, PLEASE NOTE Moonrays have a bug at sunset/sunrise


//---Volumetric light strength--//
#define SUNRISEnSET		1.5	//default is 2.0
#define NOON			1.25		//default is 0.2 for least amount of haze at the cost of effect
#define NIGHT			0.65		//default is 0.7 for least amount of haze at the cost of effect, 1.5 for best looking but lots of haze
#define IN_SIDE_RAYS	2.5		//strength of rays when indoors, daytime
#define IN_SIDE_RAYS_NIGHT 100.3		//strength of rays when indoors, night

#define NUM_RAYS		1


#define NO_UNDERWATER_RAYS


//----------End CONFIGURABLE GodRays----------//

/////////////////////////END OF CONFIGURABLE VARIABLES/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////END OF CONFIGURABLE VARIABLES/////////////////////////////////////////////////////////////////////////////////////////////////////////////

/* DRAWBUFFERS:2 */

const bool gcolorMipmapEnabled = true;
const bool compositeMipmapEnabled = true;


uniform sampler2D gcolor;
uniform sampler2D gdepth;
uniform sampler2D gdepthtex;
uniform sampler2D depthtex1;
uniform sampler2D gnormal;
uniform sampler2D composite;
uniform sampler2D noisetex;

uniform float near;
uniform float far;
uniform float viewWidth;
uniform float viewHeight;
uniform float rainStrength;
uniform float wetness;
uniform float aspectRatio;
uniform float frameTimeCounter;
uniform int worldTime;
uniform int isEyeInWater;
uniform int moonPhase;

uniform mat4 gbufferProjection;
uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferModelViewInverse;
uniform mat4 gbufferModelView;
uniform mat4 shadowModelViewInverse;

uniform vec3 cameraPosition;
uniform ivec2 eyeBrightnessSmooth;

varying vec4 texcoord;

varying vec3 lightVector;
varying vec3 upVector;
uniform vec3 sunPosition;

varying float timeSunriseSunset;
varying float timeSunrise;
varying float timeNoon;
varying float timeSunset;
varying float timeMidnight;
varying float timeSkyDark;

varying vec3 colorSunlight;
varying vec3 colorSkylight;

#define ANIMATION_SPEED 1.0

//#define ANIMATE_USING_WORLDTIME

#ifdef ANIMATE_USING_WORLDTIME
	#define FRAME_TIME worldTime * ANIMATION_SPEED / 20.0
#else
	#define FRAME_TIME frameTimeCounter * ANIMATION_SPEED
#endif


/////////////////////////FUNCTIONS/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////FUNCTIONS/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

float saturate(float x) {
	return clamp(x, 0.0, 1.0);
}

vec3 GetNormals(in vec2 coord) {
	vec3 normal = vec3(0.0);
	normal = texture2DLod(gnormal, coord.st, 0).rgb;
	normal = normal * 2.0 - 1.0;

	normal = normalize(normal);

	return normal;
}

float GetDepth(in vec2 coord) {
	return texture2D(gdepthtex, coord).x;
}

float ExpToLinearDepth(in float depth) {
	return 2.0 * near * far / (far + near - (2.0 * depth - 1.0) * (far - near));
}

float GetDepthLinear(vec2 coord) {
    return 2.0 * near * far / (far + near - (2.0 * texture2D(gdepthtex, coord).x - 1.0) * (far - near));
}

vec4 GetViewSpacePosition(in vec2 coord) {	//Function that calculates the screen-space position of the objects in the scene using the depth texture and the texture coordinates of the full-screen quad
	float depth = GetDepth(coord);
	vec4 fragposition = gbufferProjectionInverse * vec4(coord.s * 2.0 - 1.0, coord.t * 2.0 - 1.0, 2.0 * depth - 1.0, 1.0);
	fragposition /= fragposition.w;

	return fragposition;
}

float GetMaterialIDs(in vec2 coord) {			//Function that retrieves the texture that has all material IDs stored in it
	return texture2D(gdepth, coord).r;
}

float GetSunlightVisibility(in vec2 coord) {
	return texture2D(gdepth, coord).g;
}

float cubicPulse(float c, float w, float x) {
	x = abs(x - c);
	if (x > w) return 0.0;
	x /= w;

	return 1.0 - x * x * (3.0 - 2.0 * x);
}

bool GetMaterialMask(in vec2 coord, in int ID, in float matID) {
	matID = floor(matID * 255.0);

	if (matID == ID) {
		return true;
	} else {
		return false;
	}
}

bool GetSkyMask(in vec2 coord, in float matID) {
	matID = floor(matID * 255.0);

	if (matID < 1.0 || matID > 254.0)
	{
		return true;
	} else {
		return false;
	}
}

float GetMetallic(in vec2 coord) {
	return texture2D(composite, coord).r;
}

float GetSmoothness(in vec2 coord) {
	return pow(texture2D(composite, coord).b, 2.2);
}



float GetSolidSunlightVisibility(in vec2 coord) {
	return texture2D(composite, coord).a;
}

bool GetWaterMask(in float matID) {					//Function that returns "true" if a pixel is water, and "false" if a pixel is not water.
	matID = floor(matID * 255.0);

	if (matID >= 35.0 && matID <= 51) {
		return true;
	} else {
		return false;
	}
}

bool GetWaterMask(in vec2 coord) {					//Function that returns "true" if a pixel is water, and "false" if a pixel is not water.
	float matID = floor(GetMaterialIDs(coord) * 255.0);

	if (matID >= 35.0 && matID <= 51) {
		return true;
	} else {
		return false;
	}
}

float GetWaterMaskFloat(in vec2 coord) {
	float matID = floor(GetMaterialIDs(coord) * 255.0);

	if (matID >= 35.0 && matID <= 51) {
		return 0.0;
	} else {
		return 1.0;
	}
}

float GetLightmapSky(in vec2 coord) {
	return texture2D(gdepth, texcoord.st).b;
}

vec3 convertScreenSpaceToWorldSpace(vec2 co) {
    vec4 fragposition = gbufferProjectionInverse * vec4(vec3(co, texture2DLod(gdepthtex, co, 0).x) * 2.0 - 1.0, 1.0);
    fragposition /= fragposition.w;

    return fragposition.xyz;
}

vec3 convertCameraSpaceToScreenSpace(vec3 cameraSpace) {
    vec4 clipSpace = gbufferProjection * vec4(cameraSpace, 1.0);
    vec3 NDCSpace = clipSpace.xyz / clipSpace.w;
    vec3 screenSpace = 0.5 * NDCSpace + 0.5;
		screenSpace.z = 0.1;

    return screenSpace;
}

vec3 	CalculateNoisePattern1(vec2 offset, float size) {
	vec2 coord = texcoord.st;

	coord *= vec2(viewWidth, viewHeight);
	coord = mod(coord + offset, vec2(size));
	coord /= 64.0;

	return texture2D(noisetex, coord).xyz;
}

float noise(in float offset) {
	vec2 coord = texcoord.st + vec2(offset);
	float noise = clamp(fract(sin(dot(coord ,vec2(12.9898,78.233))) * 43758.5453),0.0,1.0)*2.0-1.0;
	return noise;
}

float noise(in vec2 coord, in float offset) {
	coord += vec2(offset);
	float noise = clamp(fract(sin(dot(coord ,vec2(12.9898,78.233))) * 43758.5453),0.0,1.0)*2.0-1.0;

	return noise;
}

void 	DoNightEye(inout vec3 color) {					//Desaturates any color input at night, simulating the rods in the human eye
	float amount = 0.8; 												//How much will the new desaturated and tinted image be mixed with the original image
	vec3 rodColor = vec3(0.2, 0.5, 1.0); 		//Cyan color that humans percieve when viewing extremely low light levels via rod cells in the eye
	float colorDesat = dot(color, vec3(1.0)); 	//Desaturated color

	color = mix(color, vec3(colorDesat) * rodColor, timeSkyDark * amount);
}

float Get3DNoise(in vec3 pos) {
	return texture2D(noisetex, pos.xz / 64.0).x;
}

vec3 fresnel(vec3 R0, float vdoth) {
    return R0 + (vec3(1.0) - R0) * max(0.0, pow(1.0 - vdoth, 5));
}


/////////////////////////STRUCTS///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////STRUCTS///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

struct MaskStruct {

	float matIDs;

	bool sky;
	bool land;
	bool tallGrass;
	bool leaves;
	bool ice;
	bool hand;
	bool translucent;
	bool glow;
	bool goldBlock;
	bool ironBlock;
	bool diamondBlock;
	bool emeraldBlock;
	bool sand;
	bool sandstone;
	bool stone;
	bool cobblestone;
	bool wool;

	bool torch;
	bool lava;
	bool glowstone;
	bool fire;

	bool water;

};

struct Ray {
	vec3 dir;
	vec3 origin;
};

struct Plane {
	vec3 normal;
	vec3 origin;
};

struct SurfaceStruct {
	MaskStruct 		mask;			//Material ID Masks

	//Properties that are required for lighting calculation
		vec3 	color;					//Diffuse texture aka "color texture"
		vec3 	normal;					//Screen-space surface normals
		float 	depth;					//Scene depth
		float 	linearDepth;			//Scene depth

		float 	rDepth;
		float  	metallic;
		vec3 	baseReflectivity;
		float 	smoothness;
		Ray 	viewRay;

		vec4 	viewSpacePosition;
		vec4 	worldSpacePosition;
		vec3 	worldLightVector;
		vec3  	upVector;
		vec3 	lightVector;

		float 	sunlightVisibility;

		vec4 	reflection;

		float 	cloudAlpha;
} surface;

struct Intersection {
	vec3 pos;
	float distance;
	float angle;
};

/////////////////////////STRUCT FUNCTIONS//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////STRUCT FUNCTIONS//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void 	CalculateMasks(inout MaskStruct mask) {
	mask.sky 			= GetSkyMask(texcoord.st, mask.matIDs);
	mask.land	 		= !mask.sky;
	mask.tallGrass 		= GetMaterialMask(texcoord.st, 2, mask.matIDs);
	mask.leaves	 		= GetMaterialMask(texcoord.st, 3, mask.matIDs);
	mask.ice		 	= GetMaterialMask(texcoord.st, 4, mask.matIDs);
	mask.hand	 		= GetMaterialMask(texcoord.st, 5, mask.matIDs);
	mask.translucent	= GetMaterialMask(texcoord.st, 6, mask.matIDs);

	mask.glow	 		= GetMaterialMask(texcoord.st, 10, mask.matIDs);

	mask.goldBlock 		= GetMaterialMask(texcoord.st, 20, mask.matIDs);
	mask.ironBlock 		= GetMaterialMask(texcoord.st, 21, mask.matIDs);
	mask.diamondBlock	= GetMaterialMask(texcoord.st, 22, mask.matIDs);
	mask.emeraldBlock	= GetMaterialMask(texcoord.st, 23, mask.matIDs);
	mask.sand	 		= GetMaterialMask(texcoord.st, 24, mask.matIDs);
	mask.sandstone 		= GetMaterialMask(texcoord.st, 25, mask.matIDs);
	mask.stone	 		= GetMaterialMask(texcoord.st, 26, mask.matIDs);
	mask.cobblestone	= GetMaterialMask(texcoord.st, 27, mask.matIDs);
	mask.wool			= GetMaterialMask(texcoord.st, 28, mask.matIDs);

	mask.torch 			= GetMaterialMask(texcoord.st, 30, mask.matIDs);
	mask.lava 			= GetMaterialMask(texcoord.st, 31, mask.matIDs);
	mask.glowstone 		= GetMaterialMask(texcoord.st, 32, mask.matIDs);
	mask.fire 			= GetMaterialMask(texcoord.st, 33, mask.matIDs);

	mask.water 			= GetWaterMask(mask.matIDs);
}

vec4 ComputeRaytraceReflection(in vec3 normal, in float smoothness, in vec3 base_reflectivity, in float metallic, in vec3 fragcolor) {
	float reflectionRange = 2.0;

	float stepRefinementAmount = .1;
	int maxRefinements = 0;

	vec2 screenSpacePosition2D = texcoord.st;
  	vec3 cameraSpacePosition = convertScreenSpaceToWorldSpace(screenSpacePosition2D);

	float initialStepAmount = mix(1.0, 4.0, pow(length((gbufferModelViewInverse * vec4(cameraSpacePosition, 1.0)).xz) / 144.0, 2.0));

	vec3 cameraSpaceNormal = normal;

	vec3 cameraSpaceViewDir = normalize(cameraSpacePosition);
	vec3 old_camera_space_view_vector = normalize(reflect(cameraSpaceViewDir,cameraSpaceNormal));
	vec3 cameraSpaceVector = initialStepAmount * old_camera_space_view_vector;
	vec3 oldPosition = cameraSpacePosition;
	vec3 cameraSpaceVectorPosition = oldPosition + cameraSpaceVector;
	vec3 currentPosition = convertCameraSpaceToScreenSpace(cameraSpaceVectorPosition);
	vec4 color = vec4(pow(texture2D(gcolor, screenSpacePosition2D).rgb, vec3(3.0 + 1.2)), 0.0);
	int numRefinements = 0;
	int count = 0;
	vec2 finalSamplePos = vec2(0.0);

  	while(count < far/initialStepAmount*reflectionRange) {
    	if(currentPosition.x < 0 || currentPosition.x > 1 ||
         currentPosition.y < 0 || currentPosition.y > 1 ||
		   	 currentPosition.z < 0 || currentPosition.z > 1) {
		  	 break;
			}

    	vec2 samplePos = currentPosition.xy;
    	float sampleDepth = convertScreenSpaceToWorldSpace(samplePos).z;

	    float currentDepth = cameraSpaceVectorPosition.z;
	    float diff = sampleDepth - currentDepth;
	    float error = length(cameraSpaceVector);

	  	if(diff >= 0 && diff <= error * 1.00) {
				finalSamplePos = samplePos;
				break;
			}

			cameraSpaceVector *= 2.5;	//Each step gets bigger
    	cameraSpaceVectorPosition += cameraSpaceVector;
			currentPosition = convertCameraSpaceToScreenSpace(cameraSpaceVectorPosition);
    	count++;
  	}

	float distance_travelled = length(oldPosition - currentPosition);
	float r = float(distance_travelled) + 4.0;
	r *= (1.0 - smoothness) * 0.8;

	color = pow(texture2DLod(gcolor, finalSamplePos, r), vec4(2.2));

	#ifdef GODRAYS
		color.a = 1.0;
	#endif

	#ifdef VOLUMETRIC_LIGHT
		color.a = 1.0;
	#endif

	if (finalSamplePos.x == 0.0 || finalSamplePos.y == 0.0) {
		color.a = 0.0;
	}

	float ndotv = dot(normal, old_camera_space_view_vector);
	ndotv = clamp(ndotv, 0.0, 1.0);
	vec3 fresnel_ray = fresnel(base_reflectivity, ndotv);

	color.rgb *= 0.5;
	color.rgb = ((vec3(1.0) - fresnel_ray) * fragcolor * (1.0 - metallic)) + (color.rgb * fresnel_ray * smoothness);

	color.a *= clamp(1 - pow(distance(vec2(0.5), finalSamplePos)*2.0, 2.0), 0.0, 1.0);
  	return color;
}

float CalculateLuminance(in vec3 color) {
	return (color.r * 0.2126 + color.g * 0.7152 + color.b * 0.0722);
}

float CalculateSunglow(in SurfaceStruct surface) {
	float curve = 4.0;

	vec3 npos = normalize(surface.viewSpacePosition.xyz);
	vec3 halfVector2 = normalize(-surface.lightVector + npos);
	float factor = 1.0 - dot(halfVector2, npos);

	return factor * factor * factor * factor;
}

float CalculateReflectedSunglow(in SurfaceStruct surface) {
	float curve = 4.0;

	vec3 npos = normalize(surface.viewSpacePosition.xyz);
	surface.lightVector = reflect(surface.lightVector, surface.normal);
	vec3 halfVector2 = normalize(-surface.lightVector + npos);
	float factor = 1.0 - dot(halfVector2, npos);

	return factor * factor * factor * factor;
}

float CalculateAntiSunglow(in SurfaceStruct surface) {
	float curve = 4.0;

	vec3 npos = normalize(surface.viewSpacePosition.xyz);
	vec3 halfVector2 = normalize(surface.lightVector + npos);
	float factor = 1.0 - dot(halfVector2, npos);

	return factor * factor * factor * factor;
}

float CalculateSunspot(in SurfaceStruct surface) {
	float curve = 1.0;

	vec3 npos = normalize(surface.viewSpacePosition.xyz);
	vec3 halfVector2 = normalize(-surface.lightVector + npos);

	float sunProximity = abs(1.0 - dot(halfVector2, npos));

	float sizeFactor = surface.smoothness * 0.9;

	float sunSpot = (clamp(sunProximity, sizeFactor, 0.96) - sizeFactor) / (0.96 - sizeFactor);
	sunSpot = pow(cubicPulse(1.0, 1.0, sunSpot), 2.0);

	float result = sunSpot / ((1.0 - surface.smoothness) * 20.0 + 0.1);
	result *= surface.sunlightVisibility;

	return result;
}

vec3 ComputeReflectedSkyGradient(in SurfaceStruct surface) {
	float curve = 5.0;
	surface.viewSpacePosition.xyz = reflect(surface.viewSpacePosition.xyz, surface.normal);
	vec3 npos = normalize(surface.viewSpacePosition.xyz);

	vec3 halfVector2 = normalize(-surface.upVector + npos);
	float skyGradientFactor = dot(halfVector2, npos);
	float skyGradientRaw = skyGradientFactor;
	float skyDirectionGradient = skyGradientFactor;

	if (dot(halfVector2, npos) > 0.75)
		skyGradientFactor = 1.5 - skyGradientFactor;

	skyGradientFactor = pow(skyGradientFactor, curve);

	vec3 skyColor = CalculateLuminance(pow(gl_Fog.color.rgb, vec3(2.2))) * colorSkylight;

	skyColor *= mix(skyGradientFactor, 1.0, clamp((0.12 - (timeNoon * 0.1)) + rainStrength, 0.0, 1.0));
	skyColor *= pow(skyGradientFactor, 2.5) + 0.2;
	skyColor *= (pow(skyGradientFactor, 1.1) + 0.425) * 0.5;
	skyColor.g *= skyGradientFactor * 3.0 + 1.0;

	vec3 linFogColor = pow(gl_Fog.color.rgb, vec3(2.2));

	float fogLum = max(max(linFogColor.r, linFogColor.g), linFogColor.b);

	float fadeSize = 0.0;

	float fade1 = clamp(skyGradientFactor - 0.05 - fadeSize, 0.0, 0.2 + fadeSize) / (0.2 + fadeSize);
	fade1 = fade1 * fade1 * (3.0 - 2.0 * fade1);
	vec3 color1 = vec3(5.0, 2.0, 0.7) * 0.25;
	color1 = mix(color1, vec3(1.0, 0.55, 0.2), vec3(timeSunrise + timeSunset));

	skyColor *= mix(vec3(1.0), color1, vec3(fade1));

	float fade2 = clamp(skyGradientFactor - 0.11 - fadeSize, 0.0, 0.2 + fadeSize) / (0.2 + fadeSize);
	vec3 color2 = vec3(1.7, 1.0, 0.8) * 0.5;
	color2 = mix(color2, vec3(1.0, 0.15, 0.5), vec3(timeSunrise + timeSunset));

	skyColor *= mix(vec3(1.0), color2, vec3(fade2 * 0.5));

	float horizonGradient = 1.0 - distance(skyDirectionGradient, 0.72 + fadeSize) / (0.72 + fadeSize);
	horizonGradient = pow(horizonGradient, 10.0);
	horizonGradient = max(0.0, horizonGradient);

	float sunglow = CalculateSunglow(surface);
	horizonGradient *= sunglow * 2.0+ (0.65 - timeSunrise * 0.55 - timeSunset * 0.55);

	vec3 horizonColor1 = vec3(1.5, 1.5, 1.5);
	horizonColor1 = mix(horizonColor1, vec3(1.5, 1.95, 0.5) * 2.0, vec3(timeSunrise + timeSunset));

	vec3 horizonColor2 = vec3(1.5, 1.2, 0.8) * 1.0;
	horizonColor2 = mix(horizonColor2, vec3(1.9, 0.6, 0.4) * 2.0, vec3(timeSunrise + timeSunset));

	skyColor *= mix(vec3(1.0), horizonColor1, vec3(horizonGradient) * (1.0 - timeMidnight));
	skyColor *= mix(vec3(1.0), horizonColor2, vec3(pow(horizonGradient, 2.0)) * (1.0 - timeMidnight));

	float grayscale = fogLum / 20.0;
	grayscale /= 3.0;

	float rainSkyBrightness = 1.2;
	rainSkyBrightness *= mix(0.05, 10.0, timeMidnight);

	skyColor = mix(skyColor, vec3(grayscale * colorSkylight.r) * 0.06 * vec3(0.85, 0.85, 1.0), vec3(rainStrength));
	skyColor /= fogLum;

	float antiSunglow = CalculateAntiSunglow(surface);

	skyColor *= 1.0 + pow(sunglow, 1.1) * (7.0 + timeNoon * 1.0) * (1.0 - rainStrength);
	skyColor *= mix(vec3(1.0), colorSunlight * 11.0, clamp(vec3(sunglow) * (1.0 - timeMidnight) * (1.0 - rainStrength), vec3(0.0), vec3(1.0)));
	skyColor *= 1.0 + antiSunglow * 2.0 * (1.0 - rainStrength);

	if (surface.mask.water) {
		vec3 sunspot = vec3(CalculateSunspot(surface)) * colorSunlight;
		sunspot *= 50.0;
		sunspot *= 1.0 - timeMidnight;
		sunspot *= 1.0 - rainStrength;

		skyColor += sunspot;
	}

	skyColor *= pow(1.0 - clamp(skyGradientRaw - 0.75, 0.0, 0.25) / 0.25, 3.0);
	skyColor *= mix(1.0, 4.5, timeNoon);

	return skyColor;
}

Intersection RayPlaneIntersectionWorld(in Ray ray, in Plane plane) {
	float rayPlaneAngle = dot(ray.dir, plane.normal);

	float planeRayDist = 100000000.0;
	vec3 intersectionPos = ray.dir * planeRayDist;

	if (rayPlaneAngle > 0.0001 || rayPlaneAngle < -0.0001) {
		planeRayDist = dot((plane.origin), plane.normal) / rayPlaneAngle;
		intersectionPos = ray.dir * planeRayDist;
		intersectionPos = -intersectionPos;

		intersectionPos += cameraPosition.xyz;
	}

	Intersection i;

	i.pos = intersectionPos;
	i.distance = planeRayDist;
	i.angle = rayPlaneAngle;

	return i;
}

float GetCoverage(in float coverage, in float density, in float clouds) {
	clouds = clamp(clouds - (1.0 - coverage), 0.0, 1.0 - density) / (1.0 - density);
	clouds = max(0.0, clouds * 1.1 - 0.1);
	clouds = clouds = clouds * clouds * (3.0 - 2.0 * clouds);

	return clouds;
}

vec4 CloudColor2(in vec4 worldPosition, in float sunglow, in vec3 worldLightVector, in float altitude, in float thickness, const bool isShadowPass) {
	float cloudHeight = altitude;
	float cloudDepth  = thickness;
	float cloudUpperHeight = cloudHeight + (cloudDepth / 2.0);
	float cloudLowerHeight = cloudHeight - (cloudDepth / 2.0);

	worldPosition.xz /= 1.0 + max(0.0, length(worldPosition.xz - cameraPosition.xz) / 5000.0);

	vec3 p = worldPosition.xyz / 150.0;
	float t = frameTimeCounter * 0.25;

	p += (Get3DNoise(p * 2.0 + vec3(0.0, t * 0.01, 0.0)) * 2.0 - 1.0) * 0.1;
	p.z -= (Get3DNoise(p * 0.25 + vec3(0.0, t * 0.01, 0.0)) * 2.0 - 1.0) * 0.5;
	p.x -= (Get3DNoise(p * 0.125 + vec3(0.0, t * 0.01, 0.0)) * 2.0 - 1.0) * 1.2;
	p.xz -= (Get3DNoise(p * 0.0525 + vec3(0.0, t * 0.01, 0.0)) * 2.0 - 1.0) * 1.7;


	p.x *= 0.5;
	p.x -= t * 0.01;

	vec3 p1 = p * vec3(1.0, 0.5, 1.0)  + vec3(0.0, t * 0.01, 0.0);
	float noise  = 	Get3DNoise(p * vec3(1.0, 0.5, 1.0) + vec3(0.0, t * 0.01, 0.0));	p *= 2.0;	p.x -= t * 0.057;	vec3 p2 = p;
	noise += (2.0 - abs(Get3DNoise(p) * 2.0 - 0.0)) * (0.55);	p *= 3.0;	p.xz -= t * 0.035;	p.x *= 2.0;	vec3 p3 = p;
	noise += (3.0 - abs(Get3DNoise(p) * 3.0 - 0.0)) * (0.085); p *= 3.0;	p.xz -= t * 0.035;	vec3 p4 = p;
	noise += (3.0 - abs(Get3DNoise(p) * 3.0 - 0.0)) * (0.045); p *= 3.0;	p.xz -= t * 0.035;

	if (!isShadowPass) {
		noise += ((Get3DNoise(p))) * (0.049);
		p *= 3.0;
		noise += ((Get3DNoise(p))) * (0.024);
	}

	noise /= 2.175;

	float next_moon_phase = moonPhase + 1;

	if(float(moonPhase) == 7) {
		next_moon_phase = 0;
	}

	float moon_phase_smooth = mix(moonPhase, next_moon_phase, float(worldTime) / 24000.0);

	float dynWeather = ((abs(float(moon_phase_smooth) - 3) + 1) / 5) + 0.5;

	//cloud edge
	float rainy = mix(wetness, 1.0, rainStrength);

	float coverage = CLOUD_PLANE_Coverage + rainy * 0.335;
	coverage = min(mix(coverage * dynWeather, 0.87, rainStrength), 0.87);

	float dist = length(worldPosition.xz - cameraPosition.xz);
	coverage *= max(0.0, 1.0 - dist / mix(7000.0, 3000.0, rainStrength));

	float density = 0.0;

	if (isShadowPass) {
		return vec4(GetCoverage(coverage + 0.2, density + 0.2, noise));
	}

	noise = GetCoverage(coverage, density, noise);

	const float lightOffset = 0.2;

	float sundiff = Get3DNoise(p1 + worldLightVector.xyz * lightOffset);
	sundiff += (2.0 - abs(Get3DNoise(p2 + worldLightVector.xyz * lightOffset / 2.0) * 2.0 - 0.0)) * (0.55);

	float largeSundiff = sundiff;
	largeSundiff = -GetCoverage(coverage, 0.0, largeSundiff * 1.3);

	sundiff += (3.0 - abs(Get3DNoise(p3 + worldLightVector.xyz * lightOffset / 5.0) * 3.0 - 0.0)) * (0.055);
	sundiff += (3.0 - abs(Get3DNoise(p4 + worldLightVector.xyz * lightOffset / 8.0) * 3.0 - 0.0)) * (0.025);
	sundiff /= 1.5;
	sundiff = -GetCoverage(coverage * 1.0, 0.0, sundiff);

	float secondOrder 	= pow(clamp(sundiff * 1.1 + 1.35, 0.0, 1.0), 7.0);
	float firstOrder 	= pow(clamp(largeSundiff * 1.1 + 1.56, 0.0, 1.0), 3.0);

	float directLightFalloff = firstOrder * secondOrder;
	float anisoBackFactor = mix(clamp(pow(noise, 1.6) * 2.5, 0.0, 1.0), 1.0, pow(sunglow, 1.0));

	directLightFalloff *= anisoBackFactor;
	directLightFalloff *= mix(1.5, 1.0, pow(sunglow, 0.5));



	vec3 colorDirect = colorSunlight * 100.915;
	colorDirect = mix(colorDirect, colorDirect * vec3(0.2, 0.5, 1.0), timeMidnight);
	colorDirect *= 1.0 + pow(sunglow, 5.0) * 600.0 * pow(directLightFalloff, 1.1) * (1.0 - rainStrength);


	vec3 colorAmbient = mix(colorSkylight, colorSunlight * 2.0, vec3(0.15)) * 0.07;
	colorAmbient *= mix(1.0, 0.3, timeMidnight);
	colorAmbient = mix(colorAmbient, colorAmbient * 3.0 + colorSunlight * 0.05, vec3(clamp(pow(1.0 - noise, 12.0) * 1.0, 0.0, 1.0)));


	directLightFalloff *= 2.0;
	directLightFalloff *= mix(1.0, 0.125, rainStrength);

	vec3 color = mix(colorAmbient, colorDirect, vec3(min(1.0, directLightFalloff)));

	vec4 result = vec4(color.rgb, noise);

	return result;
}

void ReflectedCloudPlane(inout vec3 color, in SurfaceStruct surface) {
	//Initialize view ray
	vec3 viewVector = normalize(surface.viewSpacePosition.xyz);
	viewVector = reflect(viewVector, surface.normal.xyz);
	vec4 worldVector = gbufferModelViewInverse * (vec4(-viewVector.xyz, 0.0));

	surface.viewRay.dir = normalize(worldVector.xyz);
	surface.viewRay.origin = vec3(0.0);

	float sunglow = CalculateReflectedSunglow(surface);

	float cloudsAltitude = 540.0;
	float cloudsThickness = 150.0;

	float cloudsUpperLimit = cloudsAltitude + cloudsThickness * 0.5;
	float cloudsLowerLimit = cloudsAltitude - cloudsThickness * 0.5;

	float density = 1.0;

	float planeHeight = cloudsUpperLimit;
	float stepSize = 25.5;
	planeHeight -= cloudsThickness * 0.85;

	Plane pl;
	pl.origin = vec3(0.0, cameraPosition.y - planeHeight, 0.0);
	pl.normal = vec3(0.0, 1.0, 0.0);

	Intersection i = RayPlaneIntersectionWorld(surface.viewRay, pl);

	if (i.angle < 0.0) {
		vec4 cloudSample = CloudColor2(vec4(i.pos.xyz * 0.5 + vec3(30.0), 1.0), sunglow, surface.worldLightVector, cloudsAltitude, cloudsThickness, false);
		cloudSample.a = min(1.0, cloudSample.a * density);

		color.rgb = mix(color.rgb, cloudSample.rgb * 0.18, cloudSample.a);

		cloudSample = CloudColor2(vec4(i.pos.xyz * 0.65 + vec3(10.0) + vec3(i.pos.z * 0.5, 0.0, 0.0), 1.0), sunglow, surface.worldLightVector, cloudsAltitude, cloudsThickness, false);
		cloudSample.a = min(1.0, cloudSample.a * density);

		color.rgb = mix(color.rgb, cloudSample.rgb * 0.18, cloudSample.a);
	}
}

void CloudPlane(inout SurfaceStruct surface) {
	//Initialize view ray
	vec4 worldVector = gbufferModelViewInverse * (-GetViewSpacePosition(texcoord.st));

	surface.viewRay.dir = normalize(worldVector.xyz);
	surface.viewRay.origin = vec3(0.0);

	float sunglow = CalculateSunglow(surface);

	float cloudsAltitude = 540.0;
	float cloudsThickness = 150.0;

	float cloudsUpperLimit = cloudsAltitude + cloudsThickness * 0.5;
	float cloudsLowerLimit = cloudsAltitude - cloudsThickness * 0.5;

	float density = 1.0;

	float planeHeight = cloudsUpperLimit;
	float stepSize = 25.5;
	planeHeight -= cloudsThickness * 0.85;

	Plane pl;
	pl.origin = vec3(0.0, cameraPosition.y - planeHeight, 0.0);
	pl.normal = vec3(0.0, 1.0, 0.0);

	Intersection i = RayPlaneIntersectionWorld(surface.viewRay, pl);

	vec3 original = surface.color.rgb;

	if (i.angle < 0.0) {
		if (i.distance < surface.linearDepth || surface.mask.sky) {
			vec4 cloudSample = CloudColor2(vec4(i.pos.xyz * 0.5 + vec3(30.0), 1.0), sunglow, surface.worldLightVector, cloudsAltitude, cloudsThickness, false);
			cloudSample.a = min(1.0, cloudSample.a * density);

			surface.color.rgb = mix(surface.color.rgb, cloudSample.rgb * 0.001, cloudSample.a);

			cloudSample = CloudColor2(vec4(i.pos.xyz * 0.65 + vec3(210.0) + vec3(i.pos.z * 0.5, 0.0, 0.0), 1.0), sunglow, surface.worldLightVector, cloudsAltitude, cloudsThickness, false);
			cloudSample.a = min(1.0, cloudSample.a * density);

			surface.color.rgb = mix(surface.color.rgb, cloudSample.rgb * 0.001, cloudSample.a);
		}
	}

	surface.color.rgb = mix(surface.color.rgb, original, surface.cloudAlpha);
}

vec3 ComputeFakeSkyReflection(in SurfaceStruct surface) {

	vec3 cameraSpacePosition = convertScreenSpaceToWorldSpace(texcoord.st);
	vec3 cameraSpaceViewDir = normalize(cameraSpacePosition);
	vec3 color = vec3(0.0);

	color = ComputeReflectedSkyGradient(surface);
	ReflectedCloudPlane(color.rgb, surface);
	color *= 0.006;
	color *= mix(1.0, 20000.0, timeSkyDark);

	float ndotv = dot(surface.normal, reflect(cameraSpaceViewDir, surface.normal));
	vec3 fresnel_ray = fresnel(surface.baseReflectivity, ndotv);

	color = ((vec3(1.0) - fresnel_ray) * surface.color * (1.0 - surface.metallic)) + (color * fresnel_ray * surface.smoothness);

	return color;
}

void CalculatePBRReflections(inout SurfaceStruct surface) {
	surface.baseReflectivity = mix(vec3(0.14), surface.color * 5000000, surface.metallic);

	bool defaultItself = true;

	surface.rDepth = 0.0;

	if(surface.mask.sky) {
		return;
	}

	if(surface.mask.water) {
		surface.smoothness = 1.0;
	}

	vec4 reflection = vec4(0);

	#if NUM_RAYS > 0
		for(int i = 0; i < NUM_RAYS; i++) {
			vec3 reflect_normal = surface.normal;
			reflection += ComputeRaytraceReflection(reflect_normal, surface.smoothness, surface.baseReflectivity, surface.metallic, surface.color);
		}
		reflection /= float(NUM_RAYS);

	#endif


	float surfaceLightmap = GetLightmapSky(texcoord.st);
		surfaceLightmap += GetSolidSunlightVisibility(texcoord.st) * (1-rainStrength);

	vec3 fakeSkyReflection = ComputeFakeSkyReflection(surface);

	vec3 noSkyToReflect = vec3(0.0);

	if (defaultItself) {
		noSkyToReflect = surface.color.rgb;
	}

	fakeSkyReflection.rgb = mix(noSkyToReflect, fakeSkyReflection.rgb, clamp(surfaceLightmap * 16 - 5, 0.0, 1.0));	// Occlude the sky when MC has no sky lighting
	reflection.rgb = mix(reflection.rgb, fakeSkyReflection.rgb, pow(vec3(1.0 - reflection.a), vec3(10.1)));			// Mix in the sky reflections when the raytraced reflection can't resolve

	surface.color.rgb = reflection.rgb;
	surface.reflection = reflection;
}

void CalculatePBRHighlight(inout SurfaceStruct surface) {
	if (!surface.mask.sky && !surface.mask.water) {

		vec3 halfVector = normalize(lightVector - normalize(surface.viewSpacePosition.xyz));
		float HdotN = max(0.0, dot(halfVector, surface.normal.xyz));

		float gloss = surface.smoothness;

		HdotN = clamp(HdotN * (1.0 + gloss * 0.01), 0.0, 1.0);

		vec3 spec = vec3(pow(HdotN, gloss * 8000.0 + 10.0));

		vec3 fresnel = fresnel(surface.baseReflectivity, HdotN);

		spec *= fresnel;
		spec *= surface.sunlightVisibility;

		spec *= gloss * 9000.0 + 10.0;
		spec *= 1.0 - rainStrength;

		vec3 specularHighlight = spec * mix(colorSunlight, vec3(0.2, 0.5, 1.0) * 0.0005, vec3(timeMidnight));

		surface.color += specularHighlight * 0.000005;
	}
}

vec4 TextureSmooth(in sampler2D tex, in vec2 coord, in float level) {
	vec2 res = vec2(viewWidth, viewHeight);
	coord = coord * res + 0.5;

	vec2 i = floor(coord);
	vec2 f = fract(coord);

	f = f * f * (3.0 - 2.0 * f);

	coord = i + f;
	coord = (coord - 0.5) / res;

	return texture2D(tex, coord, level);
}

void SmoothSky(inout SurfaceStruct surface) {
	const float cloudHeight = 170.0;
	const float cloudDepth = 60.0;
	const float cloudMaxHeight = cloudHeight + cloudDepth * 0.5;
	const float cloudMinHeight = cloudHeight - cloudDepth * 0.5;

	float cameraHeight = cameraPosition.y;
	float surfaceHeight = surface.worldSpacePosition.y;

	vec3 combined = pow(TextureSmooth(gcolor, texcoord.st, 2).rgb, vec3(2.2));

	vec3 original = surface.color;

	if (surface.cloudAlpha > 0.0001) {
		surface.color = combined;
	}

	if (cameraHeight < cloudMinHeight && surfaceHeight < cloudMinHeight - 10.0 && surface.mask.land) {
		surface.color = original;
	}

	if (cameraHeight > cloudMaxHeight && surfaceHeight > cloudMaxHeight && surface.mask.land) {
		surface.color = original;
	}
}

void FixNormals(inout vec3 normal, in vec3 viewPosition) {
	vec3 V = normalize(viewPosition.xyz);
	vec3 N = normal;

	float NdotV = dot(N, V);

	N = normalize(mix(normal, -V, clamp(pow((NdotV * 1.0), 1.0), 0.0, 1.0)));
	N = normalize(N + -V * 0.1 * clamp(NdotV + 0.4, 0.0, 1.0));

	normal = N;
}

float getnoise(vec2 pos) {
	return abs(fract(sin(dot(pos , vec2(18.9898,28.633))) * 4378.5453));
}

vec4 textureSmooth(in sampler2D tex, in vec2 coord) {
	vec2 res = vec2(64.0, 64.0);

	coord *= res;
	coord += 0.5;

	vec2 whole = floor(coord);
	vec2 part  = fract(coord);

	part.x = part.x * part.x * (3.0 - 2.0 * part.x);
	part.y = part.y * part.y * (3.0 - 2.0 * part.y);

	coord = whole + part;

	coord -= 0.5;
	coord /= res;

	return texture2D(tex, coord);
}

float AlmostIdentity(in float x, in float m, in float n) {
	if (x > m) return x;

	float a = 2.0 * n - m;
	float b = 2.0 * m - 3.0 * n;
	float t = x / m;

	return (a * t + b) * t * t + n;
}

float GetWaves(vec3 position) {
	float speed = 0.7;

  vec2 p = position.xz / 20.0;
  p.xy -= position.y / 20.0;
  p.x = -p.x;

  p.x += (FRAME_TIME / 40.0) * speed;
  p.y -= (FRAME_TIME / 40.0) * speed;

  float weight = 1.0;
  float weights = weight;

  float allwaves = 0.0;

  float wave = textureSmooth(noisetex, (p * vec2(2.0, 1.2))  + vec2(0.0,  p.x * 2.1) ).x; 			p /= 2.1; 	/*p *= pow(2.0, 1.0);*/ 	p.y -= (FRAME_TIME / 20.0) * 0.6; p.x -= (FRAME_TIME / 30.0) * speed;
  allwaves += wave;

  weight = 4.1;
  weights += weight;
  wave = textureSmooth(noisetex, (p * vec2(2.0, 1.4))  + vec2(0.0,  -p.x * 2.1) ).x;	p /= 1.5;/*p *= pow(2.0, 2.0);*/ 	p.x += (FRAME_TIME / 20.0) * speed;
  wave *= weight;
  allwaves += wave;

  weight = 17.25;
  weights += weight;
  wave = (textureSmooth(noisetex, (p * vec2(1.0, 0.75))  + vec2(0.0,  p.x * 1.1) ).x);		p /= 1.5; 	p.x -= (FRAME_TIME / 55.0) * speed;
  wave *= weight;
  allwaves += wave;

  weight = 15.25;
  weights += weight;
  wave = (textureSmooth(noisetex, (p * vec2(1.0, 0.75))  + vec2(0.0,  -p.x * 1.7) ).x);		p /= 1.9; 	p.x += (FRAME_TIME / 155.0) * 0.8;
  wave *= weight;
  allwaves += wave;

  weight = 29.25;
  weights += weight;
  wave = abs(textureSmooth(noisetex, (p * vec2(1.0, 0.8))  + vec2(0.0,  -p.x * 1.7) ).x * 2.0 - 1.0);		p /= 2.0; 	p.x += (FRAME_TIME / 155.0) * speed;
  wave = 1.0 - AlmostIdentity(wave, 0.2, 0.1);
  wave *= weight;
  allwaves += wave;

  allwaves /= weights;

  return allwaves;
}

vec3 GetWavesNormal(vec3 position) {

	float WAVE_HEIGHT = 1.0;

	const float sampleDistance = 11.0;

	position -= vec3(0.005, 0.0, 0.005) * sampleDistance;

	float wavesCenter = GetWaves(position);
	float wavesLeft = GetWaves(position + vec3(0.01 * sampleDistance, 0.0, 0.0));
	float wavesUp   = GetWaves(position + vec3(0.0, 0.0, 0.01 * sampleDistance));

	vec3 wavesNormal;
	wavesNormal.r = wavesCenter - wavesLeft;
	wavesNormal.g = wavesCenter - wavesUp;

	wavesNormal.r *= 10.0 * WAVE_HEIGHT / sampleDistance;
	wavesNormal.g *= 10.0 * WAVE_HEIGHT / sampleDistance;

	wavesNormal.b = sqrt(1.0 - wavesNormal.r * wavesNormal.r - wavesNormal.g * wavesNormal.g);
	wavesNormal.rgb = normalize(wavesNormal.rgb);

	return wavesNormal.rgb;
}

void WaterRefraction(inout SurfaceStruct surface) {
	if (surface.mask.water) {
		vec3 wavesNormal = GetWavesNormal(surface.worldSpacePosition.xyz + cameraPosition.xyz).xzy;
		float waterDepth = ExpToLinearDepth(texture2D(depthtex1, texcoord.st).x);
		float waterDepthDiff = waterDepth - surface.linearDepth;
		float refractAmount = saturate(waterDepthDiff / 1.0);

		vec4 wnv = gbufferModelView * vec4(wavesNormal.xyz, 0.0);
		vec3 wavesNormalView = normalize(wnv.xyz);
		vec4 nv = gbufferModelView * vec4(0.0, 1.0, 0.0, 0.0);
		nv.xyz = normalize(nv.xyz);

		wavesNormalView.xy -= nv.xy;

		float aberration = 0.15;
		float refractionAmount = 1.0;

		vec2 refractCoord0 = texcoord.st - wavesNormalView.xy * refractAmount * (refractionAmount) / (surface.linearDepth + 0.0001);
		vec2 refractCoord1 = texcoord.st - wavesNormalView.xy * refractAmount * (refractionAmount + aberration) / (surface.linearDepth + 0.0001);
		vec2 refractCoord2 = texcoord.st - wavesNormalView.xy * refractAmount * (refractionAmount + aberration * 2.0) / (surface.linearDepth + 0.0001);


		vec2 pixelSize = 1.0 / vec2(viewWidth, viewHeight);
		vec2 minCoord  = pixelSize;
		vec2 maxCoord  = 1.0 - pixelSize;

		refractCoord0 = clamp(refractCoord0, minCoord, maxCoord);
		refractCoord1 = clamp(refractCoord1, minCoord, maxCoord);
		refractCoord2 = clamp(refractCoord2, minCoord, maxCoord);


		float mask0 = 1 - GetWaterMaskFloat(refractCoord0);
		float mask1 = 1 - GetWaterMaskFloat(refractCoord1);
		float mask2 = 1 - GetWaterMaskFloat(refractCoord2);

		vec3 spillSample = pow(texture2D(gcolor, texcoord.xy).rgb, vec3(2.2));

		surface.color.r = pow(texture2DLod(gcolor, refractCoord0.xy, 0).r, (2.2)) * mask0 + spillSample.r * (1 - mask0);
		surface.color.g = pow(texture2DLod(gcolor, refractCoord1.xy, 0).g, (2.2)) * mask1 + spillSample.g * (1 - mask1);
		surface.color.b = pow(texture2DLod(gcolor, refractCoord2.xy, 0).b, (2.2)) * mask2 + spillSample.b * (1 - mask2);
	}
}

vec3 GetCrepuscularRays(in SurfaceStruct surface) {
	if (isEyeInWater > 0.9) {
		return vec3(0.0);
  }

	if (rainStrength > 0.9) {
		return vec3(0.0);
  }

	float vlSample = texture2DLod(gcolor, texcoord.st, 3.2).a;

	float sunglow = 1.0 - CalculateSunglow(surface);
  sunglow = 1.0 / (pow(sunglow, 1.0) * 9.0 + 0.001);
	sunglow += CalculateAntiSunglow(surface) * 0.2;

	vec3 raysSun = vec3(vlSample);
  vec3 raysSuns = vec3(vlSample);
  vec3 raysNight = vec3(vlSample);
	vec3 raysSunIn = vec3(vlSample);
	vec3 raysAtmosphere = vec3(vlSample);

	raysSunIn.rgb *= sunglow;
  raysSunIn.rgb *= colorSunlight;

  raysSun.rgb *= sunglow * timeNoon;
  raysSun.rgb *= colorSunlight * timeNoon;

  raysSuns.rgb *= sunglow * timeSunriseSunset;
  raysSuns.rgb *= colorSunlight * timeSunriseSunset;

  raysNight.rgb *= colorSunlight * timeMidnight;

  raysAtmosphere *= pow(colorSkylight, vec3(1.3));

	vec3 rays = raysSuns.rgb * SUNRISEnSET + raysAtmosphere.rgb * 0.5 * 0.0;
  rays += raysSun.rgb * NOON + raysAtmosphere.rgb * 0.5 * 0.0;
	rays += raysNight.rgb * NIGHT + raysAtmosphere.rgb * 0.5 * 0.0;


	vec3 Inrays = raysSunIn.rgb * IN_SIDE_RAYS + raysAtmosphere.rgb * 0.5 * 0.0;
	Inrays += raysNight.rgb * IN_SIDE_RAYS_NIGHT + raysAtmosphere.rgb * 0.5 * 0.0;
	Inrays *= mix(1.0, 0.0, pow(eyeBrightnessSmooth.y / 240.0, 3.0));


  DoNightEye(rays);

	return ((rays * 0.0058) * 0.0049 + Inrays * 0.00001) * pow(1 - rainStrength, 2.0);
}


vec3 Get2DGodraysRays(in SurfaceStruct surface) {
	vec4 tpos = vec4(sunPosition, 1.0) * gbufferProjection;
	tpos = vec4(tpos.xyz / tpos.w, 1.0);

	vec2 pos1 = tpos.xy / tpos.z;
	vec2 lightPos = pos1 * 0.5 + 0.5;
	float gr = 0.0;

	#ifdef NO_UNDERWATER_RAYS
		if (isEyeInWater > 0.9) {
			return vec3(0.0);
		}
	#endif

	if (rainStrength > 0.9) {
		return vec3(0.0);
	}

	float truepos = sign(sunPosition.z); //temporary fix that check if the sun/moon position is correct

	if (truepos < 0.05) {
		vec2 deltaTextCoord = vec2(texcoord.st - lightPos.xy);
		vec2 textCoord = texcoord.st;
		deltaTextCoord *= 1.0 / float(NUM_SAMPLES) * grdensity;
		float illuminationDecay = 1.0;
		gr = 0.0;
		float avgdecay = 0.0;
		float distx = abs(texcoord.x * aspectRatio - lightPos.x * aspectRatio);
		float disty = abs(texcoord.y - lightPos.y);
		illuminationDecay = pow(max(1.0 - sqrt(distx * distx + disty * disty), 0.0), 7.8);

		#ifdef VOLUMETRIC_LIGHT
			illuminationDecay *= mix(0.0, 1.0, pow(eyeBrightnessSmooth.y / 240.0, 3.0));
		#endif

		float fallof = 1.0;
		const int nSteps = 9;
		const float blurScale = 0.002;
		deltaTextCoord = normalize(deltaTextCoord);
		int center = (nSteps - 1) / 2;
		vec3 blur = vec3(0.0);
		float tw = 0.0;
		float sigma = 0.25;
		float A = 1.0/sqrt(2.0 * 3.14159265359 * sigma);
		textCoord -= deltaTextCoord * center * blurScale;

		for(int i=0; i < nSteps ; i++) {
			textCoord += deltaTextCoord * blurScale;
			float dist = (i - float(center)) / center;
			float weight = A * exp(-(dist * dist) / (4.0 * sigma));
			float sample = texture2D(gdepth, textCoord).a * weight;

			tw += weight;
			gr += sample;
		}

		return colorSunlight * exposure * (gr / tw) * pow(1 - rainStrength, 2.0) * illuminationDecay * timeNoon * 2 * pow(1 - rainStrength, 2.0);
	}
}

vec3 Get2DMoonGodraysRays(in SurfaceStruct surface) {
	vec4 tpos = vec4(sunPosition, 1.0) * gbufferProjection;
	tpos = vec4(tpos.xyz / tpos.w, 1.0);

	vec2 pos1 = tpos.xy / tpos.z;
	vec2 lightPos = pos1 * 0.5 + 0.5;
	float gr = 0.0;

	#ifdef NO_UNDERWATER_RAYS
		if (isEyeInWater > 0.9) {
			return vec3(0.0);
		}
	#endif

	if (rainStrength > 0.9) {
		return vec3(0.0);
	}

	float truepos = sign(sunPosition.z); //temporary fix that check if the sun/moon position is correct

	tpos = vec4(-sunPosition, 1.0) * gbufferProjection;
	tpos = vec4(tpos.xyz / tpos.w, 1.0);
	pos1 = tpos.xy / tpos.z;
	lightPos = pos1 * 0.5 + 0.5;

	if (truepos > 0.05) {
		vec2 deltaTextCoord = vec2(texcoord.st - lightPos.xy);
		vec2 textCoord = texcoord.st;
		deltaTextCoord *= 1.0 / float(NUM_SAMPLES) * grdensity;
		float illuminationDecay = 1.0;

		gr = 0.0;
		float avgdecay = 0.0;
		float distx = abs(texcoord.x * aspectRatio - lightPos.x * aspectRatio);
		float disty = abs(texcoord.y - lightPos.y);
		illuminationDecay = pow(max(1.0 - sqrt(distx * distx + disty * disty), 0.0), 5.0);

		#ifdef VOLUMETRIC_LIGHT
			illuminationDecay *= mix(0.0, 1.0, pow(eyeBrightnessSmooth.y / 240.0, 3.0));
		#endif

		float fallof = 1.0;
		const int nSteps = 9;
		const float blurScale = 0.002;
		deltaTextCoord = normalize(deltaTextCoord);
		int center = (nSteps - 1) / 2;
		vec3 blur = vec3(0.0);
		float tw = 0.0;
		float sigma = 0.25;
		float A = 1.0 / sqrt(2.0 * 3.14159265359 * sigma);
		textCoord -= deltaTextCoord * center * blurScale;

		for(int i=0; i < nSteps ; i++) {
			textCoord += deltaTextCoord * blurScale;
			float dist = (i - float(center)) / center;
			float weight = A * exp(-(dist * dist) / (2.0 * sigma));
			float sample = texture2D(gdepth, textCoord).a * weight;

			tw += weight;
			gr += sample;
		}
		return 5.0 * colorSunlight * Moon_exposure * (gr / tw) * pow(1 - rainStrength, 2.0) * illuminationDecay / 2.5 * truepos * timeMidnight * pow(1 - rainStrength, 2.0);
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////MAIN/////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////
void main() {

	surface.color = pow(texture2DLod(gcolor, texcoord.st, 0).rgb, vec3(2.2));
	surface.normal = GetNormals(texcoord.st);
	surface.depth = GetDepth(texcoord.st);
	surface.linearDepth = ExpToLinearDepth(surface.depth); 				//Get linear scene depth
	surface.viewSpacePosition = GetViewSpacePosition(texcoord.st);
	surface.worldSpacePosition = gbufferModelViewInverse * surface.viewSpacePosition;
	FixNormals(surface.normal, surface.viewSpacePosition.xyz);
	surface.lightVector = lightVector;
	surface.sunlightVisibility = GetSolidSunlightVisibility(texcoord.st);
	surface.upVector 	= upVector;
	vec4 wlv = shadowModelViewInverse * vec4(0.0, 0.0, 0.0, 1.0);
	surface.worldLightVector 	= normalize(wlv.xyz);

	surface.metallic = GetMetallic(texcoord.st);
	surface.smoothness = GetSmoothness(texcoord.st);

	surface.mask.matIDs = GetMaterialIDs(texcoord.st);
	CalculateMasks(surface.mask);

	surface.cloudAlpha = 0.0;
	#ifdef SMOOTH_SKY
		surface.cloudAlpha = texture2D(composite, texcoord.st, 3).g;
		SmoothSky(surface);
	#endif

	#ifdef CLOUD_PLANE_Cloud
		CloudPlane(surface);
	#endif

	#ifdef Water_Refraction
		WaterRefraction(surface);
	#endif

	CalculatePBRReflections(surface);
	CalculatePBRHighlight(surface);

	#ifdef VOLUMETRIC_LIGHT
		surface.color.rgb += GetCrepuscularRays(surface);
	#endif

	#ifdef GODRAYS
		surface.color.rgb += Get2DGodraysRays(surface);
	#endif

	#ifdef MOONRAYS
		surface.color.rgb += Get2DMoonGodraysRays(surface);
	#endif

	surface.color = pow(surface.color, vec3(1.0 / 2.2));
	gl_FragData[0] = vec4(surface.color, 1.0);
}
