#version 120
#extension GL_ARB_shader_texture_lod : enable

/*
 _______ _________ _______  _______  _
(  ____ \\__   __/(  ___  )(  ____ )( )
| (    \/   ) (   | (   ) || (    )|| |
| (_____    | |   | |   | || (____)|| |
(_____  )   | |   | |   | ||  _____)| |
      ) |   | |   | |   | || (      (_)
/\____) |   | |   | (___) || )       _
\_______)   )_(   (_______)|/       (_)

This work is licensed under the Creative Commons Attribution-NonCommercial-NoDerivatives 4.0 International License.
 To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-nd/4.0/.

*/


/////////ADJUSTABLE VARIABLES//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////ADJUSTABLE VARIABLES//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#define SHADOW_MAP_BIAS 0.8

#define Global_Illumination
#define GI_FILTER_QUALITY 0.5 		//[0.5 1.5 2.5 3.5 4.5 5.5 6.5 7.5] //Sets the actual quality of the GI

//#define ENABLE_SOFT_SHADOWS
#define PCSS						// Contact-hardening shadows
//#define COLORED_PCSS				// Makes the PCSS shadows colored when the shadows are cast by stained glass. Currently horribly broken
#define USE_RANDOM_ROTATION			// Randomly rotates the shadow filter, removing shadow banding
//#define HSSRS //Brand new method created by dethraid, makes perfect hard shadows and is to be used with PCSS

//HSSRS Settings, note, playing with these can result in unwanted effects
#define MAX_RAY_LENGTH              0.8
#define NUM_RAY_STEPS               100
#define RAY_DEPTH_BIAS              0.05

#define Brightness 1.25				//[0.10 0.25 0.50 0.75 1.0 1.25 1.35 1.5]
#define Shadow_Brightness 0.55		//[0.05 0.1 0.25 0.35 0.45 0.55 0.63 0.7 1.0 3.0 5.0 10.0]

#define Torch_Brightness 0.008		//[0.005 0.008 0.01 0.04 0.06 0.08 0.1]

#define HELD_LIGHT				//Dynamic Torch Light when in player hand

#define CAVE_BRIGHTNESS	0.0007		//[0.0002 0.0005 0.0007 0.003]

#define RAIN_FOG
#define FOG_DENSITY	0.0030			//default is 0.0018 and is best if using RainFog2 from final[0.0018 0.0025 0.0030 0.0038]

#define ATMOSPHERIC_FOG
#define NO_ATMOSPHERIC_FOG_INSIDE		//removes distant fog in caves/buildings
#define MORNING_FOG
#define EVENING_FOG

//----------3D clouds----------//
#define VOLUMETRIC_CLOUDS
#define SOFT_FLUFFY_CLOUDS				// dissable to fully remove dither Pattern ripple, adds a little pixel noise on cloud edge
#define CLOUD_DISPERSE 10.0          // increase this for thicker clouds and so that they don't fizzle away when you fly close to them, 10 is default Dont Go Over 30 will lag and maybe crash

#define Cloud3Height 250				//[100 120 140 160 180 200 220 240 250] //Sets the Volumetric clouds3 Height

//----------New 2D clouds----------//

#define CLOUD_PLANE_Coverage 0.40		// CLOUD_PLANE_Coverage. 0.20 = Lowest Cover. 0.60 = Highest Cover [0.20 0.30 0.40 0.50 0.60 0.70]

//----------End CONFIGURABLE 2D Clouds----------//

//----------New Cloud Shadows----------//

#define CLOUD_SHADOW
#define DYNAMIC_WEATHER

//----------End CONFIGURABLE Cloud Shadows----------//

#define UnderwaterFog					//dissable for clear underwater

#define Water_DepthFog

//----------2D GodRays----------//
#define GODRAYS
	const float grdensity = 0.7;
	const int NUM_SAMPLES = 10;			//increase this for better quality at the cost of performance /10 is default
	const float grnoise = 1.0;		//amount of noise /1.0 is default

#define GODRAY_LENGTH 0.75			//default is 0.65, to increase the distance/length of the godrays at the cost of slight increase of sky brightness

#define MOONRAYS					//Make sure if you enable/disable this to do the same in Composite1, PLEASE NOTE Moonrays have a bug at sunset/sunrise

#define VOLUMETRIC_LIGHT			//True GodRays, not 2D ScreenSpace
//----------End CONFIGURABLE 2D GodRays----------//

////----------This feature is connected to ATMOSPHERIC_FOG----------//
//#define NEW_UNDERWATER

//#define GTX500_FIX					//disable this to fix strange colours on GTX 500 cards

/////////INTERNAL VARIABLES////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////INTERNAL VARIABLES////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Do not change the name of these variables or their type. The Shaders Mod reads these lines and determines values to send to the inner-workings
// of the shaders mod. The shaders mod only reads these lines and doesn't actually know the real value assigned to these variables in GLSL.
// Some of these variables are critical for proper operation. Change at your own risk.

const int shadowMapResolution       = 2048; // Shadow Resolution. 1024 = Lowest Quality. 4096 = Highest Quality [1024 2048 3072 4096]
const float shadowDistance          = 140;  // shadowDistance. 60 = Lowest Quality. 200 = Highest Quality [60 100 120 160 180 200]
const float shadowIntervalSize      = 4.0;
const bool shadowHardwareFiltering0 = true;

const bool shadowtex1Mipmap = true;
const bool shadowtex1Nearest = true;
const bool shadowcolor0Mipmap = true;
const bool shadowcolor0Nearest = false;
const bool shadowcolor1Mipmap = true;
const bool shadowcolor1Nearest = false;

const int RA8    = 0;
const int RA16   = 0;
const int RGA8   = 0;
const int RGBA8  = 0;
const int RGBA16 = 0;

const int gcolorFormat    = RGBA16;
const int gdepthFormat    = RGBA8;
const int gnormalFormat   = RGBA16;
const int compositeFormat = RGBA8;
const int gaux1Format     = RGBA16;
const int gaux3Format     = RGBA16;

const float eyeBrightnessHalflife = 10.0;
const float centerDepthHalflife   = 2.0;
const float wetnessHalflife       = 200.0;
const float drynessHalflife       = 40.0;

const float	sunPathRotation       = -40.0;
const float ambientOcclusionLevel = 0.5;

const int noiseTextureResolution  = 64;


//END OF INTERNAL VARIABLES//

/* DRAWBUFFERS:0136 */

const bool gaux1MipmapEnabled = true;
const bool gaux2MipmapEnabled = true;

#define BANDING_FIX_FACTOR 1.0

uniform sampler2D gcolor;
uniform sampler2D gdepth;
uniform sampler2D gdepthtex;
uniform sampler2D depthtex1;
uniform sampler2D gnormal;
uniform sampler2D composite;
uniform sampler2DShadow shadow;
uniform sampler2D shadowtex1;
uniform sampler2D shadowcolor0;
uniform sampler2D noisetex;
uniform sampler2D gaux1;
uniform sampler2D gaux2;
uniform sampler2D gaux3;

varying vec4 texcoord;
varying vec3 lightVector;
varying vec3 upVector;

uniform mat4 gbufferProjection;
uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferModelViewInverse;
uniform mat4 shadowProjection;
uniform mat4 shadowModelView;
uniform mat4 shadowModelViewInverse;
uniform vec3 sunPosition;
uniform vec3 cameraPosition;

uniform float near;
uniform float far;
uniform float viewWidth;
uniform float viewHeight;
uniform float rainStrength;
uniform float wetness;
uniform float aspectRatio;
uniform float frameTimeCounter;
uniform float sunAngle;

uniform int isEyeInWater;
uniform int worldTime;
uniform int moonPhase;
uniform ivec2 eyeBrightnessSmooth;

varying float timeSunrise;
varying float timeSunset;
varying float timeSunriseSunset;
varying float timeNoon;
varying float timeMidnight;
varying float timeSkyDark;
varying float transition_fading;

varying vec3 colorSunlight;
varying vec3 colorSkylight;
varying vec3 colorBouncedSunlight;
varying vec3 colorScatteredSunlight;
varying vec3 colorTorchlight;
varying vec3 colorWaterMurk;
varying vec3 colorWaterBlue;

uniform int heldBlockLightValue;

/////////////////////////FUNCTIONS/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////FUNCTIONS/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

float saturate(float x) {
	return clamp(x, 0.0, 1.0);
}

//Get gbuffer textures
vec3 GetAlbedoLinear(in vec2 coord) {			//Function that retrieves the diffuse texture and convert it into linear space.
	return pow(texture2D(gcolor, coord).rgb, vec3(2.2));
}

vec3 GetWaterNormals(in vec2 coord) {				//Function that retrieves the screen space surface normals. Used for lighting calculations
	return normalize(texture2DLod(gnormal, coord.st, 0).rgb * 2.0 - 1.0);
}

vec3 GetNormals(in vec2 coord) {				//Function that retrieves the screen space surface normals. Used for lighting calculations
	return normalize(texture2DLod(gaux2, coord.st, 0).rgb * 2.0 - 1.0);
}

float GetDepth(in vec2 coord) {					//Function that retrieves the scene depth. 0 - 1, higher values meaning farther away
	return texture2D(gdepthtex, coord).r;
}

float GetDepthSolid(in vec2 coord) {					//Function that retrieves the scene depth. 0 - 1, higher values meaning farther away
	return texture2D(depthtex1, coord).r;
}

float GetDepthLinear(in vec2 coord) {					//Function that retrieves the scene depth. 0 - 1, higher values meaning farther away
	return 2.0 * near * far / (far + near - (2.0 * texture2D(gdepthtex, coord).x - 1.0) * (far - near));
}

float ExpToLinearDepth(in float depth) {
	return 2.0 * near * far / (far + near - (depth * 2.0 - 1.0) * (far - near));
}

float getnoise(vec2 pos) {
	return abs(fract(sin(dot(pos ,vec2(18.9898,28.633))) * 4378.5453));
}

//Lightmaps
float 	GetLightmapTorch(in vec2 coord) {			//Function that retrieves the lightmap of light emitted by emissive blocks like torches and lava
	float lightmap = texture2D(gdepth, coord).g;

	//Apply inverse square law and normalize for natural light falloff
	lightmap 		= clamp(lightmap * 1.10, 0.0, 1.0);
	lightmap 		= 1.0 - lightmap;
	lightmap 		*= 5.6;
	lightmap 		= 1.0 / pow((lightmap + 0.8), 2.0);
	lightmap 		-= 0.02435;


	lightmap 		= max(0.0, lightmap);
	lightmap 		*= Torch_Brightness;
	lightmap 		= clamp(lightmap, 0.0, 1.0);
	lightmap 		= pow(lightmap, 0.9);
	return lightmap;
}

float 	GetLightmapSky(in vec2 coord) {			//Function that retrieves the lightmap of light emitted by the sky. This is a raw value from 0 (fully dark) to 1 (fully lit) regardless of time of day
	return pow(texture2D(gdepth, coord).b, 4.3);
}

//Specularity
float 	GetMetallic(in vec2 coord) {			//Function that retrieves how reflective any surface/pixel is in the scene. Used for reflections and specularity
	return texture2D(composite, coord.st).g;
}

float 	GetSmoothness(in vec2 coord) {			//Function that retrieves how reflective any surface/pixel is in the scene. Used for reflections and specularity
	return pow(texture2D(composite, coord.st).r, 2.2);
}

float	GetEmission(in vec2 coord) {
	return texture2D(composite, coord.st).b;
}

float 	GetAO(in vec2 coord) {
	return texture2D(composite, coord.st).a;
}

//Material IDs
float 	GetMaterialIDs(in vec2 coord) {			//Function that retrieves the texture that has all material IDs stored in it
	return texture2D(gdepth, coord).r;
}

float 	GetMaterialMask(in vec2 coord ,const in int ID, in float matID) {
	matID = (matID * 255.0);

	//Catch last part of sky
	if (matID > 254.0) {
		matID = 0.0;
	}

	if (matID == ID) {
		return 1.0;
	} else {
		return 0.0;
	}
}

float GetWaterMask(in vec2 coord, in float matID) { //Function that returns "true" if a pixel is water, and "false" if a pixel is not water.
	matID = (matID * 255.0);

	if (matID >= 35.0 && matID <= 51) {
		return 1.0;
	} else {
		return 0.0;
	}
}

//Surface calculations
vec4 GetScreenSpacePosition(in vec2 coord) { //Function that calculates the screen-space position of the objects in the scene using the depth texture and the texture coordinates of the full-screen quad
	float depth = GetDepth(coord);
	depth += float(GetMaterialMask(coord, 5, GetMaterialIDs(coord))) * 0.38;

	vec4 fragposition = gbufferProjectionInverse * vec4(coord.s * 2.0 - 1.0, coord.t * 2.0 - 1.0, 2.0 * depth - 1.0, 1.0);
	fragposition /= fragposition.w;

	return fragposition;
}

vec4 GetScreenSpacePositionSolid(in vec2 coord, in float solidDepth) { //Function that calculates the screen-space position of the objects in the scene using the depth texture and the texture coordinates of the full-screen quad
	solidDepth += float(GetMaterialMask(coord, 5, GetMaterialIDs(coord))) * 0.38;

	vec4 fragposition = gbufferProjectionInverse * vec4(vec3(coord, solidDepth) * 2.0 - 1.0, 1.0);
	     fragposition /= fragposition.w;

	return fragposition;
}

vec4 GetScreenSpacePosition(in vec2 coord, in float depth) { //Function that calculates the screen-space position of the objects in the scene using the depth texture and the texture coordinates of the full-screen quad
	vec4 fragposition = gbufferProjectionInverse * vec4(coord.s * 2.0 - 1.0, coord.t * 2.0 - 1.0, 2.0 * depth - 1.0, 1.0);
	fragposition /= fragposition.w;

	return fragposition;
}

vec3 getCameraSpacePosition(vec2 coord, in float depth) {
	vec4 fragposition = gbufferProjectionInverse * vec4(coord.s * 2.0 - 1.0, coord.t * 2.0 - 1.0, 2.0 * depth - 1.0, 1.0);
		 fragposition /= fragposition.w;
	return fragposition.xyz;
}

vec2 getCoordFromCameraSpace(in vec3 position) {
    vec4 viewSpacePosition1 = gbufferProjection * vec4(position, 1);
    vec2 ndcSpacePosition = viewSpacePosition1.xy / viewSpacePosition1.w;
    return ndcSpacePosition * 0.5 + 0.5;
}

vec4 	GetCloudSpacePosition(in vec2 coord, in float depth, in float distanceMult) {
	float linDepth = depth;

	float expDepth = (far * (linDepth - near)) / (linDepth * (far - near));

	//Convert texture coordinates and depth into view space
	vec4 viewPos = gbufferProjectionInverse * vec4(coord.s * 2.0 - 1.0, coord.t * 2.0 - 1.0, 2.0 * expDepth - 1.0, 1.0);
	viewPos /= viewPos.w;

	//Convert from view space to world space
	vec4 worldPos = gbufferModelViewInverse * viewPos;

	worldPos.xyz *= distanceMult;
	worldPos.xyz += cameraPosition.xyz;

	return worldPos;
}

void 	DoNightEye(inout vec3 color) {			//Desaturates any color input at night, simulating the rods in the human eye
	float amount = 0.8; 						//How much will the new desaturated and tinted image be mixed with the original image
	vec3 rodColor = vec3(0.2, 0.5, 1.25); 	//Cyan color that humans percieve when viewing extremely low light levels via rod cells in the eye
	float colorDesat = dot(color, vec3(1.0)); 	//Desaturated color

	color = mix(color, vec3(colorDesat) * rodColor, timeSkyDark * amount);
}

float 	LinearToExponentialDepth(in float linDepth) {
	return (far * (linDepth - near)) / (linDepth * (far - near));
}

void 	DoLowlightEye(inout vec3 color) {			//Desaturates any color input at night, simulating the rods in the human eye
	float amount = 0.8; 						//How much will the new desaturated and tinted image be mixed with the original image
	vec3 rodColor = vec3(0.2, 0.5, 1.0); 	//Cyan color that humans percieve when viewing extremely low light levels via rod cells in the eye
	float colorDesat = dot(color, vec3(1.0)); 	//Desaturated color

	color = mix(color, vec3(colorDesat) * rodColor, amount);
}

float 	CalculateLuminance(in vec3 color) {
	return (color.r * 0.2126 + color.g * 0.7152 + color.b * 0.0722);
}

vec3 	Glowmap(in vec3 albedo, in float mask, in float curve, in vec3 emissiveColor) {
	vec3 color = albedo * (mask);
	color = pow(color, vec3(curve));
	color = vec3(CalculateLuminance(color));
	color *= emissiveColor;

	return color;
}

float  	CalculateDitherPattern() {
	const int[4] ditherPattern = int[4] (0, 2, 1, 4);

	vec2 count = vec2(0.0);
	count.x = floor(mod(texcoord.s * viewWidth, 2.0));
	count.y = floor(mod(texcoord.t * viewHeight, 2.0));

	int dither = ditherPattern[int(count.x) + int(count.y) * 2];

	return float(dither) / 4.0;
}

float  	CalculateDitherPattern1() {
	const int[16] ditherPattern = int[16] (0 , 8 , 2 , 10,
									 	   12, 4 , 14, 6 ,
									 	   3 , 11, 1,  9 ,
									 	   15, 7 , 13, 5 );

	vec2 count = vec2(0.0);
	count.x = floor(mod(texcoord.s * viewWidth, 4.0));
	count.y = floor(mod(texcoord.t * viewHeight, 4.0));

	int dither = ditherPattern[int(count.x) + int(count.y) * 4];

	return float(dither) / 16.0;
}

float  	CalculateDitherPattern2() {
	const int[64] ditherPattern = int[64] ( 1, 49, 13, 61,  4, 52, 16, 64,
										   33, 17, 45, 29, 36, 20, 48, 32,
										    9, 57,  5, 53, 12, 60,  8, 56,
										   41, 25, 37, 21, 44, 28, 40, 24,
										    3, 51, 15, 63,  2, 50, 14, 62,
										   35, 19, 47, 31, 34, 18, 46, 30,
										   11, 59,  7, 55, 10, 58,  6, 54,
										   43, 27, 39, 23, 42, 26, 38, 22);

	vec2 count = vec2(0.0);
	count.x = floor(mod(texcoord.s * viewWidth, 8.0));
	count.y = floor(mod(texcoord.t * viewHeight, 8.0));

	int dither = ditherPattern[int(count.x) + int(count.y) * 8];

	return float(dither) / 64.0;
}

vec3 	CalculateNoisePattern1(vec2 offset, float size) {
	vec2 coord = texcoord.st;

	coord *= vec2(viewWidth, viewHeight);
	coord = mod(coord + offset, vec2(size));
	coord /= noiseTextureResolution;

	return texture2D(noisetex, coord).xyz;
}

/////////////////////////STRUCTS////////////////////////////////////////////////////////////////////////////
/////////////////////////STRUCTS////////////////////////////////////////////////////////////////////////////

struct MCLightmapStruct {		//Lightmaps directly from MC engine
	float torch;				//Light emitted from torches and other emissive blocks
	float sky;					//Light coming from the sky

	vec3 torchVector; 			//Vector in screen space that represents the direction of average light transfered
	vec3 skyVector;
} mcLightmap;

struct SpecularAttributesStruct {			//Specular surface shading attributes
	float smoothness;			//How smooth or rough a specular surface is
	vec3  specularColor;		//The color of specular reflection
	float metallic;				//from 0 - 1. 0 representing non-metallic, 1 representing fully metallic.
	float fresnelPower; 		//Curve of fresnel effect. Higher values mean the surface has to be viewed at more extreme angles to see reflectance
};

struct SkyStruct { 				//All sky shading attributes
	vec3 	albedo;				//Diffuse texture aka "color texture" of the sky
	vec3 	tintColor; 			//Color that will be multiplied with the sky to tint it
	vec3 	sunglow;			//Color that will be added to the sky simulating scattered light arond the sun/moon
	vec3 	sunSpot; 			//Actual sun surface
};

struct WaterStruct {
	vec3 albedo;
};

struct MaskStruct {

	float matIDs;

	float sky;
	float land;
	float grass;
	float leaves;
	float ice;
	float hand;
	float translucent;
	float glow;
	float goldBlock;
	float ironBlock;
	float diamondBlock;
	float emeraldBlock;
	float sand;
	float sandstone;
	float stone;
	float cobblestone;
	float wool;
	float clouds;
	float doubleTall;
	float vines;

	float torch;
	float lava;
	float glowstone;
	float fire;

	float water;

	float volumeCloud;

};

struct CloudsStruct {
	vec3 albedo;
};

struct Ray {
	vec3 dir;
	vec3 origin;
};

struct Plane {
	vec3 normal;
	vec3 origin;
};

struct SurfaceStruct { //Surface shading properties, attributes, and functions
	SpecularAttributesStruct specular; //Contains all specular surface attributes

	SkyStruct 	    sky;    //Sky shading attributes and properties
	WaterStruct 	water;  //Water shading attributes and properties
	CloudsStruct 	clouds;

	//Properties that are required for lighting calculation
	vec3  albedo;      //Diffuse texture aka "color texture"
	float linearDepth; //Linear depth

	vec4  viewSpacePosition1;  //Vector representing the screen-space position of the surface
	vec4  viewSpacePosition; //Vector representing the screen-space position of the surface
	vec3  viewVector;           //Vector representing the viewing direction
	Ray   viewRay;
	float NdotL;                //dot(normal, lightVector). used for direct lighting calculation

	float shadow;
	float cloudShadow;

	float cloudAlpha;
} surface;

struct LightmapStruct {     //Lighting information to light the scene. These are untextured colored lightmaps to be multiplied with albedo to get the final lit and textured image.
	vec3 sunlight;          //Direct light from the sun
	vec3 skylight;          //Ambient light from the sky
	vec3 bouncedSunlight;   //Fake bounced light, coming from opposite of sun direction and adding to ambient light
	vec3 scatteredSunlight; //Fake scattered sunlight, coming from same direction as sun and adding to ambient light
	vec3 scatteredUpLight;  //Fake GI from ground
	vec3 torchlight;        //Light emitted from torches and other emissive blocks
	vec3 nolight;           //Base ambient light added to everything. For lighting caves so that the player can barely see even when no lights are present
	vec3 specular;          //Reflected direct light from sun
	vec3 translucent;       //Light on the backside of objects representing thin translucent materials
	vec3 sky;               //Color and brightness of the sky itself
	vec3 underwater;        //underwater lightmap
	vec3 heldLight;
} lightmap;

struct ShadingStruct {         //Shading calculation variables
	vec3    direct;
	float 	bounced;            //Fake bounced sunlight
	float 	skylight;           //Light coming from sky
	float 	scattered;          //Fake scattered sunlight
	float   scatteredUp;        //Fake GI from ground
	vec3 	sunlightVisibility; //Shadows
	float 	heldLight;
} shading;

struct GlowStruct {
	vec3 torch;
	vec3 lava;
	vec3 glowstone;
	vec3 fire;
	vec3 emission;
};

struct FinalStruct {			//Final textured and lit images sorted by what is illuminating them.
	GlowStruct 		glow;		//Struct containing emissive material final images

	vec3 lighting;				// The lighting frtom all sources
	vec3 translucent;			//Light on the backside of objects representing thin translucent materials
	vec3 sky;					//Color and brightness of the sky itself
	vec3 underwater;			//underwater colors
	vec3 torchlight;

} final;

struct Intersection {
	vec3 pos;
	float distance;
	float angle;
};

/////////////////////////STRUCT FUNCTIONS////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////STRUCT FUNCTIONS////////////////////////////////////////////////////////////////////////////////////////////

void CalculateMasks(inout MaskStruct mask) {
		if (isEyeInWater > 0) mask.sky = 0.0;
		else mask.sky     = GetMaterialMask(texcoord.st, 0, mask.matIDs);

		mask.land         = GetMaterialMask(texcoord.st,  1, mask.matIDs);
		mask.grass        = GetMaterialMask(texcoord.st,  2, mask.matIDs);
		mask.doubleTall   = GetMaterialMask(texcoord.st, 65, mask.matIDs);
		mask.vines        = GetMaterialMask(texcoord.st, 66, mask.matIDs);
		mask.leaves       = GetMaterialMask(texcoord.st,  3, mask.matIDs);
		mask.ice          = GetMaterialMask(texcoord.st,  4, mask.matIDs);
		mask.hand         = GetMaterialMask(texcoord.st,  5, mask.matIDs);
		mask.translucent  = GetMaterialMask(texcoord.st,  6, mask.matIDs);

		mask.glow         = GetMaterialMask(texcoord.st, 10, mask.matIDs);

		mask.goldBlock    = GetMaterialMask(texcoord.st, 20, mask.matIDs);
		mask.ironBlock    = GetMaterialMask(texcoord.st, 21, mask.matIDs);
		mask.diamondBlock = GetMaterialMask(texcoord.st, 22, mask.matIDs);
		mask.emeraldBlock = GetMaterialMask(texcoord.st, 23, mask.matIDs);
		mask.sand         = GetMaterialMask(texcoord.st, 24, mask.matIDs);
		mask.sandstone    = GetMaterialMask(texcoord.st, 25, mask.matIDs);
		mask.stone        = GetMaterialMask(texcoord.st, 26, mask.matIDs);
		mask.cobblestone  = GetMaterialMask(texcoord.st, 27, mask.matIDs);
		mask.wool         = GetMaterialMask(texcoord.st, 28, mask.matIDs);

		mask.clouds       = GetMaterialMask(texcoord.st, 29, mask.matIDs);

		mask.torch        = GetMaterialMask(texcoord.st, 30, mask.matIDs);
		mask.lava         = GetMaterialMask(texcoord.st, 31, mask.matIDs);
		mask.glowstone    = GetMaterialMask(texcoord.st, 32, mask.matIDs);
		mask.fire         = GetMaterialMask(texcoord.st, 33, mask.matIDs);

		mask.water        = GetWaterMask(texcoord.st, mask.matIDs);

		mask.volumeCloud  = 0.0;
}

vec2 calculateRayTracedShadows(in vec3 origin, in vec3 direction) {
  vec3 curPos = origin;
  vec2 curCoord = getCoordFromCameraSpace(curPos);
  vec3 noise = texture2D(noisetex, texcoord.st * vec2(viewWidth / noiseTextureResolution, viewHeight / noiseTextureResolution)).rgb;
  direction = normalize(direction + noise * 0.025);
  direction = normalize(direction) * (MAX_RAY_LENGTH / NUM_RAY_STEPS);

  //The basic idea here is the the ray goes forward until it's behind something,
  //then slowly moves forward until it's in front of something.
  for(int i = 0; i < NUM_RAY_STEPS; i++) {
    curPos += direction;
    curCoord = getCoordFromCameraSpace(curPos);
		//curPos = normalize(curPos);

    if(curCoord.x < 0 || curCoord.x > 1 || curCoord.y < 0 || curCoord.y > 1) {
      //If we're here, the ray has gone off-screen so we can't reflect anything
      return vec2(1.0);
    }

		const float DepthCorrection = 0.05;
		float depth = GetDepth(curCoord.st);

    float worldDepth = getCameraSpacePosition(curCoord, depth).z;
		worldDepth -= DepthCorrection * depth;

    float rayDepth = curPos.z;
    float depthDiff = (worldDepth - rayDepth);
    float maxDepthDiff = length(direction) + RAY_DEPTH_BIAS;

    if(depthDiff > 0 && depthDiff < maxDepthDiff) {
      return vec2(0.0, length(origin - curPos) / MAX_RAY_LENGTH / 2.35);
    }
  }
	//If we're here, we couldn't find anything to reflect within the alloted number of steps
	return vec2(1.0);
}

//Surface
void CalculateNdotL(inout float NdotL, in vec3 normal) { //Calculates direct sunlight without visibility check
	NdotL = dot(normal, lightVector);
}

float CalculateDirectLighting(in float NdotL, in MaskStruct mask) {
	//Tall grass translucent shading
	if (mask.grass > 0.5 || mask.doubleTall > 0.5 || mask.vines > 0.5) {
		return 1.0;
	//Leaves
	} else if(mask.leaves > 0.5) {
		return 1.0;
	//clouds
	} else if(mask.clouds > 0.5) {
		return 0.5;
	} else if (mask.ice > 0.5) {
		return pow(NdotL * 0.5 + 0.5, 2.0);
	//Default lambert shading
	} else {
		return max(0.0, NdotL * 0.99 + 0.01);
	}
}

mat2 rotateDirection(vec2 angle, vec2 angleInverse, vec2 rotation) {
    return mat2(angleInverse.x * rotation.x - angle.y * rotation.y,
                angle.x * rotation.y + angleInverse.y * rotation.x,
								-angle.x * -rotation.x + -angle.y * -rotation.y,
						    -angleInverse.x * -rotation.y - -angleInverse.y * -rotation.x) / 4.0;
}

//Optifine temp fix, this does nothing except trick optifine into thinking these are doing something
#ifdef ENABLE_SOFT_SHADOWS
float doNothing;
#endif

#ifdef PCSS
float doNothing1;
#endif

vec3 CalculateSunlightVisibility(in vec4 viewSpacePosition1, in ShadingStruct shadingStruct) {                              //Calculates shadows
	if (rainStrength > 0.99)
		return vec3(1.0);

	if (length(shadingStruct.direct) > 0.0) {
    vec4 worldposition = gbufferModelViewInverse * viewSpacePosition1;  //Transform from screen space to world space
    worldposition = shadowModelView * worldposition; //Transform from world space to shadow space
	  worldposition = shadowProjection * worldposition;
	  worldposition /= worldposition.w;

		float dist = sqrt(dot(worldposition.xy, worldposition.xy)); //Compensate for shadow distortion
		vec2 pos = abs(worldposition.xy * 1.165);
		dist = pow(pow(pos.x, 8) + pow(pos.y, 8), 1.0 / 8.0);
	  float distortFactor = (1.0 - SHADOW_MAP_BIAS) + dist * SHADOW_MAP_BIAS;

	  worldposition.xy *= 1.0 / distortFactor;
	  worldposition.z /= 4.0;
	  worldposition = worldposition * 0.5 + 0.5;  //Transform from shadow space to shadow map coordinates
                                                                                                                                                      //Multiplier used to fade out shadows at distance
	  vec3 shading = vec3(0.0);

    float diffthresh = dist * 1.0 + 0.10;
    diffthresh *= 3.0 / (shadowMapResolution / 2048.0);

		#if defined ENABLE_SOFT_SHADOWS
			float numBlockers = 0.0;
			float numSamples = 0.0;

			int PCFSizeHalf = 3;

			float penumbraSize = 0.5;

			#ifdef USE_RANDOM_ROTATION
				float rotateAmount = texture2D(noisetex, texcoord.st * vec2(viewWidth / noiseTextureResolution, viewHeight / noiseTextureResolution)).r * 2.0 - 1.0;
        mat2 kernelRotation = mat2(cos(rotateAmount), -sin(rotateAmount), sin(rotateAmount), cos(rotateAmount));
			#endif

			for( int i = -PCFSizeHalf; i <= PCFSizeHalf; i++ ) {
				for( int j = -PCFSizeHalf; j <= PCFSizeHalf; j++ ) {
					vec2 sampleCoord = vec2( j, i ) / shadowMapResolution;
			    sampleCoord *= penumbraSize;

			    #ifdef USE_RANDOM_ROTATION
			      sampleCoord = kernelRotation * sampleCoord;
			    #endif

					float shadowDepth = shadow2DLod(shadow, vec3(worldposition.st + sampleCoord, worldposition.z - 0.0006 * diffthresh), 0).r;
					numBlockers +=  step(worldposition.z - shadowDepth, 0.0006);
					numSamples++;
				}
			}

			shading = vec3(numBlockers / numSamples);

		#elif defined PCSS

			float vpsSpread = 0.4 / distortFactor;
			vec3 noise = CalculateNoisePattern1(vec2(0.0), 64.0);

			float avgDepth = 0.0;
			float minDepth = 11.0;
			int c;

			for (int i = -1; i <= 1; i++) {
				for (int j = -1; j <= 1; j++) {
					vec2 angle = noise.xy * 3.14159 * 2.0;
					mat2 rotation = mat2(cos(angle.x), -sin(angle.x), sin(angle.y), cos(angle.y));

					vec2 lookupCoord = worldposition.xy + (vec2(i, j) / shadowMapResolution) * rotation * vpsSpread;
					float depthSample = texture2DLod(shadowtex1, lookupCoord, 2).x;
					minDepth = min(minDepth, depthSample);
					avgDepth += pow(clamp(worldposition.z - depthSample, 0.0, 0.15), 1.7);
					c++;
				}
			}

			avgDepth /= c;
			avgDepth = pow(avgDepth, 1.0 / 2.0);

			// float penumbraSize = min(abs(worldposition.z - minDepth), 0.15);
			float penumbraSize = avgDepth;

			int count = 0;
			float spread = penumbraSize * 0.0062 * vpsSpread + 0.45 / shadowMapResolution;

			diffthresh *= 1.0 + avgDepth * 40.0;

			for (float i = -2.0; i <= 2.0; i += 1.0) {
				for (float j = -2.0; j <= 2.0; j += 1.0) {
					float angle = noise.x * 3.14159 * 2.0;
					mat2 rotation = mat2(cos(angle), -sin(angle), sin(angle), cos(angle));

					vec2 coord = vec2(i, j) * rotation;
					vec3 shadowCoord = vec3(worldposition.st + coord * spread, worldposition.z - 0.0001 * diffthresh);

					#ifdef COLORED_PCSS
						float shadowDepth = shadow2DLod(shadow, shadowCoord, 0).x;
						vec3 colorSample = texture2D(shadowcolor0, shadowCoord.st).xyz;

						shading += mix(vec3(0.0), colorSample, shadowDepth);
					#else
						shading += shadow2DLod(shadow, shadowCoord, 0).x;
					#endif
					count += 1;
				}
			}
			shading /= count;

		#else
			shading = shadow2DLod(shadow, vec3(worldposition.st, worldposition.z - 0.0006 * diffthresh), 0).x;
		#endif

		//Ray Traced
		#ifdef HSSRS
			if(length(viewSpacePosition1.xyz) < 10) {
				vec2 raytracedShadow = calculateRayTracedShadows(viewSpacePosition1.xyz, lightVector);
				shading = min(raytracedShadow.xxx, shading);
			}
		#endif

		shading = shading * pow(1 - rainStrength, 2.0);
		shading = min(shading, transition_fading);

		return shading; //Use vec3 to hold color information

  } else {
		return vec3(0.0);
	}
}

float SolidSunlightVisibility(in vec4 viewSpacePositionSolid) {
	vec4 worldposition = gbufferModelViewInverse * viewSpacePositionSolid;  //Transform from screen space to world space
	worldposition = shadowModelView * worldposition; //Transform from world space to shadow space
	worldposition = shadowProjection * worldposition;
	worldposition /= worldposition.w;

	float dist = sqrt(dot(worldposition.xy, worldposition.xy)); //Compensate for shadow distortion
	vec2 pos = abs(worldposition.xy * 1.165);
	dist = pow(pow(pos.x, 8) + pow(pos.y, 8), 1.0 / 8.0);
	float distortFactor = (1.0 - SHADOW_MAP_BIAS) + dist * SHADOW_MAP_BIAS;

	worldposition.xy *= 1.0 / distortFactor;
	worldposition.z /= 4.0;
	worldposition = worldposition * 0.5 + 0.5;  //Transform from shadow space to shadow map coordinates

	float diffthresh = dist * 1.0 + 0.10;
	diffthresh *= 3.0 / (shadowMapResolution / 2048.0);

	float shading = 1.0;

	float numBlockers = 0.0;
	float numSamples = 0.0;

	int PCFSizeHalf = 1;

	float penumbraSize = 0.5;

	#ifdef USE_RANDOM_ROTATION
		float rotateAmount = texture2D(noisetex, texcoord.st * vec2(viewWidth / noiseTextureResolution, viewHeight / noiseTextureResolution)).r * 2.0 - 1.0;
		mat2 kernelRotation = mat2(cos(rotateAmount), -sin(rotateAmount), sin(rotateAmount), cos(rotateAmount));
	#endif

	for( int i = -PCFSizeHalf; i <= PCFSizeHalf; i++ ) {
		for( int j = -PCFSizeHalf; j <= PCFSizeHalf; j++ ) {
			vec2 sampleCoord = vec2( j * 3, i * 3) / shadowMapResolution;
			sampleCoord *= penumbraSize;

			#ifdef USE_RANDOM_ROTATION
				sampleCoord = kernelRotation * sampleCoord;
			#endif

			float shadowDepth = shadow2DLod(shadow, vec3(worldposition.st + sampleCoord, worldposition.z - 0.0006 * diffthresh), 0).r;
			numBlockers +=  step(worldposition.z - shadowDepth, 0.0006);
			numSamples++;
		}
	}

	shading = numBlockers / numSamples;

	return shading;
}

float CalculateBouncedSunlight(in float NdotL) {
	float bounced = clamp(-NdotL + 0.95, 0.0, 1.95) / 1.95;

	return pow(bounced, 3.0);
}

float CalculateScatteredSunlight(in float NdotL) {
	return clamp(NdotL * 0.75 + 0.25, 0.0, 1.0);
}

float CalculateSkylight(in vec3 normal, in MaskStruct mask) {
	if (mask.clouds > 0.5) {
		return 1.0;
	} else if (mask.grass > 0.5) {
		return 1.0;
	} else {
		float skylight = dot(normal, upVector);
		skylight = skylight * 0.4 + 0.6;

		return skylight;
	}
}

float CalculateScatteredUpLight(in vec3 normal) {
	float scattered = dot(normal, upVector);
	scattered = scattered * 0.5 + 0.5;
	scattered = 1.0 - scattered;

	return scattered;
}

float CalculateHeldLightShading(in vec4 viewSpacePosition1) {
	vec3 lightPos = vec3(0.0);
	vec3 lightVector = normalize(lightPos - viewSpacePosition1.xyz);
	float lightDist = length(lightPos.xyz - viewSpacePosition1.xyz);

	float atten = 1.0 / (pow(lightDist, 2.0) + 0.001);
	float NdotL = 1.0;

	return atten * NdotL;
}

float CalculateSunglow(in vec4 viewSpacePosition1) {
	float curve = 4.0;

	vec3 npos = normalize(viewSpacePosition1.xyz);
	vec3 halfVector2 = normalize(-lightVector + npos);
	float factor = 1.0 - dot(halfVector2, npos);

	return factor * factor * factor * factor;
}

float CalculateAntiSunglow(in vec4 viewSpacePosition1) {
	float curve = 4.0;

	vec3 npos = normalize(viewSpacePosition1.xyz);
	vec3 halfVector2 = normalize(lightVector + npos);
	float factor = 1.0 - dot(halfVector2, npos);

	return factor * factor * factor * factor;
}

bool CalculateSunspot(in vec4 viewSpacePosition1) {
	//circular sun
	float curve = 1.0;

	vec3 npos = normalize(viewSpacePosition1.xyz);
	vec3 halfVector2 = normalize(-lightVector + npos);

	float sunProximity = 1.0 - dot(halfVector2, npos);

	if (sunProximity > 0.96 && sunAngle > 0.0 && sunAngle < 0.5) {
		return true;
	} else {
		return false;
	}
}

void AddSkyGradient(inout vec3 color, in vec4 viewSpacePosition1) {
	float curve = 3.5;
	vec3 npos = normalize(viewSpacePosition1.xyz);
	vec3 halfVector2 = normalize(-upVector + npos);
	float skyGradientFactor = dot(halfVector2, npos);
	float skyDirectionGradient = skyGradientFactor;

	skyGradientFactor = pow(skyGradientFactor, curve);
	color *= mix(skyGradientFactor, 1.0, clamp((0.145 - (timeNoon * 0.1)) + rainStrength, 0.0, 1.0));

	vec3 skyBlueColor = vec3(0.25, 0.4, 1.0) * 2.5;
	skyBlueColor.g *= skyGradientFactor * 0.5 + 0.75;
	skyBlueColor = mix(skyBlueColor, vec3(1.0, 0.9, 0.5), vec3(timeSkyDark));
	skyBlueColor *= mix(vec3(1.0), vec3(1.0, 1.0, 0.5), vec3(timeSunriseSunset));

	float fade1 = clamp(skyGradientFactor - 0.15, 0.0, 0.2) / 0.2;
	vec3 color1 = vec3(1.0, 1.3, 1.0);

	color *= mix(skyBlueColor, color1, vec3(fade1));

	float fade2 = clamp(skyGradientFactor - 0.18, 0.0, 0.2) / 0.2;
	vec3 color2 = vec3(1.7, 1.0, 0.8);
	color2 = mix(color2, vec3(1.0, 0.15, 0.0), vec3(timeSunriseSunset));

	color *= mix(vec3(1.0), color2, vec3(fade2 * 0.5));

	float horizonGradient = 1.0 - distance(skyDirectionGradient, 0.72) / 0.72;
	horizonGradient = pow(horizonGradient, 10.0);
	horizonGradient = max(0.0, horizonGradient);

	float sunglow = CalculateSunglow(viewSpacePosition1);
	//horizonGradient *= sunglow * 2.0 + (0.65 - timeSunriseSunset * 0.55 - timeSunriseSunset * 0.55);

	vec3 horizonColor1 = vec3(1.5, 1.5, 1.5);
	horizonColor1 = mix(horizonColor1, vec3(1.5, 1.95, 1.5) * 2.0, vec3(timeSunriseSunset));
	vec3 horizonColor2 = vec3(1.5, 1.2, 0.8) * 1.0;
	horizonColor2 = mix(horizonColor2, vec3(1.9, 0.6, 0.4) * 2.0, vec3(timeSunriseSunset));

	color *= mix(vec3(1.0), horizonColor1, vec3(horizonGradient) * (1.0 - timeMidnight));
	color *= mix(vec3(1.0), horizonColor2, vec3(pow(horizonGradient, 2.0)) * (1.0 - timeMidnight));

	float grayscale = color.r + color.g + color.b;
	grayscale /= 3.0;

	color = mix(color, vec3(grayscale) * 1.4, vec3(rainStrength));
}

void AddSunglow(inout vec3 color, in vec4 viewSpacePosition1) {
	float sunglowFactor = CalculateSunglow(viewSpacePosition1);
	float antiSunglowFactor = CalculateAntiSunglow(viewSpacePosition1);

	color *= 1.0 + pow(sunglowFactor, 1.1) * (1.5 + timeNoon * 1.0) * (1.0 - rainStrength);
	color *= mix(vec3(1.0), colorSunlight * 5.0, pow(clamp(vec3(sunglowFactor) * (1.0 - timeMidnight) * (1.0 - rainStrength), vec3(0.0), vec3(1.0)), vec3(2.0)));

	color *= 1.0 + antiSunglowFactor * 2.0 * (1.0 - rainStrength);
}

void AddCloudGlow(inout vec3 color, in vec4 viewSpacePosition, in MaskStruct mask) {
	float glow = CalculateSunglow(viewSpacePosition);
	glow = pow(glow, 1.0);

	float mult = mix(50.0, 800.0, timeSkyDark);

	color.rgb *= 1.0 + glow * mult * (mask.clouds);
}

void CalculateUnderwaterFog(inout vec3 finalComposite, in float linearDepth) {
	#ifdef UnderwaterFog
	if (isEyeInWater <= 0) return;

	vec3 fogColor = colorWaterMurk * vec3(colorSkylight);

	float fogFactor = linearDepth / 100.0;
	fogFactor = min(fogFactor, 0.7);
	fogFactor = sin(fogFactor * 3.1415 * 0.5);
	fogFactor = sqrt(fogFactor);


	finalComposite.rgb  = mix(finalComposite.rgb, fogColor * 0.002, vec3(fogFactor));
	finalComposite.rgb *= mix(vec3(1.0), pow(colorWaterBlue, vec3(4.0)), vec3(fogFactor));
	#endif
}

void CalculateRainFog(inout vec3 color, in vec4 viewSpacePosition1, in MaskStruct mask) {
	#ifdef RAIN_FOG
	vec3 fogColor = colorSkylight * 0.055;

	float fogDensity = 0.0018 * rainStrength;
	fogDensity *= mix(0.0, 1.0, pow(eyeBrightnessSmooth.y / 240.0, 6.0));
	float visibility = 1.0 / (pow(exp(distance(viewSpacePosition1.xyz, vec3(0.0)) * fogDensity), 1.0));

	float fogFactor = 1.0 - visibility;
	fogFactor = clamp(fogFactor, 0.0, 1.0);
	fogFactor = mix(fogFactor, 1.0, float(mask.sky) * 0.8 * rainStrength);
	fogFactor = mix(fogFactor, 1.0, float(mask.clouds) * 0.8 * rainStrength);

	color = mix(color, fogColor, vec3(fogFactor));
	#endif
}

void CalculateAtmosphericScattering(inout vec3 color, in MaskStruct mask, in vec4 viewSpacePosition1, float linearDepth, in vec3 sunSpot) {
	#ifdef ATMOSPHERIC_FOG

	#ifdef NEW_UNDERWATER
	if (isEyeInWater > 0) return;
	#endif

	vec3 fogColor = colorSkylight * 0.11;

	float sat = 0.5;
	fogColor.r = fogColor.r * (0.0 + sat) - (fogColor.g + fogColor.b) * 0.0 * sat;
	fogColor.g = fogColor.g * (0.0 + sat) - (fogColor.r + fogColor.b) * 0.0 * sat;
	fogColor.b = fogColor.b * (0.0 + sat) - (fogColor.r + fogColor.g) * 0.0 * sat;

	float sunglow = CalculateSunglow(viewSpacePosition1);
	vec3 sunColor = colorSunlight;

	fogColor += mix(vec3(0.0), sunColor, sunglow * 0.8);

	float fogDensity = 0.01;

	#ifdef MORNING_FOG
		fogDensity += 0.04 * timeSunriseSunset * 0.25;
	#endif

	#ifdef 	EVENING_FOG
		fogDensity += 0.04 * timeSunriseSunset * 0.25;
	#endif

	float visibility = 1.26 / (pow(exp(linearDepth * fogDensity), 1.0));

	float fogFactor = 1.0 - visibility;
	fogFactor = clamp(fogFactor, 0.0, 1.0);
	fogFactor = pow(fogFactor, 2.7);
	fogFactor = mix(fogFactor, 0.0, min(1.0, sunSpot.r));
	fogFactor *= mix(1.0, 0.25, float(mask.sky));
	fogFactor *= mix(1.0, 0.75, float(mask.clouds));

	#ifdef NO_ATMOSPHERIC_FOG_INSIDE
		fogFactor *= mix(0.0, 1.0, pow(eyeBrightnessSmooth.y / 240.0, 3.0));
	#endif

	float redshift = 1.20;

	//scatter away high frequency light
	color.b *= 1.0 - clamp(fogFactor * 1.65 * redshift, 0.0, 0.75);
	color.g *= 1.0 - fogFactor * 0.2* redshift;
	color.g *= 1.0 - clamp(fogFactor - 0.26, 0.0, 1.0) * 0.5* redshift;

	//add scattered low frequency light
	color += fogColor * fogFactor * 1.0;
	#endif
}

Intersection RayPlaneIntersectionWorld(in Ray ray, in Plane plane) {
	float rayPlaneAngle = dot(ray.dir, plane.normal);

	float planeRayDist = 100000000.0;
	vec3 intersectionPos = ray.dir * planeRayDist;

	if (rayPlaneAngle > 0.0001 || rayPlaneAngle < -0.0001) {
		planeRayDist = dot((plane.origin), plane.normal) / rayPlaneAngle;
		intersectionPos = ray.dir * planeRayDist;
		intersectionPos = -intersectionPos;

		intersectionPos += cameraPosition.xyz;
	}

	Intersection i;

	i.pos = intersectionPos;
	i.distance = planeRayDist;
	i.angle = rayPlaneAngle;

	return i;
}

Intersection RayPlaneIntersection(in Ray ray, in Plane plane) {
	float rayPlaneAngle = dot(ray.dir, plane.normal);

	float planeRayDist = 100000000.0;
	vec3 intersectionPos = ray.dir * planeRayDist;

	if (rayPlaneAngle > 0.0001 || rayPlaneAngle < -0.0001) {
		planeRayDist = dot((plane.origin - ray.origin), plane.normal) / rayPlaneAngle;
		intersectionPos = ray.origin + ray.dir * planeRayDist;
	}

	Intersection i;

	i.pos = intersectionPos;
	i.distance = planeRayDist;
	i.angle = rayPlaneAngle;

	return i;
}

float Get3DNoise(in vec3 pos) {
	pos.z += 0.0;
	vec3 p = floor(pos);
	vec3 f = fract(pos);

	vec2 uv =  (p.xy + p.z * vec2(17.0)) + f.xy;
	vec2 uv2 = (p.xy + (p.z + 1.0) * vec2(17.0)) + f.xy;
	vec2 coord =  (uv  + 0.5) / noiseTextureResolution;
	vec2 coord2 = (uv2 + 0.5) / noiseTextureResolution;
	float xy1 = texture2D(noisetex, coord).x;
	float xy2 = texture2D(noisetex, coord2).x;
	return mix(xy1, xy2, f.z);
}

float Get3DNoise3(in vec3 pos) {
	pos.z += 0.0;
	pos.xyz += 0.5;

	vec3 p = floor(pos);
	vec3 f = fract(pos);

	f.x = f.x * f.x * (3.0 - 2.0 * f.x);
	f.y = f.y * f.y * (3.0 - 2.0 * f.y);
	f.z = f.z * f.z * (3.0 - 2.0 * f.z);

	vec2 uv =  (p.xy + p.z * vec2(17.0)) + f.xy;
	vec2 uv2 = (p.xy + (p.z + 1.0) * vec2(17.0)) + f.xy;
	uv += 0.5;
	uv2 += 0.5;

	vec2 coord =  (uv  + 0.5) / noiseTextureResolution;
	vec2 coord2 = (uv2 + 0.5) / noiseTextureResolution;

	float xy1 = texture2D(noisetex, coord).x;
	float xy2 = texture2D(noisetex, coord2).x;

	return mix(xy1, xy2, f.z);
}

float GetCoverage(in float coverage, in float density, in float clouds) {
	clouds = clamp(clouds - (1.0 - coverage), 0.0, 1.0 -density) / (1.0 - density);
	clouds = max(0.0, clouds * 1.1 - 0.1);

	return clouds;
}

float GetCoverage2(in float coverage, in float density, in float clouds) {
	clouds = clamp(clouds - (1.0 - coverage), 0.0, 1.0 -density) / (1.0 - density);
	clouds = max(0.0, clouds * 1.1 - 0.1);
	clouds = clouds = clouds * clouds * (3.0 - 2.0 * clouds);

	return clouds;
}

vec4 volumeCloudColor(in vec4 worldPosition, in float sunglow, in vec3 worldLightVector) {
	float cloudHeight = Cloud3Height;
	float cloudDepth  = 120.0;


	float cloudUpperHeight = cloudHeight + (cloudDepth / 2.0);
	float cloudLowerHeight = cloudHeight - (cloudDepth / 2.0);

	if (worldPosition.y < cloudLowerHeight || worldPosition.y > cloudUpperHeight) {
		return vec4(0.0);
	} else {

		vec3 p = worldPosition.xyz / 150.0;

		float t = frameTimeCounter * 2 ;
		p.x -= t * 0.02;

		vec3 p1 = p * vec3(1.0, 0.5, 1.0)  + vec3(0.0, t * 0.01, 0.0);
		float noise = Get3DNoise(p) * 1.0;	p *= 4.0;	p.x += t * 0.02; vec3 p2 = p;
		noise += (1.0 - abs(Get3DNoise(p) * 3.0 - 1.0)) * 0.20;	p *= 3.0;	p.xz += t * 0.05;
		noise += (1.0 - abs(Get3DNoise(p) * 3.0 - 1.5)-0.2) * 0.065;	p.xz -=t * 0.165;	p.xz += t * 0.05;
		noise += (1.0 - abs(Get3DNoise(p) * 3.0 - 1.0)) * 0.065;	p *= 2.0; //Seems to be turbulance
		noise += (1.0 - abs(Get3DNoise3(p) * 2.0 - 1.0)) * 0.015;
		noise /= 1.2;

		const float lightOffset = 0.3;

		float heightGradient = clamp(( - (cloudLowerHeight - worldPosition.y) / (cloudDepth * 1.0)), 0.0, 1.0);
		float heightGradient2 = clamp(( - (cloudLowerHeight - (worldPosition.y + worldLightVector.y * lightOffset * 150.0)) / (cloudDepth * 1.0)), 0.0, 1.0);

		float cloudAltitudeWeight = 1.0 - clamp(distance(worldPosition.y, cloudHeight) / (cloudDepth / 2.0), 0.0, 1.0);
		cloudAltitudeWeight = (-cos(cloudAltitudeWeight * 3.1415)) * 0.5 + 0.5;
		cloudAltitudeWeight = pow(cloudAltitudeWeight, mix(0.33, 0.8, rainStrength));

		float cloudAltitudeWeight2 = 1.0 - clamp(distance(worldPosition.y + worldLightVector.y * lightOffset * 150.0, cloudHeight) / (cloudDepth / 2.0), 0.0, 1.0);
		cloudAltitudeWeight2 = (-cos(cloudAltitudeWeight2 * 3.1415)) * 0.5 + 0.5;
		cloudAltitudeWeight2 = pow(cloudAltitudeWeight2, mix(0.33, 0.8, rainStrength));

		noise *= cloudAltitudeWeight;

		float next_moon_phase = moonPhase + 1;

		if(float(moonPhase) == 7) {
			next_moon_phase = 0;
		}

		float moon_phase_smooth = mix(moonPhase, next_moon_phase, float(worldTime) / 24000.0);

		float dynWeather;

		#ifdef DYNAMIC_WEATHER
			dynWeather = (((abs(float(moon_phase_smooth) - 2) + 1) / 6) + 0.5);
		#endif

		//cloud edge
		float rainy = mix(wetness, 1.0, rainStrength);
		float coverage = 0.48 + rainy * 0.335;
		coverage = min(mix(coverage * dynWeather, 0.77, rainStrength), 0.77);

		float dist = length(worldPosition.xz - cameraPosition.xz);
		coverage *= max(0.0, 1.0 - dist / 40000.0);

		float density = 0.90;
		noise = GetCoverage2(coverage, density, noise);
		noise = pow(noise, 1.5);

		if(noise <= 0.001) {
			return vec4(0.0, 0.0, 0.0, 0.0);
		}


		float sundiff = Get3DNoise3(p1 + worldLightVector.xyz * lightOffset);
		sundiff += (1.0 - abs(Get3DNoise3(p2 + worldLightVector.xyz * lightOffset / 2.0) * 1.0 - 0.5) - 0.1) * 0.55;
		sundiff *= 0.955;
		sundiff *= cloudAltitudeWeight2;

		float preCoverage = sundiff;
		sundiff = -GetCoverage2(coverage * 1.0, density * 0.5, sundiff);
		float sundiff2 = -GetCoverage2(coverage * 1.0, 0.0, preCoverage);
		float firstOrder 	= pow(clamp(sundiff * 1.2 + 1.7, 0.0, 1.0), 8.0);
		float secondOrder 	= pow(clamp(sundiff2 * 1.2 + 1.1, 0.0, 1.0), 4.0);

		float anisoBackFactor = mix(clamp(pow(noise, 1.6) * 2.5, 0.0, 1.0), 1.0, pow(sunglow, 1.0));
		firstOrder *= anisoBackFactor * 0.99 + 0.01;
		secondOrder *= anisoBackFactor * 1.19 + 0.9;

		float directLightFalloff = clamp(pow(-(cloudLowerHeight - worldPosition.y) / cloudDepth, 3.5), 0.0, 1.0);
		directLightFalloff *= mix(	clamp(pow(noise, 0.9), 0.0, 1.0), 	clamp(pow(1.0 - noise, 10.3), 0.0, 0.5), 	pow(sunglow, 0.2));

		vec3 colorDirect = colorSunlight * 12.5;
		colorDirect = mix(colorDirect, colorDirect * vec3(0.1, 0.2, 0.3), timeMidnight);
		colorDirect = mix(colorDirect, colorDirect * vec3(0.2, 0.2, 0.2), rainStrength);
		colorDirect *= 1.0 + pow(sunglow, 4.0) * 100.0;

		float DirectDotLightVector = dot(colorDirect, lightVector);

		vec3 colorAmbient = mix(colorSkylight, colorSunlight, 0.15) * 0.065;
		colorAmbient *= mix(1.0, 0.3, timeMidnight);

		vec3 colorBounced = colorBouncedSunlight * 0.35;
		colorBounced *= pow((1.0 - heightGradient), 8.0);
		colorBounced *= anisoBackFactor + 0.5;
		colorBounced *= 1.0 - rainStrength;

		vec3 color = mix(colorAmbient, colorDirect, vec3(directLightFalloff));
		color *= 1.0;

		vec4 result = vec4(color.rgb, noise);
		return result;
	}
}

void CalculateVolumeClouds(inout vec3 color, in vec4 viewSpacePosition1, in vec3 worldLightVector, inout SurfaceStruct surface, in MaskStruct mask) {
	#ifdef VOLUMETRIC_CLOUDS
	surface.cloudAlpha = 0.0;

	vec4 worldPosition = gbufferModelViewInverse * viewSpacePosition1;
	worldPosition.xyz += cameraPosition.xyz;

	float cloudHeight = 150.0;
	float cloudDepth  = 140.0;
	float cloudDensity = 1.0;

	float startingRayDepth = far - 5.0;

	float rayDepth = startingRayDepth;
	float rayIncrement = far / CLOUD_DISPERSE;

	#ifdef SOFT_FLUFFY_CLOUDS
		rayDepth += CalculateDitherPattern1() * rayIncrement;
	#else
		rayDepth += CalculateDitherPattern2() * rayIncrement;
	#endif

	int i = 0;

	vec4 cloudSum = vec4(0.0);
	cloudSum.rgb = color.rgb;

	float sunglow = min(CalculateSunglow(viewSpacePosition1), 2.0);

	float cloudDistanceMult = 400.0 / far;

	float surfaceDistance = length(worldPosition.xyz - cameraPosition.xyz);

	while (rayDepth > 0.0) {
		//determine worldspace ray position
		vec4 rayPosition = GetCloudSpacePosition(texcoord.st, rayDepth, cloudDistanceMult);

		float rayDistance = length((rayPosition.xyz - cameraPosition.xyz) / cloudDistanceMult);

		vec4 proximity = volumeCloudColor(rayPosition, sunglow / 1.4, worldLightVector);

		proximity.a *= cloudDensity;

		if(surfaceDistance < rayDistance * cloudDistanceMult && mask.sky == 0.0)
			proximity.a = 0.0;

		cloudSum.rgb = mix(cloudSum.rgb, proximity.rgb, vec3(min(1.0, proximity.a * cloudDensity)) );
		cloudSum.a += proximity.a * cloudDensity;

		surface.cloudAlpha += proximity.a;

		//Increment ray
		rayDepth -= rayIncrement;
		i++;
	}

	color.rgb = mix(color.rgb, cloudSum.rgb, vec3(min(1.0, cloudSum.a * 50.0)));

	if (cloudSum.a > 0.00) {
		mask.volumeCloud = 1.0;
	}
	#endif
}

vec4 Calculatate2DCloudsColor(in vec4 worldPosition, in float sunglow, in vec3 worldLightVector, in float altitude, in float thickness, const bool isShadowPass) {
	float cloudHeight = altitude;
	float cloudDepth  = thickness;
	float cloudUpperHeight = cloudHeight + (cloudDepth / 2.0);
	float cloudLowerHeight = cloudHeight - (cloudDepth / 2.0);

	worldPosition.xz /= 1.0 + max(0.0, length(worldPosition.xz - cameraPosition.xz) / 5000.0);

	vec3 p = worldPosition.xyz / 150.0;
	float t = frameTimeCounter * 0.25;

	p += (Get3DNoise(p * 2.0 + vec3(0.0, t * 0.01, 0.0)) * 2.0 - 1.0) * 0.1;
	p.z -= (Get3DNoise(p * 0.25 + vec3(0.0, t * 0.01, 0.0)) * 2.0 - 1.0) * 0.5;
	p.x -= (Get3DNoise(p * 0.125 + vec3(0.0, t * 0.01, 0.0)) * 2.0 - 1.0) * 1.2;
	p.xz -= (Get3DNoise(p * 0.0525 + vec3(0.0, t * 0.01, 0.0)) * 2.0 - 1.0) * 1.7;


	p.x *= 0.5;
	p.x -= t * 0.01;

	vec3 p1 = p * vec3(1.0, 0.5, 1.0)  + vec3(0.0, t * 0.01, 0.0);
	float noise  = 	Get3DNoise(p * vec3(1.0, 0.5, 1.0) + vec3(0.0, t * 0.01, 0.0));	p *= 2.0;	p.x -= t * 0.057;	vec3 p2 = p;
	noise += (2.0 - abs(Get3DNoise(p) * 2.0 - 0.0)) * (0.55);	p *= 3.0;	p.xz -= t * 0.035;	p.x *= 2.0;	vec3 p3 = p;
	noise += (3.0 - abs(Get3DNoise(p) * 3.0 - 0.0)) * (0.085); p *= 3.0;	p.xz -= t * 0.035;	vec3 p4 = p;
	noise += (3.0 - abs(Get3DNoise(p) * 3.0 - 0.0)) * (0.045); p *= 3.0;	p.xz -= t * 0.035;

	if (!isShadowPass) {
		noise += ((Get3DNoise(p))) * (0.049);
		p *= 3.0;
		noise += ((Get3DNoise(p))) * (0.024);
	}

	noise /= 2.175;

	float next_moon_phase = moonPhase + 1;

	if(float(moonPhase) == 7) {
		next_moon_phase = 0;
	}

	float moon_phase_smooth = mix(moonPhase, next_moon_phase, float(worldTime) / 24000.0);

	float dynWeather = 1.0;

	#ifdef DYNAMIC_WEATHER
		dynWeather = ((abs(float(moon_phase_smooth) - 3) + 1) / 5) + 0.5;
	#endif

	//cloud edge
	float rainy = mix(wetness, 1.0, rainStrength);

	float coverage = CLOUD_PLANE_Coverage + rainy * 0.335;
	coverage = min(mix(coverage * dynWeather, 0.87, rainStrength), 0.87);

	float dist = length(worldPosition.xz - cameraPosition.xz);
	coverage *= max(0.0, 1.0 - dist / mix(7000.0, 3000.0, rainStrength));

	float density = 0.0;

	if (isShadowPass) {
		return vec4(GetCoverage(coverage + 0.2, density, noise));
	}

	noise = GetCoverage(coverage, density, noise);

	const float lightOffset = 0.2;

	float sundiff = Get3DNoise(p1 + worldLightVector.xyz * lightOffset);
	sundiff += (2.0 - abs(Get3DNoise(p2 + worldLightVector.xyz * lightOffset / 2.0) * 2.0 - 0.0)) * (0.55);

	float largeSundiff = sundiff;
	largeSundiff = -GetCoverage(coverage, 0.0, largeSundiff * 1.3);

	sundiff += (3.0 - abs(Get3DNoise(p3 + worldLightVector.xyz * lightOffset / 5.0) * 3.0 - 0.0)) * (0.055);
	sundiff += (3.0 - abs(Get3DNoise(p4 + worldLightVector.xyz * lightOffset / 8.0) * 3.0 - 0.0)) * (0.025);
	sundiff /= 1.5;
	sundiff = -GetCoverage(coverage * 1.0, 0.0, sundiff);

	float secondOrder 	= pow(clamp(sundiff * 1.1 + 1.35, 0.0, 1.0), 7.0);
	float firstOrder 	= pow(clamp(largeSundiff * 1.1 + 1.56, 0.0, 1.0), 3.0);

	float directLightFalloff = firstOrder * secondOrder;
	float anisoBackFactor = mix(clamp(pow(noise, 1.6) * 2.5, 0.0, 1.0), 1.0, pow(sunglow, 1.0));

	directLightFalloff *= anisoBackFactor;
	directLightFalloff *= mix(1.5, 1.0, pow(sunglow, 0.5));



	vec3 colorDirect = colorSunlight * 100.915;
	colorDirect = mix(colorDirect, colorDirect * vec3(0.2, 0.5, 1.0), timeMidnight);
	colorDirect *= 1.0 + pow(sunglow, 5.0) * 600.0 * pow(directLightFalloff, 1.1) * (1.0 - rainStrength);


	vec3 colorAmbient = mix(colorSkylight, colorSunlight * 2.0, vec3(0.15)) * 0.07;
	colorAmbient *= mix(1.0, 0.3, timeMidnight);
	colorAmbient = mix(colorAmbient, colorAmbient * 3.0 + colorSunlight * 0.05, vec3(clamp(pow(1.0 - noise, 12.0) * 1.0, 0.0, 1.0)));


	directLightFalloff *= 2.0;
	directLightFalloff *= mix(1.0, 0.125, rainStrength);

	vec3 color = mix(colorAmbient, colorDirect, vec3(min(1.0, directLightFalloff)));

	vec4 result = vec4(color.rgb, noise);

	return result;
}

float CloudShadow(in SurfaceStruct surface, in vec3 worldLightVector) {
	float cloudsAltitude = 540.0;
	float cloudsThickness = 150.0;

	float cloudsUpperLimit = cloudsAltitude + cloudsThickness * 0.5;
	float cloudsLowerLimit = cloudsAltitude - cloudsThickness * 0.5;

	float planeHeight = cloudsUpperLimit;

	planeHeight -= cloudsThickness * 0.85;

	Plane pl;
	pl.origin = vec3(0.0, planeHeight, 0.0);
	pl.normal = vec3(0.0, 1.0, 0.0);

	//Cloud shadow
	Ray surfaceToSun;
	vec4 sunDir = gbufferModelViewInverse * vec4(lightVector, 0.0);
	surfaceToSun.dir = normalize(sunDir.xyz);
	vec4 surfacePos = gbufferModelViewInverse * surface.viewSpacePosition;
	surfaceToSun.origin = surfacePos.xyz + cameraPosition.xyz;

	Intersection i = RayPlaneIntersection(surfaceToSun, pl);

	float cloudShadow = Calculatate2DCloudsColor(vec4(i.pos.xyz * 0.5, 1.0), 1.0, worldLightVector, cloudsAltitude, cloudsThickness, true).x;
	cloudShadow += Calculatate2DCloudsColor(vec4(i.pos.xyz * 0.65 + vec3(210.0) + vec3(i.pos.z * 0.5, 0.0, 0.0), 1.0), 1.0, worldLightVector, cloudsAltitude, cloudsThickness, true).x;

	float cloudDistanceMult = 400.0 / far;
	vec4 rayPosition = GetCloudSpacePosition(texcoord.st, 1.0, cloudDistanceMult);

	cloudShadow = min(cloudShadow, 0.95);
	cloudShadow = 1.0 - cloudShadow;

	return cloudShadow;
}

vec4 BilateralUpsample(const in float scale, in vec2 offset, in float depth, in vec3 normal) {
	vec2 recipres = vec2(1.0 / viewWidth, 1.0 / viewHeight);

	vec4 light = vec4(0.0);
	float weights = 0.0;

	float gi_quality = GI_FILTER_QUALITY;

	for (float i = -gi_quality; i <= gi_quality; i += 1.0) {
		for (float j = -gi_quality; j <= gi_quality; j += 1.0) {
			vec2 coord = vec2(i, j) * recipres * 2.0;

			float sampleDepth = GetDepthLinear(texcoord.st + coord * 2.0 * (exp2(scale)));
			vec3 sampleNormal = GetNormals(texcoord.st + coord * 2.0 * (exp2(scale)));
			float weight = clamp(1.0 - abs(sampleDepth - depth) / 2.0, 0.0, 1.0);
			weight *= max(0.0, dot(sampleNormal, normal) * 2.0 - 1.0);

			light +=	pow(texture2DLod(gaux1, (texcoord.st) * (1.0 / exp2(scale )) + 	offset + coord, 1), vec4(2.2, 2.2, 2.2, 1.0)) * weight;
			weights += weight;
		}
	}

	light /= max(0.00001, weights);

	if (weights < 0.01) {
		light =	pow(texture2DLod(gaux1, (texcoord.st) * (1.0 / exp2(scale 	)) + 	offset, 2), vec4(2.2, 2.2, 2.2, 1.0));
	}

	return light;
}

vec4 Delta(vec3 albedo, vec3 normal, float skylight) {
	float depth = GetDepthLinear(texcoord.st);
	vec4 delta = BilateralUpsample(1.0, vec2(0.0, 0.0), depth, normal);

	delta.rgb = delta.rgb * albedo * colorSunlight;

	delta.rgb *= 5.0 * delta.a * delta.a * (1.0 - rainStrength);

	return delta;
}

float CrepuscularRays(in float linearDepth) {
	#ifdef VOLUMETRIC_LIGHT
	float rayDepth = 0.02;
	float increment = 4.0;

	const float rayLimit = 30.0;
	float dither = CalculateDitherPattern2();

	float lightAccumulation = 0.0;
	float ambientFogAccumulation = 0.0;

	float numSteps = rayLimit / increment;

	int count = 0;

	while (rayDepth < rayLimit) {
		if(linearDepth < rayDepth + dither * increment) {
			break;
		}

		vec4 rayPosition = GetScreenSpacePosition(texcoord.st, LinearToExponentialDepth(rayDepth + dither * increment));
		rayPosition = gbufferModelViewInverse * rayPosition;

		rayPosition = shadowModelView * rayPosition;
		rayPosition = shadowProjection * rayPosition;
		rayPosition /= rayPosition.w;

		float dist = sqrt(dot(rayPosition.xy, rayPosition.xy));
		vec2 pos = abs(rayPosition.xy * 1.165);
		dist = pow(pow(pos.x, 8) + pow(pos.y, 8), 1.0 / 8.0);
    float distortFactor = (1.0 - SHADOW_MAP_BIAS) + dist * SHADOW_MAP_BIAS;
    rayPosition.xy *= 1.0 / distortFactor;
    rayPosition.z /= 4.0;
    rayPosition = rayPosition * 0.5 + 0.5;            //Transform from shadow space to shadow map coordinates

		float shadowSample = shadow2DLod(shadow, vec3(rayPosition.st, rayPosition.z), 2).x;

		lightAccumulation += shadowSample * increment;

		ambientFogAccumulation *= 1.0;

		rayDepth += increment;
		count++;
		increment *= 1.5;
	}

	lightAccumulation /= numSteps;
	ambientFogAccumulation /= numSteps;

	float rays = lightAccumulation;
	float depth = GetDepthLinear(texcoord.st);

	rays = min(rays, transition_fading);
	return rays * 0.1;
	#else
	return 1.0;
	#endif
}

///--2DGodRays--///
float Rays() {
	#ifdef GODRAYS
	float gr = 0.0;
	vec4 tpos = vec4(sunPosition, 1.0) * gbufferProjection;
	tpos = vec4(tpos.xyz / tpos.w,1.0);
	vec2 pos1 = tpos.xy / tpos.z;
	vec2 lightPos = pos1 * 0.5 + 0.5;

	float truepos = sign(sunPosition.z);

	if (truepos < 0.0) {
		vec2 deltaTextCoord = vec2(texcoord.st - lightPos.xy);
		vec2 textCoord = texcoord.st;
		deltaTextCoord *= 1.0 / float(NUM_SAMPLES) * grdensity;

		float avgdecay = 0.0;
		float distx = abs(texcoord.x * aspectRatio - lightPos.x * aspectRatio);
		float disty = abs(texcoord.y - lightPos.y);
		float fallof = 1.0;
		float noise = getnoise(textCoord);

		for(int i=0; i < NUM_SAMPLES ; i++) {
			textCoord -= deltaTextCoord;

			fallof *= GODRAY_LENGTH;
			float sample = step(texture2D(gdepth, textCoord + deltaTextCoord * noise * grnoise).r, 0.001);
			gr += sample * fallof;
		}
	}
	#ifdef MOONRAYS
		else {
			vec4 tpos = vec4(-sunPosition, 1.0) * gbufferProjection;
			tpos = vec4(tpos.xyz / tpos.w, 1.0);
			vec2 pos1 = tpos.xy / tpos.z;
			vec2 lightPos = pos1 * 0.5 + 0.5;

			if (truepos > 0.0) {
				vec2 deltaTextCoord = vec2(texcoord.st - lightPos.xy);
				vec2 textCoord = texcoord.st;
				deltaTextCoord *= 1.0 / float(NUM_SAMPLES) * grdensity;

				float avgdecay = 0.0;
				float distx = abs(texcoord.x * aspectRatio-lightPos.x * aspectRatio);
				float disty = abs(texcoord.y - lightPos.y);
				float fallof = 1.0;
				float noise = getnoise(textCoord);

				for(int i=0; i < NUM_SAMPLES ; i++) {
					textCoord -= deltaTextCoord;

					fallof *= 0.65;
					float sample = step(texture2D(gdepth, textCoord + deltaTextCoord * noise * grnoise).r, 0.001);
					gr += sample * fallof;
				}
			}
		}
	#endif
	return (gr/NUM_SAMPLES);
	#else
	return 0.0;
	#endif
}

void WaterDepthFog(inout vec3 color, in float waterMask) {
	#ifdef Water_DepthFog
	if (waterMask < 0.5 && isEyeInWater <= 0) return;

	float depth = texture2D(depthtex1, texcoord.st).x;
	float depthSolid = texture2D(gdepthtex, texcoord.st).x;

	vec4 viewSpacePosition1 = GetScreenSpacePosition(texcoord.st, depth);
	vec4 viewSpacePositionSolid = GetScreenSpacePosition(texcoord.st, depthSolid);

	vec3 viewVector = normalize(viewSpacePosition1.xyz);

	float waterDepth = distance(viewSpacePosition1.xyz, viewSpacePositionSolid.xyz);

	if (isEyeInWater > 0) {
		waterDepth = length(viewSpacePosition1.xyz) * 0.5;
		if (waterMask > 0.5) {
			waterDepth = length(viewSpacePositionSolid.xyz) * 0.5;
		}
	}

	float fogDensity = 0.30;
	float fogDensity2 = 0.010;
	float visibility = 1.0 / (pow(exp(waterDepth * fogDensity), 1.0));
	float visibility2 = 1.0 / (pow(exp(waterDepth * fogDensity2), 1.0));

	vec3 waterNormal = normalize(GetWaterNormals(texcoord.st));

	//Rainy water colour
	vec3 waterFogColors = vec3(0.05, 0.08, 0.1);	//murky water for the rainy weather

	//Depth water colour
	vec3 waterFogColors2 = vec3(0.0015, 0.004, 0.0098) * colorSunlight * pow(1-rainStrength, 2.0);	//Depth water colour, ALT (0.0028,0.0107,0.0180)
	waterFogColors2 *=mix(0.0, 1.0, pow(eyeBrightnessSmooth.y / 240.0, 10.0));

	//Underwater colour
	vec3 waterFogColor = vec3(0.1, 0.5, 0.8); //clear water, Under water fog colour
	waterFogColor *= 0.01 * dot(vec3(0.33333), colorSunlight);


	vec3 viewVectorRefracted = refract(viewVector, waterNormal, 1.0 / 1.3333);
	float scatter = 1.0 / (pow(saturate(dot(-lightVector, viewVectorRefracted) * 0.5 + 0.5) * 20.0, 2.0) + 0.1);


	if (isEyeInWater < 1) {
		waterFogColor = mix(waterFogColor, colorSunlight * 21.0 * waterFogColor, vec3(scatter));
	}

	//this is to change the water colour when raining
	if (rainStrength > 0.9) {
		waterFogColors2 *= mix(waterFogColors, colorSunlight * waterFogColors, vec3(scatter * (1.0 - rainStrength)));
	}

	color *= pow(vec3(0.7, 0.88, 1.0) * 0.99, vec3(waterDepth * 0.45 + 0.8));

	//this is to separate water fog either in water or out
	if (isEyeInWater < 0.9) {
		color = mix(waterFogColors2, color, saturate(visibility));

		if (rainStrength > 0.9) {
				color = mix(waterFogColors2, color, saturate(visibility)) * pow(1.65 - rainStrength, 1.0);
		}
	} else {
		color = mix(waterFogColor, color, saturate(visibility2));
	}
	#endif
}

////////////////////////////////////STRUCT COMPILERS////////////////////////////////////

void compileSky(inout SurfaceStruct surface, in vec4 viewSpacePosition1, in float SkyMask) {
	//Initialize sky surface properties
	surface.sky.tintColor = mix(colorSunlight, vec3(colorSunlight.r), vec3(0.8)); //Initializes the defualt tint color for the sky
	surface.sky.tintColor *= mix(1.0, 100.0, timeSkyDark); //Boost sky color at night																		//Scale sunglow back to be less intense

	surface.sky.sunSpot = float(CalculateSunspot(viewSpacePosition1)) * SkyMask * colorSunlight;
	surface.sky.sunSpot *= 1.0 - timeMidnight;
	surface.sky.sunSpot *= 1.0 - rainStrength;
	surface.sky.sunSpot *= 300.0;

	AddSkyGradient(surface.sky.albedo, viewSpacePosition1);
	AddSunglow(surface.sky.albedo, viewSpacePosition1);
}

void compileSpecular(inout SpecularAttributesStruct specular) {
	specular.metallic = GetMetallic(texcoord.st);     //Gets the reflectance/specularity of the surface
	specular.smoothness = GetSmoothness(texcoord.st);
	specular.fresnelPower = 5.0;                      //Default surface fresnel power

	specular.specularColor = mix(vec3(0.14), surface.albedo, vec3(surface.specular.metallic));
}

void compileShading(in vec4 viewSpacePosition, inout SurfaceStruct surface, in MaskStruct mask, in vec3 normal, inout ShadingStruct shading, inout MCLightmapStruct mcLightmap, in float waterMask) {
	//Calculate surface shading
	CalculateNdotL(surface.NdotL, normal);
	shading.direct = vec3(CalculateDirectLighting(surface.NdotL, mask)); //Calculate direct sunlight without visibility check (shadows)
	shading.sunlightVisibility = CalculateSunlightVisibility(surface.viewSpacePosition1, shading);  //Calculate shadows and apply them to direct lighting

	shading.direct *= shading.sunlightVisibility;
	shading.direct *= mix(1.0, 0.0, rainStrength);

	#ifdef Global_Illumination
		shading.direct *= pow(mcLightmap.sky, 0.1);
		shading.skylight = CalculateSkylight(normal, mask);					//Calculate scattered light from sky
		shading.heldLight = CalculateHeldLightShading(surface.viewSpacePosition);
	#else
		shading.direct *= pow(mcLightmap.sky, 0.1);
		shading.bounced = CalculateBouncedSunlight(surface.NdotL);  //Calculate fake bounced sunlight
		shading.scattered = CalculateScatteredSunlight(surface.NdotL);    //Calculate fake scattered sunlight
		shading.skylight = CalculateSkylight(normal, mask); //Calculate scattered light from sky
		shading.scatteredUp = CalculateScatteredUpLight(normal);
		shading.heldLight = CalculateHeldLightShading(surface.viewSpacePosition);
	#endif
}

void compileLightmap(inout SurfaceStruct surface, in MaskStruct mask, inout ShadingStruct shading, inout MCLightmapStruct mcLightmap, inout LightmapStruct lightmap, inout float ao) {
	//Colorize surface shading and store in lightmaps
	lightmap.sunlight = vec3(shading.direct) * colorSunlight;
	AddCloudGlow(lightmap.sunlight, surface.viewSpacePosition, mask);

	lightmap.skylight = vec3(mcLightmap.sky);
	lightmap.skylight *= mix(colorSkylight, colorBouncedSunlight, vec3(max(0.0, (1.0 - pow(mcLightmap.sky + 0.1, 0.45) * 1.0)))) + colorBouncedSunlight * (mix(Shadow_Brightness, 1.0, wetness)) * (1.0 - rainStrength);
	lightmap.skylight *= shading.skylight;
	lightmap.skylight *= mix(1.0, 5.0, float(mask.clouds));
	lightmap.skylight *= mix(1.0, 50.0, float(mask.clouds) * timeSkyDark);
	lightmap.skylight += mix(colorSkylight, colorSunlight, vec3(0.2)) * vec3(mcLightmap.sky) * 0.05;
	lightmap.skylight *= mix(1.0, 1.2, rainStrength);

	#ifdef Global_Illumination
		lightmap.skylight *= ao;

		lightmap.underwater = vec3(mcLightmap.sky) * colorSkylight;

		lightmap.torchlight = mcLightmap.torch * colorTorchlight;
		lightmap.torchlight *= ao;

		lightmap.nolight = vec3(0.05);
		lightmap.nolight *= ao;
	#else
		lightmap.bouncedSunlight = vec3(shading.bounced) * colorBouncedSunlight;
		lightmap.bouncedSunlight *= pow(vec3(mcLightmap.sky), vec3(1.75));
		lightmap.bouncedSunlight *= mix(1.0, 0.25, timeSunrise + timeSunset);
		lightmap.bouncedSunlight *= mix(1.0, 0.0, rainStrength);


		lightmap.scatteredSunlight = vec3(shading.scattered) * colorScatteredSunlight * (1.0 - rainStrength);
		lightmap.scatteredSunlight *= pow(vec3(mcLightmap.sky), vec3(1.0));

		lightmap.underwater = vec3(mcLightmap.sky) * colorSkylight;

		lightmap.torchlight = mcLightmap.torch * colorTorchlight;

		lightmap.nolight = vec3(0.05);

		lightmap.scatteredUpLight = vec3(shading.scatteredUp) * mix(colorSunlight, colorSkylight, vec3(0.0));
		lightmap.scatteredUpLight *= pow(mcLightmap.sky, 0.5);
		lightmap.scatteredUpLight *= mix(1.0, 0.1, rainStrength);
	#endif

	lightmap.heldLight = vec3(shading.heldLight);
	lightmap.heldLight *= colorTorchlight;
	lightmap.heldLight *= heldBlockLightValue * 0.070;

	//If eye is in water
	if (isEyeInWater > 0) {
		vec3 halfColor = mix(colorWaterMurk, vec3(1.0), vec3(0.5));
		lightmap.sunlight *= mcLightmap.sky * halfColor;
		lightmap.skylight *= halfColor;
		lightmap.bouncedSunlight *= 0.0;
		lightmap.scatteredSunlight *= halfColor;
		lightmap.nolight *= halfColor;
		lightmap.scatteredUpLight *= halfColor;
	}
}

void finalLighting(inout FinalStruct final, inout LightmapStruct lightmap, in float sunlightMult) {
	//Apply lightmaps to albedo and generate final shaded surface
	//final.glow.emission	= vec3(GetEmission(texcoord.st));
	final.lighting = lightmap.nolight * CAVE_BRIGHTNESS
	               + lightmap.sunlight * 0.9 * 1.5 * sunlightMult
	               + lightmap.skylight * 0.045
		#ifndef Global_Illumination
	               + lightmap.bouncedSunlight * 0.05 * sunlightMult
	               + lightmap.scatteredSunlight * 0.02 * sunlightMult
	               + lightmap.scatteredUpLight * 0.001 * sunlightMult
		#endif
		#ifdef HELD_LIGHT
	               + lightmap.heldLight * 0.05;
		#endif

	final.lighting = mix(final.lighting, final.lighting * 0.025, surface.specular.metallic);

	DoNightEye(final.lighting);
	final.lighting *= surface.albedo;
}

void compileFinal(inout SurfaceStruct surface, in MaskStruct mask, inout FinalStruct final) {
	final.underwater 			= surface.water.albedo * colorWaterBlue;
	final.underwater 			*= (lightmap.sunlight * 0.3) + (lightmap.skylight * 0.06) + (lightmap.torchlight * 0.0165) + (lightmap.nolight * 0.002);

	//Do night eye effect on outdoor lighting and sky
	DoNightEye(surface.sky.albedo);
	DoNightEye(final.underwater);


	final.glow.lava = Glowmap(surface.albedo, mask.lava, 3.0, vec3(1.0, 0.05, 0.00));

	final.glow.glowstone = Glowmap(surface.albedo, mask.glowstone, 1.9, colorTorchlight);

	final.glow.fire = surface.albedo * mask.fire;

	final.glow.torch = pow(surface.albedo * mask.torch, vec3(4.4));

	final.torchlight  = surface.albedo * lightmap.torchlight * (1.0 - mask.glowstone) * (1.0 - mask.lava);
	final.torchlight *= 1.0 - float(mask.glowstone);

	final.lighting = final.lighting
	               + final.torchlight     * 5.0
	               + final.glow.lava      * 2.6
	               + final.glow.glowstone * 2.1
	               + final.glow.fire      * 0.35
	               + final.glow.torch     * 1.15;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////MAIN/////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void main() {

	MaskStruct mask;
	mask.matIDs = GetMaterialIDs(texcoord.st);
	CalculateMasks(mask);

	//Initialize surface properties required for lighting calculation for any surface that is not part of the sky
	surface.albedo = GetAlbedoLinear(texcoord.st);
	surface.sky.albedo = surface.albedo * mask.sky;
	surface.albedo = pow(surface.albedo, vec3(1.4));

	vec3 normal                = GetNormals(texcoord.st);
	float depth                = GetDepth(texcoord.st);
	float solidDepth           = GetDepthSolid(texcoord.st);
	surface.linearDepth        = ExpToLinearDepth(depth);
	surface.viewSpacePosition  = GetScreenSpacePosition(texcoord.st);
	surface.viewSpacePosition1 = GetScreenSpacePositionSolid(texcoord.st, solidDepth);

	surface.viewVector  = normalize(surface.viewSpacePosition.rgb);

	vec4 wlv = shadowModelViewInverse * vec4(0.0, 0.0, 1.0, 0.0);
	vec3 worldLightVector = normalize(wlv.xyz);

	if (mask.water > 0.5) surface.albedo *= 1.9;

	surface.albedo *= 1.0 - float(mask.sky); //Remove the sky from surface albedo, because sky will be handled separately


	compileSky(surface, surface.viewSpacePosition, mask.sky);

	//Initialize MCLightmap values
	mcLightmap.torch = GetLightmapTorch(texcoord.st); //Gets the lightmap for light coming from emissive blocks
	mcLightmap.sky = GetLightmapSky(texcoord.st);     //Gets the lightmap for light coming from the sky

	compileSpecular(surface.specular);

	compileShading(surface.viewSpacePosition, surface, mask, normal, shading, mcLightmap, mask.water);

	float ao = 1.0;
	#ifdef Global_Illumination
		vec4 delta = vec4(0.0);
		delta.a = 1.0;
		delta = Delta(surface.albedo.rgb, normal, mcLightmap.sky);

		ao = delta.a;

		delta.rgb = mix(delta.rgb, delta.rgb * 0.025, surface.specular.metallic);
	#endif

	compileLightmap(surface, mask, shading, mcLightmap, lightmap, ao);

	surface.albedo.rgb = mix(surface.albedo.rgb, pow(surface.albedo.rgb, vec3(2.0)), vec3(mask.fire));

	surface.cloudShadow = 1.0;
	float sunlightMult = Brightness;

	#ifdef CLOUD_SHADOW
		surface.cloudShadow = CloudShadow(surface, worldLightVector);
		sunlightMult = surface.cloudShadow * 2.0 + 0.1;
	#endif

	finalLighting(final, lightmap, sunlightMult);
	compileFinal(surface, mask, final);

	vec3 finalComposite	= mix(final.lighting, vec3(1.0), final.glow.emission);

	//Apply sky to final composite
	surface.sky.albedo *= 0.85;
	surface.sky.albedo = surface.sky.albedo * surface.sky.tintColor + surface.sky.sunglow + surface.sky.sunSpot;

	finalComposite += surface.sky.albedo;		//Add sky to final image

	#ifdef Global_Illumination
		finalComposite += delta.rgb * sunlightMult;
	#endif

	CalculateUnderwaterFog(finalComposite, surface.linearDepth);

	CalculateRainFog(finalComposite.rgb, surface.viewSpacePosition, mask);

	CalculateAtmosphericScattering(finalComposite.rgb, mask, surface.viewSpacePosition, surface.linearDepth, surface.sky.sunSpot);

	float Get2DGodRays = Rays();

	CalculateVolumeClouds(finalComposite.rgb, surface.viewSpacePosition, worldLightVector, surface, mask);

	WaterDepthFog(finalComposite, mask.water);

	finalComposite *= 0.0007; //Scale image down for HDR

	finalComposite = pow(finalComposite, vec3(1.0 / 2.2));               //Convert final image into gamma 0.45 space to compensate for gamma 2.2 on displays
	finalComposite = pow(finalComposite, vec3(1.0 / BANDING_FIX_FACTOR)); //Convert final image into banding fix space to help reduce color banding

	vec4 finalCompositeCompiled = vec4(finalComposite, 1.0);

	finalCompositeCompiled.a = CrepuscularRays(surface.linearDepth);

	gl_FragData[0] = vec4(finalCompositeCompiled);
	gl_FragData[1] = vec4(mask.matIDs, length(shading.sunlightVisibility) * surface.cloudShadow * pow(mcLightmap.sky, 0.2), mcLightmap.sky, Get2DGodRays);
	gl_FragData[2] = vec4(surface.specular.metallic, surface.cloudAlpha, surface.specular.smoothness, SolidSunlightVisibility(surface.viewSpacePosition));
}
