#version 120

#define WATER_REFLECTIONS
#define DISTANT_FOG
#define GROUND_FOG

//reflection - raytrace settings
const float stp = 0.075; // starting step (0.1)
const float ref = 0.1007; // refinement (step reduced by 0.1)
const float inc = 2.4; // log increment factor (log 2.4 search)
const int maxf = 5;     // number of refinements (5 + 1 times)
const int eff = 60;     // effort (do 60 number of searches)

varying vec4 texcoord;

varying vec4 lightVector;
varying vec3 sunlight;

varying vec3 caustics;

varying float transition_fading;
varying float fog_level;
varying float groundFog_level;
varying float gp00;
varying float gp11;
varying float gp22;
varying float gp32;
varying float gp2232;
varying float igp00;
varying float igp11;

uniform sampler2D gcolor;
uniform sampler2D gdepth;   // heavily bump water normal from gbuffer_water.fsh
uniform sampler2D gnormal;
uniform sampler2D composite;
uniform sampler2D gaux1;
uniform sampler2D gaux2;    // rain from gbuffer_weather.fsh
uniform sampler2D gaux3;    // clouds from gbuffer_textured.fsh
uniform sampler2D depthtex0;
uniform sampler2D depthtex1;
uniform sampler2D depthtex2;
uniform sampler2D noisetex;

uniform mat4 gbufferProjection;
uniform mat4 gbufferModelViewInverse;
uniform mat4 gbufferPreviousProjection;
uniform mat4 gbufferPreviousModelView;

uniform vec3 sunPosition;
uniform vec3 cameraPosition;
uniform vec3 previousCameraPosition;

uniform float far;
uniform float near;
uniform float rainStrength;
uniform float wetness;
uniform float frameTimeCounter;
uniform float viewWidth;
uniform float viewHeight;

uniform int isEyeInWater;
uniform int worldTime;

float pw = 1.0 / viewWidth;
float ph = 1.0 / viewHeight;

vec3 proj(in vec3 pos) {
    return vec3(pos.x * gp00, pos.y * gp11, gp2232) / pos.z + 0.5;
}

vec3 invproj(in vec3 p) {
    vec3 pos = p - 0.5;
    float z = gp32 / (pos.z + gp22);
    return vec3(pos.x * igp00, pos.y * igp11, -1.0) * z;
}

float fadeout(in vec2 coord) {
    float cdist = distance(coord, vec2(0.5));
    return 1.01 + cdist * cdist * (cdist - 1.0) * 6.8;
}

vec4 raytrace(in vec3 eye, in vec3 refvector, in float eyez) {
    float delta = stp;
    float sum = delta * 1.5;  // offset to remove artifacts
    int sr = 0;
    for(int i = 0; i < eff; ++i) {
        vec3 offset = eye + sum * refvector;
        vec3 pos = proj(offset);
        if (pos.z > 1.0 || pos.z < 0.0 || pos.y > 1.0 || pos.y < 0.0 || pos.x > 1.0 || pos.x < 0.0) break;
        float posz = texture2D(depthtex2, pos.st).r;    // remove objects before reflection zone
        float dist = length(offset) - length(invproj(vec3(pos.st, posz)));    // depthtex2 - no hand effect
        if (posz > eyez - 0.001 && dist > -0.0001 && dist < delta * sum + float(sr) * -0.0225 + 0.15) {
            if (sr > maxf) {
                vec4 color = texture2D(composite, pos.st);
                color.a = fadeout(pos.st);
                return color;
            }
            ++sr;
            sum -= delta;
            delta *= ref;
        }
        delta *= inc;
        sum += delta;
    }
    return vec4(0.0);
}

float ld(in float depth) {
    return (2.0 * near) / (far + near - depth * (far - near));
}

//////////////////////////////main//////////////////////////////

void main() {
    vec4 color = texture2D(gcolor, texcoord.st);

    float matflag = texture2D(gaux1, texcoord.xy).g;
    float hand = 1.0;
    if(matflag > 0.35 && matflag < 0.45) {
        hand = 0.3;    // scale rain effect on hand (0.0 - 0.9)
    }
    vec3 eye = invproj(vec3(texcoord.st, texture2D(depthtex0, texcoord.st).r));
    if (isEyeInWater < 1) {
        vec3 normal = texture2D(gnormal, texcoord.st).rgb * 2.0 - 1.0;

        const float fland = 0.05;
        const float fwater = 0.95;
        if (matflag > fland && matflag < fwater) {
            float ssao = 0.0;
            float range = clamp(distance(proj(eye + vec3(1.0)).xy, texcoord.xy), 7.5 * pw, 60.0 * pw);
            float dist = ld(texture2D(depthtex0, texcoord.st).r);
            for (float i = 0.0; i < 5.1; i += 1.3) {
                for (float j = 0.25; j < 1.1; j += 0.25) {
                    vec2 offset = vec2(cos(float(i)), sin(float(i))) * (float(j) * 0.25) * range + texcoord.xy;
                    vec3 offpos = invproj(vec3(offset, texture2D(depthtex0, offset).r));
                    float angle = min(1.0 - dot(normal, normalize(offpos - eye)), 1.0);
                    float ldist = min(abs(ld(texture2D(depthtex0, offset).r) - dist) * 65.0, 1.0);
                    float effect = min(ldist * ldist + angle * angle, 1.0);
                    ssao += effect * effect * effect;
                }
            }
            ssao *= 0.0625;
            color.rgb *= ssao * 0.5 + 0.5;
        }

        color.rgb += texture2D(gaux2, texcoord.xy).rgb * texture2D(gaux2, texcoord.xy).a * 0.5 * hand;	// draw rain

        vec3 fogclr = mix(gl_Fog.color.rgb, vec3(0.3, 0.4, 0.5), 0.2);
        vec3 neye = normalize(eye);
        vec3 nnormal = normalize(normal);
        const float freflect = 0.85;
        if (matflag > freflect) {
#ifdef WATER_REFLECTIONS
            float fresnel = clamp(pow(1.0 + dot(nnormal, neye), 5.0), 0.0, 1.0);
            vec3 fnormal = mix(normal, texture2D(gdepth, texcoord.st).rgb * 2.0 - 1.0, fresnel);
            vec4 reflection = raytrace(eye, normalize(reflect(neye, normalize(fnormal))), texture2D(depthtex2, texcoord.st).r);
            reflection.rgb = mix(fogclr, reflection.rgb, reflection.a); // add sky/fog color to reflection
            float reflectionStrength = matflag * 5.0 - 3.8;
            color.rgb += reflection.rgb * fresnel * reflectionStrength * (1.0 - rainStrength * 0.6);
            //color.r = mod(-eye.z, 5.0) > 0.95 * 5.0? 1.0 : color.r;   // show lines on water
            color.rgb += color.a * reflection.rgb * reflectionStrength * 0.5;  // spec from composite.fsh here to remove shadows
            color.r *= 0.8;    // set 80 percent red on water
#endif
        }
        if (matflag > fwater) {
            float f0 = 0.3 * max(dot(caustics, nnormal), 0.00001); // basic caustics
            f0 *= step(f0, 0.02);  // remove artifacts from flowing water
            color.g += f0;
        }
        if (hand > 0.9) {
            vec3 feye = mix(eye, invproj(vec3(texcoord.st, texture2D(depthtex1, texcoord.st).r)), texture2D(gaux3, texcoord.st).a); // fog cloud needs depthtex1, water needs depthtex0
            vec4 worldpos = gbufferModelViewInverse * vec4(feye, 1.0);
#ifdef DISTANT_FOG
            float density = 96.0 / max(120.0 + worldpos.y, 52.7) - 0.8;
            float dist = (1.0 + wetness * 0.2 + rainStrength * 0.2) * 280.0 / clamp(280.0 - length(feye) * fog_level * 215.0 / far, 132.0, 242.0) - 1.15 + wetness * 0.1;

            float volumetric_cone = pow(max(dot(neye, lightVector.xyz), 0.00001), 5.0) * lightVector.a;
            fogclr += sunlight * transition_fading * volumetric_cone * (0.9 - rainStrength * 0.8);   // inject moon color into the fog

            float fog = clamp(mix(density, dist, 0.6), 0.0, 1.0);
            fog *= 1.0 - 0.5 * step(matflag, fland) * (1.0 - rainStrength * 0.9);  // reduce fog in sky

            color.rgb = mix(color.rgb, fogclr, fog);
            //color.r = (range > 175.0 && range < 176.0) ? 1.0 : color.r;
            //color.g = (worldpos.y > -52.0 && worldpos.y < -51.0) ? 1.0 : color.g;
            //color.b = (worldpos.y > -0.1 && worldpos.y < 0.0) ? 1.0 : color.b;
#endif
#ifdef GROUND_FOG
            if ((matflag > 0.55 && matflag < 0.65) || (matflag > 0.05 && matflag < 0.15) || (matflag > freflect)) {
                vec2 ncoord = (worldpos.xz + cameraPosition.xz) * 0.22;
                ncoord.x += frameTimeCounter * 0.125;
                ncoord.y += worldpos.y * 0.11;
                float noise = texture2D(noisetex, fract(ncoord * 0.03125)).r;
                noise += texture2D(noisetex, fract(ncoord * 0.0625)).r * 0.5;
                noise += texture2D(noisetex, fract(ncoord * 0.125)).r * 0.25;
                noise += texture2D(noisetex, fract(ncoord * 0.25)).r * 0.125;
                noise += texture2D(noisetex, fract(ncoord * 0.5)).r * 0.0625;
                noise += texture2D(noisetex, fract(ncoord * 1.0)).r * 0.03125;
                noise += texture2D(noisetex, fract(ncoord * 2.0)).r * 0.015625;
                float range = length(feye);
                float ground = max(22.0 * range - range * range, 0.0) * 0.001;
                float groundFog = ground * noise * groundFog_level * 1.2 * (1.0 - rainStrength * 0.8);
                //color.b = (cameraPosition.y > 121.0) ? 1.0 : color.b;
                color.rgb = mix(color.rgb, vec3(0.2, 0.4, 0.6), groundFog);
                
                // mountain fog
                float height = clamp((cameraPosition.y - 90.0) / 20.0, 0.0, 1.5);
                float mountainFog = ground * noise * height;
                color.rgb = mix(color.rgb, vec3(0.8, 0.8, 0.8), mountainFog);
            }
#endif
        }
    }
    else {  // eye in water
        color = texture2D(gcolor, vec2(texcoord.s + 0.001 * sin(frameTimeCounter * 5.0 + texcoord.t * 20.0), texcoord.t));   // basic underwater wave
    }

    //calculate sun occlusion at vec2(0.0) for lens and sky occlusion at vec2(1.0) for sunrays
    float visiblesun = 0.0;
    vec4 tpos = vec4(sunPosition, 1.0) * gbufferProjection; // a transposed projection
    vec2 lightPos = (tpos.xy / tpos.z) * 0.5 + 0.5;
    if (((texcoord.x < pw && texcoord.y < ph) || ((texcoord.x + pw) > 1.0 && (texcoord.y + ph) > 1.0)) && (lightPos.x < 1.4 && lightPos.x > -0.4 && lightPos.y < 1.4 && lightPos.y > -0.4 && sunPosition.z < -0.4)) {
        const float fsun = 0.05;
        float mult = 12.0 + 144.0 * texcoord.x;
        for (int i = 0; i < 9; ++i) {
            for (int j = 0; j < 9; ++j) {
                vec2 offsun = lightPos + vec2(pw * (float(i) - 4.0) * mult, ph * (float(j) - 4.0) * mult);
                if (offsun.x < 1.0 && offsun.x > 0.0 && offsun.y < 1.0 && offsun.y > 0.0 && texture2D(gaux1, offsun).g < fsun) {
                    visiblesun += 1.0;
                }
            }
        }
        visiblesun /= 81.0;    // 9.0 x 9.0 = 81.0
        visiblesun *= 1.0 - rainStrength * 0.9;
        visiblesun *= clamp((13050.0 - float(worldTime)) * 0.002, 0.0, 1.0) + clamp((float(worldTime) - 22950.0) * 0.002, 0.0, 1.0);
    }
    
    vec4 wpos = gbufferModelViewInverse * vec4(eye, 1.0);  // screen space motion blur
    wpos.xyz += cameraPosition - previousCameraPosition;
    vec4 prevPos = gbufferPreviousProjection * gbufferPreviousModelView * wpos;
    vec2 velocity = vec2(prevPos.xy / prevPos.w * 0.5 + 0.5 - texcoord.xy);
    velocity = max(velocity * 0.5 + 0.5, 0.00001);
    velocity.x = pow(velocity.x, 3.0);
    velocity.y = pow(velocity.y, 3.0);

/* DRAWBUFFERS:06 */
    gl_FragData[0] = vec4(color.rgb, visiblesun);    // gcolor in final.fsh
    gl_FragData[1] = vec4(velocity, 1.0, 1.0);  // gaux3 in final.fsh
}

//////////////////////////////main//////////////////////////////
