#version 450 compatibility

#define SHADOW_MAP_BIAS 0.80
#define FRAGMENT_SCALE 1.0
#define VERTEX_SCALE 0.5

#define EXTENDED_SHADOW_DISTANCE

#define Global_Illumination

#define GI_QUALITY 2.0 //[1.0 1.5 2.0 3.0 4.0] //sets the Quality of the GI Calculation
#define GI_Boost true

//#define SEUS_GI

#define NEW_HQ_GI				//Turn off for lower Quality New GI
#define NEW_GI_QUALITY 128 //[128 256 512]

//#define HBAO			//Continuum's new AO (By Joey) 
#define SSAO			//Sonic Ethers AO

//////////////////////////////INTERNAL VARIABLES////////////////////////////////////////////////////////////
//////////////////////////////INTERNAL VARIABLES////////////////////////////////////////////////////////////
//Do not change the name of these variables or their type. The Shaders Mod reads these lines and determines values to send to the inner-workings
//of the shaders mod. The shaders mod only reads these lines and doesn't actually know the real value assigned to these variables in GLSL.
//Some of these variables are critical for proper operation. Change at your own risk.

const int       shadowMapResolution  = 2048;		 // Shadow Resolution. 1024 = Lowest Quality. 4096 = Highest Quality [1024 2048 3072 4096]
const float 	shadowDistance 			= 140;	// shadowDistance. 60 = Lowest Quality. 200 = Highest Quality [60 100 120 160 180 200]
const float 	shadowIntervalSize 		= 4.0;
const bool 		shadowHardwareFiltering0 = true;

const bool 		shadowtex1Mipmap = true;
const bool 		shadowtex1Nearest = true;
const bool 		shadowcolor0Mipmap = true;
const bool 		shadowcolor0Nearest = false;
const bool 		shadowcolor1Mipmap = true;
const bool 		shadowcolor1Nearest = false;

const int 		R8 						= 0;
const int 		RG8 					= 0;
const int 		RGB8 					= 1;
const int 		RGBA8 					= 1;
const int 		RGB16 					= 2;
const int 		RGBA16 					= 2;
const int 		colortex0Format 		= RGB16;
const int 		colortex1Format 		= RGB8;
const int 		colortex2Format 		= RGB16;
const int 		colortex3Format 		= RGB8;
const int 		colortex4Format 		= RGBA8;
const int 		colortex5Format 		= RGB8;

const float 	wetnessHalflife 		= 100.0;
const float 	drynessHalflife 		= 40.0;
const float 	centerDepthHalflife 	= 2.0;
const float 	eyeBrightnessHalflife 	= 10.0;

const float		sunPathRotation 		= -40.0;
const float 	ambientOcclusionLevel 	= 0.65;

const int 		noiseTextureResolution  = 64;

uniform sampler2D colortex2;
uniform sampler2D colortex1;
uniform sampler2D gdepthtex;
uniform sampler2D shadowtex1;
uniform sampler2D shadowcolor;
uniform sampler2D shadowcolor1;
uniform sampler2D noisetex;
uniform sampler2D gdepth;
uniform sampler2DShadow shadow;

uniform mat4 gbufferModelViewInverse;
uniform mat4 shadowModelView;
uniform mat4 shadowModelViewInverse;
uniform mat4 shadowProjection;
uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferProjection;

uniform vec3 previousCameraPosition;

uniform float viewWidth;
uniform float viewHeight;
uniform float aspectRatio;
uniform float rainStrength;
uniform float sunAngle;
uniform float frameTimeCounter;

uniform int isEyeInWater;

in vec3 lightVector;

in vec2 texcoord;

/* DRAWBUFFERS:46 */

struct MaskStruct {
	float materialIDs;
	float matIDs;

	float fullbright;
	float bit1;
	float bit2;
	float bit3;

	float sky;

	float grass;
	float leaves;
	float water;
} mask;

//////////////////////////////FUNCTIONS////////////////////////////////////////////////////////////
//////////////////////////////FUNCTIONS////////////////////////////////////////////////////////////

vec3 GetNormals(in vec2 coord) {
	return texture2DLod(colortex2, coord.st, 0).xyz * 2.0 - 1.0;
}

float GetDepth(in vec2 coord) {
	return texture2D(gdepthtex, coord.st).x;
}

vec4 GetViewSpacePosition(in vec2 coord) {		//Function that calculates the view-space position of the objects in the scene using the depth texture and the texture coordinates of the full-screen quad
	float depth = GetDepth(coord);
	vec4 fragposition = gbufferProjectionInverse * vec4(coord.s * 2.0 - 1.0, coord.t * 2.0 - 1.0, depth * 2.0 - 1.0, 1.0);
		 fragposition /= fragposition.w;

	return fragposition;
}

vec4 GetViewSpacePosition(in vec2 coord, in float depth) {		//Function that calculates the view-space position of the objects in the scene using the depth texture and the texture coordinates of the full-screen quad
	vec4 fragposition = gbufferProjectionInverse * vec4(vec3(coord.st, depth) * 2.0 - 1.0, 1.0);
		 fragposition /= fragposition.w;

	return fragposition;
}

vec4 ViewSpaceToWorldSpace(in vec4 viewSpacePosition) {
	vec4 pos = gbufferModelViewInverse * viewSpacePosition;
	return pos / pos.w;
}

vec4 WorldSpaceToShadowSpace(in vec4 worldSpacePosition) {
	vec4 pos = shadowProjection * shadowModelView * worldSpacePosition;
	return pos /= pos.w;
}

vec4 BiasShadowProjection(in vec4 projectedShadowSpacePosition) {
	#ifndef EXTENDED_SHADOW_DISTANCE
		float dist = length(projectedShadowSpacePosition.xy);
	#else
		vec2 pos = abs(projectedShadowSpacePosition.xy * 1.165);
		float dist = pow(pow(pos.x, 8) + pow(pos.y, 8), 1.0 / 8.0);
	#endif

	float distortFactor = (1.0 - SHADOW_MAP_BIAS) + dist * SHADOW_MAP_BIAS;

	projectedShadowSpacePosition.xy /= distortFactor;

	#ifdef EXTENDED_SHADOW_DISTANCE
		projectedShadowSpacePosition.z /= 4.0;
	#endif

	return projectedShadowSpacePosition;
}

vec2 BiasShadowMap(in vec2 ShadowMapPosition) {
	ShadowMapPosition = ShadowMapPosition * 2.0 - 1.0;

	#ifndef EXTENDED_SHADOW_DISTANCE
		float dist = length(ShadowMapPosition.xy);
	#else
		vec2 pos = abs(ShadowMapPosition.xy * 1.165);
		float dist = pow(pow(pos.x, 8) + pow(pos.y, 8), 1.0 / 8.0);
	#endif

	float distortFactor = (1.0 - SHADOW_MAP_BIAS) + dist * SHADOW_MAP_BIAS;

	ShadowMapPosition /= distortFactor;

	ShadowMapPosition = ShadowMapPosition * 0.5 + 0.5;

	return ShadowMapPosition;
}

vec3 CalculateNoisePattern1(const float size) {
	vec2 coord = texcoord * VERTEX_SCALE;

	coord *= vec2(viewWidth, viewHeight);
	coord = mod(coord, vec2(size));
	coord /= noiseTextureResolution;

	return texture2D(noisetex, coord).xyz;
}


float 	GetMaterialIDs(in vec2 coord) {			//Function that retrieves the texture that has all material IDs stored in it
	return texture2D(gdepth, coord).r;
}

float 	GetMaterialMask(in vec2 coord ,const in int ID, in float matID) {
	matID = (matID * 255.0f);

	//Catch last part of sky
	if (matID > 254.0f) {
		matID = 0.0f;
	}

	if (matID == ID) {
		return 1.0f;
	} else {
		return 0.0f;
	}
}

float GetWaterMask(in vec2 coord, in float matID) {					//Function that returns "true" if a pixel is water, and "false" if a pixel is not water.
	matID = (matID * 255.0f);

	if (matID >= 35.0f && matID <= 51) {
		return 1.0f;
	} else {
		return 0.0f;
	}
}


void CalculateMasks(inout MaskStruct mask, in vec2 coord) {
	mask.materialIDs	= GetMaterialIDs(coord);
	mask.matIDs			= mask.materialIDs;

	mask.sky 			= GetMaterialMask(texcoord.st, 0, mask.matIDs);

	mask.grass 			= GetMaterialMask(texcoord.st, 2, mask.matIDs);
	mask.leaves	 		= GetMaterialMask(texcoord.st, 3, mask.matIDs);
	mask.water		 	= GetWaterMask(texcoord.st, mask.matIDs);
}

vec2 RotateDirection(vec2 Dir, vec2 CosSin) {
    return vec2(Dir.x * CosSin.x - Dir.y * CosSin.y,
                Dir.x * CosSin.y + Dir.y * CosSin.x);
}

float calculateHBAO(in vec2 coord) {
	//AO Modifiers..
	const float samplingRadius = 0.6;
	const int samplingDirections = 8;
	const float samplingStep = 0.004;
	const int numSampleSteps = 4;
	const float tangentBias = 0.2;
	const float PI = 3.14159265;
	//End AO Modifiers

	vec3 viewPosition = GetViewSpacePosition(coord).xyz;
	vec3 viewNormal = GetNormals(coord);

	float total = 0.0;
	float sampleDirectionIncrement = 2.0 * PI / float(samplingDirections);

	vec2 randomRotation = CalculateNoisePattern1(3).xy;

	for(uint i = 0; i < samplingDirections; i++) {
		//Holding off jittering as long as possible.
		float samplingAngle = i * sampleDirectionIncrement; //Azimuth angle theta in the paper.
		vec2 sampleDirection = vec2(cos(samplingAngle * randomRotation.x), sin(samplingAngle * randomRotation.y));

		
			//Random Rotation Defined by nVidia
			sampleDirection = RotateDirection(vec2(cos(samplingAngle), sin(samplingAngle)), randomRotation.xy);
			sampleDirection.y = -sampleDirection.y;
		

		float tangentAngle = acos(dot(vec3(sampleDirection, 0.0), viewNormal)) - (0.5 * PI) + tangentBias;
		float horizonAngle = tangentAngle;
		vec3 lastDiff = vec3(0.0);

		for(uint j = 0; j < numSampleSteps; j++) {
			//Marching Time
			vec2 sampleOffset = float(j + 1) * samplingStep * sampleDirection;
			vec2 offset = coord + sampleOffset;

			vec3 offsetViewPosition = GetViewSpacePosition(offset).xyz;
			vec3 differential = offsetViewPosition - viewPosition;

			if(length(differential) < samplingRadius) { //skip samples outside local sample space
				lastDiff = differential;
				float elevationAngle = atan(differential.z / length(differential.xy));
				horizonAngle = max(horizonAngle, elevationAngle);
			}
		}
		float normalDiff = length(lastDiff) / samplingRadius;
		float atten = 1.0 - pow(normalDiff, 2);

		float AO = clamp(atten * (sin(horizonAngle) - sin(tangentAngle)), 0.0, 1.0);
		total += 1.0 - AO;
	}
	total /= samplingDirections;

	return total;
}

float CalculateAO(in vec4 viewSpacePosition, in vec3 normal, in vec2 coord, const in int sampleCount, in vec3 dither) {
	//Determine origin position
	vec3 origin = viewSpacePosition.xyz;

	vec3 randomRotation = normalize(dither.xyz * vec3(2.0, 2.0, 1.0) - vec3(1.0, 1.0, 0.0));

	vec3 tangent = normalize(randomRotation - normal * dot(randomRotation, normal));
	vec3 bitangent = cross(normal, tangent);
	mat3 tbn = mat3(tangent, bitangent, normal);

	float aoRadius   = 1.0;
	float zThickness = 0.25 * -viewSpacePosition.z;

	vec3 	samplePosition 		= vec3(0.0);
	vec4 	sampleViewSpace 	= vec4(0.0);
	float 	sampleDepth 		= 0.0;

	float ao = 0.0;

	for (int i = 0; i < sampleCount; i++) {
		vec3 kernel = vec3(texture2D(noisetex, vec2(0.1 + i / 64.0)).x * 2.0 - 1.0,
						   texture2D(noisetex, vec2(0.1 + i / 64.0)).y * 2.0 - 1.0,
						   texture2D(noisetex, vec2(0.1 + i / 64.0)).z);

		kernel = normalize(kernel);
		kernel *= dither.x + 0.01;

		samplePosition = tbn * kernel;
		samplePosition = origin + samplePosition * aoRadius;

		sampleViewSpace = gbufferProjection * vec4(samplePosition, 0.0);
		sampleViewSpace.xyz /= sampleViewSpace.w;
		sampleViewSpace.xyz = sampleViewSpace.xyz * 0.5 + 0.5;

		//Check depth at sample point
		sampleDepth = GetViewSpacePosition(sampleViewSpace.xy).z;

		//If point is behind geometry, buildup AO
		if (sampleDepth >= samplePosition.z && sampleDepth - samplePosition.z < zThickness) {
			ao += 1.0;
		}
	}

	ao /= sampleCount;
	ao = 1.0 - ao;

	return ao;
}

vec3 CalculateGI(in vec2 coord, in vec4 viewSpacePosition, in vec3 normal, const in float radius, const in float quality, in vec3 noisePattern, in MaskStruct mask) {
	float NdotL = dot(normal, lightVector);

	vec3 shadowSpaceNormal = (shadowModelView * gbufferModelViewInverse * vec4(normal, 0.0)).xyz;

	vec4 position = ViewSpaceToWorldSpace(viewSpacePosition);
		 position = WorldSpaceToShadowSpace(position);
		 position = position * 0.5 + 0.5;

	#ifndef EXTENDED_SHADOW_DISTANCE
		if (position.x <= 0.0 || position.x >= 1.0 ||
			position.y <= 0.0 || position.y >= 1.0 ||
			position.z <= 0.0 || position.z >= 1.0
			) return vec3(0.0);
	#endif

	float fademult 	= 0.15;
	//float lightMult	= clamp(1.0 - (length(viewSpacePosition.xyz) - shadowDistance) / shadowDistance, 0.0, 1.0);
	float lightMult	= 1.0;

	if (GI_Boost) {
		vec4 biasPos = BiasShadowProjection(position * 2.0 - 1.0) * 0.5 + 0.5;
		float sunlight = shadow2DLod(shadow, vec3(biasPos.xyz), 0).x;
		lightMult *= clamp(1.0 - NdotL * 4.0 * pow(sunlight, 8.0), 0.0 , 1.0);
		if (lightMult < 0.01) return vec3(0.0);

		float skylight = texture2D(colortex1, coord).b;
		if (skylight <= 0.01) return vec3(0.0);
	}

	const float range		= 2.0;
	const float A			= range * radius / 2048.0;
	const float interval	= 1.0 / quality;
	float depthLOD			= 2.0 * clamp(1.0 - length(viewSpacePosition.xyz) / shadowDistance, 0.0, 1.0);
	float sampleLOD			= 5.0 * clamp(1.0 - length(viewSpacePosition.xyz) / shadowDistance, 0.0, 1.0);
	vec2 V					= noisePattern.xy - 0.5;
	vec3 light				= vec3(0.0);
	int samples				= 0;

	for (float I = -range; I <= range; I += interval) {
		for (float O = -range; O <= range; O += interval) {
			vec2 randomPos		= (vec2(I, O) + V * interval) * A;
			vec3 samplePos		= vec3(position.xy + randomPos, 0.0);
			vec2 biasedSample	= BiasShadowMap(samplePos.xy);
			samplePos.z			= texture2DLod(shadowtex1, biasedSample, depthLOD).x;
			#ifdef EXTENDED_SHADOW_DISTANCE
				samplePos.z		= ((samplePos.z * 2.0 - 1.0) * 4.0) * 0.5 + 0.5;
			#endif
			vec3 sampleDir		= normalize(samplePos.xyz - position.xyz);
			vec3 shadowNormal	= texture2DLod(shadowcolor1, biasedSample, sampleLOD).xyz * 2.0 - 1.0;
			shadowNormal.xy		*= -1.0;
			float NdotS			= max(0.0, dot(shadowSpaceNormal, sampleDir * vec3(1.0, 1.0, -1.0)));
			float SdotN			= max(0.0, dot(shadowNormal, sampleDir));

			if (mask.leaves + mask.grass > 0.5) NdotS = 1.0;

			float falloff = length(samplePos.xyz - position.xyz);
			falloff = max(falloff, 0.005);
			falloff = 1.0 / (pow(falloff * (13600.0 / radius), 2.0) + 0.0001 * interval);
			falloff = max(0.0, falloff - 9e-05);

			vec3 sampleColor = pow(texture2DLod(shadowcolor, biasedSample, sampleLOD).rgb, vec3(2.2));

			light += sampleColor * falloff * SdotN * NdotS;
		}
	}

	light /= pow(4.0 / interval + 1.0, 2.0);

	light *= mix(0.0, 1.0, lightMult);

	return light * 400.0;// / radius * 16.0;
}

vec3 CalculateGINew_LQ(in vec2 coord, in vec4 viewSpacePosition, in vec3 normal, const in float radius, in vec3 noisePattern, in MaskStruct mask) {
	vec3 shadowSpaceNormal = (shadowModelView * gbufferModelViewInverse * vec4(normal, 0.0)).xyz;

	vec4 position = ViewSpaceToWorldSpace(viewSpacePosition);
	position = WorldSpaceToShadowSpace(position);
	position = position * 0.5 + 0.5;

	float sampleLOD	= 3.0 * clamp(1.0 - length(viewSpacePosition.xyz) / shadowDistance, 0.0, 1.0);

	vec2 noiseOffset = noisePattern.xy - 0.5;
	noiseOffset *= 3;
	vec3 light = vec3(0.0);

	const int sampleRadius =  7;
	const float pi = 3.14;

	if(length(viewSpacePosition.xyz) < 64) {
		for(int i = 0; i < NEW_GI_QUALITY; i++) {
			float percentage_done = float(i) / float(NEW_GI_QUALITY);
			float dist_from_center = sampleRadius * percentage_done;

			float theta = percentage_done * (NEW_GI_QUALITY / 16) * pi;
			vec2 offset = vec2(cos(theta), sin(theta)) * (dist_from_center * 6);
			offset += noiseOffset;
			offset /= shadowMapResolution;

			vec3 samplePos = vec3(position.xy + offset, 0.0);
			vec2 biasedSample	= BiasShadowMap(samplePos.xy);
			samplePos.z	= texture2DLod(shadowtex1, biasedSample, 0.0).x;

			#ifdef EXTENDED_SHADOW_DISTANCE
				samplePos.z	= ((samplePos.z * 2.0 - 1.0) * 4.0) * 0.5 + 0.5;
			#endif

			vec3 sampleDir = normalize(samplePos.xyz - position.xyz);
			vec3 shadowNormal	= texture2DLod(shadowcolor1, biasedSample, 0).xyz * 2.0 - 1.0;
			shadowNormal.xy	*= -1.0;

			float viewNormalCoeff	= max(0.0, dot(shadowSpaceNormal, sampleDir * vec3(1.0, 1.0, -1.0)));
			float shadowNormalCoeff	= max(0.0, dot(shadowNormal, sampleDir));

			if (mask.leaves + mask.grass > 0.5) viewNormalCoeff = 1.0;

			float falloff = length(samplePos.xyz - position.xyz);
			falloff = 1.0 / (pow(falloff * (40000.0 / radius), 2.0) + 0.00001);
			falloff = max(0.0, falloff);

			vec3 sampleColor = pow(texture2DLod(shadowcolor, biasedSample, sampleLOD).rgb, vec3(2.2));

			light += sampleColor * falloff * shadowNormalCoeff * viewNormalCoeff;
		}
	}
	return light * 4;
}

vec3 CalculateGINew(in vec2 coord, in vec4 viewSpacePosition, in vec3 normal, const in float radius, const in float quality, in vec3 noisePattern, in MaskStruct mask) {
	float NdotL = dot(normal, lightVector);

	vec3 shadowSpaceNormal = (shadowModelView * gbufferModelViewInverse * vec4(normal, 0.0)).xyz;

	vec4 position = ViewSpaceToWorldSpace(viewSpacePosition);
		 position = WorldSpaceToShadowSpace(position);
		 position = position * 0.5 + 0.5;

	#ifndef EXTENDED_SHADOW_DISTANCE
		if (position.x <= 0.0 || position.x >= 1.0 ||
			position.y <= 0.0 || position.y >= 1.0 ||
			position.z <= 0.0 || position.z >= 1.0
			) return vec3(0.0);
	#endif

	float fademult 	= 0.15;
	float lightMult	= 1.0;

	if (GI_Boost) {
		vec4 biasPos = BiasShadowProjection(position * 2.0 - 1.0) * 0.5 + 0.5;
		float sunlight = shadow2DLod(shadow, vec3(biasPos.xyz), 0).x;
		lightMult *= clamp(1.0 - NdotL * 4.0 * pow(sunlight, 8.0), 0.0 , 1.0);
		if (lightMult < 0.01) return vec3(0.0);

		float skylight = texture2D(colortex1, coord).b;
		if (skylight <= 0.01) return vec3(0.0);
	}

	const float interval = 1.0 / quality;

	float sampleLOD	= 3.0 * clamp(1.0 - length(viewSpacePosition.xyz) / shadowDistance, 0.0, 1.0);

	vec2 noiseOffset = noisePattern.xy - 0.5;
	noiseOffset *= 3;
	vec3 light = vec3(0.0);
	int samples	= 0;

	#define GI_SAMPLE_RADIUS 7
	#define PI 3.14

	for(int i = 0; i < NEW_GI_QUALITY; i++) {
		float percentage_done = float(i) / float(NEW_GI_QUALITY);
		float dist_from_center = GI_SAMPLE_RADIUS * percentage_done;

		float theta = percentage_done * (NEW_GI_QUALITY / 16) * PI;
		vec2 offset = vec2(cos(theta), sin(theta)) * (dist_from_center * 6);
		offset += noiseOffset;
		offset /= shadowMapResolution;

		vec3 samplePos = vec3(position.xy + offset, 0.0);
		vec2 biasedSample	= BiasShadowMap(samplePos.xy);
		samplePos.z	= texture2DLod(shadowtex1, biasedSample, 0.0).x;

		#ifdef EXTENDED_SHADOW_DISTANCE
			samplePos.z	= ((samplePos.z * 2.0 - 1.0) * 4.0) * 0.5 + 0.5;
		#endif

		vec3 sampleDir = normalize(samplePos.xyz - position.xyz);
		vec3 shadowNormal	= texture2DLod(shadowcolor1, biasedSample, 0).xyz * 2.0 - 1.0;
		shadowNormal.xy	*= -1.0;

	//return shadowNormal;

		float viewNormalCoeff	= max(0.0, dot(shadowSpaceNormal, sampleDir * vec3(1.0, 1.0, -1.0)));
		float shadowNormalCoeff	= max(0.0, dot(shadowNormal, sampleDir));

		if (mask.leaves + mask.grass > 0.5) viewNormalCoeff = 1.0;

		float falloff = length(samplePos.xyz - position.xyz);
		falloff = max(falloff, 0.005);
		falloff = 1.0 / (pow(falloff * (40000.0 / radius), 2.0) + 0.0001);
		falloff = max(0.0, falloff - 9e-05);

		vec3 sampleColor = pow(texture2DLod(shadowcolor, biasedSample, sampleLOD).rgb, vec3(2.2));
		//return sampleColor;

		light += sampleColor * falloff * shadowNormalCoeff * viewNormalCoeff;
	}

	light /= NEW_GI_QUALITY * 5 * radius;
	light *= mix(0.0, 1.0, lightMult);

	return light * 12000.0;
}

//////////////////////////////MAIN////////////////////////////////////////////////////////////
//////////////////////////////MAIN////////////////////////////////////////////////////////////

void main() {

	CalculateMasks(mask, texcoord);
	if (mask.sky + mask.water > 0.5) { gl_FragData[0] = vec4(0.0, 0.0, 0.0, 1.0); return; }

	vec3	normal 				= GetNormals(texcoord);
	float	depth  				= GetDepth(texcoord);
	vec4	viewSpacePosition	= GetViewSpacePosition(texcoord, depth);
	vec3	noisePattern		= CalculateNoisePattern1(4);

	vec4 light = vec4(0.0, 0.0, 0.0, 1.0);

	#ifdef HBAO
		light.a = calculateHBAO(texcoord.st);
	#endif
	
	#ifdef SSAO
		light.a		= CalculateAO(viewSpacePosition, normal, texcoord, 8, noisePattern);
	#endif

	if (isEyeInWater > 0.5 || rainStrength > 0.99) { gl_FragData[0] = light; return; }
#ifdef Global_Illumination
	#ifdef NEW_HQ_GI
		light.rgb	= CalculateGINew(texcoord, viewSpacePosition, normal, 64.0, 8.0, noisePattern, mask);
	#else
		light.rgb	= CalculateGINew_LQ(texcoord, viewSpacePosition, normal, 16.0, noisePattern, mask);
	#endif
	
	#ifdef SEUS_GI
		light.rgb	= CalculateGI(texcoord, viewSpacePosition, normal, 16.0, GI_QUALITY, noisePattern, mask);
	#endif
#endif
	gl_FragData[0] = vec4(pow(light.rgb, vec3(1.0 / 2.2)), light.a);
}
