scriptname sslMatchMaker extends ReferenceAlias
; The version number for SexLabMatchMaker
int version = 5

; Load the SexLab framework for use
SexLabFramework property SexLab auto

; Actor selection storage
Actor[] Slots


; Trigger lock to make sure we don't accidentally call TriggerSex() more than once
; on the same set of actors after another's spell effect expires.
bool locked

; Our spells
Spell property MatchMakerSpellTarget auto
Spell property MatchMakerSpellSelf auto
Spell property MatchMakerSpellBuff auto

; NO LONGER USED.
; These properties are no longer used as of version 3
; Replaced by Actor[] Slots
ReferenceAlias[] property ActorList auto
ReferenceAlias property ActorSelection00 auto
ReferenceAlias property ActorSelection01 auto
ReferenceAlias property ActorSelection02 auto


; Initializes the MatchMaker script back to a state with no registered actors
function ResetSlots()
	; Empty out the Slots array by initializing it to an empty array.
	Actor[] emptyslots
	Slots = emptyslots
	; Unlock our TriggerSex() function for future use.
	locked = false
endFunction

; Called on first startup with mod enabled
event OnInit()
	; Give player the irrestible aura spells that trigger everything here.
	Game.GetPlayer().AddSpell(MatchMakerSpellTarget, true)
	Game.GetPlayer().AddSpell(MatchMakerSpellSelf, true)
	; Init slots and locked status
	ResetSlots()
endEvent

; Reset registered actors and lock whenever player reloads their game.
; Ensures we start out in a useable state on every load; just in case.
event OnPlayerLoadGame()
	ResetSlots()
endEvent

; Utility function to check if the actor is slotted already or not
; Returns true if they are in the Slots array already
bool function IsSlotted(actor Position)
	return Slots.Find(Position) != -1
endFunction

; Called from spell, validates the target and places actor in storage for scene use.
; Returns True if the actor is successfully added to storage, False if they fail to validate.
function RegisterActor(actor Position)

	; Validate we can add the actor before trying to register them.

	; IsSlotted() will check if they are already registered or not.
	if IsSlotted(Position)
		; Position is already slotted, fail and show some flavor text.
		Debug.Notification(Position.GetLeveledActorBase().GetName() + " is already irrestible!")
		return ; They didn't pass validation; don't continue with this function.

	; Check and make sure we aren't already over our limit of 3 actors
	; Also checks if they are a valid actor to animate in the eyes of SexLab
	; Slots.Length >= 3 makes sure our Slots array has room for them with a max of 3.
	; ValidateActor() will return a negative number if they are rejected.
	elseIf Slots.Length >= 3 || SexLab.ValidateActor(Position) < 1
		; Flavor text for the failure.
		Debug.Notification("Your spell fizzles and fails to affect " + Position.GetLeveledActorBase().GetName())
		return ; They didn't pass validation; don't continue with this function.

	endIf

	; Still here? The actor has passed our validation checks then.

	; Slot the actor into our registration array.
	Slots = sslUtility.PushActor(Position, Slots)
	; Flavor text for the successful cast
	Debug.Notification(Position.GetLeveledActorBase().GetName() + " has an irrestible aura about them")

	; If we have our limit of 3 actors now, lets not wait and just start the scene now.
	if Slots.Length >= 3
		TriggerSex()
	endIf

endFunction

; Takes our current Slots array and attempts to start a sex scene using the contained actors, however many actors it has, between 1 and 3.
function TriggerSex()

	; Ensure we aren't triggering a second time from another actors spell effect expiring while we are still processing this one
	; Also make sure we even have the actors to do anything first.
	if locked || Slots.Length == 0
		return ; TriggerSex() is already processing OR no actor slots filled. Stop now and don't trigger any sex. 
	endIf

	; Set our lock to prevent any calls to TriggerSex() on our current actor slots.
	locked = true

	; Claim an available animation thread from SexLab we can add our actors to.
	sslThreadModel Thread = SexLab.NewThread()

	; All animations so far in SexLab expect the female to the first actor given to it. 
	; So make sure we have the female actors in front of the array so animation positions are properly set
	; SortActors() from SexLab will return the given array with females moved to the front of the array by default.
	Slots = SexLab.SortActors(Slots)

	; Loop through registered actors to the thread from our now sorted Slots array.  
	int i
	while i < Slots.Length
		; Add looped actor to the thread. AddActor() will return -1 if they failed to be added to the thread.
		if Thread.AddActor(Slots[i]) == -1
			; The actor failed to be added to the thread for some reason, so lets not continue.
			; Send error message to the debug log.
			Debug.Trace("--- SexLab MatchMaker --- Failed to slot the actor '" + Slots[i].GetLeveledActorBase().GetName() + "' into the animation thread["+Thread.tid+"]!")
			; Unlock the animation thread so it clears itself and doesn't try to continue with any actors we may have successfully added.
			Thread.UnlockThread()
			; Clear our actors & unlock our TriggerSex() function for future use.
			ResetSlots()
			return ; Stop the function now since we failed to add the actor.
		endIf
		i += 1
	endWhile

	; Set our custom hook name "MatchMaker" in the thread.
	Thread.SetHook("MatchMaker")
	; Register our custom event "EndBuff" onto the AnimationEnd hook, which is triggered once
	; the actors have finished and begun to reset.
	; _MatchMaker makes it a non global hook that is called from our above Thread.SetHook("MatchMaker"),
	; so only this scene will call the EndBuff event and not other animations started outside of MatchMaker.
	RegisterForModEvent("AnimationEnd_MatchMaker", "EndBuff")

	; We'll keep the scene basic and let SexLab and player configuration figure out most of the stuff on it's own.
	; So let's start the animation now.
	Thread.StartThread()

	; We are done with our registered actors, so reset our lock and clear out our registered actors.
	ResetSlots()

endFunction


; Our buff hook, called from RegisterForModEvent("AnimationEnd_MatchMaker", "EndBuff")
; AnimationEnd is sent by SexLab called once the sex animation has fully stopped
event EndBuff(string eventName, string argString, float argNum, form sender)

	; Get our list of actors at the end
	Actor[] Positions = SexLab.HookActors(argString)

	; Loop through our list of actors
	int i = Positions.Length
	while i
		i -= 1
		; Add our buff spell to the looped actor
		Positions[i].AddSpell(MatchMakerSpellBuff, false)
	endWhile

endEvent