﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;

namespace FemaleTroops
{
    public class Config
    {
        private const string CONFIG_FILE_NAME = "femaletroops.ini";

        public bool HideHelmet { get; set; } = true;
        public bool DoHeroes { get; set; } = false;
        public bool DoCompanions { get; set; } = false;
        public bool DoInfantry { get; set; } = true;
        public bool DoSoldiers { get; set; } = true;
        public bool DoArchers { get; set; } = true;
        public bool DoCavalry { get; set; } = true;
        /// <summary>
        /// Are we turning everything to male or female?
        /// </summary>
        public bool ToMale { get; set; } = false;
        /// <summary>
        /// Which cultures should we convert? 'all' means all
        /// </summary>
        public string[] TargetCultures { get; set; } = new string[] { };
        /// <summary>
        /// List of string ids, such as 'imperial_recruit' to convert
        /// </summary>
        public string[] TargetUnits { get; set; } = new string[] { };

        public static Config LoadFromFile()
        {
            if(!File.Exists(CONFIG_FILE_NAME))
            {
                return CreateDefaultConfigFile();
            }
            var config = new Config();
            var lines = File.ReadAllLines(CONFIG_FILE_NAME);
            var kvp = lines.Select((l) => FromLine(l)).ToDictionary(l1 => l1.Name, l2 => l2.Value);

            config.HideHelmet = GetBoolean(kvp, nameof(HideHelmet), config.HideHelmet);
            config.DoHeroes = GetBoolean(kvp, nameof(DoHeroes), config.DoHeroes);
            config.DoCompanions = GetBoolean(kvp, nameof(DoCompanions), config.DoCompanions);
            config.DoInfantry = GetBoolean(kvp, nameof(DoInfantry), config.DoInfantry);
            config.DoSoldiers = GetBoolean(kvp, nameof(DoSoldiers), config.DoSoldiers);
            config.DoArchers = GetBoolean(kvp, nameof(DoArchers), config.DoArchers);
            config.DoCavalry = GetBoolean(kvp, nameof(DoCavalry), config.DoCavalry);
            config.ToMale = GetBoolean(kvp, nameof(ToMale), config.ToMale);

            config.TargetCultures = GetStringArray(kvp, nameof(TargetCultures), config.TargetCultures);
            config.TargetUnits = GetStringArray(kvp, nameof(TargetUnits), config.TargetUnits);

            return config;
        }

        private static bool GetBoolean(Dictionary<string, string> kvp, string name, bool defaultValue)
        {
            if(kvp.TryGetValue(name, out var value))
            {
                return value == "true";
            }
            return defaultValue;
        }

        private static string[] GetStringArray(Dictionary<string, string> kvp, string name, string[] defaultValue)
        {
            if (kvp.TryGetValue(name, out var values) && values.Length > 0)
            {
                return values.ToLower().Split(';').Select(s => s.Trim()).ToArray();
            }
            return defaultValue;
        }

        private static Config CreateDefaultConfigFile()
        {
            File.Delete(CONFIG_FILE_NAME);
            using (var stream = File.CreateText(CONFIG_FILE_NAME))
            {
                var config = new Config();
                stream.WriteLine(ToLine(nameof(DoHeroes), config.DoHeroes, "false = Will not change hero-type units. TargetUnits takes precedence over this setting if applicable."));
                // stream.WriteLine(ToLine(nameof(DoCompanions), config.DoCompanions, "false = Will not change companions. TargetUnits takes precedence over this setting if applicable."));
                stream.WriteLine(ToLine(nameof(DoInfantry), config.DoInfantry, "false = Will not change infantry-type units, such as looters. TargetUnits takes precedence over this setting if applicable."));
                stream.WriteLine(ToLine(nameof(DoSoldiers), config.DoSoldiers, "false = Will not change soldier-type units. TargetUnits takes precedence over this setting if applicable."));
                stream.WriteLine(ToLine(nameof(DoArchers), config.DoArchers, "false = Will not change archer-type units. TargetUnits takes precedence over this setting if applicable."));
                stream.WriteLine(ToLine(nameof(DoCavalry), config.DoCavalry, "false = Will not change cavalry-type units. TargetUnits takes precedence over this setting if applicable."));
                stream.WriteLine(ToLine(nameof(HideHelmet), config.HideHelmet, "true = Will hide the helmet of units affected by the changes. false = wont."));
                stream.WriteLine(ToLine(nameof(TargetCultures), config.TargetCultures, "List of culture IDs you want the changes to apply to. Leave blank for all, otherwise list them seperated by a semi-column, i.e battanian;khuzait"));
                stream.WriteLine(ToLine(nameof(TargetUnits), config.TargetUnits, "List of unit IDs you want the changes to apply to. Leave blank for all, otherwise list them seperated by a semi-column, i.e imperial_recruit;imperial_archer"));
                stream.WriteLine(ToLine(nameof(ToMale), config.ToMale, "false = Will change the units to female. true = Will change the units to male."));
                return config;
            }
        }

        private static string ToLine(string name, string[] values, string comment = null)
        {
            var valuesStr = values == null ? "" : string.Join(";", values);
            return ToLine(name, valuesStr, comment);
        }
        private static string ToLine(string name, bool value, string comment = null)
        {
            return ToLine(name, value ? "true" : "false", comment);
        }
        private static string ToLine(string name, string value, string comment = null)
        {
            return comment == null ? $"{name}={value}" : $"{name}={value} # {comment}";
        }

        private static (string Name, string Value) FromLine(string line)
        {
            if (line.Contains("#")) // remove comment
            {
                line = line.Split('#')[0].Trim();
            }

            var parts = line.Split('=');
            if(parts.Length != 2)
            {
                throw new Exception($"Unexpected line in config: " + line);
            }
            var name = parts[0].Trim();
            var value = parts[1];
            return (name, value);
        }
    }
}
