/*

    Responsible for overwriting CSS at runtime according to CC app
    settings as defined by the end user.

*/

var ThemeManager = (function () {
    'use strict';

    /**
     * Convert the Color object to string in hexadecimal format;
     */
    function toHex(color, delta) {

        function computeValue(value, delta) {
            var computedValue = !isNaN(delta) ? value + delta : value;
            if (computedValue < 0) {
                computedValue = 0;
            } else if (computedValue > 255) {
                computedValue = 255;
            }

            computedValue = Math.floor(computedValue);

            computedValue = computedValue.toString(16);
            return computedValue.length === 1 ? "0" + computedValue : computedValue;
        }

        var hex = "";
        if (color) {
            hex = computeValue(color.red, delta) + computeValue(color.green, delta) + computeValue(color.blue, delta);
        }
        return hex;
    }


    function reverseColor(color, delta) {
        return toHex({
            red: Math.abs(255 - color.red),
            green: Math.abs(255 - color.green),
            blue: Math.abs(255 - color.blue)
        },
            delta);
    }


    function addRule(stylesheetId, selector, rule) {
        var stylesheet = document.getElementById(stylesheetId);

        if (stylesheet) {
            stylesheet = stylesheet.sheet;
            if (stylesheet.addRule) {
                stylesheet.addRule(selector, rule);
            } else if (stylesheet.insertRule) {
                stylesheet.insertRule(selector + ' { ' + rule + ' }', stylesheet.cssRules.length);
            }
        }
    }



    /**
     * Update the theme with the AppSkinInfo retrieved from the host product.
     */
    function updateThemeWithAppSkinInfo(appSkinInfo) {

        var panelBgColor = appSkinInfo.panelBackgroundColor.color;
        var bgdColor = toHex(panelBgColor);

        var darkBgdColor =  toHex(panelBgColor, 20);
        var darkerBgdColor =  toHex(panelBgColor, -10);
        var darkBrdColor =  toHex(panelBgColor, -20);

        var fontColor = "F0F0F0";
        var lightFontColor = "DDDDDD";
        if (panelBgColor.red > 122) {
            fontColor = "000000";
            lightFontColor = "333333";
        }
        var lightBgdColor = toHex(panelBgColor, -100);

        var styleId = "hostStyle";

        addRule(styleId, ".body", "background-color:" + "#" + bgdColor);
        addRule(styleId, ".body", "font-size:" + appSkinInfo.baseFontSize + "px;");
        addRule(styleId, ".body", "font-family:" + appSkinInfo.baseFontFamily);
        addRule(styleId, ".body", "color:" + "#" + fontColor);

        addRule(styleId, ".ps-input", "background-color:" + "#" + darkerBgdColor);
        addRule(styleId, ".ps-input", "border-color:" + "#" + darkBrdColor);
        addRule(styleId, ".ps-input:focus", "background-color:" + "#" + darkBgdColor);

        addRule(styleId, ".ps-button", "background-color:" + "#" + darkBgdColor);
        addRule(styleId, ".ps-button", "border-color:" + "#" + darkBrdColor);
        addRule(styleId, ".ps-button--active", "background-color:" + "#" + bgdColor);
        addRule(styleId, ".ps-button:hover", "background-color:" + "#" + bgdColor);
        addRule(styleId, ".ps-button:active", "background-color:" + "#" + darkBgdColor);

        addRule(styleId, ".ps-darker-bg", "background-color:" + "#" + darkerBgdColor);
        addRule(styleId, ".ps-light-font", "color:" + "#" + lightFontColor);
    }


    function onAppThemeColorChanged(event) {
        var skinInfo = JSON.parse(window.__adobe_cep__.getHostEnvironment()).appSkinInfo;
        updateThemeWithAppSkinInfo(skinInfo);
    }


    function init(csInterface) {
        updateThemeWithAppSkinInfo(csInterface.hostEnvironment.appSkinInfo);
        csInterface.addEventListener(CSInterface.THEME_COLOR_CHANGED_EVENT, onAppThemeColorChanged);
    }

    return {
        init: init
    };

}());
