package.path = package.path .. ";data/scripts/lib/?.lua"
package.path = package.path .. ";data/scripts/?.lua"
package.path = package.path .. ";?"


include("randomext")
local Xsotan = include("story/xsotan")
local SpawnUtility = include("spawnutility")
local ShipUtility = include("shiputility")
local SectorGenerator = include ("SectorGenerator")
local SectorTurretGenerator = include ("sectorturretgenerator")


if onServer() then

local data = {}
data.countXsotanDestroyed = 0
data.maxBackgroundXsotan = 15
data.level2Spawned = false
data.level3Spawned = false
data.level4Spawned = false
data.level5Spawned = false
data.despawn = false
data.active = false
data.precursorFightTimer = 10 * 60

function getUpdateInterval()
    if data.despawn then
        return 0.5
    else
        return 6
    end
end

function initialize()
    data.precursorFightTimer = 10 * 60
    Sector():registerCallback("onDestroyed", "onDestroyed")
    Sector():registerCallback("onPlayerEntered", "onPlayerEntered")
    Sector():registerCallback("onPlayerLeft", "onPlayerLeft")
end

function updateServer(timeStep)
    local players = {Sector():getPlayers()}
    if players and #players > 0 then
        local server = Server()
        if not data.active then
            data.active = server:getValue("xsotan_swarm_active")
            if data.active then
                for i = 1, 7 do
                    spawnBackgroundXsotan() -- spawn even if player is in sector already
                end
            end
        end

        if not data.despawn and data.active then
            -- event is running
            spawnBackgroundXsotan()

            if not data.level2Spawned and miniBossSlain() and data.countXsotanDestroyed >= 10 then
                spawnLevel2()
            elseif not data.level3Spawned and miniBossSlain() and data.countXsotanDestroyed >= 25 then
                spawnLevel3()
            elseif not data.level4Spawned and miniBossSlain() and data.countXsotanDestroyed >= 45 then
                spawnLevel4()
            elseif not data.level5Spawned and miniBossSlain() and data.countXsotanDestroyed >= 50 then
                spawnLevel5()
            elseif data.level5Spawned and miniBossSlain() then
                finishUp(true) -- if it was in time player wins
            end

            -- check if precursor fight is ongoing
            if data.level5Spawned then
                data.precursorFightTimer = data.precursorFightTimer - timeStep
                if data.precursorFightTimer <= 0 then
                    server:setValue("xsotan_swarm_precursor_fight", false)
                else
                    server:setValue("xsotan_swarm_precursor_fight", true)
                end
            end


            -- check if we ran out of time
            if data.active and not server:getValue("xsotan_swarm_duration") then
                finishUp(false)
            end

        elseif data.despawn then
            -- despawn ships
            local ships = getAllSpawnedShips()
            if #ships > 0 then
                Sector():deleteEntityJumped(ships[1])
            else
                resetValues()
            end
        end
    end
end

function onPlayerEntered(playerIndex, sectorChangeType)

    local players = {Sector():getPlayers()}
    if players and #players > 1 then return end

    -- spawn first few immediately, but only if no other player is in sector
    if data.active then
        for i = 1, 7 do
            spawnBackgroundXsotan()
        end
    end
end

function onPlayerLeft(playerIndex, sectorChangeType)
    local sector = Sector()
    local players = {sector:getPlayers()}
    if #players > 0 then return end -- clean up only if the last player left

    -- clean up swarm ships
    local entities = getAllSpawnedShips()
    for _, ship in pairs(entities) do
        if ship:getValue("xsotan_destruction_limit") or ship:getValue("xsotan_swarm_boss") then
            Sector():deleteEntity(ship)
        end
    end

    -- reset values
    resetValues()
end

function resetValues()
    data.countXsotanDestroyed = 0
    data.maxBackgroundXsotan = 15
    data.level2Spawned = false
    data.level3Spawned = false
    data.level4Spawned = false
    data.level5Spawned = false
    data.despawn = false
    data.precursorFightTimer = 10 * 60
end

function spawnBackgroundXsotan()
    if countAliveXsotan() > 15 then return end

    local generator = SectorGenerator(Sector():getCoordinates())

    local xsotan = Xsotan.createShip(generator:getPositionInSector(), 1.0)
    if not valid(xsotan) then return end

    xsotan:setValue("xsotan_spawn_limit", 1)
    xsotan:setValue("xsotan_destruction_limit", 1)
    for _, p in pairs({Sector():getPlayers()}) do
        ShipAI(xsotan.id):registerEnemyFaction(p.index)
    end
    ShipAI(xsotan.id):setAggressive()
end

function spawnHenchmenXsotan(num)

    local generator = SectorGenerator(Sector():getCoordinates())

    local ships = {}
    local i
    for i = 0, num do
        local xsotan = Xsotan.createShip(generator:getPositionInSector(), 1.0)
        if not valid(xsotan) then return end

        table.insert(ships, xsotan)
        xsotan:setValue("xsotan_destruction_limit", 1)
        for _, p in pairs({Sector():getPlayers()}) do
            ShipAI(xsotan.id):registerEnemyFaction(p.index)
        end
        ShipAI(xsotan.id):setAggressive()
    end

    SpawnUtility.addEnemyBuffs(ships)
end

function spawnLevel2()
    data.level2Spawned = true

    local generator = SectorGenerator(Sector():getCoordinates())
    local xsotan = Xsotan.createShip(generator:getPositionInSector(), 10.0)
    if not valid(xsotan) then return end

    xsotan:setValue("xsotan_swarm_boss", 1)
    xsotan.title = "Xsotan Emissary"%_T
    for _, p in pairs({Sector():getPlayers()}) do
        ShipAI(xsotan.id):registerEnemyFaction(p.index)
    end
    ShipAI(xsotan.id):setAggressive()

    -- spawn henchmen
    spawnHenchmenXsotan(3)
end

function spawnLevel3()
    data.level3Spawned = true

    local generator = SectorGenerator(Sector():getCoordinates())
    local xsotan = Xsotan.createQuantum(generator:getPositionInSector(), 15.0)
    if not valid(xsotan) then return end

    Loot(xsotan.index):insert(generateUpgrade())
    Loot(xsotan.index):insert(generateUpgrade())
    WreckageCreator(xsotan.index).active = false

    xsotan:setValue("xsotan_swarm_boss", 1)
    for _, p in pairs({Sector():getPlayers()}) do
        ShipAI(xsotan.id):registerEnemyFaction(p.index)
    end
    ShipAI(xsotan.id):setAggressive()

    -- spawn henchmen
    spawnHenchmenXsotan(3)
end

function spawnLevel4()
    data.level4Spawned = true

    local generator = SectorGenerator(Sector():getCoordinates())
    local xsotan = Xsotan.createSummoner(generator:getPositionInSector(), 15.0)
    if not valid(xsotan) then return end
    WreckageCreator(xsotan.index).active = false

    local x, y = Sector():getCoordinates()
    Loot(xsotan.index):insert(InventoryTurret(SectorTurretGenerator():generate(x, y, 0, Rarity(RarityType.Exotic))))
    Loot(xsotan.index):insert(InventoryTurret(SectorTurretGenerator():generate(x, y, 0, Rarity(RarityType.Exotic))))

    xsotan:setValue("xsotan_swarm_boss", 1)
    for _, p in pairs({Sector():getPlayers()}) do
        ShipAI(xsotan.id):registerEnemyFaction(p.index)
    end
    ShipAI(xsotan.id):setAggressive()

    -- spawn henchmen
    spawnHenchmenXsotan(3)
end

function spawnLevel5()
    data.level5Spawned = true
    Server():setValue("xsotan_swarm_precursor_fight", true)

    -- spawn guardian precursor
    local generator = SectorGenerator(Sector():getCoordinates())
    local precursor = spawnPrecursor(generator:getPositionInSector(), 0.8)
    if not valid(precursor) then return end
    Sector():addScriptOnce("story/guardianprecursorbar.lua")

    -- set enemy factions
    for _, p in pairs({Sector():getPlayers()}) do
        ShipAI(precursor.id):registerEnemyFaction(p.index)
    end
    ShipAI(precursor.id):setAggressive()

    -- add loot
    Loot(precursor.index):insert(generateUpgrade())
    Loot(precursor.index):insert(generateUpgrade())
    Loot(precursor.index):insert(generateUpgrade())
    Loot(precursor.index):insert(generateUpgrade())

    -- extend global event time limit
    local server = Server()
    local timer = server:getValue("xsotan_swarm_duration")
    if timer == (30 * 60) then
        server:setValue("xsotan_swarm_duration", timer + (10 * 60))
    end

end

function spawnPrecursor(position, scale)
    position = position or Matrix()
    local volume = Balancing_GetSectorShipVolume(Sector():getCoordinates())

    volume = volume * (scale or 10)

    local x, y = Sector():getCoordinates()
    local probabilities = Balancing_GetTechnologyMaterialProbability(x, y)
    local material = Material(MaterialType.Avorion)
    local faction = Xsotan.getFaction()

    local plan = PlanGenerator.makeShipPlan(faction, volume, nil, material)
    local front = PlanGenerator.makeShipPlan(faction, volume, nil, material)
    local back = PlanGenerator.makeShipPlan(faction, volume, nil, material)
    local top = PlanGenerator.makeShipPlan(faction, volume, nil, material)
    local bottom = PlanGenerator.makeShipPlan(faction, volume, nil, material)
    local left = PlanGenerator.makeShipPlan(faction, volume, nil, material)
    local right = PlanGenerator.makeShipPlan(faction, volume, nil, material)
    local frontleft= PlanGenerator.makeShipPlan(faction, volume, nil, material)
    local frontright = PlanGenerator.makeShipPlan(faction, volume, nil, material)

    --
    attachMin(plan, back, "z")
    attachMax(plan, front, "z")
    attachMax(plan, front, "z")

    attachMin(plan, bottom, "y")
    attachMax(plan, top, "y")

    attachMin(plan, left, "x")
    attachMax(plan, right, "x")

    local self = findMaxBlock(plan, "z")
    local other = findMinBlock(frontleft, "x")
    plan:addPlanDisplaced(self.index, frontleft, other.index, self.box.center - other.box.center)

    local other = findMaxBlock(frontright, "x")
    plan:addPlanDisplaced(self.index, frontright, other.index, self.box.center - other.box.center)

    Xsotan.infectPlan(plan)
    local boss = Sector():createShip(faction, "", plan, position)

    -- Xsotan have random turrets
    local numTurrets = math.max(1, Balancing_GetEnemySectorTurrets(x, y) / 2)

    ShipUtility.addTurretsToCraft(boss, Xsotan.createPlasmaTurret(), numTurrets, numTurrets)
    ShipUtility.addTurretsToCraft(boss, Xsotan.createLaserTurret(), numTurrets, numTurrets)
    ShipUtility.addTurretsToCraft(boss, Xsotan.createRailgunTurret(), numTurrets, numTurrets)
    ShipUtility.addBossAntiTorpedoEquipment(boss)

    boss.title = "Wormhole Guardian Prototype"%_t
    boss.crew = boss.minCrew
    boss.shieldDurability = boss.shieldMaxDurability

    AddDefaultShipScripts(boss)

    ShipAI(boss.id):setAggressive()
    boss:addScriptOnce("story/xsotanbehaviour.lua")
    boss:setValue("is_xsotan", 1)
    boss:setValue("xsotan_swarm_boss", 1)
    WreckageCreator(boss.index).active = false

    Boarding(boss).boardable = false

    return boss
end

function attachMax(plan, attachment, dimStr)
    local self = findMaxBlock(plan, dimStr)
    local other = findMinBlock(attachment, dimStr)

    plan:addPlanDisplaced(self.index, attachment, other.index, self.box.center - other.box.center)
end

function attachMin(plan, attachment, dimStr)
    local self = findMinBlock(plan, dimStr)
    local other = findMaxBlock(attachment, dimStr)

    plan:addPlanDisplaced(self.index, attachment, other.index, self.box.center - other.box.center)
end

function finishUp(success)
    -- all Xsotan are despawned one by one
    data.despawn = true
    data.active = false

    -- set global success
    local server = Server()
    if not server:getValue("xsotan_swarm_success") and success then
        server:setValue("xsotan_swarm_success", true)
    end

end

function getAllSpawnedShips()
    local entities = {Sector():getEntitiesByType(EntityType.Ship)}
    local spawnedShips = {}
    for _, ship in pairs(entities) do
        if ship:getValue("xsotan_destruction_limit") or ship:getValue("xsotan_swarm_boss") then
            table.insert(spawnedShips, ship)
        end
    end
    return spawnedShips
end

function generateUpgrade()
    local upgrades = {
        "data/scripts/systems/arbitrarytcs.lua",
        "data/scripts/systems/militarytcs.lua",
        "data/scripts/systems/defensesystem.lua",
        "data/scripts/systems/radarbooster.lua",
        "data/scripts/systems/scannerbooster.lua",
        "data/scripts/systems/weaknesssystem.lua",
        "data/scripts/systems/resistancesystem.lua",
    }
    local randUpgrades = random():getInt(1, #upgrades)
    local upgradeName = upgrades[randUpgrades]

    local rarities = {2, 2, 2, 3, 3, 3, 3, 4}
    local randRarities = random():getInt(1, #rarities)
    local rarity = rarities[randRarities]

    return SystemUpgradeTemplate(upgradeName, Rarity(rarity), random():createSeed())
end

function countAliveXsotan()
    local count = 0
    local entities = {Sector():getEntitiesByType(EntityType.Ship)}
    for _, ship in pairs(entities) do
        if ship:getValue("xsotan_spawn_limit") then
            count = count + 1
        end
    end
    return count
end

function miniBossSlain()
    local entities = {Sector():getEntitiesByType(EntityType.Ship)}
    for _, ship in pairs(entities) do
        if ship:getValue("xsotan_swarm_boss") then
            return false
        end
    end
    return true
end

function onDestroyed(index)
    local entity = Entity(index)
    if valid(entity) and entity:getValue("xsotan_destruction_limit") then
        data.countXsotanDestroyed = data.countXsotanDestroyed + 1
    end
end

function secure()
    return data
end

function restore(data_in)
    data = data_in
end

end
