package.path = package.path .. ";data/scripts/lib/?.lua"
package.path = package.path .. ";data/scripts/?.lua"
package.path = package.path .. ";data/scripts/entity/?.lua"

include("utility")
include("stringutility")
include("callable")
include("structuredmission")

include("defaultscripts")
MissionUT = include("missionutility")
local Adventurer = include("story/missionadventurer")
local AdventurerGuide = include("story/adventurerguide")
local ShipGenerator = include("shipgenerator")

-- mission.tracing = true

-- data
mission.data.title = "Station Founding"%_T
mission.data.brief = "Station Founding"%_T
mission.data.icon = "data/textures/icons/graduate-cap.png"
mission.data.priority = 5
mission.data.description = {}
mission.data.location = {}

-- custom data
mission.data.custom.adventurerId = nil
mission.data.custom.shipId = nil
mission.data.custom.createdStationfounder = false
mission.data.custom.createdStation = false
mission.data.custom.thirdDialogStarted = false
mission.data.custom.tmpCoords = {}
mission.data.custom.secondMailRead = false
mission.data.custom.lastMailRead = false


-- phases
mission.globalPhase = {}
mission.globalPhase.onAbandon = function()
    local entity = Sector():getEntitiesByScript("data/scripts/entity/story/missionadventurer.lua")
    if entity then Sector():deleteEntityJumped(entity) end
end

-- Phase 1: Player comes into sector => spawn Adventurer and set descriptions
mission.phases[1] = {}
mission.phases[1].onBegin = function()
    if onServer() then
        mission.data.description[1] = {text = "Your friend, ${name} the Adventurer, wants to teach you how to found your own station!"%_T, arguments = {name = getAdventurerName()}}
        mission.data.location.x, mission.data.location.y = Sector():getCoordinates()
        createAdventurer()
        mission.data.custom.adventurerName = getAdventurerName()
    end
    nextPhase()
end
mission.phases[1].noBossEncountersTargetSector = true
mission.phases[1].noPlayerEventsTargetSector = true

-- Phase 2: Wait until Adventurer is finished spawning - start first dialog
mission.phases[2] = {}
mission.phases[2].triggers = {}
mission.phases[2].triggers[1] =
    {
        condition = function() return checkAdventurerCreated() end,
        callback = function () return startFirstDialog() end
    }

-- Phase 3: Check if player built a ship with station founder
mission.phases[3] = {}
mission.phases[3].onBeginServer = function()
   Player():registerCallback("onShipCreationFinished", "onShipCreationFinished")
end
mission.phases[3].onRestore = function()
    if onServer() then
        Player():registerCallback("onShipCreationFinished", "onShipCreationFinished")
    end
end

-- Phase 4: Recheck that Adventurer is still here and continue with a dialog (or a mail if player went elsewhere)
mission.phases[4] = {}
mission.phases[4].onBegin = function()
    mission.data.description[2].fulfilled = true
    mission.data.description[3] = {text = "Turn a station founder ship into a station. Careful: You'll need at least 4 Million Credits!"%_T, bulletPoint = true, fulfilled = false}
    if onClient() then
        local adventurerHere = checkAdventurerCreated()
        if adventurerHere then
            startSecondDialog()
        else
            sendSecondDialogMail()
        end
    else
        Player():registerCallback("onTransformedToStation", "onTransformedToStation")
    end
end
mission.phases[4].onRestore = function()
    if onServer() then
        Player():registerCallback("onTransformedToStation", "onTransformedToStation")
    end
end


-- Phase 5: Recheck that Adventurer is still here and finish up with a dialog (or a mail if player went elsewhere)
mission.phases[5] = {}
mission.phases[5].onBegin = function()
    if onClient() then
        local adventurerHere = checkAdventurerCreated()
        if adventurerHere then
            startLastDialog()
        else
            sendLastDialogMail()
            finishUp()
        end
    end
end
mission.phases[5].timers = {}
mission.phases[5].timers[1] = {callback = function() finishUp() end}
mission.phases[5].showUpdateOnBegin = true


-- helper functions
function getAdventurerName()
    local player = Player()
    local faction = Galaxy():getNearestFaction(player:getHomeSectorCoordinates())
    local language = faction:getLanguage()
    language.seed = Server().seed
    return language:getName()
end

function createAdventurer()
    if onClient() then invokeServerFunction("createAdventurer") return end

    local adventShip = AdventurerGuide.spawnMissionAdventurer(Player())
    if not adventShip then
        adventShip = Sector():getEntitiesByScript("data/scripts/entity/story/missionadventurer.lua")
    end
    adventShip.invincible = true
    MissionUT.deleteOnPlayersLeft(adventShip)
    mission.data.custom.adventurerId = adventShip.id.string
    adventShip:invokeFunction("story/missionadventurer.lua", "setInteractingScript", "player/missions/tutorials/foundstationtutorial.lua")
    sync()
end
callable(nil, "createAdventurer")

function checkAdventurerCreated()
    if onServer() then invokeClientFunction(Player(), "checkAdventurerCreated") return false end

    if not mission.data.custom.adventurerId then return false end
    return Entity(mission.data.custom.adventurerId) ~= nil
end

function onShipCreationFinished(senderInfo, shipIndex)
    local ship = Entity(shipIndex)
    if ship.factionIndex == Player().index then
        if ship:hasScript("data/scripts/entity/stationfounder.lua") then
            mission.data.custom.createdStationfounder = true
        end
    end
    if mission.data.custom.createdStationfounder == true then
        Player():unregisterCallback("onShipCreationFinished", "onShipCreationFinished")
        goToNextPhase()
    end
end

function onTransformedToStation(senderInfo, stationIndex)
    local station = Entity(stationIndex)
    local stationWasCreated = 0
    if station.factionIndex == Player().index then
        mission.data.custom.createdStation = true
        stationWasCreated = 1
    end
    if mission.data.custom.createdStation == true and stationWasCreated == 1 then
        goToNextPhase()
    end
end

function startFirstDialog()

    if onServer() then
        invokeClientFunction(Player(), "startFirstDialog")
    return end

    local dialog = {}
    local dialog2 = {}
    local dialog3 = {}
    local dialogPost = {}
    local dialogAbandon = {}

    dialog.text = "Hello!\n\nYou seem like someone who would enjoy owning a station.\n\nDo you want me to show you the quickest way to get one? The quickest legal way, that is."%_t
    dialog.answers =
    {
        {answer = "Sure."%_t, followUp = dialog2},
        {answer = "Not right now."%_t, followUp = dialogPost},
        {answer = "No thanks, I already know how to do it."%_t, followUp = dialogAbandon}
    }

    dialog2.text = "See that shipyard over there? You can have ships built at any shipyard.\n\nIf you want to found a station, you need to make sure to check the box 'Station Founder' when you give your order to the crew at the shipyard."%_t
    dialog2.answers = {{answer = "Thanks, I'll check it out."%_t, followUp = dialog3}}

    dialog3.text = "Make sure to bring enough money! You'll need at least 4 Million to found one.\n\nIt doesn't matter if you don't have the money right now, though. You can always do it later and use this ship as a normal ship for now!"%_t
    dialog3.onEnd = "onEndFirstDialog"

    dialogPost.text = "Oh, okay. Until later then."%_t
    dialogPost.onEnd = "onDialogPost"

    dialogAbandon.text = "Oh, okay."%_t
    dialogAbandon.onEnd = "onDialogAbandon"

    local entity = Entity(mission.data.custom.adventurerId)
    entity:invokeFunction("story/missionadventurer.lua", "setData", true, true, dialog)
end

function onDialogPost()
    if onClient() then invokeServerFunction("onDialogPost") return end

    -- let adventurer jump away
    local adventurer = Entity(mission.data.custom.adventurerId)
    if adventurer then
        Sector():deleteEntityJumped(adventurer)
    end

    terminate() -- mission silently dies
end
callable(nil, "onDialogPost")

function onDialogAbandon()
    if onClient() then invokeServerFunction("onDialogAbandon") return end

    -- let adventurer jump away
    local adventurer = Entity(mission.data.custom.adventurerId)
    if adventurer then
        Sector():deleteEntityJumped(adventurer)
    end

    Player():setValue("tutorial_foundstation_accomplished", true)
    showMissionAbandoned()
    terminate() -- mission silently dies
end
callable(nil, "onDialogAbandon")

function onEndFirstDialog()
    if onClient() then
        invokeServerFunction("onEndFirstDialog")
        return
    end
    mission.data.description[2] = {text = "Select 'Station Founder' when having a ship built at a shipyard."%_T, bulletPoint = true, fulfilled = false}
    if mission.currentPhase == mission.phases[2] then
        goToNextPhase()
    end
end
callable(nil, "onEndFirstDialog")

function startSecondDialog()
    if onServer() then invokeClientFunction(Player(), "startSecondDialog") return end

    local dialog = {}
    dialog.text = "Well done. Now you have a ship that you can use to found a station. Just enter it, fly it to where you want the station to be, and found a station the same way you would normally found a ship."%_t
    dialog.answers = {{answer = "Thank you."%_t}}

    local entity = Sector():getEntitiesByScript("story/missionadventurer.lua")
    entity:invokeFunction("story/missionadventurer.lua", "setData", true, false, dialog)
end

function sendSecondDialogMail()
    if onClient() then invokeServerFunction("sendSecondDialogMail") return end

    local mail = Mail()
    mail.text = Format("Well done. Now you have a ship you can use to found a station. Just enter it, fly it to where you want the station to be, and found a station the same way you would normally found a ship.\n\nGreetings,\n%s"%_T, getAdventurerName())
    mail.header = "A step in the right direction /* Mail Subject */"%_T
    mail.sender = Format("%s, the Adventurer"%_T, getAdventurerName())
    mail.id = "Tutorial_FoundStation1"

    Player():addMail(mail)
end
callable(nil, "sendSecondDialogMail")

function startLastDialog()
    if onServer() then invokeClientFunction(Player(), "startLastDialog") return end

    local dialog = {}
    dialog.text = "Good job, you founded a station! It's still a little small. You should expand it and hire and assign more crew. The next time I come to this sector, I'm sure I'll find a magnificent station!"%_t
    dialog.answers = {{answer = "Thank you."%_t}}
    dialog.onEnd = "finishUp"

    local entity = Sector():getEntitiesByScript("story/missionadventurer.lua")
    entity:invokeFunction("story/missionadventurer.lua", "setData", true, false, dialog)
end

function sendLastDialogMail()
    if onClient() then invokeServerFunction("sendLastDialogMail") return end

    local mail = Mail()
    mail.text = Format("Good job, you founded a station! It's still a little small. You should expand it and hire and assign more crew.\n\nThe next time I come to this sector, I'm sure I'll find a magnificent station!\n\nGreetings,\n%s"%_T, getAdventurerName())
    mail.header = "Congratulations on your station! /* Mail Subject */"%_T
    mail.sender = Format("%s, the Adventurer"%_T, getAdventurerName())
    mail.id = "Tutorial_FoundStation2"

    Player():addMail(mail)
end
callable(nil, "sendLastDialogMail")

function onShipCreationFinished(senderInfo, shipIndex)
    local ship = Entity(shipIndex)
    if ship and ship.factionIndex == Player().index then
        if ship:hasScript("data/scripts/entity/stationfounder.lua") then
           mission.data.custom.createdStationfounder = true
        end
    end

    if mission.data.custom.createdStationfounder == true and mission.currentPhase == mission.phases[3] then
        goToNextPhase()
    end
end

function finishUp()
    if onClient() then invokeServerFunction("finishUp") return end

    local entity = Entity(mission.data.custom.adventurerId)
    if entity then
        Sector():deleteEntityJumped(entity)
    end
    Player():setValue("tutorial_foundstation_accomplished", true)
    accomplish()
end
callable(nil, "finishUp")

function goToNextPhase()
    if onClient() then invokeServerFunction("goToNextPhase") return
    else
        nextPhase()
    end
end
callable(nil, "goToNextPhase")

