package.path = package.path .. ";data/scripts/lib/?.lua"
package.path = package.path .. ";data/scripts/?.lua"

include ("utility")
include ("stringutility")
include ("callable")
include ("galaxy")
include ("faction")
include("randomext")
include ("structuredmission")
MissionUT = include ("missionutility")

local AdventurerGuide = include ("story/adventurerguide")

-- mission data
mission.data.title = "Overcome the Barrier"%_T
mission.data.brief = "Overcome the Barrier"%_T
mission.data.icon = "data/textures/icons/story-mission.png"
mission.data.priority = 10
mission.data.description = {}
mission.data.description[1] = "Find a way to cross the Barrier."%_T
mission.data.description[2] = {text = "Travel to the Barrier"%_T, bulletPoint = true, fulfilled = false}

-- custom
mission.data.custom.firstRoundThrough = true
mission.data.custom.foundNoArtifacts = false

-- mission phases
mission.globalPhase = {}
mission.globalPhase.distanceChecks = {}
mission.globalPhase.distanceChecks[1] = -- used for adventurer spawns, to have him fly to the player but stop a short distance away
    {
        distance = 1000,
        onLower = function() onAdventurerCloseEnough() end
    }

mission.phases[1] = {}
mission.phases[1].onBeginServer = function()
    mission.data.description[1] = "After the cataclysmic catastrophe 200 years back, a ring of torn hyperspace fabric appeared around the center of the galaxy. Rumors say that there are huge asteroid fields with a new element called Avorion inside of it."%_T

    -- check to which phase we should switch
    local inventorySystems = checkInventory()
    local shipSystems = checkShipUpgrades()
    local count = 0
    for i = 1, 8, 1 do
        if (inventorySystems and inventorySystems[i]) or (shipSystems and shipSystems[i]) then
            count = count + 1
        end
    end

    if count == 0 then mission.data.custom.foundNoArtifacts = true end
    if count == 8 then setPhase(3) return end
    if count < 8 then nextPhase() return end
end

mission.phases[2] =  {}
mission.phases[2].onBeginServer = function()
    mission.data.description[4] = nil
    if mission.data.custom.firstRoundThrough then -- only talk to player once, he then knows he has to find the artifacts
        mission.data.custom.firstRoundThrough = false

        mission.data.description[2].fulfilled = true
        mission.data.description[3] = {text = "Find the Xsotan artifacts"%_T, bulletPoint = true, fulfilled = false}

        createAdventurer()
        ShipAI(mission.data.custom.adventurerId):setFollow(Player().craft) -- have adventurer move closer to the player
        mission.globalPhase.distanceChecks[1].id = mission.data.custom.adventurerId
        mission.phases[2].timers[1].time = 5
    end
end
mission.phases[2].onSectorEntered = function(x,y)
    if checkDistance(x, y) then
        local inventorySystems = checkInventory()
        local shipSystems = checkShipUpgrades()
        local count = 0
        for i = 1, 8, 1 do
            if (inventorySystems and inventorySystems[i]) or (shipSystems and shipSystems[i]) then
                count = count + 1
            end
        end

        if count == 8 then setPhase(3) return end
        if count < 8 then setPhase(2) return end
    end
end
mission.phases[2].timers = {}
mission.phases[2].timers[1] = {callback = function() onGoFindDialog() end}

mission.phases[3] = {}
mission.phases[3].onBeginServer = function()
    mission.data.description[2].fulfilled = true
    mission.data.description[3] = {text = "Find the eight Xsotan artifacts"%_T, bulletPoint = true, fulfilled = true}
    mission.data.description[4] = {text = "Find a way over the Barrier"%_T, bulletPoint = true, fulfilled = false}

    createAdventurer()
    ShipAI(mission.data.custom.adventurerId):setFollow(Player().craft) -- have adventurer move closer to the player
    mission.globalPhase.distanceChecks[1].id = mission.data.custom.adventurerId

    mission.phases[3].timers[1].time = 4
end
mission.phases[3].timers = {}
mission.phases[3].timers[1] = {callback = function() onFoundArtifactsDialog() end}

mission.phases[4] = {}
mission.phases[4].onBeginServer = function()
    mission.globalPhase.distanceChecks[1].id = mission.data.custom.adventurerId
end
mission.phases[4].onSectorEntered = function(x,y)
    if checkCrossedBarrier(x,y) then
        createAdventurer()
        nextPhase()
    end
end
mission.phases[4].showUpdateOnEnd = true

mission.phases[5] = {}
mission.phases[5].triggers = {}
mission.phases[5].triggers[1] =
{
    condition = function() return checkAdventurerCreated() end,
    callback = function () return onLastDialog() end,
}
mission.phases[5].onSectorLeft = function()
    if onServer() then
        local mail = lastDialogMail()
        local player = Player()
        player:addMail(mail)
        player:setValue("tutorial_barrier_accomplished", true)
        accomplish()
    end
end


-- helper functions
function checkInventory()
    local player = Player()
    local inventory = player:getInventory()
    local upgrades = inventory:getItemsByType(InventoryItemType.SystemUpgrade)
    local count = {}

    for _, u in pairs(upgrades) do
        local upgrade = u.item
        if upgrade.rarity == Rarity(RarityType.Legendary) then
            if upgrade.script == "data/scripts/systems/teleporterkey1.lua" then
                count[1] = true
            elseif upgrade.script == "data/scripts/systems/teleporterkey2.lua" then
                count[2] = true
            elseif upgrade.script == "data/scripts/systems/teleporterkey3.lua" then
                count[3] = true
            elseif upgrade.script == "data/scripts/systems/teleporterkey4.lua" then
                count[4] = true
            elseif upgrade.script == "data/scripts/systems/teleporterkey5.lua" then
                count[5] = true
            elseif upgrade.script == "data/scripts/systems/teleporterkey6.lua" then
                count[6] = true
            elseif upgrade.script == "data/scripts/systems/teleporterkey7.lua" then
                count[7] = true
            elseif upgrade.script == "data/scripts/systems/teleporterkey8.lua" then
                count[8] = true
            end
        end
    end
    return count
end

function checkShipUpgrades()
    local count = {}

    local shipNames = {Player():getShipNames()}
    for _, name in pairs(shipNames) do
        for system, permannt in pairs(Player():getShipSystems(name)) do
            if system.script == "data/scripts/systems/teleporterkey1.lua" then
                count[1] = true
            elseif system.script == "data/scripts/systems/teleporterkey2.lua" then
                count[2] = true
            elseif system.script == "data/scripts/systems/teleporterkey3.lua" then
                count[3] = true
            elseif system.script == "data/scripts/systems/teleporterkey4.lua" then
                count[4] = true
            elseif system.script == "data/scripts/systems/teleporterkey5.lua" then
                count[5] = true
            elseif system.script == "data/scripts/systems/teleporterkey6.lua" then
                count[6] = true
            elseif system.script == "data/scripts/systems/teleporterkey7.lua" then
                count[7] = true
            elseif system.script == "data/scripts/systems/teleporterkey8.lua" then
                count[8] = true
            end
        end
    end
    return count
end

function checkDistance(x, y)
    local player = Player()
    if not player.craft then return false end
    local distReach = player.craft.hyperspaceJumpReach
    local distBarrier = math.sqrt((x*x)+(y*y)) - 150

    if distBarrier < 0 then setPhase(4) return false end
    if distReach < distBarrier then return false end
    return true
end

function checkCrossedBarrier(x, y)
    local player = Player()
    local distBarrier = math.sqrt((x*x)+(y*y)) - 150
    return distBarrier < 0
end

function createAdventurer()
    if onClient() then invokeServerFunction("createAdventurer") return end
    local adventShip = AdventurerGuide.spawnMissionAdventurer(Player())
    if not adventShip then
        adventShip = Sector():getEntitiesByScript("data/scripts/entity/story/missionadventurer.lua")
    end
    adventShip.invincible = true
    MissionUT.deleteOnPlayersLeft(adventShip)
    mission.data.custom.adventurerId = adventShip.id.string
    adventShip:invokeFunction("story/missionadventurer.lua", "setInteractingScript", "player/missions/tutorials/barriertutorial.lua")
end
callable(nil, "createAdventurer")

function checkAdventurerCreated()
    if onServer() then invokeClientFunction(Player(), "checkAdventurerCreated") return false end

    if not mission.data.custom.adventurerId then return false end
    return Entity(mission.data.custom.adventurerId) ~= nil
end

function getAdventurerName()
    local player = Player()
    local faction = Galaxy():getNearestFaction(player:getHomeSectorCoordinates())
    local language = faction:getLanguage()
    language.seed = Server().seed
    return language:getName()
end

function onGoFindDialog()
    if onServer() then
        sync()
        invokeClientFunction(Player(),"onGoFindDialog")
        return
    end

    local dialog = {}
    if mission.data.custom.foundNoArtifacts then
        dialog.text = "Hey there!\n\nYou've found the Barrier. Have you found any of the Xsotan artifacts?\n\nMy source says that the only way over the Barrier is to have all eight of these artifacts."%_t
    else
        dialog.text = "Hey there!\n\nI see that you've got one of those weird artifacts.\n\nI've heard that there are actually eight of them.\n\nMy source told me that you can cross the Barrier if you have all eight."%_t
    end
    dialog.answers =
    {
        {answer = "Thanks for the information!"%_t}
    }

    local entity = Entity(mission.data.custom.adventurerId)
    if not entity then return end
    entity:invokeFunction("story/missionadventurer.lua", "setData", true, true, dialog)
end

function onFoundArtifactsDialog()
    if onServer() then
        invokeClientFunction(Player(),"onFoundArtifactsDialog")
        if not ShipAI(mission.data.custom.adventurerId) then return end
        ShipAI(mission.data.custom.adventurerId):setIdle()
        return
    end

    local dialog = {}
    dialog.text = "Hey there!\nYou found all eight artifacts. Good job!\n\nIt should be possible to get over the Barrier using those...\n\nAs soon as you find a way I'll catch up to you!"%_t
    dialog.answers =
    {
        {answer = "See you then."%_t}
    }
    dialog.onEnd = "onFoundArtifactsDialogEnd"

    local entity = Entity(mission.data.custom.adventurerId)
    if not entity then return end
    entity:invokeFunction("story/missionadventurer.lua", "setData", true, true, dialog)
end

function onFoundArtifactsDialogEnd()
    if onClient() then invokeServerFunction("onFoundArtifactsDialogEnd") return end

    -- have adventurer despawn after a while
    local entity = Entity(mission.data.custom.adventurerId)
    if not entity then return end
    entity:addScriptOnce("entity/utility/delayeddelete.lua", random():getFloat(6, 10))

    if mission.currentPhase == mission.phases[3] then
        nextPhase()
    end
end
callable(nil, "onFoundArtifactsDialogEnd")

function onLastDialog()
    if onServer() then
        invokeClientFunction(Player(), "onLastDialog")
        return
    end

    local dialog = {}
    local praiseGlad = {}
    local praise = {}
    dialog.text = "You did it! We did it! A dream come true! Now we'll find Avorion and get rich selling it!"%_t
    dialog.answers =
    {
        {answer = "Yeah, thank you for your help!"%_t, followUp = praiseGlad},
        {answer = "Yeah, I'm the best."%_t, followUp = praise}
    }

    praiseGlad.text = "Thank YOU for your help! This is so amazing. Good luck on your search for Avorion!"%_t
    praiseGlad.answers = {{answer = "To you, too!"%_t}}
    praiseGlad.onEnd = "onLastDialogEnd"

    praise.text = "Haha! This is so amazing. Good luck on your search for Avorion!"%_t
    praise.answers = {{answer = "To you, too!"%_t}}
    praise.onEnd = "onLastDialogEnd"

    local entity = Entity(mission.data.custom.adventurerId)
    if not entity then return end
    entity:invokeFunction("story/missionadventurer.lua", "setData", true, true, dialog)
end

function onLastDialogEnd()
    if onClient() then invokeServerFunction("onLastDialogEnd") return end

    -- have adventurer despawn after a while
    local player = Player()
    Entity(mission.data.custom.adventurerId):addScriptOnce("entity/utility/delayeddelete.lua", random():getFloat(6, 10))
    player:setValue("tutorial_barrier_accomplished", true)

    -- spawn new mission for player: Find Avorion
    if not player:getValue("tutorial_findavorion_accomplished") then
        player:addScriptOnce("data/scripts/player/missions/tutorials/findavoriontutorial.lua")
    end

    accomplish()
end
callable(nil, "onLastDialogEnd")

function lastDialogMail()
    local mail = Mail()
    mail.text = Format("Hi there,\n\nI found the wormhole you opened. So amazing! A dream come true! Now we'll find Avorion and get rich selling it!\n\nThank you for your help. I knew it, together we'd make it!\n%1%, the Adventurer"%_T, getAdventurerName())
    mail.header = "Crossed the barrier today!/* Mail Subject */"%_T
    mail.sender = Format("%1%, the Adventurer"%_T, getAdventurerName())
    mail.id = "Tutorial_Barrier"
    return mail
end

function onAdventurerCloseEnough()
    ShipAI(mission.data.custom.adventurerId):setPassive() -- adventurer should stop
end

function onHailRejected()
    if onClient() then invokeServerFunction("onHailRejected") return end

    if mission.currentPhase == mission.phases[3] then setPhase(4) end
end
callable(nil, "onHailRejected")

function onHailAccepted()
    if onClient() then invokeServerFunction("onHailAccepted") end

    -- stop repeating trigger for hail as soon as player accepts
    if mission.currentPhase == mission.phases[5] then mission.phases[5].triggers[1].repeating = nil end
end
callable(nil, "onHailAccepted")
