package.path = package.path .. ";data/scripts/lib/?.lua"
package.path = package.path .. ";data/scripts/?.lua"

include ("structuredmission")
include ("utility")
include ("callable")
include ("goods")
include ("randomext")
include ("relations")
include ("stringutility")
include ("galaxy")

local Dialog = include ("dialogutility")
local SectorSpecifics = include ("sectorspecifics")
local Balancing = include ("galaxy")
local AsyncPirateGenerator = include("asyncpirategenerator")
local ShipGenerator = include ("shipgenerator")
local Placer = include("placer")
local SectorGenerator = include ("SectorGenerator")

--mission.tracing = true


-- custom mission data
mission.data.custom.enoughCargoSpace = 0
mission.data.custom.cargoSpaceChecked = 0
mission.data.custom.amountPotatoes = 45
mission.data.custom.sectorsVisited = 0
mission.data.custom.newX = 0
mission.data.custom.newY = 0
mission.data.custom.piratesX = 0
mission.data.custom.piratesY = 0
mission.data.custom.freighterPirates = {}
mission.data.custom.freighterPiratesCreated = 0
mission.data.custom.leader = nil
mission.data.custom.factionId = nil
mission.data.custom.pirateIds = {}
mission.data.custom.stationId = nil
mission.data.custom.balanceReward = 1

-- mission data
mission.data.title = "Investigate Missing Freighters"%_t
mission.data.brief = mission.data.title

mission.data.description = {}

mission.data.accomplishMessage = "..."%_t
mission.data.failMessage = "..."%_t

--utility for mail if mission leads to rift
mission.globalPhase.onBeginServer = function()
    local player = Player()
    player:registerCallback("onMailRead", "onMailRead")
end
mission.globalPhase.onRestore = function()
    Player():registerCallback("onMailRead", "onMailRead")
end

-- Phase 1: Calculate all necessary values
mission.phases[1] = {}
mission.phases[1].onBeginServer = function()
    local stationName = Entity(mission.data.giver.id).name
    local stationTitle = Entity(mission.data.giver.id).title
    mission.data.description[1] = {text = "For some time now, freighters taking a certain trade route have been disappearing. Pretend to be a freighter yourself and investigate. Report to ${stationTitle} ${stationName} if you have ${cargoSpace} free cargo space."%_T, arguments = {stationTitle = stationTitle, stationName = stationName, cargoSpace = mission.data.custom.amountPotatoes}}
    checkForCargoSpace()
end
mission.phases[1].onBeginClient = function()
        local stationName = Entity(mission.data.giver.id).name
        local stationTitle = Entity(mission.data.giver.id).title
        mission.data.description[1] = {text = "For some time now, freighters taking a certain trade route have been disappearing. Pretend to be a freighter yourself and investigate. Report to ${stationTitle} ${stationName} if you have ${cargoSpace} free cargo space."%_T, arguments = {stationTitle = stationTitle, stationName = stationName, cargoSpace = mission.data.custom.amountPotatoes}}
end
mission.phases[1].updateClient = function()
    if mission.data.custom.cargoSpaceChecked == 1 then
        nextPhase()
    end
end

-- Phase 2: Have player accept mission and add cargo
mission.phases[2] = {}
mission.phases[2].onBeginClient = function()
    if not Entity(mission.data.giver.id) then
        displayMissionAccomplishedText("MISSION ABORTED"%_t, ("The station that hired you no longer exists.")%_t)
        finish()
    else
        ScriptUI(mission.data.giver.id):interactShowDialog(startDialog(), 0)
    end
end
mission.phases[2].onStartDialog = function(entityId)
    if tostring(mission.data.giver.id) == tostring(entityId) then
        local ui = ScriptUI(mission.data.giver.id)
        ui:addDialogOption("I'm ready to find those freighters."%_t, "checkForCargoSpaceAfterReturn")
    end
end

-- Phase 3: Give player the points on the way
mission.phases[3] = {}
mission.phases[3].onSectorEntered = function()
    if onClient() then return end
    local currentX, currentY = Sector():getCoordinates()
    if currentX == mission.data.custom.newX and currentY == mission.data.custom.newY and mission.data.custom.sectorsVisited == 1 then
        calculateNextLocation()
        nextPhase()
    end
end

mission.phases[4] = {}
mission.phases[4].onSectorEntered = function()
    if onClient() then return end
    local currentX, currentY = Sector():getCoordinates()
    if currentX == mission.data.custom.newX and currentY == mission.data.custom.newY and mission.data.custom.sectorsVisited == 2 then
        calculateNextLocation()
        nextPhase()
    end
end

mission.phases[5] = {}
mission.phases[5].onSectorEntered = function()
    if onClient() then return end
    local currentX, currentY = Sector():getCoordinates()
    if currentX == mission.data.custom.newX and currentY == mission.data.custom.newY and mission.data.custom.sectorsVisited == 3 then
        calculateNextLocation()
        nextPhase()
    end
end

mission.phases[6] = {}
mission.phases[6].onSectorEntered = function()
    if onClient() then return end
    local currentX, currentY = Sector():getCoordinates()
    if currentX == mission.data.custom.piratesX and currentY == mission.data.custom.piratesY and mission.data.custom.sectorsVisited == 4 then
        createPirateFaction(mission.data.custom.piratesX, mission.data.custom.piratesY)
    end
end

-- Phase 4: Meet the fake pirates
mission.phases[7] = {}
mission.phases[7].triggers = {}
mission.phases[7].triggers[1] =
    {
        condition = function() -- check if the freighter pirates were created
            local sector = Sector()
            for _, pirate in pairs(mission.data.custom.freighterPirates) do
                if not Entity(pirate) then return false end
            end
            return true
        end,
        callback = function () return startPirateDialog() end
    }

-- Phase 5: Meet the real pirates
mission.phases[8] = {}
mission.phases[8].onTargetLocationEntered = function()
    if onClient() then return end
    createPirates(pirateFaction) -- this is async - go to next phase as soon as it's finished
end

-- Phase 6: start dialog with pirates and fight them
mission.phases[9] = {}
mission.phases[9].triggers = {}
mission.phases[9].triggers[1] =
{
    condition = function() return checkStationCreated() end,
    callback = function () return onStartRealPirateDialog() end,
}
-- Phase 7: Check if pirates were defeated
mission.phases[10] = {}
mission.phases[10].triggers = {}
mission.phases[10].triggers[1] =
{
condition = function() -- check if all real pirates were destroyed
        local sector = Sector()
        local x, y = Sector():getCoordinates()
        for _, pirate in pairs(mission.data.custom.pirateIds) do
            if (x ~= mission.data.custom.piratesX) and (y ~= mission.data.custom.piratesY) then return false end
            if Entity(pirate) then return false end
        end
        if ((x ~= mission.data.custom.piratesX) and (y ~= mission.data.custom.piratesY)) or Entity(mission.data.custom.stationId) then return false end
        return true
    end,
    callback = function () return sendFreightersComeHomeMail() end,
}

mission.phases[11] = {}
mission.phases[11].triggers = {}
mission.phases[11].triggers[1] =
{
    condition = function() -- check if all freighter pirates were destroyed
        local sector = Sector()
        for _, pirate in pairs(mission.data.custom.freighterPirates) do
            local x, y = Sector():getCoordinates()
            if (x ~= mission.data.custom.piratesX) and (y ~= mission.data.custom.piratesY) then return false end
            if Entity(pirate) then return false end
        end
        return true
    end,
    callback = function () return sendFreighterPiratesDefeatedMail() end,
}

-- helper functions
function goNext()
    if onClient() then invokeServerFunction("goNext") return end
    nextPhase()
end
callable(nil, "goNext")

function syncValues(values) -- additional sync just for custom data
    if onServer() then
        invokeClientFunction(Player(), "syncValues", mission.data.custom)
    else
        mission.data.custom = values
    end
end
callable(nil, "syncValues")

function startDialog()
    local d0_Start = {}
    local d1_Finish = {}

    d0_Start.text = "You want to help us investigate what happened to our freighters? Every cargo ship we send on that route simply disappears. But no military ship has found anything. Would you pretend to be a freighter and find out what happened to all the others?"%_t
    d0_Start.answers = {
        {answer = "I think I can do that."%_t, followUp = secondDialog()},
    }

    return d0_Start
end

function secondDialog()
    local d0_EnoughCargoSpace1 = {}
    local d0_EnoughCargoSpace2 = {}
    local d1_NotEnoughCargoSpace = {}

    d0_EnoughCargoSpace1.text = "We will give you some cargo, so you can pretend to be a freighter. Expect dire consequences if you don't return it."%_t
    d0_EnoughCargoSpace1.answers = {{answer = "I will do my best."%_t, followUp = d0_EnoughCargoSpace2}}
    d0_EnoughCargoSpace2.text = "Very well. We are going to send you the route our freighters usually take."%_t
    d0_EnoughCargoSpace2.onEnd = "addFakeCargo"

    d1_NotEnoughCargoSpace.text = "You don't have enough cargo space to take our cargo. Please come back once you have enough cargo space."%_t

    if mission.data.custom.enoughCargoSpace == 1 then
        return d0_EnoughCargoSpace1
    else
        return d1_NotEnoughCargoSpace
    end

end

function checkForCargoSpace()
    if onClient() then invokeServerFunction("checkForCargoSpace") return end

    local player = Player()
    local ship = player.craft
    if not ship or ship.freeCargoSpace == nil or ship.freeCargoSpace < mission.data.custom.amountPotatoes then
        mission.data.custom.enoughCargoSpace = 0
    else
        mission.data.custom.enoughCargoSpace = 1
    end
    local cargoSpaceCheckedCounter = 1
    cargoSpaceChecked(cargoSpaceCheckedCounter)
end
callable(nil, "checkForCargoSpace")

function checkForCargoSpaceAfterReturn()
    if onClient() then invokeServerFunction("checkForCargoSpaceAfterReturn") return end
    local player = Player()
    local ship = player.craft
    if not ship or ship.freeCargoSpace == nil or ship.freeCargoSpace <= mission.data.custom.amountPotatoes then -- adjust free cargo space to goods size
        mission.data.custom.enoughCargoSpace = 0
    else
        mission.data.custom.enoughCargoSpace = 1
    end
    syncValues()
    returnedToStation()
end
callable(nil, "checkForCargoSpaceAfterReturn")

function cargoSpaceChecked(cargoSpaceCheckedCounter)
    if cargoSpaceCheckedCounter == 1 and mission.currentPhase == mission.phases[1] then
        mission.data.custom.cargoSpaceChecked = 1
        nextPhase()
    end
end

function returnedToStation()
    if onServer() then invokeClientFunction(Player(), "returnedToStation") return end
    ScriptUI(mission.data.giver.id):interactShowDialog(secondDialog(), 0)
end

function addFakeCargo()
    if onClient() then invokeServerFunction("addFakeCargo") return end

    local player = Player()
    local ship = player.craft
    ship:addCargo(goods["Potato"]:good(), mission.data.custom.amountPotatoes)
    calculateNextLocation()
    nextPhase()
end
callable(nil, "addFakeCargo")

function abortMission()
    if onClient() then invokeServerFunction("abortMission") return end
    mission.data.failMessage = nil
    fail() -- there is no "MISSION ABORTED" so we use failed
end
callable(nil, "abortMission")

function calculateNextLocation(counter)
    if onClient() then
        invokeServerFunction("calculateNextLocation", counter)
        return
    end

    local counter = counter or 0

    local centerX, centerY = Sector():getCoordinates()
    local randomX = math.random(1,3)
    local randomY = math.random(1,3)

    local newX = 0
    local newY = 0
    if centerX <= 0 then
        newX = centerX + randomX
    elseif centerX > 0 then
        newX = centerX - randomX
    end
    if centerY <= 0 then
        newY = centerY + randomY
    elseif centerY > 0 then
        newY = centerY - randomY
    end

    local specs = SectorSpecifics()
    local regular, offgrid, blocked, home = specs:determineContent(newX, newY, Server().seed)
    if blocked then
        counter = counter + 1
        if counter <= 2 then
            calculateNextLocation(counter)
        else
            --write new description on end of other descriptions and fulfill all other descriptions
            local newDescription = mission.data.custom.sectorsVisited + 2
            local oldDescription = mission.data.custom.sectorsVisited + 1
            mission.data.description[oldDescription].fulfilled = true
            mission.data.description[newDescription] = "It seems as if we lost track of the trading route. We sent you a compensation for your efforts."%_T
            mission.data.reward.credits = mission.data.reward.credits * 0.5
            mission.data.location = nil
            Player():addMail(createMail())
            reward()
            accomplish()
        end

        return
    end


    if mission.data.custom.sectorsVisited == 0 then
        mission.data.description[2] = {text = "The freighters usually go to (${xCoord}:${yCoord}) first."%_T, arguments = {xCoord = newX, yCoord = newY}, bulletPoint = true, fulfilled = false}
    elseif mission.data.custom.sectorsVisited == 1 then
        mission.data.description[2].fulfilled = true
        mission.data.description[3] = {text = "Afterwards, they travel to (${xCoord}:${yCoord})."%_T, arguments = {xCoord = newX, yCoord = newY}, bulletPoint = true, fulfilled = false}
    elseif mission.data.custom.sectorsVisited == 2 then
        mission.data.description[3].fulfilled = true
        mission.data.description[4] = {text = "Then they jump to (${xCoord}:${yCoord})."%_T, arguments = {xCoord = newX, yCoord = newY}, bulletPoint = true, fulfilled = false}
    elseif mission.data.custom.sectorsVisited == 3 then
        getTargetLocation(centerX, centerY, randomX)
        mission.data.description[4].fulfilled = true
        mission.data.description[5] = {text = "Their next stop usually is (${xCoord}:${yCoord})."%_T, arguments = {xCoord = mission.data.custom.piratesX, yCoord = mission.data.custom.piratesY}, bulletPoint = true, fulfilled = false}
    elseif mission.data.custom.sectorsVisited == 4 then
        getTargetLocation(newX, newY, randomX)
    end

    mission.data.custom.sectorsVisited = mission.data.custom.sectorsVisited + 1

    if mission.data.custom.sectorsVisited <= 3 then
        local target = {}
        mission.data.custom.newY = newY
        mission.data.custom.newX = newX
        target.x = newX
        target.y = newY
        mission.data.location = target
    else
        local target = {}
        target.x = mission.data.custom.piratesX
        target.y = mission.data.custom.piratesY
        mission.data.location = target
    end
    syncValues()
end
callable(nil, "calculateNextLocation")

function createMail()
    local mail = Mail()
    mail.text = Format("Hello,\n\nit seems that our information was outdated and the trade route now leads through a rift. We apologize for the inconvenience and transferred a compensation for your time to your account.\n\nGreetings,\n${name}"%_T % {name = tostring(Faction(mission.data.giver.factionIndex).name)})
    mail.header = "Investigate Missing Freighters/* Mail Subject */"%_T
    mail.sender = Format("%s"%_T, tostring(Faction(mission.data.giver.factionIndex).name))
    mail.id = "Investigate_Missing_Freighters"
    return mail
end

function getTargetLocation(x, y, radius)
     if onClient() then invokeServerFunction("getTargetLocation") return end
     local x, y = Sector():getCoordinates()
     local radiusMax = radius or 2
     local coords = {}
     coords.x, coords.y = MissionUT.getSector(x, y, 1, radiusMax, false, false, false, false)

     if (x >= 0 and coords.x >= x) or (x < 0 and coords.x <= x) or (y >= 0 and coords.y >= y) or (y < 0 and coords.y <= y) then
         if radiusMax < 5 then
         radiusMax = radiusMax + 1
         end
         return getTargetLocation(x, y, radiusMax)
     end
     mission.data.custom.piratesX = coords.x
     mission.data.custom.piratesY = coords.y
     local target = {}
     target.x = coords.x
     target.y = coords.y
     mission.data.location = target
     syncValues()
     return coords.x, coords.y
 end
 callable(nil, "getTargetLocation")

 function createPirateFaction(x, y)

     local seed = Seed(string.join({GameSeed(), x, y, "investigatemissingfreighters"}, "-"))
     math.randomseed(seed);

     local generator = SectorGenerator(x, y)

     local language = Language(Seed(makeFastHash(seed.value, x, y)))
     local factionNameBase = language:getFactionName()
     local factionName = ("The " .. factionNameBase .. " Pirates")

     local faction = Galaxy():findFaction(factionName)
     if not faction then
         faction = Galaxy():createFaction(factionName, x, y)
     end

     mission.data.custom.factionId = faction.index

     -- create ships
     for i = 1, 6 do
         local ship = ShipGenerator.createDefender(faction, generator:getPositionInSector())
         ShipAI(ship.id):setAggressive()
         table.insert(mission.data.custom.freighterPirates, ship.id.string)
     end

     mission.data.custom.leader = mission.data.custom.freighterPirates[1]

     Galaxy():setFactionRelationStatus(faction, Faction(), RelationStatus.Ceasefire, false, false)
     syncValues()
     nextPhase()
end

function startPirateDialog()
    if onServer() then return end
    local leader = mission.data.custom.leader
    ScriptUI(mission.data.custom.leader):interactShowDialog(pirateDialog(), 0)
end

function pirateDialog()

    if onServer() then invokeClientFunction(Player(), "pirateDialog") return end

    local dialog1 = {}
    local dialog2 = {}
    local dialog3 = {}
    local dialog4 = {}
    local dialog5 = {}
    local dialog6 = {}
    local dialog7 = {}

    dialog1.text = "If you want to live, you have only one option now!"%_t
    dialog1.answers =
    {
        {answer = "I'm listening."%_t, followUp = dialog2},
        {answer = "I know which one that is. I'll kill you!"%_t, followUp = dialog6},
        {answer = "You're the pirates that destroyed all those freighters!"%_t, followUp = dialog3}
    }

    dialog2.text = "You're going to have to become a pirate and fight for us."%_t
    dialog2.answers =
    {
        {answer = "Never!"%_t, followUp = dialog6},
        {answer = "Why would I do that?"%_t, followUp = dialog3}
    }

    dialog3.text = "We didn't want to become pirates. But they attacked us and told us if we didn't fight for them, they would go after our families."%_t
    dialog3.answers =
    {
        {answer = "Who did that?"%_t, followUp = dialog4},
        {answer = "That's a nice excuse! You're the ones attacking innocent ships, now you're going to have to live with the consequences! Or rather, die with the consequences."%_t, followUp = dialog6}
    }

    dialog4.text = string.format("The pirates that have their base in sector (${xCoord}:${yCoord}). They now have a giant army, because they force everybody to fight for them instead of killing them."%_T % {xCoord = mission.data.custom.piratesX, yCoord = mission.data.custom.piratesY})
    dialog4.answers =
    {
        {answer = "Then why don't you turn on them?"%_t, followUp = dialog5},
        {answer = "You're no better than them now. You're just pirates as well, and you should be punished!"%_t, followUp = dialog6}
    }

    dialog5.text = "They're too strong for us. If we fail, they are going to go after our families."%_t
    dialog5.answers =
    {
        {answer = "Fine, I will try to defeat them for you. But if I ever see one of you committing any crime again, I will hunt all of you down!"%_t, followUp = dialog7}
    }

    dialog6.text = "You are outnumbered. Prepare to die!"%_t
    dialog6.onEnd = "fightTheFreighterPirates"

    dialog7.text = "Thank you so much, we will never attack anyone ever again!"%_t
    dialog7.onEnd = "doNotFightFreighterPirates"

    return dialog1
end

function fightTheFreighterPirates()
    mission.data.description[5].fulfilled = true
    mission.data.description[6] = {text = "Defeat the pirates."%_T, bulletPoint = true, fulfilled = false}

    if onClient() then invokeServerFunction("fightTheFreighterPirates") return end
    local faction = Faction(mission.data.custom.factionId)
    Galaxy():setFactionRelationStatus(faction, Faction(), RelationStatus.War, true, true)
    setPhase(11)
end
callable(nil, "fightTheFreighterPirates")

function doNotFightFreighterPirates()
    calculateNextLocation()
    findThePirateBase()
end

function findThePirateBase()

    mission.data.description[5].fulfilled = true
    mission.data.description[6] = {text = "Find the pirate base."%_T, bulletPoint = true, fulfilled = false}

    if onClient() then invokeServerFunction("findThePirateBase") return end

    local sector = Sector()
    for _, pirate in pairs(mission.data.custom.freighterPirates) do
        if not Entity(pirate) then return
        end
    end
    mission.data.custom.freighterPiratesCreated = 1
    local factionId = mission.data.giver.factionIndex

    local sector = Sector()
    for _, pirate in pairs(mission.data.custom.freighterPirates) do
        Entity(pirate).factionIndex = factionId
    end

    local faction = Faction(mission.data.giver.factionIndex)
    changeRelations(faction, Faction(), 15000)
    -- go to phase(8)
    nextPhase()
end
callable(nil, "findThePirateBase")

function getPositionInSector()
    local position = vec3(math.random(), math.random(), math.random());
    local dist = getFloat(-5000, 5000)
    position = position * dist

    -- create a random up, right and look vector
    local up = vec3(math.random(), math.random(), math.random())
    local look = vec3(math.random(), math.random(), math.random())
    local mat = MatrixLookUp(look, up)
    mat.pos = position

    return mat
end

function createPirates(pirateFaction)
    -- create ships
    local generator = AsyncPirateGenerator(nil, onPiratesCreated)
    local numShips = 3

    mission.data.custom.pirateFactionId = generator:getPirateFaction().index

    generator:startBatch()
    for i = 1, numShips do
        if i <= 2 then
            generator:createScaledBandit(getPositionInSector())
        else
            generator:createScaledPirate(getPositionInSector())
        end
    end
    generator:endBatch()

    -- create station
    local coords = {}
    coords.x, coords.y = Sector():getCoordinates()
    local sectorGenerator = SectorGenerator(mission.data.custom.piratesX, mission.data.custom.piratesY)
    local pirateFaction = Faction(mission.data.custom.pirateFactionId)

    local station = sectorGenerator:createStation(pirateFaction, "data/scripts/entity/merchants/shipyard.lua")
    station:addScriptOnce("entity/ai/patrolpeacefully.lua")
    mission.data.custom.stationId = station.id.string
    station.shieldDurability = station.shieldDurability * 0.1
    if station.shieldDurability and station.shieldDurability > 0 then
        local damage = station.shieldMaxDurability * 0.2
        station:damageShield(damage, station.translationf, station.id)
    else
        local damage = (station.maxDurability or 0) * 0.8
        station:inflictDamage(damage, 1, DamageType.Physical, 0, vec3(), station.id)
    end
    syncValues()
end

function onPiratesCreated(generated)
    for _, pirate in pairs(generated) do
        table.insert(mission.data.custom.pirateIds, pirate.id.string)
        pirate:addScriptOnce("entity/ai/patrolpeacefully.lua") -- don't attack yet, but fly around so that background looks more alive
    end
    -- set phase (9)
    nextPhase()
end

function checkStationCreated()
    if onServer() then return end
    -- check only on client
    local entities = {Sector():getEntitiesByType(EntityType.Station)}
    for _, ent in pairs(entities) do
        if ent.id.string == mission.data.custom.stationId then
            return true
        end
    end
    return false
end

function onStartRealPirateDialog()
    local station = Entity(mission.data.custom.stationId)
    if not station then return end
    ScriptUI(station):interactShowDialog(realPirateDialog(), 0)
end

function realPirateDialog()
    local dialog1 = {}
    local dialog2 = {}

    dialog1.text = "What are you doing here?"%_t
    dialog1.answers = {

    {answer = "I heard you are forcing innocent freighters to fight for you. I will put a stop to that."%_t, followUp = dialog2},
        {answer = "(attack)"%_t, followUp = dialog2},
    }

    dialog2.text = "Let's see you try!"%_t
    dialog2.onEnd = "fightPirates"

    return dialog1
end

function fightPirates()
    if onClient() then
        -- make sure Server knows we fight now
        invokeServerFunction("fightPirates")
    end
    -- set pirates aggressive
    for _, pirateId in pairs(mission.data.custom.pirateIds) do
        if not pirateId then goto continue end
        local pirate = Entity(pirateId)
        if not pirate then goto continue end
        pirate:removeScript("entity/ai/patrolpeacefully.lua")
        pirate:addScriptOnce("entity/ai/patrol.lua")
        ::continue::
    end

    -- set station aggressive as well
    if mission.data.custom.stationId then
        local station = Entity(mission.data.custom.stationId)
        station:removeScript("entity/ai/patrolpeacefully.lua")
        station:addScriptOnce("entity/ai/patrol.lua")
    end
    -- set phase 10
    nextPhase()
end
callable(nil, "fightPirates")

function sendFreightersComeHomeMail()
    if onClient() then invokeServerFunction("sendFreightersComeHomeMail") return end
    local mail = Mail()
    mail.text = "Thank you! \n\nOur people have returned home safely. \n\nWe are in your debt.\n\nYou may keep the cargo we gave you.\n\nGreetings,\n\nLieutenant Omask."%_T
    mail.header = "Thank you for returning our people /* Mail Subject */"%_T
    mail.sender = "Lieutenant Omask"%_T
    mail.id = "FreightersComeHomeMail"

    mission.data.custom.balanceReward = 1.5
    Player():addMail(mail)
    reward()
    accomplish()
end
callable(nil, "sendFreightersComeHomeMail")

function sendFreighterPiratesDefeatedMail()
    if onClient() then invokeServerFunction("sendFreighterPiratesDefeatedMail") return end
    local mail = Mail()
    mail.text = "Thank you! \n\nWe heard you defeated some pirates that were lurking on the trade route. \n\nYou may keep the cargo we gave you.\n\nGreetings,\n\nLieutenant Omask."%_T
    mail.header = "Thank you for defeating the pirates /* Mail Subject */"%_T
    mail.sender = "Lieutenant Omask"%_T
    mail.id = "FreighterPiratesDefeatedMail"

    Player():addMail(mail)
    reward()
    accomplish()
end
callable(nil, "sendFreighterPiratesDefeatedMail")


mission.makeBulletin = function(station)
    local x, y = Sector():getCoordinates()
    local startingPosition = {x = x, y = y}

    local target = {}
    target.x, target.y = Sector():getCoordinates()
    --check if distance to barrier is bigger than distance traveled in mission  to avoid jumps over barrier
    local distanceToBarrier = math.abs((target.x * target.x + target.y * target.y) - (Balancing_GetBlockRingMax() * Balancing_GetBlockRingMax()))
    if not target.x or not target.y or distanceToBarrier < 25 then return end
    mission.data.location = target

    local generator = AsyncPirateGenerator()
    local faction = generator:getPirateFaction()

    local balancing = Balancing.GetSectorRichnessFactor(Sector():getCoordinates()) + mission.data.custom.balanceReward
    reward = {credits = 10000 * balancing, relations = 4000, paymentMessage = "Earned %1% Credits for finding out what happened to the freighters"%_T}
    punishment = {relations = reward.relations}

    local bulletin =
    {
        -- data for the bulletin board
        brief = mission.data.brief,
        title = mission.data.title,
        description = mission.data.description[1],
        difficulty = "Normal /*difficulty*/"%_T,
        reward = "¢${reward}"%_T,
        script = "missions/investigatemissingfreighters.lua",
        formatArguments = {x = target.x, y = target.y, reward = createMonetaryString(reward.credits)},
        msg = "Investigate the missing freighters."%_T,
        giverTitle = station.title,
        giverTitleArgs = station:getTitleArguments(),
        onAccept = [[
            local self, player = ...
            player:sendChatMessage(Entity(self.arguments[1].giver), 0, self.msg, self.formatArguments.x, self.formatArguments.y)
        ]],

        -- data that's important for our own mission
        arguments = {{
            giver = station.id,
            location = target,
            reward = reward,
            punishment = punishment,
            pirateFactionId = faction.index,
            startingPosition = startingPosition
        }},
    }

    return bulletin
end

