package.path = package.path .. ";data/scripts/lib/?.lua"

include("utility")

local cCounter = 0
local function c()
    cCounter = cCounter + 1
    return cCounter
end

-- All relation change types. When you want to change relations and the type doesn't fit into any category, use 'nil'.
-- Using 'nil' in changeRelations() will not make the change be influenced by faction traits or characteristics.
-- If the relation change should be influenced by a faction trait but doesn't exist in the below list,
-- then a new RelationChangeType must most likely be introduced.
RelationChangeType =
{
    -- placeholder for 'nothing special happens', also used when 'nil' is passed
    Default = c(),

    -- bad combat
    CraftDestroyed = c(),
    ShieldsDamaged = c(),
    HullDamaged = c(),
    Boarding = c(),

    -- good combat
    CombatSupport = c(), -- when the player helps the AI faction out in combat

    -- illegal activities
    Smuggling = c(),
    Raiding = c(),
    GeneralIllegal = c(),

    -- commerce
    ServiceUsage = c(), -- ie. repair dock, shipyard, etc.
    ResourceTrade = c(), -- resources
    GoodsTrade = c(), -- goods & cargo
    EquipmentTrade = c(), -- upgrades and other items
    WeaponsTrade = c(), -- turrets, torpedoes and fighters
    Commerce = c(), -- everything else / general commerce

    -- other
    Tribute = c(),

}

RelationChangeNames = {}
RelationChangeNames[RelationChangeType.Default] = "Default"
RelationChangeNames[RelationChangeType.CraftDestroyed] = "CraftDestroyed"
RelationChangeNames[RelationChangeType.ShieldsDamaged] = "ShieldsDamaged"
RelationChangeNames[RelationChangeType.HullDamaged] = "HullDamaged"
RelationChangeNames[RelationChangeType.Boarding] = "Boarding"
RelationChangeNames[RelationChangeType.CombatSupport] = "CombatSupport"
RelationChangeNames[RelationChangeType.Smuggling] = "Smuggling"
RelationChangeNames[RelationChangeType.Raiding] = "Raiding"
RelationChangeNames[RelationChangeType.GeneralIllegal] = "GeneralIllegal"
RelationChangeNames[RelationChangeType.ServiceUsage] = "ServiceUsage"
RelationChangeNames[RelationChangeType.ResourceTrade] = "ResourceTrade"
RelationChangeNames[RelationChangeType.GoodsTrade] = "GoodsTrade"
RelationChangeNames[RelationChangeType.EquipmentTrade] = "EquipmentTrade"
RelationChangeNames[RelationChangeType.WeaponsTrade] = "WeaponsTrade"
RelationChangeNames[RelationChangeType.Commerce] = "Commerce"
RelationChangeNames[RelationChangeType.Tribute] = "Tribute"

RelationChangeMaxCap = {}
RelationChangeMaxCap[RelationChangeType.ServiceUsage] = 45000
RelationChangeMaxCap[RelationChangeType.ResourceTrade] = 45000
RelationChangeMaxCap[RelationChangeType.GoodsTrade] = 65000
RelationChangeMaxCap[RelationChangeType.EquipmentTrade] = 75000
RelationChangeMaxCap[RelationChangeType.WeaponsTrade] = 75000
RelationChangeMaxCap[RelationChangeType.Commerce] = 50000
RelationChangeMaxCap[RelationChangeType.Tribute] = 0

RelationChangeMinCap = {}
RelationChangeMinCap[RelationChangeType.Smuggling] = -75000
RelationChangeMinCap[RelationChangeType.GeneralIllegal] = -75000


function hardCapLoss(relations, delta, threshold)
    if not threshold then return delta end
    if delta > 0 then return delta end

    return -math.min(-delta, math.max(100, relations - threshold))
end

function hardCapGain(relations, delta, threshold)
    if not threshold then return delta end
    if delta < 0 then return delta end

    return math.min(delta, math.max(0, threshold - relations))
end

function softCapLoss(relations, delta, threshold)
    if not threshold then return delta end
    if delta > 0 then return delta end
    if relations <= threshold then return delta end

    return -math.min(-delta, relations - threshold)
end

function softCapGain(relations, delta, threshold)
    if not threshold then return delta end
    if delta < 0 then return delta end
    if relations >= threshold then return delta end

    return math.min(delta, threshold - relations)
end

function getInteractingFactions(a, b)
    local galaxy = Galaxy()

    if type(a) == "number" then a = Faction(a) end
    if not a then return end

    if type(b) == "number" then b = Faction(b) end
    if not b then return end

    -- is one of the two a player faction and the other an ai faction?
    local player -- we handle alliances and players the same in this case
    local ai
    if (a.isAlliance or a.isPlayer) and b.isAIFaction then
        if a.isAlliance then
            player = Alliance(a.index)
        else
            player = Player(a.index)
        end

        ai = b
    elseif (b.isAlliance or b.isPlayer) and a.isAIFaction then
        if b.isAlliance then
            player = Alliance(b.index)
        else
            player = Player(b.index)
        end

        ai = a
    end

    return a, b, ai, player
end

function changeRelations(a, b, delta, changeType, notifyA, notifyB)

    if not delta or delta == 0 then return end

    local a, b, ai, player = getInteractingFactions(a, b)
    if a.isAIFaction and b.isAIFaction then return end
    if a.index == b.index then return end
    if a.alwaysAtWar or b.alwaysAtWar then return end

    local galaxy = Galaxy()

    local relations = galaxy:getFactionRelations(a, b)
    local status = galaxy:getFactionRelationStatus(a, b)
    local newStatus

    if player and ai then
        -- just to make sure that a 'nil' won't cause crashes
        changeType = changeType or RelationChangeType.Default

        local statusChangeThresholdOffset = getStatusChangeThresholdOffset(ai)

--        print ("Relation Change: " .. delta .. " (".. RelationChangeNames[changeType] .. ")")

        delta = getCustomFactionRelationDelta(ai, delta, changeType)

        -- cap reputation gain or loss to a maximum (ie. trading won't improve relations beyond a certain point)
        local uncappedDelta = delta
        if delta > 0 then
            delta = hardCapGain(relations, delta, RelationChangeMaxCap[changeType])
        elseif delta < 0 then
            delta = hardCapLoss(relations, delta, RelationChangeMinCap[changeType])
        end

        if status == RelationStatus.Neutral then
--            print ("Current Sqtatus: Neutral")

            -- Craft destruction and boarding are considered acts of war
            -- damaging shields and hull can be, too, but they must be handled with more context, so we can't handle them here
            if uncappedDelta < 0 then
                if changeType == RelationChangeType.CraftDestroyed
                        or changeType == RelationChangeType.Boarding
                        or changeType == RelationChangeType.Raiding then

                    -- if relations are this bad, war is declared
                    if relations + delta < (-80000 + statusChangeThresholdOffset) then
                        newStatus = RelationStatus.War
                    end
                end

                if changeType == RelationChangeType.HullDamaged then
                    -- doing something bad at -100k means war
                    if uncappedDelta < 0 and relations <= math.max(-100000, -100000 + statusChangeThresholdOffset) then
                        newStatus = RelationStatus.War
                    end
                end
            end

        elseif status == RelationStatus.Ceasefire then
--            print ("Current Status: Ceasefire")

            -- Craft destruction and boarding are considered acts of war
            -- damaging shields and hull can be, too, but they must be handled with more context, so we can't handle them here
            if uncappedDelta < 0
                    and (changeType == RelationChangeType.CraftDestroyed
                    or changeType == RelationChangeType.Boarding
                    or changeType == RelationChangeType.Raiding) then

                -- declare war
                newStatus = RelationStatus.War
            end

            -- a ceasefire turns to Neutral when relations become good enough
            if uncappedDelta > 0 and relations + delta > (-30000 + statusChangeThresholdOffset) then
                -- turn neutral
                newStatus = RelationStatus.Neutral
            end

            -- reaching -100k during ceasefire means war
            if uncappedDelta < 0 and relations + delta <= math.max(-100000, -100000 + statusChangeThresholdOffset) then
                newStatus = RelationStatus.War
            end

        elseif status == RelationStatus.War then
--            print ("Current Status: War")
            -- less gain when at war
            if uncappedDelta > 0 then delta = delta / 2 end

        elseif status == RelationStatus.Allies then
--            print ("Current Status: Allies")

            if changeType == RelationChangeType.Raiding or changeType == RelationChangeType.Boarding then
                -- raiding or boarding an allied ship will turn the pact to neutral
                newStatus = RelationStatus.Neutral
            else
                -- less loss when allied
                if uncappedDelta < 0 then delta = delta / 2 end

                -- also there are soft caps for losing reputation (if a big loss would go beyond the threshold, it's stopped by the threshold)
                delta = softCapLoss(relations, delta, 90000)
                delta = softCapLoss(relations, delta, 80000)

                if relations + delta < (75000 + statusChangeThresholdOffset) then
                    newStatus = RelationStatus.Neutral
                end
            end
        end

        if uncappedDelta < 0 and relations + delta < (-80000 + statusChangeThresholdOffset) then
            terminateReconstructionTreaty(ai, player)
        end


    end

    -- finally change relations and set a new status, if there is one
    galaxy:changeFactionRelations(a, b, delta, notifyA, notifyB)

    if newStatus and newStatus ~= status then
        setRelationStatus(a, b, newStatus, notifyA, notifyB)
    end
end

function setRelationStatus(a, b, status, notifyA, notifyB)
    local a, b, ai, player = getInteractingFactions(a, b)
    if a.isAIFaction and b.isAIFaction then return end

    local galaxy = Galaxy()
    local relations = galaxy:getFactionRelations(a, b)
    local statusBefore = galaxy:getFactionRelationStatus(a, b)

    if status == RelationStatus.War then
        -- war means no more reconstruction
        if player and ai then
            terminateReconstructionTreaty(ai, player)
        end
    elseif status == RelationStatus.Ceasefire then
        -- set to "neutral" immediately when relations go from "War, but many points" to "Ceasefire, but many points"
        if relations > -30000 then status = RelationStatus.Neutral end
    end

    if statusBefore ~= status then
        if player and ai then
            local key = "statuschange_timestamp_" .. tostring(player.index)
            ai:setValue(key, Server().unpausedRuntime)
        end

        galaxy:setFactionRelationStatus(a, b, status, notifyA, notifyB)

        if player and ai then
            if status == RelationStatus.War and player.isPlayer then
                local mail = Mail()
                mail.header = "Declaration of War"%_T
                mail.sender = ai.name
                local text = "This is a declaration of war. The sender of this declaration and %s are now officially at war with each other.\nDue to recent events that cannot be tolerated, the sender is forced to defend their territory against threats like this."%_T
                mail.text = Format(text, player.name)
                player:addMail(mail)
            end
        end
    end
end

function getStatusChangeThresholdOffset(ai)
    local trusting = ai:getTrait("trusting")

    -- very trusting: less points required for relation changes
    -- very mistrustful: more points required for relation changes
    if trusting > 0.85 then
        return -10000
    elseif trusting > 0.5 then
        return -5000
    elseif trusting < -0.85 then
        return 10000
    elseif trusting < -0.5 then
        return 5000
    end

    return 0
end

function terminateReconstructionTreaty(ai, player)
    if player.isAlliance then return end

    local rx, ry = player:getRespawnSectorCoordinates()
    local hx, hy = player:getHomeSectorCoordinates()

    if hx ~= rx or hy ~= ry then
        local reconstructionFaction = Galaxy():getControllingFaction(rx, ry)
        if reconstructionFaction and reconstructionFaction.index == ai.index then
            local hx, hy = player:getHomeSectorCoordinates()

            player:setRespawnSectorCoordinates(hx, hy)

            local mail = Mail()
            mail.header = "Termination of Reconstruction Site"%_T
            mail.sender = ai.name
            mail.text = "Dear Ex-Customer,\n\nDue to recent, very concerning developments in the politics of our two factions, we were forced to terminate our mutual Reconstruction Agreement. Upon destruction, your drone will no longer be reconstructed at our Repair Dock.\nOnce relations between our two factions improve, you're welcome to sign another Reconstruction Site treaty. Don't be afraid to come back once you're welcome here again.\n\nBest wishes,\nRepair Dock Management"%_T
            player:addMail(mail)
        end
    end

end


RelationChangeMultipliers = {}
RelationChangeMultipliers[RelationChangeType.Default] = {}

RelationChangeMultipliers[RelationChangeType.CraftDestroyed] = {forgiving = 1, peaceful = -2, careful = -1}
RelationChangeMultipliers[RelationChangeType.ShieldsDamaged] = {forgiving = 1, careful = -1}
RelationChangeMultipliers[RelationChangeType.HullDamaged] = {forgiving = 1, peaceful = -2, careful = -1}
RelationChangeMultipliers[RelationChangeType.Boarding] = {forgiving = 1, peaceful = -2, careful = -1}

RelationChangeMultipliers[RelationChangeType.CombatSupport] = {peaceful = 2, aggressive = -1, careful = 1, honorable = 1}

RelationChangeMultipliers[RelationChangeType.Smuggling] = {forgiving = 1, careful = -1, generous = 1, greedy = -1, opportunistic = 1, honorable = -1, mistrustful = -1}
RelationChangeMultipliers[RelationChangeType.Raiding] = {forgiving = 1, peaceful = -2, careful = -1, greedy = -1, honorable = -1, mistrustful = -1}
RelationChangeMultipliers[RelationChangeType.GeneralIllegal] = {forgiving = 1, generous = 1, greedy = -1, honorable = -1, mistrustful = -1}

RelationChangeMultipliers[RelationChangeType.ServiceUsage] = {peaceful = 1, greedy = 1, brave = -1}
RelationChangeMultipliers[RelationChangeType.ResourceTrade] = {greedy = 1, brave = -1}
RelationChangeMultipliers[RelationChangeType.GoodsTrade] = {peaceful = 1, greedy = 1, brave = -1}
RelationChangeMultipliers[RelationChangeType.EquipmentTrade] = {peaceful = 1, greedy = 1, brave = -1}
RelationChangeMultipliers[RelationChangeType.WeaponsTrade] = {aggressive = 1, greedy = 1, brave = -1}
RelationChangeMultipliers[RelationChangeType.Commerce] = {peaceful = 1, greedy = 1, brave = -1}

RelationChangeMultipliers[RelationChangeType.Tribute] = {peaceful = 1, aggressive = -1, careful = -1, greedy = 2, mistrustful = -1}

function getCustomFactionRelationDelta(aiFaction, delta, changeType)
    local multipliers = RelationChangeMultipliers[changeType]
    if not multipliers then return end

    for trait, multiplier in pairs(multipliers) do
        local value = aiFaction:getTrait(trait)

        -- traits only kick in if they actually mean something
        if value and value >= 0.5 then
            -- value is between 0 and 1
            -- multiplier is between -2 and 2
            -- in total, 'factor' can vary between -2x and 2x (x being the value below)
            local factor = value * multiplier * 0.25
            delta = delta + math.abs(delta) * factor
        end
    end

    return delta
end
