package.path = package.path .. ";data/scripts/lib/?.lua"
package.path = package.path .. ";data/scripts/?.lua"

include ("utility")
include ("stringutility")
include ("randomext")
include ("galaxy")
local FactionPacks = include ("factionpacks")
local StyleGenerator = include ("internal/stylegenerator.lua")

local StationStyleTypes =
{
    {script = "/factory.lua", factoryStyle = "Factory", subType = StationSubType.Factory},
    {script = "/factory.lua", factoryStyle = "Farm", subType = StationSubType.Farm},
    {script = "/factory.lua", factoryStyle = "Mine", subType = StationSubType.Mine},
    {script = "/factory.lua", factoryStyle = "Collector", subType = StationSubType.Collector},
    {script = "/factory.lua", factoryStyle = "SolarPowerPlant", subType = StationSubType.SolarPowerPlant},
    {script = "/factory.lua", factoryStyle = "Ranch", subType = StationSubType.Ranch},
    {script = "/factory.lua", subType = StationSubType.Factory},

    {script = "/shipyard.lua", subType = StationSubType.Shipyard},
    {script = "/repairdock.lua", subType = StationSubType.RepairDock},
    {script = "/resourcetrader.lua", subType = StationSubType.ResourceDepot},
    {script = "/tradingpost.lua", subType = StationSubType.TradingPost},
    {script = "/equipmentdock.lua", subType = StationSubType.EquipmentDock},
    {script = "/smugglersmarket.lua", subType = StationSubType.SmugglersMarket},
    {script = "/scrapyard.lua", subType = StationSubType.Scrapyard},

    {script = "/fighterfactory.lua", subType = StationSubType.FighterFactory},
    {script = "/turretfactory.lua", subType = StationSubType.TurretFactory},
    {script = "/turretfactorysupplier.lua", subType = StationSubType.TradingPost},

    {script = "/biotope.lua", subType = StationSubType.Biotope},
    {script = "/casino.lua", subType = StationSubType.Casino},
    {script = "/habitat.lua", subType = StationSubType.Habitat},
    {script = "/militaryoutpost.lua", subType = StationSubType.MilitaryOutpost},
    {script = "/headquarters.lua", subType = StationSubType.Headquarters},
    {script = "/researchstation.lua", subType = StationSubType.ResearchStation},

}




local PlanGenerator = {}

function findMaxBlock(plan, dimStr)

    local result
    local maximum = -math.huge
    for i = 0, plan.numBlocks - 1 do
        local block = plan:getNthBlock(i)

        local d = block.box.upper[dimStr]
        if d > maximum then
            result = block
            maximum = d
        end
    end

    return result
end

function findMinBlock(plan, dimStr)

    local result
    local minimum = math.huge
    for i = 0, plan.numBlocks - 1 do
        local block = plan:getNthBlock(i)

        local d = block.box.lower[dimStr]
        if d < minimum then
            result = block
            minimum = d
        end
    end

    return result
end

PlanGenerator.findMinBlock = findMinBlock
PlanGenerator.findMaxBlock = findMaxBlock

function PlanGenerator.selectMaterial(faction)
    local probabilities = Balancing_GetTechnologyMaterialProbability(faction:getHomeSectorCoordinates())
    local material = Material(getValueFromDistribution(probabilities))

    local sector = Sector()
    if sector then
        local x, y = sector:getCoordinates()
        local distFromCenter = length(vec2(x, y))

        if material.value == 6 and distFromCenter > Balancing_GetBlockRingMin() then
            material.value = 5
        end
    end

    return material
end

function PlanGenerator.getShipStyleDefaultName() return "Ship /* Craft classification */"%_T end
function PlanGenerator.getShipStyle(faction, styleName)
    styleName = styleName or PlanGenerator.getShipStyleDefaultName()

    local style = faction:getPlanStyle(styleName)
    if style then return style end

    local styleGenerator = StyleGenerator(faction.index)
    local style = styleGenerator:makeShipStyle(Seed(styleName))

    faction:addPlanStyle(styleName, style)

    return style
end

function PlanGenerator.getCarrierStyleDefaultName() return "Carrier /* Craft classification */"%_T end
function PlanGenerator.getCarrierStyle(faction, styleName)
    styleName = styleName or PlanGenerator.getCarrierStyleDefaultName()

    local style = faction:getPlanStyle(styleName)
    if style then return style end

    local styleGenerator = StyleGenerator(faction.index)
    local style = styleGenerator:makeCarrierStyle(Seed(styleName))

    faction:addPlanStyle(styleName, style)

    return style
end

function PlanGenerator.getFreighterStyleDefaultName() return "Freighter /* Craft classification */"%_T end
function PlanGenerator.getFreighterStyle(faction, styleName)
    styleName = styleName or PlanGenerator.getFreighterStyleDefaultName()

    local style = faction:getPlanStyle(styleName)
    if style then return style end

    local styleGenerator = StyleGenerator(faction.index)
    local style = styleGenerator:makeFreighterStyle(Seed(styleName))

    faction:addPlanStyle(styleName, style)

    return style
end

function PlanGenerator.getMinerStyleDefaultName() return "Miner /* Craft classification */"%_T end
function PlanGenerator.getMinerStyle(faction, styleName)
    styleName = styleName or PlanGenerator.getMinerStyleDefaultName()

    local style = faction:getPlanStyle(styleName)
    if style then return style end

    local styleGenerator = StyleGenerator(faction.index)
    local style = styleGenerator:makeMinerStyle(Seed(styleName))

    faction:addPlanStyle(styleName, style)

    return style
end

function PlanGenerator.getStationStyleDefaultName() return "Station /* Station classification, vanilla station in this case*/"%_T end
function PlanGenerator.getStationStyle(faction, styleName)
    styleName = styleName or PlanGenerator.getStationStyleDefaultName()

    local style = faction:getPlanStyle(styleName)
    if style then return style end

    local styleGenerator = StyleGenerator(faction.index)
    local style = styleGenerator:makeStationStyle(Seed(styleName), StationSubType[styleName])

    faction:addPlanStyle(styleName, style)

    return style
end


function PlanGenerator.makeAsyncFreighterPlan(callback, values, faction, volume, styleName, material, sync)
    local seed = math.random(0xffffffff)

    if not material then
        material = PlanGenerator.selectMaterial(faction)
    end

    local code = [[
        package.path = package.path .. ";data/scripts/lib/?.lua"
        package.path = package.path .. ";data/scripts/?.lua"

        local FactionPacks = include ("factionpacks")
        local PlanGenerator = include ("plangenerator")

        function run(styleName, seed, volume, material, factionIndex, ...)

            local faction = Faction(factionIndex)

            if not volume then
                volume = Balancing_GetSectorShipVolume(faction:getHomeSectorCoordinates());
                local deviation = Balancing_GetShipVolumeDeviation();
                volume = volume * deviation
            end

            local plan = FactionPacks.getFreighterPlan(faction, volume, material)
            if plan then return plan, ... end

            local style = PlanGenerator.getFreighterStyle(faction, styleName)
            local plan = GeneratePlanFromStyle(style, Seed(seed), volume, 5000, nil, material)

            return plan, ...
        end
    ]]

    if sync then
        return execute(code, styleName, seed, volume, material, faction.index)
    else
        values = values or {}
        async(callback, code, styleName, seed, volume, material, faction.index, unpack(values))
    end
end

function PlanGenerator.makeFreighterPlan(faction, volume, styleName, material)
    return PlanGenerator.makeAsyncFreighterPlan(nil, nil, faction, volume, styleName, material, true)
end

function PlanGenerator.makeAsyncMinerPlan(callback, values, faction, volume, styleName, material, sync)
    local seed = math.random(0xffffffff)

    if not material then
        material = PlanGenerator.selectMaterial(faction)
    end

    local code = [[
        package.path = package.path .. ";data/scripts/lib/?.lua"
        package.path = package.path .. ";data/scripts/?.lua"

        local FactionPacks = include ("factionpacks")
        local PlanGenerator = include ("plangenerator")

        function run(styleName, seed, volume, material, factionIndex, ...)

            local faction = Faction(factionIndex)

            if not volume then
                volume = Balancing_GetSectorShipVolume(faction:getHomeSectorCoordinates());
                local deviation = Balancing_GetShipVolumeDeviation();
                volume = volume * deviation
            end

            local plan = FactionPacks.getMinerPlan(faction, volume, material)
            if plan then return plan, ... end

            local style = PlanGenerator.getMinerStyle(faction, styleName)
            local plan = GeneratePlanFromStyle(style, Seed(seed), volume, 5000, 1, material)

            return plan, ...
        end
    ]]

    if sync then
        return execute(code, styleName, seed, volume, material, faction.index)
    else
        values = values or {}
        async(callback, code, styleName, seed, volume, material, faction.index, unpack(values))
    end
end

function PlanGenerator.makeMinerPlan(faction, volume, styleName, material)
    return PlanGenerator.makeAsyncMinerPlan(nil, nil, faction, volume, styleName, material, true)
end

function PlanGenerator.makeAsyncCarrierPlan(callback, values, faction, volume, styleName, material, sync)
    local seed = math.random(0xffffffff)

    if not material then
        material = PlanGenerator.selectMaterial(faction)
    end

    local code = [[
        package.path = package.path .. ";data/scripts/lib/?.lua"
        package.path = package.path .. ";data/scripts/?.lua"

        local FactionPacks = include ("factionpacks")
        local PlanGenerator = include ("plangenerator")

        function run(styleName, seed, volume, material, factionIndex, ...)

            local faction = Faction(factionIndex)
            if not volume then
                volume = Balancing_GetSectorShipVolume(faction:getHomeSectorCoordinates());
                local deviation = Balancing_GetShipVolumeDeviation();
                volume = volume * deviation
            end

            local plan = FactionPacks.getCarrierPlan(faction, volume, material)
            if plan then return plan, ... end

            local style = PlanGenerator.getCarrierStyle(faction, styleName)
            local plan = GeneratePlanFromStyle(style, Seed(seed), volume, 5000, nil, material)

            return plan, ...
        end
    ]]

    if sync then
        return execute(code, styleName, seed, volume, material, faction.index)
    else
        values = values or {}
        async(callback, code, styleName, seed, volume, material, faction.index, unpack(values))
    end
end

function PlanGenerator.makeCarrierPlan(faction, volume, styleName, material)
    return PlanGenerator.makeAsyncCarrierPlan(nil, nil, faction, volume, styleName, material, true)
end

function PlanGenerator.makeAsyncShipPlan(callback, values, faction, volume, styleName, material, sync)
    local seed = math.random(0xffffffff)

    if not material then
        material = PlanGenerator.selectMaterial(faction)
    end

    local code = [[
        package.path = package.path .. ";data/scripts/lib/?.lua"
        package.path = package.path .. ";data/scripts/?.lua"

        local FactionPacks = include ("factionpacks")
        local PlanGenerator = include ("plangenerator")

        function run(styleName, seed, volume, material, factionIndex, ...)

            local faction = Faction(factionIndex)
            if not volume then
                volume = Balancing_GetSectorShipVolume(faction:getHomeSectorCoordinates());
                local deviation = Balancing_GetShipVolumeDeviation();
                volume = volume * deviation
            end

            local plan = FactionPacks.getShipPlan(faction, volume, material)
            if plan then return plan, ... end

            local style = PlanGenerator.getShipStyle(faction, styleName)

            plan = GeneratePlanFromStyle(style, Seed(seed), volume, 6000, 1, material)
            return plan, ...
        end
    ]]

    if sync then
        return execute(code, styleName, seed, volume, material, faction.index)
    else
        values = values or {}
        async(callback, code, styleName, seed, volume, material, faction.index, unpack(values))
    end
end

function PlanGenerator.makeShipPlan(faction, volume, styleName, material)
    return PlanGenerator.makeAsyncShipPlan(nil, nil, faction, volume, styleName, material, true)
end

function PlanGenerator.makeFighterPlan(factionIndex, seed, material)
    local plan = FactionPacks.getFighterPlan(factionIndex, nil, material)
    if plan then return plan end

    local styleGenerator = StyleGenerator(factionIndex)
    local style = styleGenerator:makeFighterStyle(seed)
    local plan = GeneratePlanFromStyle(style, Seed(tostring(seed) .. "+1"), 5000, 200, nil, material)

    return plan
end

function PlanGenerator.makeXsotanShipPlan(volume, material)
    local styleGenerator = StyleGenerator(1337)

    local seed = math.random(0xffffffff)
    local style = styleGenerator:makeXsotanShipStyle(seed)
    local plan = GeneratePlanFromStyle(style, Seed(tostring(seed)), volume, 5000, nil, material)
    return plan
end

function PlanGenerator.makeXsotanCarrierPlan(volume, material)
    local styleGenerator = StyleGenerator(1337)

    local seed = math.random(0xffffffff)
    local style = styleGenerator:makeXsotanCarrierStyle(seed)
    local plan = GeneratePlanFromStyle(style, Seed(tostring(seed)), volume, 5000, nil, material)
    return plan
end

function PlanGenerator.determineStationStyleFromScriptArguments(scriptName, arg1)
    if not scriptName then return "Station" end

    local result = nil
    for _, style in pairs(StationStyleTypes) do

        if scriptName:ends(style.script) then
            if style.factoryStyle then
                if arg1 and style.factoryStyle == arg1.factoryStyle then
                    result = style.subType
                    break
                end
            else
                result = style.subType
                break
            end
        end
    end

    if not result then
        eprint("Warning: No station style found for " .. scriptName)
        result = StationSubType.Default
    end

    for k, v in pairs(StationSubType) do
        if v == result then return k end
    end

    return "Station"
end


function PlanGenerator.makeStationPlan(faction, styleName, seed, volume, material)
    seed = seed or math.random(0xffffffff)
    if not volume then
        volume = Balancing_GetSectorStationVolume(faction:getHomeSectorCoordinates())
        volume = volume * Balancing_GetStationVolumeDeviation()
    end

    material = material or PlanGenerator.selectMaterial(faction)

    local plan = FactionPacks.getStationPlan(faction, volume, material, styleName)
    if plan then return plan end

    local style = PlanGenerator.getStationStyle(faction, styleName)

    local plan = GeneratePlanFromStyle(style, Seed(seed), volume, 10000, nil, material)
    return plan, seed, volume, material
end

function PlanGenerator.makeBigAsteroidPlan(size, resources, material, iterations)

    iterations = iterations or 10

    local directions = {
        vec3(1, 0, 0), vec3(-1, 0, 0),
        vec3(0, 1, 0), vec3(0, -1, 0),
        vec3(0, 0, 1), vec3(0, 0, -1)
    }

    local plan = PlanGenerator.makeSmallAsteroidPlan(1.0, resources, material)

    local centers = {plan:getBlock(0)}
    local numCenters = 1

    for i = 1, iterations do

        local center = centers[getInt(1, numCenters)]
        local dir = directions[getInt(1, 6)]

        -- make a plan and attach it to the selected center in the selected direction
        local other = PlanGenerator.makeSmallAsteroidPlan(1.0, resources, material)
        local otherCenter = other:getBlock(0)

        local displacement = (center.box.size + otherCenter.box.size) * dir * 0.5
        other:displace(displacement)

        local index = plan:addPlan(center.index, other, otherCenter.index)

        table.insert(centers, plan:getBlock(index))
        numCenters = numCenters + 1

    end

    local scale = size / plan.radius
    plan:scale(vec3(scale, scale, scale))

    return plan
end

function PlanGenerator.makeSmallHiddenResourcesAsteroidPlan(size, material)
    return PlanGenerator.makeSmallAsteroidPlanEx(size, 2, material)
end

function PlanGenerator.makeSmallAsteroidPlan(size, resources, material)
    local resourceType = 0
    if resources then resourceType = 1 end

    return PlanGenerator.makeSmallAsteroidPlanEx(size, resourceType, material)
end

function PlanGenerator.makeSmallAsteroidPlanEx(size, resourceType, material)

    resourceType = resourceType or 0
    material = material or Material(0)

    local plan = BlockPlan()

    local color = material.blockColor

    local from = 0.1
    local to = 0.5

    local center
    local border
    local edge
    local corner

    if resourceType == 0 then
        center = BlockType.Stone
        border = BlockType.Stone
        edge = BlockType.StoneEdge
        corner = BlockType.StoneCorner
    elseif resourceType == 1 then -- normal rich stone
        center = BlockType.RichStone
        border = BlockType.RichStone
        edge = BlockType.RichStoneEdge
        corner = BlockType.RichStoneCorner
    elseif resourceType == 2 then -- hidden super rich stone
        center = BlockType.SuperRichStone
        border = BlockType.Stone
        edge = BlockType.StoneEdge
        corner = BlockType.StoneCorner

        to = 0.25
    end


    local ls = vec3(getFloat(from, to), getFloat(from, to), getFloat(from, to))
    local us = vec3(getFloat(from, to), getFloat(from, to), getFloat(from, to))
    local s = vec3(1, 1, 1) - ls - us

    local hls = ls * 0.5
    local hus = us * 0.5
    local hs = s * 0.5



    local ci = plan:addBlock(vec3(0, 0, 0), s, -1, -1, color, material, Matrix(), center)

    -- top bottom
    plan:addBlock(vec3(0, hs.y + hus.y, 0), vec3(s.x, us.y, s.z), ci, -1, color, material, Matrix(), border)
    plan:addBlock(vec3(0, -hs.y - hls.y, 0), vec3(s.x, ls.y, s.z), ci, -1, color, material, Matrix(), border)

    -- left right
    plan:addBlock(vec3(hs.x + hus.x, 0, 0), vec3(us.x, s.y, s.z), ci, -1, color, material, Matrix(), border)
    plan:addBlock(vec3(-hs.x - hls.x, 0, 0), vec3(ls.x, s.y, s.z), ci, -1, color, material, Matrix(), border)

    -- front back
    plan:addBlock(vec3(0, 0, hs.z + hus.z), vec3(s.x, s.y, us.z), ci, -1, color, material, Matrix(), border)
    plan:addBlock(vec3(0, 0, -hs.z - hls.z), vec3(s.x, s.y, ls.z), ci, -1, color, material, Matrix(), border)


    -- top left right
    plan:addBlock(vec3(hs.x + hus.x, hs.y + hus.y, 0), vec3(us.x, us.y, s.z), ci, -1, color, material, MatrixLookUp(vec3(-1, 0, 0), vec3(0, 1, 0)), edge)
    plan:addBlock(vec3(-hs.x - hls.x, hs.y + hus.y, 0), vec3(ls.x, us.y, s.z), ci, -1, color, material, MatrixLookUp(vec3(1, 0, 0), vec3(0, 1, 0)), edge)

    -- top front back
    plan:addBlock(vec3(0, hs.y + hus.y, hs.z + hus.z), vec3(s.x, us.y, us.z), ci, -1, color, material, MatrixLookUp(vec3(0, 0, -1), vec3(0, 1, 0)), edge)
    plan:addBlock(vec3(0, hs.y + hus.y, -hs.z - hls.z), vec3(s.x, us.y, ls.z), ci, -1, color, material, MatrixLookUp(vec3(0, 0, 1), vec3(0, 1, 0)), edge)

    -- bottom left right
    plan:addBlock(vec3(hs.x + hus.x, -hs.y - hls.y, 0), vec3(us.x, ls.y, s.z), ci, -1, color, material, MatrixLookUp(vec3(-1, 0, 0), vec3(0, -1, 0)), edge)
    plan:addBlock(vec3(-hs.x - hls.x, -hs.y - hls.y, 0), vec3(ls.x, ls.y, s.z), ci, -1, color, material, MatrixLookUp(vec3(1, 0, 0), vec3(0, -1, 0)), edge)

    -- bottom front back
    plan:addBlock(vec3(0, -hs.y - hls.y, hs.z + hus.z), vec3(s.x, ls.y, us.z), ci, -1, color, material, MatrixLookUp(vec3(0, 0, -1), vec3(0, -1, 0)), edge)
    plan:addBlock(vec3(0, -hs.y - hls.y, -hs.z - hls.z), vec3(s.x, ls.y, ls.z), ci, -1, color, material, MatrixLookUp(vec3(0, 0, 1), vec3(0, -1, 0)), edge)

    -- middle left right
    plan:addBlock(vec3(hs.x + hus.x, 0, -hs.z - hls.z), vec3(us.x, s.y, ls.z), ci, -1, color, material, MatrixLookUp(vec3(-1, 0, 0), vec3(0, 0, -1)), edge)
    plan:addBlock(vec3(-hs.x - hls.x, 0, -hs.z - hls.z), vec3(ls.x, s.y, ls.z), ci, -1, color, material, MatrixLookUp(vec3(1, 0, 0), vec3(0, 0, -1)), edge)

    -- middle front back
    plan:addBlock(vec3(hs.x + hus.x, 0, hs.z + hus.z), vec3(us.x, s.y, us.z), ci, -1, color, material, MatrixLookUp(vec3(-1, 0, 0), vec3(0, 0, 1)), edge)
    plan:addBlock(vec3(-hs.x - hls.x, 0, hs.z + hus.z), vec3(ls.x, s.y, us.z), ci, -1, color, material, MatrixLookUp(vec3(1, 0, 0), vec3(0, 0, 1)), edge)


    -- top edges
    -- left right
    plan:addBlock(vec3(hs.x + hus.x, hs.y + hus.y, -hs.z - hls.z), vec3(us.x, us.y, ls.z), ci, -1, color, material, MatrixLookUp(vec3(-1, 0, 0), vec3(0, 1, 0)), corner)
    plan:addBlock(vec3(-hs.x - hls.x, hs.y + hus.y, -hs.z - hls.z), vec3(ls.x, us.y, ls.z), ci, -1, color, material, MatrixLookUp(vec3(1, 0, 0), vec3(0, 0, -1)), corner)

    -- front back
    plan:addBlock(vec3(hs.x + hus.x, hs.y + hus.y, hs.z + hus.z), vec3(us.x, us.y, us.z), ci, -1, color, material, MatrixLookUp(vec3(-1, 0, 0), vec3(0, 0, 1)), corner)
    plan:addBlock(vec3(-hs.x - hls.x, hs.y + hus.y, hs.z + hus.z), vec3(ls.x, us.y, us.z), ci, -1, color, material, MatrixLookUp(vec3(1, 0, 0), vec3(0, 1, 0)), corner)

    -- bottom edges
    -- left right
    plan:addBlock(vec3(hs.x + hus.x, -hs.y - hls.y, -hs.z - hls.z), vec3(us.x, ls.y, ls.z), ci, -1, color, material, MatrixLookUp(vec3(0, 0, 1), vec3(0, -1, 0)), corner)
    plan:addBlock(vec3(-hs.x - hls.x, -hs.y - hls.y, -hs.z - hls.z), vec3(ls.x, ls.y, ls.z), ci, -1, color, material, MatrixLookUp(vec3(1, 0, 0), vec3(0, -1, 0)), corner)

    -- front back
    plan:addBlock(vec3(hs.x + hus.x, -hs.y - hls.y, hs.z + hus.z), vec3(us.x, ls.y, us.z), ci, -1, color, material, MatrixLookUp(vec3(-1, 0, 0), vec3(0, -1, 0)), corner)
    plan:addBlock(vec3(-hs.x - hls.x, -hs.y - hls.y, hs.z + hus.z), vec3(ls.x, ls.y, us.z), ci, -1, color, material, MatrixLookUp(vec3(0, 0, -1), vec3(0, -1, 0)), corner)

    plan:scale(vec3(getFloat(0.3, 1.5), getFloat(0.3, 1.5), getFloat(0.3, 1.5)))

    local r = size * 2.0 / plan.radius
    plan:scale(vec3(r, r, r))

    plan.convex = true

    return plan
end

function PlanGenerator.makeGatePlan(seed, color1, color2, color3)

    local r = random()
    local random = r

    if seed then random = Random(seed) end

    local bright = color1 or ColorRGB(0.5, 0.5, 0.5)
    local dark = color2 or ColorRGB(0.25, 0.25, 0.25)
    local colored = color3 or ColorHSV(random:getFloat(0, 360), random:getFloat(0.5, 0.7), random:getFloat(0.4, 0.6))
    local iron = Material()
    local orientation = Matrix()
    local block = BlockType.BlankHull
    local edge = BlockType.EdgeHull

    local slopes = random:getFloat() < 0.5 and true or false
    local lightLines = random:getFloat() < 0.35 and true or false
    local rings = false
    local rings2 = false
    local rings3 = false
    local bubbleLights = false
    local secondaryLine = random:getFloat() < 0.5 and true or false
    local secondaryArms = random:getFloat() < 0.35 and true or false

    if not lightLines then
        rings = random:getFloat() < 0.5 and true or false
        rings2 = random:getFloat() < 0.5 and true or false
    end

    if not slopes and not rings then
        rings3 = true
    end

    if not lightLines then
        bubbleLights = true
    end

    local segment = BlockPlan()

    -- make main arm
    -- create 2 possible thicknesses for the default blocks
    local t1 = random:getFloat(1.3, 1.75)
    local t2 = random:getFloat(0.75, 1.3)

    -- choose from the 2 thicknesses
    local ta =  t2
    local tb = random:getFloat() < 0.5 and t1 or t2
    local tc = random:getFloat() < 0.5 and t1 or t2

    local ca = random:getFloat() < 0.5 and bright or dark
    local cb = random:getFloat() < 0.5 and bright or dark
    local cc = random:getFloat() < 0.5 and bright or dark
    local cd = bright

    if not slopes then cd = colored end

    local root = segment:addBlock(vec3(0, 1 + 1, 0), vec3(ta, 2, ta), -1, -1, ca, iron, orientation, block)
    local a = root

    local b = segment:addBlock(vec3(0, 1 + 3, 0), vec3(tb, 2, tb), a, -1, cb, iron, orientation, block)
    local c = segment:addBlock(vec3(0, 1 + 5, 0), vec3(tc, 2, tc), b, -1, cc, iron, orientation, block)
    local d = segment:addBlock(vec3(0, 1 + 7, 0), vec3(2.5, 2.5, 2.5), c, -1, cd, iron, orientation, block)

    -- antennas front back
    -- segment:addBlock(vec3(0, 1 + 7, 2), vec3(0.2, 0.2, 2.5), last, -1, white, iron, orientation, block)
    -- segment:addBlock(vec3(0, 1 + 7, -2), vec3(0.2, 0.2, 2.5), last, -1, white, iron, orientation, block)

    -- antennas outside
    local antennas = random:getInt(2, 4)

    for i = 1, antennas do
        local p = random:getVector(-1, 1)
        local f = random:getFloat(0.25, 1.75)
        p.x = 0;

        local s = vec3(2.5, 0.05, 0.05) * f
        segment:addBlock(vec3(2 + s.x * 0.5 - 1, 1 + 7, 0) + p, s, last, -1, bright, iron, orientation, block)
    end

    if secondaryLine then
        segment:addBlock(vec3(1, 1 + 2.875, 0), vec3(0.5, 5.75, 0.5), a, -1, ca, iron, MatrixLookUp(vec3(0, 1, 0), vec3(0, 0, -1)), BlockType.Light)
    end

    if bubbleLights then
        segment:addBlock(vec3(0, 1 + 7, 1.5), vec3(0.75, 0.75, 2), d, -1, ca, iron, MatrixLookUp(vec3(0, 1, 0), vec3(0, 0, -1)), BlockType.Light)
        segment:addBlock(vec3(0, 1 + 7, -1.5), vec3(0.75, 0.75, 2), d, -1, ca, iron, MatrixLookUp(vec3(0, 1, 0), vec3(0, 0, 1)), BlockType.Light)
    end

    if rings then
        local h = 0.1

        segment:addBlock(vec3(0, 1 + 1, 0), vec3(ta, h, ta) + h, a, -1, ca, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 3, 0), vec3(tb, h, tb) + h, b, -1, cb, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 5, 0), vec3(tc, h, tc) + h, c, -1, cc, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 7, 0), vec3(2.5, h, 2.5) + h, d, -1, bright, iron, orientation, block)
    end

    if rings2 then
        local h = 0.1

        segment:addBlock(vec3(0, 1 + 1, 0), vec3(h, 2.5, ta) + h, a, -1, ca, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 3, 0), vec3(h, 2.5, tb) + h, b, -1, cb, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 5, 0), vec3(h, 2.5, tc) + h, c, -1, cc, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 7, 0), vec3(h, 2.5, 2.5) + h, d, -1, bright, iron, orientation, block)
    end

    if rings3 then
        local h = 0.5

        segment:addBlock(vec3(0, 1 + 1, 0), vec3(ta, h, ta) + h, a, -1, ca, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 3, 0), vec3(tb, h, tb) + h, b, -1, cb, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 5, 0), vec3(tc, h, tc) + h, c, -1, cc, iron, orientation, block)
    end

    if lightLines then
        local h = 0.05

        local block = BlockType.Glow
        local color = copy(colored)
        color.value = 1.0

        segment:addBlock(vec3(0, 1 + 1, 0), vec3(h, 2, ta) + h, a, -1, color, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 3, 0), vec3(h, 2, tb) + h, b, -1, color, iron, orientation, block)
        segment:addBlock(vec3(0, 1 + 5, 0), vec3(h, 2, tc) + h, c, -1, color, iron, orientation, block)

        if random:getFloat() < 0.5 then
            segment:addBlock(vec3(0, 1 + 7, 0), vec3(2.5, h, 2.5) + h, d, -1, color, iron, orientation, block)
        else
            segment:addBlock(vec3(0, 1 + 7, 0), vec3(h, 2.5, 2.5) + h, d, -1, color, iron, orientation, block)
        end
    end

    if slopes then
        -- slope segments
        local slopeWidth = random:getFloat(t1 + 0.1, 2.5) -- 2.0 to 2.5, but always smaller than the biggest last element
        local slopeColor = colored -- one of the 3
        local slopeHeight = random:getFloat(0.5, 1.5)
        local slopeDist = random:getFloat() < 0.15 and slopeHeight * 0.125 or random:getFloat(slopeHeight * 0.15, slopeHeight * 0.5)
        local slopeStart = random:getFloat(3.0, 5.0)

        local w = slopeWidth
        local hw = w * 0.5
        local h = slopeHeight
        local hh = h * 0.5
        local m1 = MatrixLookUp(vec3(1, 0, 0), vec3(0, 1, 0))
        local m2 = MatrixLookUp(vec3(-1, 0, 0), vec3(0, -1, 0))

        for p = slopeStart, 7, slopeDist * 2.0 do
            segment:addBlock(vec3(-hw * 0.5, p, 0), vec3(hw, hh, w), last, -1, slopeColor, iron, m1, edge)
            segment:addBlock(vec3(hw * 0.5, p + hh, 0), vec3(hw, hh, w), last, -1, slopeColor, iron, m1, edge)

            segment:addBlock(vec3(-hw * 0.5, p - hh, 0), vec3(hw, hh, w), last, -1, slopeColor, iron, m2, edge)
            segment:addBlock(vec3(hw * 0.5, p, 0), vec3(hw, hh, w), last, -1, slopeColor, iron, m2, edge)
        end
    end

    local arm = nil
    if secondaryArms then
        arm = copy(segment)
    end

    local size = vec3(2, 2, 2)
    local plan = BlockPlan()

    local root = plan:addBlock(vec3(0, 0, 0), vec3(20, 20, 0.1), -1, -1, bright, iron, orientation, BlockType.Portal)

    plan:addBlock(vec3(0, -10, 0), size, root, -1, bright, iron, orientation, block)
    plan:addBlock(vec3(0, 10, 0), size, root, -1, bright, iron, orientation, block)

    plan:addBlock(vec3(10, 0, 0), size, root, -1, bright, iron, orientation, block)
    plan:addBlock(vec3(-10, 0, 0), size, root, -1, bright, iron, orientation, block)

    -- default
    segment:displace(vec3(10, 0, 0))
    plan:addPlan(0, segment, 0)

    -- mirrored down
    segment:mirror(vec3(0, 1, 0), vec3(0, 0, 0))
    plan:addPlan(0, segment, 0)

    -- mirrored down other side
    segment:mirror(vec3(1, 0, 0), vec3(0, 0, 0))
    plan:addPlan(0, segment, 0)

    -- default other side
    segment:mirror(vec3(0, 1, 0), vec3(0, 0, 0))
    plan:addPlan(0, segment, 0)

    -- turned
    segment:rotate(vec3(0, 0, 1), 1)
    plan:addPlan(0, segment, 0)

    segment:mirror(vec3(1, 0, 0), vec3(0, 0, 0))
    plan:addPlan(0, segment, 0)

    segment:mirror(vec3(0, 1, 0), vec3(0, 0, 0))
    plan:addPlan(0, segment, 0)

    segment:mirror(vec3(1, 0, 0), vec3(0, 0, 0))
    plan:addPlan(0, segment, 0)

    if secondaryArms then
        if random:getFloat() < 0.5 then
            arm:rotate(vec3(0, 0, 1), 1)
            arm:displace(vec3(-10, 0, 0))
            plan:addPlan(0, arm, 0)

            arm:mirror(vec3(1, 0, 0), vec3(0, 0, 0))
            plan:addPlan(0, arm, 0)
        else
            arm:rotate(vec3(0, 0, 1), 1)
            arm:rotate(vec3(0, 1, 0), 1)

            arm:displace(vec3(-10, 0, 0))
            plan:addPlan(0, arm, 0)

            arm:mirror(vec3(1, 0, 0), vec3(0, 0, 0))
            plan:addPlan(0, arm, 0)

            if random:getFloat() < 0.5 then
                arm:mirror(vec3(0, 0, 1), vec3(0, 0, 0))
                plan:addPlan(0, arm, 0)

                arm:mirror(vec3(1, 0, 0), vec3(0, 0, 0))
                plan:addPlan(0, arm, 0)
            end

        end
    end

    local scale = 6
    plan:scale(vec3(scale, scale, scale))
    return plan
end

function PlanGenerator.makeBeaconPlan(colors)
    local container = PlanGenerator.makeContainerPlan(colors)

    container:scale(vec3(0.5, 0.5, 2))


    local maxZ = findMaxBlock(container, "z")
    local minZ = findMinBlock(container, "z")

    container:addBlock(maxZ.box.position + vec3(0, 0, maxZ.box.size.z), maxZ.box.size, maxZ.index, -1, maxZ.color, maxZ.material, MatrixLookUp(vec3(1, 0, 0), vec3(0, 0, 1)), BlockType.Light)
    container:addBlock(minZ.box.position - vec3(0, 0, minZ.box.size.z), minZ.box.size, minZ.index, -1, minZ.color, minZ.material, MatrixLookUp(vec3(1, 0, 0), vec3(0, 0, -1)), BlockType.Light)



    return container
end

function PlanGenerator.makeContainerPlan(colors)
    return PlanGenerator.makeGenericContainerPlan(colors, BlockType.CargoBay)
end

function PlanGenerator.makeCrewQuartersPlan(colors)
    return PlanGenerator.makeGenericContainerPlan(colors, BlockType.Quarters)
end

function PlanGenerator.makeGenericContainerPlan(colors_in, blockType)
    local plan = BlockPlan()

    -- create root block
    local root = plan:addBlock(vec3(0, 0, 0), vec3(2, 2, 2), -1, -1, ColorRGB(1, 1, 1), Material(), Matrix(), blockType)


    local brightColor = getFloat(0.75, 1);
    local darkColor = getFloat(0.2, 0.6);

    local colors = colors_in
    if colors == nil then
        colors = {}
        table.insert(colors, ColorRGB(brightColor, brightColor, brightColor))
        table.insert(colors, ColorRGB(darkColor, darkColor, darkColor))
        table.insert(colors, ColorRGB(math.random(), math.random(), math.random()))
    end

    -- maybe add to front, back, top, bottom
    if math.random() < 0.2 then
        local size = math.random() * 1.5 + 0.5 -- 0.5 to 2.0

        local color = colors[math.random(1, 3)]

        plan:addBlock(vec3(0, 1 + size / 2, 0), vec3(size, size, size), root, -1, color, Material(), Matrix(), blockType)
        plan:addBlock(vec3(0, -(1 + size / 2), 0), vec3(size, size, size), root, -1, color, Material(), Matrix(), blockType)
    end

    if math.random() < 0.2 then
        local size = math.random() * 1.5 + 0.5 -- 0.5 to 2.0

        local color = colors[math.random(1, 3)]

        plan:addBlock(vec3(1 + size / 2, 0, 0), vec3(size, size, size), root, -1, color, Material(), Matrix(), blockType)
        plan:addBlock(vec3(-(1 + size / 2), 0, 0), vec3(size, size, size), root, -1, color, Material(), Matrix(), blockType)
    end



    -- now add to the sides
    local blockPairs = {}
    local added = 0
    local maxAdded = math.random() * 2.5 + 1.5 -- 1.5 to 4.0
    while added < maxAdded do
        local thickness;
        local size = math.random() * 1.0 + 1.5 -- 1.5 to 2.5

        if math.random() < 0.3 then
            thickness = math.random() * 0.2 + 0.1 -- 0.1 to 0.3
        else
            thickness = math.random() * 1.5 + 0.5 -- 0.5 to 2.0
        end

        local color = colors[math.random(1, 3)]

        local a = plan:addBlock(vec3(0, 0, added + thickness / 2), vec3(size, size, thickness), root, -1, color, Material(), Matrix(), blockType)
        local b = plan:addBlock(vec3(0, 0, -(added + thickness / 2)), vec3(size, size, thickness), root, -1, color, Material(), Matrix(), blockType)

        if thickness > 1.0 then
            table.insert(blockPairs, {a = a, b = b})
        end

        added = added + thickness
    end

    for i, blocks in pairs(blockPairs) do
        if math.random() < 0.3 then
            -- add x blocks
            local newWidth = math.random() * 0.3 + 0.7 -- 0.7 to 1.0
            local newThick = math.random() * 0.2 + 0.1 -- 0.1 to 0.3
            local newSize = vec3(newThick, newWidth, newWidth)

            -- block a
            local size = plan:getBlock(blocks.a).box.size

            -- +x
            local newPos = plan:getBlock(blocks.a).box.position
            newPos.x = newPos.x + (size.x / 2 + newSize.x / 2)
            plan:addBlock(newPos, newSize, blocks.a, -1, ColorRGB(1, 1, 1), Material(), Matrix(), blockType)

            -- -x
            local newPos = plan:getBlock(blocks.a).box.position
            newPos.x = newPos.x - (size.x / 2 + newSize.x / 2)
            plan:addBlock(newPos, newSize, blocks.a, -1, ColorRGB(1, 1, 1), Material(), Matrix(), blockType)


            -- block b
            local size = plan:getBlock(blocks.b).box.size

            -- +x
            local newPos = plan:getBlock(blocks.b).box.position
            newPos.x = newPos.x + (size.x / 2 + newSize.x / 2)
            plan:addBlock(newPos, newSize, blocks.b, -1, ColorRGB(1, 1, 1), Material(), Matrix(), blockType)

            -- -x
            local newPos = plan:getBlock(blocks.b).box.position
            newPos.x = newPos.x - (size.x / 2 + newSize.x / 2)
            plan:addBlock(newPos, newSize, blocks.b, -1, ColorRGB(1, 1, 1), Material(), Matrix(), blockType)

        end

        if math.random() < 0.3 then
            -- add x blocks
            local newWidth = getFloat(0.7, 1.0) -- 0.7 to 1.0
            local newThick = getFloat(0.1, 0.5) -- 0.1 to 0.5
            local newSize = vec3(newWidth, newThick, newWidth)

            -- block a
            local size = plan:getBlock(blocks.a).box.size

            -- +y
            local newPos = plan:getBlock(blocks.a).box.position
            newPos.y = newPos.y + (size.y / 2 + newSize.y / 2)
            plan:addBlock(newPos, newSize, blocks.a, -1, ColorRGB(1, 1, 1), Material(), Matrix(), blockType)

            -- -y
            local newPos = plan:getBlock(blocks.a).box.position
            newPos.y = newPos.y - (size.y / 2 + newSize.y / 2)
            plan:addBlock(newPos, newSize, blocks.a, -1, ColorRGB(1, 1, 1), Material(), Matrix(), blockType)


            -- block b
            local size = plan:getBlock(blocks.b).box.size

            -- +y
            local newPos = plan:getBlock(blocks.b).box.position
            newPos.y = newPos.y + (size.y / 2 + newSize.y / 2)
            plan:addBlock(newPos, newSize, blocks.b, -1, ColorRGB(1, 1, 1), Material(), Matrix(), blockType)

            -- -y
            local newPos = plan:getBlock(blocks.b).box.position
            newPos.y = newPos.y - (size.y / 2 + newSize.y / 2)
            plan:addBlock(newPos, newSize, blocks.b, -1, ColorRGB(1, 1, 1), Material(), Matrix(), blockType)

        end

    end

    local scale = getFloat(0.8, 1.3)
    plan:scale(vec3(scale, scale, scale))

    return plan
end


return PlanGenerator
