
package.path = package.path .. ";data/scripts/lib/?.lua"
package.path = package.path .. ";data/scripts/?.lua"

include ("randomext")
include ("galaxy")
include ("utility")
include ("stringutility")
include ("defaultscripts")
include ("goods")
include ("merchantutility")
ShipGenerator = include ("shipgenerator")
PlanGenerator = include ("plangenerator")

--local SectorGenerator = include ("SectorGenerator")

local AsteroidFieldGenerator = {}
AsteroidFieldGenerator.__index = AsteroidFieldGenerator

local function new(x, y)
    assert(type(x) == "number" and type(y) == "number", "New AsteroidFieldGenerator expects 2 numbers")

    return setmetatable({coordX = x, coordY = y}, AsteroidFieldGenerator)
end


-- every new function that gets added here must also be added to the SectorGenerator!


function AsteroidFieldGenerator:getPositionInSector(maxDist)
    -- deliberately not [-1;1] to avoid round sectors
    -- see getUniformPositionInSector(maxDist) below
    local position = vec3(math.random(), math.random(), math.random());
    local dist = 0
    if maxDist == nil then
        dist = getFloat(-5000, 5000)
    else
        dist = getFloat(-maxDist, maxDist)
    end

    position = position * dist

    -- create a random up vector
    local up = vec3(math.random(), math.random(), math.random())

    -- create a random right vector
    local look = vec3(math.random(), math.random(), math.random())

    -- create the look vector from them
    local mat = MatrixLookUp(look, up)
    mat.pos = position

    return mat
end

-- returns an asteroid type, based on the sector's position in the galaxy
function AsteroidFieldGenerator:getAsteroidType()
    local probabilities = Balancing_GetMaterialProbability(self.coordX, self.coordY)
    return Material(getValueFromDistribution(probabilities))
end

function AsteroidFieldGenerator:createClaimableAsteroid(position)
    local desc = AsteroidDescriptor()
    desc:removeComponent(ComponentType.MineableMaterial)
    desc:addComponents(
       ComponentType.Owner,
       ComponentType.FactionNotifier
       )

    desc.position = position or self:getPositionInSector()
    desc:setMovePlan(PlanGenerator.makeBigAsteroidPlan(100, false, Material(0)))
    desc:addScript("claim.lua")

    return Sector():createEntity(desc)
end

-- creates an asteroid
function AsteroidFieldGenerator:createSmallAsteroid(translation, size, resources, material)
    --acquire a random seed for the asteroid
    local plan = PlanGenerator.makeSmallAsteroidPlan(size, resources, material)
    plan.accumulatingHealth = false

    local position = MatrixLookUp(vec3(math.random(), math.random(), math.random()), vec3(math.random(), math.random(), math.random()))
    position.pos = translation

    local asteroid = Sector():createAsteroid(plan, resources, position)

    if resources then
        asteroid.isObviouslyMineable = true
    end

    return asteroid
end

function AsteroidFieldGenerator:createHiddenTreasureAsteroid(translation, size, material)
    --acquire a random seed for the asteroid
    local plan = PlanGenerator.makeSmallHiddenResourcesAsteroidPlan(size, material)
    plan.accumulatingHealth = false

    local position = MatrixLookUp(vec3(math.random(), math.random(), math.random()), vec3(math.random(), math.random(), math.random()))
    position.pos = translation

    local asteroid = Sector():createAsteroid(plan, true, position)
    asteroid.isObviouslyMineable = false
    return asteroid
end

-- create an asteroid field. this field is already placed randomly in the sector.
function AsteroidFieldGenerator:createAsteroidFieldEx(numAsteroids, fieldSize, minAsteroidSize, maxAsteroidSize, hasResources, probability)

    probability = probability or 0.05

    local asteroidsWithResources = numAsteroids * probability
    if not hasResources then asteroidsWithResources = 0 end

    local mat = self:getPositionInSector()
    local asteroids = {}

    for i = 1, numAsteroids do
        local resources = false
        if asteroidsWithResources > 0 then
            resources = true
            asteroidsWithResources = asteroidsWithResources - 1
        end

        -- create asteroid size from those min/max values and the actual value
        local size
        local hiddenTreasure = false

        if math.random() < 0.15 then
            size = lerp(math.random(), 0, 1.0, minAsteroidSize, maxAsteroidSize);
            if resources then
                resources = false
                asteroidsWithResources = asteroidsWithResources + 1
            end
        else
            size = lerp(math.random(), 0, 2.5, minAsteroidSize, maxAsteroidSize);
        end

        if math.random() < (1 / 50) then
            hiddenTreasure = true
        end

        -- create the local position in the field
        local angle = getFloat(0, math.pi * 2.0)
        local height = getFloat(-fieldSize / 5, fieldSize / 5)

        local distFromCenter = getFloat(0, fieldSize * 0.75)
        local asteroidPosition = vec3(math.sin(angle) * distFromCenter, height, math.cos(angle) * distFromCenter)

        asteroidPosition = mat:transformCoord(asteroidPosition)

        local material = self:getAsteroidType()

        local asteroid = nil
        if hiddenTreasure then
            asteroid = self:createHiddenTreasureAsteroid(asteroidPosition, size, material)
        else
            asteroid = self:createSmallAsteroid(asteroidPosition, size, resources, material)
        end
        table.insert(asteroids, asteroid)
    end

    return mat, asteroids
end

-- create an asteroid field with lots of tiny asteroids. This field is already placed randomly in the sector.
function AsteroidFieldGenerator:createAsteroidFieldWithTinyAsteroidsEx(numAsteroids, fieldSize, minAsteroidSize, maxAsteroidSize, hasResources, probability)

    numAsteroids = numAsteroids * 2

    probability = probability or 0.05

    local asteroidsWithResources = numAsteroids * probability
    if not hasResources then asteroidsWithResources = 0 end

    local mat = self:getPositionInSector()
    local asteroids = {}

    for i = 1, numAsteroids do
        local resources = false
        if asteroidsWithResources > 0 then
            resources = true
            asteroidsWithResources = asteroidsWithResources - 1
        end

        -- create asteroid size from those min/max values and the actual value
        local size
        local hiddenTreasure = false

            size = lerp(0.01, 0, 2.5, minAsteroidSize/2, maxAsteroidSize/2)

        if math.random() < (1 / 50) then
            hiddenTreasure = true
        end

        -- create the local position in the field
        local angle = getFloat(0, math.pi * 2.0)
        local height = getFloat(-fieldSize / 5, fieldSize / 5)

        local distFromCenter = getFloat(0, fieldSize * 0.75)
        local asteroidPosition = vec3(math.sin(angle) * distFromCenter, height, math.cos(angle) * distFromCenter)

        asteroidPosition = mat:transformCoord(asteroidPosition)

        local material = self:getAsteroidType()

        local asteroid = nil
       if hiddenTreasure then
            asteroid = self:createHiddenTreasureAsteroid(asteroidPosition, size, material)
        else
            asteroid = self:createSmallAsteroid(asteroidPosition, size, resources, material)
        end
        table.insert(asteroids, asteroid)
    end
    return mat, asteroids
end

-- create a spiral asteroid field. This field is already placed randomly in the sector.
function AsteroidFieldGenerator:createSpiralAsteroidFieldEx(numAsteroids, fieldSize, minAsteroidSize, maxAsteroidSize, hasResources, probability)

    numAsteroids = 60 -- right now, all spiral asteroid fields have the same number of asteroids (240 = layers * numAsteroids)
    local layers = 4 -- the number of "layers" the spiral has (how "fat" it is). The fatter it is, the more asteroids it contains

    probability = probability or 0.05

    local asteroidsWithResources = numAsteroids * probability
    if not hasResources then asteroidsWithResources = 0 end

    local mat = self:getPositionInSector()

    -- variables for creating a logarithmic spiral (see wikipedia)
    local rOld = 1
    local phi = 0 -- the angle of the individual asteroids (increases for each new asteroid)
    local c = 5 -- the distance between the asteroids (don't make too big or there might be overflow)
    local a = 0.4 -- the general size of the spiral (don't make too big or there might be overflow)
    local k = 0.3 -- the tightness of the coils of the spiral (don't make too big or there might be overflow)

    local asteroids = {}

    -- spirals can have 1, 2, 3 or 4 arms
    local numArms = math.random(1,4)
    if numArms == 2 or numArms == 3 then layers = 6 end -- we need more asteroids, otherwise the arms are too thin (4 arms have to be very thin or they look too crowded)
    local offset = 0
    local counter = 1

    for j = 1, layers do
        if numArms == 1 then
            offset = 0
        elseif numArms == 2 then
            if counter == 1 or counter == 2 or counter == 3 then offset = 0
            elseif counter == 4 or counter == 5 or counter == 6 then offset = math.rad(180)  end
        elseif numArms == 3 then
            if counter == 1 or counter == 2 then offset = 0
            elseif counter == 3 or counter == 4 then offset = math.rad(120)
            elseif counter == 5 or counter == 6 then offset = math.rad(240)  end
        elseif numArms == 4 then
            if counter == 1 then offset = math.rad(0)
            elseif counter == 2 then offset = math.rad(90)
            elseif counter == 3 then offset = math.rad(180)
            elseif counter == 4 then offset = math.rad(270) end
        end

        for i = 1, numAsteroids do
            local resources = false
            if asteroidsWithResources > 0 then
                resources = true
                asteroidsWithResources = asteroidsWithResources - 1
            end
            -- create asteroid size from those min/max values and the actual value
            local size
            local hiddenTreasure = false

            if math.random() < 0.15 then
                size = lerp(math.random(), 0, 1.0, minAsteroidSize, maxAsteroidSize);
                if resources then
                    resources = false
                    asteroidsWithResources = asteroidsWithResources + 1
                end
            else
                size = lerp(math.random(), 0, 2.5, minAsteroidSize, maxAsteroidSize);
            end

            if math.random() < (1 / 50) then
                hiddenTreasure = true
            end

            -- create the spiral
            local tempOne = (a * c) / (rOld)
            phi = phi + tempOne
            local tempTwo = k * phi
            local tempThree = math.pow(2,tempTwo)
            local r = a * tempThree
            rOld = tempThree
            local x = r * (math.cos(phi + offset))
            local xcoord = (mat.pos.x + (x * 100)) -- the x coordinate based on the spiral's position in the sector
            local y = r * (math.sin(phi + offset))
            local ycoord = (mat.pos.y + (y * 100)) -- the y coordinate based on the spiral's position in the sector

            local asteroidPosition = vec3(xcoord, ycoord, mat.pos.z)
            asteroidPosition = mat:transformCoord(asteroidPosition)
            local material = self:getAsteroidType()

            local asteroid = nil
            if hiddenTreasure then
                asteroid = self:createHiddenTreasureAsteroid(asteroidPosition, size, material)
            else
                asteroid = self:createSmallAsteroid(asteroidPosition, size, resources, material)
            end
            table.insert(asteroids, asteroid)
         end
        counter = counter + 1
        phi = 0
        rOld = 1
        mat.pos.z = mat.pos.z + 20
    end
    return mat, asteroids
end

-- create a donut asteroid field. This field is already placed randomly in the sector.
function AsteroidFieldGenerator:createRingAsteroidFieldEx(numAsteroids, fieldSize, minAsteroidSize, maxAsteroidSize, hasResources, probability)

    numAsteroids = 50 -- the number of asteroids that can potentially be in one ring of one layer. Only about a third of them are actually created
    local rings = 6 -- the higher the number of rings, the smaller the hole in the middle
    local layers = 4 -- the higher the number of layers, the fatter the ring --> actual number of asteroids is about:  numAsteroids/3 * layers * rings (here: about 400)

    probability = probability or 0.05

    local asteroidsWithResources = numAsteroids * probability
    if not hasResources then asteroidsWithResources = 0 end

    local mat = self:getPositionInSector()
    local zcoord = mat.pos.z

    local radius = 15
    local angle = 1

    local asteroids = {}

    for k = 1, layers do
        radius = 15
        angle = 1

        for j = 1, rings do
            for i = 1, numAsteroids do
                local resources = false
                if asteroidsWithResources > 0 then
                    resources = true
                    asteroidsWithResources = asteroidsWithResources - 1
                end
                -- create asteroid size from those min/max values and the actual value
                local size
                local hiddenTreasure = false

                if math.random() < 0.15 then
                    size = lerp(math.random(), 0, 1.0, minAsteroidSize, maxAsteroidSize);
                    if resources then
                        resources = false
                        asteroidsWithResources = asteroidsWithResources + 1
                    end
                else
                    size = lerp(math.random(), 0, 2.5, minAsteroidSize, maxAsteroidSize);
                end

                if math.random() < (1 / 50) then
                    hiddenTreasure = true
                end

                -- create the circle
                local phi = math.rad((angle / numAsteroids) * 360);
                local offset = (math.random(1, 6)) / 10
                angle = angle + 1 + offset

                local x = radius * (math.cos(phi))
                local xcoord = (mat.pos.x + (x * 100)) -- the x coordinate based on the circle's position in the sector
                local y = radius * (math.sin(phi))
                local ycoord = (mat.pos.y + (y * 100)) -- the y coordinate based on the circle's position in the sector

                local asteroidPosition = vec3(xcoord, ycoord, zcoord + (math.random(40, 60)))
                asteroidPosition = mat:transformCoord(asteroidPosition)
                local material = self:getAsteroidType()

                local asteroid = nil
                if hiddenTreasure then
                    local random = math.random(1,3)
                    if random == 1  then
                        asteroid = self:createHiddenTreasureAsteroid(asteroidPosition, size, material)
                    end
                else
                    local random = math.random(1,3)
                    if random == 1 then
                        asteroid = self:createSmallAsteroid(asteroidPosition, size, resources, material)
                    end
                end
                table.insert(asteroids, asteroid)
            end
            radius = radius - 0.5
            angle = 1
        end
        zcoord = zcoord + 50
    end
    return mat, asteroids
end


-- create a ball (hedgehog) asteroid field. This field is either placed randomly in the sector or at the position that is given in the parameters
function AsteroidFieldGenerator:createBallAsteroidFieldEx(numAsteroids, fieldSize, minAsteroidSize, maxAsteroidSize, hasResources, probability, position)
    -- the parameter position must be a vec3 with the coordinates of the spot you want to spawn the center of the ball on

    local layers = 10
    numAsteroids = layers * 26 -- the number of asteroids, only the number of layers should be changed to generate a different number of asteroids

    probability = probability or 0.05

    local asteroidsWithResources = numAsteroids * probability
    if not hasResources then asteroidsWithResources = 0 end

    local mat = self:getPositionInSector()
    if position ~= nil then
        mat.position = position
    end

    local radius = 5

    local asteroids = {}
    local coordinates ={} -- saves the coordinates of the points on the ball

    -- 26 evently spaced points on a ball are calculated, then a certain number of balls (= layers) are placed inside each other with a new radius
    for i = 1, layers do

        -- the coordinates of the 26 evenly spaced points one ball are calculated individually
        table.insert(coordinates, vec3((radius * math.sin(math.rad(45)) * math.cos(math.rad(0)  ) * 100) + mat.pos.x, (radius * math.sin(math.rad(45)) * math.sin(math.rad(0)  ) * 100) + mat.pos.y, (radius * math.cos(math.rad(45)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(45)) * math.cos(math.rad(45) ) * 100) + mat.pos.x, (radius * math.sin(math.rad(45)) * math.sin(math.rad(45) ) * 100) + mat.pos.y, (radius * math.cos(math.rad(45)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(45)) * math.cos(math.rad(90) ) * 100) + mat.pos.x, (radius * math.sin(math.rad(45)) * math.sin(math.rad(90) ) * 100) + mat.pos.y, (radius * math.cos(math.rad(45)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(45)) * math.cos(math.rad(135)) * 100) + mat.pos.x, (radius * math.sin(math.rad(45)) * math.sin(math.rad(135)) * 100) + mat.pos.y, (radius * math.cos(math.rad(45)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(45)) * math.cos(math.rad(180)) * 100) + mat.pos.x, (radius * math.sin(math.rad(45)) * math.sin(math.rad(180)) * 100) + mat.pos.y, (radius * math.cos(math.rad(45)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(45)) * math.cos(math.rad(225)) * 100) + mat.pos.x, (radius * math.sin(math.rad(45)) * math.sin(math.rad(225)) * 100) + mat.pos.y, (radius * math.cos(math.rad(45)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(45)) * math.cos(math.rad(270)) * 100) + mat.pos.x, (radius * math.sin(math.rad(45)) * math.sin(math.rad(270)) * 100) + mat.pos.y, (radius * math.cos(math.rad(45)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(45)) * math.cos(math.rad(315)) * 100) + mat.pos.x, (radius * math.sin(math.rad(45)) * math.sin(math.rad(315)) * 100) + mat.pos.y, (radius * math.cos(math.rad(45)) * 100) + mat.pos.z))

        table.insert(coordinates, vec3((radius * math.sin(math.rad(90)) * math.cos(math.rad(0)  ) * 100) + mat.pos.x, (radius * math.sin(math.rad(90)) * math.sin(math.rad(0)  ) * 100) + mat.pos.y, (radius * math.cos(math.rad(90)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(90)) * math.cos(math.rad(45) ) * 100) + mat.pos.x, (radius * math.sin(math.rad(90)) * math.sin(math.rad(45) ) * 100) + mat.pos.y, (radius * math.cos(math.rad(90)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(90)) * math.cos(math.rad(90) ) * 100) + mat.pos.x, (radius * math.sin(math.rad(90)) * math.sin(math.rad(90) ) * 100) + mat.pos.y, (radius * math.cos(math.rad(90)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(90)) * math.cos(math.rad(135)) * 100) + mat.pos.x, (radius * math.sin(math.rad(90)) * math.sin(math.rad(135)) * 100) + mat.pos.y, (radius * math.cos(math.rad(90)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(90)) * math.cos(math.rad(180)) * 100) + mat.pos.x, (radius * math.sin(math.rad(90)) * math.sin(math.rad(180)) * 100) + mat.pos.y, (radius * math.cos(math.rad(90)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(90)) * math.cos(math.rad(225)) * 100) + mat.pos.x, (radius * math.sin(math.rad(90)) * math.sin(math.rad(225)) * 100) + mat.pos.y, (radius * math.cos(math.rad(90)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(90)) * math.cos(math.rad(270)) * 100) + mat.pos.x, (radius * math.sin(math.rad(90)) * math.sin(math.rad(270)) * 100) + mat.pos.y, (radius * math.cos(math.rad(90)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(90)) * math.cos(math.rad(315)) * 100) + mat.pos.x, (radius * math.sin(math.rad(90)) * math.sin(math.rad(315)) * 100) + mat.pos.y, (radius * math.cos(math.rad(90)) * 100) + mat.pos.z))

        table.insert(coordinates, vec3((radius * math.sin(math.rad(135)) * math.cos(math.rad(0)  ) * 100) + mat.pos.x, (radius * math.sin(math.rad(135)) * math.sin(math.rad(0)  ) * 100) + mat.pos.y, (radius * math.cos(math.rad(135)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(135)) * math.cos(math.rad(45) ) * 100) + mat.pos.x, (radius * math.sin(math.rad(135)) * math.sin(math.rad(45) ) * 100) + mat.pos.y, (radius * math.cos(math.rad(135)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(135)) * math.cos(math.rad(90) ) * 100) + mat.pos.x, (radius * math.sin(math.rad(135)) * math.sin(math.rad(90) ) * 100) + mat.pos.y, (radius * math.cos(math.rad(135)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(135)) * math.cos(math.rad(135)) * 100) + mat.pos.x, (radius * math.sin(math.rad(135)) * math.sin(math.rad(135)) * 100) + mat.pos.y, (radius * math.cos(math.rad(135)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(135)) * math.cos(math.rad(180)) * 100) + mat.pos.x, (radius * math.sin(math.rad(135)) * math.sin(math.rad(180)) * 100) + mat.pos.y, (radius * math.cos(math.rad(135)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(135)) * math.cos(math.rad(225)) * 100) + mat.pos.x, (radius * math.sin(math.rad(135)) * math.sin(math.rad(225)) * 100) + mat.pos.y, (radius * math.cos(math.rad(135)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(135)) * math.cos(math.rad(270)) * 100) + mat.pos.x, (radius * math.sin(math.rad(135)) * math.sin(math.rad(270)) * 100) + mat.pos.y, (radius * math.cos(math.rad(135)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(135)) * math.cos(math.rad(315)) * 100) + mat.pos.x, (radius * math.sin(math.rad(135)) * math.sin(math.rad(315)) * 100) + mat.pos.y, (radius * math.cos(math.rad(135)) * 100) + mat.pos.z))

        table.insert(coordinates, vec3((radius * math.sin(math.rad(0)) * math.cos(math.rad(180)) * 100) + mat.pos.x, (radius * math.sin(math.rad(0)) * math.sin(math.rad(180)) * 100) + mat.pos.y, (radius * math.cos(math.rad(0)) * 100) + mat.pos.z))
        table.insert(coordinates, vec3((radius * math.sin(math.rad(0)) * math.cos(math.rad(0)) * 100) + mat.pos.x, (radius * math.sin(math.rad(0)) * math.sin(math.rad(0)) * 100) + mat.pos.y, (radius * math.cos(math.rad(0)) * 100) + mat.pos.z))

    radius = radius + 0.8

    end

    for i = 1, numAsteroids do
        local resources = false
        if asteroidsWithResources > 0 then
            resources = true
            asteroidsWithResources = asteroidsWithResources - 1
        end

        -- create asteroid size from those min/max values and the actual value
        local size
        local hiddenTreasure = false

        if math.random() < 0.15 then
            size = lerp(math.random(), 0, 1.0, minAsteroidSize, maxAsteroidSize);
            if resources then
                resources = false
                asteroidsWithResources = asteroidsWithResources + 1
            end
        else
            size = lerp(math.random(), 0, 2.5, minAsteroidSize, maxAsteroidSize);
        end

        if math.random() < (1 / 50) then
            hiddenTreasure = true
        end

        -- create the local position in the field

        local asteroidPosition = coordinates[i] -- all the coordinates that were previously calculated are used

        local material = self:getAsteroidType()

        local asteroid = nil
        if hiddenTreasure then
            asteroid = self:createHiddenTreasureAsteroid(asteroidPosition, size, material)
        else
            asteroid = self:createSmallAsteroid(asteroidPosition, size, resources, material)
        end
        table.insert(asteroids, asteroid)
    end

    return mat, asteroids
end


-- create a forest asteroid field. This field is already placed randomly in the sector.
function AsteroidFieldGenerator:createForestAsteroidFieldEx(numAsteroids, fieldSize, minAsteroidSize, maxAsteroidSize, hasResources, probability)

    numAsteroids = 250 -- the number of asteroids

    probability = probability or 0.05

    local asteroidsWithResources = numAsteroids * probability
    if not hasResources then asteroidsWithResources = 0 end

    local mat = self:getPositionInSector()
    local xcoord = mat.pos.x
    local ycoord = mat.pos.y
    local zcoord = mat.pos.z

    local asteroids = {}

    local counter = 0
    local angle = getFloat(0, math.pi * 2.0)
    local height = getFloat(-fieldSize / 5, fieldSize / 5)
    local distFromCenter = getFloat(0, fieldSize * 0.75)

    for i = 1, numAsteroids do
        local resources = false
            if asteroidsWithResources > 0 then
                resources = true
                asteroidsWithResources = asteroidsWithResources - 1
            end
            -- create asteroid size from those min/max values and the actual value
            local size
            local hiddenTreasure = false

            if math.random() < 0.15 then
                size = lerp(math.random(), 0, 1.0, minAsteroidSize, maxAsteroidSize);
                if resources then
                    resources = false
                    asteroidsWithResources = asteroidsWithResources + 1
                end
            else
                size = lerp(math.random(), 0, 2.5, minAsteroidSize, maxAsteroidSize);
            end

            if math.random() < (1 / 50) then
                hiddenTreasure = true
            end

            zcoord = zcoord + 40
            counter = counter + 1
            local randomHeight = math.random(4,9)

            if counter == randomHeight or counter >= 10 then

                zcoord = mat.pos.z
                counter = 0
                angle = getFloat(0, math.pi * 2.0)
                height = getFloat(-fieldSize / 5, fieldSize / 5)
                distFromCenter = getFloat(0, fieldSize * 0.75)

            end

            local asteroidPosition = vec3(math.sin(angle) * distFromCenter, height, zcoord)

            asteroidPosition = mat:transformCoord(asteroidPosition)
            local material = self:getAsteroidType()

            local asteroid = nil
            if hiddenTreasure then
                asteroid = self:createHiddenTreasureAsteroid(asteroidPosition, size, material)
            else
                asteroid = self:createSmallAsteroid(asteroidPosition, size, resources, material)
            end
            table.insert(asteroids, asteroid)
        end
    return mat, asteroids
end

function AsteroidFieldGenerator:createAsteroidFieldWithTinyAsteroids(probability)
    local size = getFloat(0.5, 1.25)

    return self:createAsteroidFieldWithTinyAsteroidsEx(300 * size, 400 * size, 5.0, 25.0, true, probability);
end

function AsteroidFieldGenerator:createSpiralAsteroidField(probability)
    local size = getFloat(0.5, 1.0)

    return self:createSpiralAsteroidFieldEx(300 * size, 1800 * size, 5.0, 25.0, true, probability);
end

function AsteroidFieldGenerator:createRingAsteroidField(probability)
    local size = getFloat(0.5, 1.0)

    return self:createRingAsteroidFieldEx(300 * size, 1800 * size, 5.0, 25.0, true, probability);
end

function AsteroidFieldGenerator:createForestAsteroidField(probability)
    local size = getFloat(0.5, 1.0)

    return self:createForestAsteroidFieldEx(300 * size, 1800 * size, 5.0, 25.0, true, probability);
end

function AsteroidFieldGenerator:createBallAsteroidField(probability, position)
    local size = getFloat(0.5, 1.0)

    return self:createBallAsteroidFieldEx(300 * size, 1800 * size, 5.0, 25.0, true, probability, position);
end

return setmetatable({new = new}, {__call = function(_, ...) return new(...) end})
