package.path = package.path .. ";data/scripts/lib/?.lua"
package.path = package.path .. ";data/scripts/?.lua"

include("stringutility")
include("galaxy")

local function getQuadrant(hx, hy, x, y)
    if x >= hx then
        if y >= hy then
            return 2
        else
            return 4
        end
    else
        if y >= hy then
            return 1
        else
            return 3
        end
    end
end

local toYesNo = function(line, value)
    if value then
        line.rtext = "Yes"%_t
        line.rcolor = ColorRGB(0.3, 1.0, 0.3)
    else
        line.rtext = "No"%_t
        line.rcolor = ColorRGB(1.0, 0.3, 0.3)
    end

    return line
end

local QuadrantNames = {}
QuadrantNames[1] = "North West"%_t
QuadrantNames[2] = "North East"%_t
QuadrantNames[3] = "South West"%_t
QuadrantNames[4] = "South East"%_t

function create(item, rarity, factionIndex, hx, hy, x, y)

    item.stackable = true
    item.depleteOnUse = true
    item.icon = "data/textures/icons/o-brick.png"
    item.rarity = rarity
    item:setValue("subtype", "FactionMapSegment")
    item:setValue("factionIndex", factionIndex)

    local price = 0
    if rarity.value >= RarityType.Exotic then
        item.name = "Faction Territory Map"%_t
        price = 500000
    elseif rarity.value >= RarityType.Exceptional then
        item.name = "Faction Quadrant Map"%_t
        price = 250000
    elseif rarity.value >= RarityType.Rare then
        item.name = "Explorer's Quadrant Map"%_t
        price = 30000
    else
        item.name = "Traveler's Quadrant Map"%_t
        price = 10000
    end

    item.price = price * Balancing_GetSectorRichnessFactor(hx, hy)

    local quadrant = getQuadrant(hx, hy, x, y)
    item:setValue("quadrant", quadrant)

    local tooltip = Tooltip()
    tooltip.icon = item.icon

    local title = item.name

    local headLineSize = 25
    local headLineFontSize = 15
    local line = TooltipLine(headLineSize, headLineFontSize)
    line.ctext = title
    line.ccolor = item.rarity.color
    tooltip:addLine(line)

    -- empty line
    tooltip:addLine(TooltipLine(14, 14))

    local line = TooltipLine(18, 14)
    line.ltext = "Faction"%_t
    line.rtext = "${faction:"..factionIndex.."}"
    line.icon = "data/textures/icons/flying-flag.png"
    line.iconColor = ColorRGB(0.8, 0.8, 0.8)
    tooltip:addLine(line)

    -- empty line
    tooltip:addLine(TooltipLine(14, 14))

    local line = TooltipLine(18, 14)
    line.ltext = "Quadrant"%_t
    line.icon = "data/textures/icons/o-brick.png"
    line.iconColor = ColorRGB(0.8, 0.8, 0.8)

    if rarity.value >= RarityType.Exotic then
        toYesNo(line, true)
        line.rtext = "Full Territory"%_t
    else
        line.rtext = QuadrantNames[quadrant]
    end

    tooltip:addLine(line)

    -- empty line
    tooltip:addLine(TooltipLine(14, 14))

    local line = TooltipLine(18, 14)
    line.ltext = "Gate Network"%_t
    line.icon = "data/textures/icons/patrol.png"
    line.iconColor = ColorRGB(0.8, 0.8, 0.8)
    tooltip:addLine(toYesNo(line, true))

    local line = TooltipLine(18, 14)
    line.ltext = "Additional Sectors"%_t
    line.icon = "data/textures/icons/diamonds.png"
    line.iconColor = ColorRGB(0.8, 0.8, 0.8)
    tooltip:addLine(toYesNo(line, rarity.value >= RarityType.Rare))

    local line = TooltipLine(18, 14)
    line.ltext = "Sector Stations"%_t
    line.icon = "data/textures/icons/checklist.png"
    line.iconColor = ColorRGB(0.8, 0.8, 0.8)
    tooltip:addLine(toYesNo(line, rarity.value >= RarityType.Exceptional))


    -- empty line
    tooltip:addLine(TooltipLine(14, 14))

    local line = TooltipLine(20, 15)
    line.ltext = "Depleted on Use"%_t
    line.lcolor = ColorRGB(1.0, 1.0, 0.3)
    tooltip:addLine(line)


    -- empty line
    tooltip:addLine(TooltipLine(14, 14))

    local line = TooltipLine(18, 14)
    line.ltext = "Can be activated by the player"%_t
    tooltip:addLine(line)

    local line = TooltipLine(18, 14)
    line.ltext = "Unveils a quadrant of the faction's territory."%_t
    tooltip:addLine(line)

    local line = TooltipLine(18, 14)
    line.ltext = "Territory info is loaded into the Galaxy Map"%_t
    tooltip:addLine(line)

    -- empty line
    tooltip:addLine(TooltipLine(14, 14))

    local line = TooltipLine(18, 14)
    line.ltext = "A chip with territory information."%_t
    line.lcolor = ColorRGB(0.4, 0.4, 0.4)
    tooltip:addLine(line)

    item:setTooltip(tooltip)

    return item
end


local code = [[
package.path = package.path .. ";data/scripts/lib/?.lua"
package.path = package.path .. ";data/scripts/?.lua"

include("stringutility")
local FactoryPredictor = include ("factorypredictor")
local SectorSpecifics = include ("sectorspecifics")
local GatesMap = include ("gatesmap")

local gatesMap

local function fillSectorView(view, specs, withOffgrid, withContent)

    local x, y = specs.coordinates.x, specs.coordinates.y
    local contents = specs.generationTemplate.contents(x, y)

    view:setCoordinates(x, y)

    if specs.gates then
        local connections = gatesMap:getConnectedSectors(specs.coordinates)

        local gateDestinations = {}
        for _, connection in pairs(connections) do
            table.insert(gateDestinations, ivec2(connection.x, connection.y))
        end

        view:setGateDestinations(unpack(gateDestinations))
    end

    view.factionIndex = specs.factionIndex

    if withContent then
        local stations = contents.stations - (contents.neighborTradingPosts or 0)
        view.influence = view:calculateInfluence(stations)
        view.numStations = contents.stations
        view.numShips = contents.ships

        local titles = {}

        for i = 1, (contents.shipyards or 0) do table.insert(titles, NamedFormat("Shipyard"%_t, {})) end
        for i = 1, (contents.repairDocks or 0) do table.insert(titles, NamedFormat("Repair Dock"%_t, {})) end
        for i = 1, (contents.scrapyards or 0) do table.insert(titles, NamedFormat("Scrapyard"%_t, {})) end
        for i = 1, (contents.resourceDepots or 0) do table.insert(titles, NamedFormat("Resource Depot"%_t, {})) end
        for i = 1, (contents.equipmentDocks or 0) do table.insert(titles, NamedFormat("Equipment Dock"%_t, {})) end
        for i = 1, (contents.turretFactories or 0) do table.insert(titles, NamedFormat("Turret Factory"%_t, {})) end
        for i = 1, (contents.turretFactorySuppliers or 0) do table.insert(titles, NamedFormat("Turret Factory Supplier"%_t, {})) end
        for i = 1, (contents.fighterFactories or 0) do table.insert(titles, NamedFormat("Fighter Factory"%_t, {})) end
        for i = 1, (contents.headquarters or 0) do table.insert(titles, NamedFormat("Headquarter"%_t, {})) end
        for i = 1, (contents.casinos or 0) do table.insert(titles, NamedFormat("Casino"%_t, {})) end
        for i = 1, (contents.biotopes or 0) do table.insert(titles, NamedFormat("Biotope"%_t, {})) end
        for i = 1, (contents.habitats or 0) do table.insert(titles, NamedFormat("Habitat"%_t, {})) end
        for i = 1, (contents.researchStations or 0) do table.insert(titles, NamedFormat("Research Station"%_t, {})) end
        for i = 1, (contents.militaryOutposts or 0) do table.insert(titles, NamedFormat("Military Outpost"%_t, {})) end
        for i = 1, (contents.resistanceOutposts or 0) do table.insert(titles, NamedFormat("Resistance Outpost"%_t, {})) end
        for i = 1, (contents.planetaryTradingPosts or 0) do table.insert(titles, NamedFormat("Planetary Trading Post"%_t, {})) end
        for i = 1, (contents.smugglersMarkets or 0) do table.insert(titles, NamedFormat("Smuggler's Market"%_t, {})) end
        for i = 1, (contents.tradingPosts or 0) do table.insert(titles, NamedFormat("Trading Post"%_t, {})) end
        for i = 1, (contents.neighborTradingPosts or 0) do table.insert(titles, NamedFormat("Trading Post"%_t, {})) end

        local factories = contents.factories or 0
        if factories > 0 then
            local productions = FactoryPredictor.generateFactoryProductions(x, y, factories)
            for i = 1, factories do
                local production = productions[i]
                local str, args = formatFactoryName(production)

                table.insert(titles, NamedFormat(str, args))
            end
        end

        local mines = contents.mines or 0
        if mines > 0 then
            local productions = FactoryPredictor.generateMineProductions(x, y, mines)
            for i = 1, mines do
                local production = productions[i]
                local str, args = formatFactoryName(production)

                table.insert(titles, NamedFormat(str, args))
            end
        end

        if #titles ~= contents.stations then
            eprint ("mismatch: %i %i", x, y)
        end

        view:setStationTitles(unpack(titles))
    end

end

function run(playerIndex, factionIndex, withOffgrid, withContent, quadrant)
    local timer = HighResolutionTimer()
    timer:start()

    gatesMap = GatesMap(GameSeed())

    local player = Player(playerIndex)
    local faction = Faction(factionIndex)

    local hx, hy = faction:getHomeSectorCoordinates()

    -- +----+----+
    -- | NW | NE |
    -- | 1  | 2  |
    -- +--- H ---+
    -- | SW | SE |
    -- | 3  | 4  |
    -- +----+----+

    local startX = hx - 200
    local endX = hx + 200
    local startY = hy - 200
    local endY = hy + 200

    if quadrant then
        if quadrant == 3 then
            endX = hx
            endY = hy
        elseif quadrant == 4 then
            startX = hx
            endY = hy
        elseif quadrant == 1 then
            endX = hx
            startY = hy
        elseif quadrant == 2 then
            startX = hx
            startY = hy
        end
    end

    if startX < -500 then startX = -500 end
    if startY < -500 then startY = -500 end

    if endX > 500 then endX = 500 end
    if endY > 500 then endY = 500 end

    -- print ("h: %i %i, s: %i %i, e: %i %i, q: %i", hx, hy, startX, startY, endX, endY, quadrant)

    local specs = SectorSpecifics()

    local seed = GameSeed()
    for x = startX, endX do
        for y = startY, endY do
            local regular, offgrid, dust = specs.determineFastContent(x, y, seed)

            if regular then
                specs:initialize(x, y, seed)

                if specs.regular and specs.factionIndex == factionIndex
                        and specs.generationTemplate
                        and (withOffgrid or specs.gates) then
                    local view = player:getKnownSector(x, y) or SectorView()

                    if not view.visited then
                        fillSectorView(view, specs, withOffgrid, withContent)

                        player:updateKnownSectorPreserveNote(view)
                    end

                end
            end
            ::continuey::
        end
        ::continuex::
    end

    local view = player:getKnownSector(hx, hy) or SectorView()
    view:setCoordinates(hx, hy)
    view.note = "Home Sector"
    player:updateKnownSectorPreserveNote(view)

    player:sendChatMessage("", ChatMessageType.Information, "Map information added to the Galaxy Map."%_t)
    -- print ("done: %s", timer.seconds)
end

]]

function activate(item)

    local player = Player()

    local factionIndex = item:getValue("factionIndex")
    if not factionIndex then return false end

    local faction = Faction(factionIndex)
    if not faction.isAIFaction then return false end

    local quadrant = item:getValue("quadrant") or 1
    if item.rarity.value >= RarityType.Exotic then quadrant = nil end

    local withContent = item.rarity.value >= RarityType.Exceptional
    local withOffgrid = item.rarity.value >= RarityType.Rare

    async("", code, player.index, factionIndex, withOffgrid, withContent, quadrant)

    return true
end
