
package.path = package.path .. ";data/scripts/lib/?.lua"
include("utility")

-- Don't remove or alter the following comment, it tells the game the namespace this script lives in. If you remove it, the script will break.
-- namespace FactionEconomy
FactionEconomy = {}

if onServer() then

function FactionEconomy.getUpdateInterval()
    return 15 * 60
end

function FactionEconomy.initialize(preventExecution)
    if preventExecution == true then return end -- for tests

    async("", FactionEconomy.getCode())
end

function FactionEconomy.getCode()
    return [[
    package.path = package.path .. ";data/scripts/lib/?.lua"

    local FactionsMap = include("factionsmap")
    local SectorSpecifics = include("sectorspecifics")
    local FactionEconomyUtility = include("factioneconomyutility")

    function getSectorHash(x, y)
        return x * 10000 + y
    end

    function run()
        local profiler = Profiler("FactionEconomy")

        local galaxy = Galaxy()
        local existingFactions = {}
        local homeSectors = {}

        profiler:section("clear deprecated data")
        clearDeprecatedData()
        profiler:done()

        profiler:section("get faction indices")

        local seed = GameSeed()
        local factionsMap = FactionsMap(seed)
        for _, index in pairs(factionsMap:getFactionIndices()) do
            if galaxy:aiFactionExists(index) then
                existingFactions[index] = true
            end

            local homeSector = factionsMap:getHomeSector(index)
            homeSectors[getSectorHash(homeSector.x, homeSector.y)] = true
        end

        profiler:done()

        profiler:section("sector contents")
        local numSectorsByFaction, numStationsByFaction = gatherData(seed, factionsMap, homeSectors, existingFactions, profiler)
        profiler:done()

--        profiler:section("debug prints")
--        printDebugInformation(numSectorsByFaction, numStationsByFaction)
--        profiler:done()

        profiler:section("set faction values")
        refreshEconomyValues(numSectorsByFaction, numStationsByFaction, existingFactions)
--        profiler:print()
    end

    function clearDeprecatedData()
        for x = -499, 500 do
            for y = -499, 500 do
                setGlobal(getSectorHash(x, y), nil)
            end
        end
    end

    function gatherData(seed, factionsMap, homeSectors, existingFactions, profiler)
        local galaxy = Galaxy()

        local startX = -499
        local startY = -499
        local endX = 500
        local endY = 500

        local specs = SectorSpecifics()

        local numSectorsByFaction = {}
        for factionIndex, _ in pairs(existingFactions) do
            numSectorsByFaction[factionIndex] = 0
        end

        local numStationsByFaction = {}

        for _, index in pairs(factionsMap:getFactionIndices()) do
            numStationsByFaction[index] = 0
        end

        local seedInt32 = seed.int32
        local passageMap = PassageMap(seed)

        for x = startX, endX do
            for y = startY, endY do

                profiler:reenter("fast content")
                local regular = specs.determineRegular(x, y, seedInt32)
                profiler:done()

                if not regular and homeSectors[getSectorHash(x, y)] then
                    -- home sectors do not care about being regular
                    regular = true
                end

                if not regular then goto nextSector end


                profiler:reenter("passage map")
                local passable = passageMap:passable(x, y)
                profiler:done()

                if not passable then goto nextSector end

                profiler:reenter("sector stats")
                local sectorView = galaxy:getSectorView(x, y)
                profiler:done()

                local controllingFactionIndex
                local numStations = 0

                if sectorView then
                    for factionIndex, numStations in pairs(sectorView:getStationsByFaction()) do
                        if numStationsByFaction[factionIndex] ~= nil then
                            numStationsByFaction[factionIndex] = numStationsByFaction[factionIndex] + numStations
                        end
                    end

                    controllingFactionIndex = sectorView.factionIndex

                else
                    profiler:reenter("sector specifics")
                    specs:initialize(x, y, seed)
                    profiler:done()

                    if specs.regular then
                        controllingFactionIndex = specs.factionIndex

                        -- only get detailed contents if the faction actually already exists
                        if controllingFactionIndex and existingFactions[controllingFactionIndex] and specs.generationTemplate then

--                            print("templates: " .. specs.generationTemplate.path)

                            profiler:reenter("sector stations")
                            local contentsTable = {specs.generationTemplate.contents(x, y)}

                            updateStationsFromContents(contentsTable, numStationsByFaction, specs.factionIndex, specs.generationTemplate.path)

                            profiler:done()

                        end
                    end
                end

                if controllingFactionIndex then
                    numSectorsByFaction[controllingFactionIndex] = (numSectorsByFaction[controllingFactionIndex] or 0) + 1
                end

                ::nextSector::
            end
        end

        return numSectorsByFaction, numStationsByFaction
    end

    function updateStationsFromContents(contentsTable, numStationsByFaction, factionIndex, templatePath)
        local contents = contentsTable[1]
        local faction = contents.faction or factionIndex

        if numStationsByFaction[faction] == nil then return end

        local numStations = 0

        numStations = numStations + (contents.biotopes or 0)
        numStations = numStations + (contents.casinos or 0)
        numStations = numStations + (contents.equipmentDocks or 0)
        numStations = numStations + (contents.factories or 0)
        numStations = numStations + (contents.fighterFactories or 0)
        numStations = numStations + (contents.habitats or 0)
        numStations = numStations + (contents.headquarters or 0)
        numStations = numStations + (contents.militaryOutposts or 0)
        numStations = numStations + (contents.mines or 0)
        numStations = numStations + (contents.planetaryTradingPosts or 0)
        numStations = numStations + (contents.repairDocks or 0)
        numStations = numStations + (contents.researchStations or 0)
        numStations = numStations + (contents.resourceDepots or 0)
        numStations = numStations + (contents.scrapyards or 0)
        numStations = numStations + (contents.smugglersMarkets or 0)
        numStations = numStations + (contents.tradingPosts or 0)
        numStations = numStations + (contents.turretFactories or 0)
        numStations = numStations + (contents.turretFactorySuppliers or 0)

        -- don't count neighbor trading posts
        -- they are in another faction's territory and would prevent eradicating a faction by only visiting sectors by that faction
--        if contents.neighborTradingPosts then
--            if type(contentsTable[4]) == "userdata" then
--                if contentsTable[4].__avoriontype == "Faction" then
--                    local neighborFaction = contentsTable[4]
--                    numStationsByFaction[neighborFaction] = numStationsByFaction[neighborFaction] + contents.neighborTradingPosts
--                else
--                    print("unexpected error: sector with neighbor trading post: userdata is not a faction.")
--                end
--            elseif type(contentsTable[4]) == "table" then
--                for neighborFaction, _ in pairs(contentsTable[4]) do
--                    numStationsByFaction[neighborFaction] = numStationsByFaction[neighborFaction] + 1
--                end
--            else
--                print("unexpected error: sector with neighbor trading posts: invalid type for return value 4")
--            end
--        end

        -- don't add shipyards generated in piratestation sectors, they belong to pirates
        -- this is not actually required because pirate sectors are offgrid
        -- added for the sake of completeness
        if templatePath ~= "sectors/piratestation" then
            numStations = numStations + (contents.shipyards or 0)
        end

        numStationsByFaction[faction] = numStationsByFaction[faction] + numStations
    end

    function printDebugInformation(numSectorsByFaction, numStationsByFaction)
        local minSectors, maxSectors
        local avgSectors = 0
        local totalFactions = 0

        for factionIndex, numSectors in pairs(numSectorsByFaction) do
            if minSectors == nil or numSectors < minSectors then minSectors = numSectors end
            if maxSectors == nil or numSectors > maxSectors then maxSectors = numSectors end

            avgSectors = avgSectors + numSectors
            totalFactions = totalFactions + 1

--            print("faction " .. factionIndex .. " has " .. numSectors .. " sectors")
        end

        avgSectors = avgSectors / totalFactions
--        print("min sectors: " .. minSectors .. ", max sectors: " .. maxSectors .. ", avg sectors: " .. avgSectors)


        local minStations, maxStations
        local avgStations = 0
        local totalStations = 0

        for factionIndex, numStations in pairs(numStationsByFaction) do
            if minStations == nil or numStations < minStations then minStations = numStations end
            if maxStations == nil or numStations > maxStations then maxStations = numStations end

            avgStations = avgStations + numStations

--            print("faction: " .. factionIndex .. ", stations: " .. numStations .. ", sectors: " .. (numSectorsByFaction[factionIndex] or 1) .. ", stations per sector: " .. numStations / (numSectorsByFaction[factionIndex] or 1))
        end

        avgStations = avgStations / totalFactions
--        print("min stations: " .. minStations .. ", max stations: " .. maxStations .. ", avg stations: " .. avgStations)
    end

    function refreshEconomyValues(numSectorsByFaction, numStationsByFaction, existingFactions)
        for factionIndex, numSectors in pairs(numSectorsByFaction) do
            if not existingFactions[factionIndex] then goto continue end

            local faction = Faction(factionIndex)
            if not faction then goto continue end

            local areaFactor = math.log(1 + numSectors / 100, 2)

            -- base values
            local militaryPower = areaFactor
            local constructionPower = areaFactor

            local aggressive = faction:getTrait("aggressive") or 0 -- opposite: "peaceful"
            local influence = aggressive * aggressive
            if aggressive < 0 then influence = -influence end
            influence = influence * 0.6 + 1
            militaryPower = militaryPower * influence

            local maxMilitaryShips = math.ceil(40 * militaryPower)
            local maxConstruction = math.ceil(3 * constructionPower)

            -- update values
            local currentMilitaryShips = faction:getValue("military_ships") or maxMilitaryShips
            local currentConstruction = faction:getValue("construction") or maxConstruction


            -- time to restock military ships
            -- not shown: dependence on aggressive trait
            -- sectors | power | ticks | avg. time
            --      50 | ~0.58 |  ~3.4 | 51 minutes
            --     100 |  1    |   2   | 30 minutes
            --     150 | ~1.32 |  ~1.5 | 23 minutes

            local restockMilitary = faction:getValue("restock_military")
            if restockMilitary ~= nil then
                restockMilitary = restockMilitary - militaryPower

                while restockMilitary <= 0 do
                    currentMilitaryShips = math.min(currentMilitaryShips + 4, maxMilitaryShips)
                    restockMilitary = restockMilitary + 2
                end
            else
                restockMilitary = 2
            end

            local restockConstruction = faction:getValue("restock_construction")
            if restockConstruction ~= nil then
                restockConstruction = restockConstruction - 1

                if restockConstruction <= 0 then
                    currentConstruction = math.min(currentConstruction + math.ceil(math.max(1, constructionPower)), maxConstruction)
                    restockConstruction = restockConstruction + 3
                end
            else
                restockConstruction = 3
            end

            -- calculate the number of stations
            local numStations = numStationsByFaction[factionIndex] or 0

            if numSectors == 0 and numStations == 0 then
                currentMilitaryShips = 0
                currentConstruction = 0
                maxMilitaryShips = 0
                maxConstruction = 0

                FactionEconomyUtility.setFactionEradicated(faction)
            end

            -- set faction values
            faction:setValue("num_stations", numStations)

            faction:setValue("restock_military", restockMilitary)
            faction:setValue("restock_construction", restockConstruction)

            faction:setValue("military_ships", currentMilitaryShips)
            faction:setValue("construction", currentConstruction)

            faction:setValue("max_military_ships", maxMilitaryShips)
            faction:setValue("max_construction", maxConstruction)

            ::continue::
        end
    end
    ]]
end

function FactionEconomy.runImmediately()
    execute(FactionEconomy.getCode())
end

function FactionEconomy.update()
    async("", FactionEconomy.getCode())
end


end
