package.path = package.path .. ";data/scripts/lib/?.lua"

include ("callable")

-- Don't remove or alter the following comment, it tells the game the namespace this script lives in. If you remove it, the script will break.
-- namespace Spy
Spy = {}
Spy.interactionThreshold = -80000
Spy.targetFaction = nil

function Spy.interactionPossible(playerIndex, option)
    return CheckFactionInteraction(playerIndex, Spy.interactionThreshold)
end

function Spy.initialize()
    if onServer() then
        local station = Entity()
        if station.title == "" then
            station.title = "Smuggler's Market"%_t
        end

        -- find a faction to spy on
        local faction = Galaxy():getNearestFaction(Sector():getCoordinates())
        Spy.targetFaction = faction.index
    end

    if onClient() then
        EntityIcon().icon = "data/textures/icons/pixel/crate.png"

        -- request target faction from server
        Spy.uiInitialized = false
        invokeServerFunction("sync")
    end
end

function Spy.sync(useAlliance)
    local player = Player(callingPlayer)
    if not player then return end

    local interactingFaction = player
    if useAlliance then
        interactingFaction = player.alliance
    end

    if not interactingFaction then return end

    local key = "spy_timestamp_" .. tostring(Spy.targetFaction)
    local timeStamp = interactingFaction:getValue(key)
    local timeLeft = nil
    if timeStamp then
        timeLeft = math.max(0, timeStamp - Server().unpausedRuntime)
    end

    invokeClientFunction(player, "receiveData", Spy.targetFaction, timeLeft)
end
callable(Spy, "sync")

function Spy.receiveData(targetFaction, timeLeft)
    Spy.targetFaction = targetFaction
    Spy.timeLeft = timeLeft

--    print("client: target faction: " .. tostring(targetFaction) .. ", time left: " .. tostring(timeLeft))
    Spy.refreshUI()
end

function Spy.initUI()
    local res = getResolution()
    local size = vec2(600, 275)

    local menu = ScriptUI()
    local window = menu:createWindow(Rect(res * 0.5 - size * 0.5, res * 0.5 + size * 0.5))

    window.caption = "Hire Spy"%_t
    window.showCloseButton = 1
    window.moveable = 1
    menu:registerWindow(window, "Hire Spy"%_t);

    local icon = window:createPicture(Rect(vec2(200, 200)), "data/textures/icons/domino-mask.png")
    icon.isIcon = true
    icon.color = ColorARGB(0.5, 0, 0, 0)

    local lister = UIVerticalLister(Rect(window.size), 5, 10)
    Spy.targetLabel = window:createLabel(lister:nextRect(20), "", 14)

    Spy.offerLines = {}
    Spy.durationByButton = {}

    for _, duration in pairs({15, 30, 60}) do
        local splitter = UIArbitraryVerticalSplitter(lister:nextRect(30), 10, 0, 300, 450)

        local frameRect = splitter:partition(0)
        frameRect.upper = splitter:partition(1).upper
        window:createFrame(frameRect)

        local labelRect = splitter:partition(0)
        labelRect.lower = labelRect.lower + vec2(10, 0)
        local descriptionLabel = window:createLabel(labelRect, "", 14)
        descriptionLabel:setLeftAligned()

        local priceRect = splitter:partition(1)
        priceRect.upper = priceRect.upper - vec2(10, 0)
        local priceLabel = window:createLabel(priceRect, "", 14)
        priceLabel:setRightAligned()

        local button = window:createButton(splitter:partition(2), "Hire"%_t, "onHirePressed")

        table.insert(Spy.offerLines, {duration = duration, description = descriptionLabel, price = priceLabel, button = button.index})
        Spy.durationByButton[button.index] = duration
    end

    local rect = lister:nextRect(90)
    Spy.remainingTimeLabel = window:createLabel(rect, "", 14)
    Spy.remainingTimeLabel:setCenterAligned()

    local center = Spy.remainingTimeLabel.rect.center
    local iconSize = vec2(100, 100)
    icon.rect = Rect(center - iconSize, center + iconSize)

    local label = window:createLabel(lister:nextRect(50), "Spies will infiltrate the neighboring faction and give you information about their faction strength."%_t, 14)
    label.wordBreak = true

    Spy.uiInitialized = true
end

function Spy.refreshUI()
    if not Spy.uiInitialized then return end

    local stationFaction = Faction()
    local customerFaction = Spy.getCustomerFaction()

    for i, line in pairs(Spy.offerLines) do
        line.description.caption = "Hire spies for ${duration} minutes"%_t % {duration = line.duration}
        local price = Spy.getPriceAndTax(line.duration, stationFaction, customerFaction)
        line.price.caption = "¢${price}"%_t % {price = createMonetaryString(price)}
    end

    local faction
    if Spy.targetFaction ~= nil then
        faction = Faction(Spy.targetFaction)
    end

    local factionName = "Unavailable"%_t
    if faction then factionName = faction.translatedName end

    Spy.targetLabel.caption = "Target faction: ${name}"%_t % {name = factionName}
    if Spy.timeLeft == nil or Spy.timeLeft == 0 then
        Spy.remainingTimeLabel.caption = "The target is currently not under surveillance."%_t
    else
        local remaining, tbl = createDigitalTimeString(Spy.timeLeft)
        local timeUnit = "Seconds"%_t
        if tbl.hours > 0 then
            timeUnit = "Hours"%_t
        elseif tbl.minutes > 0 then
            timeUnit = "Minutes"%_t
        end

        Spy.remainingTimeLabel.caption = "Time remaining: ${remaining} ${unit}"%_t % {remaining = remaining, unit = timeUnit}
    end
end

function Spy.onShowWindow()
    local customer = Spy.getCustomerFaction()

    invokeServerFunction("sync", customer.isAlliance)
end

function Spy.getCustomerFaction()
    local customer = Player()
    local ship = customer.craft
    if ship.factionIndex == customer.allianceIndex then
        return customer.alliance
    end

    return customer
end

function Spy.onHirePressed(button)
    local duration = Spy.durationByButton[button.index]
    if duration == nil then return end

    local shipIndex = Player().craftIndex
    invokeServerFunction("hireSpy", shipIndex, duration)
end

function Spy.hireSpy(shipIndex, duration)
    local shipFaction, ship, player = getInteractingFactionByShip(shipIndex, callingPlayer, AlliancePrivilege.SpendResources)
    if not shipFaction then return end

    local station = Entity()
    local stationFaction = Faction()

    local targetFaction = Faction(Spy.targetFaction)
    if not Spy.targetFaction or not targetFaction then
        player:sendChatMessage(station.title, ChatMessageType.Normal, "Sorry, we can't offer our services at the moment."%_T)
        return
    end

    local price, tax = Spy.getPriceAndTax(duration, stationFaction, shipFaction)
--    print("hire: ${duration} minutes, ${price} credits" % {duration = duration, price = price})
--    print("faction to be spied on: " .. Spy.targetFaction)

    local canPay, msg, args = shipFaction:canPay(costs)
    if not canPay then
        player:sendChatMessage(station.title, 1, msg, unpack(args))
        return
    end

    receiveTransactionTax(station, tax)
    shipFaction:pay("Paid %1% Credits to spy on a faction"%_T, price)

    local key = "spy_timestamp_" .. tostring(Spy.targetFaction)
    local unpausedRuntime = Server().unpausedRuntime

    -- stack spies
    local currentTimestamp = shipFaction:getValue(key) or unpausedRuntime
    if currentTimestamp < unpausedRuntime then currentTimestamp = unpausedRuntime end

    local timestamp = currentTimestamp + duration * 60
    shipFaction:setValue(key, timestamp)

    player:invokeFunction("data/scripts/player/background/spymails.lua", "onSpyHired", Spy.targetFaction, timestamp)

    -- sync
    invokeClientFunction(player, "onShowWindow")
end
callable(Spy, "hireSpy")

function Spy.getPriceAndTax(duration, stationFaction, buyerFaction)
    local price = math.ceil((12 + 30 / 60 * (duration - 15)) * (1 + GetFee(stationFaction, buyerFaction))) * 1000
    local tax = round(price * 0.2)

    if stationFaction.index == buyerFaction.index then
        price = price - tax
        -- don't pay out for the second time
        tax = 0
    end

    return price, tax
end
